# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['trustpilot']

package_data = \
{'': ['*']}

install_requires = \
['aiohttp>=3.6.2,<4.0.0', 'click>=7.1.1,<8.0.0', 'requests>=2.23.0,<3.0.0']

setup_kwargs = {
    'name': 'trustpilot',
    'version': '8.0.0',
    'description': 'trustpilot api client including cli tool',
    'long_description': '# trustpilot\n\n[![Build Status](https://travis-ci.org/trustpilot/python-trustpilot.svg?branch=master)](https://travis-ci.org/trustpilot/python-trustpilot) [![Latest Version](https://img.shields.io/pypi/v/trustpilot.svg)](https://pypi.python.org/pypi/trustpilot) [![Python Support](https://img.shields.io/pypi/pyversions/trustpilot.svg)](https://pypi.python.org/pypi/trustpilot)\n\nPython HTTP client for [Trustpilot](https://developers.trustpilot.com/).\n\n### Features\n\n- Extends the [`requests.Session`](http://docs.python-requests.org/en/master/api/#requests.Session) class with automatic authentication for public and private endpoints\n- GET, POST, PUT, DELETE, HEAD, OPTIONS and PATCH methods are exposed on module level\n- Implements session factory and default singleton session\n- Provides a simple hook system\n- CLI tool with basic HTTP commands\n\n## Installation\n\nInstall the package from [PyPI](http://pypi.python.org/pypi/) using [pip](https://pip.pypa.io/):\n\n```\npip install trustpilot\n```\n\n## Getting Started\n\nThis client is using the [Requests](http://docs.python-requests.org/en/master/) library. Responses are standard [`requests.Response`](http://docs.python-requests.org/en/master/api/#requests.Response) objects. You can use it as a factory or as a singleton.\n\n### Use the singleton session\n\nUse the built-in `default session` to instantiate a globally accessible session.\n\n```python\nfrom trustpilot import client\nclient.default_session.setup(\n    api_host="https://api.trustpilot.com",\n    api_version="v1",\n    api_key="YOUR_API_KEY",\n    api_secret="YOUR_API_SECRET",\n    username="YOUR_TRUSTPILOT_BUSINESS_USERNAME",\n    password="YOUR_TRUSTPILOT_BUSINESS_PASSWORD"\n)\nresponse = client.get("/foo/bar")\n```\n\nYou can rely on environment variables for the setup of sessions so\n\n```bash\n$ env\nTRUSTPILOT_API_HOST=foobar.com\nTRUSTPILOT_API_VERSION=v1\nTRUSTPILOT_API_KEY=foo\nTRUSTPILOT_API_SECRET=bar\nTRUSTPILOT_USERNAME=username\nTRUSTPILOT_PASSWORD=password\n```\n\nWill work with the implicit `default_session` and the `TrustpilotSession.setup` method.\n\n```python\nfrom trustpilot import client\nclient.get("/foo/bar")\n```\n\n### Instantiate your own session\n\nYou can create as many sessions as you like, as long as you pass them around yourself.\n\n```python\nfrom trustpilot import client\nsession = client.TrustpilotSession(\n    api_host="https://api.trustpilot.com",\n    api_version="v1",\n    api_key="YOUR_API_KEY",\n    api_secret="YOUR_API_SECRET",\n    username="YOUR_TRUSTPILOT_BUSINESS_USERNAME",\n    password="YOUR_TRUSTPILOT_BUSINESS_PASSWORD"\n)\nresponse = session.get("/foo/bar")\n```\n\n## Async client\n\nSince version `3.0.0` you are able to use the `async_client` for `asyncio` usecases.\n\nTo use the default `async_client` session, using `env-vars` for settings, import is as following:\n\n```python\nimport asyncio\nfrom trustpilot import async_client\nloop = asyncio.get_event_loop()\n\nasync def get_response():\n    response = await async_client.get(\'/foo/bar\')\n    response_json = await response.json()\n\nloop.run_until_complete(get_response())\n```\n\nOr instantiate the session yourself with:\n\n```python\nimport asyncio\nfrom trustpilot import async_client\nloop = asyncio.get_event_loop()\n\nsession = async_client.TrustpilotAsyncSession(\n    api_host="https://api.trustpilot.com",\n    api_version="v1",\n    api_key="YOUR_API_KEY",\n    api_secret="YOUR_API_SECRET",\n    username="YOUR_TRUSTPILOT_BUSINESS_USERNAME",\n    password="YOUR_TRUSTPILOT_BUSINESS_PASSWORD"\n)\n\nasync def get_response():\n    response = await session.get(\'/foo/bar\')\n    response_json = await response.json()\n\nloop.run_until_complete(get_response())\n```\n\n### Advanced async usage\n\nThe async client uses an _asynccontextmanager_ under the hood to perform the supported request methods.\nA side effect of the implementation is that it buffers up all the content before returning it to the calling scope.\n\nYou can get around this limitation by using the _asynccontextmanager_ directly like in the next example.\n\n**Example with stream reading the raw aiohttp response object:**\n\n```\nimport asyncio\nfrom trustpilot import async_client\nloop = asyncio.get_event_loop()\n\nasync def get_response():\n    async with session.request_context_manager(\'get\', "/v1/foo/bar") as resp:\n        result = True\n        while True:\n            chunk = resp.content.read(8)\n            if not chunk:\n                break\n            result += chunk\n    return result\n\nloop.run_until_complete(get_response())\n```\n\n## Setup User Agent\n\nA UserAgent header can be specified in two ways:\n\n1. By populating the `TRUSTPILOT_USER_AGENT` environment var\n2. By creating your own (async/sync)-client instance, or calling `setup` on the `default_session`, and supplying the kwargs `user_agent=foobar`\n\nIf no user-agent is given it will autopopulate using the function in `get_user_agent` function in [auth.py](./trustpilot/auth.py)\n\n## CLI\n\nA command line tool `trustpilot_api_client` is bundled with the module. To invoke it, use:\n\n```bash\nUsage: trustpilot_api_client [OPTIONS] COMMAND [ARGS]...\n\nOptions:\n  --host TEXT               host name\n  --version TEST            api version\n  --key TEXT                api key\n  --secret TEXT             api secret\n  --token_issuer_host TEXT  token issuer host name\n  --username TEXT           Trustpilot username\n  --password TEXT           Trustpilot password\n  -c TEXT                   json config file name\n  -v, --verbose             verbosity level\n  --help                    Show this message and exit.\n\nCommands:\n  create-access-token  Get an access token\n  delete               Send a DELETE request\n  get                  Send a GET request\n  post                 Send a POST request with specified data\n  put                  Send a PUT request with specified data\n```\n\nIn order to use the **-c** option please supply the filename of a JSON in the following format:\n\n```json\n{\n  "TRUSTPILOT_API_HOST": "foo",\n  "TRUSTPILOT_API_VERSION": "v1",\n  "TRUSTPILOT_API_KEY": "bar",\n  "TRUSTPILOT_API_SECRET": "baz",\n  "TRUSTPILOT_USERNAME": "username",\n  "TRUSTPILOT_PASSWORD": "password"\n}\n```\n',
    'author': 'Johannes Valbjørn',
    'author_email': 'jgv@trustpilot.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/trustpilot/python-trustpilot',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
