"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoScaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class AutoScaler extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const launchTemplate = this.getLT(props.templateProps, props.asgName);
        const tgArn = this.getTG(props.tgProps, props.templateProps.vpc.vpcName);
        new aws_autoscaling_1.CfnAutoScalingGroup(this, props.asgName, {
            maxSize: props.maxSize,
            minSize: props.minSize,
            autoScalingGroupName: props.asgName,
            launchTemplate: {
                version: launchTemplate.versionNumber,
                launchTemplateId: launchTemplate.launchTemplateId,
                launchTemplateName: launchTemplate.launchTemplateName,
            },
            targetGroupArns: tgArn,
            tags: props.tags,
            availabilityZones: this.getZones(props.subnets),
            vpcZoneIdentifier: props.subnets,
            healthCheckGracePeriod: 300,
        });
        if (tgArn.length) {
            this.targetGroupArn = tgArn[0];
        }
        this.targetGroupArn = '';
    }
    getVPC(props) {
        const stackVPC = aws_ec2_1.Vpc.fromLookup(this, props.vpcName, {
            isDefault: false,
            vpcId: props.vpcName,
        });
        return stackVPC;
    }
    getRole(props, asgName) {
        var _b;
        if (props.type == 'existing') {
            const role = iam.Role.fromRoleArn(this, asgName + '-stackRole', props.roleArn);
            return role;
        }
        else {
            const role = new iam.Role(this, asgName + '-stackRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                roleName: asgName + '-role',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2SpotFleetTaggingRole',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/ReadOnlyAccess',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/AmazonEC2FullAccess',
            });
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'ec2:DescribeTags',
                ],
            }));
            role.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                resources: ['*'],
                actions: [
                    'iam:ListUsers',
                    'iam:GetGroup',
                ],
                sid: 'VisualEditor0',
            }));
            (_b = props.additionalPolicies) === null || _b === void 0 ? void 0 : _b.forEach(policyDoc => {
                role.addToPolicy(aws_iam_1.PolicyStatement.fromJson(policyDoc));
            });
            return role;
        }
    }
    getSG(props, vpc, asgName) {
        var _b, _c, _d;
        if (props.type == 'existing') {
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, 'stack-sg', props.sgGroupId);
            return securityGroup;
        }
        else {
            const sgProps = {
                securityGroupName: (_b = props.securityGroupName) !== null && _b !== void 0 ? _b : 'stack-sg-group',
                vpc: vpc,
                allowAllOutbound: (_c = props.allowAllOutbound) !== null && _c !== void 0 ? _c : true,
                disableInlineRules: (_d = props.disableInlineRules) !== null && _d !== void 0 ? _d : false,
            };
            const securityGroup = new aws_ec2_1.SecurityGroup(this, asgName + '-stack-sg', sgProps);
            props.ingressRules.forEach(ingress => {
                var _b;
                const ingressSG = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, ingress.sourceSG + ingress.port.toString(), ingress.sourceSG);
                securityGroup.connections.allowFrom(ingressSG, aws_ec2_1.Port.tcp(ingress.port), (_b = ingress.description) !== null && _b !== void 0 ? _b : 'Application port');
            });
            return securityGroup;
        }
    }
    getBD(props) {
        const bd = {
            deviceName: props.name,
            volume: aws_ec2_1.BlockDeviceVolume.ebs(props.size, {
                volumeType: props.type,
                deleteOnTermination: true,
            }),
        };
        return bd;
    }
    getLT(props, asgName) {
        if (props.type == 'existing') {
            const launchTemplate = aws_ec2_1.LaunchTemplate.fromLaunchTemplateAttributes(this, props.templateName, props.existingAttributes);
            return launchTemplate;
        }
        else {
            const launchTemplate = new aws_ec2_1.LaunchTemplate(this, props.templateName, {
                launchTemplateName: props.templateName,
                instanceType: new aws_ec2_1.InstanceType(props.instanceType),
                machineImage: aws_ec2_1.MachineImage.lookup({
                    name: props.amiImageId,
                }),
                securityGroup: this.getSG(props.securityGroup, this.getVPC(props.vpc), asgName),
                role: this.getRole(props.role, asgName),
                detailedMonitoring: false,
                blockDevices: [this.getBD(props.blockDevice)],
                keyName: props.sshKey,
            });
            return launchTemplate;
        }
    }
    getTG(props, vpcId) {
        if (props != undefined) {
            const tg = new aws_elasticloadbalancingv2_1.CfnTargetGroup(this, props.name, {
                name: props.name,
                healthCheckEnabled: true,
                healthCheckPath: props.healthPath,
                ...((props.protocol == 'GRPC') ? { protocol: 'HTTP' } : { protocol: props.protocol }),
                ...((props.protocol == 'GRPC') ? { protocolVersion: 'GRPC' } : {}),
                healthCheckTimeoutSeconds: props.timeout,
                healthCheckPort: String(props.port),
                port: props.port,
                vpcId: vpcId,
            });
            return [tg.ref];
        }
        else {
            return [];
        }
    }
    getZones(subnets) {
        var availabilityZones = [];
        subnets.forEach(subnet => {
            const net = aws_ec2_1.Subnet.fromSubnetAttributes(this, subnet, {
                availabilityZone: 'dummy',
                subnetId: subnet,
            });
            availabilityZones.push(net.availabilityZone);
        });
        return availabilityZones;
    }
}
exports.AutoScaler = AutoScaler;
_a = JSII_RTTI_SYMBOL_1;
AutoScaler[_a] = { fqn: "@smallcase/aws-cdk-microservice.AutoScaler", version: "0.0.1" };
//# sourceMappingURL=data:application/json;base64,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