import fnmatch
import os
import pathlib
import shutil
import tempfile
import zipfile

from hatchling.builders.hooks.plugin.interface import BuildHookInterface
from literary.commands.build import LiteraryBuildApp
from literary.config import find_literary_config, load_literary_config


def mangle_attribute(cls, name):
    return f"_{cls.__name__}__{name}"


class LiteraryBuildHook(BuildHookInterface):
    PLUGIN_NAME = 'literary'
    PTH_IMPORT_HOOK_COMMENT = (
        "# The following line was automatically generated by literary_build_hatch"
    )
    PTH_IMPORT_HOOK_TEMPLATE = (
        "import literary.hook; literary.hook.install_import_hook()"
    )

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)

        # Find literary config file
        root_path = pathlib.Path(self.root)
        config_path = find_literary_config(root_path)
        if config_path.parent != root_path:
            raise RuntimeError("missing literary config")

        # Build Python files
        self._builder = LiteraryBuildApp(config_file=config_path)

        # Ensure that we own build directory
        if self._builder.generated_path == root_path:
            raise RuntimeError("cannot generate inside root")

        # Store original wheel metadata constructor
        self._build_config = getattr(
            self, mangle_attribute(BuildHookInterface, "build_config")
        )
        self._metadata_constructor = self._build_config.core_metadata_constructor

    def _require_packages(self, packages):
        # Patch metadata constructor
        def core_metadata_constructor(metadata, extra_dependencies=()):
            return self._metadata_constructor(
                metadata, tuple(extra_dependencies) + tuple(packages)
            )

        setattr(
            self._build_config,
            mangle_attribute(type(self._build_config), "core_metadata_constructor"),
            core_metadata_constructor,
        )

    def _patch_editable_wheel(self, artifact):
        # As we cannot edit the wheel, we'll create a temporary file
        dst_fd, dst_path = tempfile.mkstemp()

        # To patch-in support for Literary, we'll augment the existing editables
        # support by appending to the first `.pth` created.
        has_import_hook = False
        with zipfile.ZipFile(artifact, "r") as src, zipfile.ZipFile(
            dst_path, "w"
        ) as dst:
            for info in src.infolist():
                # If we've found a .pth file
                if not has_import_hook and fnmatch.fnmatch(info.filename, "*.pth"):
                    # Patch-in Literary support
                    literary_hook_parts = self.PTH_IMPORT_HOOK_TEMPLATE.format(
                        # Import hook needs to know about the root packages directory
                        root_path=str(self._builder.packages_path)
                    ).splitlines()
                    # Build a single executable line
                    literary_hook_line = ";".join(
                        [l for l in literary_hook_parts if l.strip()]
                    )
                    # Concatenate with the existing hook
                    new_pth_contents = '\n'.join(
                        [
                            *src.read(info).decode().splitlines(),
                            self.PTH_IMPORT_HOOK_COMMENT,
                            literary_hook_line,
                        ]
                    )
                    # Write modified path config
                    dst.writestr(info, new_pth_contents)
                    # Don't visit this branch again
                    has_import_hook = True
                else:
                    dst.writestr(info, src.read(info))

        # Move new archive to replace existing artifact
        shutil.move(dst_path, artifact)

    def initialize(self, version, build_data):
        print("INI", self.target_name, version)
        if self.target_name == "wheel":
            # For editable wheels, we don't want to build anything for Literary
            # Instead, we just want to patch the final wheel to support the import hook
            if version == "editable":
                self._require_packages(("literary>=3.0.2",))

            # We only want to generate files for standard wheels
            elif version == "standard":
                self._builder.build()

    def finalize(self, version, build_data, artifact):
        # We can inject our editable support to wheels
        if self.target_name == "wheel" and version == "editable":
            self._patch_editable_wheel(artifact)

    def clean(self, versions):
        if hasattr(self._builder, "clean"):
            self._builder.clean()
