from abc import abstractmethod
from typing import Iterable, Tuple, Union

from physrisk.data.data_requests import EventDataResponse
from physrisk.kernel.impact_curve import ImpactCurve

from ..data import EventDataRequest
from .assets import Asset
from .curve import ExceedanceCurve
from .events import RiverineInundation
from .hazard_event_distrib import HazardEventDistrib
from .model import Model
from .vulnerability_distrib import VulnerabilityDistrib


class VulnerabilityModel(Model):
    """A model that"""

    def __init__(self, *, model: str, event_type: type, impact_bin_edges):
        self.event_type = event_type
        self.model = model
        self.impact_bin_edges = impact_bin_edges

    def get_event_data_requests(
        self, asset: Asset, *, scenario: str, year: int
    ) -> Union[EventDataRequest, Iterable[EventDataRequest]]:
        return EventDataRequest(
            self.event_type, asset.longitude, asset.latitude, scenario=scenario, year=year, model=self.model
        )

    def get_distributions(
        self, asset: Asset, event_data_responses: Iterable[EventDataResponse]
    ) -> Tuple[VulnerabilityDistrib, HazardEventDistrib]:

        (event_data,) = event_data_responses
        intensity_curve = ExceedanceCurve(1.0 / event_data.return_periods, event_data.intensities)
        intensity_bin_edges, probs = intensity_curve.get_probability_bins()
        intensity_bin_centres = (intensity_bin_edges[1:] + intensity_bin_edges[:-1]) / 2

        vul = VulnerabilityDistrib(
            type(RiverineInundation),
            intensity_bin_edges,
            self.impact_bin_edges,
            self.get_impact_curve(intensity_bin_centres).to_prob_matrix(self.impact_bin_edges),
        )

        event = HazardEventDistrib(type(RiverineInundation), intensity_bin_edges, probs)
        return vul, event

    @abstractmethod
    def get_impact_curve(self, intensities) -> ImpactCurve:
        """Get the impact curves for the intensitis specified."""
        ...
