"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'can create receipt rules with second after first'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    receiptRuleName: 'FirstRule',
                },
                {
                    enabled: false,
                    receiptRuleName: 'SecondRule',
                    recipients: ['hello@aws.com'],
                    scanEnabled: true,
                    tlsPolicy: lib_1.TlsPolicy.REQUIRE
                }
            ]
        });
        // THEN
        assert_1.expect(stack).toMatch({
            "Resources": {
                "RuleSetE30C6C48": {
                    "Type": "AWS::SES::ReceiptRuleSet"
                },
                "RuleSetRule023C3B8E1": {
                    "Type": "AWS::SES::ReceiptRule",
                    "Properties": {
                        "Rule": {
                            "Name": "FirstRule",
                            "Enabled": true
                        },
                        "RuleSetName": {
                            "Ref": "RuleSetE30C6C48"
                        }
                    }
                },
                "RuleSetRule117041B57": {
                    "Type": "AWS::SES::ReceiptRule",
                    "Properties": {
                        "Rule": {
                            "Enabled": false,
                            "Name": "SecondRule",
                            "Recipients": [
                                "hello@aws.com"
                            ],
                            "ScanEnabled": true,
                            "TlsPolicy": "Require"
                        },
                        "RuleSetName": {
                            "Ref": "RuleSetE30C6C48"
                        },
                        "After": {
                            "Ref": "RuleSetRule023C3B8E1"
                        }
                    }
                }
            }
        });
        test.done();
    },
    'import receipt rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const receiptRule = lib_1.ReceiptRule.fromReceiptRuleName(stack, 'ImportedRule', 'MyRule');
        const receiptRuleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        receiptRuleSet.addRule('MyRule', {
            after: receiptRule
        });
        // THEN
        assert_1.expect(stack).toMatch({
            "Resources": {
                "RuleSetE30C6C48": {
                    "Type": "AWS::SES::ReceiptRuleSet"
                },
                "RuleSetMyRule60B1D107": {
                    "Type": "AWS::SES::ReceiptRule",
                    "Properties": {
                        "Rule": {
                            "Enabled": true
                        },
                        "RuleSetName": {
                            "Ref": "RuleSetE30C6C48"
                        },
                        "After": "MyRule"
                    }
                }
            },
        });
        test.done();
    },
    'can add actions in rule props'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        ruleSet.addRule('Rule', {
            actions: [
                {
                    bind: () => ({
                        stopAction: {
                            scope: 'RuleSet'
                        }
                    })
                }
            ]
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            "Rule": {
                "Actions": [
                    {
                        "StopAction": {
                            "Scope": "RuleSet"
                        }
                    }
                ],
                "Enabled": true
            },
            "RuleSetName": {
                "Ref": "RuleSetE30C6C48"
            }
        }));
        test.done();
    },
    'can add action with addAction'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        const rule = ruleSet.addRule('Rule');
        rule.addAction({
            bind: () => ({
                stopAction: {
                    scope: 'RuleSet'
                }
            })
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            "Rule": {
                "Actions": [
                    {
                        "StopAction": {
                            "Scope": "RuleSet"
                        }
                    }
                ],
                "Enabled": true
            },
            "RuleSetName": {
                "Ref": "RuleSetE30C6C48"
            }
        }));
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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