# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/04_data_generator.ipynb.

# %% auto 0
__all__ = ['DataGenerator']

# %% ../nbs/04_data_generator.ipynb 4
import re
from typing import List, Callable, Tuple, Union

import torch
import torch.nn.functional as F

from torchtyping import TensorType
from einops import rearrange
from langchain import PromptTemplate

from .api import BaseAPI
from .api import CalculatorAPI

# %% ../nbs/04_data_generator.ipynb 5
class DataGenerator:
    def __init__(self, config: dict, model: Callable, tokenizer: Callable, apis: List[BaseAPI]):
        start_character = config["data_generator"]["api_start_character"]
        end_character = config["data_generator"]["api_end_character"]
        output_character = config["data_generator"]["api_output_character"]
        
        # add a space, because when the model generate a token, it's also include a "space"
        self.api_start_token = tokenizer(f' {start_character}', return_tensors="pt")["input_ids"][0]
        self.api_end_token = tokenizer(end_character, return_tensors="pt")["input_ids"][0]
        self.api_output_token = tokenizer(f'{output_character}', return_tensors="pt")["input_ids"][0]
        
        self.top_k = config["data_generator"]["top_k"]
        self.sampling_threshold = config["data_generator"]["sampling_threshold"]
        self.filtering_threshold = config["data_generator"]["filtering_threshold"]
        
        self.apis = apis
        self.model = model
        self.tokenizer = tokenizer
        
        # TODO: handle for cases that the sentence contains ".\n\n"
        self.pad_token_id = tokenizer.pad_token_id
        self.eos_token_id = tokenizer(".\n\n")["input_ids"][0]
    
    def extract_api_request_content(self, text: str, api_name: str) -> str:
        start_tag = f"{api_name}("
        end_tag = ")"
        start_idx = text.find(start_tag)
        if start_idx == -1:
            return None
        start_idx += len(start_tag)
        end_idx = text.find(end_tag, start_idx)
        if end_idx == -1:
            return None
        return text[start_idx:end_idx]
    
    def extract_api_syntax(self, sentence: str, api_name: str) -> str:
        pattern = r"\[{}\(.*?\)\]".format(api_name)
        matches = re.findall(pattern, sentence)
        return matches
    
    def sample_api_position(
        self,
        prompt_ids: TensorType["batch_size", "seq_len"], # the ids of the prompt
    ) -> Tuple[
        TensorType["batch_size", "n_positions"], # The positions of api call
        TensorType["batch_size", "seq_len"] # The generated text
    ]:
        # TODO: add support batch
        
        # the ids of the prompt and generated_ids
        prompt_and_generated_ids = prompt_ids
        # only the ids of the generated_ids
        generated_ids = torch.tensor([])
        api_positions = torch.tensor([])
        i = torch.tensor([0])
        
        with torch.no_grad():    
            while True:
                logits = self.model(
                    input_ids=prompt_and_generated_ids.unsqueeze(0),
                ).logits

                last_logit = logits[0, -1, :]
                probs = torch.softmax(last_logit, dim=-1)
                
                # find the top k tokens for api call
                # TODO: add filter by larger than sampling_threshold
                top_k_tokens = torch.topk(probs, k=5, dim=-1)
                
                if self.api_start_token in top_k_tokens.indices:
                    # api_position = torch.tensor([len(generated_ids)]) # the current idx
                    api_positions = torch.cat([api_positions, i], dim=0)
                
                # sampling a token
                # next_token = torch.multinomial(probs, num_samples=1)
                next_token = torch.argmax(probs, dim=-1)
                next_token = next_token.unsqueeze(0)
                
                prompt_and_generated_ids = torch.cat([prompt_and_generated_ids, next_token], dim=0)
                generated_ids = torch.cat([generated_ids, next_token], dim=0)
                
                if next_token == self.eos_token_id:
                    break
                else:
                    i += 1
        
        return api_positions.long(), generated_ids.long()

    def obtain_api_response(
        self,
        prompt_ids: TensorType["batch_size", "seq_len"],
        positions: TensorType["batch_size", "n_positions"],
        generated_ids: TensorType["batch_size", "seq_len"]
    ) -> TensorType["batch_size", "n_positions", "seq_len"]:
        
        MAX_PAD = 50
        PAD_TOKEN = self.tokenizer.pad_token_id
        
        # the ids before the start of an api call
        pre_api_ids = torch.tensor([])

        for position in positions:
            text_ids = torch.cat([generated_ids[:position], self.api_start_token], dim=0)
            padded_text_ids = F.pad(text_ids, pad=(MAX_PAD - text_ids.shape[-1], 0), value=PAD_TOKEN)
            
            pre_api_ids = torch.cat([
                pre_api_ids,
                rearrange(padded_text_ids, "... -> 1 ...")
            ])
        
        PROMPT_LENGTH = len(prompt_ids)
        
        # TODO: optimzie this
        prompt_and_pre_api_ids = torch.tensor([])
        for x in pre_api_ids:
            prompt_and_pre_api_ids = torch.cat([
                prompt_and_pre_api_ids,
                torch.cat([prompt_ids, x]).unsqueeze(0)
            ], dim=0)
                     
        with torch.no_grad():
            candidates = self.model.generate(
                input_ids=prompt_and_pre_api_ids.long(),
                eos_token_id=self.eos_token_id,
                max_new_tokens=50,
            )
        
        # filter out the prompt template
        # only keep the generated ids
        candidates = candidates[:, PROMPT_LENGTH:]
        
        return candidates
    
    def _generate_conditioning_prompts(
        self,
        candidate_ids: TensorType["batch_size", "n_candidates", "seq_len"],
    ):
        calculator_api = CalculatorAPI()
        conditioning_api_ids = torch.tensor([])

        API_NAME = "Calculator"
        MAX_PAD = 100
        PAD_TOKEN = self.tokenizer.pad_token_id

        for text_ids in candidate_ids:
            # the ids of the prediction
            text = self.tokenizer.decode(text_ids, skip_special_tokens=True)
            
            api_request_content = self.extract_api_request_content(text, api_name=API_NAME)
            api_response = calculator_api(api_request_content)
            api_response_ids = self.tokenizer(api_response, return_tensors="pt")["input_ids"][0]
            # Format: "-> [api_response]"
            api_response_with_arrow_ids = torch.cat([self.api_output_token, api_response_ids], dim=0)
            
            api_syntax = self.extract_api_syntax(text, api_name=API_NAME)
            api_syntax_ids = self.tokenizer(api_syntax, return_tensors="pt")["input_ids"][0]
            api_syntax_with_response_ids = torch.cat([api_syntax_ids[:-1], api_response_with_arrow_ids, api_syntax_ids[-1:]])
            api_syntax_without_response_ids = torch.cat([api_syntax_ids[:-1], self.api_output_token, api_syntax_ids[-1:]])
                              
            # padded_api_without_response = rearrange(
            #     F.pad(api_syntax_without_response_ids, pad=(0, (MAX_PAD - api_syntax_without_response_ids.shape[-1])), value=PAD_TOKEN),
            #     "... -> 1 ..."
            # )
            # padded_api_with_response = rearrange(
            #     F.pad(api_syntax_with_response_ids, pad=(0, (MAX_PAD - api_syntax_with_response_ids.shape[-1])), value=PAD_TOKEN),
            #     "... -> 1 ..."
            # )
            padded_api_without_response = rearrange(
                F.pad(api_syntax_without_response_ids, pad=((MAX_PAD - api_syntax_without_response_ids.shape[-1]), 0), value=PAD_TOKEN),
                "... -> 1 ..."
            )
            padded_api_with_response = rearrange(
                F.pad(api_syntax_with_response_ids, pad=((MAX_PAD - api_syntax_with_response_ids.shape[-1]), 0), value=PAD_TOKEN),
                "... -> 1 ..."
            )
        
            padded_api_call = torch.cat([
                padded_api_without_response,
                padded_api_with_response
            ], dim=0)
            padded_api_call = rearrange(padded_api_call, "... -> 1 ...")
            
            conditioning_api_ids = torch.cat([conditioning_api_ids, padded_api_call], dim=0).long()
                    
        return conditioning_api_ids

    def _compute_weight(self, t: int) -> Union[int, float]:
        """Compute the weight in the loss function."""
        return max(0, 1-0.2*t)

    
    def _normalize_weights(self, augmented_text_ids):
        """Normalize the weight of each position in a sequence."""
        for api_start_position in augmented_text_ids["api_start_positions"].values():
            total_weight = sum([seq_position["unnormalized_weight"] for seq_position in api_start_position["seq_positions"].values()])
            for seq_position in api_start_position["seq_positions"].values():
                seq_position["normalized_weight"] = seq_position["unnormalized_weight"] / total_weight
        
        return augmented_text_ids
    
    def _calculate_weighted_loss(self, augmented_text_ids):
        for position in augmented_text_ids["api_start_positions"]:        
            seq_positions = augmented_text_ids["api_start_positions"][position]["seq_positions"]
            for i in seq_positions:
                losses = seq_positions[i]["losses"]
                weights = seq_positions[i]["normalized_weight"]
                seq_positions[i]["weighted_losses"] = -losses * weights
        
        return augmented_text_ids
    
    def _calculate_loss(self, augmented_text_ids):
        losses = {}
        for position in augmented_text_ids["api_start_positions"]:        
            seq_positions = augmented_text_ids["api_start_positions"][position]["seq_positions"]
            three_loss = [0, 0, 0]            
            for i in seq_positions:
                three_loss[0] += seq_positions[i]["weighted_losses"][0]
                three_loss[1] += seq_positions[i]["weighted_losses"][1]
                three_loss[2] += seq_positions[i]["weighted_losses"][2]
            losses[position] = three_loss
            
        return losses

    def filter_api_candidate_by_threshold(self, losses, candidates, threshold):
        filtered_augmented_text_ids = []
        for i, position in enumerate(losses):
            negative_loss = min(losses[position][0], losses[position][1])
            positive_loss = losses[position][2]
            
            if negative_loss - positive_loss >= threshold:
                filtered_augmented_text_ids.append(candidates[i])
        
        return filtered_augmented_text_ids
    
    def _convert_prompt_dict_to_list_input_ids(self, augmented_text_ids):
        # input_ids = torch.tensor([])
        input_ids = []
        for _, api_start_position_dict in augmented_text_ids["api_start_positions"].items():
            for _, seq_position_dict in api_start_position_dict["seq_positions"].items():
                for x in seq_position_dict["prompt_ids"]:
                    # print(x.shape)
                    input_ids.append(x)
                    # input_ids = torch.cat([input_ids, x.unsqueeze(0)], dim=0)
        return input_ids
    
    def _convert_prompt_dict_to_list_target_ids(self, augmented_text_ids):
        target_ids = []
        for _, api_start_position_dict in augmented_text_ids["api_start_positions"].items():
            for _, seq_position_dict in api_start_position_dict["seq_positions"].items():
                target_ids.append(seq_position_dict["target_ids"])
        return target_ids

    def filter_api( 
        self,
        text_ids,
        api_start_idxs,
        candidates: TensorType["batch_size", "n_positions", "seq_len"]
    ):
        conditioning_api_ids = self._generate_conditioning_prompts(candidates)
        
        ######
        
        MAX_PAD = 50
        FILTER_THRESHOLD = 0.23
        PAD_TOKEN = self.pad_token_id
        SPACE_TOKEN = self.tokenizer(". ", return_tensors="pt")["input_ids"][0]
        API_LENGTH = 100
        augmented_text_ids = {"api_start_positions": {}}
        
        for idx, api_ids in zip(api_start_idxs, conditioning_api_ids):
            idx = idx.item()
            seq_len = len(text_ids)
            augmented_text_ids["api_start_positions"][idx] = {
                "seq_positions": {}
            }

            j = idx
            while j <= seq_len - 1:
                # if the model predic
                if j == 1:
                    j += 1
                    continue
                
                # in the formua, from x_1 to x_j (include x_j)
                # => generate_ids[:j]
                conditioning_text_ids = text_ids[:j]
                api_and_text_ids = torch.stack([
                    F.pad(conditioning_text_ids, pad=(API_LENGTH + len(SPACE_TOKEN), 0), value=PAD_TOKEN), # [text_ids]
                    torch.cat([api_ids[0], SPACE_TOKEN, conditioning_text_ids], dim=0), # [api->, text_ids]
                    torch.cat([api_ids[1], SPACE_TOKEN, conditioning_text_ids], dim=0), # [api->result, text_ids]
                ], dim=0)
                                
                # api_and_text_ids = conditioning_api_ids[]
                # the next token after x_j
                next_token_ids = text_ids[j]
                augmented_text_ids["api_start_positions"][idx]["seq_positions"][j] = {
                    "prompt_ids": api_and_text_ids,
                    "unnormalized_weight": self._compute_weight(t=j-idx),
                    "losses": [],
                    "target_ids": next_token_ids
                }
                j += 1
        
        augmented_text_ids = self._normalize_weights(augmented_text_ids)
        input_ids = self._convert_prompt_dict_to_list_input_ids(augmented_text_ids)
        target_ids = self._convert_prompt_dict_to_list_target_ids(augmented_text_ids)
        
        # convert input_ids to tensor
        MAX_PAD = 10
        # input_ids = torch.tensor([[F.pad(x.long(), pad=(50-x.shape[-1], 0), value=PAD_TOKEN) for x in input_ids]])

        padded_input_ids = torch.tensor([])
        for x in input_ids:
            padded_input_ids = torch.cat([
                padded_input_ids,
                F.pad(x.long(), pad=(50-x.shape[-1], 0), value=PAD_TOKEN).unsqueeze(0)
            ], dim=0)
            
        output = self.model(input_ids=padded_input_ids.long())
        logits = output.logits[:, -1, :]
        probs = F.softmax(logits, dim=-1)
        
        log_probs = torch.tensor([])

        i = 0
        for x in target_ids:
            log_probs = torch.cat([log_probs, probs[i:i+3][:, x].log().unsqueeze(0)], dim=0)
            i += 3
            
        for _, api_start_position_dict in augmented_text_ids["api_start_positions"].items():
            for _, seq_position_dict in api_start_position_dict["seq_positions"].items():
                seq_position_dict["losses"] = log_probs[:1].squeeze(0)
                log_probs = log_probs[1:]
        
        augmented_text_ids = self._calculate_weighted_loss(augmented_text_ids)
        losses = self._calculate_loss(augmented_text_ids)
        filtered_candidate_ids = self.filter_api_candidate_by_threshold(losses, candidates, threshold=FILTER_THRESHOLD)
        
        return filtered_candidate_ids
    
    def generate(
        self,
        prompt_tempalte: PromptTemplate,
        text: str,
    ) -> TensorType["batch_size", "seq_len"]:
        # TODO: add support batch
        prompt = prompt_tempalte.format(input=text)
        prompt_ids = self.tokenizer(prompt, return_tensors="pt")["input_ids"][0]
    
        # sampling positions
        api_start_idxs, generated_ids = self.sample_api_position(prompt_ids)
        
        # obtaining api responses
        candidates = self.obtain_api_response(prompt_ids, api_start_idxs, generated_ids)

        # filtering
        text_ids = self.tokenizer(text, return_tensors="pt")["input_ids"][0]
        filtered_candidate_ids = self.filter_api(text_ids, api_start_idxs, candidates)
                
        return filtered_candidate_ids
