"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserPoolDomain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const cognito_generated_1 = require("./cognito.generated");
/**
 * Define a user pool domain.
 *
 * @stability stable
 */
class UserPoolDomain extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        if (!!props.customDomain === !!props.cognitoDomain) {
            throw new Error('One of, and only one of, cognitoDomain or customDomain must be specified');
        }
        if (((_b = props.cognitoDomain) === null || _b === void 0 ? void 0 : _b.domainPrefix) &&
            !core_1.Token.isUnresolved((_c = props.cognitoDomain) === null || _c === void 0 ? void 0 : _c.domainPrefix) &&
            !/^[a-z0-9-]+$/.test(props.cognitoDomain.domainPrefix)) {
            throw new Error('domainPrefix for cognitoDomain can contain only lowercase alphabets, numbers and hyphens');
        }
        this.isCognitoDomain = !!props.cognitoDomain;
        const domainName = ((_d = props.cognitoDomain) === null || _d === void 0 ? void 0 : _d.domainPrefix) || ((_e = props.customDomain) === null || _e === void 0 ? void 0 : _e.domainName);
        const resource = new cognito_generated_1.CfnUserPoolDomain(this, 'Resource', {
            userPoolId: props.userPool.userPoolId,
            domain: domainName,
            customDomainConfig: props.customDomain ? { certificateArn: props.customDomain.certificate.certificateArn } : undefined,
        });
        this.domainName = resource.ref;
    }
    /**
     * Import a UserPoolDomain given its domain name.
     *
     * @stability stable
     */
    static fromDomainName(scope, id, userPoolDomainName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.domainName = userPoolDomainName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The domain name of the CloudFront distribution associated with the user pool domain.
     *
     * @stability stable
     */
    get cloudFrontDomainName() {
        if (!this.cloudFrontCustomResource) {
            const sdkCall = {
                service: 'CognitoIdentityServiceProvider',
                action: 'describeUserPoolDomain',
                parameters: {
                    Domain: this.domainName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.domainName),
            };
            this.cloudFrontCustomResource = new custom_resources_1.AwsCustomResource(this, 'CloudFrontDomainName', {
                resourceType: 'Custom::UserPoolCloudFrontDomainName',
                onCreate: sdkCall,
                onUpdate: sdkCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                    // DescribeUserPoolDomain only supports access level '*'
                    // https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazoncognitouserpools.html#amazoncognitouserpools-actions-as-permissions
                    resources: ['*'],
                }),
            });
        }
        return this.cloudFrontCustomResource.getResponseField('DomainDescription.CloudFrontDistribution');
    }
    /**
     * The URL to the hosted UI associated with this domain.
     *
     * @stability stable
     */
    baseUrl() {
        if (this.isCognitoDomain) {
            return `https://${this.domainName}.auth.${core_1.Stack.of(this).region}.amazoncognito.com`;
        }
        return `https://${this.domainName}`;
    }
    /**
     * The URL to the sign in page in this domain using a specific UserPoolClient.
     *
     * @param client [disable-awslint:ref-via-interface] the user pool client that the UI will use to interact with the UserPool.
     * @param options options to customize the behaviour of this method.
     * @stability stable
     */
    signInUrl(client, options) {
        var _b;
        let responseType;
        if (client.oAuthFlows.authorizationCodeGrant) {
            responseType = 'code';
        }
        else if (client.oAuthFlows.implicitCodeGrant) {
            responseType = 'token';
        }
        else {
            throw new Error('signInUrl is not supported for clients without authorizationCodeGrant or implicitCodeGrant flow enabled');
        }
        const path = (_b = options.signInPath) !== null && _b !== void 0 ? _b : '/login';
        return `${this.baseUrl()}${path}?client_id=${client.userPoolClientId}&response_type=${responseType}&redirect_uri=${options.redirectUri}`;
    }
}
exports.UserPoolDomain = UserPoolDomain;
_a = JSII_RTTI_SYMBOL_1;
UserPoolDomain[_a] = { fqn: "@aws-cdk/aws-cognito.UserPoolDomain", version: "1.96.0" };
//# sourceMappingURL=data:application/json;base64,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