import { IResource, Resource, Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IUserPool } from './user-pool';
import { ClientAttributes } from './user-pool-attr';
import { IUserPoolResourceServer, ResourceServerScope } from './user-pool-resource-server';
/**
 * Types of authentication flow.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html
 * @stability stable
 */
export interface AuthFlow {
    /**
     * Enable admin based user password authentication flow.
     *
     * @default false
     * @stability stable
     */
    readonly adminUserPassword?: boolean;
    /**
     * Enable custom authentication flow.
     *
     * @default false
     * @stability stable
     */
    readonly custom?: boolean;
    /**
     * Enable auth using username & password.
     *
     * @default false
     * @stability stable
     */
    readonly userPassword?: boolean;
    /**
     * Enable SRP based authentication.
     *
     * @default false
     * @stability stable
     */
    readonly userSrp?: boolean;
}
/**
 * OAuth settings to configure the interaction between the app and this client.
 *
 * @stability stable
 */
export interface OAuthSettings {
    /**
     * OAuth flows that are allowed with this client.
     *
     * @default {authorizationCodeGrant:true,implicitCodeGrant:true}
     * @see - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
     * @stability stable
     */
    readonly flows?: OAuthFlows;
    /**
     * List of allowed redirect URLs for the identity providers.
     *
     * @default - ['https://example.com'] if either authorizationCodeGrant or implicitCodeGrant flows are enabled, no callback URLs otherwise.
     * @stability stable
     */
    readonly callbackUrls?: string[];
    /**
     * List of allowed logout URLs for the identity providers.
     *
     * @default - no logout URLs
     * @stability stable
     */
    readonly logoutUrls?: string[];
    /**
     * OAuth scopes that are allowed with this client.
     *
     * @default [OAuthScope.PHONE,OAuthScope.EMAIL,OAuthScope.OPENID,OAuthScope.PROFILE,OAuthScope.COGNITO_ADMIN]
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
     * @stability stable
     */
    readonly scopes?: OAuthScope[];
}
/**
 * Types of OAuth grant flows.
 *
 * @see - the 'Allowed OAuth Flows' section at https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @stability stable
 */
export interface OAuthFlows {
    /**
     * Initiate an authorization code grant flow, which provides an authorization code as the response.
     *
     * @default false
     * @stability stable
     */
    readonly authorizationCodeGrant?: boolean;
    /**
     * The client should get the access token and ID token directly.
     *
     * @default false
     * @stability stable
     */
    readonly implicitCodeGrant?: boolean;
    /**
     * Client should get the access token and ID token from the token endpoint using a combination of client and client_secret.
     *
     * @default false
     * @stability stable
     */
    readonly clientCredentials?: boolean;
}
/**
 * OAuth scopes that are allowed with this client.
 *
 * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-app-idp-settings.html
 * @stability stable
 */
export declare class OAuthScope {
    /**
     * Grants access to the 'phone_number' and 'phone_number_verified' claims.
     *
     * Automatically includes access to `OAuthScope.OPENID`.
     *
     * @stability stable
     */
    static readonly PHONE: OAuthScope;
    /**
     * Grants access to the 'email' and 'email_verified' claims.
     *
     * Automatically includes access to `OAuthScope.OPENID`.
     *
     * @stability stable
     */
    static readonly EMAIL: OAuthScope;
    /**
     * Returns all user attributes in the ID token that are readable by the client.
     *
     * @stability stable
     */
    static readonly OPENID: OAuthScope;
    /**
     * Grants access to all user attributes that are readable by the client Automatically includes access to `OAuthScope.OPENID`.
     *
     * @stability stable
     */
    static readonly PROFILE: OAuthScope;
    /**
     * Grants access to Amazon Cognito User Pool API operations that require access tokens, such as UpdateUserAttributes and VerifyUserAttribute.
     *
     * @stability stable
     */
    static readonly COGNITO_ADMIN: OAuthScope;
    /**
     * Custom scope is one that you define for your own resource server in the Resource Servers.
     *
     * The format is 'resource-server-identifier/scope'.
     *
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pools-define-resource-servers.html
     * @stability stable
     */
    static custom(name: string): OAuthScope;
    /**
     * Adds a custom scope that's tied to a resource server in your stack.
     *
     * @stability stable
     */
    static resourceServer(server: IUserPoolResourceServer, scope: ResourceServerScope): OAuthScope;
    /**
     * The name of this scope as recognized by CloudFormation.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-cognito-userpoolclient.html#cfn-cognito-userpoolclient-allowedoauthscopes
     * @stability stable
     */
    readonly scopeName: string;
    private constructor();
}
/**
 * Identity providers supported by the UserPoolClient.
 *
 * @stability stable
 */
export declare class UserPoolClientIdentityProvider {
    /**
     * Allow users to sign in using 'Sign In With Apple'.
     *
     * A `UserPoolIdentityProviderApple` must be attached to the user pool.
     *
     * @stability stable
     */
    static readonly APPLE: UserPoolClientIdentityProvider;
    /**
     * Allow users to sign in using 'Facebook Login'.
     *
     * A `UserPoolIdentityProviderFacebook` must be attached to the user pool.
     *
     * @stability stable
     */
    static readonly FACEBOOK: UserPoolClientIdentityProvider;
    /**
     * Allow users to sign in using 'Google Login'.
     *
     * A `UserPoolIdentityProviderGoogle` must be attached to the user pool.
     *
     * @stability stable
     */
    static readonly GOOGLE: UserPoolClientIdentityProvider;
    /**
     * Allow users to sign in using 'Login With Amazon'.
     *
     * A `UserPoolIdentityProviderAmazon` must be attached to the user pool.
     *
     * @stability stable
     */
    static readonly AMAZON: UserPoolClientIdentityProvider;
    /**
     * Allow users to sign in directly as a user of the User Pool.
     *
     * @stability stable
     */
    static readonly COGNITO: UserPoolClientIdentityProvider;
    /**
     * Specify a provider not yet supported by the CDK.
     *
     * @param name name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     * @stability stable
     */
    static custom(name: string): UserPoolClientIdentityProvider;
    /**
     * The name of the identity provider as recognized by CloudFormation property `SupportedIdentityProviders`.
     *
     * @stability stable
     */
    readonly name: string;
    private constructor();
}
/**
 * Options to create a UserPoolClient.
 *
 * @stability stable
 */
export interface UserPoolClientOptions {
    /**
     * Name of the application client.
     *
     * @default - cloudformation generated name
     * @stability stable
     */
    readonly userPoolClientName?: string;
    /**
     * Whether to generate a client secret.
     *
     * @default false
     * @stability stable
     */
    readonly generateSecret?: boolean;
    /**
     * The set of OAuth authentication flows to enable on the client.
     *
     * @default - all auth flows disabled
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/amazon-cognito-user-pools-authentication-flow.html
     * @stability stable
     */
    readonly authFlows?: AuthFlow;
    /**
     * Turns off all OAuth interactions for this client.
     *
     * @default false
     * @stability stable
     */
    readonly disableOAuth?: boolean;
    /**
     * OAuth settings for this client to interact with the app.
     *
     * An error is thrown when this is specified and `disableOAuth` is set.
     *
     * @default - see defaults in `OAuthSettings`. meaningless if `disableOAuth` is set.
     * @stability stable
     */
    readonly oAuth?: OAuthSettings;
    /**
     * Whether Cognito returns a UserNotFoundException exception when the user does not exist in the user pool (false), or whether it returns another type of error that doesn't reveal the user's absence.
     *
     * @default true for new stacks
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/cognito-user-pool-managing-errors.html
     * @stability stable
     */
    readonly preventUserExistenceErrors?: boolean;
    /**
     * The list of identity providers that users should be able to use to sign in using this client.
     *
     * @default - supports all identity providers that are registered with the user pool. If the user pool and/or
     * identity providers are imported, either specify this option explicitly or ensure that the identity providers are
     * registered with the user pool using the `UserPool.registerIdentityProvider()` API.
     * @stability stable
     */
    readonly supportedIdentityProviders?: UserPoolClientIdentityProvider[];
    /**
     * Validity of the ID token.
     *
     * Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity.
     *
     * @default Duration.minutes(60)
     * @see https://docs.aws.amazon.com/en_us/cognito/latest/developerguide/amazon-cognito-user-pools-using-tokens-with-identity-providers.html#amazon-cognito-user-pools-using-the-id-token
     * @stability stable
     */
    readonly idTokenValidity?: Duration;
    /**
     * Validity of the refresh token.
     *
     * Values between 60 minutes and 10 years are valid.
     *
     * @default Duration.days(30)
     * @see https://docs.aws.amazon.com/en_us/cognito/latest/developerguide/amazon-cognito-user-pools-using-tokens-with-identity-providers.html#amazon-cognito-user-pools-using-the-refresh-token
     * @stability stable
     */
    readonly refreshTokenValidity?: Duration;
    /**
     * Validity of the access token.
     *
     * Values between 5 minutes and 1 day are valid. The duration can not be longer than the refresh token validity.
     *
     * @default Duration.minutes(60)
     * @see https://docs.aws.amazon.com/en_us/cognito/latest/developerguide/amazon-cognito-user-pools-using-tokens-with-identity-providers.html#amazon-cognito-user-pools-using-the-access-token
     * @stability stable
     */
    readonly accessTokenValidity?: Duration;
    /**
     * The set of attributes this client will be able to read.
     *
     * @default - all standard and custom attributes
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-attribute-permissions-and-scopes
     * @stability stable
     */
    readonly readAttributes?: ClientAttributes;
    /**
     * The set of attributes this client will be able to write.
     *
     * @default - all standard and custom attributes
     * @see https://docs.aws.amazon.com/cognito/latest/developerguide/user-pool-settings-attributes.html#user-pool-settings-attribute-permissions-and-scopes
     * @stability stable
     */
    readonly writeAttributes?: ClientAttributes;
}
/**
 * Properties for the UserPoolClient construct.
 *
 * @stability stable
 */
export interface UserPoolClientProps extends UserPoolClientOptions {
    /**
     * The UserPool resource this client will have access to.
     *
     * @stability stable
     */
    readonly userPool: IUserPool;
}
/**
 * Represents a Cognito user pool client.
 *
 * @stability stable
 */
export interface IUserPoolClient extends IResource {
    /**
     * Name of the application client.
     *
     * @stability stable
     * @attribute true
     */
    readonly userPoolClientId: string;
}
/**
 * Define a UserPool App Client.
 *
 * @stability stable
 */
export declare class UserPoolClient extends Resource implements IUserPoolClient {
    /**
     * Import a user pool client given its id.
     *
     * @stability stable
     */
    static fromUserPoolClientId(scope: Construct, id: string, userPoolClientId: string): IUserPoolClient;
    /**
     * Name of the application client.
     *
     * @stability stable
     */
    readonly userPoolClientId: string;
    /**
     * The OAuth flows enabled for this client.
     *
     * @stability stable
     */
    readonly oAuthFlows: OAuthFlows;
    private readonly _userPoolClientName?;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: UserPoolClientProps);
    /**
     * The client name that was specified via the `userPoolClientName` property during initialization, throws an error otherwise.
     *
     * @stability stable
     */
    get userPoolClientName(): string;
    private configureAuthFlows;
    private configureOAuthFlows;
    private configureOAuthScopes;
    private configurePreventUserExistenceErrors;
    private configureIdentityProviders;
    private configureTokenValidity;
    private validateDuration;
}
