import os
from typing import List, Optional

from dstack.backend.local.common import (
    create_secret,
    delete_object,
    delete_secret,
    get_object,
    get_secret_value,
    list_objects,
    put_object,
    put_secret_value,
)
from dstack.core.error import SecretError
from dstack.core.repo import RepoAddress
from dstack.core.secret import Secret


def list_secret_names(path: str, repo_address: RepoAddress) -> List[str]:
    root = os.path.join(path, "secrets", repo_address.path())
    secret_head_prefix = f"l;"
    response = list_objects(Root=root, Prefix=secret_head_prefix)
    secret_names = []
    for obj in response:
        secret_name = obj[len(secret_head_prefix) :]
        secret_names.append(secret_name)
    return secret_names


def get_secret(path: str, repo_address: RepoAddress, secret_name: str) -> Optional[Secret]:
    root = os.path.join(path, "secrets", repo_address.path())
    try:
        return Secret(
            secret_name,
            get_secret_value(
                SecretId=f"/dstack/secrets/{repo_address.path()}/{secret_name}",
                Root=root,
            ),
        )
    except SecretError:
        return None


def add_secret(path: str, repo_address: RepoAddress, secret: Secret):
    root = os.path.join(path, "secrets", repo_address.path())
    secret_id = f"/dstack/secrets/{repo_address.path()}/{secret.secret_name}"
    create_secret(
        SecretId=secret_id,
        SecretString=secret.secret_value,
        Description="Generated by dstack",
        Root=root,
    )
    key = f"l;{secret.secret_name}"
    put_object(Body="", Root=root, Key=key)


def update_secret(path: str, repo_address: RepoAddress, secret: Secret):
    root = os.path.join(path, "secrets", repo_address.path())
    secret_id = f"/dstack/secrets/{repo_address.path()}/{secret.secret_name}"
    put_secret_value(SecretId=secret_id, SecretString=secret.secret_value, Root=root)
    key = f"l;{secret.secret_name}"
    put_object(Body="", Root=root, Key=key)


def remove_secret(path: str, repo_address: RepoAddress, secret_name: str):
    root = os.path.join(path, "secrets", repo_address.path())
    delete_secret(SecretId=f"/dstack/secrets/{repo_address.path()}/{secret_name}", Root=root)
    key = f"l;{secret_name}"
    delete_object(Root=root, Key=key)
