#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

"""Visual Studio project reader/writer."""

import gyp.common
import gyp.easy_xml as easy_xml


class Writer(object):
  """Visual Studio XML tool file writer."""

  def __init__(self, tool_file_path, name):
    """Initializes the tool file.

    Args:
      tool_file_path: Path to the tool file.
      name: Name of the tool file.
    """
    self.tool_file_path = tool_file_path
    self.name = name
    self.rules_section = ['Rules']

  def AddCustomBuildRule(self, name, cmd, description,
                         additional_dependencies,
                         outputs, extensions):
    """Adds a rule to the tool file.

    Args:
      name: Name of the rule.
      description: Description of the rule.
      cmd: Command line of the rule.
      additional_dependencies: other files which may trigger the rule.
      outputs: outputs of the rule.
      extensions: extensions handled by the rule.
    """
    rule = ['CustomBuildRule',
            {'Name': name,
             'ExecutionDescription': description,
             'CommandLine': cmd,
             'Outputs': ';'.join(outputs),
             'FileExtensions': ';'.join(extensions),
             'AdditionalDependencies':
                 ';'.join(additional_dependencies)
            }]
    self.rules_section.append(rule)

  def WriteIfChanged(self):
    """Writes the tool file."""
    content = ['VisualStudioToolFile',
               {'Version': '8.00',
                'Name': self.name
               },
               self.rules_section
               ]
    easy_xml.WriteXmlIfChanged(content, self.tool_file_path,
                               encoding="Windows-1252")
