"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const glue = require("../lib");
function createTable(props) {
    const stack = new cdk.Stack();
    new glue.Table(stack, 'table', Object.assign(Object.assign({}, props), { database: new glue.Database(stack, 'db', {
            databaseName: 'database_name',
        }), dataFormat: glue.DataFormat.Json }));
}
module.exports = {
    'unpartitioned JSON table'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        const tableStack = new cdk.Stack(app, 'table');
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.UNENCRYPTED);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::S3::Bucket', {
            Type: "AWS::S3::Bucket",
            DeletionPolicy: "Retain",
            UpdateReplacePolicy: "Retain"
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "db:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'partitioned JSON table'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        const tableStack = new cdk.Stack(app, 'table');
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            partitionKeys: [{
                    name: 'year',
                    type: glue.Schema.SMALL_INT
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.UNENCRYPTED);
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "db:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                PartitionKeys: [
                    {
                        Name: "year",
                        Type: "smallint"
                    }
                ],
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'compressed table'(test) {
        const stack = new cdk.Stack();
        const database = new glue.Database(stack, 'Database', {
            databaseName: 'database',
        });
        const table = new glue.Table(stack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            compressed: true,
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                Ref: "DatabaseB269D8BB"
            },
            TableInput: {
                Name: "table",
                Description: "table generated by CDK",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: true,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "TableBucketDA42407C"
                                },
                                "/data/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'table.node.defaultChild'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const database = new glue.Database(stack, 'Database', {
            databaseName: 'database',
        });
        // WHEN
        const table = new glue.Table(stack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            compressed: true,
            dataFormat: glue.DataFormat.Json,
        });
        // THEN
        test.ok(table.node.defaultChild instanceof glue.CfnTable);
        test.done();
    },
    'encrypted table': {
        'SSE-S3'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.S3_MANAGED,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.S3_MANAGED);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: "AES256"
                            }
                        }
                    ]
                }
            }));
            test.done();
        },
        'SSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion",
                                "kms:GenerateDataKey"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                },
                Description: "Created by Table/Bucket"
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    "Fn::GetAtt": [
                                        "TableBucketKey3E9F984A",
                                        "Arn"
                                    ]
                                },
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'SSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            test.notEqual(table.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion",
                                "kms:GenerateDataKey"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    "Fn::GetAtt": [
                                        "MyKey6AB29FA6",
                                        "Arn"
                                    ]
                                },
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'SSE-KMS_MANAGED'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS_MANAGED,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS_MANAGED);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: "aws:kms"
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Name: "table",
                    Description: "table generated by CDK",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion",
                                "kms:GenerateDataKey"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion",
                                "kms:GenerateDataKey"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "TableBucketDA42407C"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly passed bucket and key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const bucket = new s3.Bucket(stack, 'Bucket');
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                bucket,
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                "kms:Create*",
                                "kms:Describe*",
                                "kms:Enable*",
                                "kms:List*",
                                "kms:Put*",
                                "kms:Update*",
                                "kms:Revoke*",
                                "kms:Disable*",
                                "kms:Get*",
                                "kms:Delete*",
                                "kms:ScheduleKeyDeletion",
                                "kms:CancelKeyDeletion",
                                "kms:GenerateDataKey"
                            ],
                            Effect: "Allow",
                            Principal: {
                                AWS: {
                                    "Fn::Join": [
                                        "",
                                        [
                                            "arn:",
                                            {
                                                Ref: "AWS::Partition"
                                            },
                                            ":iam::",
                                            {
                                                Ref: "AWS::AccountId"
                                            },
                                            ":root"
                                        ]
                                    ]
                                }
                            },
                            Resource: "*"
                        }
                    ],
                    Version: "2012-10-17"
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: "AWS::AccountId"
                },
                DatabaseName: {
                    Ref: "DatabaseB269D8BB"
                },
                TableInput: {
                    Description: "table generated by CDK",
                    Name: "table",
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: "col",
                                Type: "string"
                            }
                        ],
                        Compressed: false,
                        InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                        Location: {
                            "Fn::Join": [
                                "",
                                [
                                    "s3://",
                                    {
                                        Ref: "Bucket83908E77"
                                    },
                                    "/data/"
                                ]
                            ]
                        },
                        OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                        SerdeInfo: {
                            SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: "EXTERNAL_TABLE"
                }
            }));
            test.done();
        }
    },
    'explicit s3 bucket and prefix'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const stack = new cdk.Stack(app, 'app');
        const bucket = new s3.Bucket(stack, 'ExplicitBucket');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        new glue.Table(stack, 'Table', {
            database,
            bucket,
            s3Prefix: 'prefix/',
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            dataFormat: glue.DataFormat.Json,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: "AWS::AccountId"
            },
            DatabaseName: {
                "Fn::ImportValue": "db:ExportsOutputRefDatabaseB269D8BB88F4B1C4"
            },
            TableInput: {
                Description: "table generated by CDK",
                Name: "table",
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: "col",
                            Type: "string"
                        }
                    ],
                    Compressed: false,
                    InputFormat: "org.apache.hadoop.mapred.TextInputFormat",
                    Location: {
                        "Fn::Join": [
                            "",
                            [
                                "s3://",
                                {
                                    Ref: "ExplicitBucket0AA51A3F"
                                },
                                "/prefix/"
                            ]
                        ]
                    },
                    OutputFormat: "org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat",
                    SerdeInfo: {
                        SerializationLibrary: "org.openx.data.jsonserde.JsonSerDe"
                    },
                    StoredAsSubDirectories: false
                },
                TableType: "EXTERNAL_TABLE"
            }
        }));
        test.done();
    },
    'grants': {
        'read only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantRead(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchDeletePartition",
                                "glue:BatchGetPartition",
                                "glue:GetPartition",
                                "glue:GetPartitions",
                                "glue:GetTable",
                                "glue:GetTables",
                                "glue:GetTableVersions"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":table/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        },
        'write only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchCreatePartition",
                                "glue:BatchDeletePartition",
                                "glue:CreatePartition",
                                "glue:DeletePartition",
                                "glue:UpdatePartition"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":table/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:DeleteObject*",
                                "s3:PutObject*",
                                "s3:Abort*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        },
        'read and write'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantReadWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                "glue:BatchDeletePartition",
                                "glue:BatchGetPartition",
                                "glue:GetPartition",
                                "glue:GetPartitions",
                                "glue:GetTable",
                                "glue:GetTables",
                                "glue:GetTableVersions",
                                "glue:BatchCreatePartition",
                                "glue:BatchDeletePartition",
                                "glue:CreatePartition",
                                "glue:DeletePartition",
                                "glue:UpdatePartition"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":glue:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":table/",
                                        {
                                            Ref: "DatabaseB269D8BB"
                                        },
                                        "/",
                                        {
                                            Ref: "Table4C2D914F"
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "s3:GetObject*",
                                "s3:GetBucket*",
                                "s3:List*",
                                "s3:DeleteObject*",
                                "s3:PutObject*",
                                "s3:Abort*"
                            ],
                            Effect: "Allow",
                            Resource: [
                                {
                                    "Fn::GetAtt": [
                                        "TableBucketDA42407C",
                                        "Arn"
                                    ]
                                },
                                {
                                    "Fn::Join": [
                                        "",
                                        [
                                            {
                                                "Fn::GetAtt": [
                                                    "TableBucketDA42407C",
                                                    "Arn"
                                                ]
                                            },
                                            "/data/"
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "UserDefaultPolicy1F97781E",
                Users: [
                    {
                        Ref: "User00B015A1"
                    }
                ]
            }));
            test.done();
        }
    },
    'validate': {
        'at least one column'(test) {
            test.throws(() => {
                createTable({
                    columns: [],
                    tableName: 'name',
                });
            }, undefined, 'you must specify at least one column for the table');
            test.done();
        },
        'unique column names'(test) {
            test.throws(() => {
                createTable({
                    tableName: 'name',
                    columns: [{
                            name: 'col1',
                            type: glue.Schema.STRING
                        }, {
                            name: 'col1',
                            type: glue.Schema.STRING
                        }]
                });
            }, undefined, "column names and partition keys must be unique, but 'col1' is duplicated.");
            test.done();
        },
        'unique partition keys'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                partitionKeys: [{
                        name: 'p1',
                        type: glue.Schema.STRING
                    }, {
                        name: 'p1',
                        type: glue.Schema.STRING
                    }]
            }), undefined, "column names and partition keys must be unique, but 'p1' is duplicated");
            test.done();
        },
        'column names and partition keys are all unique'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                partitionKeys: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }]
            }), "column names and partition keys must be unique, but 'col1' is duplicated");
            test.done();
        },
        'can not specify an explicit bucket and encryption'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.KMS
            }), undefined, 'you can not specify encryption settings if you also provide a bucket');
            test.done();
        },
        'can explicitly pass bucket if Encryption undefined'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if Unencrypted'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if ClientSideKms'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS
            }));
            test.done();
        }
    },
    'Table.fromTableArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const table = glue.Table.fromTableArn(stack, 'boom', 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
        // THEN
        test.deepEqual(table.tableArn, 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
        test.deepEqual(table.tableName, 'tbl1');
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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