"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix || 'data/';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed === undefined ? false : props.compressed;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories === undefined ? false : props.storedAsSubDirectories,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className
                    },
                },
                tableType: 'EXTERNAL_TABLE'
            }
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`
        });
        this.node.defaultChild = tableResource;
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).parseArn(tableArn).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.s3Prefix);
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.s3Prefix);
        return ret;
    }
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
}
exports.Table = Table;
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but 'p1' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey
    };
}
const readPermissions = [
    'glue:BatchDeletePartition',
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersions'
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition'
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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