"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const glue_generated_1 = require("./glue.generated");
/**
 * A Glue database.
 */
class Database extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.databaseName,
        });
        if (props.locationUri) {
            this.locationUri = props.locationUri;
        }
        else {
            const bucket = new s3.Bucket(this, 'Bucket');
            this.locationUri = `s3://${bucket.bucketName}/${props.databaseName}`;
        }
        this.catalogId = core_1.Stack.of(this).account;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput: {
                name: this.physicalName,
                locationUri: this.locationUri
            }
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = this.getResourceNameAttribute(resource.ref);
        this.databaseArn = this.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName,
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = core_1.Stack.of(this).formatArn({
            service: 'glue',
            resource: 'catalog'
        });
    }
    static fromDatabaseArn(scope, id, databaseArn) {
        const stack = core_1.Stack.of(scope);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.databaseArn = databaseArn;
                this.databaseName = stack.parseArn(databaseArn).resourceName;
                this.catalogArn = stack.formatArn({ service: 'glue', resource: 'catalog' });
                this.catalogId = stack.account;
            }
        }
        return new Import(scope, id);
    }
}
exports.Database = Database;
//# sourceMappingURL=data:application/json;base64,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