"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Absolute class name of the Hadoop `InputFormat` to use when reading table files.
 */
class InputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.InputFormat = InputFormat;
/**
 * An InputFormat for plain text files. Files are broken into lines. Either linefeed or
 * carriage-return are used to signal end of line. Keys are the position in the file, and
 * values are the line of text.
 *
 * @see https://hadoop.apache.org/docs/stable/api/org/apache/hadoop/mapred/TextInputFormat.html
 */
InputFormat.TEXT_INPUT_FORMAT = new InputFormat('org.apache.hadoop.mapred.TextInputFormat');
/**
 * Absolute class name of the Hadoop `OutputFormat` to use when writing table files.
 */
class OutputFormat {
    constructor(className) {
        this.className = className;
    }
}
exports.OutputFormat = OutputFormat;
/**
 * Writes text data with a null key (value only).
 *
 * @see https://hive.apache.org/javadocs/r2.2.0/api/org/apache/hadoop/hive/ql/io/HiveIgnoreKeyTextOutputFormat.html
 */
OutputFormat.HIVE_IGNORE_KEY_TEXT_OUTPUT_FORMAT = new OutputFormat('org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat');
/**
 * Serialization library to use when serializing/deserializing (SerDe) table records.
 *
 * @see https://cwiki.apache.org/confluence/display/Hive/SerDe
 */
class SerializationLibrary {
    constructor(className) {
        this.className = className;
    }
}
exports.SerializationLibrary = SerializationLibrary;
/**
 * @see https://cwiki.apache.org/confluence/display/Hive/LanguageManual+DDL#LanguageManualDDL-JSON
 */
SerializationLibrary.HIVE_JSON = new SerializationLibrary('org.apache.hive.hcatalog.data.JsonSerDe');
/**
 * @see https://github.com/rcongiu/Hive-JSON-Serde
 */
SerializationLibrary.OPENX_JSON = new SerializationLibrary('org.openx.data.jsonserde.JsonSerDe');
var DataFormat;
(function (DataFormat) {
    /**
     * Stored as plain text files in JSON format.
     *
     * Uses OpenX Json SerDe for serialization and deseralization.
     *
     * @see https://docs.aws.amazon.com/athena/latest/ug/json.html
     */
    DataFormat.Json = {
        inputFormat: InputFormat.TEXT_INPUT_FORMAT,
        outputFormat: OutputFormat.HIVE_IGNORE_KEY_TEXT_OUTPUT_FORMAT,
        serializationLibrary: SerializationLibrary.OPENX_JSON
    };
})(DataFormat = exports.DataFormat || (exports.DataFormat = {}));
//# sourceMappingURL=data:application/json;base64,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