import * as apigwy from '@aws-cdk/aws-apigatewayv2-alpha';
import { RemovalPolicy } from 'aws-cdk-lib';
import * as cf from 'aws-cdk-lib/aws-cloudfront';
import * as dynamodb from 'aws-cdk-lib/aws-dynamodb';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as s3 from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
/**
 * Properties to initialize an instance of `MicroAppsSvcs`.
 */
export interface MicroAppsSvcsProps {
    /**
     * RemovalPolicy override for child resources
     *
     * Note: if set to DESTROY the S3 buckes will have `autoDeleteObjects` set to `true`
     *
     * @default - per resource default
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * S3 bucket for deployed applications
     */
    readonly bucketApps: s3.IBucket;
    /**
     * CloudFront Origin Access Identity for the deployed applications bucket
     */
    readonly bucketAppsOAI: cf.OriginAccessIdentity;
    /**
     * S3 bucket for staged applications (prior to deploy)
     */
    readonly bucketAppsStaging: s3.IBucket;
    /**
     * API Gateway v2 HTTP for Router and app
     */
    readonly httpApi: apigwy.HttpApi;
    /**
     * Application environment, passed as `NODE_ENV`
     * to the Router and Deployer Lambda functions
     */
    readonly appEnv: string;
    /**
     * Optional asset name root
     *
     * @example microapps
     * @default - resource names auto assigned
     */
    readonly assetNameRoot?: string;
    /**
     * Optional asset name suffix
     *
     * @example -dev-pr-12
     * @default none
     */
    readonly assetNameSuffix?: string;
    /**
     * Use a strict S3 Bucket Policy that prevents applications
     * from reading/writing/modifying/deleting files in the S3 Bucket
     * outside of the path that is specific to their app/version.
     *
     * This setting should be used when applications are less than
     * fully trusted.
     *
     * @default false
     */
    readonly s3StrictBucketPolicy?: boolean;
    /**
     * Applies when using s3StrictBucketPolicy = true
     *
     * IAM Role or IAM User names to exclude from the DENY rules on the S3 Bucket Policy.
     *
     * Roles that are Assumed must instead have their AROA added to `s3PolicyBypassAROAs`.
     *
     * Typically any admin roles / users that need to view or manage the S3 Bucket
     * would be added to this list.
     *
     * @example ['arn:aws:iam::1234567890123:role/AdminAccess', 'arn:aws:iam::1234567890123:user/MyAdminUser']
     *
     * @see s3PolicyBypassAROAs
     */
    readonly s3PolicyBypassPrincipalARNs?: string[];
    /**
     * Applies when using s3StrictBucketPolicy = true
     *
     * AROAs of the IAM Role to exclude from the DENY rules on the S3 Bucket Policy.
     * This allows sessions that assume the IAM Role to be excluded from the
     * DENY rules on the S3 Bucket Policy.
     *
     * Typically any admin roles / users that need to view or manage the S3 Bucket
     * would be added to this list.
     *
     * Roles / users that are used directly, not assumed, can be added to `s3PolicyBypassRoleNames` instead.
     *
     * Note: This AROA must be specified to prevent this policy from locking
     * out non-root sessions that have assumed the admin role.
     *
     * The notPrincipals will only match the role name exactly and will not match
     * any session that has assumed the role since notPrincipals does not allow
     * wildcard matches and does not do wildcard matches implicitly either.
     *
     * The AROA must be used because there are only 3 Principal variables available:
     *  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
     *  aws:username, aws:userid, aws:PrincipalTag
     *
     * For an assumed role, aws:username is blank, aws:userid is:
     *  [unique id AKA AROA for Role]:[session name]
     *
     * Table of unique ID prefixes such as AROA:
     *  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
     *
     * The name of the role is simply not available for an assumed role and, if it was,
     * a complicated comparison would be requierd to prevent exclusion
     * of applying the Deny Rule to roles from other accounts.
     *
     * To get the AROA with the AWS CLI:
     *   aws iam get-role --role-name ROLE-NAME
     *   aws iam get-user -–user-name USER-NAME
     *
     * @example [ 'AROA1234567890123' ]
     *
     * @see s3StrictBucketPolicy
     */
    readonly s3PolicyBypassAROAs?: string[];
    /**
     * Path prefix on the root of the deployment
     *
     * @example dev/
     * @default none
     */
    readonly rootPathPrefix?: string;
}
/**
 * Represents a MicroApps Services
 */
export interface IMicroAppsSvcs {
    /**
     * DynamoDB table used by Router, Deployer, and Release console app
     */
    readonly table: dynamodb.ITable;
    /**
     * Lambda function for the Deployer
     */
    readonly deployerFunc: lambda.IFunction;
    /**
     * Lambda function for the Router
     */
    readonly routerFunc: lambda.IFunction;
}
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
export declare class MicroAppsSvcs extends Construct implements IMicroAppsSvcs {
    private _table;
    get table(): dynamodb.ITable;
    private _deployerFunc;
    get deployerFunc(): lambda.IFunction;
    private _routerFunc;
    get routerFunc(): lambda.IFunction;
    constructor(scope: Construct, id: string, props?: MicroAppsSvcsProps);
}
//# sourceMappingURL=MicroAppsSvcs.d.ts.map