"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsCF = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cf = require("aws-cdk-lib/aws-cloudfront");
const cforigins = require("aws-cdk-lib/aws-cloudfront-origins");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
const ReverseDomain_1 = require("./utils/ReverseDomain");
/**
 * Create a new MicroApps CloudFront Distribution.
 */
class MicroAppsCF extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props must be set');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        const { domainNameEdge, domainNameOrigin, httpApi, removalPolicy, certEdge, assetNameRoot, assetNameSuffix, r53Zone, bucketLogs, bucketAppsOrigin, rootPathPrefix, createAPIPathRoute = true, } = props;
        const apigwyOriginRequestPolicy = MicroAppsCF.createAPIOriginPolicy(this, {
            assetNameRoot,
            assetNameSuffix,
            domainNameEdge,
        });
        //
        // Determine URL of the origin FQDN
        //
        let httpOriginFQDN = 'invalid.pwrdrvr.com';
        if (domainNameOrigin !== undefined) {
            httpOriginFQDN = domainNameOrigin;
        }
        else {
            httpOriginFQDN = `${httpApi.apiId}.execute-api.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`;
        }
        //
        // CloudFront Distro
        //
        const apiGwyOrigin = new cforigins.HttpOrigin(httpOriginFQDN, {
            protocolPolicy: cf.OriginProtocolPolicy.HTTPS_ONLY,
            originSslProtocols: [cf.OriginSslPolicy.TLS_V1_2],
        });
        this._cloudFrontDistro = new cf.Distribution(this, 'cft', {
            comment: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : domainNameEdge,
            domainNames: domainNameEdge !== undefined ? [domainNameEdge] : undefined,
            certificate: certEdge,
            httpVersion: cf.HttpVersion.HTTP2,
            defaultBehavior: {
                allowedMethods: cf.AllowedMethods.ALLOW_ALL,
                cachePolicy: cf.CachePolicy.CACHING_DISABLED,
                compress: true,
                originRequestPolicy: apigwyOriginRequestPolicy,
                origin: apiGwyOrigin,
                viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            },
            enableIpv6: true,
            priceClass: cf.PriceClass.PRICE_CLASS_100,
            logBucket: bucketLogs,
            logFilePrefix: props.domainNameEdge
                ? `${ReverseDomain_1.reverseDomain(props.domainNameEdge)}/cloudfront-raw/`
                : undefined,
        });
        if (removalPolicy !== undefined) {
            this._cloudFrontDistro.applyRemovalPolicy(removalPolicy);
        }
        // Add routes to the CloudFront Distribution
        MicroAppsCF.addRoutes(scope, {
            apiGwyOrigin,
            bucketAppsOrigin,
            distro: this._cloudFrontDistro,
            apigwyOriginRequestPolicy: apigwyOriginRequestPolicy,
            rootPathPrefix,
            createAPIPathRoute,
        });
        //
        // Create the edge name for the CloudFront distro
        //
        if (r53Zone !== undefined) {
            const rrAppsEdge = new r53.RecordSet(this, 'edge-arecord', {
                recordName: domainNameEdge,
                recordType: r53.RecordType.A,
                target: r53.RecordTarget.fromAlias(new r53targets.CloudFrontTarget(this._cloudFrontDistro)),
                zone: r53Zone,
            });
            if (removalPolicy !== undefined) {
                rrAppsEdge.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    /**
     * Create or get the origin request policy
     *
     * If a custom domain name is NOT used for the origin then a policy
     * will be created.
     *
     * If a custom domain name IS used for the origin then the ALL_VIEWER
     * policy will be returned.  This policy passes the Host header to the
     * origin, which is fine when using a custom domain name on the origin.
     *
     * @param scope
     * @param props
     */
    static createAPIOriginPolicy(scope, props) {
        const { assetNameRoot, assetNameSuffix, domainNameEdge } = props;
        let apigwyOriginRequestPolicy = cf.OriginRequestPolicy.ALL_VIEWER;
        if (domainNameEdge === undefined) {
            // When not using a custom domain name we must limit down the origin policy to
            // prevent it from passing the Host header (distribution_id.cloudfront.net) to
            // apigwy which will then reject it with a 403 because it does not match the
            // execute-api name that apigwy is expecting.
            //
            // 2021-12-28 - There is a bug in the name generation that causes the same asset
            // in different stacks to have the same generated name.  We have to make the id
            // in all cases to ensure the generated name is unique.
            apigwyOriginRequestPolicy = new cf.OriginRequestPolicy(scope, `apigwy-origin-policy-${aws_cdk_lib_1.Stack.of(scope).stackName}`, {
                comment: assetNameRoot ? `${assetNameRoot}-apigwy${assetNameSuffix}` : undefined,
                originRequestPolicyName: assetNameRoot
                    ? `${assetNameRoot}-apigwy${assetNameSuffix}`
                    : undefined,
                cookieBehavior: cf.OriginRequestCookieBehavior.all(),
                queryStringBehavior: cf.OriginRequestQueryStringBehavior.all(),
                headerBehavior: cf.OriginRequestHeaderBehavior.allowList('user-agent', 'referer'),
            });
        }
        return apigwyOriginRequestPolicy;
    }
    /**
     * Add API Gateway and S3 routes to an existing CloudFront Distribution
     * @param _scope
     * @param props
     */
    static addRoutes(_scope, props) {
        const { apiGwyOrigin, bucketAppsOrigin, distro, apigwyOriginRequestPolicy, rootPathPrefix = '', createAPIPathRoute = true, } = props;
        //
        // Add Behaviors
        //
        const s3BehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_GET_HEAD_OPTIONS,
            cachePolicy: cf.CachePolicy.CACHING_OPTIMIZED,
            compress: true,
            originRequestPolicy: cf.OriginRequestPolicy.CORS_S3_ORIGIN,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        const apiGwyBehaviorOptions = {
            allowedMethods: cf.AllowedMethods.ALLOW_ALL,
            // TODO: Caching needs to be set by the app response
            cachePolicy: cf.CachePolicy.CACHING_DISABLED,
            compress: true,
            originRequestPolicy: apigwyOriginRequestPolicy,
            viewerProtocolPolicy: cf.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        //
        // If a route specifically has `/api/` in it, send it to API Gateway
        // This is needed to catch routes that have periods in the API path data,
        // such as: /release/0.0.0/api/update/default/release/0.0.0
        //
        if (createAPIPathRoute) {
            distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/api/*'), apiGwyOrigin, apiGwyBehaviorOptions);
        }
        //
        // All static assets are assumed to have a dot in them
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*/*/*.*'), bucketAppsOrigin, s3BehaviorOptions);
        //
        // Everything that isn't a static asset is going to API Gateway
        // There is no trailing slash because Serverless Next.js wants
        // go load pages at /release/0.0.3 (with no trailing slash).
        //
        distro.addBehavior(path_1.posix.join(rootPathPrefix, '/*'), apiGwyOrigin, apiGwyBehaviorOptions);
    }
    get cloudFrontDistro() {
        return this._cloudFrontDistro;
    }
}
exports.MicroAppsCF = MicroAppsCF;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsCF[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsCF", version: "0.2.10" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiTWljcm9BcHBzQ0YuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvTWljcm9BcHBzQ0YudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwrQkFBMEM7QUFFMUMsNkNBQXdEO0FBR3hELGlEQUFpRDtBQUNqRCxnRUFBZ0U7QUFDaEUsK0NBQStDO0FBQy9DLDhEQUE4RDtBQUU5RCwyQ0FBdUM7QUFDdkMseURBQXNEO0FBNkt0RDs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLHNCQUFTO0lBc0h4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztTQUN0QztRQUVELElBQ0UsQ0FBQyxLQUFLLENBQUMsT0FBTyxLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsY0FBYyxLQUFLLFNBQVMsQ0FBQztZQUNuRSxDQUFDLEtBQUssQ0FBQyxPQUFPLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLEVBQ25FO1lBQ0EsTUFBTSxJQUFJLEtBQUssQ0FBQywyRUFBMkUsQ0FBQyxDQUFDO1NBQzlGO1FBRUQsTUFBTSxFQUNKLGNBQWMsRUFDZCxnQkFBZ0IsRUFDaEIsT0FBTyxFQUNQLGFBQWEsRUFDYixRQUFRLEVBQ1IsYUFBYSxFQUNiLGVBQWUsRUFDZixPQUFPLEVBQ1AsVUFBVSxFQUNWLGdCQUFnQixFQUNoQixjQUFjLEVBQ2Qsa0JBQWtCLEdBQUcsSUFBSSxHQUMxQixHQUFHLEtBQUssQ0FBQztRQUVWLE1BQU0seUJBQXlCLEdBQUcsV0FBVyxDQUFDLHFCQUFxQixDQUFDLElBQUksRUFBRTtZQUN4RSxhQUFhO1lBQ2IsZUFBZTtZQUNmLGNBQWM7U0FDZixDQUFDLENBQUM7UUFFSCxFQUFFO1FBQ0YsbUNBQW1DO1FBQ25DLEVBQUU7UUFDRixJQUFJLGNBQWMsR0FBVyxxQkFBcUIsQ0FBQztRQUNuRCxJQUFJLGdCQUFnQixLQUFLLFNBQVMsRUFBRTtZQUNsQyxjQUFjLEdBQUcsZ0JBQWdCLENBQUM7U0FDbkM7YUFBTTtZQUNMLGNBQWMsR0FBRyxHQUFHLE9BQU8sQ0FBQyxLQUFLLGdCQUFnQixpQkFBRyxDQUFDLE1BQU0sZ0JBQWdCLENBQUM7U0FDN0U7UUFFRCxFQUFFO1FBQ0Ysb0JBQW9CO1FBQ3BCLEVBQUU7UUFDRixNQUFNLFlBQVksR0FBRyxJQUFJLFNBQVMsQ0FBQyxVQUFVLENBQUMsY0FBYyxFQUFFO1lBQzVELGNBQWMsRUFBRSxFQUFFLENBQUMsb0JBQW9CLENBQUMsVUFBVTtZQUNsRCxrQkFBa0IsRUFBRSxDQUFDLEVBQUUsQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDO1NBQ2xELENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLEVBQUUsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtZQUN4RCxPQUFPLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQyxHQUFHLGFBQWEsR0FBRyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUMsY0FBYztZQUM5RSxXQUFXLEVBQUUsY0FBYyxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN4RSxXQUFXLEVBQUUsUUFBUTtZQUNyQixXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxLQUFLO1lBQ2pDLGVBQWUsRUFBRTtnQkFDZixjQUFjLEVBQUUsRUFBRSxDQUFDLGNBQWMsQ0FBQyxTQUFTO2dCQUMzQyxXQUFXLEVBQUUsRUFBRSxDQUFDLFdBQVcsQ0FBQyxnQkFBZ0I7Z0JBQzVDLFFBQVEsRUFBRSxJQUFJO2dCQUNkLG1CQUFtQixFQUFFLHlCQUF5QjtnQkFDOUMsTUFBTSxFQUFFLFlBQVk7Z0JBQ3BCLG9CQUFvQixFQUFFLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUI7YUFDaEU7WUFDRCxVQUFVLEVBQUUsSUFBSTtZQUNoQixVQUFVLEVBQUUsRUFBRSxDQUFDLFVBQVUsQ0FBQyxlQUFlO1lBQ3pDLFNBQVMsRUFBRSxVQUFVO1lBQ3JCLGFBQWEsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDakMsQ0FBQyxDQUFDLEdBQUcsNkJBQWEsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLGtCQUFrQjtnQkFDMUQsQ0FBQyxDQUFDLFNBQVM7U0FDZCxDQUFDLENBQUM7UUFDSCxJQUFJLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDL0IsSUFBSSxDQUFDLGlCQUFpQixDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzFEO1FBRUQsNENBQTRDO1FBQzVDLFdBQVcsQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFO1lBQzNCLFlBQVk7WUFDWixnQkFBZ0I7WUFDaEIsTUFBTSxFQUFFLElBQUksQ0FBQyxpQkFBaUI7WUFDOUIseUJBQXlCLEVBQUUseUJBQXlCO1lBQ3BELGNBQWM7WUFDZCxrQkFBa0I7U0FDbkIsQ0FBQyxDQUFDO1FBRUgsRUFBRTtRQUNGLGlEQUFpRDtRQUNqRCxFQUFFO1FBRUYsSUFBSSxPQUFPLEtBQUssU0FBUyxFQUFFO1lBQ3pCLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO2dCQUN6RCxVQUFVLEVBQUUsY0FBYztnQkFDMUIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDNUIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLElBQUksVUFBVSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO2dCQUMzRixJQUFJLEVBQUUsT0FBTzthQUNkLENBQUMsQ0FBQztZQUNILElBQUksYUFBYSxLQUFLLFNBQVMsRUFBRTtnQkFDL0IsVUFBVSxDQUFDLGtCQUFrQixDQUFDLGFBQWEsQ0FBQyxDQUFDO2FBQzlDO1NBQ0Y7SUFDSCxDQUFDO0lBMU5EOzs7Ozs7Ozs7Ozs7T0FZRztJQUNJLE1BQU0sQ0FBQyxxQkFBcUIsQ0FDakMsS0FBZ0IsRUFDaEIsS0FBbUM7UUFFbkMsTUFBTSxFQUFFLGFBQWEsRUFBRSxlQUFlLEVBQUUsY0FBYyxFQUFFLEdBQUcsS0FBSyxDQUFDO1FBRWpFLElBQUkseUJBQXlCLEdBQTRCLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxVQUFVLENBQUM7UUFDM0YsSUFBSSxjQUFjLEtBQUssU0FBUyxFQUFFO1lBQ2hDLDhFQUE4RTtZQUM5RSw4RUFBOEU7WUFDOUUsNEVBQTRFO1lBQzVFLDZDQUE2QztZQUM3QyxFQUFFO1lBQ0YsZ0ZBQWdGO1lBQ2hGLCtFQUErRTtZQUMvRSx1REFBdUQ7WUFDdkQseUJBQXlCLEdBQUcsSUFBSSxFQUFFLENBQUMsbUJBQW1CLENBQ3BELEtBQUssRUFDTCx3QkFBd0IsbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQ25EO2dCQUNFLE9BQU8sRUFBRSxhQUFhLENBQUMsQ0FBQyxDQUFDLEdBQUcsYUFBYSxVQUFVLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUVoRix1QkFBdUIsRUFBRSxhQUFhO29CQUNwQyxDQUFDLENBQUMsR0FBRyxhQUFhLFVBQVUsZUFBZSxFQUFFO29CQUM3QyxDQUFDLENBQUMsU0FBUztnQkFDYixjQUFjLEVBQUUsRUFBRSxDQUFDLDJCQUEyQixDQUFDLEdBQUcsRUFBRTtnQkFDcEQsbUJBQW1CLEVBQUUsRUFBRSxDQUFDLGdDQUFnQyxDQUFDLEdBQUcsRUFBRTtnQkFDOUQsY0FBYyxFQUFFLEVBQUUsQ0FBQywyQkFBMkIsQ0FBQyxTQUFTLENBQUMsWUFBWSxFQUFFLFNBQVMsQ0FBQzthQUNsRixDQUNGLENBQUM7U0FDSDtRQUVELE9BQU8seUJBQXlCLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsU0FBUyxDQUFDLE1BQWlCLEVBQUUsS0FBdUI7UUFDaEUsTUFBTSxFQUNKLFlBQVksRUFDWixnQkFBZ0IsRUFDaEIsTUFBTSxFQUNOLHlCQUF5QixFQUN6QixjQUFjLEdBQUcsRUFBRSxFQUNuQixrQkFBa0IsR0FBRyxJQUFJLEdBQzFCLEdBQUcsS0FBSyxDQUFDO1FBRVYsRUFBRTtRQUNGLGdCQUFnQjtRQUNoQixFQUFFO1FBQ0YsTUFBTSxpQkFBaUIsR0FBMEI7WUFDL0MsY0FBYyxFQUFFLEVBQUUsQ0FBQyxjQUFjLENBQUMsc0JBQXNCO1lBQ3hELFdBQVcsRUFBRSxFQUFFLENBQUMsV0FBVyxDQUFDLGlCQUFpQjtZQUM3QyxRQUFRLEVBQUUsSUFBSTtZQUNkLG1CQUFtQixFQUFFLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQyxjQUFjO1lBQzFELG9CQUFvQixFQUFFLEVBQUUsQ0FBQyxvQkFBb0IsQ0FBQyxpQkFBaUI7U0FDaEUsQ0FBQztRQUNGLE1BQU0scUJBQXFCLEdBQTBCO1lBQ25ELGNBQWMsRUFBRSxFQUFFLENBQUMsY0FBYyxDQUFDLFNBQVM7WUFDM0Msb0RBQW9EO1lBQ3BELFdBQVcsRUFBRSxFQUFFLENBQUMsV0FBVyxDQUFDLGdCQUFnQjtZQUM1QyxRQUFRLEVBQUUsSUFBSTtZQUNkLG1CQUFtQixFQUFFLHlCQUF5QjtZQUM5QyxvQkFBb0IsRUFBRSxFQUFFLENBQUMsb0JBQW9CLENBQUMsaUJBQWlCO1NBQ2hFLENBQUM7UUFFRixFQUFFO1FBQ0Ysb0VBQW9FO1FBQ3BFLHlFQUF5RTtRQUN6RSwyREFBMkQ7UUFDM0QsRUFBRTtRQUNGLElBQUksa0JBQWtCLEVBQUU7WUFDdEIsTUFBTSxDQUFDLFdBQVcsQ0FDaEIsWUFBUyxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsWUFBWSxDQUFDLEVBQzVDLFlBQVksRUFDWixxQkFBcUIsQ0FDdEIsQ0FBQztTQUNIO1FBRUQsRUFBRTtRQUNGLHNEQUFzRDtRQUN0RCxFQUFFO1FBQ0YsTUFBTSxDQUFDLFdBQVcsQ0FDaEIsWUFBUyxDQUFDLElBQUksQ0FBQyxjQUFjLEVBQUUsVUFBVSxDQUFDLEVBQzFDLGdCQUFnQixFQUNoQixpQkFBaUIsQ0FDbEIsQ0FBQztRQUVGLEVBQUU7UUFDRiwrREFBK0Q7UUFDL0QsOERBQThEO1FBQzlELDREQUE0RDtRQUM1RCxFQUFFO1FBQ0YsTUFBTSxDQUFDLFdBQVcsQ0FBQyxZQUFTLENBQUMsSUFBSSxDQUFDLGNBQWMsRUFBRSxJQUFJLENBQUMsRUFBRSxZQUFZLEVBQUUscUJBQXFCLENBQUMsQ0FBQztJQUNoRyxDQUFDO0lBR0QsSUFBVyxnQkFBZ0I7UUFDekIsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUM7SUFDaEMsQ0FBQzs7QUFwSEgsa0NBNE5DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgcG9zaXggYXMgcG9zaXhQYXRoIH0gZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBhcGlnd3kgZnJvbSAnQGF3cy1jZGsvYXdzLWFwaWdhdGV3YXl2Mi1hbHBoYSc7XG5pbXBvcnQgeyBBd3MsIFJlbW92YWxQb2xpY3ksIFN0YWNrIH0gZnJvbSAnYXdzLWNkay1saWInO1xuLy8gaW1wb3J0ICogYXMgYXBpZ3d5Y2ZuIGZyb20gJ2F3cy1jZGstbGliL2F3cy1hcGlnYXRld2F5djInO1xuaW1wb3J0ICogYXMgYWNtIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jZXJ0aWZpY2F0ZW1hbmFnZXInO1xuaW1wb3J0ICogYXMgY2YgZnJvbSAnYXdzLWNkay1saWIvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgY2ZvcmlnaW5zIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZGZyb250LW9yaWdpbnMnO1xuaW1wb3J0ICogYXMgcjUzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1yb3V0ZTUzJztcbmltcG9ydCAqIGFzIHI1M3RhcmdldHMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXJvdXRlNTMtdGFyZ2V0cyc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyByZXZlcnNlRG9tYWluIH0gZnJvbSAnLi91dGlscy9SZXZlcnNlRG9tYWluJztcblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgTWljcm9BcHBzIENsb3VkRnJvbnRcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJTWljcm9BcHBzQ0Yge1xuICByZWFkb25seSBjbG91ZEZyb250RGlzdHJvOiBjZi5EaXN0cmlidXRpb247XG59XG5cbi8qKlxuICogUHJvcGVydGllcyB0byBpbml0aWFsaXplIGFuIGluc3RhbmNlIG9mIGBNaWNyb0FwcHNDRmAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTWljcm9BcHBzQ0ZQcm9wcyB7XG4gIC8qKlxuICAgKiBSZW1vdmFsUG9saWN5IG92ZXJyaWRlIGZvciBjaGlsZCByZXNvdXJjZXNcbiAgICpcbiAgICogTm90ZTogaWYgc2V0IHRvIERFU1RST1kgdGhlIFMzIGJ1Y2tlcyB3aWxsIGhhdmUgYGF1dG9EZWxldGVPYmplY3RzYCBzZXQgdG8gYHRydWVgXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gcGVyIHJlc291cmNlIGRlZmF1bHRcbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xuXG4gIC8qKlxuICAgKiBTMyBidWNrZXQgb3JpZ2luIGZvciBkZXBsb3llZCBhcHBsaWNhdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldEFwcHNPcmlnaW46IGNmb3JpZ2lucy5TM09yaWdpbjtcblxuICAvKipcbiAgICogUzMgYnVja2V0IGZvciBDbG91ZEZyb250IGxvZ3NcbiAgICovXG4gIHJlYWRvbmx5IGJ1Y2tldExvZ3M/OiBzMy5JQnVja2V0O1xuXG4gIC8qKlxuICAgKiBDbG91ZEZyb250IERpc3RyaWJ1dGlvbiBkb21haW4gbmFtZVxuICAgKlxuICAgKiBAZXhhbXBsZSBhcHBzLnB3cmRydnIuY29tXG4gICAqIEBkZWZhdWx0IGF1dG8tYXNzaWduZWRcbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVFZGdlPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBUEkgR2F0ZXdheSBjdXN0b20gb3JpZ2luIGRvbWFpbiBuYW1lXG4gICAqXG4gICAqIEBleGFtcGxlIGFwcHMucHdyZHJ2ci5jb21cbiAgICogQGRlZmF1bHQgLSByZXRyaWV2ZWQgZnJvbSBodHRwQXBpLCBpZiBwb3NzaWJsZVxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZU9yaWdpbj86IHN0cmluZztcblxuICAvKipcbiAgICogQVBJIEdhdGV3YXkgdjIgSFRUUCBBUEkgZm9yIGFwcHNcbiAgICovXG4gIHJlYWRvbmx5IGh0dHBBcGk6IGFwaWd3eS5IdHRwQXBpO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhc3NldCBuYW1lIHJvb3RcbiAgICpcbiAgICogQGV4YW1wbGUgbWljcm9hcHBzXG4gICAqIEBkZWZhdWx0IC0gcmVzb3VyY2UgbmFtZXMgYXV0byBhc3NpZ25lZFxuICAgKi9cbiAgcmVhZG9ubHkgYXNzZXROYW1lUm9vdD86IHN0cmluZztcblxuICAvKipcbiAgICogT3B0aW9uYWwgYXNzZXQgbmFtZSBzdWZmaXhcbiAgICpcbiAgICogQGV4YW1wbGUgLWRldi1wci0xMlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBhc3NldE5hbWVTdWZmaXg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEFDTSBDZXJ0aWZpY2F0ZSB0aGF0IGNvdmVycyBgZG9tYWluTmFtZUVkZ2VgIG5hbWVcbiAgICovXG4gIHJlYWRvbmx5IGNlcnRFZGdlPzogYWNtLklDZXJ0aWZpY2F0ZTtcblxuICAvKipcbiAgICogUm91dGU1MyB6b25lIGluIHdoaWNoIHRvIGNyZWF0ZSBvcHRpb25hbCBgZG9tYWluTmFtZUVkZ2VgIHJlY29yZFxuICAgKi9cbiAgcmVhZG9ubHkgcjUzWm9uZT86IHI1My5JSG9zdGVkWm9uZTtcblxuICAvKipcbiAgICogUGF0aCBwcmVmaXggb24gdGhlIHJvb3Qgb2YgdGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uXG4gICAqXG4gICAqIEBleGFtcGxlIGRldi9cbiAgICovXG4gIHJlYWRvbmx5IHJvb3RQYXRoUHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gZXh0cmEgQmVoYXZpb3IgKFJvdXRlKSBmb3IgL2FwaS8gdGhhdCBhbGxvd3NcbiAgICogQVBJIHJvdXRlcyB0byBoYXZlIGEgcGVyaW9kIGluIHRoZW0uXG4gICAqXG4gICAqIFdoZW4gZmFsc2UgQVBJIHJvdXRlcyB3aXRoIGEgcGVyaW9kIGluIHRoZSBwYXRoIHdpbGwgZ2V0IHJvdXRlZCB0byBTMy5cbiAgICpcbiAgICogV2hlbiB0cnVlIEFQSSByb3V0ZXMgdGhhdCBjb250YWluIC9hcGkvIGluIHRoZSBwYXRoIHdpbGwgZ2V0IHJvdXRlZCB0byBBUEkgR2F0ZXdheVxuICAgKiBldmVuIGlmIHRoZXkgaGF2ZSBhIHBlcmlvZCBpbiB0aGUgcGF0aC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRlQVBJUGF0aFJvdXRlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciB0aGUgYENyZWF0ZUFQSU9yaWdpblBvbGljeWBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDcmVhdGVBUElPcmlnaW5Qb2xpY3lPcHRpb25zIHtcbiAgLyoqXG4gICAqIE9wdGlvbmFsIGFzc2V0IG5hbWUgcm9vdFxuICAgKlxuICAgKiBAZXhhbXBsZSBtaWNyb2FwcHNcbiAgICogQGRlZmF1bHQgLSByZXNvdXJjZSBuYW1lcyBhdXRvIGFzc2lnbmVkXG4gICAqL1xuICByZWFkb25seSBhc3NldE5hbWVSb290Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBPcHRpb25hbCBhc3NldCBuYW1lIHN1ZmZpeFxuICAgKlxuICAgKiBAZXhhbXBsZSAtZGV2LXByLTEyXG4gICAqIEBkZWZhdWx0IG5vbmVcbiAgICovXG4gIHJlYWRvbmx5IGFzc2V0TmFtZVN1ZmZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogRWRnZSBkb21haW4gbmFtZSB1c2VkIGJ5IENsb3VkRnJvbnQgLSBJZiBzZXQgYSBjdXN0b21cbiAgICogT3JpZ2luUmVxdWVzdFBvbGljeSB3aWxsIGJlIGNyZWF0ZWQgdGhhdCBwcmV2ZW50c1xuICAgKiB0aGUgSG9zdCBoZWFkZXIgZnJvbSBiZWluZyBwYXNzZWQgdG8gdGhlIG9yaWdpbi5cbiAgICovXG4gIHJlYWRvbmx5IGRvbWFpbk5hbWVFZGdlPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGBBZGRSb3V0ZXNgXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkUm91dGVzT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBBUEkgR2F0ZXdheSBDbG91ZEZyb250IE9yaWdpbiBmb3IgQVBJIGNhbGxzXG4gICAqL1xuICByZWFkb25seSBhcGlHd3lPcmlnaW46IGNmLklPcmlnaW47XG5cbiAgLyoqXG4gICAqIFMzIEJ1Y2tldCBDbG91ZEZyb250IE9yaWdpbiBmb3Igc3RhdGljIGFzc2V0c1xuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0QXBwc09yaWdpbjogY2ZvcmlnaW5zLlMzT3JpZ2luO1xuXG4gIC8qKlxuICAgKiBDbG91ZEZyb250IERpc3RyaWJ1dGlvbiB0byBhZGQgdGhlIEJlaGF2aW9ycyAoUm91dGVzKSB0b1xuICAgKi9cbiAgcmVhZG9ubHkgZGlzdHJvOiBjZi5EaXN0cmlidXRpb247XG5cbiAgLyoqXG4gICAqIE9yaWdpbiBSZXF1ZXN0IHBvbGljeSBmb3IgQVBJIEdhdGV3YXkgT3JpZ2luXG4gICAqL1xuICByZWFkb25seSBhcGlnd3lPcmlnaW5SZXF1ZXN0UG9saWN5OiBjZi5JT3JpZ2luUmVxdWVzdFBvbGljeTtcblxuICAvKipcbiAgICogUGF0aCBwcmVmaXggb24gdGhlIHJvb3Qgb2YgdGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uXG4gICAqXG4gICAqIEBleGFtcGxlIGRldi9cbiAgICovXG4gIHJlYWRvbmx5IHJvb3RQYXRoUHJlZml4Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBDcmVhdGUgYW4gZXh0cmEgQmVoYXZpb3IgKFJvdXRlKSBmb3IgL2FwaS8gdGhhdCBhbGxvd3NcbiAgICogQVBJIHJvdXRlcyB0byBoYXZlIGEgcGVyaW9kIGluIHRoZW0uXG4gICAqXG4gICAqIFdoZW4gZmFsc2UgQVBJIHJvdXRlcyB3aXRoIGEgcGVyaW9kIGluIHRoZSBwYXRoIHdpbGwgZ2V0IHJvdXRlZCB0byBTMy5cbiAgICpcbiAgICogV2hlbiB0cnVlIEFQSSByb3V0ZXMgdGhhdCBjb250YWluIC9hcGkvIGluIHRoZSBwYXRoIHdpbGwgZ2V0IHJvdXRlZCB0byBBUEkgR2F0ZXdheVxuICAgKiBldmVuIGlmIHRoZXkgaGF2ZSBhIHBlcmlvZCBpbiB0aGUgcGF0aC5cbiAgICpcbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgY3JlYXRlQVBJUGF0aFJvdXRlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBDcmVhdGUgYSBuZXcgTWljcm9BcHBzIENsb3VkRnJvbnQgRGlzdHJpYnV0aW9uLlxuICovXG5leHBvcnQgY2xhc3MgTWljcm9BcHBzQ0YgZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJTWljcm9BcHBzQ0Yge1xuICAvKipcbiAgICogQ3JlYXRlIG9yIGdldCB0aGUgb3JpZ2luIHJlcXVlc3QgcG9saWN5XG4gICAqXG4gICAqIElmIGEgY3VzdG9tIGRvbWFpbiBuYW1lIGlzIE5PVCB1c2VkIGZvciB0aGUgb3JpZ2luIHRoZW4gYSBwb2xpY3lcbiAgICogd2lsbCBiZSBjcmVhdGVkLlxuICAgKlxuICAgKiBJZiBhIGN1c3RvbSBkb21haW4gbmFtZSBJUyB1c2VkIGZvciB0aGUgb3JpZ2luIHRoZW4gdGhlIEFMTF9WSUVXRVJcbiAgICogcG9saWN5IHdpbGwgYmUgcmV0dXJuZWQuICBUaGlzIHBvbGljeSBwYXNzZXMgdGhlIEhvc3QgaGVhZGVyIHRvIHRoZVxuICAgKiBvcmlnaW4sIHdoaWNoIGlzIGZpbmUgd2hlbiB1c2luZyBhIGN1c3RvbSBkb21haW4gbmFtZSBvbiB0aGUgb3JpZ2luLlxuICAgKlxuICAgKiBAcGFyYW0gc2NvcGVcbiAgICogQHBhcmFtIHByb3BzXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGNyZWF0ZUFQSU9yaWdpblBvbGljeShcbiAgICBzY29wZTogQ29uc3RydWN0LFxuICAgIHByb3BzOiBDcmVhdGVBUElPcmlnaW5Qb2xpY3lPcHRpb25zLFxuICApOiBjZi5JT3JpZ2luUmVxdWVzdFBvbGljeSB7XG4gICAgY29uc3QgeyBhc3NldE5hbWVSb290LCBhc3NldE5hbWVTdWZmaXgsIGRvbWFpbk5hbWVFZGdlIH0gPSBwcm9wcztcblxuICAgIGxldCBhcGlnd3lPcmlnaW5SZXF1ZXN0UG9saWN5OiBjZi5JT3JpZ2luUmVxdWVzdFBvbGljeSA9IGNmLk9yaWdpblJlcXVlc3RQb2xpY3kuQUxMX1ZJRVdFUjtcbiAgICBpZiAoZG9tYWluTmFtZUVkZ2UgPT09IHVuZGVmaW5lZCkge1xuICAgICAgLy8gV2hlbiBub3QgdXNpbmcgYSBjdXN0b20gZG9tYWluIG5hbWUgd2UgbXVzdCBsaW1pdCBkb3duIHRoZSBvcmlnaW4gcG9saWN5IHRvXG4gICAgICAvLyBwcmV2ZW50IGl0IGZyb20gcGFzc2luZyB0aGUgSG9zdCBoZWFkZXIgKGRpc3RyaWJ1dGlvbl9pZC5jbG91ZGZyb250Lm5ldCkgdG9cbiAgICAgIC8vIGFwaWd3eSB3aGljaCB3aWxsIHRoZW4gcmVqZWN0IGl0IHdpdGggYSA0MDMgYmVjYXVzZSBpdCBkb2VzIG5vdCBtYXRjaCB0aGVcbiAgICAgIC8vIGV4ZWN1dGUtYXBpIG5hbWUgdGhhdCBhcGlnd3kgaXMgZXhwZWN0aW5nLlxuICAgICAgLy9cbiAgICAgIC8vIDIwMjEtMTItMjggLSBUaGVyZSBpcyBhIGJ1ZyBpbiB0aGUgbmFtZSBnZW5lcmF0aW9uIHRoYXQgY2F1c2VzIHRoZSBzYW1lIGFzc2V0XG4gICAgICAvLyBpbiBkaWZmZXJlbnQgc3RhY2tzIHRvIGhhdmUgdGhlIHNhbWUgZ2VuZXJhdGVkIG5hbWUuICBXZSBoYXZlIHRvIG1ha2UgdGhlIGlkXG4gICAgICAvLyBpbiBhbGwgY2FzZXMgdG8gZW5zdXJlIHRoZSBnZW5lcmF0ZWQgbmFtZSBpcyB1bmlxdWUuXG4gICAgICBhcGlnd3lPcmlnaW5SZXF1ZXN0UG9saWN5ID0gbmV3IGNmLk9yaWdpblJlcXVlc3RQb2xpY3koXG4gICAgICAgIHNjb3BlLFxuICAgICAgICBgYXBpZ3d5LW9yaWdpbi1wb2xpY3ktJHtTdGFjay5vZihzY29wZSkuc3RhY2tOYW1lfWAsXG4gICAgICAgIHtcbiAgICAgICAgICBjb21tZW50OiBhc3NldE5hbWVSb290ID8gYCR7YXNzZXROYW1lUm9vdH0tYXBpZ3d5JHthc3NldE5hbWVTdWZmaXh9YCA6IHVuZGVmaW5lZCxcblxuICAgICAgICAgIG9yaWdpblJlcXVlc3RQb2xpY3lOYW1lOiBhc3NldE5hbWVSb290XG4gICAgICAgICAgICA/IGAke2Fzc2V0TmFtZVJvb3R9LWFwaWd3eSR7YXNzZXROYW1lU3VmZml4fWBcbiAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgIGNvb2tpZUJlaGF2aW9yOiBjZi5PcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3IuYWxsKCksXG4gICAgICAgICAgcXVlcnlTdHJpbmdCZWhhdmlvcjogY2YuT3JpZ2luUmVxdWVzdFF1ZXJ5U3RyaW5nQmVoYXZpb3IuYWxsKCksXG4gICAgICAgICAgaGVhZGVyQmVoYXZpb3I6IGNmLk9yaWdpblJlcXVlc3RIZWFkZXJCZWhhdmlvci5hbGxvd0xpc3QoJ3VzZXItYWdlbnQnLCAncmVmZXJlcicpLFxuICAgICAgICB9LFxuICAgICAgKTtcbiAgICB9XG5cbiAgICByZXR1cm4gYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgQVBJIEdhdGV3YXkgYW5kIFMzIHJvdXRlcyB0byBhbiBleGlzdGluZyBDbG91ZEZyb250IERpc3RyaWJ1dGlvblxuICAgKiBAcGFyYW0gX3Njb3BlXG4gICAqIEBwYXJhbSBwcm9wc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhZGRSb3V0ZXMoX3Njb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBBZGRSb3V0ZXNPcHRpb25zKSB7XG4gICAgY29uc3Qge1xuICAgICAgYXBpR3d5T3JpZ2luLFxuICAgICAgYnVja2V0QXBwc09yaWdpbixcbiAgICAgIGRpc3RybyxcbiAgICAgIGFwaWd3eU9yaWdpblJlcXVlc3RQb2xpY3ksXG4gICAgICByb290UGF0aFByZWZpeCA9ICcnLFxuICAgICAgY3JlYXRlQVBJUGF0aFJvdXRlID0gdHJ1ZSxcbiAgICB9ID0gcHJvcHM7XG5cbiAgICAvL1xuICAgIC8vIEFkZCBCZWhhdmlvcnNcbiAgICAvL1xuICAgIGNvbnN0IHMzQmVoYXZpb3JPcHRpb25zOiBjZi5BZGRCZWhhdmlvck9wdGlvbnMgPSB7XG4gICAgICBhbGxvd2VkTWV0aG9kczogY2YuQWxsb3dlZE1ldGhvZHMuQUxMT1dfR0VUX0hFQURfT1BUSU9OUyxcbiAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX09QVElNSVpFRCxcbiAgICAgIGNvbXByZXNzOiB0cnVlLFxuICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeTogY2YuT3JpZ2luUmVxdWVzdFBvbGljeS5DT1JTX1MzX09SSUdJTixcbiAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBjZi5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICB9O1xuICAgIGNvbnN0IGFwaUd3eUJlaGF2aW9yT3B0aW9uczogY2YuQWRkQmVoYXZpb3JPcHRpb25zID0ge1xuICAgICAgYWxsb3dlZE1ldGhvZHM6IGNmLkFsbG93ZWRNZXRob2RzLkFMTE9XX0FMTCxcbiAgICAgIC8vIFRPRE86IENhY2hpbmcgbmVlZHMgdG8gYmUgc2V0IGJ5IHRoZSBhcHAgcmVzcG9uc2VcbiAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX0RJU0FCTEVELFxuICAgICAgY29tcHJlc3M6IHRydWUsXG4gICAgICBvcmlnaW5SZXF1ZXN0UG9saWN5OiBhcGlnd3lPcmlnaW5SZXF1ZXN0UG9saWN5LFxuICAgICAgdmlld2VyUHJvdG9jb2xQb2xpY3k6IGNmLlZpZXdlclByb3RvY29sUG9saWN5LlJFRElSRUNUX1RPX0hUVFBTLFxuICAgIH07XG5cbiAgICAvL1xuICAgIC8vIElmIGEgcm91dGUgc3BlY2lmaWNhbGx5IGhhcyBgL2FwaS9gIGluIGl0LCBzZW5kIGl0IHRvIEFQSSBHYXRld2F5XG4gICAgLy8gVGhpcyBpcyBuZWVkZWQgdG8gY2F0Y2ggcm91dGVzIHRoYXQgaGF2ZSBwZXJpb2RzIGluIHRoZSBBUEkgcGF0aCBkYXRhLFxuICAgIC8vIHN1Y2ggYXM6IC9yZWxlYXNlLzAuMC4wL2FwaS91cGRhdGUvZGVmYXVsdC9yZWxlYXNlLzAuMC4wXG4gICAgLy9cbiAgICBpZiAoY3JlYXRlQVBJUGF0aFJvdXRlKSB7XG4gICAgICBkaXN0cm8uYWRkQmVoYXZpb3IoXG4gICAgICAgIHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyovKi9hcGkvKicpLFxuICAgICAgICBhcGlHd3lPcmlnaW4sXG4gICAgICAgIGFwaUd3eUJlaGF2aW9yT3B0aW9ucyxcbiAgICAgICk7XG4gICAgfVxuXG4gICAgLy9cbiAgICAvLyBBbGwgc3RhdGljIGFzc2V0cyBhcmUgYXNzdW1lZCB0byBoYXZlIGEgZG90IGluIHRoZW1cbiAgICAvL1xuICAgIGRpc3Ryby5hZGRCZWhhdmlvcihcbiAgICAgIHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyovKi8qLionKSxcbiAgICAgIGJ1Y2tldEFwcHNPcmlnaW4sXG4gICAgICBzM0JlaGF2aW9yT3B0aW9ucyxcbiAgICApO1xuXG4gICAgLy9cbiAgICAvLyBFdmVyeXRoaW5nIHRoYXQgaXNuJ3QgYSBzdGF0aWMgYXNzZXQgaXMgZ29pbmcgdG8gQVBJIEdhdGV3YXlcbiAgICAvLyBUaGVyZSBpcyBubyB0cmFpbGluZyBzbGFzaCBiZWNhdXNlIFNlcnZlcmxlc3MgTmV4dC5qcyB3YW50c1xuICAgIC8vIGdvIGxvYWQgcGFnZXMgYXQgL3JlbGVhc2UvMC4wLjMgKHdpdGggbm8gdHJhaWxpbmcgc2xhc2gpLlxuICAgIC8vXG4gICAgZGlzdHJvLmFkZEJlaGF2aW9yKHBvc2l4UGF0aC5qb2luKHJvb3RQYXRoUHJlZml4LCAnLyonKSwgYXBpR3d5T3JpZ2luLCBhcGlHd3lCZWhhdmlvck9wdGlvbnMpO1xuICB9XG5cbiAgcHJpdmF0ZSBfY2xvdWRGcm9udERpc3RybzogY2YuRGlzdHJpYnV0aW9uO1xuICBwdWJsaWMgZ2V0IGNsb3VkRnJvbnREaXN0cm8oKTogY2YuRGlzdHJpYnV0aW9uIHtcbiAgICByZXR1cm4gdGhpcy5fY2xvdWRGcm9udERpc3RybztcbiAgfVxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBNaWNyb0FwcHNDRlByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ3Byb3BzIG11c3QgYmUgc2V0Jyk7XG4gICAgfVxuXG4gICAgaWYgKFxuICAgICAgKHByb3BzLnI1M1pvbmUgPT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kb21haW5OYW1lRWRnZSAhPT0gdW5kZWZpbmVkKSB8fFxuICAgICAgKHByb3BzLnI1M1pvbmUgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5kb21haW5OYW1lRWRnZSA9PT0gdW5kZWZpbmVkKVxuICAgICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdJZiBlaXRoZXIgb2YgcjUzWm9uZSBvciBkb21haW5OYW1lRWRnZSBhcmUgc2V0IHRoZW4gdGhlIG90aGVyIG11c3QgYmUgc2V0Jyk7XG4gICAgfVxuXG4gICAgY29uc3Qge1xuICAgICAgZG9tYWluTmFtZUVkZ2UsXG4gICAgICBkb21haW5OYW1lT3JpZ2luLFxuICAgICAgaHR0cEFwaSxcbiAgICAgIHJlbW92YWxQb2xpY3ksXG4gICAgICBjZXJ0RWRnZSxcbiAgICAgIGFzc2V0TmFtZVJvb3QsXG4gICAgICBhc3NldE5hbWVTdWZmaXgsXG4gICAgICByNTNab25lLFxuICAgICAgYnVja2V0TG9ncyxcbiAgICAgIGJ1Y2tldEFwcHNPcmlnaW4sXG4gICAgICByb290UGF0aFByZWZpeCxcbiAgICAgIGNyZWF0ZUFQSVBhdGhSb3V0ZSA9IHRydWUsXG4gICAgfSA9IHByb3BzO1xuXG4gICAgY29uc3QgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSA9IE1pY3JvQXBwc0NGLmNyZWF0ZUFQSU9yaWdpblBvbGljeSh0aGlzLCB7XG4gICAgICBhc3NldE5hbWVSb290LFxuICAgICAgYXNzZXROYW1lU3VmZml4LFxuICAgICAgZG9tYWluTmFtZUVkZ2UsXG4gICAgfSk7XG5cbiAgICAvL1xuICAgIC8vIERldGVybWluZSBVUkwgb2YgdGhlIG9yaWdpbiBGUUROXG4gICAgLy9cbiAgICBsZXQgaHR0cE9yaWdpbkZRRE46IHN0cmluZyA9ICdpbnZhbGlkLnB3cmRydnIuY29tJztcbiAgICBpZiAoZG9tYWluTmFtZU9yaWdpbiAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICBodHRwT3JpZ2luRlFETiA9IGRvbWFpbk5hbWVPcmlnaW47XG4gICAgfSBlbHNlIHtcbiAgICAgIGh0dHBPcmlnaW5GUUROID0gYCR7aHR0cEFwaS5hcGlJZH0uZXhlY3V0ZS1hcGkuJHtBd3MuUkVHSU9OfS5hbWF6b25hd3MuY29tYDtcbiAgICB9XG5cbiAgICAvL1xuICAgIC8vIENsb3VkRnJvbnQgRGlzdHJvXG4gICAgLy9cbiAgICBjb25zdCBhcGlHd3lPcmlnaW4gPSBuZXcgY2ZvcmlnaW5zLkh0dHBPcmlnaW4oaHR0cE9yaWdpbkZRRE4sIHtcbiAgICAgIHByb3RvY29sUG9saWN5OiBjZi5PcmlnaW5Qcm90b2NvbFBvbGljeS5IVFRQU19PTkxZLFxuICAgICAgb3JpZ2luU3NsUHJvdG9jb2xzOiBbY2YuT3JpZ2luU3NsUG9saWN5LlRMU19WMV8yXSxcbiAgICB9KTtcbiAgICB0aGlzLl9jbG91ZEZyb250RGlzdHJvID0gbmV3IGNmLkRpc3RyaWJ1dGlvbih0aGlzLCAnY2Z0Jywge1xuICAgICAgY29tbWVudDogYXNzZXROYW1lUm9vdCA/IGAke2Fzc2V0TmFtZVJvb3R9JHthc3NldE5hbWVTdWZmaXh9YCA6IGRvbWFpbk5hbWVFZGdlLFxuICAgICAgZG9tYWluTmFtZXM6IGRvbWFpbk5hbWVFZGdlICE9PSB1bmRlZmluZWQgPyBbZG9tYWluTmFtZUVkZ2VdIDogdW5kZWZpbmVkLFxuICAgICAgY2VydGlmaWNhdGU6IGNlcnRFZGdlLFxuICAgICAgaHR0cFZlcnNpb246IGNmLkh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgZGVmYXVsdEJlaGF2aW9yOiB7XG4gICAgICAgIGFsbG93ZWRNZXRob2RzOiBjZi5BbGxvd2VkTWV0aG9kcy5BTExPV19BTEwsXG4gICAgICAgIGNhY2hlUG9saWN5OiBjZi5DYWNoZVBvbGljeS5DQUNISU5HX0RJU0FCTEVELFxuICAgICAgICBjb21wcmVzczogdHJ1ZSxcbiAgICAgICAgb3JpZ2luUmVxdWVzdFBvbGljeTogYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSxcbiAgICAgICAgb3JpZ2luOiBhcGlHd3lPcmlnaW4sXG4gICAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBjZi5WaWV3ZXJQcm90b2NvbFBvbGljeS5SRURJUkVDVF9UT19IVFRQUyxcbiAgICAgIH0sXG4gICAgICBlbmFibGVJcHY2OiB0cnVlLFxuICAgICAgcHJpY2VDbGFzczogY2YuUHJpY2VDbGFzcy5QUklDRV9DTEFTU18xMDAsXG4gICAgICBsb2dCdWNrZXQ6IGJ1Y2tldExvZ3MsXG4gICAgICBsb2dGaWxlUHJlZml4OiBwcm9wcy5kb21haW5OYW1lRWRnZVxuICAgICAgICA/IGAke3JldmVyc2VEb21haW4ocHJvcHMuZG9tYWluTmFtZUVkZ2UpfS9jbG91ZGZyb250LXJhdy9gXG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgIH0pO1xuICAgIGlmIChyZW1vdmFsUG9saWN5ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRoaXMuX2Nsb3VkRnJvbnREaXN0cm8uYXBwbHlSZW1vdmFsUG9saWN5KHJlbW92YWxQb2xpY3kpO1xuICAgIH1cblxuICAgIC8vIEFkZCByb3V0ZXMgdG8gdGhlIENsb3VkRnJvbnQgRGlzdHJpYnV0aW9uXG4gICAgTWljcm9BcHBzQ0YuYWRkUm91dGVzKHNjb3BlLCB7XG4gICAgICBhcGlHd3lPcmlnaW4sXG4gICAgICBidWNrZXRBcHBzT3JpZ2luLFxuICAgICAgZGlzdHJvOiB0aGlzLl9jbG91ZEZyb250RGlzdHJvLFxuICAgICAgYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeTogYXBpZ3d5T3JpZ2luUmVxdWVzdFBvbGljeSxcbiAgICAgIHJvb3RQYXRoUHJlZml4LFxuICAgICAgY3JlYXRlQVBJUGF0aFJvdXRlLFxuICAgIH0pO1xuXG4gICAgLy9cbiAgICAvLyBDcmVhdGUgdGhlIGVkZ2UgbmFtZSBmb3IgdGhlIENsb3VkRnJvbnQgZGlzdHJvXG4gICAgLy9cblxuICAgIGlmIChyNTNab25lICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIGNvbnN0IHJyQXBwc0VkZ2UgPSBuZXcgcjUzLlJlY29yZFNldCh0aGlzLCAnZWRnZS1hcmVjb3JkJywge1xuICAgICAgICByZWNvcmROYW1lOiBkb21haW5OYW1lRWRnZSxcbiAgICAgICAgcmVjb3JkVHlwZTogcjUzLlJlY29yZFR5cGUuQSxcbiAgICAgICAgdGFyZ2V0OiByNTMuUmVjb3JkVGFyZ2V0LmZyb21BbGlhcyhuZXcgcjUzdGFyZ2V0cy5DbG91ZEZyb250VGFyZ2V0KHRoaXMuX2Nsb3VkRnJvbnREaXN0cm8pKSxcbiAgICAgICAgem9uZTogcjUzWm9uZSxcbiAgICAgIH0pO1xuICAgICAgaWYgKHJlbW92YWxQb2xpY3kgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICByckFwcHNFZGdlLmFwcGx5UmVtb3ZhbFBvbGljeShyZW1vdmFsUG9saWN5KTtcbiAgICAgIH1cbiAgICB9XG4gIH1cbn1cbiJdfQ==