"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsAPIGwy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const logs = require("aws-cdk-lib/aws-logs");
const r53 = require("aws-cdk-lib/aws-route53");
const r53targets = require("aws-cdk-lib/aws-route53-targets");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps API Gateway HTTP API endpoint.
 */
class MicroAppsAPIGwy extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        if ((props.r53Zone === undefined && props.domainNameEdge !== undefined) ||
            (props.r53Zone !== undefined && props.domainNameEdge === undefined)) {
            throw new Error('If either of r53Zone or domainNameEdge are set then the other must be set');
        }
        if (props.domainNameOrigin !== undefined && props.certOrigin === undefined) {
            throw new Error('If domainNameOrigin is set then certOrigin must be set');
        }
        if ((props.domainNameEdge === undefined && props.certOrigin !== undefined) ||
            (props.domainNameEdge !== undefined && props.certOrigin === undefined)) {
            throw new Error('If either of domainNameEdge or certOrigin are set then the other must be set');
        }
        const { r53Zone, domainNameEdge, domainNameOrigin, certOrigin, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix, } = props;
        // API Gateway uses the `id` string as the gateway name without
        // any randomization... we have to make sure the name is unique-ish
        const apigatewayName = assetNameRoot
            ? `${assetNameRoot}${assetNameSuffix}`
            : `${aws_cdk_lib_1.Stack.of(this).stackName}-microapps`;
        //
        // APIGateway domain names for CloudFront and origin
        //
        this._httpApi = new apigwy.HttpApi(this, 'gwy', {
            apiName: apigatewayName,
            createDefaultStage: false,
        });
        if (removalPolicy !== undefined) {
            this._httpApi.applyRemovalPolicy(removalPolicy);
        }
        // Create the stage
        const stage = new apigwy.HttpStage(this, 'stage', {
            httpApi: this._httpApi,
            autoDeploy: true,
            // If rootPathPrefix is not defined this will be the $default stage
            stageName: rootPathPrefix,
        });
        if (domainNameEdge !== undefined && certOrigin !== undefined) {
            // Create Custom Domains for API Gateway
            const dnAppsEdge = new apigwy.DomainName(this, 'microapps-apps-edge-dn', {
                domainName: domainNameEdge,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                dnAppsEdge.applyRemovalPolicy(removalPolicy);
            }
            // Create the edge domain name mapping for the API
            const apiMapping = new apigwy.ApiMapping(this, 'mapping', {
                api: this._httpApi,
                domainName: dnAppsEdge,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            apiMapping.node.addDependency(dnAppsEdge);
        }
        if (domainNameOrigin !== undefined && certOrigin !== undefined) {
            this._dnAppsOrigin = new apigwy.DomainName(this, 'origin-dn', {
                domainName: domainNameOrigin,
                certificate: certOrigin,
            });
            if (removalPolicy !== undefined) {
                this._dnAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
        // Enable access logs on API Gateway
        const apiAccessLogs = new logs.LogGroup(this, 'logs', {
            logGroupName: apigatewayName
                ? `/aws/apigwy/${apigatewayName}`
                : `/aws/apigwy/${this.httpApi.httpApiName}`,
            retention: logs.RetentionDays.TWO_WEEKS,
        });
        if (removalPolicy !== undefined) {
            apiAccessLogs.applyRemovalPolicy(removalPolicy);
        }
        const cfnStage = stage.node.defaultChild;
        cfnStage.accessLogSettings = {
            destinationArn: apiAccessLogs.logGroupArn,
            format: JSON.stringify({
                requestId: '$context.requestId',
                userAgent: '$context.identity.userAgent',
                sourceIp: '$context.identity.sourceIp',
                requestTime: '$context.requestTime',
                requestTimeEpoch: '$context.requestTimeEpoch',
                httpMethod: '$context.httpMethod',
                path: '$context.path',
                status: '$context.status',
                protocol: '$context.protocol',
                responseLength: '$context.responseLength',
                domainName: '$context.domainName',
                authorizerError: '$context.authorizer.error',
            }),
        };
        // Create a logging role
        // Tips: https://github.com/aws/aws-cdk/issues/11100
        const apiGwyLogRole = new iam.Role(this, 'logs-role', {
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonAPIGatewayPushToCloudWatchLogs'),
            ],
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        });
        apiAccessLogs.grantWrite(apiGwyLogRole);
        //
        // Let API Gateway accept requests using domainNameOrigin
        // That is the origin URI that CloudFront uses for this gateway.
        // The gateway will refuse the traffic if it doesn't have the
        // domain name registered.
        //
        if (this._dnAppsOrigin !== undefined) {
            const mappingAppsApis = new apigwy.ApiMapping(this, 'api-map-origin', {
                api: this._httpApi,
                domainName: this._dnAppsOrigin,
                stage,
            });
            // 2022-01-16 - CDK is still generating CloudFormation with no dependency
            // between the R53 RecordSet and the Mapping
            mappingAppsApis.node.addDependency(this._dnAppsOrigin);
            if (removalPolicy !== undefined) {
                mappingAppsApis.applyRemovalPolicy(removalPolicy);
            }
        }
        //
        // Create the origin name for API Gateway
        //
        if (r53Zone !== undefined && this._dnAppsOrigin) {
            const rrAppsOrigin = new r53.ARecord(this, 'origin-arecord', {
                zone: r53Zone,
                recordName: domainNameOrigin,
                target: r53.RecordTarget.fromAlias(new r53targets.ApiGatewayv2DomainProperties(this._dnAppsOrigin.regionalDomainName, this._dnAppsOrigin.regionalHostedZoneId)),
            });
            if (removalPolicy !== undefined) {
                rrAppsOrigin.applyRemovalPolicy(removalPolicy);
            }
        }
    }
    get dnAppsOrigin() {
        return this._dnAppsOrigin;
    }
    get httpApi() {
        return this._httpApi;
    }
}
exports.MicroAppsAPIGwy = MicroAppsAPIGwy;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsAPIGwy[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsAPIGwy", version: "0.2.10" };
//# sourceMappingURL=data:application/json;base64,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