"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyint = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
class MicroAppsSvcs extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        this._table = new dynamodb.Table(this, 'table', {
            tableName: assetNameRoot ? `${assetNameRoot}${assetNameSuffix}` : undefined,
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            partitionKey: {
                name: 'PK',
                type: dynamodb.AttributeType.STRING,
            },
            sortKey: {
                name: 'SK',
                type: dynamodb.AttributeType.STRING,
            },
            removalPolicy,
        });
        //
        // Router Lambda Function
        //
        // Create Router Lambda Function
        const routerFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                DATABASE_TABLE_NAME: this._table.tableName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
            // This is for local dev
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else {
            // Create Router Lambda Layer
            const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                removalPolicy,
            });
            this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                layers: [routerDataFiles],
                ...routerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._routerFunc.applyRemovalPolicy(removalPolicy);
        }
        const policyReadTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [`${bucketApps.bucketArn}/*`],
        });
        for (const router of [this._routerFunc]) {
            router.addToRolePolicy(policyReadTarget);
            // Give the Router access to DynamoDB table
            this._table.grantReadData(router);
            this._table.grant(router, 'dynamodb:DescribeTable');
        }
        // Create alias for Router
        const routerAlias = this._routerFunc.addAlias('CurrentVersion');
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                APIGWY_ID: httpApi.httpApiId,
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        //
        // Deloyer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        // Grant full control over the API we created
        const policyAPIManage = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:*'],
            resources: [
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
            ],
        });
        this._deployerFunc.addToRolePolicy(policyAPIManage);
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        // This creates an integration and a router
        const route = new apigwy.HttpRoute(this, 'route-default', {
            httpApi,
            routeKey: apigwy.HttpRouteKey.DEFAULT,
            integration: new apigwyint.HttpLambdaIntegration('router-integration', routerAlias),
        });
        let routeArn = route.routeArn;
        // Remove the trailing `/` on the ARN, which is not correct
        if (routeArn.endsWith('/')) {
            routeArn = routeArn.slice(0, routeArn.length - 1);
        }
        // Grant API Gateway permission to invoke the Lambda
        new lambda.CfnPermission(this, 'router-invoke', {
            action: 'lambda:InvokeFunction',
            functionName: this._routerFunc.functionName,
            principal: 'apigateway.amazonaws.com',
            sourceArn: routeArn,
        });
    }
    get table() {
        return this._table;
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.2.10" };
//# sourceMappingURL=data:application/json;base64,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