"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomSynthesizer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const utils_1 = require("./utils");
/**
 * As a best practice organizations enforce policies which require all custom IAM Roles created to be defined under
 * a specific path and permission boundary.
 * In order to adhere with such compliance requirements, the CDK bootstrapping is often customized
 * (refer: https://docs.aws.amazon.com/cdk/v2/guide/bootstrapping.html#bootstrapping-customizing).
 * So, we need to ensure that parallel customization is applied during synthesis phase.
 * This Custom Synthesizer is used to modify the default path of the following IAM Roles internally used by CDK:
 *  - deploy role
 *  - file-publishing-role
 *  - image-publishing-role
 *  - cfn-exec-role
 *  - lookup-role
 * @see PermissionsBoundaryAspect
 *
 * Example Usage:
 * ```ts
 *    new DbStack(app, config.id('apiDbStack'), {
 *      env: {account: '123456789012', region: 'us-east-1'},
 *      synthesizer: new CustomSynthesizer('/banking/dev/'),
 *     });
 * ```
 */
class CustomSynthesizer extends aws_cdk_lib_1.DefaultStackSynthesizer {
    constructor(rolePath) {
        super({
            deployRoleArn: CustomSynthesizer.qualifiedRole('deploy-role', rolePath),
            fileAssetPublishingRoleArn: CustomSynthesizer.qualifiedRole('file-publishing-role', rolePath),
            imageAssetPublishingRoleArn: CustomSynthesizer.qualifiedRole('image-publishing-role', rolePath),
            cloudFormationExecutionRole: CustomSynthesizer.qualifiedRole('cfn-exec-role', rolePath),
            lookupRoleArn: CustomSynthesizer.qualifiedRole('lookup-role', rolePath),
        });
    }
    static qualifiedRole(roleName, rolePath) {
        return 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role' + utils_1.Utils.wrap(rolePath, '/') + 'cdk-${Qualifier}-' + roleName + '-${AWS::AccountId}-${AWS::Region}';
    }
}
exports.CustomSynthesizer = CustomSynthesizer;
_a = JSII_RTTI_SYMBOL_1;
CustomSynthesizer[_a] = { fqn: "ez-constructs.CustomSynthesizer", version: "0.0.23" };
//# sourceMappingURL=data:application/json;base64,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