"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SimpleCodebuildProject = exports.GitEvent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const ez_construct_1 = require("../ez-construct");
const utils_1 = require("../lib/utils");
const secure_bucket_1 = require("../secure-bucket");
/**
 * The Github events which should trigger this build.
 */
var GitEvent;
(function (GitEvent) {
    GitEvent["PULL_REQUEST"] = "pull_request";
    GitEvent["PUSH"] = "push";
    GitEvent["ALL"] = "all";
})(GitEvent = exports.GitEvent || (exports.GitEvent = {}));
/**
 * Most of the cases,a developer will use CodeBuild setup to perform simple CI tasks such as:
 * - Build and test your code on a PR
 * - Run a specific script based on a cron schedule.
 * Also, they might want:
 * - artifacts like testcase reports to be available via Reports UI and/or S3.
 * - logs to be available via CloudWatch Logs.
 *
 * However, there can be additional organizational retention policies, for example retaining logs for a particular period of time.
 * With this construct, you can easily create a basic CodeBuild project with many opinated defaults that are compliant with FISMA and NIST.
 *
 * Example, creates a project named `my-project`, with artifacts going to my-project-artifacts-<accountId>-<region>
 *  and logs going to `/aws/codebuild/my-project` log group with a retention period of 90 days and 14 months respectively.
 *
 * ```ts
 *    new SimpleCodebuildProject(stack, 'MyProject')
 *      .projectName('myproject')
 *      .gitRepoUrl('https://github.com/bijujoseph/cloudbiolinux.git')
 *      .gitBaseBranch('main')
 *      .triggerEvent(GitEvent.PULL_REQUEST)
 *      .buildSpecPath('buildspecs/my-pr-checker.yml')
 *      .assemble();
 * ```
 *
 */
class SimpleCodebuildProject extends ez_construct_1.EzConstruct {
    constructor(scope, id) {
        super(scope, id);
        this._gitBaseBranch = 'develop';
        this._grantReportGroupPermissions = true;
        this._privileged = false;
        this._triggerOnPushToBranches = [];
        this._computType = aws_codebuild_1.ComputeType.MEDIUM;
        this._buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_5_0;
        this._envVariables = {};
        this.scope = scope;
        this.id = id;
    }
    /**
     * The underlying codebuild project that is created by this construct.
     */
    get project() {
        return this._project;
    }
    /**
     * A convenient way to set the project environment variables.
     * The values set here will be presnted on the UI when build with overriding is used.
     * @param name - The environment variable name
     * @param envVar - The environment variable value
     *Example:
     *
     * ```ts
     *  project
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'abcd})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: '/dev/thatkey, type: BuildEnvironmentVariableType.PARAMETER_STORE})
     *    .addEnvironmentVariable('MY_ENV_VAR', {value: 'arn:of:secret, type: BuildEnvironmentVariableType.SECRETS_MANAGER});
     * ```
     */
    addEnv(name, envVar) {
        this._envVariables[name] = envVar;
        return this;
    }
    /**
     * The name of the codebuild project
     * @param projectName - a valid name string
     */
    projectName(projectName) {
        this._projectName = projectName;
        return this;
    }
    /**
     * The description of the codebuild project
     * @param projectDescription - a valid description string
     */
    projectDescription(projectDescription) {
        this._projectDescription = projectDescription;
        return this;
    }
    /**
     * The compute type to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-compute-types.html
     * @param computeType
     */
    computeType(computeType) {
        this._computType = computeType;
        return this;
    }
    /**
     * The build image to use
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     * @see https://docs.aws.amazon.com/cdk/api/v1/docs/@aws-cdk_aws-codebuild.IBuildImage.html
     * @param buildImage
     */
    buildImage(buildImage) {
        this._buildImage = buildImage;
        return this;
    }
    /**
     * The build image to use
     * @param ecrRepoName - the ecr repository name
     * @param imageTag - the image tag
     */
    ecrBuildImage(ecrRepoName, imageTag) {
        let repo = aws_ecr_1.Repository.fromRepositoryName(this.scope, 'ClaimsCodeBuildBaseImageRepository', ecrRepoName);
        return this.buildImage(aws_codebuild_1.LinuxBuildImage.fromEcrRepository(repo, imageTag));
    }
    /**
     * The vpc network interfaces to add to the codebuild
     * @see https://docs.aws.amazon.com/cdk/api/v1/docs/aws-codebuild-readme.html#definition-of-vpc-configuration-in-codebuild-project
     * @param vpcId
     */
    inVpc(vpcId) {
        this._vpcId = vpcId;
        return this;
    }
    /**
     * Set privileged mode of execution. Usually needed if this project builds Docker images,
     * and the build environment image you chose is not provided by CodeBuild with Docker support.
     * By default, Docker containers do not allow access to any devices.
     * Privileged mode grants a build project's Docker container access to all devices
     * https://docs.aws.amazon.com/codebuild/latest/userguide/change-project-console.html#change-project-console-environment
     * @param p - true/false
     */
    privileged(p) {
        this._privileged = p;
        return this;
    }
    /**
     * The build spec file path
     * @param buildSpecPath - relative location of the build spec file
     */
    buildSpecPath(buildSpecPath) {
        this._buildSpecPath = buildSpecPath;
        return this;
    }
    /**
     * The github or enterprise github repository url
     * @param gitRepoUrl
     */
    gitRepoUrl(gitRepoUrl) {
        this._gitRepoUrl = gitRepoUrl;
        return this;
    }
    /**
     * The main branch of the github project.
     * @param branch
     */
    gitBaseBranch(branch) {
        this._gitBaseBranch = branch;
        return this;
    }
    /**
     * Triggers build on push to specified branches
     * @param branches
     */
    triggerOnPushToBranches(branches) {
        this._triggerOnPushToBranches.push(...branches);
        return this;
    }
    /**
     * The Github events that can trigger this build.
     * @param event
     */
    triggerBuildOnGitEvent(event) {
        this._triggerOnGitEvent = event;
        return this;
    }
    /**
     * The cron schedule on which this build gets triggerd.
     * @param schedule
     */
    triggerBuildOnSchedule(schedule) {
        this._triggerOnSchedule = schedule;
        return this;
    }
    /**
     * The name of the bucket to store the artifacts.
     * By default the buckets will get stored in `<project-name>-artifacts` bucket.
     * This function can be used to ovrride the default behavior.
     * @param artifactBucket - a valid existing Bucket reference or bucket name to use.
     */
    artifactBucket(artifactBucket) {
        this._artifactBucket = artifactBucket;
        return this;
    }
    overrideProjectProps(props) {
        var _b;
        let projectName = this._projectName ? this._projectName : utils_1.Utils.kebabCase(this.id);
        let description = this._projectDescription ? this._projectDescription : `Codebuild description for ${projectName}`;
        let defaults = {
            projectName,
            description,
            grantReportGroupPermissions: this._grantReportGroupPermissions,
        };
        // set the default environment if not provided.
        if (utils_1.Utils.isEmpty(props.environment)) {
            // @ts-ignore
            defaults.environment = {
                buildImage: this._buildImage ? this._buildImage : aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                privileged: this._privileged,
                computeType: this._computType,
                environmentVariables: this._envVariables,
            };
        }
        // add default logging if not provided
        if (utils_1.Utils.isEmpty(props.logging)) {
            // @ts-ignore
            defaults.logging = {
                cloudWatch: {
                    logGroup: new aws_logs_1.LogGroup(this, 'ProjectLogGroup', {
                        logGroupName: `/aws/codebuild/${utils_1.Utils.kebabCase(projectName)}`,
                        retention: aws_logs_1.RetentionDays.THIRTEEN_MONTHS,
                    }),
                },
            };
        }
        // create source if not provided in props
        if (utils_1.Utils.isEmpty(props.source)) {
            // @ts-ignore
            defaults.source = this.createSource(this._gitRepoUrl, this._gitBaseBranch, this._triggerOnGitEvent, this._triggerOnPushToBranches);
        }
        // create artifact bucket if one was not provided
        if (utils_1.Utils.isEmpty(props.artifacts)) {
            // @ts-ignore
            defaults.artifacts = this.createArtifacts((_b = this._artifactBucket) !== null && _b !== void 0 ? _b : `${this._projectName}-artifacts`);
        }
        // create the build spec if one was not provided
        if (utils_1.Utils.isEmpty(props.buildSpec) && !utils_1.Utils.isEmpty(this._buildSpecPath)) {
            // @ts-ignore
            defaults.buildSpec = this.createBuildSpec(this._buildSpecPath);
        }
        // create vpc interface if needed
        if (!utils_1.Utils.isEmpty(this._vpcId)) {
            // @ts-ignore
            defaults.vpc = aws_ec2_1.Vpc.fromLookup(this, 'VPC', { vpcId: this._vpcId });
        }
        this._props = Object.assign({}, defaults, props);
        return this;
    }
    assemble(defaultProps) {
        // create the default project properties
        this.overrideProjectProps(defaultProps !== null && defaultProps !== void 0 ? defaultProps : {});
        // create a codebuild project
        let project = new aws_codebuild_1.Project(this.scope, 'Project', this._props);
        // run above project on a schedule ?
        if (this._triggerOnSchedule) {
            new aws_events_1.Rule(this.scope, 'ScheduleRule', {
                schedule: this._triggerOnSchedule,
                targets: [new aws_events_targets_1.CodeBuildProject(project)],
            });
        }
        this._project = project;
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-CB4', 'Artifacts produced by this project are encrypted using `aws/s3` key, and also at rest by S3.');
        utils_1.Utils.suppressNagRule(this._project, 'AwsSolutions-IAM5', 'There is an artifact bucket per project. The wild card IAM policies are scoped to this project and only used to manage artifacts produced by the project.');
        return this;
    }
    /**
     * Create an S3 bucket for storing artifacts produced by the codebuild project
     * @param bucketName - the s3 bucket
     * @private
     */
    createBucket(bucketName) {
        return new secure_bucket_1.SecureBucket(this, 'ProjectArtifactBucket')
            .bucketName(bucketName)
            .objectsExpireInDays(90)
            .assemble()
            .bucket;
    }
    /**
     * Creates an S3 artifact store for storing the objects produced by the codebuild project
     * @param artifactBucket - a bucket object or bucket name
     * @private
     */
    createArtifacts(artifactBucket) {
        let theBucket = (typeof artifactBucket === 'string') ? this.createBucket(artifactBucket) : artifactBucket;
        return aws_codebuild_1.Artifacts.s3({
            bucket: theBucket,
            includeBuildId: true,
            packageZip: true,
        });
    }
    /**
     * Creates a Github or Enterprise Githb repo source object
     * @param repoUrl - the url of the repo
     * @param base - the main or base branch used by the repo
     * @param gitEvent - the github events that can trigger builds
     * @private
     */
    createSource(repoUrl, base, gitEvent, branches) {
        let webhook = gitEvent && true;
        let repoDetails = utils_1.Utils.parseGithubUrl(repoUrl);
        let webhookFilter = this.createWebHookFilters(base, gitEvent, branches);
        if (repoDetails.enterprise == true) {
            return aws_codebuild_1.Source.gitHubEnterprise({
                httpsCloneUrl: repoUrl,
                webhook,
                webhookFilters: webhookFilter,
            });
        }
        return aws_codebuild_1.Source.gitHub({
            owner: repoDetails.owner,
            repo: repoDetails.repo,
            webhook,
            webhookFilters: webhookFilter,
        });
    }
    /**
     * Creates a webhook filter object
     * @param base - the base branch
     * @param gitEvent - the github event
     * @private
     */
    createWebHookFilters(base, gitEvent, branches) {
        // @ts-ignore
        let fg1 = null;
        let fgList = [];
        if (!gitEvent)
            return undefined;
        if (gitEvent == GitEvent.PULL_REQUEST) {
            fg1 = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED);
            if (base) {
                fg1 = fg1.andBaseBranchIs(base);
            }
        }
        if (gitEvent == GitEvent.PUSH) {
            fg1 = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH);
        }
        if (gitEvent == GitEvent.ALL) {
            fg1 = aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH, aws_codebuild_1.EventAction.PULL_REQUEST_CREATED, aws_codebuild_1.EventAction.PULL_REQUEST_UPDATED, aws_codebuild_1.EventAction.PULL_REQUEST_REOPENED, aws_codebuild_1.EventAction.PULL_REQUEST_MERGED);
        }
        fgList.push(fg1);
        branches === null || branches === void 0 ? void 0 : branches.forEach(branch => {
            fgList.push(aws_codebuild_1.FilterGroup.inEventOf(aws_codebuild_1.EventAction.PUSH).andHeadRefIs(branch));
        });
        return fgList;
    }
    /**
     * Loads the build spec associated with the given codebuild project
     * @param buildSpecPath - location of the build spec file.
     * @private
     */
    createBuildSpec(buildSpecPath) {
        return aws_codebuild_1.BuildSpec.fromSourceFilename(buildSpecPath);
    }
}
exports.SimpleCodebuildProject = SimpleCodebuildProject;
_a = JSII_RTTI_SYMBOL_1;
SimpleCodebuildProject[_a] = { fqn: "ez-constructs.SimpleCodebuildProject", version: "0.0.23" };
//# sourceMappingURL=data:application/json;base64,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