"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubActionsIdentityProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
/**
 * Github Actions as OpenID Connect Identity Provider for AWS IAM.
 * There can be only one (per AWS Account).
 *
 * Use `fromAccount` to retrieve a reference to existing Github OIDC provider.
 *
 * @see https://docs.github.com/en/actions/deployment/security-hardening-your-deployments/configuring-openid-connect-in-amazon-web-services
 */
class GithubActionsIdentityProvider extends iam.OpenIdConnectProvider {
    /**
     * Define a new Github OpenID Connect Identity PRovider for AWS IAM.
     * There can be only one (per AWS Account).
     *
     * @param scope CDK Stack or Construct to which the provider is assigned to
     * @param id CDK Construct ID given to the construct
     *
     * @example
     * new GithubActionsIdentityProvider(scope, "GithubProvider");
     */
    constructor(scope, id) {
        super(scope, id, {
            url: `https://${GithubActionsIdentityProvider.issuer}`,
            thumbprints: GithubActionsIdentityProvider.thumbprints,
            clientIds: ['sts.amazonaws.com'],
        });
    }
    /**
     * Retrieve a reference to existing Github OIDC provider in your AWS account.
     * An AWS account can only have single Github OIDC provider configured into it,
     * so internally the reference is made by constructing the ARN from AWS
     * Account ID & Github issuer URL.
     *
     * @param scope CDK Stack or Construct to which the provider is assigned to
     * @param id CDK Construct ID given to the construct
     * @returns a CDK Construct representing the Github OIDC provider
     *
     * @example
     * GithubActionsIdentityProvider.fromAccount(scope, "GithubProvider");
     */
    static fromAccount(scope, id) {
        const accountId = cdk.Stack.of(scope).account;
        const providerArn = `arn:aws:iam::${accountId}:oidc-provider/${GithubActionsIdentityProvider.issuer}`;
        return iam.OpenIdConnectProvider.fromOpenIdConnectProviderArn(scope, id, providerArn);
    }
}
exports.GithubActionsIdentityProvider = GithubActionsIdentityProvider;
_a = JSII_RTTI_SYMBOL_1;
GithubActionsIdentityProvider[_a] = { fqn: "aws-cdk-github-oidc.GithubActionsIdentityProvider", version: "2.2.1" };
GithubActionsIdentityProvider.issuer = 'token.actions.githubusercontent.com';
GithubActionsIdentityProvider.thumbprints = [
    'a031c46782e6e6c662c2c87c76da9aa62ccabd8e',
    '6938fd4d98bab03faadb97b34396831e3780aea1',
];
//# sourceMappingURL=data:application/json;base64,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