# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['bas_style_kit_jinja_templates']

package_data = \
{'': ['*'],
 'bas_style_kit_jinja_templates': ['templates/includes/*',
                                   'templates/layouts/*',
                                   'templates/macros/*',
                                   'templates/patterns/*']}

install_requires = \
['Jinja2>=2.10.1,<3.0.0']

setup_kwargs = {
    'name': 'bas-style-kit-jinja-templates',
    'version': '0.5.0',
    'description': 'A set of Jinja2 templates implementing the BAS Style Kit',
    'long_description': '# BAS Style Kit Jinja Templates\n\nA set of [Jinja2](http://jinja.pocoo.org) templates implementing the [BAS Style Kit](https://style-kit.web.bas.ac.uk).\n\n## Installation\n\n### Pip package\n\nThe recommended way to install these templates is via its PyPi package,\n[`bas-style-kit-jekyll-templates`](https://pypi.org/project/bas-style-kit-jinja-templates/).\n\n## Usage\n\n### Quickstart\n\nFor a typical Flask application add this wherever your Flask application is defined:\n\n```python\nfrom flask import Flask, render_template\nfrom jinja2 import PrefixLoader, PackageLoader, FileSystemLoader\n\nfrom bas_style_kit_jinja_templates import BskTemplates\n\napp = Flask(__name__)\napp.jinja_loader = PrefixLoader({\n    \'app\': FileSystemLoader(\'templates\'),\n    \'bas_style_kit\': PackageLoader(\'bas_style_kit_jinja_templates\'),\n})\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\n\n# Required/recommended settings\napp.config[\'BSK_TEMPLATES\'].site_title = \'Example service\'\napp.config[\'BSK_TEMPLATES\'].site_description = \'Service to act as an example\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_brand_text = \'Example service\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_development_phase = \'beta\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_feedback_href = \'/feedback\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_footer_policies_cookies_href = \'/legal/cookies\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_footer_policies_copyright_href = \'/legal/copyright\'\napp.config[\'BSK_TEMPLATES\'].bsk_site_footer_policies_privacy_href = \'/legal/privacy\'\n\n# Optional - add a custom CSS file with a relative URL\napp.config[\'BSK_TEMPLATES\'].site_styles.append({\'href\': \'/css/app.css\'})\n# Optional - add a custom JS file with a SRI value\napp.config[\'BSK_TEMPLATES\'].site_scripts.append({\'href\': \'https://example.com/js/example.js\', \'integrity\': \'abc123\'})\n# Optional - enable Google Analytics\napp.config[\'BSK_TEMPLATES\'].site_analytics[\'id\'] = \'1234\'\n# Optional - choose between the `bsk-container` (used by default) and `bsk-container-fluid` layout container\napp.config[\'BSK_TEMPLATES\'].bsk_container_classes = [\'bsk-container\']\n# Optional - add navigation menu items\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_primary.append({\'value\': \'Item\', \'href\': \'#\'})\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_secondary.append({\n    \'value\': \'Dropdown\',\n    \'items\': [\n        {\'value\': \'Sub-item 1\', \'href\': \'#\', \'target\': \'_blank\'}\n    ]\n})\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_launcher.append({\'value\': \'Related service\', \'href\': \'https://example.com\'})\n\n\n@app.route(\'/\')\ndef index():\n    # noinspection PyUnresolvedReferences\n    return render_template(f"app/index.j2")\n```\n\nWhere `app/index.j2` is a view located in `templates/index.j2` which extends an application layout:\n\n```jinja2\n{% extends \'app/layouts/app.j2\' %}\n{% block main_content %}\n<p>Index view content...</p>\n{% endblock %}\n```\n\nThis layout in turn extends a layout provided by this package:\n\n```jinja2\n{% extends \'bas_style_kit/layouts/bsk_standard.j2\' %}\n```\n\n### Using a page pattern\n\nTo create a page in an application based on a [Page pattern](#page-patterns), such as the\n[page not found](https://style-kit.web.bas.ac.uk/patterns/page-not-found/) pattern, create a template (e.g.\n`templates/errors/404.j2`) with the following:\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_page-not-found.j2\' %}\n```\n\nTo use this template as the 404 error handler in a Flask application for example:\n\n```python\n@app.errorhandler(404)\ndef page_not_found(e):\n    # note that we set the 404 status explicitly\n    return render_template(\'app/errors/404.j2\'), 404\n```\n\n### Using a component pattern\n\nTo use a [Component pattern](#component-pattern), such as the\n[ORCID iD](https://style-kit-testing.web.bas.ac.uk/patterns/orcid-id/) pattern, import and call the relevant Macro:\n\n```jinja2\n{% from "bas_style_kit/macros/bsk_pattern_orcid_id.j2" import pattern_orcid_id %}\n\n<p>{{ pattern_orcid_id(\'https://sandbox.orcid.org/0000-0001-8373-6934\') }}</p>\n```\n\n### Using custom CSS/JS\n\nNon-Style Kit CSS an/or JavaScript resources can be included either as references to files, or as inline content.\n\n**Note:** This won\'t work if you are using the [Blank layout](#layouts).\n\n* CSS resources are outputted in the [styles block](#blocks), at the end of the `<head>` element\n* JS resources are outputted in the [scripts block](#blocks), at the end of the `<body>` element\n\n#### Using custom CSS/JS files\n\n* CSS files are added as a resource object to the `site_styles` property of the `BskTemplates` class instance\n* JS files are added as a resource object to the `site_scripts` property of the `BskTemplates` class instance\n\nFiles will be included after the Style Kit\'s own resources (where a Style Kit layout is used) to ensure they have\npriority.\n\nEach file reference consists of an object with these properties:\n\n| Property    | Data Type | Required | Allowed Values | Example Value                                         |\n| ----------- | --------- | -------- | -------------- | ----------------------------------------------------- |\n| `href`      | String    | Yes      | Any URL        | */css/app.css* / *https://example.com/js/app.js*      |\n| `integrity` | String    | No       | Any SRI value  | *sha256-ClILH8AIH4CkAybtlKhzqqQUYR4eSDiNTK5LIWfF4qQ=* |\n\nFor example (using a Flask application with a `css/app.css` file with the default\n[static files route](http://flask.pocoo.org/docs/1.0/tutorial/static/)):\n\n```python\napp.config[\'bsk_templates\'] = BskTemplates()\napp.config[\'bsk_templates\'].site_styles.append({\'href\': \'/static/css/app.css\'})\n```\n\nThe `integrity` property is used to specify a\n[Subresource Integrity (SRI)](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) value for\na resource. If specified an `integrity` attribute and will be added to the generated markup. A `crossorigin`\nattribute will also be added for\n[Cross-Origin Resource Sharing (CORS)](https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS) support with a\nhard-coded, anonymous, value.\n\n#### Using custom CSS/JS inline content\n\n* CSS content should be appended to the [styles block](#blocks)\n* JS content should be appended to the [scripts block](#blocks)\n* inline content will be added after any files to ensure they have priority\n\n**Note:** To append to a block use `{{ super() }}`, rather than replacing the contents of a block.\n\nFor example (using a Jinja template):\n\n```jinja2\n{% block scripts %}\n    {{ super() }}\n    console.log(\'jQuery version: \' + jQuery.fn.jquery);\n{% endblock %}\n```\n\n### Navigation menu items\n\nWhen using the [bsk_standard layout](#layouts), a [navbar](https://style-kit.web.bas.ac.uk/components/navbar/) is\nincluded as part of the \'standard header\', which consists of a cookie banner, navbar and site development phase banner.\n\nThis navbar consists of three menus (and other elements, documented elsewhere):\n\n* a primary navigation menu - aligned left, after [brand elements](#navbar-branding)\n* a secondary navigation menu - aligned right, before the launcher menu\n* a navigation launcher menu - aligned right, after the secondary navigation menu\n\nThe navigation launcher is a restricted menu, used to link to other BAS websites and applications. By default it\ncontains links to the [BAS public website](https://www.bas.ac.uk) and the [BAS data catalogue](https://data.bas.ac.uk).\nOther websites and applications can be added as well where relevant.\n\n* primary navigation menu items should be added to the `BskTemplates.bsk_site_nav_primary` variable\n* secondary navigation menu items should be added to the `BskTemplates.bsk_site_nav_secondary` variable\n* navigation launcher menu items should be added to the `BskTemplates.bsk_site_nav_launcher` variable\n\nThe primary and secondary navigation menu\'s support:\n\n* [navbar items](https://style-kit.web.bas.ac.uk/components/navbar/#item)\n* [navbar drop-down menus](https://style-kit.web.bas.ac.uk/components/navbar/#drop-down-menus)\n* [navbar drop-down menu items](https://style-kit.web.bas.ac.uk/components/navbar/#drop-down-menus)\n\nThe navigation launcher menu, which is implemented as a drop-down menu, supports:\n\n* [navbar drop-down menu items](https://style-kit.web.bas.ac.uk/components/navbar/#drop-down-menus)\n\nMenu item objects have the following properties:\n\n| Property    | Data Type | Required | Allowed Values             | Example Value                              | Notes                                 |\n| ----------- | --------- | -------- | -------------------------- | ------------------------------------------ | ------------------------------------- |\n| `value`     | String    | Yes      | Any string                 | *About*                                    | -                                     |\n| `href`      | String    | Yes      | Any URL                    | */about* / *https://www.example.com/about* | Ignored if `items` set                |\n| `items`     | Array     | No       | Array of menu item objects | -                                          | Ignored for navigation launcher items |\n| *Any*       | String    | No       | As per attribute           | -                                          | Arbitrary attribute/value key/values  |\n\n**Note:** The `items` property is only recursed once, deeper objects will be ignored.\n\nFor example (using a Flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_primary.push({\'value\': \'Item\', \'href\': \'/about\'})\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_primary.push({\'value\': \'Another Item\', \'href\': \'#\', \'target\': \'_blank\'})\n```\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_secondary.push({\n    \'value\': \'Dropdown\',\n    \'items\': [\n        {\'value\': \'Sub-item 1\', \'href\': \'https://www.example.com\'}\n    ]\n})\n```\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_launcher.push({\'value\': \'Related service\', \'href\': \'https://example.com\'})\n```\n\n#### Active navigation items\n\nThese templates will automatically add the `.bsk-active` class to the relevant navigation item, and if relevant,\nsub-item, where its `href` attribute exactly matches the current URL given by `{{ request.path }}`.\n\nFor example for a navigation item `{\'value\': \'About\', \'href\': \'/about\'}`, when visiting `https://www.example/about`,\nthe about navigation item will be made active, as the current path `/about` matches the `href` attribute.\n\nThis support doesn\'t support URL patterns, such as `/foo/{id}` where `{id}` is a dynamic value. In these cases the\n`active_nav_item` variable can be set to the `href` value of a navigation item to make it active explicitly.\n\nFor example (using flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_primary.push({\'value\': \'Foo\', \'href\': \'/foo\'})\n\n@app.route(\'/foo/<foo_id>\')\ndef foo_details(foo_id: str):\n    foo = get_foo(foo_id)\n\n    return render_template(f"app/views/foo-details.j2", foo=foo, active_nav_item=\'/foo\')\n```\n\n### Navbar branding\n\nNavbars are used to display the name/identity of a website or application, to remind users where they are. These\nelements are referred to as \'brand\' elements within the Style Kit.\n\nSupported brand elements:\n\n* [brand text](https://style-kit.web.bas.ac.uk/components/navbar/#brand-text) - set using the\n`BskTemplates.bsk_site_nav_brand_text` property\n* [brand image](https://style-kit.web.bas.ac.uk/components/navbar/#brand-image) - set using the\n`BskTemplates.bsk_site_nav_brand_img_href` property\n\nBrand elements can be used together or individually, with fix classes applied automatically as needed.\n\nBrand elements are linked to a location specified by the `bsk_attributes.site_nav_brand_href` variable, which should be,\nand is by default, the index of each website or application (i.e. `/`).\n\nFor example (using a Flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_brand_text = \'Example service\'\n```\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_nav_brand_img_href = \'/assets/img/navbar-brand-img.png\'\n```\n\n### Site development phase\n\nThe site development phase indicates the stage of development for a website or application, e.g. alpha or live.\nConventional phases are described in the\n[Style Kit documentation](https://style-kit.web.bas.ac.uk/core/colours/#development-phase-colours).\n\nFor websites or applications that are not firmly in the \'live\' phase, a banner should be shown to inform users and\nrequest feedback. This forms part of the \'standard header\' of cookie banner, navbar and site development phase banner.\n\nIn these templates, the `BskTemplates.bsk_site_development_phase` property is used to specify the current phase for a\nwebsite or application. When using the [bsk_standard layout](#layouts) the banner will be shown automatically.\n\nTo disable this banner, use the `live-stable`. Strictly speaking this isn\'t a real phase but is recommended by these\ntemplates to distinguish between a newly released or mature and well-established website or application.\n\nFor example (using a Flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_development_phase = \'beta\'\n```\n\n#### Experimental site phase\n\nWhere a website or application is in a staging environment, or otherwise used for development/testing activities, the\nsite phase can be set to `experimental` to use the conventional\n[experimental phase](https://style-kit.web.bas.ac.uk/core/colours/#variants).\n\nFor example (using a Flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_development_phase = \'experimental\'\n```\n\n### Google analytics\n\nTo include the Google Analytics universal tracking library (gtag), set the `BskTemplates.bsk_site_analytics[\'id\']`\nproperty to relevant Google Analytics property ID.\n\n**Note:** When used the anonymise IP option in Google Analytics is enabled by default.\n\nFor example (using a Flask application):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\napp.config[\'BSK_TEMPLATES\'].bsk_site_analytics[\'id\'] = \'UA-12345678\'\n```\n\n### Footer content\n\nAdd custom footer content to the `footer_content` block. It will be shown between the\n[divider](https://style-kit.web.bas.ac.uk/components/footer/#divider) and\n[Governance](https://style-kit.web.bas.ac.uk/components/footer/#governance) footer components.\n\nIt is recommended to include a [spacer](https://style-kit.web.bas.ac.uk/components/footer/#spacer) component after any\nfooter content to balance the whitespace within the footer.\n\nFor example:\n\n```jinja2\n{% block footer_content %}\n<div>Footer content</div>\n<div role="separator" class="bsk-footer-spacer"></div>\n{% endblock %}\n```\n\nYou can also set custom classes on the footer element by appending to the `bsk_footer_classes` list, or replacing all\nclasses by overriding the list.\n\n### Patterns\n\n[Patterns](https://style-kit.web.bas.ac.uk/patterns/) demonstrate preferred ways to ask information from, or\nprovide information to, end users for various tasks.\n\nThere are two types of pattern used in the Style Kit and these templates:\n\n* [Pages](#page-patterns) - standalone pages designed to be used with or without customisation\n* [Components](#component-patterns) - inline elements designed to be used without customisation using [Macros](#macros)\n\n#### Page patterns\n\nPage patterns define content for common pages such as [*Page not found* (404) pages](#using-a-page-pattern).\n\nThese templates implement page patterns as layouts/views. Blocks can be used to provide required or additional\ninformation as needed.\n\n##### Page not found pattern\n\nFor example:\n\n(basic variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_page-not-found.j2\' %}\n```\n\n##### Problem with this service pattern\n\nBlocks:\n\n`pattern_content`\n: General content, including contact information.\n\nFor example:\n\n(basic variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_problem-with-service.j2\' %}\n\n{% block pattern_content %}\n<p>Contact the <a href="mailto:team@example.com">Example team</a> for more information.</p>\n{% endblock %}\n```\n\n##### Service unavailable pattern\n\nVariables:\n\n`availability`\n: Set to `replaced` for the [replaced](https://style-kit.web.bas.ac.uk/patterns/service-unavailable/#replaced) variant.\n  Set to `closed` for the [closed](https://style-kit.web.bas.ac.uk/patterns/service-unavailable/#closed) variant.\n\nBlocks:\n\n`pattern_content`\n: General content, including contact information.\n\nFor example:\n\n(basic variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_service-unavailable.j2\' %}\n\n{% block pattern_content %}\nContact the <a href="mailto:servicedesk.bas.ac.uk">BAS IT Service Desk</a> for more information.\n{% endblock %}\n```\n\n(closed variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_service-unavailable.j2\' %}\n\n{% set availability = \'closed\' %}\n\n{% block pattern_content %}\nContact the <a href="mailto:servicedesk.bas.ac.uk">BAS IT Service Desk</a> for more information.\n{% endblock %}\n```\n\n##### Start pattern\n\nVariables:\n\n`call_to_action_url`\n: Set to the href the call to action button should go to\n\n`call_to_action_variant`\n: Set to `default` for a standard \'start now\' call to action button\n  Set to `sign-in-microsoft` for a combined \'sign-in to continue\' and \'start now\' button\n\nBlocks:\n\n`pattern_content_uses`\n: Set to an unordered list of items for the \'use this service to:\' list\n\n`pattern_content`\n: General content, including \'more information\' for contact information and \'before you begin\' section if needed\n\nFor example:\n\n(basic variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_start.j2\' %}\n\n{% set call_to_action_url = \'#\' %}\n\n{% block pattern_content_uses %}\n<ul>\n    <li>A task</li>\n    <li>Another task</li>\n</ul>\n{% endblock %}\n\n{% block pattern_content %}\n    <section class="bsk-before-you-start">\n        <h2 class="bsk-h3">Before you start</h2>\n        <p>Before you start information</p>\n    </section>\n    <section class="bsk-more-information">\n        <h2 class="bsk-h3">More information</h2>\n        <p>Contact the <a href="mailto:servicedesk.bas.ac.uk">BAS IT Service Desk</a> for more information.</p>\n    </section>\n{% endblock %}\n```\n\n(more information variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_start.j2\' %}\n\n{% set call_to_action_url = \'#\' %}\n{% set call_to_action_variant = \'sign-in-microsoft\' %}\n\n{% block pattern_content_uses %}\n    <ul>\n        <li>A task</li>\n        <li>Another task</li>\n    </ul>\n{% endblock %}\n\n{% block pattern_content %}\n    <section class="bsk-before-you-start">\n        <h2 class="bsk-h3">Before you start</h2>\n        <p>Before you start information</p>\n    </section>\n    <section class="bsk-more-information">\n        <h2 class="bsk-h3">More information</h2>\n        <p>Contact the <a href="mailto:servicedesk.bas.ac.uk">BAS IT Service Desk</a> for more information.</p>\n        <p>More information</p>\n    </section>\n{% endblock %}\n```\n\n##### Sign-in (Microsoft) pattern\n\nVariables:\n\n`call_to_action_url`\n: Set to the href the call to action button should go to\n\nBlocks:\n\n`pattern_content`\n: Additional, optional, content if needed\n\nFor example:\n\n(basic variant)\n\n```jinja2\n{% extends \'bas_style_kit/patterns/bsk_sign-in-microsoft.j2\' %}\n\n{% set call_to_action_url = \'#\' %}\n```\n\n#### Component patterns\n\nThese templates include macros for all component patterns. Macro parameters are used for customising each instance of\nthe component.\n\n**Note:** Macro parameters are positional, meaning you need to ensure values are provided in the right order to work.\nSee the [Style Kit documentation](https://style-kit.web.bas.ac.uk/) for general information on using these patterns.\n\n##### Item type header pattern\n\nParameters:\n\n`item_type`\n: The type or kind of thing the item is, e.g. if the item is a person, it\'s type is \'person\'.\n\n`item_title`\n: A label specific to the item, e.g. if the item is a person their name.\n\n```jinja2\n{{ pattern_item_type_header(\'Item type\', \'Item title\') }}\n```\n\nFor example:\n\n```jinja2\n{{ pattern_item_type_header(\'Person\', \'Connie Watson\') }}\n```\n\n##### ORCID ID pattern\n\nParameters:\n\n`orcid_id`\n: The ORCID iD of an individual as a URL\n\n```jinja2\n{{ pattern_orcid_id(\'orcid_id\') }}\n```\n\nFor example:\n\n```jinja2\n{{ pattern_orcid_id(\'https://sandbox.orcid.org/0000-0001-8373-6934\') }}\n```\n\n## Components\n\nAll components are located in the `bas_style_kit_jinja_templates` package. Variables are\ndefined in `bas_style_kit_jinja_templates/__init__.py`, with all other components defined in\n`bas_style_kit_jinja_templates//templates/`.\n\nComponents that are specific to the Style Kit are prefixed with `bsk--` or `bsk_`.\n\n### Jinja setup\n\nThese templates require a *PrefixLoader* and *PackageLoader*\n[Jinja loader](http://jinja.pocoo.org/docs/2.10/api/#loaders) to be loaded into an application:\n\n```python\nloader = PrefixLoader({\n    \'bas_style_kit\': PackageLoader(\'bas-style-kit-jekyll-templates\')\n})\n```\n\nTo set [Variable](#variables) values used in these templates, a `BskTemplates` class instance is needed. These templates\nassume this instance will be available in Jinja\'s environment as `config.BSK_TEMPLATES`\n\nFor Flask applications this will occur automatically by adding the class instance to the\n[Flask config object](https://flask.palletsprojects.com/en/1.1.x/config/):\n\n```python\napp.config[\'BSK_TEMPLATES\'] = BskTemplates()\n```\n\nTypically applications have their own templates too, which can be loaded under a different prefix (such as `app`) using\na relevant loader, such as the default *FileSystemloader*.\n\n```python\nloader = PrefixLoader({\n    \'app\': FileSystemLoader(\'templates\'),\n    \'bas_style_kit\': PackageLoader(\'bas-style-kit-jekyll-templates\')\n})\n```\n\nThe use of a *PrefixLoader* means references to resources should include a prefix and a deliminator (`/` by default).\n\nFor example calling an application layout would change from:\n\n```jinja2\n{% extends "layouts/base.j2" %}\n```\n\nTo:\n\n```jinja2\n{% extends "app/layouts/base.j2" %}\n```\n\nOr to use a layout from these templates:\n\n```jinja2\n{% extends "bas_style_kit/layouts/bsk_base.j2" %}\n```\n\n### Layouts\n\nLayouts are \'base\' templates from which views or other layouts inherit. Layouts in these templates are hierarchical,\nwith each layout extending the last in this order:\n\n1. `blank.j2`: lowest level layout, intentionally as minimal as possible and not intended for direct use, unless\n    non-HTML output is needed\n2. `html.j2`: defines a minimal, accessible, HTML5 structure with some recommended best practices for cross-platform\n    compatibility\n3. `bsk_base.j2`: intentionally implements the BAS Style Kit as minimally as possible and not intended for direct use,\n    unless the `bsk_standard.j2` layout is unsuitable\n4. `bsk_standard.j2`: defines an opinionated, conventional, page layout with a \'standard\' header/footer, recommended as\n    a base for application/website layouts\n\nLayouts can be used using the `extend` keyword and defining content in the relevant block:\n\n| Layout            | Content Block  |\n| ----------------- | -------------- |\n| `blank.j2`        | `content`      |\n| `html.j2`         | `main_content` |\n| `bsk_base.j2`     | `main_content` |\n| `bsk_standard.j2` | `main_content` |\n\nFor example:\n\n```jinja2\n{% extends \'bas_style_kit/layouts/bsk_standard.j2\' %}\n\n{% block main_content %}\nLayout content\n{% endblock %}\n```\n\n### Blocks\n\n[Blocks](http://jinja.pocoo.org/docs/2.10/templates/#template-inheritance) are used for template inheritance and provide\na logical structure/hierarchy.\n\nBlocks are defined in [Layouts](#layouts), typically with default content using [Includes](#includes). Some blocks are\nempty, designed for user content or extensibility.\n\nTo implement or override a block, redefine it in a template or view:\n\n```jinja2\n{% block example_block %}\ncontent ...\n{% endblock %}\n```\n\nTo append to a block, without overriding its existing content, use the special `{{ super() }}` variable:\n\n```jinja2\n{% block example_block %}\n{{ super() }}\ncontent ...\n{% endblock %}\n```\n\n### Includes\n\n[Includes](http://jinja.pocoo.org/docs/2.10/templates/#include) are used for organising content, to make management\neasier, and to allow common elements to be used in multiple places, typically in [Blocks](#blocks).\n\nFor example the content needed for [using Google Analytics](#google-analytics) is encapsulated in the\n`body--analytics-script.j2` include.\n\n### Macros\n\n[Macros](http://jinja.pocoo.org/docs/2.10/templates/#macros) are used to provide configurable, reusable, functionality.\n\nThey are used within other components, such as the [navigation menus](#navigation-menu-items) macro for processing\nprimary and secondary navigation menus the same way, and to implement [Component patterns](#component-patterns).\n\n### Variables\n\nVarious elements in these templates are configurable, such as the name of the application or website, or the CSS/JS\nresources to include. A Python class `BskTemplates` is used to configure these elements and which should be passed to\nthe [Jinja environment](#jinja-setup).\n\nThese variables should be changed or set for each website or application:\n\n`site_title`\n: Name of the application or website\n\n`site_description`\n: Description of the application or website\n\n`site_analytics.id`\n: Google Analytics property ID\n\n`bsk_site_nav_brand_text`\n: Name of the application or website\n\n`bsk_site_nav_primary`\n: [Primary navigation menu items](#navigation-menu-items)\n\n`bsk_site_development_phase`\n: [Site development phase](#site-development-phase)\n\n`bsk_site_feedback_href`\n: URL or `mailto:` value for application or website feedback\n\n`bsk_site_footer_policies_cookies_href`\n: URL to application or website cookies policy\n\n`bsk_site_footer_policies_copyright_href`\n: URL to application or website copyright notice\n\n`bsk_site_footer_policies_privacy_href`\n: URL to application or website privacy policy\n\nThese variables may, but don\'t need to be, changed or set for each website or application:\n\n`site_styles`\n: Array of additional CSS files\n\n`site_scripts`\n: Array of additional JS files\n\n`container_classes`\n: Array of non-Style Kit classes which set the layout of content, including main content and headers/footers which\n  should align the same way\n\n`main_content_classes`\n: Array of non-Style Kit classes which should only be applied to main page content\n\n`bsk_container_classes`\n: Array of Style Kit classes which should set the layout of content, including main content and headers/footers which\n  should align the same way\n\n`bsk_main_content_classes`\n: Array of Style Kit classes which should only be applied to main page content\n\n`bsk_footer_classes`\n: Array of Style Kit classes which should be applied to the standard footer element\n\n`bsk_site_favicon`\n: Name of the [Favicon](https://style-kit.web.bas.ac.uk/core/favicon) to include (valid options: [`default`])\n\n`bsk_site_nav_secondary`\n: [Secondary navigation menu items](#navigation-menu-items)\n\n`bsk_site_nav_brand_img_href`\n: URL to [Navbar brand image](#navbar-branding)\n\n`bsk_site_nav_brand_href`\n: URL for [Navbar brand elements](#navbar-branding), which should be the index or home of the application or website\n\n`bsk_site_nav_launcher`\n: [Navigation launcher items](#navigation-menu-items)\n\nThese variables do not normally, and should not, need to be changed or set:\n\n`site_back_to_top_target_id`\n: ID of the anchor element representing the top of the current page/view\n\n`site_main_content_target_id`\n: ID of the element representing the main content in a page/view (i.e. skipping navigation elements)\n\n`bsk_site_footer_ogl_symbol_a_href`\n: URL to the Open Government Licence symbol\n\n`bsk_site_footer_ogl_text_href`\n: URL to the Open Government Licence text (i.e. the actual licence)\n\n`bsk_site_footer_ogl_text_version`\n: Version of the Open Government Licence used\n\nThese variables must not be changed and should be treated as read only:\n\n`templates_version`\n: Version of these templates\n\n`bsk_version`\n: Version of the Style Kit used by these templates\n\n\n## Development\n\nA local Flask application defined in `app.py` is used to develop this project.\n\nEnsure templates and configuration options for use in this project are defined within the\n`bas_style_kit_jinja_templates` package.\n\n### Development environment\n\nThe `:latest` container image is used for developing this project. It can run locally using Docker and Docker Compose:\n\n```\n$ docker login docker-registry.data.bas.ac.uk\n$ docker-compose pull app\n```\n\nTo run/test application commands:\n\n```\n$ docker-compose up\n```\n\n### Code Style\n\nPEP-8 style and formatting guidelines must be used for this project, with the exception of the 80 character line limit.\n\n[Black](https://github.com/psf/black) is used to ensure compliance, configured in `pyproject.toml`.\n\nBlack can be [integrated](https://black.readthedocs.io/en/stable/editor_integration.html#pycharm-intellij-idea) with a\nrange of editors, such as PyCharm, to perform formatting automatically.\n\nTo apply formatting manually:\n\n```\n$ docker-compose run app black bas_style_kit_jinja_templates/\n```\n\nTo check compliance manually:\n\n```\n$ docker-compose run app black --check bas_style_kit_jinja_templates\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Python version\n\nWhen upgrading to a new version of Python, ensure the following are also checked and updated where needed:\n\n* `Dockerfile`:\n    * base stage image (e.g. `FROM python:3.X-alpine` to `FROM python:3.Y-alpine`)\n* `pyproject.toml`:\n    * `[tool.poetry.dependencies]`\n        * `python` (e.g. `python = "^3.X"` to `python = "^3.Y"`)\n    * `[tool.black]`\n        * `target-version` (e.g. `target-version = [\'py3X\']` to `target-version = [\'py3Y\']`)\n\n### Dependencies\n\nPython dependencies for this project are managed with [Poetry](https://python-poetry.org) in `pyproject.toml`.\n\nNon-code files, such as static files, can also be included in the [Python package](#python-package) using the\ninclude key in `pyproject.toml`.\n\nTo add a new (development) dependency:\n\n```\n$ docker-compose run app ash\n$ poetry add [dependency] (--dev)\n```\n\nThen rebuild the development container and push to GitLab:\n\n```\n$ docker-compose build app\n$ docker-compose push app\n```\n\nAll dependencies should be periodically reviewed and updated as new versions are released.\n\n#### Static security scanning\n\nTo ensure the security of this API, source code is checked against Bandit for issues such as not sanitising user inputs\nor using weak cryptography.\n\n**Warning:** Bandit is a static analysis tool and can\'t check for issues that are only be detectable when running the\napplication. As with all security tools, Bandit is an aid for spotting common mistakes, not a guarantee of secure code.\n\nThrough [Continuous Integration](#continuous-integration), each commit is tested.\n\nTo check locally:\n\n```shell\n$ docker-compose run app bandit -r .\n```\n\n## Testing\n\n### Continuous Integration\n\nAll commits will trigger a Continuous Integration process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\nPip dependencies are also [checked and monitored for vulnerabilities](#dependency-vulnerability-scanning).\n\n## Distribution\n\nThis project is distributed as a Python package, hosted in\n[PyPi](https://pypi.org/project/bas-style-kit-jinja-templates/).\n\nSource and binary packages are built and published automatically using [Poetry](https://python-poetry.org) through\n[Continuous Deployment](#continuous-deployment).\n\nPackage versions are determined automatically using the `support/python-packaging/parse_version.py` script.\n\n### Continuous Deployment\n\nA Continuous Deployment process using GitLab\'s CI/CD platform is configured in `.gitlab-ci.yml`. This will:\n\n* build the source and binary artefacts for this project\n* publish built artefacts for this project to the relevant PyPi repository\n\nThis process will deploy changes to [PyPi testing](https://test.pypi.org/) on all commits to the master branch.\n\nThis process will deploy changes to [PyPi](https://pypi.org/) on all tagged commits.\n\n## Release procedure\n\n### At release:\n\n1. create a `release` branch\n2. remove `-develop` suffix from `templates_version` variable in `bas_style_kit_jinja_templates/__init__.py`\n3. close release in CHANGELOG.md\n4. push changes, merge the `release` branch into `master` and tag with version\n\nThe project will be built and published to PyPi automatically through [Continuous Deployment](#continuous-deployment).\n\n### After release:\n\n1. create a `next-release` branch\n2. bump `templates_version` variable in `bas_style_kit_jinja_templates/__init__.py`\n3. bump `version` variable in `setup.py`\n4. push changes and merge the `next-release` branch into `master`\n\n## Feedback\n\nThe maintainer of this project is the BAS Web & Applications Team, they can be contacted at:\n[servicedesk@bas.ac.uk](mailto:servicedesk@bas.ac.uk).\n\n## Issue tracking\n\nThis project uses issue tracking, see the\n[Issue tracker](https://gitlab.data.bas.ac.uk/web-apps/bsk/bas-style-kit-jinja2-templates/issues) for more information.\n\n**Note:** Read & write access to this issue tracker is restricted. Contact the project maintainer to request access.\n\n## License\n\n© UK Research and Innovation (UKRI), 2019 - 2020, British Antarctic Survey.\n\nYou may use and re-use this software and associated documentation files free of charge in any format or medium, under\nthe terms of the Open Government Licence v3.0.\n\nYou may obtain a copy of the Open Government Licence at\n[http://www.nationalarchives.gov.uk/doc/open-government-licence/](http://www.nationalarchives.gov.uk/doc/open-government-licence/)\n',
    'author': 'Felix Fennell',
    'author_email': 'felnne@bas.ac.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/antarctica/bas-style-kit-jinja-templates',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
