# PyAlgoTrade
# 
# Copyright 2011 Gabriel Martin Becedillas Ruiz
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#   http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
.. moduleauthor:: Gabriel Martin Becedillas Ruiz <gabriel.becedillas@gmail.com>
"""

import collections
import types

from pyalgotrade import dataseries

class EventWindow:
	"""An EventWindow class is responsible for making calculation over a moving window of values.

	:param windowSize: The size of the window. Must be greater than 0.
	:type windowSize: int.
	:param skipNone: True if None values should not be included in the window.
	:type skipNone: boolean.

	.. note::
		This is a base class and should not be used directly.
	"""

	def __init__(self, windowSize, skipNone=True):
		assert(windowSize > 0)
		assert(type(windowSize) == types.IntType)
		self.__values = collections.deque(maxlen=windowSize)
		self.__windowSize = windowSize
		self.__skipNone = skipNone

	def onNewValue(self, dateTime, value):
		if value != None or not self.__skipNone:
			self.__values.append(value)

	def getValues(self):
		"""Returns the values in the window."""
		return self.__values

	def getWindowSize(self):
		"""Returns the window size."""
		return self.__windowSize

	def windowFull(self):
		return len(self.__values) == self.__windowSize

	def getValue(self):
		"""Override to calculate a value using the values in the window."""
		raise NotImplementedError()

class EventBasedFilter(dataseries.SequenceDataSeries):
	"""An EventBasedFilter class is responsible for capturing new values in a :class:`pyalgotrade.dataseries.DataSeries`
	and using an :class:`EventWindow` to calculate new values.

	:param dataSeries: The DataSeries instance being filtered.
	:type dataSeries: :class:`pyalgotrade.dataseries.DataSeries`.
	:param eventWindow: The EventWindow instance to use to calculate new values.
	:type eventWindow: :class:`EventWindow`.
	:param maxLen: The maximum number of values to hold. If not None, it must be greater than 0.
		Once a bounded length is full, when new items are added, a corresponding number of items are discarded from the opposite end.
	:type maxLen: int.
	"""

	def __init__(self, dataSeries, eventWindow, maxLen=dataseries.DEFAULT_MAX_LEN):
		dataseries.SequenceDataSeries.__init__(self, maxLen)
		self.__dataSeries = dataSeries
		self.__dataSeries.getNewValueEvent().subscribe(self.__onNewValue)
		self.__eventWindow = eventWindow

	def __onNewValue(self, dataSeries, dateTime, value):
		# Let the event window perform calculations.
		self.__eventWindow.onNewValue(dateTime, value)
		# Get the resulting value
		newValue = self.__eventWindow.getValue()
		# Add the new value.
		self.appendWithDateTime(dateTime, newValue)

	def getDataSeries(self):
		"""Returns the :class:`pyalgotrade.dataseries.DataSeries` being filtered."""
		return self.__dataSeries

	def getEventWindow(self):
		"""Returns the :class:`EventWindow` instance to use to calculate new values."""
		return self.__eventWindow

