# coding: utf-8

# (C) Copyright IBM Corp. 2021, 2022, 2023.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# IBM OpenAPI SDK Code Generator Version: 3.68.2-ac7def68-20230310-195410

"""
The IBM Cloud Virtual Private Cloud (VPC) API can be used to programmatically provision
and manage virtual server instances, along with subnets, volumes, load balancers, and
more.

API Version: 2023-03-28
"""

from datetime import datetime
from enum import Enum
from typing import Dict, List
import base64
import json
import logging
import sys

from ibm_cloud_sdk_core import BaseService, DetailedResponse, get_query_param
from ibm_cloud_sdk_core.authenticators.authenticator import Authenticator
from ibm_cloud_sdk_core.get_authenticator import get_authenticator_from_environment
from ibm_cloud_sdk_core.utils import convert_model, datetime_to_string, string_to_datetime

from .common import get_sdk_headers

##############################################################################
# Service
##############################################################################

class VpcV1(BaseService):
    """The vpc V1 service."""

    DEFAULT_SERVICE_URL = 'https://us-south.iaas.cloud.ibm.com/v1'
    DEFAULT_SERVICE_NAME = 'vpc'

    @classmethod
    def new_instance(cls,
                     version: str = '2023-03-28',
                     service_name: str = DEFAULT_SERVICE_NAME,
                     generation: int = 2,
                    ) -> 'VpcV1':
        """
        Return a new client for the vpc service using the specified parameters and
               external configuration.

        :param str version: The API version, in format `YYYY-MM-DD`. For the API
               behavior documented here, specify any date between `2022-09-13` and today's
               date (UTC).
        """
        if version is None:
            raise ValueError('version must be provided')

        authenticator = get_authenticator_from_environment(service_name)
        service = cls(
            version,
            authenticator,
            generation,
            )
        service.configure_service(service_name)
        return service

    def __init__(self,
                 version: str = '2023-03-28',
                 authenticator: Authenticator = None,
                 generation: int = 2,
                ) -> None:
        """
        Construct a new client for the vpc service.

        :param str version: The API version, in format `YYYY-MM-DD`. For the API
               behavior documented here, specify any date between `2022-09-13` and today's
               date (UTC).

        :param Authenticator authenticator: The authenticator specifies the authentication mechanism.
               Get up to date information from https://github.com/IBM/python-sdk-core/blob/main/README.md
               about initializing the authenticator of your choice.
        """
        if version is None:
            raise ValueError('version must be provided')

        BaseService.__init__(self,
                             service_url=self.DEFAULT_SERVICE_URL,
                             authenticator=authenticator)
        self.generation = generation
        self.version = version


    #########################
    # VPCs
    #########################


    def list_vpcs(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        classic_access: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPCs.

        This request lists all VPCs in the region. A VPC is a virtual network that belongs
        to an account and provides logical isolation from other networks. A VPC is made up
        of resources in one or more zones. VPCs are regional, and each VPC can contain
        resources in multiple zones in a region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param bool classic_access: (optional) Filters the collection to VPCs with
               the specified `classic_access` value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPCCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpcs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'classic_access': classic_access,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpcs'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpc(self,
        *,
        address_prefix_management: str = None,
        classic_access: bool = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a VPC.

        This request creates a new VPC from a VPC prototype object. The prototype object
        is structured in the same way as a retrieved VPC, and contains the information
        necessary to create the new VPC.

        :param str address_prefix_management: (optional) Indicates whether a
               [default address
               prefix](https://cloud.ibm.com/docs/vpc?topic=vpc-configuring-address-prefixes)
               will be automatically created for each zone in this VPC. If `manual`, this
               VPC will be created with no default address prefixes.
               Since address prefixes are managed identically regardless of whether they
               were automatically created, the value is not preserved as a VPC property.
        :param bool classic_access: (optional) Indicates whether this VPC will be
               connected to Classic Infrastructure. If true, this VPC's resources will
               have private network connectivity to the account's Classic Infrastructure
               resources. Only one VPC, per region, may be connected in this way. This
               value is set at creation and subsequently immutable.
        :param str name: (optional) The name for this VPC. The name must not be
               used by another VPC in the region. If unspecified, the name will be a
               hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'address_prefix_management': address_prefix_management,
            'classic_access': classic_access,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpcs'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpc(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPC.

        This request deletes a VPC. This operation cannot be reversed. For this request to
        succeed, the VPC must not contain any instances, subnets, public gateways, or
        endpoint gateways. All security groups and network ACLs associated with the VPC
        are automatically deleted. All flow log collectors with `auto_delete` set to
        `true` targeting the VPC or any resource in the VPC are automatically deleted.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC.

        This request retrieves a single VPC specified by the identifier in the URL.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpc(self,
        id: str,
        vpc_patch: 'VPCPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPC.

        This request updates a VPC's name.

        :param str id: The VPC identifier.
        :param VPCPatch vpc_patch: The VPC patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPC` object
        """

        if not id:
            raise ValueError('id must be provided')
        if vpc_patch is None:
            raise ValueError('vpc_patch must be provided')
        if isinstance(vpc_patch, VPCPatch):
            vpc_patch = convert_model(vpc_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpc_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_default_network_acl(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC's default network ACL.

        This request retrieves the default network ACL for the VPC specified by the
        identifier in the URL. The default network ACL is applied to any new subnets in
        the VPC which do not specify a network ACL.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DefaultNetworkACL` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_default_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}/default_network_acl'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_default_routing_table(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC's default routing table.

        This request retrieves the default routing table for the VPC specified by the
        identifier in the URL. The default routing table is associated with any subnets in
        the VPC which have not been explicitly associated with another routing table.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DefaultRoutingTable` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_default_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}/default_routing_table'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_default_security_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC's default security group.

        This request retrieves the default security group for the VPC specified by the
        identifier in the URL. Resources created in this VPC that allow a security group
        to be optionally specified will use this security group by default.

        :param str id: The VPC identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DefaultSecurityGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_default_security_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{id}/default_security_group'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_vpc_address_prefixes(self,
        vpc_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all address prefixes for a VPC.

        This request lists all address pool prefixes for a VPC.

        :param str vpc_id: The VPC identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefixCollection` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_address_prefixes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/address_prefixes'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpc_address_prefix(self,
        vpc_id: str,
        cidr: str,
        zone: 'ZoneIdentity',
        *,
        is_default: bool = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an address prefix for a VPC.

        This request creates a new prefix from a prefix prototype object. The prototype
        object is structured in the same way as a retrieved prefix, and contains the
        information necessary to create the new prefix.

        :param str vpc_id: The VPC identifier.
        :param str cidr: The IPv4 range of the address prefix, expressed in CIDR
               format. The range must not overlap with any existing address prefixes in
               the VPC or any of the following reserved address ranges:
                 - `127.0.0.0/8` (IPv4 loopback addresses)
                 - `161.26.0.0/16` (IBM services)
                 - `166.8.0.0/14` (Cloud Service Endpoints)
                 - `169.254.0.0/16` (IPv4 link-local addresses)
                 - `224.0.0.0/4` (IPv4 multicast addresses)
               The prefix length of the address prefix's CIDR must be between `/9`
               (8,388,608 addresses) and `/29` (8 addresses).
        :param ZoneIdentity zone: The zone this address prefix will reside in.
        :param bool is_default: (optional) Indicates whether this will be the
               default address prefix for this zone in this VPC. If `true`, the VPC must
               not have a default address prefix for this zone.
        :param str name: (optional) The name for this address prefix. The name must
               not be used by another address prefix for the VPC. If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if cidr is None:
            raise ValueError('cidr must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        zone = convert_model(zone)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'cidr': cidr,
            'zone': zone,
            'is_default': is_default,
            'name': name,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/address_prefixes'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpc_address_prefix(self,
        vpc_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an address prefix.

        This request deletes a prefix. This operation cannot be reversed. The request will
        fail if any subnets use addresses from this prefix.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/address_prefixes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_address_prefix(self,
        vpc_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an address prefix.

        This request retrieves a single prefix specified by the identifier in the URL.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/address_prefixes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpc_address_prefix(self,
        vpc_id: str,
        id: str,
        address_prefix_patch: 'AddressPrefixPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an address prefix.

        This request updates a prefix with the information in a provided prefix patch. The
        prefix patch object is structured in the same way as a retrieved prefix and
        contains only the information to be updated.

        :param str vpc_id: The VPC identifier.
        :param str id: The prefix identifier.
        :param AddressPrefixPatch address_prefix_patch: The prefix patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `AddressPrefix` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if address_prefix_patch is None:
            raise ValueError('address_prefix_patch must be provided')
        if isinstance(address_prefix_patch, AddressPrefixPatch):
            address_prefix_patch = convert_model(address_prefix_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_address_prefix')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(address_prefix_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/address_prefixes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_vpc_routes(self,
        vpc_id: str,
        *,
        zone_name: str = None,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all routes in a VPC's default routing table.

        This request lists all routes in the VPC's default routing table. Each route is
        zone-specific and directs any packets matching its destination CIDR block to a
        `next_hop` IP address. The most specific route matching a packet's destination
        will be used. If multiple equally-specific routes exist, traffic will be
        distributed across them.

        :param str vpc_id: The VPC identifier.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RouteCollection` object

        Deprecated: this method is deprecated and may be removed in a future release.
        """

        logging.warning('A deprecated operation has been invoked: list_vpc_routes')

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_routes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'zone.name': zone_name,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpc_route(self,
        vpc_id: str,
        destination: str,
        zone: 'ZoneIdentity',
        *,
        action: str = None,
        name: str = None,
        next_hop: 'RoutePrototypeNextHop' = None,
        priority: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a route in a VPC's default routing table.

        This request creates a new route in the VPC's default routing table. The route
        prototype object is structured in the same way as a retrieved route, and contains
        the information necessary to create the new route. The request will fail if the
        new route will cause a loop.

        :param str vpc_id: The VPC identifier.
        :param str destination: The destination of the route. At most two routes
               per `zone` in a table can have the same `destination` and `priority`, and
               only if both routes have an `action` of `deliver` and the `next_hop` is an
               IP address.
        :param ZoneIdentity zone: The zone to apply the route to. (Traffic from
               subnets in this zone will be
               subject to this route.).
        :param str action: (optional) The action to perform with a packet matching
               the route:
               - `delegate`: delegate to system-provided routes
               - `delegate_vpc`: delegate to system-provided routes, ignoring
               Internet-bound routes
               - `deliver`: deliver the packet to the specified `next_hop`
               - `drop`: drop the packet.
        :param str name: (optional) The name for this route. The name must not be
               used by another route in the routing table. Names starting with `ibm-` are
               reserved for system-provided routes, and are not allowed. If unspecified,
               the name will be a hyphenated list of randomly-selected words.
        :param RoutePrototypeNextHop next_hop: (optional) If `action` is `deliver`,
               the next hop that packets will be delivered to. For other `action`
               values, it must be omitted or specified as `0.0.0.0`.
        :param int priority: (optional) The priority of this route. Smaller values
               have higher priority.
               If a routing table contains multiple routes with the same `zone` and
               `destination`, the route with the highest priority (smallest value) is
               selected. If two routes have the same `destination` and `priority`, traffic
               is distributed between them.
               At most two routes per `zone` in a table can have the same `destination`
               and
               `priority`, and only if both routes have an `action` of `deliver` and the
               `next_hop` is an IP address.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object

        Deprecated: this method is deprecated and may be removed in a future release.
        """

        logging.warning('A deprecated operation has been invoked: create_vpc_route')

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if destination is None:
            raise ValueError('destination must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        zone = convert_model(zone)
        if next_hop is not None:
            next_hop = convert_model(next_hop)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'destination': destination,
            'zone': zone,
            'action': action,
            'name': name,
            'next_hop': next_hop,
            'priority': priority,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpc_route(self,
        vpc_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPC route.

        This request deletes a route. This operation cannot be reversed.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse

        Deprecated: this method is deprecated and may be removed in a future release.
        """

        logging.warning('A deprecated operation has been invoked: delete_vpc_route')

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_route(self,
        vpc_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC route.

        This request retrieves a single route specified by the identifier in the URL.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object

        Deprecated: this method is deprecated and may be removed in a future release.
        """

        logging.warning('A deprecated operation has been invoked: get_vpc_route')

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpc_route(self,
        vpc_id: str,
        id: str,
        route_patch: 'RoutePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPC route.

        This request updates a route with the information in a provided route patch. The
        route patch object is structured in the same way as a retrieved route and contains
        only the information to be updated.

        :param str vpc_id: The VPC identifier.
        :param str id: The route identifier.
        :param RoutePatch route_patch: The route patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object

        Deprecated: this method is deprecated and may be removed in a future release.
        """

        logging.warning('A deprecated operation has been invoked: update_vpc_route')

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if route_patch is None:
            raise ValueError('route_patch must be provided')
        if isinstance(route_patch, RoutePatch):
            route_patch = convert_model(route_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(route_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_vpc_routing_tables(self,
        vpc_id: str,
        *,
        start: str = None,
        limit: int = None,
        is_default: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all routing tables for a VPC.

        This request lists all routing tables for a VPC. Each subnet in a VPC is
        associated with a routing table, which controls delivery of packets sent on that
        subnet according to the action of the most specific matching route in the table.
        If multiple equally-specific routes exist, traffic will be distributed across
        them. If no routes match, delivery will be controlled by the system's built-in
        routes.

        :param str vpc_id: The VPC identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param bool is_default: (optional) Filters the collection to routing tables
               with the specified `is_default` value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTableCollection` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_routing_tables')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'is_default': is_default,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpc_routing_table(self,
        vpc_id: str,
        *,
        accept_routes_from: List['ResourceFilter'] = None,
        name: str = None,
        route_direct_link_ingress: bool = None,
        route_internet_ingress: bool = None,
        route_transit_gateway_ingress: bool = None,
        route_vpc_zone_ingress: bool = None,
        routes: List['RoutePrototype'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a routing table for a VPC.

        This request creates a routing table from a routing table prototype object. The
        prototype object is structured in the same way as a retrieved routing table, and
        contains the information necessary to create the new routing table.

        :param str vpc_id: The VPC identifier.
        :param List[ResourceFilter] accept_routes_from: (optional) The filters
               specifying the resources that may create routes in this routing table.
               At present, only the `resource_type` filter is permitted, and only the
               `vpn_server` value is supported, but filter support is expected to expand
               in the future.
        :param str name: (optional) The name for this routing table. The name must
               not be used by another routing table in the VPC. If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param bool route_direct_link_ingress: (optional) If set to `true`, this
               routing table will be used to route traffic that originates from [Direct
               Link](https://cloud.ibm.com/docs/dl) to this VPC. The VPC must not already
               have a routing table with this property set to `true`.
               Incoming traffic will be routed according to the routing table with one
               exception: routes with an `action` of `deliver` are treated as `drop`
               unless the `next_hop` is an IP address bound to a network interface on a
               subnet in the route's `zone`. Therefore, if an incoming packet matches a
               route with a `next_hop` of an internet-bound IP address or a VPN gateway
               connection, the packet will be dropped.
               If [Classic
               Access](https://cloud.ibm.com/docs/vpc?topic=vpc-setting-up-access-to-classic-infrastructure)
               is enabled for this VPC, and this property is set to `true`, its incoming
               traffic will also be routed according to this routing table.
        :param bool route_internet_ingress: (optional) If set to `true`, this
               routing table will be used to route traffic that originates from the
               internet. For this to succeed, the VPC must not already have a routing
               table with this property set to `true`.
               Incoming traffic will be routed according to the routing table with two
               exceptions:
               - Traffic destined for IP addresses associated with public gateways will
               not be
                 subject to routes in this routing table.
               - Routes with an action of deliver are treated as drop unless the
               `next_hop` is an
                 IP address bound to a network interface on a subnet in the route's
               `zone`.
                 Therefore, if an incoming packet matches a route with a `next_hop` of an
                 internet-bound IP address or a VPN gateway connection, the packet will be
               dropped.
        :param bool route_transit_gateway_ingress: (optional) If set to `true`,
               this routing table will be used to route traffic that originates from
               [Transit Gateway](https://cloud.ibm.com/docs/transit-gateway) to this VPC.
               The VPC must not already have a routing table with this property set to
               `true`.
               Incoming traffic will be routed according to the routing table with one
               exception: routes with an `action` of `deliver` are treated as `drop`
               unless the `next_hop` is an IP address bound to a network interface on a
               subnet in the route's `zone`. Therefore, if an incoming packet matches a
               route with a `next_hop` of an internet-bound IP address or a VPN gateway
               connection, the packet will be dropped.
        :param bool route_vpc_zone_ingress: (optional) If set to `true`, this
               routing table will be used to route traffic that originates from subnets in
               other zones in this VPC. The VPC must not already have a routing table with
               this property set to `true`.
               Incoming traffic will be routed according to the routing table with one
               exception: routes with an `action` of `deliver` are treated as `drop`
               unless the `next_hop` is an IP address within the VPC's address prefix
               ranges. Therefore, if an incoming packet matches a route with a `next_hop`
               of an internet-bound IP address or a VPN gateway connection, the packet
               will be dropped.
        :param List[RoutePrototype] routes: (optional) The prototype objects for
               routes to create for this routing table. If unspecified, the routing table
               will be created with no routes.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTable` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if accept_routes_from is not None:
            accept_routes_from = [convert_model(x) for x in accept_routes_from]
        if routes is not None:
            routes = [convert_model(x) for x in routes]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'accept_routes_from': accept_routes_from,
            'name': name,
            'route_direct_link_ingress': route_direct_link_ingress,
            'route_internet_ingress': route_internet_ingress,
            'route_transit_gateway_ingress': route_transit_gateway_ingress,
            'route_vpc_zone_ingress': route_vpc_zone_ingress,
            'routes': routes,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id']
        path_param_values = self.encode_path_vars(vpc_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpc_routing_table(self,
        vpc_id: str,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPC routing table.

        This request deletes a routing table.  A routing table cannot be deleted if it is
        associated with any subnets in the VPC. Additionally, a VPC's default routing
        table cannot be deleted. This operation cannot be reversed.

        :param str vpc_id: The VPC identifier.
        :param str id: The routing table identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_routing_table(self,
        vpc_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC routing table.

        This request retrieves a single routing table specified by the identifier in the
        URL.

        :param str vpc_id: The VPC identifier.
        :param str id: The routing table identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTable` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpc_routing_table(self,
        vpc_id: str,
        id: str,
        routing_table_patch: 'RoutingTablePatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPC routing table.

        This request updates a routing table with the information in a provided routing
        table patch. The patch object is structured in the same way as a retrieved table
        and contains only the information to be updated.

        :param str vpc_id: The VPC identifier.
        :param str id: The routing table identifier.
        :param RoutingTablePatch routing_table_patch: The routing table patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTable` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if routing_table_patch is None:
            raise ValueError('routing_table_patch must be provided')
        if isinstance(routing_table_patch, RoutingTablePatch):
            routing_table_patch = convert_model(routing_table_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(routing_table_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_vpc_routing_table_routes(self,
        vpc_id: str,
        routing_table_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all routes in a VPC routing table.

        This request lists all routes in a VPC routing table. If subnets are associated
        with this routing table, delivery of packets sent on a subnet is performed
        according to the action of the most specific matching route in the table (provided
        the subnet and route are in the same zone). If multiple equally-specific routes
        exist, the route with the highest priority will be used. If two matching routes
        have the same destination and priority, traffic will be distributed between them.
        If no routes match, delivery will be controlled by the system's built-in routes.

        :param str vpc_id: The VPC identifier.
        :param str routing_table_id: The routing table identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RouteCollection` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not routing_table_id:
            raise ValueError('routing_table_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpc_routing_table_routes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'routing_table_id']
        path_param_values = self.encode_path_vars(vpc_id, routing_table_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpc_routing_table_route(self,
        vpc_id: str,
        routing_table_id: str,
        destination: str,
        zone: 'ZoneIdentity',
        *,
        action: str = None,
        name: str = None,
        next_hop: 'RoutePrototypeNextHop' = None,
        priority: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a route in a VPC routing table.

        This request creates a new VPC route from a VPC route prototype object. The
        prototype object is structured in the same way as a retrieved VPC route and
        contains the information necessary to create the route.

        :param str vpc_id: The VPC identifier.
        :param str routing_table_id: The routing table identifier.
        :param str destination: The destination of the route. At most two routes
               per `zone` in a table can have the same `destination` and `priority`, and
               only if both routes have an `action` of `deliver` and the `next_hop` is an
               IP address.
        :param ZoneIdentity zone: The zone to apply the route to. (Traffic from
               subnets in this zone will be
               subject to this route.).
        :param str action: (optional) The action to perform with a packet matching
               the route:
               - `delegate`: delegate to system-provided routes
               - `delegate_vpc`: delegate to system-provided routes, ignoring
               Internet-bound routes
               - `deliver`: deliver the packet to the specified `next_hop`
               - `drop`: drop the packet.
        :param str name: (optional) The name for this route. The name must not be
               used by another route in the routing table. Names starting with `ibm-` are
               reserved for system-provided routes, and are not allowed. If unspecified,
               the name will be a hyphenated list of randomly-selected words.
        :param RoutePrototypeNextHop next_hop: (optional) If `action` is `deliver`,
               the next hop that packets will be delivered to. For other `action`
               values, it must be omitted or specified as `0.0.0.0`.
        :param int priority: (optional) The priority of this route. Smaller values
               have higher priority.
               If a routing table contains multiple routes with the same `zone` and
               `destination`, the route with the highest priority (smallest value) is
               selected. If two routes have the same `destination` and `priority`, traffic
               is distributed between them.
               At most two routes per `zone` in a table can have the same `destination`
               and
               `priority`, and only if both routes have an `action` of `deliver` and the
               `next_hop` is an IP address.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not routing_table_id:
            raise ValueError('routing_table_id must be provided')
        if destination is None:
            raise ValueError('destination must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        zone = convert_model(zone)
        if next_hop is not None:
            next_hop = convert_model(next_hop)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpc_routing_table_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'destination': destination,
            'zone': zone,
            'action': action,
            'name': name,
            'next_hop': next_hop,
            'priority': priority,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'routing_table_id']
        path_param_values = self.encode_path_vars(vpc_id, routing_table_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpc_routing_table_route(self,
        vpc_id: str,
        routing_table_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPC routing table route.

        This request deletes a VPC route. This operation cannot be reversed. Only VPC
        routes with an `origin` of `user` are allowed to be deleted.

        :param str vpc_id: The VPC identifier.
        :param str routing_table_id: The routing table identifier.
        :param str id: The VPC routing table route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not routing_table_id:
            raise ValueError('routing_table_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpc_routing_table_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpc_id', 'routing_table_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, routing_table_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpc_routing_table_route(self,
        vpc_id: str,
        routing_table_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPC routing table route.

        This request retrieves a single VPC route specified by the identifier in the URL
        path.

        :param str vpc_id: The VPC identifier.
        :param str routing_table_id: The routing table identifier.
        :param str id: The VPC routing table route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not routing_table_id:
            raise ValueError('routing_table_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpc_routing_table_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'routing_table_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, routing_table_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpc_routing_table_route(self,
        vpc_id: str,
        routing_table_id: str,
        id: str,
        route_patch: 'RoutePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPC routing table route.

        This request updates a VPC route with the information provided in a route patch
        object. The patch object is structured in the same way as a retrieved VPC route
        and needs to contain only the information to be updated. Only VPC routes with an
        `origin` of `user` are allowed to be updated.

        :param str vpc_id: The VPC identifier.
        :param str routing_table_id: The routing table identifier.
        :param str id: The VPC routing table route identifier.
        :param RoutePatch route_patch: The VPC route patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Route` object
        """

        if not vpc_id:
            raise ValueError('vpc_id must be provided')
        if not routing_table_id:
            raise ValueError('routing_table_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if route_patch is None:
            raise ValueError('route_patch must be provided')
        if isinstance(route_patch, RoutePatch):
            route_patch = convert_model(route_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpc_routing_table_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(route_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpc_id', 'routing_table_id', 'id']
        path_param_values = self.encode_path_vars(vpc_id, routing_table_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpcs/{vpc_id}/routing_tables/{routing_table_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Subnets
    #########################


    def list_subnets(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        routing_table_id: str = None,
        routing_table_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all subnets.

        This request lists all subnets in the region. Subnets are contiguous ranges of IP
        addresses specified in CIDR block notation. Each subnet is within a particular
        zone and cannot span multiple zones or regions.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str routing_table_id: (optional) Filters the collection to subnets
               attached to the routing table with the specified identifier.
        :param str routing_table_name: (optional) Filters the collection to subnets
               attached to the routing table with the specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SubnetCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_subnets')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'routing_table.id': routing_table_id,
            'routing_table.name': routing_table_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/subnets'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_subnet(self,
        subnet_prototype: 'SubnetPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a subnet.

        This request creates a new subnet from a subnet prototype object. The prototype
        object is structured in the same way as a retrieved subnet, and contains the
        information necessary to create the new subnet. For this request to succeed, the
        prototype's CIDR block must not overlap with an existing subnet in the VPC.

        :param SubnetPrototype subnet_prototype: The subnet prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if subnet_prototype is None:
            raise ValueError('subnet_prototype must be provided')
        if isinstance(subnet_prototype, SubnetPrototype):
            subnet_prototype = convert_model(subnet_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_subnet')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(subnet_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/subnets'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_subnet(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a subnet.

        This request deletes a subnet. This operation cannot be reversed. For this request
        to succeed, the subnet must not be referenced by any network interfaces, VPN
        gateways, or load balancers. A delete operation automatically detaches the subnet
        from any network ACLs, public gateways, or endpoint gateways. All flow log
        collectors with `auto_delete` set to `true` targeting the subnet or any resource
        in the subnet are automatically deleted.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_subnet')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_subnet(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a subnet.

        This request retrieves a single subnet specified by the identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_subnet(self,
        id: str,
        subnet_patch: 'SubnetPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a subnet.

        This request updates a subnet with the information in a provided subnet patch. The
        subnet patch object is structured in the same way as a retrieved subnet and
        contains only the information to be updated.

        :param str id: The subnet identifier.
        :param SubnetPatch subnet_patch: The subnet patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Subnet` object
        """

        if not id:
            raise ValueError('id must be provided')
        if subnet_patch is None:
            raise ValueError('subnet_patch must be provided')
        if isinstance(subnet_patch, SubnetPatch):
            subnet_patch = convert_model(subnet_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_subnet')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(subnet_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_subnet_network_acl(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a subnet's attached network ACL.

        This request retrieves the network ACL attached to the subnet specified by the
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/network_acl'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def replace_subnet_network_acl(self,
        id: str,
        network_acl_identity: 'NetworkACLIdentity',
        **kwargs
    ) -> DetailedResponse:
        """
        Replace the network ACL for a subnet.

        This request replaces the existing network ACL for a subnet with the network ACL
        specified in the request body.

        :param str id: The subnet identifier.
        :param NetworkACLIdentity network_acl_identity: The network ACL identity.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if not id:
            raise ValueError('id must be provided')
        if network_acl_identity is None:
            raise ValueError('network_acl_identity must be provided')
        if isinstance(network_acl_identity, NetworkACLIdentity):
            network_acl_identity = convert_model(network_acl_identity)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='replace_subnet_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_acl_identity)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/network_acl'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def unset_subnet_public_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Detach a public gateway from a subnet.

        This request detaches the public gateway from the subnet specified by the subnet
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='unset_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/public_gateway'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_subnet_public_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a subnet's attached public gateway.

        This request retrieves the public gateway attached to the subnet specified by the
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/public_gateway'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def set_subnet_public_gateway(self,
        id: str,
        public_gateway_identity: 'PublicGatewayIdentity',
        **kwargs
    ) -> DetailedResponse:
        """
        Attach a public gateway to a subnet.

        This request attaches the public gateway, specified in the request body, to the
        subnet specified by the subnet identifier in the URL. The public gateway must have
        the same VPC and zone as the subnet.

        :param str id: The subnet identifier.
        :param PublicGatewayIdentity public_gateway_identity: The public gateway
               identity.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        if public_gateway_identity is None:
            raise ValueError('public_gateway_identity must be provided')
        if isinstance(public_gateway_identity, PublicGatewayIdentity):
            public_gateway_identity = convert_model(public_gateway_identity)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='set_subnet_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(public_gateway_identity)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/public_gateway'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_subnet_routing_table(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a subnet's attached routing table.

        This request retrieves the routing table attached to the subnet specified by the
        identifier in the URL.

        :param str id: The subnet identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTable` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/routing_table'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def replace_subnet_routing_table(self,
        id: str,
        routing_table_identity: 'RoutingTableIdentity',
        **kwargs
    ) -> DetailedResponse:
        """
        Replace the routing table for a subnet.

        This request replaces the existing routing table for a subnet with the routing
        table specified in the request body.
        For this request to succeed, the routing table `route_direct_link_ingress`,
        `route_transit_gateway_ingress`, and `route_vpc_zone_ingress` properties must be
        `false`.

        :param str id: The subnet identifier.
        :param RoutingTableIdentity routing_table_identity: The routing table
               identity.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RoutingTable` object
        """

        if not id:
            raise ValueError('id must be provided')
        if routing_table_identity is None:
            raise ValueError('routing_table_identity must be provided')
        if isinstance(routing_table_identity, RoutingTableIdentity):
            routing_table_identity = convert_model(routing_table_identity)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='replace_subnet_routing_table')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(routing_table_identity)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{id}/routing_table'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_subnet_reserved_ips(self,
        subnet_id: str,
        *,
        start: str = None,
        limit: int = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all reserved IPs in a subnet.

        This request lists all reserved IPs in a subnet. A reserved IP resource will exist
        for every address in the subnet which is not available for use.

        :param str subnet_id: The subnet identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIPCollection` object
        """

        if not subnet_id:
            raise ValueError('subnet_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_subnet_reserved_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['subnet_id']
        path_param_values = self.encode_path_vars(subnet_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{subnet_id}/reserved_ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_subnet_reserved_ip(self,
        subnet_id: str,
        *,
        address: str = None,
        auto_delete: bool = None,
        name: str = None,
        target: 'ReservedIPTargetPrototype' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Reserve an IP in a subnet.

        This request reserves an IP address in a subnet. If the provided prototype object
        includes an `address`, the address must not already be reserved.

        :param str subnet_id: The subnet identifier.
        :param str address: (optional) The IP address to reserve, which must not
               already be reserved on the subnet.
               If unspecified, an available address on the subnet will automatically be
               selected.
        :param bool auto_delete: (optional) Indicates whether this reserved IP
               member will be automatically deleted when either
               `target` is deleted, or the reserved IP is unbound. Must be `false` if the
               reserved IP is unbound.
        :param str name: (optional) The name for this reserved IP. The name must
               not be used by another reserved IP in the subnet. Names starting with
               `ibm-` are reserved for provider-owned resources, and are not allowed. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param ReservedIPTargetPrototype target: (optional) The target to bind this
               reserved IP to.  The target must be in the same VPC.
               At present, only endpoint gateway targets are supported.  The endpoint
               gateway must
               not be already bound to a reserved IP in the subnet's zone.
               If unspecified, the reserved IP will be created unbound.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not subnet_id:
            raise ValueError('subnet_id must be provided')
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_subnet_reserved_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'address': address,
            'auto_delete': auto_delete,
            'name': name,
            'target': target,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['subnet_id']
        path_param_values = self.encode_path_vars(subnet_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{subnet_id}/reserved_ips'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_subnet_reserved_ip(self,
        subnet_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a reserved IP.

        This request releases a reserved IP. This operation cannot be reversed.
        For this request to succeed, the reserved IP must not be required by another
        resource, such as a network interface for which it is the primary IP. A
        provider-owned reserved IP is not allowed to be deleted.

        :param str subnet_id: The subnet identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not subnet_id:
            raise ValueError('subnet_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_subnet_reserved_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['subnet_id', 'id']
        path_param_values = self.encode_path_vars(subnet_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{subnet_id}/reserved_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_subnet_reserved_ip(self,
        subnet_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a reserved IP.

        This request retrieves a single reserved IP specified by the identifier in the
        URL.

        :param str subnet_id: The subnet identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not subnet_id:
            raise ValueError('subnet_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_subnet_reserved_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['subnet_id', 'id']
        path_param_values = self.encode_path_vars(subnet_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{subnet_id}/reserved_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_subnet_reserved_ip(self,
        subnet_id: str,
        id: str,
        reserved_ip_patch: 'ReservedIPPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a reserved IP.

        This request updates a reserved IP with the information in a provided reserved IP
        patch. The reserved IP patch object is structured in the same way as a retrieved
        reserved IP and contains only the information to be updated.
        A provider-owned reserved IP is not allowed to be updated.

        :param str subnet_id: The subnet identifier.
        :param str id: The reserved IP identifier.
        :param ReservedIPPatch reserved_ip_patch: The reserved IP patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not subnet_id:
            raise ValueError('subnet_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if reserved_ip_patch is None:
            raise ValueError('reserved_ip_patch must be provided')
        if isinstance(reserved_ip_patch, ReservedIPPatch):
            reserved_ip_patch = convert_model(reserved_ip_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_subnet_reserved_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(reserved_ip_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['subnet_id', 'id']
        path_param_values = self.encode_path_vars(subnet_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/subnets/{subnet_id}/reserved_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Images
    #########################


    def list_images(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        name: str = None,
        visibility: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all images.

        This request lists all images available in the region. An image provides source
        data for a volume. Images are either system-provided, or created from another
        source, such as importing from Cloud Object Storage.
        The images will be sorted by their `created_at` property values, with the newest
        first. Images with identical `created_at` values will be secondarily sorted by
        ascending `id` property values.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str visibility: (optional) Filters the collection to images with the
               specified `visibility`.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ImageCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_images')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'visibility': visibility,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/images'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_image(self,
        image_prototype: 'ImagePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create an image.

        This request creates a new image from an image prototype object. The prototype
        object is structured in the same way as a retrieved image, and contains the
        information necessary to create the new image. If an image is being imported, a
        URL to the image file on object storage must be specified. If an image is being
        created from an existing volume, that volume must be specified.

        :param ImagePrototype image_prototype: The image prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if image_prototype is None:
            raise ValueError('image_prototype must be provided')
        if isinstance(image_prototype, ImagePrototype):
            image_prototype = convert_model(image_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_image')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(image_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/images'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_image(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an image.

        This request deletes an image. This operation cannot be reversed. A
        system-provided image is not allowed to be deleted. Additionally, an image cannot
        be deleted if it:
        - has a `status` of `deleting`
        - has a `status` of `pending` with a `status_reasons` code of
        `image_request_in_progress`
        - has `catalog_offering.managed` set to `true`.

        :param str id: The image identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_image')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/images/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_image(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an image.

        This request retrieves a single image specified by the identifier in the URL.

        :param str id: The image identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_image')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/images/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_image(self,
        id: str,
        image_patch: 'ImagePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an image.

        This request updates an image with the information in a provided image patch. The
        image patch object is structured in the same way as a retrieved image and contains
        only the information to be updated. A system-provided image is not allowed to be
        updated. An image with a `status` of `deleting` cannot be updated.

        :param str id: The image identifier.
        :param ImagePatch image_patch: The image patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Image` object
        """

        if not id:
            raise ValueError('id must be provided')
        if image_patch is None:
            raise ValueError('image_patch must be provided')
        if isinstance(image_patch, ImagePatch):
            image_patch = convert_model(image_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_image')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(image_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/images/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_operating_systems(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all operating systems.

        This request lists all operating systems in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `OperatingSystemCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_operating_systems')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/operating_systems'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_operating_system(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an operating system.

        This request retrieves a single operating system specified by the name in the URL.

        :param str name: The operating system name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `OperatingSystem` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_operating_system')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/operating_systems/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Keys
    #########################


    def list_keys(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all keys.

        This request lists all keys in the region. A key contains a public SSH key which
        may be installed on instances when they are created. Private keys are not stored.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `KeyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_keys')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/keys'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_key(self,
        public_key: str,
        *,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        type: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a key.

        This request creates a new SSH key from an key prototype object. The prototype
        object is structured in the same way as a retrieved key, and contains the
        information necessary to create the new key. The public key value must be
        provided.

        :param str public_key: A unique public SSH key to import, in OpenSSH format
               (consisting of three space-separated fields: the algorithm name,
               base64-encoded key, and a comment). The algorithm and comment fields may be
               omitted, as only the key field is imported.
        :param str name: (optional) The name for this key. The name must not be
               used by another key in the region. If unspecified, the name will be a
               hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param str type: (optional) The crypto-system used by this key.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if public_key is None:
            raise ValueError('public_key must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_key')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'public_key': public_key,
            'name': name,
            'resource_group': resource_group,
            'type': type,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/keys'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_key(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a key.

        This request deletes a key. This operation cannot be reversed.

        :param str id: The key identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_key')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/keys/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_key(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a key.

        This request retrieves a single key specified by the identifier in the URL.

        :param str id: The key identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_key')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/keys/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_key(self,
        id: str,
        key_patch: 'KeyPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a key.

        This request updates a key's name.

        :param str id: The key identifier.
        :param KeyPatch key_patch: The key patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Key` object
        """

        if not id:
            raise ValueError('id must be provided')
        if key_patch is None:
            raise ValueError('key_patch must be provided')
        if isinstance(key_patch, KeyPatch):
            key_patch = convert_model(key_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_key')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(key_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/keys/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Instances
    #########################


    def list_instance_profiles(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List all instance profiles.

        This request lists provisionable [instance
        profiles](https://cloud.ibm.com/docs/vpc?topic=vpc-profiles) in the region. An
        instance profile specifies the performance characteristics and pricing model for
        an instance.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instance/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_profile(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance profile.

        This request retrieves a single instance profile specified by the name in the URL.

        :param str name: The instance profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceProfile` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_profile')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance/profiles/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_instance_templates(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List all instance templates.

        This request lists all instance templates in the region.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceTemplateCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_templates')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instance/templates'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_template(self,
        instance_template_prototype: 'InstanceTemplatePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create an instance template.

        This request creates a new instance template. The prototype object is structured
        in the same way as a retrieved instance template, and contains the information
        necessary to provision a new instance from the template.
        If a `source_template` is specified in the prototype object, its contents are
        copied into the new template prior to copying any other properties provided in the
        prototype object.

        :param InstanceTemplatePrototype instance_template_prototype: The instance
               template prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceTemplate` object
        """

        if instance_template_prototype is None:
            raise ValueError('instance_template_prototype must be provided')
        if isinstance(instance_template_prototype, InstanceTemplatePrototype):
            instance_template_prototype = convert_model(instance_template_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_template')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_template_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instance/templates'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_template(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance template.

        This request deletes the instance template. This operation cannot be reversed.

        :param str id: The instance template identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_template')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance/templates/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_template(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance template.

        This request retrieves a single instance template specified by the identifier in
        the URL.

        :param str id: The instance template identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceTemplate` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_template')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance/templates/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_template(self,
        id: str,
        instance_template_patch: 'InstanceTemplatePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance template.

        This request updates an instance template with the information provided in the
        instance template patch. The instance template patch object is structured in the
        same way as a retrieved instance template and contains only the information to be
        updated.

        :param str id: The instance template identifier.
        :param InstanceTemplatePatch instance_template_patch: The instance template
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceTemplate` object
        """

        if not id:
            raise ValueError('id must be provided')
        if instance_template_patch is None:
            raise ValueError('instance_template_patch must be provided')
        if isinstance(instance_template_patch, InstanceTemplatePatch):
            instance_template_patch = convert_model(instance_template_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_template')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_template_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance/templates/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instances(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        name: str = None,
        vpc_id: str = None,
        vpc_crn: str = None,
        vpc_name: str = None,
        dedicated_host_id: str = None,
        dedicated_host_crn: str = None,
        dedicated_host_name: str = None,
        placement_group_id: str = None,
        placement_group_crn: str = None,
        placement_group_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all instances.

        This request lists all instances in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str vpc_id: (optional) Filters the collection to resources in the
               VPC with the specified identifier.
        :param str vpc_crn: (optional) Filters the collection to resources in the
               VPC with the specified CRN.
        :param str vpc_name: (optional) Filters the collection to resources in the
               VPC with the exact specified name.
        :param str dedicated_host_id: (optional) Filters the collection to
               instances on the dedicated host with the specified identifier.
        :param str dedicated_host_crn: (optional) Filters the collection to
               instances on the dedicated host with the specified CRN.
        :param str dedicated_host_name: (optional) Filters the collection to
               instances on the dedicated host with the specified name.
        :param str placement_group_id: (optional) Filters the collection to
               instances in the placement group with the specified identifier.
        :param str placement_group_crn: (optional) Filters the collection to
               instances in the placement group with the specified CRN.
        :param str placement_group_name: (optional) Filters the collection to
               instances in the placement group with the specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instances')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'vpc.id': vpc_id,
            'vpc.crn': vpc_crn,
            'vpc.name': vpc_name,
            'dedicated_host.id': dedicated_host_id,
            'dedicated_host.crn': dedicated_host_crn,
            'dedicated_host.name': dedicated_host_name,
            'placement_group.id': placement_group_id,
            'placement_group.crn': placement_group_crn,
            'placement_group.name': placement_group_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instances'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance(self,
        instance_prototype: 'InstancePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create an instance.

        This request provisions a new instance from an instance prototype object. The
        prototype object is structured in the same way as a retrieved instance, and
        contains the information necessary to provision the new instance. The instance is
        automatically started.

        :param InstancePrototype instance_prototype: The instance prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if instance_prototype is None:
            raise ValueError('instance_prototype must be provided')
        if isinstance(instance_prototype, InstancePrototype):
            instance_prototype = convert_model(instance_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instances'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance.

        This request deletes an instance. This operation cannot be reversed. Any floating
        IPs associated with the instance's network interfaces are implicitly
        disassociated. All flow log collectors with `auto_delete` set to `true` targeting
        the instance and/or the instance's network interfaces are automatically deleted.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance.

        This request retrieves a single instance specified by the identifier in the URL.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance(self,
        id: str,
        instance_patch: 'InstancePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance.

        This request updates an instance with the information in a provided instance
        patch. The instance patch object is structured in the same way as a retrieved
        instance and contains only the information to be updated.

        :param str id: The instance identifier.
        :param InstancePatch instance_patch: The instance patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Instance` object
        """

        if not id:
            raise ValueError('id must be provided')
        if instance_patch is None:
            raise ValueError('instance_patch must be provided')
        if isinstance(instance_patch, InstancePatch):
            instance_patch = convert_model(instance_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_instance_initialization(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve initialization configuration for an instance.

        This request retrieves configuration variables used to initialize the instance,
        such as SSH keys and the Windows administrator password.

        :param str id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceInitialization` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_initialization')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{id}/initialization'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_action(self,
        instance_id: str,
        type: str,
        *,
        force: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an instance action.

        This request creates a new action which will be queued up to run as soon as any
        pending or running actions have completed.

        :param str instance_id: The instance identifier.
        :param str type: The type of action.
        :param bool force: (optional) If set to true, the action will be forced
               immediately, and all queued actions deleted. Ignored for the start action.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceAction` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if type is None:
            raise ValueError('type must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_action')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'type': type,
            'force': force,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/actions'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def create_instance_console_access_token(self,
        instance_id: str,
        console_type: str,
        *,
        force: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a console access token for an instance.

        This request creates a new single-use console access token for an instance. All
        console configuration is provided at token create time, and the token is
        subsequently used in the `access_token` query parameter for the WebSocket request.
         The access token is only valid for a short period of time, and a maximum of one
        token is valid for a given instance at a time.

        :param str instance_id: The instance identifier.
        :param str console_type: The instance console type for which this token may
               be used.
        :param bool force: (optional) Indicates whether to disconnect an existing
               serial console session as the serial console cannot be shared.  This has no
               effect on VNC consoles.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceConsoleAccessToken` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if console_type is None:
            raise ValueError('console_type must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_console_access_token')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'console_type': console_type,
            'force': force,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/console_access_token'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instance_disks(self,
        instance_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all disks on an instance.

        This request lists all disks on an instance.  A disk is a block device that is
        locally attached to the instance's physical host and is also referred to as
        instance storage. By default, the listed disks are sorted by their `created_at`
        property values, with the newest disk first.

        :param str instance_id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceDiskCollection` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_disks')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/disks'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_disk(self,
        instance_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance disk.

        This request retrieves a single instance disk specified by the identifier in the
        URL.

        :param str instance_id: The instance identifier.
        :param str id: The instance disk identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceDisk` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_disk(self,
        instance_id: str,
        id: str,
        instance_disk_patch: 'InstanceDiskPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance disk.

        This request updates the instance disk with the information in a provided patch.

        :param str instance_id: The instance identifier.
        :param str id: The instance disk identifier.
        :param InstanceDiskPatch instance_disk_patch: The instance disk patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceDisk` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if instance_disk_patch is None:
            raise ValueError('instance_disk_patch must be provided')
        if isinstance(instance_disk_patch, InstanceDiskPatch):
            instance_disk_patch = convert_model(instance_disk_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_disk_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instance_network_interfaces(self,
        instance_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all network interfaces on an instance.

        This request lists all network interfaces on an instance. A network interface is
        an abstract representation of a network interface card and connects an instance to
        a subnet. While each network interface can attach to only one subnet, multiple
        network interfaces can be created to attach to multiple subnets. Multiple
        interfaces may also attach to the same subnet.

        :param str instance_id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterfaceUnpaginatedCollection` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_network_interfaces')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_network_interface(self,
        instance_id: str,
        subnet: 'SubnetIdentity',
        *,
        allow_ip_spoofing: bool = None,
        name: str = None,
        primary_ip: 'NetworkInterfaceIPPrototype' = None,
        security_groups: List['SecurityGroupIdentity'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a network interface on an instance.

        This request creates a new network interface from a network interface prototype
        object. The prototype object is structured in the same way as a retrieved network
        interface, and contains the information necessary to create the new network
        interface. Any subnet in the instance's VPC may be specified, even if it is
        already attached to another network interface. Addresses on the network interface
        must be within the specified subnet's CIDR blocks.

        :param str instance_id: The instance identifier.
        :param SubnetIdentity subnet: The associated subnet.
        :param bool allow_ip_spoofing: (optional) Indicates whether source IP
               spoofing is allowed on this interface. If false, source IP spoofing is
               prevented on this interface. If true, source IP spoofing is allowed on this
               interface.
        :param str name: (optional) The name for network interface. The name must
               not be used by another network interface on the virtual server instance. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param NetworkInterfaceIPPrototype primary_ip: (optional) The primary IP
               address to bind to the network interface. This can be specified using
               an existing reserved IP, or a prototype object for a new reserved IP.
               If an existing reserved IP or a prototype object with an address is
               specified, it must
               be available on the network interface's subnet. Otherwise, an available
               address on the
               subnet will be automatically selected and reserved.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this network interface. If unspecified, the VPC's default
               security group is used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if subnet is None:
            raise ValueError('subnet must be provided')
        subnet = convert_model(subnet)
        if primary_ip is not None:
            primary_ip = convert_model(primary_ip)
        if security_groups is not None:
            security_groups = [convert_model(x) for x in security_groups]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'subnet': subnet,
            'allow_ip_spoofing': allow_ip_spoofing,
            'name': name,
            'primary_ip': primary_ip,
            'security_groups': security_groups,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_network_interface(self,
        instance_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a network interface.

        This request deletes a network interface. This operation cannot be reversed. Any
        floating IPs associated with the network interface are implicitly disassociated.
        All flow log collectors with `auto_delete` set to `true` targeting the network
        interface are automatically deleted. The primary network interface is not allowed
        to be deleted.

        :param str instance_id: The instance identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_network_interface(self,
        instance_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a network interface.

        This request retrieves a single network interface specified by the identifier in
        the URL.

        :param str instance_id: The instance identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_network_interface(self,
        instance_id: str,
        id: str,
        network_interface_patch: 'NetworkInterfacePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a network interface.

        This request updates a network interface with the information provided in a
        network interface patch object. The network interface patch object is structured
        in the same way as a retrieved network interface and needs to contain only the
        information to be updated.

        :param str instance_id: The instance identifier.
        :param str id: The network interface identifier.
        :param NetworkInterfacePatch network_interface_patch: The network interface
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkInterface` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if network_interface_patch is None:
            raise ValueError('network_interface_patch must be provided')
        if isinstance(network_interface_patch, NetworkInterfacePatch):
            network_interface_patch = convert_model(network_interface_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_interface_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instance_network_interface_floating_ips(self,
        instance_id: str,
        network_interface_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all floating IPs associated with a network interface.

        This request lists all floating IPs associated with a network interface.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIPUnpaginatedCollection` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_network_interface_floating_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'network_interface_id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def remove_instance_network_interface_floating_ip(self,
        instance_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Disassociate a floating IP from a network interface.

        This request disassociates the specified floating IP from the specified network
        interface.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='remove_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_network_interface_floating_ip(self,
        instance_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve associated floating IP.

        This request retrieves a specified floating IP address if it is associated with
        the network interface and instance specified in the URL.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def add_instance_network_interface_floating_ip(self,
        instance_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Associate a floating IP with a network interface.

        This request associates the specified floating IP with the specified network
        interface, replacing any existing association. For this request to succeed, the
        existing floating IP must not be required by another resource, such as a public
        gateway. A request body is not required, and if provided, is ignored.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='add_instance_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_instance_network_interface_ips(self,
        instance_id: str,
        network_interface_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all reserved IPs bound to a network interface.

        This request lists all reserved IPs bound to a network interface.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIPCollectionNetworkInterfaceContext` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_network_interface_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'network_interface_id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_network_interface_ip(self,
        instance_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve bound reserved IP.

        This request retrieves the specified reserved IP address if it is bound to the
        network interface and instance specified in the URL.

        :param str instance_id: The instance identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_network_interface_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/network_interfaces/{network_interface_id}/ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_instance_volume_attachments(self,
        instance_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all volumes attachments on an instance.

        This request lists all volume attachments on an instance. A volume attachment
        connects a volume to an instance. Each instance may have many volume attachments
        but each volume attachment connects exactly one instance to exactly one volume.

        :param str instance_id: The instance identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachmentCollection` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_volume_attachments')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/volume_attachments'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_volume_attachment(self,
        instance_id: str,
        volume: 'VolumeAttachmentPrototypeVolume',
        *,
        delete_volume_on_instance_delete: bool = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a volume attachment on an instance.

        This request creates a new volume attachment from a volume attachment prototype
        object, connecting a volume to an instance. For this request to succeed, the
        specified volume must not be busy. The prototype object is structured in the same
        way as a retrieved volume attachment, and contains the information necessary to
        create the new volume attachment.

        :param str instance_id: The instance identifier.
        :param VolumeAttachmentPrototypeVolume volume: An existing volume to attach
               to the instance, or a prototype object for a new volume.
        :param bool delete_volume_on_instance_delete: (optional) Indicates whether
               deleting the instance will also delete the attached volume.
        :param str name: (optional) The name for this volume attachment. The name
               must not be used by another volume attachment on the instance. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if volume is None:
            raise ValueError('volume must be provided')
        volume = convert_model(volume)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'volume': volume,
            'delete_volume_on_instance_delete': delete_volume_on_instance_delete,
            'name': name,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id']
        path_param_values = self.encode_path_vars(instance_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/volume_attachments'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_volume_attachment(self,
        instance_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a volume attachment.

        This request deletes a volume attachment. This operation cannot be reversed, but a
        new volume attachment may subsequently be created for the volume.  For this
        request to succeed, the volume must not be busy.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/volume_attachments/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_volume_attachment(self,
        instance_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a volume attachment.

        This request retrieves a single volume attachment specified by the identifier in
        the URL.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/volume_attachments/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_volume_attachment(self,
        instance_id: str,
        id: str,
        volume_attachment_patch: 'VolumeAttachmentPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a volume attachment.

        This request updates a volume attachment with the information provided in a volume
        attachment patch object. The volume attachment patch object is structured in the
        same way as a retrieved volume attachment and needs to contain only the
        information to be updated.

        :param str instance_id: The instance identifier.
        :param str id: The volume attachment identifier.
        :param VolumeAttachmentPatch volume_attachment_patch: The volume attachment
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeAttachment` object
        """

        if not instance_id:
            raise ValueError('instance_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if volume_attachment_patch is None:
            raise ValueError('volume_attachment_patch must be provided')
        if isinstance(volume_attachment_patch, VolumeAttachmentPatch):
            volume_attachment_patch = convert_model(volume_attachment_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_volume_attachment')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(volume_attachment_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_id', 'id']
        path_param_values = self.encode_path_vars(instance_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instances/{instance_id}/volume_attachments/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Instance groups
    #########################


    def list_instance_groups(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all instance groups.

        This request lists all instance groups in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_groups')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instance_groups'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_group(self,
        instance_template: 'InstanceTemplateIdentity',
        subnets: List['SubnetIdentity'],
        *,
        application_port: int = None,
        load_balancer: 'LoadBalancerIdentity' = None,
        load_balancer_pool: 'LoadBalancerPoolIdentity' = None,
        membership_count: int = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an instance group.

        This request creates a new instance group.

        :param InstanceTemplateIdentity instance_template: Instance template to use
               when creating new instances.
               Instance groups are not compatible with instance templates that specify
               `true` for
               `default_trusted_profile.auto_link`.
        :param List[SubnetIdentity] subnets: The subnets to use when creating new
               instances.
        :param int application_port: (optional) The port to use for new load
               balancer pool members created by this instance group.
               This property must be specified if and only if `load_balancer_pool` has
               been specified.
        :param LoadBalancerIdentity load_balancer: (optional) The load balancer
               associated with the specified load balancer pool.
               Required if `load_balancer_pool` is specified.
               At present, only load balancers in the `application` family are supported.
        :param LoadBalancerPoolIdentity load_balancer_pool: (optional) If
               specified, the load balancer pool this instance group will manage. A pool
               member
               will be created for each instance created by this group.
               If specified, `load_balancer` and `application_port` must also be
               specified.
        :param int membership_count: (optional) The number of instances in the
               instance group.
        :param str name: (optional) The name for this instance group. The name must
               not be used by another instance group in the region. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroup` object
        """

        if instance_template is None:
            raise ValueError('instance_template must be provided')
        if subnets is None:
            raise ValueError('subnets must be provided')
        instance_template = convert_model(instance_template)
        subnets = [convert_model(x) for x in subnets]
        if load_balancer is not None:
            load_balancer = convert_model(load_balancer)
        if load_balancer_pool is not None:
            load_balancer_pool = convert_model(load_balancer_pool)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'instance_template': instance_template,
            'subnets': subnets,
            'application_port': application_port,
            'load_balancer': load_balancer,
            'load_balancer_pool': load_balancer_pool,
            'membership_count': membership_count,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/instance_groups'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance group.

        This request deletes an instance group. This operation cannot be reversed. Any
        instances associated with the group will be deleted.

        :param str id: The instance group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance group.

        This request retrieves a single instance group specified by identifier in the URL.

        :param str id: The instance group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_group(self,
        id: str,
        instance_group_patch: 'InstanceGroupPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance group.

        This request updates an instance group with the information provided instance
        group patch. The instance group patch object is structured in the same way as a
        retrieved instance group and contains only the information to be updated.

        :param str id: The instance group identifier.
        :param InstanceGroupPatch instance_group_patch: The instance group patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        if instance_group_patch is None:
            raise ValueError('instance_group_patch must be provided')
        if isinstance(instance_group_patch, InstanceGroupPatch):
            instance_group_patch = convert_model(instance_group_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_load_balancer(self,
        instance_group_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance group load balancer.

        This request unbinds the instance group from the load balancer pool, and deletes
        the load balancer pool members.

        :param str instance_group_id: The instance group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_load_balancer')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id']
        path_param_values = self.encode_path_vars(instance_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/load_balancer'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_instance_group_managers(self,
        instance_group_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all managers for an instance group.

        This request lists all managers for an instance group.

        :param str instance_group_id: The instance group identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerCollection` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_group_managers')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id']
        path_param_values = self.encode_path_vars(instance_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_group_manager(self,
        instance_group_id: str,
        instance_group_manager_prototype: 'InstanceGroupManagerPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a manager for an instance group.

        This request creates a new instance group manager.

        :param str instance_group_id: The instance group identifier.
        :param InstanceGroupManagerPrototype instance_group_manager_prototype: The
               instance group manager prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManager` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if instance_group_manager_prototype is None:
            raise ValueError('instance_group_manager_prototype must be provided')
        if isinstance(instance_group_manager_prototype, InstanceGroupManagerPrototype):
            instance_group_manager_prototype = convert_model(instance_group_manager_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_group_manager')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id']
        path_param_values = self.encode_path_vars(instance_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_manager(self,
        instance_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance group manager.

        This request deletes an instance group manager. This operation cannot be reversed.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group manager identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_manager')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_group_manager(self,
        instance_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance group manager.

        This request retrieves a single instance group manager specified by identifier in
        the URL.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group manager identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManager` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_group_manager')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_group_manager(self,
        instance_group_id: str,
        id: str,
        instance_group_manager_patch: 'InstanceGroupManagerPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance group manager.

        This request updates an instance group manager with the information provided
        instance group manager patch.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group manager identifier.
        :param InstanceGroupManagerPatch instance_group_manager_patch: The instance
               group manager patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManager` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if instance_group_manager_patch is None:
            raise ValueError('instance_group_manager_patch must be provided')
        if isinstance(instance_group_manager_patch, InstanceGroupManagerPatch):
            instance_group_manager_patch = convert_model(instance_group_manager_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_group_manager')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instance_group_manager_actions(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all actions for an instance group manager.

        This request lists all instance group actions for an instance group manager.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerActionsCollection` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_group_manager_actions')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_group_manager_action(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        instance_group_manager_action_prototype: 'InstanceGroupManagerActionPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create an instance group manager action.

        This request creates a new instance group manager action.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param InstanceGroupManagerActionPrototype
               instance_group_manager_action_prototype: The instance group manager action
               prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerAction` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if instance_group_manager_action_prototype is None:
            raise ValueError('instance_group_manager_action_prototype must be provided')
        if isinstance(instance_group_manager_action_prototype, InstanceGroupManagerActionPrototype):
            instance_group_manager_action_prototype = convert_model(instance_group_manager_action_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_group_manager_action')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_action_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_manager_action(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete specified instance group manager action.

        This request deletes an instance group manager action. This operation cannot be
        reversed.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager action identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_manager_action')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_group_manager_action(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve specified instance group manager action.

        This request retrieves a single instance group manager action specified by
        identifier in the URL.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager action identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerAction` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_group_manager_action')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_group_manager_action(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        instance_group_manager_action_patch: 'InstanceGroupManagerActionPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update specified instance group manager action.

        This request updates an instance group manager action.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager action identifier.
        :param InstanceGroupManagerActionPatch instance_group_manager_action_patch:
               The instance group manager action patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerAction` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if instance_group_manager_action_patch is None:
            raise ValueError('instance_group_manager_action_patch must be provided')
        if isinstance(instance_group_manager_action_patch, InstanceGroupManagerActionPatch):
            instance_group_manager_action_patch = convert_model(instance_group_manager_action_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_group_manager_action')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_action_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/actions/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_instance_group_manager_policies(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all policies for an instance group manager.

        This request lists all policies for an instance group manager.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerPolicyCollection` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_group_manager_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_instance_group_manager_policy(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        instance_group_manager_policy_prototype: 'InstanceGroupManagerPolicyPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a policy for an instance group manager.

        This request creates a new instance group manager policy.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param InstanceGroupManagerPolicyPrototype
               instance_group_manager_policy_prototype: The instance group manager policy
               prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerPolicy` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if instance_group_manager_policy_prototype is None:
            raise ValueError('instance_group_manager_policy_prototype must be provided')
        if isinstance(instance_group_manager_policy_prototype, InstanceGroupManagerPolicyPrototype):
            instance_group_manager_policy_prototype = convert_model(instance_group_manager_policy_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_instance_group_manager_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_policy_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_manager_policy(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance group manager policy.

        This request deletes an instance group manager policy. This operation cannot be
        reversed.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_manager_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_group_manager_policy(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance group manager policy.

        This request retrieves a single instance group manager policy specified by
        identifier in the URL.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerPolicy` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_group_manager_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_group_manager_policy(self,
        instance_group_id: str,
        instance_group_manager_id: str,
        id: str,
        instance_group_manager_policy_patch: 'InstanceGroupManagerPolicyPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance group manager policy.

        This request updates an instance group manager policy.

        :param str instance_group_id: The instance group identifier.
        :param str instance_group_manager_id: The instance group manager
               identifier.
        :param str id: The instance group manager policy identifier.
        :param InstanceGroupManagerPolicyPatch instance_group_manager_policy_patch:
               The instance group manager policy patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupManagerPolicy` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not instance_group_manager_id:
            raise ValueError('instance_group_manager_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if instance_group_manager_policy_patch is None:
            raise ValueError('instance_group_manager_policy_patch must be provided')
        if isinstance(instance_group_manager_policy_patch, InstanceGroupManagerPolicyPatch):
            instance_group_manager_policy_patch = convert_model(instance_group_manager_policy_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_group_manager_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_manager_policy_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'instance_group_manager_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, instance_group_manager_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/managers/{instance_group_manager_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_memberships(self,
        instance_group_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete all memberships from an instance group.

        This request deletes all memberships of an instance group. This operation cannot
        be reversed. reversed. Any memberships that have
        `delete_instance_on_membership_delete` set to `true` will also have their
        instances deleted.

        :param str instance_group_id: The instance group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_memberships')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id']
        path_param_values = self.encode_path_vars(instance_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/memberships'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_instance_group_memberships(self,
        instance_group_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all memberships for an instance group.

        This request lists all instance group memberships for an instance group.

        :param str instance_group_id: The instance group identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupMembershipCollection` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_instance_group_memberships')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id']
        path_param_values = self.encode_path_vars(instance_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/memberships'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_instance_group_membership(self,
        instance_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an instance group membership.

        This request deletes a memberships of an instance group. This operation cannot be
        reversed. reversed. If the membership has `delete_instance_on_membership_delete`
        set to `true`, the instance will also be deleted.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group membership identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_instance_group_membership')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/memberships/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_instance_group_membership(self,
        instance_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an instance group membership.

        This request retrieves a single instance group membership specified by identifier
        in the URL.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group membership identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupMembership` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_instance_group_membership')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/memberships/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_instance_group_membership(self,
        instance_group_id: str,
        id: str,
        instance_group_membership_patch: 'InstanceGroupMembershipPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an instance group membership.

        This request updates an instance group membership with the information provided
        instance group membership patch.

        :param str instance_group_id: The instance group identifier.
        :param str id: The instance group membership identifier.
        :param InstanceGroupMembershipPatch instance_group_membership_patch: The
               instance group membership patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `InstanceGroupMembership` object
        """

        if not instance_group_id:
            raise ValueError('instance_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if instance_group_membership_patch is None:
            raise ValueError('instance_group_membership_patch must be provided')
        if isinstance(instance_group_membership_patch, InstanceGroupMembershipPatch):
            instance_group_membership_patch = convert_model(instance_group_membership_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_instance_group_membership')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(instance_group_membership_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['instance_group_id', 'id']
        path_param_values = self.encode_path_vars(instance_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/instance_groups/{instance_group_id}/memberships/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Dedicated hosts
    #########################


    def list_dedicated_host_groups(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        zone_name: str = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all dedicated host groups.

        This request lists all dedicated host groups in the region. Host groups are a
        collection of dedicated hosts for placement of instances. Each dedicated host must
        belong to one and only one group. Host groups do not span zones.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostGroupCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_dedicated_host_groups')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'zone.name': zone_name,
            'name': name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/dedicated_host/groups'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_dedicated_host_group(self,
        *,
        class_: str = None,
        family: str = None,
        zone: 'ZoneIdentity' = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a dedicated host group.

        This request creates a new dedicated host group.

        :param str class_: (optional) The dedicated host profile class for hosts in
               this group.
        :param str family: (optional) The dedicated host profile family for hosts
               in this group.
        :param ZoneIdentity zone: (optional) The zone this dedicated host group
               will reside in.
        :param str name: (optional) The name for this dedicated host group. The
               name must not be used by another dedicated host group in the region. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostGroup` object
        """

        if zone is not None:
            zone = convert_model(zone)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_dedicated_host_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'class': class_,
            'family': family,
            'zone': zone,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/dedicated_host/groups'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_dedicated_host_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a dedicated host group.

        This request deletes a dedicated host group.

        :param str id: The dedicated host group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_dedicated_host_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_host/groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_dedicated_host_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a dedicated host group.

        This request retrieves a single dedicated host group specified by the identifier
        in the URL.

        :param str id: The dedicated host group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_dedicated_host_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_host/groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_dedicated_host_group(self,
        id: str,
        dedicated_host_group_patch: 'DedicatedHostGroupPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a dedicated host group.

        This request updates a dedicated host group with the information in a provided
        dedicated host group patch. The dedicated host group patch object is structured in
        the same way as a retrieved dedicated host group and contains only the information
        to be updated.

        :param str id: The dedicated host group identifier.
        :param DedicatedHostGroupPatch dedicated_host_group_patch: The dedicated
               host group patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        if dedicated_host_group_patch is None:
            raise ValueError('dedicated_host_group_patch must be provided')
        if isinstance(dedicated_host_group_patch, DedicatedHostGroupPatch):
            dedicated_host_group_patch = convert_model(dedicated_host_group_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_dedicated_host_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(dedicated_host_group_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_host/groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_dedicated_host_profiles(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all dedicated host profiles.

        This request lists provisionable [dedicated host
        profiles](https://cloud.ibm.com/docs/vpc?topic=vpc-dh-profiles) in the region. A
        dedicated host profile specifies the hardware characteristics for a dedicated
        host.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_dedicated_host_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/dedicated_host/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_dedicated_host_profile(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a dedicated host profile.

        This request retrieves a single dedicated host profile specified by the name in
        the URL.

        :param str name: The dedicated host profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostProfile` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_dedicated_host_profile')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_host/profiles/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_dedicated_hosts(self,
        *,
        dedicated_host_group_id: str = None,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        zone_name: str = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all dedicated hosts.

        This request lists all dedicated hosts in the region.

        :param str dedicated_host_group_id: (optional) Filters the collection to
               dedicated host groups with the specified identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_dedicated_hosts')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'dedicated_host_group.id': dedicated_host_group_id,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'zone.name': zone_name,
            'name': name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/dedicated_hosts'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_dedicated_host(self,
        dedicated_host_prototype: 'DedicatedHostPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a dedicated host.

        This request creates a new dedicated host.

        :param DedicatedHostPrototype dedicated_host_prototype: The dedicated host
               prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHost` object
        """

        if dedicated_host_prototype is None:
            raise ValueError('dedicated_host_prototype must be provided')
        if isinstance(dedicated_host_prototype, DedicatedHostPrototype):
            dedicated_host_prototype = convert_model(dedicated_host_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_dedicated_host')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(dedicated_host_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/dedicated_hosts'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_dedicated_host_disks(self,
        dedicated_host_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all disks on a dedicated host.

        This request lists all disks on a dedicated host.  A disk is a physical device
        that is locally attached to the compute node. By default, the listed disks are
        sorted by their
        `created_at` property values, with the newest disk first.

        :param str dedicated_host_id: The dedicated host identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostDiskCollection` object
        """

        if not dedicated_host_id:
            raise ValueError('dedicated_host_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_dedicated_host_disks')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['dedicated_host_id']
        path_param_values = self.encode_path_vars(dedicated_host_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{dedicated_host_id}/disks'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_dedicated_host_disk(self,
        dedicated_host_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a dedicated host disk.

        This request retrieves a single dedicated host disk specified by the identifier in
        the URL.

        :param str dedicated_host_id: The dedicated host identifier.
        :param str id: The dedicated host disk identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostDisk` object
        """

        if not dedicated_host_id:
            raise ValueError('dedicated_host_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_dedicated_host_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['dedicated_host_id', 'id']
        path_param_values = self.encode_path_vars(dedicated_host_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{dedicated_host_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_dedicated_host_disk(self,
        dedicated_host_id: str,
        id: str,
        dedicated_host_disk_patch: 'DedicatedHostDiskPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a dedicated host disk.

        This request updates the dedicated host disk with the information in a provided
        patch.

        :param str dedicated_host_id: The dedicated host identifier.
        :param str id: The dedicated host disk identifier.
        :param DedicatedHostDiskPatch dedicated_host_disk_patch: The dedicated host
               disk patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHostDisk` object
        """

        if not dedicated_host_id:
            raise ValueError('dedicated_host_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if dedicated_host_disk_patch is None:
            raise ValueError('dedicated_host_disk_patch must be provided')
        if isinstance(dedicated_host_disk_patch, DedicatedHostDiskPatch):
            dedicated_host_disk_patch = convert_model(dedicated_host_disk_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_dedicated_host_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(dedicated_host_disk_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['dedicated_host_id', 'id']
        path_param_values = self.encode_path_vars(dedicated_host_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{dedicated_host_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_dedicated_host(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a dedicated host.

        This request deletes a dedicated host.

        :param str id: The dedicated host identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_dedicated_host')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_dedicated_host(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a dedicated host.

        This request retrieves a single dedicated host specified by the identifiers in the
        URL.

        :param str id: The dedicated host identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHost` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_dedicated_host')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_dedicated_host(self,
        id: str,
        dedicated_host_patch: 'DedicatedHostPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a dedicated host.

        This request updates a dedicated host with the information in a provided dedicated
        host patch. The dedicated host patch object is structured in the same way as a
        retrieved dedicated host and contains only the information to be updated.

        :param str id: The dedicated host identifier.
        :param DedicatedHostPatch dedicated_host_patch: The dedicated host patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `DedicatedHost` object
        """

        if not id:
            raise ValueError('id must be provided')
        if dedicated_host_patch is None:
            raise ValueError('dedicated_host_patch must be provided')
        if isinstance(dedicated_host_patch, DedicatedHostPatch):
            dedicated_host_patch = convert_model(dedicated_host_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_dedicated_host')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(dedicated_host_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/dedicated_hosts/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Backup policies
    #########################


    def list_backup_policies(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        name: str = None,
        tag: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all backup policies.

        This request lists all backup policies in the region. Backup policies control
        which sources are selected for backup and include a set of backup policy plans
        that provide the backup schedules and deletion triggers.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str tag: (optional) Filters the collection to resources with the
               exact tag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_backup_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'tag': tag,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/backup_policies'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_backup_policy(self,
        *,
        match_user_tags: List[str] = None,
        match_resource_types: List[str] = None,
        name: str = None,
        plans: List['BackupPolicyPlanPrototype'] = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a backup policy.

        This request creates a new backup policy from a backup policy prototype object.
        The prototype object is structured in the same way as a retrieved backup policy,
        and contains the information necessary to create the new backup policy.

        :param List[str] match_user_tags: (optional) The user tags this backup
               policy applies to. Resources that have both a matching user tag and a
               matching type will be subject to the backup policy.
        :param List[str] match_resource_types: (optional) A resource type this
               backup policy applies to. Resources that have both a matching type and a
               matching user tag will be subject to the backup policy.
        :param str name: (optional) The name for this backup policy. The name must
               not be used by another backup policy in the region. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param List[BackupPolicyPlanPrototype] plans: (optional) The prototype
               objects for backup plans to be created for this backup policy.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicy` object
        """

        if plans is not None:
            plans = [convert_model(x) for x in plans]
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_backup_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'match_user_tags': match_user_tags,
            'match_resource_types': match_resource_types,
            'name': name,
            'plans': plans,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/backup_policies'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_backup_policy_jobs(self,
        backup_policy_id: str,
        *,
        status: str = None,
        backup_policy_plan_id: str = None,
        start: str = None,
        limit: int = None,
        sort: str = None,
        source_id: str = None,
        target_snapshots_id: str = None,
        target_snapshots_crn: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all jobs for a backup policy.

        This request retrieves all jobs for a backup policy. A backup job represents the
        execution of a backup policy plan for a resource matching the policy's criteria.

        :param str backup_policy_id: The backup policy identifier.
        :param str status: (optional) Filters the collection to backup policy jobs
               with the specified status.
        :param str backup_policy_plan_id: (optional) Filters the collection to
               backup policy jobs with the backup plan with the specified identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param str source_id: (optional) Filters the collection to backup policy
               jobs with a source with the specified identifier.
        :param str target_snapshots_id: (optional) Filters the collection to
               resources with the target snapshot with the specified identifier.
        :param str target_snapshots_crn: (optional) Filters the collection to
               backup policy jobs with the target snapshot with the specified CRN.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyJobCollection` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_backup_policy_jobs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'status': status,
            'backup_policy_plan.id': backup_policy_plan_id,
            'start': start,
            'limit': limit,
            'sort': sort,
            'source.id': source_id,
            'target_snapshots[].id': target_snapshots_id,
            'target_snapshots[].crn': target_snapshots_crn,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id']
        path_param_values = self.encode_path_vars(backup_policy_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/jobs'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_backup_policy_job(self,
        backup_policy_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a backup policy job.

        This request retrieves a single backup policy job specified by the identifier in
        the URL.

        :param str backup_policy_id: The backup policy identifier.
        :param str id: The backup policy job identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyJob` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_backup_policy_job')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id', 'id']
        path_param_values = self.encode_path_vars(backup_policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/jobs/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_backup_policy_plans(self,
        backup_policy_id: str,
        *,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all plans for a backup policy.

        This request retrieves all plans for a backup policy. Backup plans provide the
        backup schedule and deletion triggers.

        :param str backup_policy_id: The backup policy identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyPlanCollection` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_backup_policy_plans')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'name': name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id']
        path_param_values = self.encode_path_vars(backup_policy_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/plans'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_backup_policy_plan(self,
        backup_policy_id: str,
        cron_spec: str,
        *,
        active: bool = None,
        attach_user_tags: List[str] = None,
        clone_policy: 'BackupPolicyPlanClonePolicyPrototype' = None,
        copy_user_tags: bool = None,
        deletion_trigger: 'BackupPolicyPlanDeletionTriggerPrototype' = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a plan for a backup policy.

        This request creates a new backup policy plan from a backup policy plan prototype
        object. The prototype object is structured in the same way as a retrieved backup
        policy plan, and contains the information necessary to create the new backup
        policy plan.
        Backups created by this plan will use the resource group of the source being
        backed up.
        Backups created by this plan will use the plan's name truncated to 46 characters,
        followed by a unique 16-character suffix.

        :param str backup_policy_id: The backup policy identifier.
        :param str cron_spec: The cron specification for the backup schedule. The
               backup policy jobs
               (which create and delete backups for this plan) will not start until this
               time, and may start for up to 90 minutes after this time.
               All backup schedules for plans in the same policy must be at least an hour
               apart.
        :param bool active: (optional) Indicates whether the plan is active.
        :param List[str] attach_user_tags: (optional) User tags to attach to each
               backup (snapshot) created by this plan. If unspecified, no user tags will
               be attached.
        :param BackupPolicyPlanClonePolicyPrototype clone_policy: (optional)
        :param bool copy_user_tags: (optional) Indicates whether to copy the
               source's user tags to the created backups (snapshots).
        :param BackupPolicyPlanDeletionTriggerPrototype deletion_trigger:
               (optional)
        :param str name: (optional) The name for this backup policy plan. The name
               must not be used by another plan for the backup policy. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyPlan` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        if cron_spec is None:
            raise ValueError('cron_spec must be provided')
        if clone_policy is not None:
            clone_policy = convert_model(clone_policy)
        if deletion_trigger is not None:
            deletion_trigger = convert_model(deletion_trigger)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_backup_policy_plan')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'cron_spec': cron_spec,
            'active': active,
            'attach_user_tags': attach_user_tags,
            'clone_policy': clone_policy,
            'copy_user_tags': copy_user_tags,
            'deletion_trigger': deletion_trigger,
            'name': name,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id']
        path_param_values = self.encode_path_vars(backup_policy_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/plans'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_backup_policy_plan(self,
        backup_policy_id: str,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a backup policy plan.

        This request deletes a backup policy plan. This operation cannot be reversed. Any
        backups that have been created by the plan will remain but will no longer be
        subject to the plan's deletion trigger. Any running jobs associated with the plan
        will run to completion before the plan is deleted.
        If the request is accepted, the backup policy plan `status` will be set to
        `deleting`. Once deletion processing completes, the backup policy plan will no
        longer be retrievable.

        :param str backup_policy_id: The backup policy identifier.
        :param str id: The backup policy plan identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyPlan` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_backup_policy_plan')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id', 'id']
        path_param_values = self.encode_path_vars(backup_policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/plans/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_backup_policy_plan(self,
        backup_policy_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a backup policy plan.

        This request retrieves a single backup policy plan specified by the identifier in
        the URL.

        :param str backup_policy_id: The backup policy identifier.
        :param str id: The backup policy plan identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyPlan` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_backup_policy_plan')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id', 'id']
        path_param_values = self.encode_path_vars(backup_policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/plans/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_backup_policy_plan(self,
        backup_policy_id: str,
        id: str,
        backup_policy_plan_patch: 'BackupPolicyPlanPatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a backup policy plan.

        This request updates a backup policy plan with the information in a provided plan
        patch. The plan patch object is structured in the same way as a retrieved backup
        policy plan and can contains only the information to be updated.

        :param str backup_policy_id: The backup policy identifier.
        :param str id: The backup policy plan identifier.
        :param BackupPolicyPlanPatch backup_policy_plan_patch: The backup policy
               plan patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicyPlan` object
        """

        if not backup_policy_id:
            raise ValueError('backup_policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if backup_policy_plan_patch is None:
            raise ValueError('backup_policy_plan_patch must be provided')
        if isinstance(backup_policy_plan_patch, BackupPolicyPlanPatch):
            backup_policy_plan_patch = convert_model(backup_policy_plan_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_backup_policy_plan')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(backup_policy_plan_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['backup_policy_id', 'id']
        path_param_values = self.encode_path_vars(backup_policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{backup_policy_id}/plans/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_backup_policy(self,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a backup policy.

        This request deletes a backup policy. This operation cannot be reversed.
        If the request is accepted, the backup policy `status` will be set to `deleting`.
        Once deletion processing completes, the backup policy will no longer be
        retrievable.

        :param str id: The backup policy identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_backup_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_backup_policy(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a backup policy.

        This request retrieves a single backup policy specified by the identifier in the
        URL.

        :param str id: The backup policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_backup_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_backup_policy(self,
        id: str,
        backup_policy_patch: 'BackupPolicyPatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a backup policy.

        This request updates a backup policy with the information in a provided backup
        policy patch. The backup policy patch object is structured in the same way as a
        retrieved backup policy and contains only the information to be updated.

        :param str id: The backup policy identifier.
        :param BackupPolicyPatch backup_policy_patch: The backup policy patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BackupPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        if backup_policy_patch is None:
            raise ValueError('backup_policy_patch must be provided')
        if isinstance(backup_policy_patch, BackupPolicyPatch):
            backup_policy_patch = convert_model(backup_policy_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_backup_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(backup_policy_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/backup_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Placement groups
    #########################


    def list_placement_groups(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all placement groups.

        This request lists all placement groups in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PlacementGroupCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_placement_groups')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/placement_groups'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_placement_group(self,
        strategy: str,
        *,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a placement group.

        This request creates a new placement group.

        :param str strategy: The strategy for this placement group
               - `host_spread`: place on different compute hosts
               - `power_spread`: place on compute hosts that use different power sources
               The enumerated values for this property may expand in the future. When
               processing this property, check for and log unknown values. Optionally halt
               processing and surface the error, or bypass the placement group on which
               the unexpected strategy was encountered.
        :param str name: (optional) The name for this placement group. The name
               must not be used by another placement group in the region. If unspecified,
               the name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PlacementGroup` object
        """

        if strategy is None:
            raise ValueError('strategy must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_placement_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'strategy': strategy,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/placement_groups'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_placement_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a placement group.

        This request deletes a placement group. This operation cannot be reversed. For
        this request to succeed, the placement group must not be associated with an
        instance.

        :param str id: The placement group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_placement_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/placement_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_placement_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a placement group.

        This request retrieves a single placement group specified by identifier in the
        URL.

        :param str id: The placement group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PlacementGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_placement_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/placement_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_placement_group(self,
        id: str,
        placement_group_patch: 'PlacementGroupPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a placement group.

        This request updates a placement group with the information provided placement
        group patch. The placement group patch object is structured in the same way as a
        retrieved placement group and contains only the information to be updated.

        :param str id: The placement group identifier.
        :param PlacementGroupPatch placement_group_patch: The placement group
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PlacementGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        if placement_group_patch is None:
            raise ValueError('placement_group_patch must be provided')
        if isinstance(placement_group_patch, PlacementGroupPatch):
            placement_group_patch = convert_model(placement_group_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_placement_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(placement_group_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/placement_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Bare metal servers
    #########################


    def list_bare_metal_server_profiles(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all bare metal server profiles.

        This request lists all [bare metal server
        profiles](https://cloud.ibm.com/docs/vpc?topic=vpc-bare-metal-servers-profile)
        available in the region. A bare metal server profile specifies the performance
        characteristics and pricing model for a bare metal server.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_server_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/bare_metal_server/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_profile(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a bare metal server profile.

        This request retrieves a single bare metal server profile specified by the name in
        the URL.

        :param str name: The bare metal server profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerProfile` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_profile')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_server/profiles/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_bare_metal_servers(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        name: str = None,
        vpc_id: str = None,
        vpc_crn: str = None,
        vpc_name: str = None,
        network_interfaces_subnet_id: str = None,
        network_interfaces_subnet_crn: str = None,
        network_interfaces_subnet_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all bare metal servers.

        This request lists all bare metal servers in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str vpc_id: (optional) Filters the collection to resources in the
               VPC with the specified identifier.
        :param str vpc_crn: (optional) Filters the collection to resources in the
               VPC with the specified CRN.
        :param str vpc_name: (optional) Filters the collection to resources in the
               VPC with the exact specified name.
        :param str network_interfaces_subnet_id: (optional) Filters the collection
               to bare metal servers on the subnet with the specified identifier.
        :param str network_interfaces_subnet_crn: (optional) Filters the collection
               to bare metal servers on the subnet with the specified CRN.
        :param str network_interfaces_subnet_name: (optional) Filters the
               collection to bare metal servers on the subnet with the specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_servers')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'vpc.id': vpc_id,
            'vpc.crn': vpc_crn,
            'vpc.name': vpc_name,
            'network_interfaces.subnet.id': network_interfaces_subnet_id,
            'network_interfaces.subnet.crn': network_interfaces_subnet_crn,
            'network_interfaces.subnet.name': network_interfaces_subnet_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/bare_metal_servers'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_bare_metal_server(self,
        initialization: 'BareMetalServerInitializationPrototype',
        primary_network_interface: 'BareMetalServerPrimaryNetworkInterfacePrototype',
        profile: 'BareMetalServerProfileIdentity',
        zone: 'ZoneIdentity',
        *,
        enable_secure_boot: bool = None,
        name: str = None,
        network_interfaces: List['BareMetalServerNetworkInterfacePrototype'] = None,
        resource_group: 'ResourceGroupIdentity' = None,
        trusted_platform_module: 'BareMetalServerTrustedPlatformModulePrototype' = None,
        vpc: 'VPCIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a bare metal server.

        This request provisions a new bare metal server from a prototype object. The
        prototype object is structured in the same way as a retrieved bare metal server,
        and contains the information necessary to provision the new bare metal server. The
        bare metal server is automatically started.

        :param BareMetalServerInitializationPrototype initialization:
        :param BareMetalServerPrimaryNetworkInterfacePrototype
               primary_network_interface: Primary network interface for the bare metal
               server.
        :param BareMetalServerProfileIdentity profile: The
               [profile](https://cloud.ibm.com/docs/vpc?topic=vpc-bare-metal-servers-profile)
               to use for this bare metal server.
        :param ZoneIdentity zone: The zone this bare metal server will reside in.
        :param bool enable_secure_boot: (optional) Indicates whether secure boot is
               enabled. If enabled, the image must support secure boot or the server will
               fail to boot.
        :param str name: (optional) The name for this bare metal server. The name
               must not be used by another bare metal server in the region. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
               The system hostname will be based on this name.
        :param List[BareMetalServerNetworkInterfacePrototype] network_interfaces:
               (optional) The additional network interfaces to create for the bare metal
               server.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param BareMetalServerTrustedPlatformModulePrototype
               trusted_platform_module: (optional)
        :param VPCIdentity vpc: (optional) The VPC this bare metal server will
               reside in.
               If specified, it must match the VPC for the subnets of the server's network
               interfaces.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServer` object
        """

        if initialization is None:
            raise ValueError('initialization must be provided')
        if primary_network_interface is None:
            raise ValueError('primary_network_interface must be provided')
        if profile is None:
            raise ValueError('profile must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        initialization = convert_model(initialization)
        primary_network_interface = convert_model(primary_network_interface)
        profile = convert_model(profile)
        zone = convert_model(zone)
        if network_interfaces is not None:
            network_interfaces = [convert_model(x) for x in network_interfaces]
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if trusted_platform_module is not None:
            trusted_platform_module = convert_model(trusted_platform_module)
        if vpc is not None:
            vpc = convert_model(vpc)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'initialization': initialization,
            'primary_network_interface': primary_network_interface,
            'profile': profile,
            'zone': zone,
            'enable_secure_boot': enable_secure_boot,
            'name': name,
            'network_interfaces': network_interfaces,
            'resource_group': resource_group,
            'trusted_platform_module': trusted_platform_module,
            'vpc': vpc,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/bare_metal_servers'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def create_bare_metal_server_console_access_token(self,
        bare_metal_server_id: str,
        console_type: str,
        *,
        force: bool = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a console access token for a bare metal server.

        This request creates a new single-use console access token for a bare metal
        server. All console configuration is provided at token create time, and the token
        is subsequently used in the `access_token` query parameter for the WebSocket
        request.  The access token is only valid for a short period of time, and a maximum
        of one token is valid for a given bare metal server at a time.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str console_type: The bare metal server console type for which this
               token may be used
               Must be `serial` for bare metal servers with a `cpu.architecture` of
               `s390x`.
        :param bool force: (optional) Indicates whether to disconnect an existing
               serial console session as the serial console cannot be shared.  This has no
               effect on VNC consoles.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerConsoleAccessToken` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if console_type is None:
            raise ValueError('console_type must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_bare_metal_server_console_access_token')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'console_type': console_type,
            'force': force,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/console_access_token'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_bare_metal_server_disks(self,
        bare_metal_server_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all disks on a bare metal server.

        This request lists all disks on a bare metal server.  A disk is a block device
        that is locally attached to the physical server.  By default, the listed disks are
        sorted by their `created_at` property values, with the newest disk first.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerDiskCollection` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_server_disks')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/disks'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_disk(self,
        bare_metal_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a bare metal server disk.

        This request retrieves a single disk specified by the identifier in the URL.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str id: The bare metal server disk identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerDisk` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_bare_metal_server_disk(self,
        bare_metal_server_id: str,
        id: str,
        bare_metal_server_disk_patch: 'BareMetalServerDiskPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a bare metal server disk.

        This request updates the bare metal server disk with the information in a provided
        patch.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str id: The bare metal server disk identifier.
        :param BareMetalServerDiskPatch bare_metal_server_disk_patch: The bare
               metal server disk patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerDisk` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if bare_metal_server_disk_patch is None:
            raise ValueError('bare_metal_server_disk_patch must be provided')
        if isinstance(bare_metal_server_disk_patch, BareMetalServerDiskPatch):
            bare_metal_server_disk_patch = convert_model(bare_metal_server_disk_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_bare_metal_server_disk')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(bare_metal_server_disk_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/disks/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_bare_metal_server_network_interfaces(self,
        bare_metal_server_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all network interfaces on a bare metal server.

        This request lists all network interfaces on a bare metal server. A network
        interface is an abstract representation of a network interface card and connects a
        bare metal server to a subnet. While each network interface can attach to only one
        subnet, multiple network interfaces can be created to attach to multiple subnets.
        Multiple interfaces may also attach to the same subnet.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerNetworkInterfaceCollection` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_server_network_interfaces')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_bare_metal_server_network_interface(self,
        bare_metal_server_id: str,
        bare_metal_server_network_interface_prototype: 'BareMetalServerNetworkInterfacePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a network interface on a bare metal server.

        This request creates a new network interface from a network interface prototype
        object. The prototype object is structured in the same way as a retrieved network
        interface, and contains the information necessary to create the new network
        interface. Any subnet in the bare metal server's VPC may be specified, even if it
        is already attached to another network interface. Addresses on the network
        interface must be within the specified subnet's CIDR blocks.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param BareMetalServerNetworkInterfacePrototype
               bare_metal_server_network_interface_prototype: The network interface
               prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerNetworkInterface` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if bare_metal_server_network_interface_prototype is None:
            raise ValueError('bare_metal_server_network_interface_prototype must be provided')
        if isinstance(bare_metal_server_network_interface_prototype, BareMetalServerNetworkInterfacePrototype):
            bare_metal_server_network_interface_prototype = convert_model(bare_metal_server_network_interface_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_bare_metal_server_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(bare_metal_server_network_interface_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_bare_metal_server_network_interface(self,
        bare_metal_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a network interface.

        This request deletes a network interface. This operation cannot be reversed. Any
        floating IPs associated with the network interface are implicitly disassociated.
        The primary network interface is not allowed to be deleted.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_bare_metal_server_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['bare_metal_server_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_network_interface(self,
        bare_metal_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a network interface.

        This request retrieves a single network interface specified by the identifier in
        the URL.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerNetworkInterface` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_bare_metal_server_network_interface(self,
        bare_metal_server_id: str,
        id: str,
        bare_metal_server_network_interface_patch: 'BareMetalServerNetworkInterfacePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a network interface.

        This request updates a network interface with the information provided in a
        network interface patch object. The network interface patch object is structured
        in the same way as a retrieved network interface and needs to contain only the
        information to be updated.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str id: The network interface identifier.
        :param BareMetalServerNetworkInterfacePatch
               bare_metal_server_network_interface_patch: The network interface patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerNetworkInterface` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if bare_metal_server_network_interface_patch is None:
            raise ValueError('bare_metal_server_network_interface_patch must be provided')
        if isinstance(bare_metal_server_network_interface_patch, BareMetalServerNetworkInterfacePatch):
            bare_metal_server_network_interface_patch = convert_model(bare_metal_server_network_interface_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_bare_metal_server_network_interface')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(bare_metal_server_network_interface_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_bare_metal_server_network_interface_floating_ips(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all floating IPs associated with a network interface.

        This request lists all floating IPs associated with a network interface.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIPUnpaginatedCollection` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_server_network_interface_floating_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'network_interface_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/floating_ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def remove_bare_metal_server_network_interface_floating_ip(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Disassociate a floating IP from a network interface.

        This request disassociates the specified floating IP from the specified network
        interface.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='remove_bare_metal_server_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['bare_metal_server_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_network_interface_floating_ip(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve associated floating IP.

        This request retrieves a specified floating IP address if it is associated with
        the network interface and bare metal server specified in the URL.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def add_bare_metal_server_network_interface_floating_ip(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Associate a floating IP with a network interface.

        This request associates the specified floating IP with the specified network
        interface. If `enable_infrastructure_nat` is `false`, this adds the IP to any
        existing associations. If `enable_infrastructure_nat` is `true`, this replaces any
        existing association.  For this request to succeed, the existing floating IP must
        not be required by another resource, such as a public gateway. A request body is
        not required, and if provided, is ignored.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='add_bare_metal_server_network_interface_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_bare_metal_server_network_interface_ips(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all reserved IPs bound to a network interface.

        This request lists all reserved IPs bound to a network interface.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIPCollectionNetworkInterfaceContext` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_bare_metal_server_network_interface_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'network_interface_id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_network_interface_ip(self,
        bare_metal_server_id: str,
        network_interface_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve bound reserved IP.

        This request retrieves the specified reserved IP address if it is bound to the
        network interface and bare metal server specified in the URL.

        :param str bare_metal_server_id: The bare metal server identifier.
        :param str network_interface_id: The network interface identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not bare_metal_server_id:
            raise ValueError('bare_metal_server_id must be provided')
        if not network_interface_id:
            raise ValueError('network_interface_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_network_interface_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['bare_metal_server_id', 'network_interface_id', 'id']
        path_param_values = self.encode_path_vars(bare_metal_server_id, network_interface_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{bare_metal_server_id}/network_interfaces/{network_interface_id}/ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_bare_metal_server(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a bare metal server.

        This request deletes a bare metal server. This operation cannot be reversed. Any
        floating IPs associated with the bare metal server's network interfaces are
        implicitly disassociated.

        :param str id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a bare metal server.

        This request retrieves a single bare metal server specified by the identifier in
        the URL.

        :param str id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServer` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_bare_metal_server(self,
        id: str,
        bare_metal_server_patch: 'BareMetalServerPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a bare metal server.

        This request updates a bare metal server with the information in a provided patch.
        The bare metal server patch object is structured in the same way as a retrieved
        bare metal server and contains only the information to be updated.

        :param str id: The bare metal server identifier.
        :param BareMetalServerPatch bare_metal_server_patch: The bare metal server
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServer` object
        """

        if not id:
            raise ValueError('id must be provided')
        if bare_metal_server_patch is None:
            raise ValueError('bare_metal_server_patch must be provided')
        if isinstance(bare_metal_server_patch, BareMetalServerPatch):
            bare_metal_server_patch = convert_model(bare_metal_server_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(bare_metal_server_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_bare_metal_server_initialization(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve initialization configuration for a bare metal server.

        This request retrieves configuration variables used to initialize the bare metal
        server, such as the image used, SSH keys, and any configured usernames and
        passwords.  These attributes can subsequently be changed manually by the user and
        so are not guaranteed to be current.

        :param str id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `BareMetalServerInitialization` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_bare_metal_server_initialization')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}/initialization'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def restart_bare_metal_server(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Restart a bare metal server.

        This request restarts a bare metal server.  It will run immediately regardless of
        the state of the server.

        :param str id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='restart_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}/restart'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def start_bare_metal_server(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Start a bare metal server.

        This request starts a bare metal server.  It will run immediately provided the
        server is stopped.

        :param str id: The bare metal server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='start_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}/start'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def stop_bare_metal_server(self,
        id: str,
        type: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Stop a bare metal server.

        This request stops a bare metal server. It will run immediately provided the
        server is running. Note: A soft stop may not complete as it relies on the
        operating system to perform the operation.

        :param str id: The bare metal server identifier.
        :param str type: The type of stop operation:
               - `soft`: signal running operating system to quiesce and shutdown cleanly
               - `hard`: immediately stop the server.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        if type is None:
            raise ValueError('type must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='stop_bare_metal_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'type': type,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/bare_metal_servers/{id}/stop'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Volumes
    #########################


    def list_volume_profiles(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all volume profiles.

        This request lists all [volume
        profiles](https://cloud.ibm.com/docs/vpc?topic=vpc-block-storage-profiles)
        available in the region. A volume profile specifies the performance
        characteristics and pricing model for a volume.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_volume_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/volume/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_volume_profile(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a volume profile.

        This request retrieves a single volume profile specified by the name in the URL.

        :param str name: The volume profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeProfile` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_volume_profile')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/volume/profiles/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_volumes(self,
        *,
        start: str = None,
        limit: int = None,
        name: str = None,
        attachment_state: str = None,
        encryption: str = None,
        operating_system_family: str = None,
        operating_system_architecture: str = None,
        zone_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all volumes.

        This request lists all volumes in the region. Volumes are network-connected block
        storage devices that may be attached to one or more instances in the same region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str attachment_state: (optional) Filters the collection to volumes
               with the specified attachment state.
        :param str encryption: (optional) Filters the collection to resources with
               the specified encryption type.
        :param str operating_system_family: (optional) Filters the collection to
               resources with the exact specified operating system family.
               This parameter also supports the values `null` and `not:null` which filter
               the collection to resources which have no operating system or any operating
               system, respectively.
        :param str operating_system_architecture: (optional) Filters the collection
               to resources with the exact specified operating system architecture.
               This parameter also supports the values `null` and `not:null` which filter
               the collection to resources which have no operating system or any operating
               system, respectively.
        :param str zone_name: (optional) Filters the collection to resources in the
               zone with the exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VolumeCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_volumes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'name': name,
            'attachment_state': attachment_state,
            'encryption': encryption,
            'operating_system.family': operating_system_family,
            'operating_system.architecture': operating_system_architecture,
            'zone.name': zone_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/volumes'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_volume(self,
        volume_prototype: 'VolumePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a volume.

        This request creates a new volume from a volume prototype object. The prototype
        object is structured in the same way as a retrieved volume, and contains the
        information necessary to create the new volume.

        :param VolumePrototype volume_prototype: The volume prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if volume_prototype is None:
            raise ValueError('volume_prototype must be provided')
        if isinstance(volume_prototype, VolumePrototype):
            volume_prototype = convert_model(volume_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_volume')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(volume_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/volumes'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_volume(self,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a volume.

        This request deletes a volume. This operation cannot be reversed. For this request
        to succeed, the volume must not be attached to any instances.

        :param str id: The volume identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_volume')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/volumes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_volume(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a volume.

        This request retrieves a single volume specified by the identifier in the URL.

        :param str id: The volume identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_volume')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/volumes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_volume(self,
        id: str,
        volume_patch: 'VolumePatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a volume.

        This request updates a volume with the information in a provided volume patch. The
        volume patch object is structured in the same way as a retrieved volume and
        contains only the information to be updated.

        :param str id: The volume identifier.
        :param VolumePatch volume_patch: The volume patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Volume` object
        """

        if not id:
            raise ValueError('id must be provided')
        if volume_patch is None:
            raise ValueError('volume_patch must be provided')
        if isinstance(volume_patch, VolumePatch):
            volume_patch = convert_model(volume_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_volume')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(volume_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/volumes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Snapshots
    #########################


    def delete_snapshots(self,
        source_volume_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a filtered collection of snapshots.

        This request deletes all snapshots created from a specific source volume.

        :param str source_volume_id: Filters the collection to resources with the
               source volume with the specified identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not source_volume_id:
            raise ValueError('source_volume_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_snapshots')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'source_volume.id': source_volume_id,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        url = '/snapshots'
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_snapshots(self,
        *,
        start: str = None,
        limit: int = None,
        tag: str = None,
        resource_group_id: str = None,
        name: str = None,
        source_volume_id: str = None,
        source_volume_crn: str = None,
        source_image_id: str = None,
        source_image_crn: str = None,
        sort: str = None,
        backup_policy_plan_id: str = None,
        clones_zone_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all snapshots.

        This request lists all snapshots in the region. A snapshot preserves the data of a
        volume at the time the snapshot is created.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str tag: (optional) Filters the collection to resources with the
               exact tag value.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str source_volume_id: (optional) Filters the collection to resources
               with the source volume with the specified identifier.
        :param str source_volume_crn: (optional) Filters the collection to
               resources with the source volume with the specified CRN.
        :param str source_image_id: (optional) Filters the collection to resources
               with the source image with the specified identifier.
               This parameter also supports the values `null` and `not:null` which filter
               the collection to resources which have no source image or any existent
               source image, respectively.
        :param str source_image_crn: (optional) Filters the collection to resources
               with the source volume with the specified CRN.
               This parameter also supports the values `null` and `not:null` which filter
               the collection to resources which have no source image or any existent
               source image, respectively.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param str backup_policy_plan_id: (optional) Filters the collection to
               backup policy jobs with the backup plan with the specified identifier.
        :param str clones_zone_name: (optional) Filters the collection to resources
               with a clone in the specified zone.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SnapshotCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_snapshots')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'tag': tag,
            'resource_group.id': resource_group_id,
            'name': name,
            'source_volume.id': source_volume_id,
            'source_volume.crn': source_volume_crn,
            'source_image.id': source_image_id,
            'source_image.crn': source_image_crn,
            'sort': sort,
            'backup_policy_plan.id': backup_policy_plan_id,
            'clones[].zone.name': clones_zone_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/snapshots'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_snapshot(self,
        snapshot_prototype: 'SnapshotPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a snapshot.

        This request creates a new snapshot from a snapshot prototype object.  The
        prototype object is structured in the same way as a retrieved snapshot, and
        contains the information necessary to provision the new snapshot.

        :param SnapshotPrototype snapshot_prototype: The snapshot prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Snapshot` object
        """

        if snapshot_prototype is None:
            raise ValueError('snapshot_prototype must be provided')
        if isinstance(snapshot_prototype, SnapshotPrototype):
            snapshot_prototype = convert_model(snapshot_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_snapshot')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(snapshot_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/snapshots'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_snapshot(self,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a snapshot.

        This request deletes a snapshot. This operation cannot be reversed.

        :param str id: The snapshot identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_snapshot')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_snapshot(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a snapshot.

        This request retrieves a single snapshot specified by the identifier in the URL.

        :param str id: The snapshot identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Snapshot` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_snapshot')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_snapshot(self,
        id: str,
        snapshot_patch: 'SnapshotPatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a snapshot.

        This request updates a snapshot's name.

        :param str id: The snapshot identifier.
        :param SnapshotPatch snapshot_patch: The snapshot patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Snapshot` object
        """

        if not id:
            raise ValueError('id must be provided')
        if snapshot_patch is None:
            raise ValueError('snapshot_patch must be provided')
        if isinstance(snapshot_patch, SnapshotPatch):
            snapshot_patch = convert_model(snapshot_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_snapshot')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(snapshot_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_snapshot_clones(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all clones for a snapshot.

        This request lists all clones for a snapshot. Use a clone to quickly restore a
        snapshot within the clone's zone.

        :param str id: The snapshot identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SnapshotCloneCollection` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_snapshot_clones')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}/clones'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_snapshot_clone(self,
        id: str,
        zone_name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a snapshot clone.

        This request deletes a snapshot clone. This operation cannot be reversed, but an
        equivalent clone may be recreated from the snapshot.

        :param str id: The snapshot identifier.
        :param str zone_name: The zone name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        if not zone_name:
            raise ValueError('zone_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_snapshot_clone')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id', 'zone_name']
        path_param_values = self.encode_path_vars(id, zone_name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}/clones/{zone_name}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_snapshot_clone(self,
        id: str,
        zone_name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a snapshot clone.

        This request retrieves a single clone specified by the snapshot identifier and
        zone name in the URL.

        :param str id: The snapshot identifier.
        :param str zone_name: The zone name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SnapshotClone` object
        """

        if not id:
            raise ValueError('id must be provided')
        if not zone_name:
            raise ValueError('zone_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_snapshot_clone')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id', 'zone_name']
        path_param_values = self.encode_path_vars(id, zone_name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}/clones/{zone_name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_snapshot_clone(self,
        id: str,
        zone_name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a clone for a snapshot.

        This request creates a new clone for a snapshot in the specified zone. A request
        body is not required, and if provided, is ignored. If the snapshot already has a
        clone in the zone, it is returned.

        :param str id: The snapshot identifier.
        :param str zone_name: The zone name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SnapshotClone` object
        """

        if not id:
            raise ValueError('id must be provided')
        if not zone_name:
            raise ValueError('zone_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_snapshot_clone')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id', 'zone_name']
        path_param_values = self.encode_path_vars(id, zone_name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/snapshots/{id}/clones/{zone_name}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Geography
    #########################


    def list_regions(self,
        **kwargs
    ) -> DetailedResponse:
        """
        List all regions.

        This request lists all regions. Each region is a separate geographic area that
        contains multiple isolated zones. Resources can be provisioned into one or more
        zones in a region. Each zone is isolated, but connected to other zones in the same
        region with low-latency and high-bandwidth links. Regions represent the top-level
        of fault isolation available. Resources deployed within a single region also
        benefit from the low latency afforded by geographic proximity.

        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `RegionCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_regions')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/regions'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_region(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a region.

        This request retrieves a single region specified by the name in the URL.

        :param str name: The region name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Region` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_region')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/regions/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_region_zones(self,
        region_name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all zones in a region.

        This request lists all zones in a region. Zones represent logically-isolated data
        centers with high-bandwidth and low-latency interconnects to other zones in the
        same region. Faults in a zone do not affect other zones.

        :param str region_name: The region name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ZoneCollection` object
        """

        if not region_name:
            raise ValueError('region_name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_region_zones')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['region_name']
        path_param_values = self.encode_path_vars(region_name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/regions/{region_name}/zones'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_region_zone(self,
        region_name: str,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a zone.

        This request retrieves a single zone specified by the region and zone names in the
        URL.

        :param str region_name: The region name.
        :param str name: The zone name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `Zone` object
        """

        if not region_name:
            raise ValueError('region_name must be provided')
        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_region_zone')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['region_name', 'name']
        path_param_values = self.encode_path_vars(region_name, name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/regions/{region_name}/zones/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Public gateways
    #########################


    def list_public_gateways(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all public gateways.

        This request lists all public gateways in the region. A public gateway is a
        virtual network device associated with a VPC, which allows access to the Internet.
        A public gateway resides in a zone and can be connected to subnets in the same
        zone only.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGatewayCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_public_gateways')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/public_gateways'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_public_gateway(self,
        vpc: 'VPCIdentity',
        zone: 'ZoneIdentity',
        *,
        floating_ip: 'PublicGatewayFloatingIPPrototype' = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a public gateway.

        This request creates a new public gateway from a public gateway prototype object.
        For this to succeed, the VPC must not already have a public gateway in the
        specified zone.
        If a floating IP is provided, it must be unbound. If a floating IP is not
        provided, one will be created and bound to the public gateway. Once a public
        gateway has been created, its floating IP cannot be unbound. A public gateway must
        be explicitly attached to each subnet it will provide connectivity for.

        :param VPCIdentity vpc: The VPC this public gateway will reside in.
        :param ZoneIdentity zone: The zone this public gateway will reside in.
        :param PublicGatewayFloatingIPPrototype floating_ip: (optional)
        :param str name: (optional) The name for this public gateway. The name must
               not be used by another public gateway in the VPC. If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if vpc is None:
            raise ValueError('vpc must be provided')
        if zone is None:
            raise ValueError('zone must be provided')
        vpc = convert_model(vpc)
        zone = convert_model(zone)
        if floating_ip is not None:
            floating_ip = convert_model(floating_ip)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'vpc': vpc,
            'zone': zone,
            'floating_ip': floating_ip,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/public_gateways'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_public_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a public gateway.

        This request deletes a public gateway. This operation cannot be reversed. For this
        request to succeed, the public gateway must not be attached to any subnets. The
        public gateway's floating IP will be automatically unbound. If the floating IP was
        created when the public gateway was created, it will be deleted.

        :param str id: The public gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/public_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_public_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a public gateway.

        This request retrieves a single public gateway specified by the identifier in the
        URL.

        :param str id: The public gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/public_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_public_gateway(self,
        id: str,
        public_gateway_patch: 'PublicGatewayPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a public gateway.

        This request updates a public gateway's name.

        :param str id: The public gateway identifier.
        :param PublicGatewayPatch public_gateway_patch: The public gateway patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `PublicGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        if public_gateway_patch is None:
            raise ValueError('public_gateway_patch must be provided')
        if isinstance(public_gateway_patch, PublicGatewayPatch):
            public_gateway_patch = convert_model(public_gateway_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_public_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(public_gateway_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/public_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Floating IPs
    #########################


    def list_floating_ips(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all floating IPs.

        This request lists all floating IPs in the region. Floating IPs allow inbound and
        outbound traffic from the Internet to an instance.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIPCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_floating_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/floating_ips'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_floating_ip(self,
        floating_ip_prototype: 'FloatingIPPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Reserve a floating IP.

        This request reserves a new floating IP.

        :param FloatingIPPrototype floating_ip_prototype: The floating IP prototype
               object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if floating_ip_prototype is None:
            raise ValueError('floating_ip_prototype must be provided')
        if isinstance(floating_ip_prototype, FloatingIPPrototype):
            floating_ip_prototype = convert_model(floating_ip_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(floating_ip_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/floating_ips'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_floating_ip(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a floating IP.

        This request disassociates (if associated) and releases a floating IP. This
        operation cannot be reversed. For this request to succeed, the floating IP must
        not be required by another resource, such as a public gateway.

        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_floating_ip(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a floating IP.

        This request retrieves a single floating IP specified by the identifier in the
        URL.

        :param str id: The floating IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_floating_ip(self,
        id: str,
        floating_ip_patch: 'FloatingIPPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a floating IP.

        This request updates a floating IP's name and/or target.

        :param str id: The floating IP identifier.
        :param FloatingIPPatch floating_ip_patch: The floating IP patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FloatingIP` object
        """

        if not id:
            raise ValueError('id must be provided')
        if floating_ip_patch is None:
            raise ValueError('floating_ip_patch must be provided')
        if isinstance(floating_ip_patch, FloatingIPPatch):
            floating_ip_patch = convert_model(floating_ip_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_floating_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(floating_ip_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/floating_ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Network ACLs
    #########################


    def list_network_acls(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all network ACLs.

        This request lists all network ACLs in the region. A network ACL defines a set of
        packet filtering (5-tuple) rules for all traffic in and out of a subnet. Both
        allow and deny rules can be defined, and rules are stateless such that reverse
        traffic in response to allowed traffic is not automatically permitted.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_network_acls')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/network_acls'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_network_acl(self,
        *,
        network_acl_prototype: 'NetworkACLPrototype' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a network ACL.

        This request creates a new stateless network ACL from a network ACL prototype
        object. The prototype object is structured in the same way as a retrieved network
        ACL, and contains the information necessary to create the new network ACL.

        :param NetworkACLPrototype network_acl_prototype: (optional) The network
               ACL prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if  network_acl_prototype is not None and isinstance(network_acl_prototype, NetworkACLPrototype):
            network_acl_prototype = convert_model(network_acl_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_acl_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/network_acls'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_network_acl(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a network ACL.

        This request deletes a network ACL. This operation cannot be reversed. For this
        request to succeed, the network ACL must not be the default network ACL for any
        VPCs, and the network ACL must not be attached to any subnets.

        :param str id: The network ACL identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_network_acl(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a network ACL.

        This request retrieves a single network ACL specified by the identifier in the
        URL.

        :param str id: The network ACL identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_network_acl(self,
        id: str,
        network_acl_patch: 'NetworkACLPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a network ACL.

        This request updates a network ACL's name.

        :param str id: The network ACL identifier.
        :param NetworkACLPatch network_acl_patch: The network ACL patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACL` object
        """

        if not id:
            raise ValueError('id must be provided')
        if network_acl_patch is None:
            raise ValueError('network_acl_patch must be provided')
        if isinstance(network_acl_patch, NetworkACLPatch):
            network_acl_patch = convert_model(network_acl_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_network_acl')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_acl_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_network_acl_rules(self,
        network_acl_id: str,
        *,
        start: str = None,
        limit: int = None,
        direction: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all rules for a network ACL.

        This request lists all rules for a network ACL. These rules can allow or deny
        traffic between a source CIDR block and a destination CIDR block over a particular
        protocol and port range.

        :param str network_acl_id: The network ACL identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str direction: (optional) Filters the collection to rules with the
               specified direction.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRuleCollection` object
        """

        if not network_acl_id:
            raise ValueError('network_acl_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_network_acl_rules')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'direction': direction,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['network_acl_id']
        path_param_values = self.encode_path_vars(network_acl_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{network_acl_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_network_acl_rule(self,
        network_acl_id: str,
        network_acl_rule_prototype: 'NetworkACLRulePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a rule for a network ACL.

        This request creates a new rule from a network ACL rule prototype object. The
        prototype object is structured in the same way as a retrieved rule, and contains
        the information necessary to create the new rule.

        :param str network_acl_id: The network ACL identifier.
        :param NetworkACLRulePrototype network_acl_rule_prototype: The network ACL
               rule prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if not network_acl_id:
            raise ValueError('network_acl_id must be provided')
        if network_acl_rule_prototype is None:
            raise ValueError('network_acl_rule_prototype must be provided')
        if isinstance(network_acl_rule_prototype, NetworkACLRulePrototype):
            network_acl_rule_prototype = convert_model(network_acl_rule_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_network_acl_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_acl_rule_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['network_acl_id']
        path_param_values = self.encode_path_vars(network_acl_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{network_acl_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_network_acl_rule(self,
        network_acl_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a network ACL rule.

        This request deletes a rule. This operation cannot be reversed.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not network_acl_id:
            raise ValueError('network_acl_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_network_acl_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['network_acl_id', 'id']
        path_param_values = self.encode_path_vars(network_acl_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{network_acl_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_network_acl_rule(self,
        network_acl_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a network ACL rule.

        This request retrieves a single rule specified by the identifier in the URL.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if not network_acl_id:
            raise ValueError('network_acl_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_network_acl_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['network_acl_id', 'id']
        path_param_values = self.encode_path_vars(network_acl_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{network_acl_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_network_acl_rule(self,
        network_acl_id: str,
        id: str,
        network_acl_rule_patch: 'NetworkACLRulePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a network ACL rule.

        This request updates a rule with the information in a provided rule patch. The
        rule patch object contains only the information to be updated. The request will
        fail if the information is not applicable to the rule's protocol.

        :param str network_acl_id: The network ACL identifier.
        :param str id: The rule identifier.
        :param NetworkACLRulePatch network_acl_rule_patch: The network ACL rule
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `NetworkACLRule` object
        """

        if not network_acl_id:
            raise ValueError('network_acl_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if network_acl_rule_patch is None:
            raise ValueError('network_acl_rule_patch must be provided')
        if isinstance(network_acl_rule_patch, NetworkACLRulePatch):
            network_acl_rule_patch = convert_model(network_acl_rule_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_network_acl_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(network_acl_rule_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['network_acl_id', 'id']
        path_param_values = self.encode_path_vars(network_acl_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/network_acls/{network_acl_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Security groups
    #########################


    def list_security_groups(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        vpc_id: str = None,
        vpc_crn: str = None,
        vpc_name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all security groups.

        This request lists all security groups in the region. Security groups provide a
        way to apply IP filtering rules to instances in the associated VPC. With security
        groups, all traffic is denied by default, and rules added to security groups
        define which traffic the security group permits. Security group rules are stateful
        such that reverse traffic in response to allowed traffic is automatically
        permitted.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str vpc_id: (optional) Filters the collection to resources in the
               VPC with the specified identifier.
        :param str vpc_crn: (optional) Filters the collection to resources in the
               VPC with the specified CRN.
        :param str vpc_name: (optional) Filters the collection to resources in the
               VPC with the exact specified name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_security_groups')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'vpc.id': vpc_id,
            'vpc.crn': vpc_crn,
            'vpc.name': vpc_name,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/security_groups'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_security_group(self,
        vpc: 'VPCIdentity',
        *,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        rules: List['SecurityGroupRulePrototype'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a security group.

        This request creates a new security group from a security group prototype object.
        The prototype object is structured in the same way as a retrieved security group,
        and contains the information necessary to create the new security group. If
        security group rules are included in the prototype object, those rules will be
        added to the security group. Each security group is scoped to one VPC. Only
        network interfaces on instances in that VPC can be added to the security group.

        :param VPCIdentity vpc: The VPC this security group will reside in.
        :param str name: (optional) The name for this security group. The name must
               not be used by another security group for the VPC. If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param List[SecurityGroupRulePrototype] rules: (optional) The prototype
               objects for rules to be created for this security group. If unspecified, no
               rules will be created, resulting in all traffic being denied.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if vpc is None:
            raise ValueError('vpc must be provided')
        vpc = convert_model(vpc)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if rules is not None:
            rules = [convert_model(x) for x in rules]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_security_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'vpc': vpc,
            'name': name,
            'resource_group': resource_group,
            'rules': rules,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/security_groups'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_security_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a security group.

        This request deletes a security group. A security group cannot be deleted if it is
        referenced by any security group targets or rules. Additionally, a VPC's default
        security group cannot be deleted. This operation cannot be reversed.

        :param str id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_security_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_security_group(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a security group.

        This request retrieves a single security group specified by the identifier in the
        URL path.

        :param str id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_security_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_security_group(self,
        id: str,
        security_group_patch: 'SecurityGroupPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a security group.

        This request updates a security group with the information provided in a security
        group patch object. The security group patch object is structured in the same way
        as a retrieved security group and contains only the information to be updated.

        :param str id: The security group identifier.
        :param SecurityGroupPatch security_group_patch: The security group patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroup` object
        """

        if not id:
            raise ValueError('id must be provided')
        if security_group_patch is None:
            raise ValueError('security_group_patch must be provided')
        if isinstance(security_group_patch, SecurityGroupPatch):
            security_group_patch = convert_model(security_group_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_security_group')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(security_group_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_security_group_rules(self,
        security_group_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all rules in a security group.

        This request lists all rules in a security group. These rules define what traffic
        the security group permits. Security group rules are stateful, such that reverse
        traffic in response to allowed traffic is automatically permitted.

        :param str security_group_id: The security group identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRuleCollection` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_security_group_rules')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id']
        path_param_values = self.encode_path_vars(security_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_security_group_rule(self,
        security_group_id: str,
        security_group_rule_prototype: 'SecurityGroupRulePrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a rule for a security group.

        This request creates a new security group rule from a security group rule
        prototype object. The prototype object is structured in the same way as a
        retrieved security group rule and contains the information necessary to create the
        rule. As part of creating a new rule in a security group, the rule is applied to
        all the networking interfaces in the security group. Rules specify which IP
        traffic a security group will allow. Security group rules are stateful, such that
        reverse traffic in response to allowed traffic is automatically permitted. A rule
        allowing inbound TCP traffic on port 80 also allows outbound TCP traffic on port
        80 without the need for an additional rule.

        :param str security_group_id: The security group identifier.
        :param SecurityGroupRulePrototype security_group_rule_prototype: The
               properties of the security group rule to be created.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if security_group_rule_prototype is None:
            raise ValueError('security_group_rule_prototype must be provided')
        if isinstance(security_group_rule_prototype, SecurityGroupRulePrototype):
            security_group_rule_prototype = convert_model(security_group_rule_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_security_group_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(security_group_rule_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id']
        path_param_values = self.encode_path_vars(security_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_security_group_rule(self,
        security_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a security group rule.

        This request deletes a security group rule. This operation cannot be reversed.
        Removing a security group rule will not end existing connections allowed by that
        rule.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_security_group_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_security_group_rule(self,
        security_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a security group rule.

        This request retrieves a single security group rule specified by the identifier in
        the URL path.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_security_group_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_security_group_rule(self,
        security_group_id: str,
        id: str,
        security_group_rule_patch: 'SecurityGroupRulePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a security group rule.

        This request updates a security group rule with the information in a provided rule
        patch object. The rule patch object contains only the information to be updated.
        The request will fail if the information is not applicable to the rule's protocol.

        :param str security_group_id: The security group identifier.
        :param str id: The rule identifier.
        :param SecurityGroupRulePatch security_group_rule_patch: The security group
               rule patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupRule` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if security_group_rule_patch is None:
            raise ValueError('security_group_rule_patch must be provided')
        if isinstance(security_group_rule_patch, SecurityGroupRulePatch):
            security_group_rule_patch = convert_model(security_group_rule_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_security_group_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(security_group_rule_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_security_group_targets(self,
        security_group_id: str,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all targets associated with a security group.

        This request lists all targets associated with a security group, to which the
        rules in the security group are applied.

        :param str security_group_id: The security group identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupTargetCollection` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_security_group_targets')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id']
        path_param_values = self.encode_path_vars(security_group_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/targets'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_security_group_target_binding(self,
        security_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Remove a target from a security group.

        This request removes a target from a security group. For this request to succeed,
        the target must be attached to at least one other security group.  The specified
        target identifier can be:
        - An instance network interface identifier
        - A bare metal server network interface identifier
        - A VPN server identifier
        - An application load balancer identifier
        - An endpoint gateway identifier
        Security groups are stateful, so any changes to a target's security groups are
        applied to new connections. Existing connections are not affected.

        :param str security_group_id: The security group identifier.
        :param str id: The security group target identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_security_group_target_binding')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/targets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_security_group_target(self,
        security_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a security group target.

        This request retrieves a single target specified by the identifier in the URL
        path. The target must be an existing target of the security group.

        :param str security_group_id: The security group identifier.
        :param str id: The security group target identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupTargetReference` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_security_group_target')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/targets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_security_group_target_binding(self,
        security_group_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Add a target to a security group.

        This request adds a resource to an existing security group. The specified target
        identifier can be:
        - An instance network interface identifier
        - A bare metal server network interface identifier
        - A VPN server identifier
        - An application load balancer identifier
        - An endpoint gateway identifier
        When a target is added to a security group, the security group rules are applied
        to the target. A request body is not required, and if provided, is ignored.

        :param str security_group_id: The security group identifier.
        :param str id: The security group target identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `SecurityGroupTargetReference` object
        """

        if not security_group_id:
            raise ValueError('security_group_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_security_group_target_binding')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['security_group_id', 'id']
        path_param_values = self.encode_path_vars(security_group_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/security_groups/{security_group_id}/targets/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # VPN gateways
    #########################


    def list_ike_policies(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all IKE policies.

        This request lists all IKE policies in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ike_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/ike_policies'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_ike_policy(self,
        authentication_algorithm: str,
        dh_group: int,
        encryption_algorithm: str,
        ike_version: int,
        *,
        key_lifetime: int = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an IKE policy.

        This request creates a new IKE policy.

        :param str authentication_algorithm: The authentication algorithm
               The `md5` and `sha1` algorithms have been deprecated.
        :param int dh_group: The Diffie-Hellman group
               Groups `2` and `5` have been deprecated.
        :param str encryption_algorithm: The encryption algorithm
               The `triple_des` algorithm has been deprecated.
        :param int ike_version: The IKE protocol version.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The name for this IKE policy. The name must not
               be used by another IKE policies in the region. If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if authentication_algorithm is None:
            raise ValueError('authentication_algorithm must be provided')
        if dh_group is None:
            raise ValueError('dh_group must be provided')
        if encryption_algorithm is None:
            raise ValueError('encryption_algorithm must be provided')
        if ike_version is None:
            raise ValueError('ike_version must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_ike_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'authentication_algorithm': authentication_algorithm,
            'dh_group': dh_group,
            'encryption_algorithm': encryption_algorithm,
            'ike_version': ike_version,
            'key_lifetime': key_lifetime,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/ike_policies'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_ike_policy(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an IKE policy.

        This request deletes an IKE policy. This operation cannot be reversed. For this
        request to succeed, there must not be any VPN gateway connections using this
        policy.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_ike_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ike_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_ike_policy(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an IKE policy.

        This request retrieves a single IKE policy specified by the identifier in the URL.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_ike_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ike_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_ike_policy(self,
        id: str,
        ike_policy_patch: 'IKEPolicyPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an IKE policy.

        This request updates the properties of an existing IKE policy.

        :param str id: The IKE policy identifier.
        :param IKEPolicyPatch ike_policy_patch: The IKE policy patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IKEPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        if ike_policy_patch is None:
            raise ValueError('ike_policy_patch must be provided')
        if isinstance(ike_policy_patch, IKEPolicyPatch):
            ike_policy_patch = convert_model(ike_policy_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_ike_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(ike_policy_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ike_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_ike_policy_connections(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN gateway connections that use a specified IKE policy.

        This request lists all VPN gateway connections that use a policy.

        :param str id: The IKE policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ike_policy_connections')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ike_policies/{id}/connections'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_ipsec_policies(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all IPsec policies.

        This request lists all IPsec policies in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicyCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ipsec_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/ipsec_policies'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_ipsec_policy(self,
        authentication_algorithm: str,
        encryption_algorithm: str,
        pfs: str,
        *,
        key_lifetime: int = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an IPsec policy.

        This request creates a new IPsec policy.

        :param str authentication_algorithm: The authentication algorithm
               The `md5` and `sha1` algorithms have been deprecated
               Must be `disabled` if and only if the `encryption_algorithm` is
               `aes128gcm16`, `aes192gcm16`, or `aes256gcm16`.
        :param str encryption_algorithm: The encryption algorithm
               The `triple_des` algorithm has been deprecated
               The `authentication_algorithm` must be `disabled` if and only if
               `encryption_algorithm` is `aes128gcm16`, `aes192gcm16`, or
               `aes256gcm16`.
        :param str pfs: Perfect Forward Secrecy
               Groups `group_2` and `group_5` have been deprecated.
        :param int key_lifetime: (optional) The key lifetime in seconds.
        :param str name: (optional) The name for this IPsec policy. The name must
               not be used by another IPsec policies in the region. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if authentication_algorithm is None:
            raise ValueError('authentication_algorithm must be provided')
        if encryption_algorithm is None:
            raise ValueError('encryption_algorithm must be provided')
        if pfs is None:
            raise ValueError('pfs must be provided')
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_ipsec_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'authentication_algorithm': authentication_algorithm,
            'encryption_algorithm': encryption_algorithm,
            'pfs': pfs,
            'key_lifetime': key_lifetime,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/ipsec_policies'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_ipsec_policy(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an IPsec policy.

        This request deletes an IPsec policy. This operation cannot be reversed. For this
        request to succeed, there must not be any VPN gateway connections using this
        policy.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_ipsec_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ipsec_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_ipsec_policy(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an IPsec policy.

        This request retrieves a single IPsec policy specified by the identifier in the
        URL.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_ipsec_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ipsec_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_ipsec_policy(self,
        id: str,
        i_psec_policy_patch: 'IPsecPolicyPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an IPsec policy.

        This request updates the properties of an existing IPsec policy.

        :param str id: The IPsec policy identifier.
        :param IPsecPolicyPatch i_psec_policy_patch: The IPsec policy patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `IPsecPolicy` object
        """

        if not id:
            raise ValueError('id must be provided')
        if i_psec_policy_patch is None:
            raise ValueError('i_psec_policy_patch must be provided')
        if isinstance(i_psec_policy_patch, IPsecPolicyPatch):
            i_psec_policy_patch = convert_model(i_psec_policy_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_ipsec_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(i_psec_policy_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ipsec_policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_ipsec_policy_connections(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN gateway connections that use a specified IPsec policy.

        This request lists all VPN gateway connections that use a policy.

        :param str id: The IPsec policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_ipsec_policy_connections')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/ipsec_policies/{id}/connections'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_gateways(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        sort: str = None,
        mode: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN gateways.

        This request lists all VPN gateways in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param str mode: (optional) Filters the collection to VPN gateways with the
               specified mode.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_gateways')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'sort': sort,
            'mode': mode,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpn_gateways'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpn_gateway(self,
        vpn_gateway_prototype: 'VPNGatewayPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a VPN gateway.

        This request creates a new VPN gateway.

        :param VPNGatewayPrototype vpn_gateway_prototype: The VPN gateway prototype
               object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if vpn_gateway_prototype is None:
            raise ValueError('vpn_gateway_prototype must be provided')
        if isinstance(vpn_gateway_prototype, VPNGatewayPrototype):
            vpn_gateway_prototype = convert_model(vpn_gateway_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpn_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_gateway_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpn_gateways'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpn_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPN gateway.

        This request deletes a VPN gateway. This operation cannot be reversed. For this
        request to succeed, the VPN gateway must not have a `status` of `pending`, and
        there must not be any VPC routes using the VPN gateway's connections as a next
        hop.

        :param str id: The VPN gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPN gateway.

        This request retrieves a single VPN gateway specified by the identifier in the
        URL.

        :param str id: The VPN gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpn_gateway(self,
        id: str,
        vpn_gateway_patch: 'VPNGatewayPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPN gateway.

        This request updates the properties of an existing VPN gateway.

        :param str id: The VPN gateway identifier.
        :param VPNGatewayPatch vpn_gateway_patch: The VPN gateway patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        if vpn_gateway_patch is None:
            raise ValueError('vpn_gateway_patch must be provided')
        if isinstance(vpn_gateway_patch, VPNGatewayPatch):
            vpn_gateway_patch = convert_model(vpn_gateway_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpn_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_gateway_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_gateway_connections(self,
        vpn_gateway_id: str,
        *,
        status: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all connections of a VPN gateway.

        This request lists all connections of a VPN gateway.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str status: (optional) Filters the collection to VPN gateway
               connections with the specified status.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionCollection` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_gateway_connections')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'status': status,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id']
        path_param_values = self.encode_path_vars(vpn_gateway_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpn_gateway_connection(self,
        vpn_gateway_id: str,
        vpn_gateway_connection_prototype: 'VPNGatewayConnectionPrototype',
        **kwargs
    ) -> DetailedResponse:
        """
        Create a connection for a VPN gateway.

        This request creates a new VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param VPNGatewayConnectionPrototype vpn_gateway_connection_prototype: The
               VPN gateway connection prototype object.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if vpn_gateway_connection_prototype is None:
            raise ValueError('vpn_gateway_connection_prototype must be provided')
        if isinstance(vpn_gateway_connection_prototype, VPNGatewayConnectionPrototype):
            vpn_gateway_connection_prototype = convert_model(vpn_gateway_connection_prototype)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_gateway_connection_prototype)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id']
        path_param_values = self.encode_path_vars(vpn_gateway_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpn_gateway_connection(self,
        vpn_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPN gateway connection.

        This request deletes a VPN gateway connection. This operation cannot be reversed.
        For this request to succeed, there must not be VPC routes using this VPN
        connection as a next hop.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_gateway_connection(self,
        vpn_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPN gateway connection.

        This request retrieves a single VPN gateway connection specified by the identifier
        in the URL.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id', 'id']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpn_gateway_connection(self,
        vpn_gateway_id: str,
        id: str,
        vpn_gateway_connection_patch: 'VPNGatewayConnectionPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPN gateway connection.

        This request updates the properties of an existing VPN gateway connection.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param VPNGatewayConnectionPatch vpn_gateway_connection_patch: The VPN
               gateway connection patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnection` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if vpn_gateway_connection_patch is None:
            raise ValueError('vpn_gateway_connection_patch must be provided')
        if isinstance(vpn_gateway_connection_patch, VPNGatewayConnectionPatch):
            vpn_gateway_connection_patch = convert_model(vpn_gateway_connection_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpn_gateway_connection')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_gateway_connection_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id', 'id']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_gateway_connection_local_cidrs(self,
        vpn_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all local CIDRs for a VPN gateway connection.

        This request lists all local CIDRs for a VPN gateway connection.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionLocalCIDRs` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_gateway_connection_local_cidrs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id', 'id']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def remove_vpn_gateway_connection_local_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Remove a local CIDR from a VPN gateway connection.

        This request removes a CIDR from a VPN gateway connection.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='remove_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def check_vpn_gateway_connection_local_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Check if the specified local CIDR exists on a VPN gateway connection.

        This request succeeds if a CIDR exists on the specified VPN gateway connection,
        and fails otherwise.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='check_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def add_vpn_gateway_connection_local_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Set a local CIDR on a VPN gateway connection.

        This request adds the specified CIDR to the specified VPN gateway connection. This
        request succeeds if the specified CIDR already exists. A request body is not
        required, and if provided, is ignored.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='add_vpn_gateway_connection_local_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/local_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_gateway_connection_peer_cidrs(self,
        vpn_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all peer CIDRs for a VPN gateway connection.

        This request lists all peer CIDRs for a VPN gateway connection.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNGatewayConnectionPeerCIDRs` object
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_gateway_connection_peer_cidrs')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_gateway_id', 'id']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def remove_vpn_gateway_connection_peer_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Remove a peer CIDR from a VPN gateway connection.

        This request removes a CIDR from a VPN gateway connection.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='remove_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def check_vpn_gateway_connection_peer_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Check if the specified peer CIDR exists on a VPN gateway connection.

        This request succeeds if a CIDR exists on the specified VPN gateway connection,
        and fails otherwise.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='check_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def add_vpn_gateway_connection_peer_cidr(self,
        vpn_gateway_id: str,
        id: str,
        cidr_prefix: str,
        prefix_length: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Set a peer CIDR on a VPN gateway connection.

        This request adds the specified CIDR to the specified VPN gateway connection. This
        request succeeds if the specified CIDR already exists. A request body is not
        required, and if provided, is ignored.
        This request is only supported for policy mode VPN gateways.

        :param str vpn_gateway_id: The VPN gateway identifier.
        :param str id: The VPN gateway connection identifier.
        :param str cidr_prefix: The address prefix part of the CIDR.
        :param str prefix_length: The prefix length part of the CIDR.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_gateway_id:
            raise ValueError('vpn_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if not cidr_prefix:
            raise ValueError('cidr_prefix must be provided')
        if not prefix_length:
            raise ValueError('prefix_length must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='add_vpn_gateway_connection_peer_cidr')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_gateway_id', 'id', 'cidr_prefix', 'prefix_length']
        path_param_values = self.encode_path_vars(vpn_gateway_id, id, cidr_prefix, prefix_length)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_gateways/{vpn_gateway_id}/connections/{id}/peer_cidrs/{cidr_prefix}/{prefix_length}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response

    #########################
    # VPN servers
    #########################


    def list_vpn_servers(self,
        *,
        name: str = None,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN servers.

        This request lists all VPN servers.

        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_servers')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'name': name,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpn_servers'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpn_server(self,
        certificate: 'CertificateInstanceIdentity',
        client_authentication: List['VPNServerAuthenticationPrototype'],
        client_ip_pool: str,
        subnets: List['SubnetIdentity'],
        *,
        client_dns_server_ips: List['IP'] = None,
        client_idle_timeout: int = None,
        enable_split_tunneling: bool = None,
        name: str = None,
        port: int = None,
        protocol: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        security_groups: List['SecurityGroupIdentity'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a VPN server.

        This request creates a new VPN server.

        :param CertificateInstanceIdentity certificate: The certificate instance
               for this VPN server.
        :param List[VPNServerAuthenticationPrototype] client_authentication: The
               methods used to authenticate VPN clients to this VPN server. VPN clients
               must authenticate against all specified methods.
        :param str client_ip_pool: The VPN client IPv4 address pool, expressed in
               CIDR format. The request must not overlap with any existing address
               prefixes in the VPC or any of the following reserved address ranges:
                 - `127.0.0.0/8` (IPv4 loopback addresses)
                 - `161.26.0.0/16` (IBM services)
                 - `166.8.0.0/14` (Cloud Service Endpoints)
                 - `169.254.0.0/16` (IPv4 link-local addresses)
                 - `224.0.0.0/4` (IPv4 multicast addresses)
               The prefix length of the client IP address pool's CIDR must be between
               `/9` (8,388,608 addresses) and `/22` (1024 addresses). A CIDR block that
               contains twice the number of IP addresses that are required to enable the
               maximum number of concurrent connections is recommended.
        :param List[SubnetIdentity] subnets: The subnets to provision this VPN
               server in.  Use subnets in different zones for high availability.
        :param List[IP] client_dns_server_ips: (optional) The DNS server addresses
               that will be provided to VPN clients connected to this VPN server.
        :param int client_idle_timeout: (optional) The seconds a VPN client can be
               idle before this VPN server will disconnect it.   Specify `0` to prevent
               the server from disconnecting idle clients.
        :param bool enable_split_tunneling: (optional) Indicates whether the split
               tunneling is enabled on this VPN server.
        :param str name: (optional) The name for this VPN server. The name must not
               be used by another VPN server in the VPC. If unspecified, the name will be
               a hyphenated list of randomly-selected words.
        :param int port: (optional) The port number to use for this VPN server.
        :param str protocol: (optional) The transport protocol to use for this VPN
               server.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this VPN server. If unspecified, the VPC's default
               security group is used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServer` object
        """

        if certificate is None:
            raise ValueError('certificate must be provided')
        if client_authentication is None:
            raise ValueError('client_authentication must be provided')
        if client_ip_pool is None:
            raise ValueError('client_ip_pool must be provided')
        if subnets is None:
            raise ValueError('subnets must be provided')
        certificate = convert_model(certificate)
        client_authentication = [convert_model(x) for x in client_authentication]
        subnets = [convert_model(x) for x in subnets]
        if client_dns_server_ips is not None:
            client_dns_server_ips = [convert_model(x) for x in client_dns_server_ips]
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if security_groups is not None:
            security_groups = [convert_model(x) for x in security_groups]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpn_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'certificate': certificate,
            'client_authentication': client_authentication,
            'client_ip_pool': client_ip_pool,
            'subnets': subnets,
            'client_dns_server_ips': client_dns_server_ips,
            'client_idle_timeout': client_idle_timeout,
            'enable_split_tunneling': enable_split_tunneling,
            'name': name,
            'port': port,
            'protocol': protocol,
            'resource_group': resource_group,
            'security_groups': security_groups,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/vpn_servers'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpn_server(self,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPN server.

        This request deletes a VPN server. This operation cannot be reversed.

        :param str id: The VPN server identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_server(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPN server.

        This request retrieves a single VPN server specified by the identifier in the URL.

        :param str id: The VPN server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServer` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpn_server(self,
        id: str,
        vpn_server_patch: 'VPNServerPatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPN server.

        This request updates the properties of an existing VPN server. Any property
        changes will cause all connected VPN clients are disconnected from this VPN server
        except for the name change.

        :param str id: The VPN server identifier.
        :param VPNServerPatch vpn_server_patch: The VPN server patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServer` object
        """

        if not id:
            raise ValueError('id must be provided')
        if vpn_server_patch is None:
            raise ValueError('vpn_server_patch must be provided')
        if isinstance(vpn_server_patch, VPNServerPatch):
            vpn_server_patch = convert_model(vpn_server_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpn_server')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_server_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_server_client_configuration(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve client configuration.

        This request retrieves OpenVPN client configuration on a single VPN server
        specified by the identifier in the URL. This configuration includes directives
        compatible with OpenVPN releases 2.4 and 2.5.

        :param str id: The VPN server identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `str` result
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_server_client_configuration')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'text/plain'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{id}/client_configuration'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_server_clients(self,
        vpn_server_id: str,
        *,
        start: str = None,
        limit: int = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN clients for a VPN server.

        This request retrieves all connected VPN clients, and any disconnected VPN clients
        that the VPN server has not yet deleted based on its auto-deletion policy.

        :param str vpn_server_id: The VPN server identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerClientCollection` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_server_clients')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id']
        path_param_values = self.encode_path_vars(vpn_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/clients'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_vpn_server_client(self,
        vpn_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPN client.

        This request disconnects and deletes the VPN client from the VPN server. The VPN
        client may reconnect unless its authentication permissions for the configured
        authentication methods (such as its client certificate) have been revoked.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN client identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_server_client')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/clients/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_server_client(self,
        vpn_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPN client.

        This request retrieves a single VPN client specified by the identifier in the URL.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN client identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerClient` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_server_client')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/clients/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def disconnect_vpn_client(self,
        vpn_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Disconnect a VPN client.

        This request disconnects the specified VPN client, and deletes the client
        according to the VPN server's auto-deletion policy. The VPN client may reconnect
        unless its authentication permissions for the configured authentication methods
        (such as its client certificate) have been revoked.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN client identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='disconnect_vpn_client')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/clients/{id}/disconnect'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_vpn_server_routes(self,
        vpn_server_id: str,
        *,
        start: str = None,
        limit: int = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all VPN routes for a VPN server.

        This request lists all VPN routes in a VPN server. All VPN routes are provided to
        the VPN client when the connection is established.  Packets received from the VPN
        client will be dropped by the VPN server if there is no VPN route matching their
        specified destinations. All VPN routes must be unique within the VPN server.

        :param str vpn_server_id: The VPN server identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerRouteCollection` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_vpn_server_routes')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id']
        path_param_values = self.encode_path_vars(vpn_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_vpn_server_route(self,
        vpn_server_id: str,
        destination: str,
        *,
        action: str = None,
        name: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a VPN route for a VPN server.

        This request creates a new VPN route in the VPN server. All VPN routes are
        provided to the VPN client when the connection is established. Packets received
        from the VPN client will be dropped by the VPN server if there is no VPN route
        matching their specified destinations. All VPN routes must be unique within the
        VPN server. destination of the packet.

        :param str vpn_server_id: The VPN server identifier.
        :param str destination: The destination to use for this VPN route in the
               VPN server. Must be unique within the VPN server. If an incoming packet
               does not match any destination, it will be dropped.
        :param str action: (optional) The action to perform with a packet matching
               the VPN route:
               - `translate`: translate the source IP address to one of the private IP
               addresses of the VPN server, then deliver the packet to target.
               - `deliver`: deliver the packet to the target.
               - `drop`: drop the packet
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the VPN route
               on which the unexpected property value was encountered.
        :param str name: (optional) The name for this VPN server route. The name
               must not be used by another route for the VPN server. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerRoute` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if destination is None:
            raise ValueError('destination must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_vpn_server_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'destination': destination,
            'action': action,
            'name': name,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id']
        path_param_values = self.encode_path_vars(vpn_server_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/routes'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_vpn_server_route(self,
        vpn_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a VPN route.

        This request deletes a VPN route. This operation cannot be reversed.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_vpn_server_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_vpn_server_route(self,
        vpn_server_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a VPN route.

        This request retrieves a single VPN route specified by the identifier in the URL.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN route identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerRoute` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_vpn_server_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_vpn_server_route(self,
        vpn_server_id: str,
        id: str,
        vpn_server_route_patch: 'VPNServerRoutePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a VPN route.

        This request updates a VPN route with the information in a provided VPN route
        patch. The VPN route patch object is structured in the same way as a retrieved VPN
        route and contains only the information to be updated.

        :param str vpn_server_id: The VPN server identifier.
        :param str id: The VPN route identifier.
        :param VPNServerRoutePatch vpn_server_route_patch: The VPN route patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `VPNServerRoute` object
        """

        if not vpn_server_id:
            raise ValueError('vpn_server_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if vpn_server_route_patch is None:
            raise ValueError('vpn_server_route_patch must be provided')
        if isinstance(vpn_server_route_patch, VPNServerRoutePatch):
            vpn_server_route_patch = convert_model(vpn_server_route_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_vpn_server_route')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(vpn_server_route_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['vpn_server_id', 'id']
        path_param_values = self.encode_path_vars(vpn_server_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/vpn_servers/{vpn_server_id}/routes/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Load balancers
    #########################


    def list_load_balancer_profiles(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all load balancer profiles.

        This request lists all load balancer profiles available in the region. A load
        balancer profile specifies the performance characteristics and pricing model for a
        load balancer.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerProfileCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_profiles')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/load_balancer/profiles'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_profile(self,
        name: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer profile.

        This request retrieves a load balancer profile specified by the name in the URL.

        :param str name: The load balancer profile name.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerProfile` object
        """

        if not name:
            raise ValueError('name must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_profile')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['name']
        path_param_values = self.encode_path_vars(name)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancer/profiles/{name}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancers(self,
        *,
        start: str = None,
        limit: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all load balancers.

        This request lists all load balancers in the region.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancers')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/load_balancers'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer(self,
        is_public: bool,
        subnets: List['SubnetIdentity'],
        *,
        datapath: 'LoadBalancerLoggingDatapathPrototype' = None,
        dns: 'LoadBalancerDNSPrototype' = None,
        listeners: List['LoadBalancerListenerPrototypeLoadBalancerContext'] = None,
        logging: 'LoadBalancerLoggingPrototype' = None,
        name: str = None,
        pools: List['LoadBalancerPoolPrototype'] = None,
        profile: 'LoadBalancerProfileIdentity' = None,
        resource_group: 'ResourceGroupIdentity' = None,
        route_mode: bool = None,
        security_groups: List['SecurityGroupIdentity'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a load balancer.

        This request creates and provisions a new load balancer.

        :param bool is_public: Indicates whether this load balancer is public.
               At present, if route mode is enabled, the load balancer must not be public.
        :param List[SubnetIdentity] subnets: The subnets to provision this load
               balancer in. The subnets must be in the same VPC. The load balancer's
               availability will depend on the availability of the zones that the subnets
               reside in.
               Load balancers in the `network` family allow only one subnet to be
               specified.
        :param LoadBalancerLoggingDatapathPrototype datapath: (optional) The
               datapath logging configuration for this load balancer.
        :param LoadBalancerDNSPrototype dns: (optional)
        :param List[LoadBalancerListenerPrototypeLoadBalancerContext] listeners:
               (optional) The listeners of this load balancer.
        :param LoadBalancerLoggingPrototype logging: (optional) The logging
               configuration to use for this load balancer. See [VPC Datapath
               Logging](https://cloud.ibm.com/docs/vpc?topic=vpc-datapath-logging) on the
               logging
               format, fields and permitted values.
               To activate logging, the load balancer profile must support the specified
               logging type.
        :param str name: (optional) The name for this load balancer. The name must
               not be used by another load balancer in the VPC.  If unspecified, the name
               will be a hyphenated list of randomly-selected words.
        :param List[LoadBalancerPoolPrototype] pools: (optional) The pools of this
               load balancer.
        :param LoadBalancerProfileIdentity profile: (optional) The profile to use
               for this load balancer.
               If unspecified, `application` will be used.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param bool route_mode: (optional) Indicates whether route mode is enabled
               for this load balancer.
               At present, public load balancers are not supported with route mode
               enabled.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this load balancer. If unspecified, the VPC's default
               security group is used.
               The load balancer profile must support security groups.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if is_public is None:
            raise ValueError('is_public must be provided')
        if subnets is None:
            raise ValueError('subnets must be provided')
        subnets = [convert_model(x) for x in subnets]
        if datapath is not None:
            datapath = convert_model(datapath)
        if dns is not None:
            dns = convert_model(dns)
        if listeners is not None:
            listeners = [convert_model(x) for x in listeners]
        if logging is not None:
            logging = convert_model(logging)
        if pools is not None:
            pools = [convert_model(x) for x in pools]
        if profile is not None:
            profile = convert_model(profile)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if security_groups is not None:
            security_groups = [convert_model(x) for x in security_groups]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'is_public': is_public,
            'subnets': subnets,
            'datapath': datapath,
            'dns': dns,
            'listeners': listeners,
            'logging': logging,
            'name': name,
            'pools': pools,
            'profile': profile,
            'resource_group': resource_group,
            'route_mode': route_mode,
            'security_groups': security_groups,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/load_balancers'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer(self,
        id: str,
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer.

        This request deletes a load balancer. This operation cannot be reversed. A load
        balancer cannot be deleted if its `provisioning_status` is `delete_pending`.

        :param str id: The load balancer identifier.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer.

        This request retrieves a single load balancer specified by the identifier in the
        URL path.

        :param str id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer(self,
        id: str,
        load_balancer_patch: 'LoadBalancerPatch',
        *,
        if_match: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer.

        This request updates a load balancer with the information in a provided load
        balancer patch. The load balancer patch object is structured in the same way as a
        retrieved load balancer and contains only the information to be updated. A load
        balancer can only be updated if its `provisioning_status` is `active`.

        :param str id: The load balancer identifier.
        :param LoadBalancerPatch load_balancer_patch: The load balancer patch.
        :param str if_match: (optional) If present, the request will fail if the
               specified ETag value does not match the resource's current ETag value.
               Required if the request body includes an array.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancer` object
        """

        if not id:
            raise ValueError('id must be provided')
        if load_balancer_patch is None:
            raise ValueError('load_balancer_patch must be provided')
        if isinstance(load_balancer_patch, LoadBalancerPatch):
            load_balancer_patch = convert_model(load_balancer_patch)
        headers = {
            'If-Match': if_match,
        }
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_statistics(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all statistics of a load balancer.

        This request lists statistics of a load balancer.

        :param str id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerStatistics` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_statistics')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{id}/statistics'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancer_listeners(self,
        load_balancer_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all listeners for a load balancer.

        This request lists all listeners for a load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_listeners')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id']
        path_param_values = self.encode_path_vars(load_balancer_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer_listener(self,
        load_balancer_id: str,
        protocol: str,
        *,
        accept_proxy_protocol: bool = None,
        certificate_instance: 'CertificateInstanceIdentity' = None,
        connection_limit: int = None,
        default_pool: 'LoadBalancerPoolIdentity' = None,
        https_redirect: 'LoadBalancerListenerHTTPSRedirectPrototype' = None,
        idle_connection_timeout: int = None,
        policies: List['LoadBalancerListenerPolicyPrototype'] = None,
        port: int = None,
        port_max: int = None,
        port_min: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a listener for a load balancer.

        This request creates a new listener for a load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param str protocol: The listener protocol. Each listener in the load
               balancer must have a unique `port` and `protocol` combination.
               Load balancers in the `network` family support `tcp` and `udp` (if
               `udp_supported` is `true`). Load balancers in the `application` family
               support `tcp`, `http` and
               `https`.
               Additional restrictions:
               - If `default_pool` is set, the pool's protocol must match, or be
               compatible with
                 the listener's protocol. At present, the compatible protocols are `http`
               and
                 `https`.
               - If `https_redirect` is set, the protocol must be `http`.
        :param bool accept_proxy_protocol: (optional) If set to `true`, this
               listener will accept and forward PROXY protocol information. Supported by
               load balancers in the `application` family (otherwise always `false`).
               Additional restrictions:
               - If this listener has `https_redirect` specified, its
               `accept_proxy_protocol` value must
                 match the `accept_proxy_protocol` value of the `https_redirect` listener.
               - If this listener is the target of another listener's `https_redirect`,
               its
                 `accept_proxy_protocol` value must match that listener's
               `accept_proxy_protocol` value.
        :param CertificateInstanceIdentity certificate_instance: (optional) The
               certificate instance to use for SSL termination. The listener must have a
               `protocol` of `https`.
        :param int connection_limit: (optional) The connection limit of the
               listener.
        :param LoadBalancerPoolIdentity default_pool: (optional) The default pool
               for this listener. If specified, the pool must:
               - Belong to this load balancer.
               - Have the same `protocol` as this listener, or have a compatible protocol.
                 At present, the compatible protocols are `http` and `https`.
               - Not already be the `default_pool` for another listener.
               If unspecified, this listener will be created with no default pool, but one
               may be
               subsequently set.
        :param LoadBalancerListenerHTTPSRedirectPrototype https_redirect:
               (optional) The target listener that requests will be redirected to. This
               listener must have a
               `protocol` of `http`, and the target listener must have a `protocol` of
               `https`.
        :param int idle_connection_timeout: (optional) The idle connection timeout
               of the listener in seconds. Supported for load balancers in the
               `application` family.
        :param List[LoadBalancerListenerPolicyPrototype] policies: (optional) The
               policy prototype objects for this listener. The load balancer must be in
               the
               `application` family.
        :param int port: (optional) The listener port number, or the inclusive
               lower bound of the port range. Each listener in the load balancer must have
               a unique `port` and `protocol` combination.
               Not supported for load balancers operating with route mode enabled.
        :param int port_max: (optional) The inclusive upper bound of the range of
               ports used by this listener. Must not be less than `port_min`.
               At present, only load balancers operating with route mode enabled, and
               public load balancers in the `network` family support different values for
               `port_min` and
               `port_max`. When route mode is enabled, the value `65535` must be
               specified.
               The specified port range must not overlap with port ranges used by other
               listeners for this load balancer using the same protocol.
        :param int port_min: (optional) The inclusive lower bound of the range of
               ports used by this listener. Must not be greater than `port_max`.
               At present, only load balancers operating with route mode enabled, and
               public load balancers in the `network` family support different values for
               `port_min` and
               `port_max`. When route mode is enabled, the value `1` must be specified.
               The specified port range must not overlap with port ranges used by other
               listeners for this load balancer using the same protocol.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if protocol is None:
            raise ValueError('protocol must be provided')
        if certificate_instance is not None:
            certificate_instance = convert_model(certificate_instance)
        if default_pool is not None:
            default_pool = convert_model(default_pool)
        if https_redirect is not None:
            https_redirect = convert_model(https_redirect)
        if policies is not None:
            policies = [convert_model(x) for x in policies]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_listener')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'protocol': protocol,
            'accept_proxy_protocol': accept_proxy_protocol,
            'certificate_instance': certificate_instance,
            'connection_limit': connection_limit,
            'default_pool': default_pool,
            'https_redirect': https_redirect,
            'idle_connection_timeout': idle_connection_timeout,
            'policies': policies,
            'port': port,
            'port_max': port_max,
            'port_min': port_min,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id']
        path_param_values = self.encode_path_vars(load_balancer_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer_listener(self,
        load_balancer_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer listener.

        This request deletes a load balancer listener. This operation cannot be reversed.
        For this operation to succeed, the listener must not be the target of another load
        balancer listener.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_listener')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_listener(self,
        load_balancer_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer listener.

        This request retrieves a single listener specified by the identifier in the URL
        path.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_listener')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer_listener(self,
        load_balancer_id: str,
        id: str,
        load_balancer_listener_patch: 'LoadBalancerListenerPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer listener.

        This request updates a load balancer listener from a listener patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The listener identifier.
        :param LoadBalancerListenerPatch load_balancer_listener_patch: The load
               balancer listener patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListener` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if load_balancer_listener_patch is None:
            raise ValueError('load_balancer_listener_patch must be provided')
        if isinstance(load_balancer_listener_patch, LoadBalancerListenerPatch):
            load_balancer_listener_patch = convert_model(load_balancer_listener_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_listener')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_listener_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancer_listener_policies(self,
        load_balancer_id: str,
        listener_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all policies for a load balancer listener.

        This request lists all policies for a load balancer listener.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_listener_policies')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer_listener_policy(self,
        load_balancer_id: str,
        listener_id: str,
        action: str,
        priority: int,
        *,
        name: str = None,
        rules: List['LoadBalancerListenerPolicyRulePrototype'] = None,
        target: 'LoadBalancerListenerPolicyTargetPrototype' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a policy for a load balancer listener.

        Creates a new policy for a load balancer listener.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str action: The policy action.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the policy on
               which the unexpected property value was encountered.
        :param int priority: Priority of the policy. Lower value indicates higher
               priority.
        :param str name: (optional) The name for this policy. The name must not be
               used by another policy for the load balancer listener. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param List[LoadBalancerListenerPolicyRulePrototype] rules: (optional) The
               rule prototype objects for this policy.
        :param LoadBalancerListenerPolicyTargetPrototype target: (optional) - If
               `action` is `forward`, specify a `LoadBalancerPoolIdentity`.
               - If `action` is `redirect`, specify a
               `LoadBalancerListenerPolicyRedirectURLPrototype`.
               - If `action` is `https_redirect`, specify a
                 `LoadBalancerListenerPolicyHTTPSRedirectPrototype`.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if action is None:
            raise ValueError('action must be provided')
        if priority is None:
            raise ValueError('priority must be provided')
        if rules is not None:
            rules = [convert_model(x) for x in rules]
        if target is not None:
            target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'action': action,
            'priority': priority,
            'name': name,
            'rules': rules,
            'target': target,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer_listener_policy(self,
        load_balancer_id: str,
        listener_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer listener policy.

        Deletes a policy of the load balancer listener. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['load_balancer_id', 'listener_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_listener_policy(self,
        load_balancer_id: str,
        listener_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer listener policy.

        Retrieve a single policy specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer_listener_policy(self,
        load_balancer_id: str,
        listener_id: str,
        id: str,
        load_balancer_listener_policy_patch: 'LoadBalancerListenerPolicyPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer listener policy.

        Updates a policy from a policy patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str id: The policy identifier.
        :param LoadBalancerListenerPolicyPatch load_balancer_listener_policy_patch:
               The listener policy patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicy` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if load_balancer_listener_policy_patch is None:
            raise ValueError('load_balancer_listener_policy_patch must be provided')
        if isinstance(load_balancer_listener_policy_patch, LoadBalancerListenerPolicyPatch):
            load_balancer_listener_policy_patch = convert_model(load_balancer_listener_policy_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_listener_policy')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_listener_policy_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancer_listener_policy_rules(self,
        load_balancer_id: str,
        listener_id: str,
        policy_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all rules of a load balancer listener policy.

        This request lists all rules of a load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRuleCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not policy_id:
            raise ValueError('policy_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_listener_policy_rules')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'policy_id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, policy_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer_listener_policy_rule(self,
        load_balancer_id: str,
        listener_id: str,
        policy_id: str,
        condition: str,
        type: str,
        value: str,
        *,
        field: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a rule for a load balancer listener policy.

        Creates a new rule for the load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str condition: The condition of the rule.
        :param str type: The type of the rule.
               Body rules are applied to form-encoded request bodies using the `UTF-8`
               character set.
        :param str value: Value to be matched for rule condition.
               If the rule type is `query` and the rule condition is not `matches_regex`,
               the value must be percent-encoded.
        :param str field: (optional) The field. This is applicable to `header`,
               `query`, and `body` rule types.
               If the rule type is `header`, this property is required.
               If the rule type is `query`, this is optional. If specified and the rule
               condition is not
               `matches_regex`, the value must be percent-encoded.
               If the rule type is `body`, this is optional.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not policy_id:
            raise ValueError('policy_id must be provided')
        if condition is None:
            raise ValueError('condition must be provided')
        if type is None:
            raise ValueError('type must be provided')
        if value is None:
            raise ValueError('value must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'condition': condition,
            'type': type,
            'value': value,
            'field': field,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'policy_id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, policy_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer_listener_policy_rule(self,
        load_balancer_id: str,
        listener_id: str,
        policy_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer listener policy rule.

        Deletes a rule from the load balancer listener policy. This operation cannot be
        reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not policy_id:
            raise ValueError('policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['load_balancer_id', 'listener_id', 'policy_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_listener_policy_rule(self,
        load_balancer_id: str,
        listener_id: str,
        policy_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer listener policy rule.

        Retrieves a single rule specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not policy_id:
            raise ValueError('policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'policy_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer_listener_policy_rule(self,
        load_balancer_id: str,
        listener_id: str,
        policy_id: str,
        id: str,
        load_balancer_listener_policy_rule_patch: 'LoadBalancerListenerPolicyRulePatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer listener policy rule.

        Updates a rule of the load balancer listener policy.

        :param str load_balancer_id: The load balancer identifier.
        :param str listener_id: The listener identifier.
        :param str policy_id: The policy identifier.
        :param str id: The rule identifier.
        :param LoadBalancerListenerPolicyRulePatch
               load_balancer_listener_policy_rule_patch: The listener policy rule patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerListenerPolicyRule` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not listener_id:
            raise ValueError('listener_id must be provided')
        if not policy_id:
            raise ValueError('policy_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if load_balancer_listener_policy_rule_patch is None:
            raise ValueError('load_balancer_listener_policy_rule_patch must be provided')
        if isinstance(load_balancer_listener_policy_rule_patch, LoadBalancerListenerPolicyRulePatch):
            load_balancer_listener_policy_rule_patch = convert_model(load_balancer_listener_policy_rule_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_listener_policy_rule')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_listener_policy_rule_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'listener_id', 'policy_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, listener_id, policy_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/listeners/{listener_id}/policies/{policy_id}/rules/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancer_pools(self,
        load_balancer_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all pools of a load balancer.

        This request lists all pools of a load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_pools')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id']
        path_param_values = self.encode_path_vars(load_balancer_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer_pool(self,
        load_balancer_id: str,
        algorithm: str,
        health_monitor: 'LoadBalancerPoolHealthMonitorPrototype',
        protocol: str,
        *,
        members: List['LoadBalancerPoolMemberPrototype'] = None,
        name: str = None,
        proxy_protocol: str = None,
        session_persistence: 'LoadBalancerPoolSessionPersistencePrototype' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a load balancer pool.

        This request creates a new pool from a pool prototype object.

        :param str load_balancer_id: The load balancer identifier.
        :param str algorithm: The load balancing algorithm.
        :param LoadBalancerPoolHealthMonitorPrototype health_monitor: The health
               monitor of this pool.
        :param str protocol: The protocol used for this load balancer pool. Load
               balancers in the `network` family support `tcp` and `udp` (if
               `udp_supported` is `true`). Load balancers in the
               `application` family support `tcp`, `http`, and `https`.
        :param List[LoadBalancerPoolMemberPrototype] members: (optional) The
               members for this load balancer pool. For load balancers in the `network`
               family, the same `port` and `target` tuple cannot be shared by a pool
               member of any other load balancer in the same VPC.
        :param str name: (optional) The name for this load balancer pool. The name
               must not be used by another pool for the load balancer. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        :param str proxy_protocol: (optional) The PROXY protocol setting for this
               pool:
               - `v1`: Enabled with version 1 (human-readable header format)
               - `v2`: Enabled with version 2 (binary header format)
               - `disabled`: Disabled
               Supported by load balancers in the `application` family (otherwise always
               `disabled`).
        :param LoadBalancerPoolSessionPersistencePrototype session_persistence:
               (optional) The session persistence of this pool.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if algorithm is None:
            raise ValueError('algorithm must be provided')
        if health_monitor is None:
            raise ValueError('health_monitor must be provided')
        if protocol is None:
            raise ValueError('protocol must be provided')
        health_monitor = convert_model(health_monitor)
        if members is not None:
            members = [convert_model(x) for x in members]
        if session_persistence is not None:
            session_persistence = convert_model(session_persistence)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_pool')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'algorithm': algorithm,
            'health_monitor': health_monitor,
            'protocol': protocol,
            'members': members,
            'name': name,
            'proxy_protocol': proxy_protocol,
            'session_persistence': session_persistence,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id']
        path_param_values = self.encode_path_vars(load_balancer_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer_pool(self,
        load_balancer_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer pool.

        This request deletes a load balancer pool. This operation cannot be reversed. The
        pool must not currently be the default pool for any listener in the load balancer.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_pool')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_pool(self,
        load_balancer_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer pool.

        This request retrieves a single pool specified by the identifier in the URL path.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_pool')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer_pool(self,
        load_balancer_id: str,
        id: str,
        load_balancer_pool_patch: 'LoadBalancerPoolPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer pool.

        This request updates a load balancer pool from a pool patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str id: The pool identifier.
        :param LoadBalancerPoolPatch load_balancer_pool_patch: The load balancer
               pool patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPool` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if load_balancer_pool_patch is None:
            raise ValueError('load_balancer_pool_patch must be provided')
        if isinstance(load_balancer_pool_patch, LoadBalancerPoolPatch):
            load_balancer_pool_patch = convert_model(load_balancer_pool_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_pool')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_pool_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_load_balancer_pool_members(self,
        load_balancer_id: str,
        pool_id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        List all members of a load balancer pool.

        This request lists all members of a load balancer pool.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMemberCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_load_balancer_pool_members')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'pool_id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_load_balancer_pool_member(self,
        load_balancer_id: str,
        pool_id: str,
        port: int,
        target: 'LoadBalancerPoolMemberTargetPrototype',
        *,
        weight: int = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a member in a load balancer pool.

        This request creates a new member and adds the member to the pool.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param int port: The port the member will receive load balancer traffic on.
               Applies only to load balancer traffic received on a listener with a single
               port. (If the traffic is received on a listener with a port range, the
               member will receive the traffic on the same port the listener received it
               on.)
               This port will also be used for health checks unless the `port` property of
               `health_monitor` property is specified.
               The port must be unique across all members for all pools associated with
               this pool's listener.
        :param LoadBalancerPoolMemberTargetPrototype target: The pool member
               target. Load balancers in the `network` family support virtual server
               instances. Load balancers in the `application` family support IP addresses.
               If the load
               balancer has route mode enabled, the member must be in a zone the load
               balancer has a
               subnet in.
        :param int weight: (optional) Weight of the server member. Applicable only
               if the pool algorithm is
               `weighted_round_robin`.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        if port is None:
            raise ValueError('port must be provided')
        if target is None:
            raise ValueError('target must be provided')
        target = convert_model(target)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'port': port,
            'target': target,
            'weight': weight,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'pool_id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members'.format(**path_param_dict)
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def replace_load_balancer_pool_members(self,
        load_balancer_id: str,
        pool_id: str,
        members: List['LoadBalancerPoolMemberPrototype'],
        **kwargs
    ) -> DetailedResponse:
        """
        Replace load balancer pool members.

        This request replaces the existing members of the load balancer pool with new
        members created from the collection of member prototype objects.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param List[LoadBalancerPoolMemberPrototype] members: The member prototype
               objects for this pool.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMemberCollection` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        if members is None:
            raise ValueError('members must be provided')
        members = [convert_model(x) for x in members]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='replace_load_balancer_pool_members')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'members': members,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'pool_id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_load_balancer_pool_member(self,
        load_balancer_id: str,
        pool_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a load balancer pool member.

        This request deletes a member from the pool. This operation cannot be reversed.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['load_balancer_id', 'pool_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_load_balancer_pool_member(self,
        load_balancer_id: str,
        pool_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a load balancer pool member.

        This request retrieves a single member specified by the identifier in the URL
        path.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'pool_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_load_balancer_pool_member(self,
        load_balancer_id: str,
        pool_id: str,
        id: str,
        load_balancer_pool_member_patch: 'LoadBalancerPoolMemberPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a load balancer pool member.

        This request updates an existing member from a member patch.

        :param str load_balancer_id: The load balancer identifier.
        :param str pool_id: The pool identifier.
        :param str id: The member identifier.
        :param LoadBalancerPoolMemberPatch load_balancer_pool_member_patch: The
               load balancer pool member patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `LoadBalancerPoolMember` object
        """

        if not load_balancer_id:
            raise ValueError('load_balancer_id must be provided')
        if not pool_id:
            raise ValueError('pool_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        if load_balancer_pool_member_patch is None:
            raise ValueError('load_balancer_pool_member_patch must be provided')
        if isinstance(load_balancer_pool_member_patch, LoadBalancerPoolMemberPatch):
            load_balancer_pool_member_patch = convert_model(load_balancer_pool_member_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_load_balancer_pool_member')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(load_balancer_pool_member_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['load_balancer_id', 'pool_id', 'id']
        path_param_values = self.encode_path_vars(load_balancer_id, pool_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/load_balancers/{load_balancer_id}/pools/{pool_id}/members/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Endpoint gateways
    #########################


    def list_endpoint_gateways(self,
        *,
        name: str = None,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all endpoint gateways.

        This request lists all endpoint gateways in the region. An endpoint gateway maps
        one or more reserved IPs in a VPC to a target outside the VPC.

        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `EndpointGatewayCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_endpoint_gateways')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'name': name,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/endpoint_gateways'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_endpoint_gateway(self,
        target: 'EndpointGatewayTargetPrototype',
        vpc: 'VPCIdentity',
        *,
        ips: List['EndpointGatewayReservedIP'] = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        security_groups: List['SecurityGroupIdentity'] = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create an endpoint gateway.

        This request creates a new endpoint gateway. An endpoint gateway maps one or more
        reserved IPs in a VPC to a target outside the VPC.

        :param EndpointGatewayTargetPrototype target: The target to use for this
               endpoint gateway. Must not already be the target of another
               endpoint gateway in the VPC.
        :param VPCIdentity vpc: The VPC this endpoint gateway will reside in.
        :param List[EndpointGatewayReservedIP] ips: (optional) The reserved IPs to
               bind to this endpoint gateway. At most one reserved IP per zone is allowed.
        :param str name: (optional) The name for this endpoint gateway. The name
               must not be used by another endpoint gateway in the VPC. If unspecified,
               the name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this endpoint gateway. If unspecified, the VPC's default
               security group is used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `EndpointGateway` object
        """

        if target is None:
            raise ValueError('target must be provided')
        if vpc is None:
            raise ValueError('vpc must be provided')
        target = convert_model(target)
        vpc = convert_model(vpc)
        if ips is not None:
            ips = [convert_model(x) for x in ips]
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        if security_groups is not None:
            security_groups = [convert_model(x) for x in security_groups]
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_endpoint_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'target': target,
            'vpc': vpc,
            'ips': ips,
            'name': name,
            'resource_group': resource_group,
            'security_groups': security_groups,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/endpoint_gateways'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def list_endpoint_gateway_ips(self,
        endpoint_gateway_id: str,
        *,
        start: str = None,
        limit: int = None,
        sort: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all reserved IPs bound to an endpoint gateway.

        This request lists all reserved IPs bound to an endpoint gateway.

        :param str endpoint_gateway_id: The endpoint gateway identifier.
        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str sort: (optional) Sorts the returned collection by the specified
               property name in ascending order. A `-` may be prepended to the name to
               sort in descending order. For example, the value `-created_at` sorts the
               collection by the `created_at` property in descending order, and the value
               `name` sorts it by the `name` property in ascending order.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIPCollectionEndpointGatewayContext` object
        """

        if not endpoint_gateway_id:
            raise ValueError('endpoint_gateway_id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_endpoint_gateway_ips')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'sort': sort,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['endpoint_gateway_id']
        path_param_values = self.encode_path_vars(endpoint_gateway_id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{endpoint_gateway_id}/ips'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def remove_endpoint_gateway_ip(self,
        endpoint_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Unbind a reserved IP from an endpoint gateway.

        This request unbinds the specified reserved IP from the specified endpoint
        gateway. If the reserved IP has `auto_delete` set to `true`, the reserved IP will
        be deleted.

        :param str endpoint_gateway_id: The endpoint gateway identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not endpoint_gateway_id:
            raise ValueError('endpoint_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='remove_endpoint_gateway_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['endpoint_gateway_id', 'id']
        path_param_values = self.encode_path_vars(endpoint_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{endpoint_gateway_id}/ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_endpoint_gateway_ip(self,
        endpoint_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a reserved IP bound to an endpoint gateway.

        This request retrieves the specified reserved IP address if it is bound to the
        endpoint gateway specified in the URL.

        :param str endpoint_gateway_id: The endpoint gateway identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not endpoint_gateway_id:
            raise ValueError('endpoint_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_endpoint_gateway_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['endpoint_gateway_id', 'id']
        path_param_values = self.encode_path_vars(endpoint_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{endpoint_gateway_id}/ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def add_endpoint_gateway_ip(self,
        endpoint_gateway_id: str,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Bind a reserved IP to an endpoint gateway.

        This request binds the specified reserved IP to the specified endpoint gateway.
        The reserved IP:
        - must currently be unbound
        - must not be in the same zone as any other reserved IP bound to the endpoint
        gateway.

        :param str endpoint_gateway_id: The endpoint gateway identifier.
        :param str id: The reserved IP identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `ReservedIP` object
        """

        if not endpoint_gateway_id:
            raise ValueError('endpoint_gateway_id must be provided')
        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='add_endpoint_gateway_ip')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['endpoint_gateway_id', 'id']
        path_param_values = self.encode_path_vars(endpoint_gateway_id, id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{endpoint_gateway_id}/ips/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PUT',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def delete_endpoint_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete an endpoint gateway.

        This request deletes an endpoint gateway. This operation cannot be reversed.
        Reserved IPs that were bound to the endpoint gateway will be released if their
        `auto_delete` property is set to true.

        :param str id: The endpoint gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_endpoint_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_endpoint_gateway(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve an endpoint gateway.

        This request retrieves a single endpoint gateway specified by the identifier in
        the URL.

        :param str id: The endpoint gateway identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `EndpointGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_endpoint_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_endpoint_gateway(self,
        id: str,
        endpoint_gateway_patch: 'EndpointGatewayPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update an endpoint gateway.

        This request updates an endpoint gateway's name.

        :param str id: The endpoint gateway identifier.
        :param EndpointGatewayPatch endpoint_gateway_patch: The endpoint gateway
               patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `EndpointGateway` object
        """

        if not id:
            raise ValueError('id must be provided')
        if endpoint_gateway_patch is None:
            raise ValueError('endpoint_gateway_patch must be provided')
        if isinstance(endpoint_gateway_patch, EndpointGatewayPatch):
            endpoint_gateway_patch = convert_model(endpoint_gateway_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_endpoint_gateway')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(endpoint_gateway_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/endpoint_gateways/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response

    #########################
    # Flow log collectors
    #########################


    def list_flow_log_collectors(self,
        *,
        start: str = None,
        limit: int = None,
        resource_group_id: str = None,
        name: str = None,
        vpc_id: str = None,
        vpc_crn: str = None,
        vpc_name: str = None,
        target_id: str = None,
        target_resource_type: str = None,
        **kwargs
    ) -> DetailedResponse:
        """
        List all flow log collectors.

        This request lists all flow log collectors in the region. A flow log collector
        summarizes data sent over one or more network interfaces within a VPC, depending
        on the chosen target.

        :param str start: (optional) A server-provided token determining what
               resource to start the page on.
        :param int limit: (optional) The number of resources to return on a page.
        :param str resource_group_id: (optional) Filters the collection to
               resources in the resource group with the specified identifier.
        :param str name: (optional) Filters the collection to resources with the
               exact specified name.
        :param str vpc_id: (optional) Filters the collection to resources in the
               VPC with the specified identifier.
        :param str vpc_crn: (optional) Filters the collection to resources in the
               VPC with the specified CRN.
        :param str vpc_name: (optional) Filters the collection to resources in the
               VPC with the exact specified name.
        :param str target_id: (optional) Filters the collection to flow log
               collectors that target the specified resource.
        :param str target_resource_type: (optional) Filters the collection to flow
               log collectors that target the specified resource type.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FlowLogCollectorCollection` object
        """

        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='list_flow_log_collectors')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
            'start': start,
            'limit': limit,
            'resource_group.id': resource_group_id,
            'name': name,
            'vpc.id': vpc_id,
            'vpc.crn': vpc_crn,
            'vpc.name': vpc_name,
            'target.id': target_id,
            'target.resource_type': target_resource_type,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/flow_log_collectors'
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def create_flow_log_collector(self,
        storage_bucket: 'LegacyCloudObjectStorageBucketIdentity',
        target: 'FlowLogCollectorTargetPrototype',
        *,
        active: bool = None,
        name: str = None,
        resource_group: 'ResourceGroupIdentity' = None,
        **kwargs
    ) -> DetailedResponse:
        """
        Create a flow log collector.

        This request creates and starts a new flow log collector from a flow log collector
        prototype object. The prototype object is structured in the same way as a
        retrieved flow log collector, and contains the information necessary to create and
        start the new flow log collector.

        :param LegacyCloudObjectStorageBucketIdentity storage_bucket: The Cloud
               Object Storage bucket where the collected flows will be logged.
               The bucket must exist and an IAM service authorization must grant
               `IBM Cloud Flow Logs` resources of `VPC Infrastructure Services` writer
               access to the bucket.
        :param FlowLogCollectorTargetPrototype target: The target this collector
               will collect flow logs for. If the target is an instance,
               subnet, or VPC, flow logs will not be collected for any network interfaces
               within the
               target that are themselves the target of a more specific flow log
               collector.
        :param bool active: (optional) Indicates whether this collector will be
               active upon creation.
        :param str name: (optional) The name for this flow log collector. The name
               must not be used by another flow log collector in the region. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param ResourceGroupIdentity resource_group: (optional) The resource group
               to use. If unspecified, the account's [default resource
               group](https://cloud.ibm.com/apidocs/resource-manager#introduction) is
               used.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FlowLogCollector` object
        """

        if storage_bucket is None:
            raise ValueError('storage_bucket must be provided')
        if target is None:
            raise ValueError('target must be provided')
        storage_bucket = convert_model(storage_bucket)
        target = convert_model(target)
        if resource_group is not None:
            resource_group = convert_model(resource_group)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='create_flow_log_collector')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = {
            'storage_bucket': storage_bucket,
            'target': target,
            'active': active,
            'name': name,
            'resource_group': resource_group,
        }
        data = {k: v for (k, v) in data.items() if v is not None}
        data = json.dumps(data)
        headers['content-type'] = 'application/json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        url = '/flow_log_collectors'
        request = self.prepare_request(method='POST',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


    def delete_flow_log_collector(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Delete a flow log collector.

        This request stops and deletes a flow log collector. This operation cannot be
        reversed.
        Collected flow logs remain available within the flow log collector's Cloud Object
        Storage bucket.

        :param str id: The flow log collector identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='delete_flow_log_collector')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/flow_log_collectors/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='DELETE',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def get_flow_log_collector(self,
        id: str,
        **kwargs
    ) -> DetailedResponse:
        """
        Retrieve a flow log collector.

        This request retrieves a single flow log collector specified by the identifier in
        the URL.

        :param str id: The flow log collector identifier.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FlowLogCollector` object
        """

        if not id:
            raise ValueError('id must be provided')
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='get_flow_log_collector')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/flow_log_collectors/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='GET',
                                       url=url,
                                       headers=headers,
                                       params=params)

        response = self.send(request, **kwargs)
        return response


    def update_flow_log_collector(self,
        id: str,
        flow_log_collector_patch: 'FlowLogCollectorPatch',
        **kwargs
    ) -> DetailedResponse:
        """
        Update a flow log collector.

        This request updates a flow log collector with the information in a provided flow
        log collector patch. The flow log collector patch object is structured in the same
        way as a retrieved flow log collector and contains only the information to be
        updated.

        :param str id: The flow log collector identifier.
        :param FlowLogCollectorPatch flow_log_collector_patch: The flow log
               collector patch.
        :param dict headers: A `dict` containing the request headers
        :return: A `DetailedResponse` containing the result, headers and HTTP status code.
        :rtype: DetailedResponse with `dict` result representing a `FlowLogCollector` object
        """

        if not id:
            raise ValueError('id must be provided')
        if flow_log_collector_patch is None:
            raise ValueError('flow_log_collector_patch must be provided')
        if isinstance(flow_log_collector_patch, FlowLogCollectorPatch):
            flow_log_collector_patch = convert_model(flow_log_collector_patch)
        headers = {}
        sdk_headers = get_sdk_headers(service_name=self.DEFAULT_SERVICE_NAME,
                                      service_version='V1',
                                      operation_id='update_flow_log_collector')
        headers.update(sdk_headers)

        params = {
            'version': self.version,
            'generation': self.generation,
        }

        data = json.dumps(flow_log_collector_patch)
        headers['content-type'] = 'application/merge-patch+json'

        if 'headers' in kwargs:
            headers.update(kwargs.get('headers'))
            del kwargs['headers']
        headers['Accept'] = 'application/json'

        path_param_keys = ['id']
        path_param_values = self.encode_path_vars(id)
        path_param_dict = dict(zip(path_param_keys, path_param_values))
        url = '/flow_log_collectors/{id}'.format(**path_param_dict)
        request = self.prepare_request(method='PATCH',
                                       url=url,
                                       headers=headers,
                                       params=params,
                                       data=data)

        response = self.send(request, **kwargs)
        return response


class ListSubnetReservedIpsEnums:
    """
    Enums for list_subnet_reserved_ips parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        ADDRESS = 'address'
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListImagesEnums:
    """
    Enums for list_images parameters.
    """

    class Visibility(str, Enum):
        """
        Filters the collection to images with the specified `visibility`.
        """
        PRIVATE = 'private'
        PUBLIC = 'public'


class ListBackupPolicyJobsEnums:
    """
    Enums for list_backup_policy_jobs parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListVolumesEnums:
    """
    Enums for list_volumes parameters.
    """

    class AttachmentState(str, Enum):
        """
        Filters the collection to volumes with the specified attachment state.
        """
        ATTACHED = 'attached'
        UNATTACHED = 'unattached'
        UNUSABLE = 'unusable'
    class Encryption(str, Enum):
        """
        Filters the collection to resources with the specified encryption type.
        """
        PROVIDER_MANAGED = 'provider_managed'
        USER_MANAGED = 'user_managed'


class ListSnapshotsEnums:
    """
    Enums for list_snapshots parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListFloatingIpsEnums:
    """
    Enums for list_floating_ips parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListNetworkAclRulesEnums:
    """
    Enums for list_network_acl_rules parameters.
    """

    class Direction(str, Enum):
        """
        Filters the collection to rules with the specified direction.
        """
        INBOUND = 'inbound'
        OUTBOUND = 'outbound'


class ListVpnGatewaysEnums:
    """
    Enums for list_vpn_gateways parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'
    class Mode(str, Enum):
        """
        Filters the collection to VPN gateways with the specified mode.
        """
        POLICY = 'policy'
        ROUTE = 'route'


class ListVpnServersEnums:
    """
    Enums for list_vpn_servers parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListVpnServerClientsEnums:
    """
    Enums for list_vpn_server_clients parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order.
        """
        CREATED_AT = 'created_at'


class ListVpnServerRoutesEnums:
    """
    Enums for list_vpn_server_routes parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListEndpointGatewayIpsEnums:
    """
    Enums for list_endpoint_gateway_ips parameters.
    """

    class Sort(str, Enum):
        """
        Sorts the returned collection by the specified property name in ascending order. A
        `-` may be prepended to the name to sort in descending order. For example, the
        value `-created_at` sorts the collection by the `created_at` property in
        descending order, and the value `name` sorts it by the `name` property in
        ascending order.
        """
        ADDRESS = 'address'
        CREATED_AT = 'created_at'
        NAME = 'name'


class ListFlowLogCollectorsEnums:
    """
    Enums for list_flow_log_collectors parameters.
    """

    class TargetResourceType(str, Enum):
        """
        Filters the collection to flow log collectors that target the specified resource
        type.
        """
        INSTANCE = 'instance'
        NETWORK_INTERFACE = 'network_interface'
        SUBNET = 'subnet'
        VPC = 'vpc'


##############################################################################
# Models
##############################################################################


class AddressPrefix():
    """
    AddressPrefix.

    :attr str cidr: The CIDR block for this prefix.
    :attr datetime created_at: The date and time that the prefix was created.
    :attr bool has_subnets: Indicates whether subnets exist with addresses from this
          prefix.
    :attr str href: The URL for this address prefix.
    :attr str id: The unique identifier for this address prefix.
    :attr bool is_default: Indicates whether this is the default prefix for this
          zone in this VPC. If a default prefix was automatically created when the VPC was
          created, the prefix is automatically named using a hyphenated list of
          randomly-selected words, but may be changed.
    :attr str name: The name for this address prefix. The name must not be used by
          another address prefix for the VPC.
    :attr ZoneReference zone: The zone this address prefix resides in.
    """

    def __init__(self,
                 cidr: str,
                 created_at: datetime,
                 has_subnets: bool,
                 href: str,
                 id: str,
                 is_default: bool,
                 name: str,
                 zone: 'ZoneReference') -> None:
        """
        Initialize a AddressPrefix object.

        :param str cidr: The CIDR block for this prefix.
        :param datetime created_at: The date and time that the prefix was created.
        :param bool has_subnets: Indicates whether subnets exist with addresses
               from this prefix.
        :param str href: The URL for this address prefix.
        :param str id: The unique identifier for this address prefix.
        :param bool is_default: Indicates whether this is the default prefix for
               this zone in this VPC. If a default prefix was automatically created when
               the VPC was created, the prefix is automatically named using a hyphenated
               list of randomly-selected words, but may be changed.
        :param str name: The name for this address prefix. The name must not be
               used by another address prefix for the VPC.
        :param ZoneReference zone: The zone this address prefix resides in.
        """
        self.cidr = cidr
        self.created_at = created_at
        self.has_subnets = has_subnets
        self.href = href
        self.id = id
        self.is_default = is_default
        self.name = name
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefix':
        """Initialize a AddressPrefix object from a json dictionary."""
        args = {}
        if 'cidr' in _dict:
            args['cidr'] = _dict.get('cidr')
        else:
            raise ValueError('Required property \'cidr\' not present in AddressPrefix JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in AddressPrefix JSON')
        if 'has_subnets' in _dict:
            args['has_subnets'] = _dict.get('has_subnets')
        else:
            raise ValueError('Required property \'has_subnets\' not present in AddressPrefix JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in AddressPrefix JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in AddressPrefix JSON')
        if 'is_default' in _dict:
            args['is_default'] = _dict.get('is_default')
        else:
            raise ValueError('Required property \'is_default\' not present in AddressPrefix JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in AddressPrefix JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError('Required property \'zone\' not present in AddressPrefix JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefix object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'cidr') and self.cidr is not None:
            _dict['cidr'] = self.cidr
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'has_subnets') and self.has_subnets is not None:
            _dict['has_subnets'] = self.has_subnets
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'is_default') and self.is_default is not None:
            _dict['is_default'] = self.is_default
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'zone') and self.zone is not None:
            if isinstance(self.zone, dict):
                _dict['zone'] = self.zone
            else:
                _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefix object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefix') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefix') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class AddressPrefixCollection():
    """
    AddressPrefixCollection.

    :attr List[AddressPrefix] address_prefixes: Collection of address prefixes.
    :attr AddressPrefixCollectionFirst first: A link to the first page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr AddressPrefixCollectionNext next: (optional) A link to the next page of
          resources. This property is present for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 address_prefixes: List['AddressPrefix'],
                 first: 'AddressPrefixCollectionFirst',
                 limit: int,
                 total_count: int,
                 *,
                 next: 'AddressPrefixCollectionNext' = None) -> None:
        """
        Initialize a AddressPrefixCollection object.

        :param List[AddressPrefix] address_prefixes: Collection of address
               prefixes.
        :param AddressPrefixCollectionFirst first: A link to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param AddressPrefixCollectionNext next: (optional) A link to the next page
               of resources. This property is present for all pages
               except the last page.
        """
        self.address_prefixes = address_prefixes
        self.first = first
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollection':
        """Initialize a AddressPrefixCollection object from a json dictionary."""
        args = {}
        if 'address_prefixes' in _dict:
            args['address_prefixes'] = [AddressPrefix.from_dict(v) for v in _dict.get('address_prefixes')]
        else:
            raise ValueError('Required property \'address_prefixes\' not present in AddressPrefixCollection JSON')
        if 'first' in _dict:
            args['first'] = AddressPrefixCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError('Required property \'first\' not present in AddressPrefixCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError('Required property \'limit\' not present in AddressPrefixCollection JSON')
        if 'next' in _dict:
            args['next'] = AddressPrefixCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError('Required property \'total_count\' not present in AddressPrefixCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'address_prefixes') and self.address_prefixes is not None:
            address_prefixes_list = []
            for v in self.address_prefixes:
                if isinstance(v, dict):
                    address_prefixes_list.append(v)
                else:
                    address_prefixes_list.append(v.to_dict())
            _dict['address_prefixes'] = address_prefixes_list
        if hasattr(self, 'first') and self.first is not None:
            if isinstance(self.first, dict):
                _dict['first'] = self.first
            else:
                _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            if isinstance(self.next, dict):
                _dict['next'] = self.next
            else:
                _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class AddressPrefixCollectionFirst():
    """
    A link to the first page of resources.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a AddressPrefixCollectionFirst object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollectionFirst':
        """Initialize a AddressPrefixCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in AddressPrefixCollectionFirst JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class AddressPrefixCollectionNext():
    """
    A link to the next page of resources. This property is present for all pages except
    the last page.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a AddressPrefixCollectionNext object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixCollectionNext':
        """Initialize a AddressPrefixCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in AddressPrefixCollectionNext JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class AddressPrefixPatch():
    """
    AddressPrefixPatch.

    :attr bool is_default: (optional) Indicates whether this is the default prefix
          for this zone in this VPC. Updating to true makes this prefix the default prefix
          for this zone in this VPC, provided the VPC currently has no default address
          prefix for this zone. Updating to false removes the default prefix for this zone
          in this VPC.
    :attr str name: (optional) The name for this address prefix. The name must not
          be used by another address prefix for the VPC.
    """

    def __init__(self,
                 *,
                 is_default: bool = None,
                 name: str = None) -> None:
        """
        Initialize a AddressPrefixPatch object.

        :param bool is_default: (optional) Indicates whether this is the default
               prefix for this zone in this VPC. Updating to true makes this prefix the
               default prefix for this zone in this VPC, provided the VPC currently has no
               default address prefix for this zone. Updating to false removes the default
               prefix for this zone in this VPC.
        :param str name: (optional) The name for this address prefix. The name must
               not be used by another address prefix for the VPC.
        """
        self.is_default = is_default
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'AddressPrefixPatch':
        """Initialize a AddressPrefixPatch object from a json dictionary."""
        args = {}
        if 'is_default' in _dict:
            args['is_default'] = _dict.get('is_default')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a AddressPrefixPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'is_default') and self.is_default is not None:
            _dict['is_default'] = self.is_default
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this AddressPrefixPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'AddressPrefixPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'AddressPrefixPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicy():
    """
    BackupPolicy.

    :attr datetime created_at: The date and time that the backup policy was created.
    :attr str crn: The CRN for this backup policy.
    :attr str href: The URL for this backup policy.
    :attr str id: The unique identifier for this backup policy.
    :attr datetime last_job_completed_at: (optional) The date and time that the most
          recent job for this backup policy completed.
          If absent, no job has yet completed for this backup policy.
    :attr str lifecycle_state: The lifecycle state of the backup policy.
    :attr List[str] match_resource_types: A resource type this backup policy applies
          to. Resources that have both a matching type and a matching user tag will be
          subject to the backup policy.
          The enumerated values for this property will expand in the future. When
          processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the backup policy on which the
          unexpected property value was encountered.
    :attr List[str] match_user_tags: The user tags this backup policy applies to.
          Resources that have both a matching user tag and a matching type will be subject
          to the backup policy.
    :attr str name: The name for this backup policy. The name is unique across all
          backup policies in the region.
    :attr List[BackupPolicyPlanReference] plans: The plans for the backup policy.
    :attr ResourceGroupReference resource_group: The resource group for this backup
          policy.
    :attr str resource_type: The resource type.
    """

    def __init__(self,
                 created_at: datetime,
                 crn: str,
                 href: str,
                 id: str,
                 lifecycle_state: str,
                 match_resource_types: List[str],
                 match_user_tags: List[str],
                 name: str,
                 plans: List['BackupPolicyPlanReference'],
                 resource_group: 'ResourceGroupReference',
                 resource_type: str,
                 *,
                 last_job_completed_at: datetime = None) -> None:
        """
        Initialize a BackupPolicy object.

        :param datetime created_at: The date and time that the backup policy was
               created.
        :param str crn: The CRN for this backup policy.
        :param str href: The URL for this backup policy.
        :param str id: The unique identifier for this backup policy.
        :param str lifecycle_state: The lifecycle state of the backup policy.
        :param List[str] match_resource_types: A resource type this backup policy
               applies to. Resources that have both a matching type and a matching user
               tag will be subject to the backup policy.
               The enumerated values for this property will expand in the future. When
               processing this property, check for and log unknown values. Optionally halt
               processing and surface the error, or bypass the backup policy on which the
               unexpected property value was encountered.
        :param List[str] match_user_tags: The user tags this backup policy applies
               to. Resources that have both a matching user tag and a matching type will
               be subject to the backup policy.
        :param str name: The name for this backup policy. The name is unique across
               all backup policies in the region.
        :param List[BackupPolicyPlanReference] plans: The plans for the backup
               policy.
        :param ResourceGroupReference resource_group: The resource group for this
               backup policy.
        :param str resource_type: The resource type.
        :param datetime last_job_completed_at: (optional) The date and time that
               the most recent job for this backup policy completed.
               If absent, no job has yet completed for this backup policy.
        """
        self.created_at = created_at
        self.crn = crn
        self.href = href
        self.id = id
        self.last_job_completed_at = last_job_completed_at
        self.lifecycle_state = lifecycle_state
        self.match_resource_types = match_resource_types
        self.match_user_tags = match_user_tags
        self.name = name
        self.plans = plans
        self.resource_group = resource_group
        self.resource_type = resource_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicy':
        """Initialize a BackupPolicy object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BackupPolicy JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError('Required property \'crn\' not present in BackupPolicy JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicy JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BackupPolicy JSON')
        if 'last_job_completed_at' in _dict:
            args['last_job_completed_at'] = string_to_datetime(_dict.get('last_job_completed_at'))
        if 'lifecycle_state' in _dict:
            args['lifecycle_state'] = _dict.get('lifecycle_state')
        else:
            raise ValueError('Required property \'lifecycle_state\' not present in BackupPolicy JSON')
        if 'match_resource_types' in _dict:
            args['match_resource_types'] = _dict.get('match_resource_types')
        else:
            raise ValueError('Required property \'match_resource_types\' not present in BackupPolicy JSON')
        if 'match_user_tags' in _dict:
            args['match_user_tags'] = _dict.get('match_user_tags')
        else:
            raise ValueError('Required property \'match_user_tags\' not present in BackupPolicy JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in BackupPolicy JSON')
        if 'plans' in _dict:
            args['plans'] = [BackupPolicyPlanReference.from_dict(v) for v in _dict.get('plans')]
        else:
            raise ValueError('Required property \'plans\' not present in BackupPolicy JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(_dict.get('resource_group'))
        else:
            raise ValueError('Required property \'resource_group\' not present in BackupPolicy JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BackupPolicy JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicy object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'last_job_completed_at') and self.last_job_completed_at is not None:
            _dict['last_job_completed_at'] = datetime_to_string(self.last_job_completed_at)
        if hasattr(self, 'lifecycle_state') and self.lifecycle_state is not None:
            _dict['lifecycle_state'] = self.lifecycle_state
        if hasattr(self, 'match_resource_types') and self.match_resource_types is not None:
            _dict['match_resource_types'] = self.match_resource_types
        if hasattr(self, 'match_user_tags') and self.match_user_tags is not None:
            _dict['match_user_tags'] = self.match_user_tags
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'plans') and self.plans is not None:
            plans_list = []
            for v in self.plans:
                if isinstance(v, dict):
                    plans_list.append(v)
                else:
                    plans_list.append(v.to_dict())
            _dict['plans'] = plans_list
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            if isinstance(self.resource_group, dict):
                _dict['resource_group'] = self.resource_group
            else:
                _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicy object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicy') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicy') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class LifecycleStateEnum(str, Enum):
        """
        The lifecycle state of the backup policy.
        """
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'
        STABLE = 'stable'
        SUSPENDED = 'suspended'
        UPDATING = 'updating'
        WAITING = 'waiting'


    class MatchResourceTypesEnum(str, Enum):
        """
        The resource type.
        """
        VOLUME = 'volume'


    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BACKUP_POLICY = 'backup_policy'


class BackupPolicyCollection():
    """
    BackupPolicyCollection.

    :attr List[BackupPolicy] backup_policies: Collection of backup policies.
    :attr BackupPolicyCollectionFirst first: A link to the first page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr BackupPolicyCollectionNext next: (optional) A link to the next page of
          resources. This property is present for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 backup_policies: List['BackupPolicy'],
                 first: 'BackupPolicyCollectionFirst',
                 limit: int,
                 total_count: int,
                 *,
                 next: 'BackupPolicyCollectionNext' = None) -> None:
        """
        Initialize a BackupPolicyCollection object.

        :param List[BackupPolicy] backup_policies: Collection of backup policies.
        :param BackupPolicyCollectionFirst first: A link to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param BackupPolicyCollectionNext next: (optional) A link to the next page
               of resources. This property is present for all pages
               except the last page.
        """
        self.backup_policies = backup_policies
        self.first = first
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyCollection':
        """Initialize a BackupPolicyCollection object from a json dictionary."""
        args = {}
        if 'backup_policies' in _dict:
            args['backup_policies'] = [BackupPolicy.from_dict(v) for v in _dict.get('backup_policies')]
        else:
            raise ValueError('Required property \'backup_policies\' not present in BackupPolicyCollection JSON')
        if 'first' in _dict:
            args['first'] = BackupPolicyCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError('Required property \'first\' not present in BackupPolicyCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError('Required property \'limit\' not present in BackupPolicyCollection JSON')
        if 'next' in _dict:
            args['next'] = BackupPolicyCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError('Required property \'total_count\' not present in BackupPolicyCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'backup_policies') and self.backup_policies is not None:
            backup_policies_list = []
            for v in self.backup_policies:
                if isinstance(v, dict):
                    backup_policies_list.append(v)
                else:
                    backup_policies_list.append(v.to_dict())
            _dict['backup_policies'] = backup_policies_list
        if hasattr(self, 'first') and self.first is not None:
            if isinstance(self.first, dict):
                _dict['first'] = self.first
            else:
                _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            if isinstance(self.next, dict):
                _dict['next'] = self.next
            else:
                _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyCollectionFirst():
    """
    A link to the first page of resources.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BackupPolicyCollectionFirst object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyCollectionFirst':
        """Initialize a BackupPolicyCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyCollectionFirst JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyCollectionNext():
    """
    A link to the next page of resources. This property is present for all pages except
    the last page.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BackupPolicyCollectionNext object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyCollectionNext':
        """Initialize a BackupPolicyCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyCollectionNext JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyJob():
    """
    BackupPolicyJob.

    :attr bool auto_delete: Indicates whether this backup policy job will be
          automatically deleted after it completes. At present, this is always `true`, but
          may be modifiable in the future.
    :attr int auto_delete_after: If `auto_delete` is `true`, the days after
          completion that this backup policy job will be deleted. This value may be
          modifiable in the future.
    :attr BackupPolicyPlanReference backup_policy_plan: The backup policy plan
          operated this backup policy job (may be
          [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
    :attr datetime completed_at: (optional) The date and time that the backup policy
          job was completed.
          If absent, the backup policy job has not yet completed.
    :attr datetime created_at: The date and time that the backup policy job was
          created.
    :attr str href: The URL for this backup policy job.
    :attr str id: The unique identifier for this backup policy job.
    :attr str job_type: The type of backup policy job.
          The enumerated values for this property will expand in the future. When
          processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the backup policy job on which the
          unexpected property value was encountered.
    :attr str resource_type: The resource type.
    :attr BackupPolicyJobSource source: The source this backup was created from (may
          be
          [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
    :attr str status: The status of the backup policy job.
          The enumerated values for this property will expand in the future. When
          processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the backup policy job on which the
          unexpected property value was encountered.
    :attr List[BackupPolicyJobStatusReason] status_reasons: The reasons for the
          current status (if any).
          The enumerated reason code values for this property will expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the resource on which the unexpected
          reason code was encountered.
    :attr List[SnapshotReference] target_snapshots: The snapshots operated on by
          this backup policy job (may be
          [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
    """

    def __init__(self,
                 auto_delete: bool,
                 auto_delete_after: int,
                 backup_policy_plan: 'BackupPolicyPlanReference',
                 created_at: datetime,
                 href: str,
                 id: str,
                 job_type: str,
                 resource_type: str,
                 source: 'BackupPolicyJobSource',
                 status: str,
                 status_reasons: List['BackupPolicyJobStatusReason'],
                 target_snapshots: List['SnapshotReference'],
                 *,
                 completed_at: datetime = None) -> None:
        """
        Initialize a BackupPolicyJob object.

        :param bool auto_delete: Indicates whether this backup policy job will be
               automatically deleted after it completes. At present, this is always
               `true`, but may be modifiable in the future.
        :param int auto_delete_after: If `auto_delete` is `true`, the days after
               completion that this backup policy job will be deleted. This value may be
               modifiable in the future.
        :param BackupPolicyPlanReference backup_policy_plan: The backup policy plan
               operated this backup policy job (may be
               [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
        :param datetime created_at: The date and time that the backup policy job
               was created.
        :param str href: The URL for this backup policy job.
        :param str id: The unique identifier for this backup policy job.
        :param str job_type: The type of backup policy job.
               The enumerated values for this property will expand in the future. When
               processing this property, check for and log unknown values. Optionally halt
               processing and surface the error, or bypass the backup policy job on which
               the unexpected property value was encountered.
        :param str resource_type: The resource type.
        :param BackupPolicyJobSource source: The source this backup was created
               from (may be
               [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
        :param str status: The status of the backup policy job.
               The enumerated values for this property will expand in the future. When
               processing this property, check for and log unknown values. Optionally halt
               processing and surface the error, or bypass the backup policy job on which
               the unexpected property value was encountered.
        :param List[BackupPolicyJobStatusReason] status_reasons: The reasons for
               the current status (if any).
               The enumerated reason code values for this property will expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected reason code was encountered.
        :param List[SnapshotReference] target_snapshots: The snapshots operated on
               by this backup policy job (may be
               [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).
        :param datetime completed_at: (optional) The date and time that the backup
               policy job was completed.
               If absent, the backup policy job has not yet completed.
        """
        self.auto_delete = auto_delete
        self.auto_delete_after = auto_delete_after
        self.backup_policy_plan = backup_policy_plan
        self.completed_at = completed_at
        self.created_at = created_at
        self.href = href
        self.id = id
        self.job_type = job_type
        self.resource_type = resource_type
        self.source = source
        self.status = status
        self.status_reasons = status_reasons
        self.target_snapshots = target_snapshots

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyJob':
        """Initialize a BackupPolicyJob object from a json dictionary."""
        args = {}
        if 'auto_delete' in _dict:
            args['auto_delete'] = _dict.get('auto_delete')
        else:
            raise ValueError('Required property \'auto_delete\' not present in BackupPolicyJob JSON')
        if 'auto_delete_after' in _dict:
            args['auto_delete_after'] = _dict.get('auto_delete_after')
        else:
            raise ValueError('Required property \'auto_delete_after\' not present in BackupPolicyJob JSON')
        if 'backup_policy_plan' in _dict:
            args['backup_policy_plan'] = BackupPolicyPlanReference.from_dict(_dict.get('backup_policy_plan'))
        else:
            raise ValueError('Required property \'backup_policy_plan\' not present in BackupPolicyJob JSON')
        if 'completed_at' in _dict:
            args['completed_at'] = string_to_datetime(_dict.get('completed_at'))
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BackupPolicyJob JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyJob JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BackupPolicyJob JSON')
        if 'job_type' in _dict:
            args['job_type'] = _dict.get('job_type')
        else:
            raise ValueError('Required property \'job_type\' not present in BackupPolicyJob JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BackupPolicyJob JSON')
        if 'source' in _dict:
            args['source'] = _dict.get('source')
        else:
            raise ValueError('Required property \'source\' not present in BackupPolicyJob JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError('Required property \'status\' not present in BackupPolicyJob JSON')
        if 'status_reasons' in _dict:
            args['status_reasons'] = [BackupPolicyJobStatusReason.from_dict(v) for v in _dict.get('status_reasons')]
        else:
            raise ValueError('Required property \'status_reasons\' not present in BackupPolicyJob JSON')
        if 'target_snapshots' in _dict:
            args['target_snapshots'] = [SnapshotReference.from_dict(v) for v in _dict.get('target_snapshots')]
        else:
            raise ValueError('Required property \'target_snapshots\' not present in BackupPolicyJob JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyJob object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'auto_delete') and self.auto_delete is not None:
            _dict['auto_delete'] = self.auto_delete
        if hasattr(self, 'auto_delete_after') and self.auto_delete_after is not None:
            _dict['auto_delete_after'] = self.auto_delete_after
        if hasattr(self, 'backup_policy_plan') and self.backup_policy_plan is not None:
            if isinstance(self.backup_policy_plan, dict):
                _dict['backup_policy_plan'] = self.backup_policy_plan
            else:
                _dict['backup_policy_plan'] = self.backup_policy_plan.to_dict()
        if hasattr(self, 'completed_at') and self.completed_at is not None:
            _dict['completed_at'] = datetime_to_string(self.completed_at)
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'job_type') and self.job_type is not None:
            _dict['job_type'] = self.job_type
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self, 'source') and self.source is not None:
            if isinstance(self.source, dict):
                _dict['source'] = self.source
            else:
                _dict['source'] = self.source.to_dict()
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'status_reasons') and self.status_reasons is not None:
            status_reasons_list = []
            for v in self.status_reasons:
                if isinstance(v, dict):
                    status_reasons_list.append(v)
                else:
                    status_reasons_list.append(v.to_dict())
            _dict['status_reasons'] = status_reasons_list
        if hasattr(self, 'target_snapshots') and self.target_snapshots is not None:
            target_snapshots_list = []
            for v in self.target_snapshots:
                if isinstance(v, dict):
                    target_snapshots_list.append(v)
                else:
                    target_snapshots_list.append(v.to_dict())
            _dict['target_snapshots'] = target_snapshots_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyJob object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyJob') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyJob') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class JobTypeEnum(str, Enum):
        """
        The type of backup policy job.
        The enumerated values for this property will expand in the future. When processing
        this property, check for and log unknown values. Optionally halt processing and
        surface the error, or bypass the backup policy job on which the unexpected
        property value was encountered.
        """
        CREATION = 'creation'
        DELETION = 'deletion'


    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BACKUP_POLICY_JOB = 'backup_policy_job'


    class StatusEnum(str, Enum):
        """
        The status of the backup policy job.
        The enumerated values for this property will expand in the future. When processing
        this property, check for and log unknown values. Optionally halt processing and
        surface the error, or bypass the backup policy job on which the unexpected
        property value was encountered.
        """
        FAILED = 'failed'
        RUNNING = 'running'
        SUCCEEDED = 'succeeded'


class BackupPolicyJobCollection():
    """
    BackupPolicyJobCollection.

    :attr BackupPolicyJobCollectionFirst first: A link to the first page of
          resources.
    :attr List[BackupPolicyJob] jobs: Collection of backup policy jobs.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr BackupPolicyJobCollectionNext next: (optional) A link to the next page of
          resources. This property is present for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'BackupPolicyJobCollectionFirst',
                 jobs: List['BackupPolicyJob'],
                 limit: int,
                 total_count: int,
                 *,
                 next: 'BackupPolicyJobCollectionNext' = None) -> None:
        """
        Initialize a BackupPolicyJobCollection object.

        :param BackupPolicyJobCollectionFirst first: A link to the first page of
               resources.
        :param List[BackupPolicyJob] jobs: Collection of backup policy jobs.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param BackupPolicyJobCollectionNext next: (optional) A link to the next
               page of resources. This property is present for all pages
               except the last page.
        """
        self.first = first
        self.jobs = jobs
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyJobCollection':
        """Initialize a BackupPolicyJobCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = BackupPolicyJobCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError('Required property \'first\' not present in BackupPolicyJobCollection JSON')
        if 'jobs' in _dict:
            args['jobs'] = [BackupPolicyJob.from_dict(v) for v in _dict.get('jobs')]
        else:
            raise ValueError('Required property \'jobs\' not present in BackupPolicyJobCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError('Required property \'limit\' not present in BackupPolicyJobCollection JSON')
        if 'next' in _dict:
            args['next'] = BackupPolicyJobCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError('Required property \'total_count\' not present in BackupPolicyJobCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyJobCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            if isinstance(self.first, dict):
                _dict['first'] = self.first
            else:
                _dict['first'] = self.first.to_dict()
        if hasattr(self, 'jobs') and self.jobs is not None:
            jobs_list = []
            for v in self.jobs:
                if isinstance(v, dict):
                    jobs_list.append(v)
                else:
                    jobs_list.append(v.to_dict())
            _dict['jobs'] = jobs_list
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            if isinstance(self.next, dict):
                _dict['next'] = self.next
            else:
                _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyJobCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyJobCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyJobCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyJobCollectionFirst():
    """
    A link to the first page of resources.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BackupPolicyJobCollectionFirst object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyJobCollectionFirst':
        """Initialize a BackupPolicyJobCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyJobCollectionFirst JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyJobCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyJobCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyJobCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyJobCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyJobCollectionNext():
    """
    A link to the next page of resources. This property is present for all pages except
    the last page.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BackupPolicyJobCollectionNext object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyJobCollectionNext':
        """Initialize a BackupPolicyJobCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyJobCollectionNext JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyJobCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyJobCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyJobCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyJobCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyJobSource():
    """
    The source this backup was created from (may be
    [deleted](https://cloud.ibm.com/apidocs/vpc#deleted-resources)).

    """

    def __init__(self) -> None:
        """
        Initialize a BackupPolicyJobSource object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
                  ", ".join(['BackupPolicyJobSourceVolumeReference']))
        raise Exception(msg)

class BackupPolicyJobStatusReason():
    """
    BackupPolicyJobStatusReason.

    :attr str code: A snake case string succinctly identifying the status reason:
          - `internal_error`: Internal error (contact IBM support)
          - `snapshot_pending`: Cannot delete backup (snapshot) in the `pending` lifecycle
          state
          - `snapshot_volume_limit`: The snapshot limit for the source volume has been
          reached
          - `source_volume_busy`: The source volume has `busy` set (after multiple
          retries).
    :attr str message: An explanation of the status reason.
    :attr str more_info: (optional) Link to documentation about this status reason.
    """

    def __init__(self,
                 code: str,
                 message: str,
                 *,
                 more_info: str = None) -> None:
        """
        Initialize a BackupPolicyJobStatusReason object.

        :param str code: A snake case string succinctly identifying the status
               reason:
               - `internal_error`: Internal error (contact IBM support)
               - `snapshot_pending`: Cannot delete backup (snapshot) in the `pending`
               lifecycle state
               - `snapshot_volume_limit`: The snapshot limit for the source volume has
               been reached
               - `source_volume_busy`: The source volume has `busy` set (after multiple
               retries).
        :param str message: An explanation of the status reason.
        :param str more_info: (optional) Link to documentation about this status
               reason.
        """
        self.code = code
        self.message = message
        self.more_info = more_info

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyJobStatusReason':
        """Initialize a BackupPolicyJobStatusReason object from a json dictionary."""
        args = {}
        if 'code' in _dict:
            args['code'] = _dict.get('code')
        else:
            raise ValueError('Required property \'code\' not present in BackupPolicyJobStatusReason JSON')
        if 'message' in _dict:
            args['message'] = _dict.get('message')
        else:
            raise ValueError('Required property \'message\' not present in BackupPolicyJobStatusReason JSON')
        if 'more_info' in _dict:
            args['more_info'] = _dict.get('more_info')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyJobStatusReason object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'code') and self.code is not None:
            _dict['code'] = self.code
        if hasattr(self, 'message') and self.message is not None:
            _dict['message'] = self.message
        if hasattr(self, 'more_info') and self.more_info is not None:
            _dict['more_info'] = self.more_info
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyJobStatusReason object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyJobStatusReason') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyJobStatusReason') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class CodeEnum(str, Enum):
        """
        A snake case string succinctly identifying the status reason:
        - `internal_error`: Internal error (contact IBM support)
        - `snapshot_pending`: Cannot delete backup (snapshot) in the `pending` lifecycle
        state
        - `snapshot_volume_limit`: The snapshot limit for the source volume has been
        reached
        - `source_volume_busy`: The source volume has `busy` set (after multiple retries).
        """
        INTERNAL_ERROR = 'internal_error'
        SNAPSHOT_PENDING = 'snapshot_pending'
        SNAPSHOT_VOLUME_LIMIT = 'snapshot_volume_limit'
        SOURCE_VOLUME_BUSY = 'source_volume_busy'


class BackupPolicyPatch():
    """
    BackupPolicyPatch.

    :attr List[str] match_user_tags: (optional) The user tags this backup policy
          applies to (replacing any existing tags). Resources that have both a matching
          user tag and a matching type will be subject to the backup policy.
    :attr str name: (optional) The name for this backup policy. The name must not be
          used by another backup policy in the region.
    """

    def __init__(self,
                 *,
                 match_user_tags: List[str] = None,
                 name: str = None) -> None:
        """
        Initialize a BackupPolicyPatch object.

        :param List[str] match_user_tags: (optional) The user tags this backup
               policy applies to (replacing any existing tags). Resources that have both a
               matching user tag and a matching type will be subject to the backup policy.
        :param str name: (optional) The name for this backup policy. The name must
               not be used by another backup policy in the region.
        """
        self.match_user_tags = match_user_tags
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPatch':
        """Initialize a BackupPolicyPatch object from a json dictionary."""
        args = {}
        if 'match_user_tags' in _dict:
            args['match_user_tags'] = _dict.get('match_user_tags')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'match_user_tags') and self.match_user_tags is not None:
            _dict['match_user_tags'] = self.match_user_tags
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlan():
    """
    BackupPolicyPlan.

    :attr bool active: Indicates whether the plan is active.
    :attr List[str] attach_user_tags: The user tags to attach to backups (snapshots)
          created by this plan.
    :attr BackupPolicyPlanClonePolicy clone_policy:
    :attr bool copy_user_tags: Indicates whether to copy the source's user tags to
          the created backups (snapshots).
    :attr datetime created_at: The date and time that the backup policy plan was
          created.
    :attr str cron_spec: The cron specification for the backup schedule. The backup
          policy jobs
          (which create and delete backups for this plan) will not start until this time,
          and may start for up to 90 minutes after this time.
          All backup schedules for plans in the same policy must be at least an hour
          apart.
    :attr BackupPolicyPlanDeletionTrigger deletion_trigger:
    :attr str href: The URL for this backup policy plan.
    :attr str id: The unique identifier for this backup policy plan.
    :attr str lifecycle_state: The lifecycle state of this backup policy plan.
    :attr str name: The name for this backup policy plan. The name is unique across
          all plans in the backup policy.
    :attr str resource_type: The resource type.
    """

    def __init__(self,
                 active: bool,
                 attach_user_tags: List[str],
                 clone_policy: 'BackupPolicyPlanClonePolicy',
                 copy_user_tags: bool,
                 created_at: datetime,
                 cron_spec: str,
                 deletion_trigger: 'BackupPolicyPlanDeletionTrigger',
                 href: str,
                 id: str,
                 lifecycle_state: str,
                 name: str,
                 resource_type: str) -> None:
        """
        Initialize a BackupPolicyPlan object.

        :param bool active: Indicates whether the plan is active.
        :param List[str] attach_user_tags: The user tags to attach to backups
               (snapshots) created by this plan.
        :param BackupPolicyPlanClonePolicy clone_policy:
        :param bool copy_user_tags: Indicates whether to copy the source's user
               tags to the created backups (snapshots).
        :param datetime created_at: The date and time that the backup policy plan
               was created.
        :param str cron_spec: The cron specification for the backup schedule. The
               backup policy jobs
               (which create and delete backups for this plan) will not start until this
               time, and may start for up to 90 minutes after this time.
               All backup schedules for plans in the same policy must be at least an hour
               apart.
        :param BackupPolicyPlanDeletionTrigger deletion_trigger:
        :param str href: The URL for this backup policy plan.
        :param str id: The unique identifier for this backup policy plan.
        :param str lifecycle_state: The lifecycle state of this backup policy plan.
        :param str name: The name for this backup policy plan. The name is unique
               across all plans in the backup policy.
        :param str resource_type: The resource type.
        """
        self.active = active
        self.attach_user_tags = attach_user_tags
        self.clone_policy = clone_policy
        self.copy_user_tags = copy_user_tags
        self.created_at = created_at
        self.cron_spec = cron_spec
        self.deletion_trigger = deletion_trigger
        self.href = href
        self.id = id
        self.lifecycle_state = lifecycle_state
        self.name = name
        self.resource_type = resource_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlan':
        """Initialize a BackupPolicyPlan object from a json dictionary."""
        args = {}
        if 'active' in _dict:
            args['active'] = _dict.get('active')
        else:
            raise ValueError('Required property \'active\' not present in BackupPolicyPlan JSON')
        if 'attach_user_tags' in _dict:
            args['attach_user_tags'] = _dict.get('attach_user_tags')
        else:
            raise ValueError('Required property \'attach_user_tags\' not present in BackupPolicyPlan JSON')
        if 'clone_policy' in _dict:
            args['clone_policy'] = BackupPolicyPlanClonePolicy.from_dict(_dict.get('clone_policy'))
        else:
            raise ValueError('Required property \'clone_policy\' not present in BackupPolicyPlan JSON')
        if 'copy_user_tags' in _dict:
            args['copy_user_tags'] = _dict.get('copy_user_tags')
        else:
            raise ValueError('Required property \'copy_user_tags\' not present in BackupPolicyPlan JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BackupPolicyPlan JSON')
        if 'cron_spec' in _dict:
            args['cron_spec'] = _dict.get('cron_spec')
        else:
            raise ValueError('Required property \'cron_spec\' not present in BackupPolicyPlan JSON')
        if 'deletion_trigger' in _dict:
            args['deletion_trigger'] = BackupPolicyPlanDeletionTrigger.from_dict(_dict.get('deletion_trigger'))
        else:
            raise ValueError('Required property \'deletion_trigger\' not present in BackupPolicyPlan JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyPlan JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BackupPolicyPlan JSON')
        if 'lifecycle_state' in _dict:
            args['lifecycle_state'] = _dict.get('lifecycle_state')
        else:
            raise ValueError('Required property \'lifecycle_state\' not present in BackupPolicyPlan JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in BackupPolicyPlan JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BackupPolicyPlan JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlan object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'active') and self.active is not None:
            _dict['active'] = self.active
        if hasattr(self, 'attach_user_tags') and self.attach_user_tags is not None:
            _dict['attach_user_tags'] = self.attach_user_tags
        if hasattr(self, 'clone_policy') and self.clone_policy is not None:
            if isinstance(self.clone_policy, dict):
                _dict['clone_policy'] = self.clone_policy
            else:
                _dict['clone_policy'] = self.clone_policy.to_dict()
        if hasattr(self, 'copy_user_tags') and self.copy_user_tags is not None:
            _dict['copy_user_tags'] = self.copy_user_tags
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'cron_spec') and self.cron_spec is not None:
            _dict['cron_spec'] = self.cron_spec
        if hasattr(self, 'deletion_trigger') and self.deletion_trigger is not None:
            if isinstance(self.deletion_trigger, dict):
                _dict['deletion_trigger'] = self.deletion_trigger
            else:
                _dict['deletion_trigger'] = self.deletion_trigger.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'lifecycle_state') and self.lifecycle_state is not None:
            _dict['lifecycle_state'] = self.lifecycle_state
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlan object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlan') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlan') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class LifecycleStateEnum(str, Enum):
        """
        The lifecycle state of this backup policy plan.
        """
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'
        STABLE = 'stable'
        SUSPENDED = 'suspended'
        UPDATING = 'updating'
        WAITING = 'waiting'


    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BACKUP_POLICY_PLAN = 'backup_policy_plan'


class BackupPolicyPlanClonePolicy():
    """
    BackupPolicyPlanClonePolicy.

    :attr int max_snapshots: The maximum number of recent snapshots (per source)
          that will keep clones.
    :attr List[ZoneReference] zones: The zone this backup policy plan will create
          snapshot clones in.
    """

    def __init__(self,
                 max_snapshots: int,
                 zones: List['ZoneReference']) -> None:
        """
        Initialize a BackupPolicyPlanClonePolicy object.

        :param int max_snapshots: The maximum number of recent snapshots (per
               source) that will keep clones.
        :param List[ZoneReference] zones: The zone this backup policy plan will
               create snapshot clones in.
        """
        self.max_snapshots = max_snapshots
        self.zones = zones

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanClonePolicy':
        """Initialize a BackupPolicyPlanClonePolicy object from a json dictionary."""
        args = {}
        if 'max_snapshots' in _dict:
            args['max_snapshots'] = _dict.get('max_snapshots')
        else:
            raise ValueError('Required property \'max_snapshots\' not present in BackupPolicyPlanClonePolicy JSON')
        if 'zones' in _dict:
            args['zones'] = [ZoneReference.from_dict(v) for v in _dict.get('zones')]
        else:
            raise ValueError('Required property \'zones\' not present in BackupPolicyPlanClonePolicy JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanClonePolicy object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'max_snapshots') and self.max_snapshots is not None:
            _dict['max_snapshots'] = self.max_snapshots
        if hasattr(self, 'zones') and self.zones is not None:
            zones_list = []
            for v in self.zones:
                if isinstance(v, dict):
                    zones_list.append(v)
                else:
                    zones_list.append(v.to_dict())
            _dict['zones'] = zones_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanClonePolicy object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanClonePolicy') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanClonePolicy') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanClonePolicyPatch():
    """
    BackupPolicyPlanClonePolicyPatch.

    :attr int max_snapshots: (optional) The maximum number of recent snapshots (per
          source) that will keep clones.
    :attr List[ZoneIdentity] zones: (optional) The zones this backup policy plan
          will create snapshot clones in. Updating this value does not change the clones
          for snapshots that have already been created by this plan.
    """

    def __init__(self,
                 *,
                 max_snapshots: int = None,
                 zones: List['ZoneIdentity'] = None) -> None:
        """
        Initialize a BackupPolicyPlanClonePolicyPatch object.

        :param int max_snapshots: (optional) The maximum number of recent snapshots
               (per source) that will keep clones.
        :param List[ZoneIdentity] zones: (optional) The zones this backup policy
               plan will create snapshot clones in. Updating this value does not change
               the clones for snapshots that have already been created by this plan.
        """
        self.max_snapshots = max_snapshots
        self.zones = zones

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanClonePolicyPatch':
        """Initialize a BackupPolicyPlanClonePolicyPatch object from a json dictionary."""
        args = {}
        if 'max_snapshots' in _dict:
            args['max_snapshots'] = _dict.get('max_snapshots')
        if 'zones' in _dict:
            args['zones'] = _dict.get('zones')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanClonePolicyPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'max_snapshots') and self.max_snapshots is not None:
            _dict['max_snapshots'] = self.max_snapshots
        if hasattr(self, 'zones') and self.zones is not None:
            zones_list = []
            for v in self.zones:
                if isinstance(v, dict):
                    zones_list.append(v)
                else:
                    zones_list.append(v.to_dict())
            _dict['zones'] = zones_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanClonePolicyPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanClonePolicyPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanClonePolicyPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanClonePolicyPrototype():
    """
    BackupPolicyPlanClonePolicyPrototype.

    :attr int max_snapshots: (optional) The maximum number of recent snapshots (per
          source) that will keep clones.
    :attr List[ZoneIdentity] zones: The zone this backup policy plan will create
          snapshot clones in.
    """

    def __init__(self,
                 zones: List['ZoneIdentity'],
                 *,
                 max_snapshots: int = None) -> None:
        """
        Initialize a BackupPolicyPlanClonePolicyPrototype object.

        :param List[ZoneIdentity] zones: The zone this backup policy plan will
               create snapshot clones in.
        :param int max_snapshots: (optional) The maximum number of recent snapshots
               (per source) that will keep clones.
        """
        self.max_snapshots = max_snapshots
        self.zones = zones

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanClonePolicyPrototype':
        """Initialize a BackupPolicyPlanClonePolicyPrototype object from a json dictionary."""
        args = {}
        if 'max_snapshots' in _dict:
            args['max_snapshots'] = _dict.get('max_snapshots')
        if 'zones' in _dict:
            args['zones'] = _dict.get('zones')
        else:
            raise ValueError('Required property \'zones\' not present in BackupPolicyPlanClonePolicyPrototype JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanClonePolicyPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'max_snapshots') and self.max_snapshots is not None:
            _dict['max_snapshots'] = self.max_snapshots
        if hasattr(self, 'zones') and self.zones is not None:
            zones_list = []
            for v in self.zones:
                if isinstance(v, dict):
                    zones_list.append(v)
                else:
                    zones_list.append(v.to_dict())
            _dict['zones'] = zones_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanClonePolicyPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanClonePolicyPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanClonePolicyPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanCollection():
    """
    BackupPolicyPlanCollection.

    :attr List[BackupPolicyPlan] plans: Collection of backup policy plans.
    """

    def __init__(self,
                 plans: List['BackupPolicyPlan']) -> None:
        """
        Initialize a BackupPolicyPlanCollection object.

        :param List[BackupPolicyPlan] plans: Collection of backup policy plans.
        """
        self.plans = plans

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanCollection':
        """Initialize a BackupPolicyPlanCollection object from a json dictionary."""
        args = {}
        if 'plans' in _dict:
            args['plans'] = [BackupPolicyPlan.from_dict(v) for v in _dict.get('plans')]
        else:
            raise ValueError('Required property \'plans\' not present in BackupPolicyPlanCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'plans') and self.plans is not None:
            plans_list = []
            for v in self.plans:
                if isinstance(v, dict):
                    plans_list.append(v)
                else:
                    plans_list.append(v.to_dict())
            _dict['plans'] = plans_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanDeletionTrigger():
    """
    BackupPolicyPlanDeletionTrigger.

    :attr int delete_after: The maximum number of days to keep each backup after
          creation.
    :attr int delete_over_count: (optional) The maximum number of recent backups to
          keep. If absent, there is no maximum.
    """

    def __init__(self,
                 delete_after: int,
                 *,
                 delete_over_count: int = None) -> None:
        """
        Initialize a BackupPolicyPlanDeletionTrigger object.

        :param int delete_after: The maximum number of days to keep each backup
               after creation.
        :param int delete_over_count: (optional) The maximum number of recent
               backups to keep. If absent, there is no maximum.
        """
        self.delete_after = delete_after
        self.delete_over_count = delete_over_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanDeletionTrigger':
        """Initialize a BackupPolicyPlanDeletionTrigger object from a json dictionary."""
        args = {}
        if 'delete_after' in _dict:
            args['delete_after'] = _dict.get('delete_after')
        else:
            raise ValueError('Required property \'delete_after\' not present in BackupPolicyPlanDeletionTrigger JSON')
        if 'delete_over_count' in _dict:
            args['delete_over_count'] = _dict.get('delete_over_count')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanDeletionTrigger object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delete_after') and self.delete_after is not None:
            _dict['delete_after'] = self.delete_after
        if hasattr(self, 'delete_over_count') and self.delete_over_count is not None:
            _dict['delete_over_count'] = self.delete_over_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanDeletionTrigger object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanDeletionTrigger') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanDeletionTrigger') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanDeletionTriggerPatch():
    """
    BackupPolicyPlanDeletionTriggerPatch.

    :attr int delete_after: (optional) The maximum number of days to keep each
          backup after creation.
    :attr int delete_over_count: (optional) The maximum number of recent backups to
          keep. Specify `null` to remove any existing maximum.
    """

    def __init__(self,
                 *,
                 delete_after: int = None,
                 delete_over_count: int = None) -> None:
        """
        Initialize a BackupPolicyPlanDeletionTriggerPatch object.

        :param int delete_after: (optional) The maximum number of days to keep each
               backup after creation.
        :param int delete_over_count: (optional) The maximum number of recent
               backups to keep. Specify `null` to remove any existing maximum.
        """
        self.delete_after = delete_after
        self.delete_over_count = delete_over_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanDeletionTriggerPatch':
        """Initialize a BackupPolicyPlanDeletionTriggerPatch object from a json dictionary."""
        args = {}
        if 'delete_after' in _dict:
            args['delete_after'] = _dict.get('delete_after')
        if 'delete_over_count' in _dict:
            args['delete_over_count'] = _dict.get('delete_over_count')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanDeletionTriggerPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delete_after') and self.delete_after is not None:
            _dict['delete_after'] = self.delete_after
        if hasattr(self, 'delete_over_count') and self.delete_over_count is not None:
            _dict['delete_over_count'] = self.delete_over_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanDeletionTriggerPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanDeletionTriggerPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanDeletionTriggerPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanDeletionTriggerPrototype():
    """
    BackupPolicyPlanDeletionTriggerPrototype.

    :attr int delete_after: (optional) The maximum number of days to keep each
          backup after creation.
    :attr int delete_over_count: (optional) The maximum number of recent backups to
          keep. If unspecified, there will be no maximum.
    """

    def __init__(self,
                 *,
                 delete_after: int = None,
                 delete_over_count: int = None) -> None:
        """
        Initialize a BackupPolicyPlanDeletionTriggerPrototype object.

        :param int delete_after: (optional) The maximum number of days to keep each
               backup after creation.
        :param int delete_over_count: (optional) The maximum number of recent
               backups to keep. If unspecified, there will be no maximum.
        """
        self.delete_after = delete_after
        self.delete_over_count = delete_over_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanDeletionTriggerPrototype':
        """Initialize a BackupPolicyPlanDeletionTriggerPrototype object from a json dictionary."""
        args = {}
        if 'delete_after' in _dict:
            args['delete_after'] = _dict.get('delete_after')
        if 'delete_over_count' in _dict:
            args['delete_over_count'] = _dict.get('delete_over_count')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanDeletionTriggerPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'delete_after') and self.delete_after is not None:
            _dict['delete_after'] = self.delete_after
        if hasattr(self, 'delete_over_count') and self.delete_over_count is not None:
            _dict['delete_over_count'] = self.delete_over_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanDeletionTriggerPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanDeletionTriggerPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanDeletionTriggerPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanPatch():
    """
    BackupPolicyPlanPatch.

    :attr bool active: (optional) Indicates whether the plan is active.
    :attr List[str] attach_user_tags: (optional) The user tags to attach to backups
          (snapshots) created by this plan. Updating this value does not change the user
          tags for backups that have already been created by this plan.
    :attr BackupPolicyPlanClonePolicyPatch clone_policy: (optional)
    :attr bool copy_user_tags: (optional) Indicates whether to copy the source's
          user tags to the created backups (snapshots).
    :attr str cron_spec: (optional) The cron specification for the backup schedule.
          The backup policy jobs
          (which create and delete backups for this plan) will not start until this time,
          and may start for up to 90 minutes after this time.
          All backup schedules for plans in the same policy must be at least an hour
          apart.
    :attr BackupPolicyPlanDeletionTriggerPatch deletion_trigger: (optional)
    :attr str name: (optional) The name for this backup policy plan. The name must
          not be used by another plan for the backup policy.
    """

    def __init__(self,
                 *,
                 active: bool = None,
                 attach_user_tags: List[str] = None,
                 clone_policy: 'BackupPolicyPlanClonePolicyPatch' = None,
                 copy_user_tags: bool = None,
                 cron_spec: str = None,
                 deletion_trigger: 'BackupPolicyPlanDeletionTriggerPatch' = None,
                 name: str = None) -> None:
        """
        Initialize a BackupPolicyPlanPatch object.

        :param bool active: (optional) Indicates whether the plan is active.
        :param List[str] attach_user_tags: (optional) The user tags to attach to
               backups (snapshots) created by this plan. Updating this value does not
               change the user tags for backups that have already been created by this
               plan.
        :param BackupPolicyPlanClonePolicyPatch clone_policy: (optional)
        :param bool copy_user_tags: (optional) Indicates whether to copy the
               source's user tags to the created backups (snapshots).
        :param str cron_spec: (optional) The cron specification for the backup
               schedule. The backup policy jobs
               (which create and delete backups for this plan) will not start until this
               time, and may start for up to 90 minutes after this time.
               All backup schedules for plans in the same policy must be at least an hour
               apart.
        :param BackupPolicyPlanDeletionTriggerPatch deletion_trigger: (optional)
        :param str name: (optional) The name for this backup policy plan. The name
               must not be used by another plan for the backup policy.
        """
        self.active = active
        self.attach_user_tags = attach_user_tags
        self.clone_policy = clone_policy
        self.copy_user_tags = copy_user_tags
        self.cron_spec = cron_spec
        self.deletion_trigger = deletion_trigger
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanPatch':
        """Initialize a BackupPolicyPlanPatch object from a json dictionary."""
        args = {}
        if 'active' in _dict:
            args['active'] = _dict.get('active')
        if 'attach_user_tags' in _dict:
            args['attach_user_tags'] = _dict.get('attach_user_tags')
        if 'clone_policy' in _dict:
            args['clone_policy'] = BackupPolicyPlanClonePolicyPatch.from_dict(_dict.get('clone_policy'))
        if 'copy_user_tags' in _dict:
            args['copy_user_tags'] = _dict.get('copy_user_tags')
        if 'cron_spec' in _dict:
            args['cron_spec'] = _dict.get('cron_spec')
        if 'deletion_trigger' in _dict:
            args['deletion_trigger'] = BackupPolicyPlanDeletionTriggerPatch.from_dict(_dict.get('deletion_trigger'))
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'active') and self.active is not None:
            _dict['active'] = self.active
        if hasattr(self, 'attach_user_tags') and self.attach_user_tags is not None:
            _dict['attach_user_tags'] = self.attach_user_tags
        if hasattr(self, 'clone_policy') and self.clone_policy is not None:
            if isinstance(self.clone_policy, dict):
                _dict['clone_policy'] = self.clone_policy
            else:
                _dict['clone_policy'] = self.clone_policy.to_dict()
        if hasattr(self, 'copy_user_tags') and self.copy_user_tags is not None:
            _dict['copy_user_tags'] = self.copy_user_tags
        if hasattr(self, 'cron_spec') and self.cron_spec is not None:
            _dict['cron_spec'] = self.cron_spec
        if hasattr(self, 'deletion_trigger') and self.deletion_trigger is not None:
            if isinstance(self.deletion_trigger, dict):
                _dict['deletion_trigger'] = self.deletion_trigger
            else:
                _dict['deletion_trigger'] = self.deletion_trigger.to_dict()
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanPrototype():
    """
    BackupPolicyPlanPrototype.

    :attr bool active: (optional) Indicates whether the plan is active.
    :attr List[str] attach_user_tags: (optional) User tags to attach to each backup
          (snapshot) created by this plan. If unspecified, no user tags will be attached.
    :attr BackupPolicyPlanClonePolicyPrototype clone_policy: (optional)
    :attr bool copy_user_tags: (optional) Indicates whether to copy the source's
          user tags to the created backups (snapshots).
    :attr str cron_spec: The cron specification for the backup schedule. The backup
          policy jobs
          (which create and delete backups for this plan) will not start until this time,
          and may start for up to 90 minutes after this time.
          All backup schedules for plans in the same policy must be at least an hour
          apart.
    :attr BackupPolicyPlanDeletionTriggerPrototype deletion_trigger: (optional)
    :attr str name: (optional) The name for this backup policy plan. The name must
          not be used by another plan for the backup policy. If unspecified, the name will
          be a hyphenated list of randomly-selected words.
    """

    def __init__(self,
                 cron_spec: str,
                 *,
                 active: bool = None,
                 attach_user_tags: List[str] = None,
                 clone_policy: 'BackupPolicyPlanClonePolicyPrototype' = None,
                 copy_user_tags: bool = None,
                 deletion_trigger: 'BackupPolicyPlanDeletionTriggerPrototype' = None,
                 name: str = None) -> None:
        """
        Initialize a BackupPolicyPlanPrototype object.

        :param str cron_spec: The cron specification for the backup schedule. The
               backup policy jobs
               (which create and delete backups for this plan) will not start until this
               time, and may start for up to 90 minutes after this time.
               All backup schedules for plans in the same policy must be at least an hour
               apart.
        :param bool active: (optional) Indicates whether the plan is active.
        :param List[str] attach_user_tags: (optional) User tags to attach to each
               backup (snapshot) created by this plan. If unspecified, no user tags will
               be attached.
        :param BackupPolicyPlanClonePolicyPrototype clone_policy: (optional)
        :param bool copy_user_tags: (optional) Indicates whether to copy the
               source's user tags to the created backups (snapshots).
        :param BackupPolicyPlanDeletionTriggerPrototype deletion_trigger:
               (optional)
        :param str name: (optional) The name for this backup policy plan. The name
               must not be used by another plan for the backup policy. If unspecified, the
               name will be a hyphenated list of randomly-selected words.
        """
        self.active = active
        self.attach_user_tags = attach_user_tags
        self.clone_policy = clone_policy
        self.copy_user_tags = copy_user_tags
        self.cron_spec = cron_spec
        self.deletion_trigger = deletion_trigger
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanPrototype':
        """Initialize a BackupPolicyPlanPrototype object from a json dictionary."""
        args = {}
        if 'active' in _dict:
            args['active'] = _dict.get('active')
        if 'attach_user_tags' in _dict:
            args['attach_user_tags'] = _dict.get('attach_user_tags')
        if 'clone_policy' in _dict:
            args['clone_policy'] = BackupPolicyPlanClonePolicyPrototype.from_dict(_dict.get('clone_policy'))
        if 'copy_user_tags' in _dict:
            args['copy_user_tags'] = _dict.get('copy_user_tags')
        if 'cron_spec' in _dict:
            args['cron_spec'] = _dict.get('cron_spec')
        else:
            raise ValueError('Required property \'cron_spec\' not present in BackupPolicyPlanPrototype JSON')
        if 'deletion_trigger' in _dict:
            args['deletion_trigger'] = BackupPolicyPlanDeletionTriggerPrototype.from_dict(_dict.get('deletion_trigger'))
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'active') and self.active is not None:
            _dict['active'] = self.active
        if hasattr(self, 'attach_user_tags') and self.attach_user_tags is not None:
            _dict['attach_user_tags'] = self.attach_user_tags
        if hasattr(self, 'clone_policy') and self.clone_policy is not None:
            if isinstance(self.clone_policy, dict):
                _dict['clone_policy'] = self.clone_policy
            else:
                _dict['clone_policy'] = self.clone_policy.to_dict()
        if hasattr(self, 'copy_user_tags') and self.copy_user_tags is not None:
            _dict['copy_user_tags'] = self.copy_user_tags
        if hasattr(self, 'cron_spec') and self.cron_spec is not None:
            _dict['cron_spec'] = self.cron_spec
        if hasattr(self, 'deletion_trigger') and self.deletion_trigger is not None:
            if isinstance(self.deletion_trigger, dict):
                _dict['deletion_trigger'] = self.deletion_trigger
            else:
                _dict['deletion_trigger'] = self.deletion_trigger.to_dict()
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BackupPolicyPlanReference():
    """
    BackupPolicyPlanReference.

    :attr BackupPolicyPlanReferenceDeleted deleted: (optional) If present, this
          property indicates the referenced resource has been deleted, and provides
          some supplementary information.
    :attr str href: The URL for this backup policy plan.
    :attr str id: The unique identifier for this backup policy plan.
    :attr str name: The name for this backup policy plan. The name is unique across
          all plans in the backup policy.
    :attr str resource_type: The resource type.
    """

    def __init__(self,
                 href: str,
                 id: str,
                 name: str,
                 resource_type: str,
                 *,
                 deleted: 'BackupPolicyPlanReferenceDeleted' = None) -> None:
        """
        Initialize a BackupPolicyPlanReference object.

        :param str href: The URL for this backup policy plan.
        :param str id: The unique identifier for this backup policy plan.
        :param str name: The name for this backup policy plan. The name is unique
               across all plans in the backup policy.
        :param str resource_type: The resource type.
        :param BackupPolicyPlanReferenceDeleted deleted: (optional) If present,
               this property indicates the referenced resource has been deleted, and
               provides
               some supplementary information.
        """
        self.deleted = deleted
        self.href = href
        self.id = id
        self.name = name
        self.resource_type = resource_type

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanReference':
        """Initialize a BackupPolicyPlanReference object from a json dictionary."""
        args = {}
        if 'deleted' in _dict:
            args['deleted'] = BackupPolicyPlanReferenceDeleted.from_dict(_dict.get('deleted'))
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BackupPolicyPlanReference JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BackupPolicyPlanReference JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in BackupPolicyPlanReference JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BackupPolicyPlanReference JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanReference object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'deleted') and self.deleted is not None:
            if isinstance(self.deleted, dict):
                _dict['deleted'] = self.deleted
            else:
                _dict['deleted'] = self.deleted.to_dict()
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanReference object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanReference') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanReference') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BACKUP_POLICY_PLAN = 'backup_policy_plan'


class BackupPolicyPlanReferenceDeleted():
    """
    If present, this property indicates the referenced resource has been deleted, and
    provides some supplementary information.

    :attr str more_info: Link to documentation about deleted resources.
    """

    def __init__(self,
                 more_info: str) -> None:
        """
        Initialize a BackupPolicyPlanReferenceDeleted object.

        :param str more_info: Link to documentation about deleted resources.
        """
        self.more_info = more_info

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BackupPolicyPlanReferenceDeleted':
        """Initialize a BackupPolicyPlanReferenceDeleted object from a json dictionary."""
        args = {}
        if 'more_info' in _dict:
            args['more_info'] = _dict.get('more_info')
        else:
            raise ValueError('Required property \'more_info\' not present in BackupPolicyPlanReferenceDeleted JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BackupPolicyPlanReferenceDeleted object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'more_info') and self.more_info is not None:
            _dict['more_info'] = self.more_info
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BackupPolicyPlanReferenceDeleted object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BackupPolicyPlanReferenceDeleted') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BackupPolicyPlanReferenceDeleted') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServer():
    """
    BareMetalServer.

    :attr int bandwidth: The total bandwidth (in megabits per second) shared across
          the bare metal server's network interfaces.
    :attr BareMetalServerBootTarget boot_target: The possible resource types for
          this property are expected to expand in the future.
    :attr BareMetalServerCPU cpu: The bare metal server CPU configuration.
    :attr datetime created_at: The date and time that the bare metal server was
          created.
    :attr str crn: The CRN for this bare metal server.
    :attr List[BareMetalServerDisk] disks: The disks for this bare metal server,
          including any disks that are associated with the
          `boot_target`.
    :attr bool enable_secure_boot: Indicates whether secure boot is enabled. If
          enabled, the image must support secure boot or the server will fail to boot.
    :attr str href: The URL for this bare metal server.
    :attr str id: The unique identifier for this bare metal server.
    :attr int memory: The amount of memory, truncated to whole gibibytes.
    :attr str name: The name for this bare metal server. The name is unique across
          all bare metal servers in the region.
    :attr List[NetworkInterfaceBareMetalServerContextReference] network_interfaces:
          The network interfaces for this bare metal server, including the primary network
          interface.
    :attr NetworkInterfaceBareMetalServerContextReference primary_network_interface:
          Primary network interface.
    :attr BareMetalServerProfileReference profile: The
          [profile](https://cloud.ibm.com/docs/vpc?topic=vpc-bare-metal-servers-profile)
          for this bare metal server.
    :attr ResourceGroupReference resource_group: The resource group for this bare
          metal server.
    :attr str resource_type: The resource type.
    :attr str status: The status of the bare metal server.
    :attr List[BareMetalServerStatusReason] status_reasons: The reasons for the
          current status (if any).
          The enumerated reason code values for this property will expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the resource on which the unexpected
          reason code was encountered.
    :attr BareMetalServerTrustedPlatformModule trusted_platform_module:
    :attr VPCReference vpc: The VPC this bare metal server resides in.
    :attr ZoneReference zone: The zone this bare metal server resides in.
    """

    def __init__(self,
                 bandwidth: int,
                 boot_target: 'BareMetalServerBootTarget',
                 cpu: 'BareMetalServerCPU',
                 created_at: datetime,
                 crn: str,
                 disks: List['BareMetalServerDisk'],
                 enable_secure_boot: bool,
                 href: str,
                 id: str,
                 memory: int,
                 name: str,
                 network_interfaces: List['NetworkInterfaceBareMetalServerContextReference'],
                 primary_network_interface: 'NetworkInterfaceBareMetalServerContextReference',
                 profile: 'BareMetalServerProfileReference',
                 resource_group: 'ResourceGroupReference',
                 resource_type: str,
                 status: str,
                 status_reasons: List['BareMetalServerStatusReason'],
                 trusted_platform_module: 'BareMetalServerTrustedPlatformModule',
                 vpc: 'VPCReference',
                 zone: 'ZoneReference') -> None:
        """
        Initialize a BareMetalServer object.

        :param int bandwidth: The total bandwidth (in megabits per second) shared
               across the bare metal server's network interfaces.
        :param BareMetalServerBootTarget boot_target: The possible resource types
               for this property are expected to expand in the future.
        :param BareMetalServerCPU cpu: The bare metal server CPU configuration.
        :param datetime created_at: The date and time that the bare metal server
               was created.
        :param str crn: The CRN for this bare metal server.
        :param List[BareMetalServerDisk] disks: The disks for this bare metal
               server, including any disks that are associated with the
               `boot_target`.
        :param bool enable_secure_boot: Indicates whether secure boot is enabled.
               If enabled, the image must support secure boot or the server will fail to
               boot.
        :param str href: The URL for this bare metal server.
        :param str id: The unique identifier for this bare metal server.
        :param int memory: The amount of memory, truncated to whole gibibytes.
        :param str name: The name for this bare metal server. The name is unique
               across all bare metal servers in the region.
        :param List[NetworkInterfaceBareMetalServerContextReference]
               network_interfaces: The network interfaces for this bare metal server,
               including the primary network interface.
        :param NetworkInterfaceBareMetalServerContextReference
               primary_network_interface: Primary network interface.
        :param BareMetalServerProfileReference profile: The
               [profile](https://cloud.ibm.com/docs/vpc?topic=vpc-bare-metal-servers-profile)
               for this bare metal server.
        :param ResourceGroupReference resource_group: The resource group for this
               bare metal server.
        :param str resource_type: The resource type.
        :param str status: The status of the bare metal server.
        :param List[BareMetalServerStatusReason] status_reasons: The reasons for
               the current status (if any).
               The enumerated reason code values for this property will expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected reason code was encountered.
        :param BareMetalServerTrustedPlatformModule trusted_platform_module:
        :param VPCReference vpc: The VPC this bare metal server resides in.
        :param ZoneReference zone: The zone this bare metal server resides in.
        """
        self.bandwidth = bandwidth
        self.boot_target = boot_target
        self.cpu = cpu
        self.created_at = created_at
        self.crn = crn
        self.disks = disks
        self.enable_secure_boot = enable_secure_boot
        self.href = href
        self.id = id
        self.memory = memory
        self.name = name
        self.network_interfaces = network_interfaces
        self.primary_network_interface = primary_network_interface
        self.profile = profile
        self.resource_group = resource_group
        self.resource_type = resource_type
        self.status = status
        self.status_reasons = status_reasons
        self.trusted_platform_module = trusted_platform_module
        self.vpc = vpc
        self.zone = zone

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServer':
        """Initialize a BareMetalServer object from a json dictionary."""
        args = {}
        if 'bandwidth' in _dict:
            args['bandwidth'] = _dict.get('bandwidth')
        else:
            raise ValueError('Required property \'bandwidth\' not present in BareMetalServer JSON')
        if 'boot_target' in _dict:
            args['boot_target'] = _dict.get('boot_target')
        else:
            raise ValueError('Required property \'boot_target\' not present in BareMetalServer JSON')
        if 'cpu' in _dict:
            args['cpu'] = BareMetalServerCPU.from_dict(_dict.get('cpu'))
        else:
            raise ValueError('Required property \'cpu\' not present in BareMetalServer JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BareMetalServer JSON')
        if 'crn' in _dict:
            args['crn'] = _dict.get('crn')
        else:
            raise ValueError('Required property \'crn\' not present in BareMetalServer JSON')
        if 'disks' in _dict:
            args['disks'] = [BareMetalServerDisk.from_dict(v) for v in _dict.get('disks')]
        else:
            raise ValueError('Required property \'disks\' not present in BareMetalServer JSON')
        if 'enable_secure_boot' in _dict:
            args['enable_secure_boot'] = _dict.get('enable_secure_boot')
        else:
            raise ValueError('Required property \'enable_secure_boot\' not present in BareMetalServer JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServer JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BareMetalServer JSON')
        if 'memory' in _dict:
            args['memory'] = _dict.get('memory')
        else:
            raise ValueError('Required property \'memory\' not present in BareMetalServer JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in BareMetalServer JSON')
        if 'network_interfaces' in _dict:
            args['network_interfaces'] = [NetworkInterfaceBareMetalServerContextReference.from_dict(v) for v in _dict.get('network_interfaces')]
        else:
            raise ValueError('Required property \'network_interfaces\' not present in BareMetalServer JSON')
        if 'primary_network_interface' in _dict:
            args['primary_network_interface'] = NetworkInterfaceBareMetalServerContextReference.from_dict(_dict.get('primary_network_interface'))
        else:
            raise ValueError('Required property \'primary_network_interface\' not present in BareMetalServer JSON')
        if 'profile' in _dict:
            args['profile'] = BareMetalServerProfileReference.from_dict(_dict.get('profile'))
        else:
            raise ValueError('Required property \'profile\' not present in BareMetalServer JSON')
        if 'resource_group' in _dict:
            args['resource_group'] = ResourceGroupReference.from_dict(_dict.get('resource_group'))
        else:
            raise ValueError('Required property \'resource_group\' not present in BareMetalServer JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BareMetalServer JSON')
        if 'status' in _dict:
            args['status'] = _dict.get('status')
        else:
            raise ValueError('Required property \'status\' not present in BareMetalServer JSON')
        if 'status_reasons' in _dict:
            args['status_reasons'] = [BareMetalServerStatusReason.from_dict(v) for v in _dict.get('status_reasons')]
        else:
            raise ValueError('Required property \'status_reasons\' not present in BareMetalServer JSON')
        if 'trusted_platform_module' in _dict:
            args['trusted_platform_module'] = BareMetalServerTrustedPlatformModule.from_dict(_dict.get('trusted_platform_module'))
        else:
            raise ValueError('Required property \'trusted_platform_module\' not present in BareMetalServer JSON')
        if 'vpc' in _dict:
            args['vpc'] = VPCReference.from_dict(_dict.get('vpc'))
        else:
            raise ValueError('Required property \'vpc\' not present in BareMetalServer JSON')
        if 'zone' in _dict:
            args['zone'] = ZoneReference.from_dict(_dict.get('zone'))
        else:
            raise ValueError('Required property \'zone\' not present in BareMetalServer JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServer object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'bandwidth') and self.bandwidth is not None:
            _dict['bandwidth'] = self.bandwidth
        if hasattr(self, 'boot_target') and self.boot_target is not None:
            if isinstance(self.boot_target, dict):
                _dict['boot_target'] = self.boot_target
            else:
                _dict['boot_target'] = self.boot_target.to_dict()
        if hasattr(self, 'cpu') and self.cpu is not None:
            if isinstance(self.cpu, dict):
                _dict['cpu'] = self.cpu
            else:
                _dict['cpu'] = self.cpu.to_dict()
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'crn') and self.crn is not None:
            _dict['crn'] = self.crn
        if hasattr(self, 'disks') and self.disks is not None:
            disks_list = []
            for v in self.disks:
                if isinstance(v, dict):
                    disks_list.append(v)
                else:
                    disks_list.append(v.to_dict())
            _dict['disks'] = disks_list
        if hasattr(self, 'enable_secure_boot') and self.enable_secure_boot is not None:
            _dict['enable_secure_boot'] = self.enable_secure_boot
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'memory') and self.memory is not None:
            _dict['memory'] = self.memory
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'network_interfaces') and self.network_interfaces is not None:
            network_interfaces_list = []
            for v in self.network_interfaces:
                if isinstance(v, dict):
                    network_interfaces_list.append(v)
                else:
                    network_interfaces_list.append(v.to_dict())
            _dict['network_interfaces'] = network_interfaces_list
        if hasattr(self, 'primary_network_interface') and self.primary_network_interface is not None:
            if isinstance(self.primary_network_interface, dict):
                _dict['primary_network_interface'] = self.primary_network_interface
            else:
                _dict['primary_network_interface'] = self.primary_network_interface.to_dict()
        if hasattr(self, 'profile') and self.profile is not None:
            if isinstance(self.profile, dict):
                _dict['profile'] = self.profile
            else:
                _dict['profile'] = self.profile.to_dict()
        if hasattr(self, 'resource_group') and self.resource_group is not None:
            if isinstance(self.resource_group, dict):
                _dict['resource_group'] = self.resource_group
            else:
                _dict['resource_group'] = self.resource_group.to_dict()
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self, 'status') and self.status is not None:
            _dict['status'] = self.status
        if hasattr(self, 'status_reasons') and self.status_reasons is not None:
            status_reasons_list = []
            for v in self.status_reasons:
                if isinstance(v, dict):
                    status_reasons_list.append(v)
                else:
                    status_reasons_list.append(v.to_dict())
            _dict['status_reasons'] = status_reasons_list
        if hasattr(self, 'trusted_platform_module') and self.trusted_platform_module is not None:
            if isinstance(self.trusted_platform_module, dict):
                _dict['trusted_platform_module'] = self.trusted_platform_module
            else:
                _dict['trusted_platform_module'] = self.trusted_platform_module.to_dict()
        if hasattr(self, 'vpc') and self.vpc is not None:
            if isinstance(self.vpc, dict):
                _dict['vpc'] = self.vpc
            else:
                _dict['vpc'] = self.vpc.to_dict()
        if hasattr(self, 'zone') and self.zone is not None:
            if isinstance(self.zone, dict):
                _dict['zone'] = self.zone
            else:
                _dict['zone'] = self.zone.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServer object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServer') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServer') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BARE_METAL_SERVER = 'bare_metal_server'


    class StatusEnum(str, Enum):
        """
        The status of the bare metal server.
        """
        DELETING = 'deleting'
        FAILED = 'failed'
        MAINTENANCE = 'maintenance'
        PENDING = 'pending'
        RESTARTING = 'restarting'
        RUNNING = 'running'
        STARTING = 'starting'
        STOPPED = 'stopped'
        STOPPING = 'stopping'


class BareMetalServerBootTarget():
    """
    The possible resource types for this property are expected to expand in the future.

    """

    def __init__(self) -> None:
        """
        Initialize a BareMetalServerBootTarget object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
                  ", ".join(['BareMetalServerBootTargetBareMetalServerDiskReference']))
        raise Exception(msg)

class BareMetalServerCPU():
    """
    The bare metal server CPU configuration.

    :attr str architecture: The CPU architecture.
    :attr int core_count: The total number of cores.
    :attr int socket_count: The total number of CPU sockets.
    :attr int threads_per_core: The total number of hardware threads per core.
    """

    def __init__(self,
                 architecture: str,
                 core_count: int,
                 socket_count: int,
                 threads_per_core: int) -> None:
        """
        Initialize a BareMetalServerCPU object.

        :param str architecture: The CPU architecture.
        :param int core_count: The total number of cores.
        :param int socket_count: The total number of CPU sockets.
        :param int threads_per_core: The total number of hardware threads per core.
        """
        self.architecture = architecture
        self.core_count = core_count
        self.socket_count = socket_count
        self.threads_per_core = threads_per_core

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerCPU':
        """Initialize a BareMetalServerCPU object from a json dictionary."""
        args = {}
        if 'architecture' in _dict:
            args['architecture'] = _dict.get('architecture')
        else:
            raise ValueError('Required property \'architecture\' not present in BareMetalServerCPU JSON')
        if 'core_count' in _dict:
            args['core_count'] = _dict.get('core_count')
        else:
            raise ValueError('Required property \'core_count\' not present in BareMetalServerCPU JSON')
        if 'socket_count' in _dict:
            args['socket_count'] = _dict.get('socket_count')
        else:
            raise ValueError('Required property \'socket_count\' not present in BareMetalServerCPU JSON')
        if 'threads_per_core' in _dict:
            args['threads_per_core'] = _dict.get('threads_per_core')
        else:
            raise ValueError('Required property \'threads_per_core\' not present in BareMetalServerCPU JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerCPU object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'architecture') and self.architecture is not None:
            _dict['architecture'] = self.architecture
        if hasattr(self, 'core_count') and self.core_count is not None:
            _dict['core_count'] = self.core_count
        if hasattr(self, 'socket_count') and self.socket_count is not None:
            _dict['socket_count'] = self.socket_count
        if hasattr(self, 'threads_per_core') and self.threads_per_core is not None:
            _dict['threads_per_core'] = self.threads_per_core
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerCPU object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerCPU') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerCPU') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerCollection():
    """
    BareMetalServerCollection.

    :attr List[BareMetalServer] bare_metal_servers: Collection of bare metal
          servers.
    :attr BareMetalServerCollectionFirst first: A link to the first page of
          resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr BareMetalServerCollectionNext next: (optional) A link to the next page of
          resources. This property is present for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 bare_metal_servers: List['BareMetalServer'],
                 first: 'BareMetalServerCollectionFirst',
                 limit: int,
                 total_count: int,
                 *,
                 next: 'BareMetalServerCollectionNext' = None) -> None:
        """
        Initialize a BareMetalServerCollection object.

        :param List[BareMetalServer] bare_metal_servers: Collection of bare metal
               servers.
        :param BareMetalServerCollectionFirst first: A link to the first page of
               resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param int total_count: The total number of resources across all pages.
        :param BareMetalServerCollectionNext next: (optional) A link to the next
               page of resources. This property is present for all pages
               except the last page.
        """
        self.bare_metal_servers = bare_metal_servers
        self.first = first
        self.limit = limit
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerCollection':
        """Initialize a BareMetalServerCollection object from a json dictionary."""
        args = {}
        if 'bare_metal_servers' in _dict:
            args['bare_metal_servers'] = [BareMetalServer.from_dict(v) for v in _dict.get('bare_metal_servers')]
        else:
            raise ValueError('Required property \'bare_metal_servers\' not present in BareMetalServerCollection JSON')
        if 'first' in _dict:
            args['first'] = BareMetalServerCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError('Required property \'first\' not present in BareMetalServerCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError('Required property \'limit\' not present in BareMetalServerCollection JSON')
        if 'next' in _dict:
            args['next'] = BareMetalServerCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError('Required property \'total_count\' not present in BareMetalServerCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'bare_metal_servers') and self.bare_metal_servers is not None:
            bare_metal_servers_list = []
            for v in self.bare_metal_servers:
                if isinstance(v, dict):
                    bare_metal_servers_list.append(v)
                else:
                    bare_metal_servers_list.append(v.to_dict())
            _dict['bare_metal_servers'] = bare_metal_servers_list
        if hasattr(self, 'first') and self.first is not None:
            if isinstance(self.first, dict):
                _dict['first'] = self.first
            else:
                _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'next') and self.next is not None:
            if isinstance(self.next, dict):
                _dict['next'] = self.next
            else:
                _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerCollectionFirst():
    """
    A link to the first page of resources.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BareMetalServerCollectionFirst object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerCollectionFirst':
        """Initialize a BareMetalServerCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerCollectionFirst JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerCollectionNext():
    """
    A link to the next page of resources. This property is present for all pages except
    the last page.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BareMetalServerCollectionNext object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerCollectionNext':
        """Initialize a BareMetalServerCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerCollectionNext JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerConsoleAccessToken():
    """
    The bare metal server console access token information.

    :attr str access_token: A URL safe single-use token used to access the console
          WebSocket.
    :attr str console_type: The bare metal server console type for which this token
          may be used.
    :attr datetime created_at: The date and time that the access token was created.
    :attr datetime expires_at: The date and time that the access token will expire.
    :attr bool force: Indicates whether to disconnect an existing serial console
          session as the serial console cannot be shared.  This has no effect on VNC
          consoles.
    :attr str href: The URL to access this bare metal server console.
    """

    def __init__(self,
                 access_token: str,
                 console_type: str,
                 created_at: datetime,
                 expires_at: datetime,
                 force: bool,
                 href: str) -> None:
        """
        Initialize a BareMetalServerConsoleAccessToken object.

        :param str access_token: A URL safe single-use token used to access the
               console WebSocket.
        :param str console_type: The bare metal server console type for which this
               token may be used.
        :param datetime created_at: The date and time that the access token was
               created.
        :param datetime expires_at: The date and time that the access token will
               expire.
        :param bool force: Indicates whether to disconnect an existing serial
               console session as the serial console cannot be shared.  This has no effect
               on VNC consoles.
        :param str href: The URL to access this bare metal server console.
        """
        self.access_token = access_token
        self.console_type = console_type
        self.created_at = created_at
        self.expires_at = expires_at
        self.force = force
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerConsoleAccessToken':
        """Initialize a BareMetalServerConsoleAccessToken object from a json dictionary."""
        args = {}
        if 'access_token' in _dict:
            args['access_token'] = _dict.get('access_token')
        else:
            raise ValueError('Required property \'access_token\' not present in BareMetalServerConsoleAccessToken JSON')
        if 'console_type' in _dict:
            args['console_type'] = _dict.get('console_type')
        else:
            raise ValueError('Required property \'console_type\' not present in BareMetalServerConsoleAccessToken JSON')
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BareMetalServerConsoleAccessToken JSON')
        if 'expires_at' in _dict:
            args['expires_at'] = string_to_datetime(_dict.get('expires_at'))
        else:
            raise ValueError('Required property \'expires_at\' not present in BareMetalServerConsoleAccessToken JSON')
        if 'force' in _dict:
            args['force'] = _dict.get('force')
        else:
            raise ValueError('Required property \'force\' not present in BareMetalServerConsoleAccessToken JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerConsoleAccessToken JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerConsoleAccessToken object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'access_token') and self.access_token is not None:
            _dict['access_token'] = self.access_token
        if hasattr(self, 'console_type') and self.console_type is not None:
            _dict['console_type'] = self.console_type
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'expires_at') and self.expires_at is not None:
            _dict['expires_at'] = datetime_to_string(self.expires_at)
        if hasattr(self, 'force') and self.force is not None:
            _dict['force'] = self.force
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerConsoleAccessToken object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerConsoleAccessToken') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerConsoleAccessToken') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class ConsoleTypeEnum(str, Enum):
        """
        The bare metal server console type for which this token may be used.
        """
        SERIAL = 'serial'
        VNC = 'vnc'


class BareMetalServerDisk():
    """
    BareMetalServerDisk.

    :attr datetime created_at: The date and time that the disk was created.
    :attr str href: The URL for this bare metal server disk.
    :attr str id: The unique identifier for this bare metal server disk.
    :attr str interface_type: The disk interface used for attaching the disk.
          - `fcp`: Attached using Fiber Channel Protocol
          - `sata`: Attached using Serial Advanced Technology Attachment
          - `nvme`: Attached using Non-Volatile Memory Express
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the resource on which the unexpected
          property value was encountered.
    :attr str name: The name for this bare metal server disk. The name is unique
          across all disks on the bare metal server.
    :attr str resource_type: The resource type.
    :attr int size: The size of the disk in GB (gigabytes).
    """

    def __init__(self,
                 created_at: datetime,
                 href: str,
                 id: str,
                 interface_type: str,
                 name: str,
                 resource_type: str,
                 size: int) -> None:
        """
        Initialize a BareMetalServerDisk object.

        :param datetime created_at: The date and time that the disk was created.
        :param str href: The URL for this bare metal server disk.
        :param str id: The unique identifier for this bare metal server disk.
        :param str interface_type: The disk interface used for attaching the disk.
               - `fcp`: Attached using Fiber Channel Protocol
               - `sata`: Attached using Serial Advanced Technology Attachment
               - `nvme`: Attached using Non-Volatile Memory Express
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected property value was encountered.
        :param str name: The name for this bare metal server disk. The name is
               unique across all disks on the bare metal server.
        :param str resource_type: The resource type.
        :param int size: The size of the disk in GB (gigabytes).
        """
        self.created_at = created_at
        self.href = href
        self.id = id
        self.interface_type = interface_type
        self.name = name
        self.resource_type = resource_type
        self.size = size

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerDisk':
        """Initialize a BareMetalServerDisk object from a json dictionary."""
        args = {}
        if 'created_at' in _dict:
            args['created_at'] = string_to_datetime(_dict.get('created_at'))
        else:
            raise ValueError('Required property \'created_at\' not present in BareMetalServerDisk JSON')
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerDisk JSON')
        if 'id' in _dict:
            args['id'] = _dict.get('id')
        else:
            raise ValueError('Required property \'id\' not present in BareMetalServerDisk JSON')
        if 'interface_type' in _dict:
            args['interface_type'] = _dict.get('interface_type')
        else:
            raise ValueError('Required property \'interface_type\' not present in BareMetalServerDisk JSON')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        else:
            raise ValueError('Required property \'name\' not present in BareMetalServerDisk JSON')
        if 'resource_type' in _dict:
            args['resource_type'] = _dict.get('resource_type')
        else:
            raise ValueError('Required property \'resource_type\' not present in BareMetalServerDisk JSON')
        if 'size' in _dict:
            args['size'] = _dict.get('size')
        else:
            raise ValueError('Required property \'size\' not present in BareMetalServerDisk JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerDisk object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'created_at') and self.created_at is not None:
            _dict['created_at'] = datetime_to_string(self.created_at)
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        if hasattr(self, 'id') and self.id is not None:
            _dict['id'] = self.id
        if hasattr(self, 'interface_type') and self.interface_type is not None:
            _dict['interface_type'] = self.interface_type
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'resource_type') and self.resource_type is not None:
            _dict['resource_type'] = self.resource_type
        if hasattr(self, 'size') and self.size is not None:
            _dict['size'] = self.size
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerDisk object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerDisk') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerDisk') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

    class InterfaceTypeEnum(str, Enum):
        """
        The disk interface used for attaching the disk.
        - `fcp`: Attached using Fiber Channel Protocol
        - `sata`: Attached using Serial Advanced Technology Attachment
        - `nvme`: Attached using Non-Volatile Memory Express
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the resource on which the unexpected
        property value was encountered.
        """
        FCP = 'fcp'
        NVME = 'nvme'
        SATA = 'sata'


    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        BARE_METAL_SERVER_DISK = 'bare_metal_server_disk'


class BareMetalServerDiskCollection():
    """
    BareMetalServerDiskCollection.

    :attr List[BareMetalServerDisk] disks: Collection of the bare metal server's
          disks.
    """

    def __init__(self,
                 disks: List['BareMetalServerDisk']) -> None:
        """
        Initialize a BareMetalServerDiskCollection object.

        :param List[BareMetalServerDisk] disks: Collection of the bare metal
               server's disks.
        """
        self.disks = disks

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerDiskCollection':
        """Initialize a BareMetalServerDiskCollection object from a json dictionary."""
        args = {}
        if 'disks' in _dict:
            args['disks'] = [BareMetalServerDisk.from_dict(v) for v in _dict.get('disks')]
        else:
            raise ValueError('Required property \'disks\' not present in BareMetalServerDiskCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerDiskCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'disks') and self.disks is not None:
            disks_list = []
            for v in self.disks:
                if isinstance(v, dict):
                    disks_list.append(v)
                else:
                    disks_list.append(v.to_dict())
            _dict['disks'] = disks_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerDiskCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerDiskCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerDiskCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerDiskPatch():
    """
    BareMetalServerDiskPatch.

    :attr str name: (optional) The name for this bare metal server disk. The name
          must not be used by another disk on the bare metal server.
    """

    def __init__(self,
                 *,
                 name: str = None) -> None:
        """
        Initialize a BareMetalServerDiskPatch object.

        :param str name: (optional) The name for this bare metal server disk. The
               name must not be used by another disk on the bare metal server.
        """
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerDiskPatch':
        """Initialize a BareMetalServerDiskPatch object from a json dictionary."""
        args = {}
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerDiskPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerDiskPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerDiskPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerDiskPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerDiskReferenceDeleted():
    """
    If present, this property indicates the referenced resource has been deleted, and
    provides some supplementary information.

    :attr str more_info: Link to documentation about deleted resources.
    """

    def __init__(self,
                 more_info: str) -> None:
        """
        Initialize a BareMetalServerDiskReferenceDeleted object.

        :param str more_info: Link to documentation about deleted resources.
        """
        self.more_info = more_info

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerDiskReferenceDeleted':
        """Initialize a BareMetalServerDiskReferenceDeleted object from a json dictionary."""
        args = {}
        if 'more_info' in _dict:
            args['more_info'] = _dict.get('more_info')
        else:
            raise ValueError('Required property \'more_info\' not present in BareMetalServerDiskReferenceDeleted JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerDiskReferenceDeleted object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'more_info') and self.more_info is not None:
            _dict['more_info'] = self.more_info
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerDiskReferenceDeleted object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerDiskReferenceDeleted') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerDiskReferenceDeleted') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerInitialization():
    """
    BareMetalServerInitialization.

    :attr ImageReference image: The image the bare metal server was provisioned
          from.
    :attr List[KeyReference] keys: The public SSH keys used at initialization.
    :attr List[BareMetalServerInitializationUserAccount] user_accounts: The user
          accounts that are created at initialization. There can be multiple account types
          distinguished by the `resource_type` property.
    """

    def __init__(self,
                 image: 'ImageReference',
                 keys: List['KeyReference'],
                 user_accounts: List['BareMetalServerInitializationUserAccount']) -> None:
        """
        Initialize a BareMetalServerInitialization object.

        :param ImageReference image: The image the bare metal server was
               provisioned from.
        :param List[KeyReference] keys: The public SSH keys used at initialization.
        :param List[BareMetalServerInitializationUserAccount] user_accounts: The
               user accounts that are created at initialization. There can be multiple
               account types distinguished by the `resource_type` property.
        """
        self.image = image
        self.keys = keys
        self.user_accounts = user_accounts

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerInitialization':
        """Initialize a BareMetalServerInitialization object from a json dictionary."""
        args = {}
        if 'image' in _dict:
            args['image'] = ImageReference.from_dict(_dict.get('image'))
        else:
            raise ValueError('Required property \'image\' not present in BareMetalServerInitialization JSON')
        if 'keys' in _dict:
            args['keys'] = [KeyReference.from_dict(v) for v in _dict.get('keys')]
        else:
            raise ValueError('Required property \'keys\' not present in BareMetalServerInitialization JSON')
        if 'user_accounts' in _dict:
            args['user_accounts'] = _dict.get('user_accounts')
        else:
            raise ValueError('Required property \'user_accounts\' not present in BareMetalServerInitialization JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerInitialization object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'image') and self.image is not None:
            if isinstance(self.image, dict):
                _dict['image'] = self.image
            else:
                _dict['image'] = self.image.to_dict()
        if hasattr(self, 'keys') and self.keys is not None:
            keys_list = []
            for v in self.keys:
                if isinstance(v, dict):
                    keys_list.append(v)
                else:
                    keys_list.append(v.to_dict())
            _dict['keys'] = keys_list
        if hasattr(self, 'user_accounts') and self.user_accounts is not None:
            user_accounts_list = []
            for v in self.user_accounts:
                if isinstance(v, dict):
                    user_accounts_list.append(v)
                else:
                    user_accounts_list.append(v.to_dict())
            _dict['user_accounts'] = user_accounts_list
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerInitialization object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerInitialization') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerInitialization') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerInitializationPrototype():
    """
    BareMetalServerInitializationPrototype.

    :attr ImageIdentity image: The image to be used when provisioning the bare metal
          server.
    :attr List[KeyIdentity] keys: The public SSH keys to install on the bare metal
          server. Keys will be made available to the bare metal server as cloud-init
          vendor data. For cloud-init enabled images, these keys will also be added as SSH
          authorized keys for the administrative user.
          For Windows images, at least one key must be specified, and one will be chosen
          to encrypt the administrator password. Keys are optional for other images, but
          if no keys are specified, the instance will be inaccessible unless the specified
          image provides another means of access.
    :attr str user_data: (optional) User data to be made available when initializing
          the bare metal server.
    """

    def __init__(self,
                 image: 'ImageIdentity',
                 keys: List['KeyIdentity'],
                 *,
                 user_data: str = None) -> None:
        """
        Initialize a BareMetalServerInitializationPrototype object.

        :param ImageIdentity image: The image to be used when provisioning the bare
               metal server.
        :param List[KeyIdentity] keys: The public SSH keys to install on the bare
               metal server. Keys will be made available to the bare metal server as
               cloud-init vendor data. For cloud-init enabled images, these keys will also
               be added as SSH authorized keys for the administrative user.
               For Windows images, at least one key must be specified, and one will be
               chosen to encrypt the administrator password. Keys are optional for other
               images, but if no keys are specified, the instance will be inaccessible
               unless the specified image provides another means of access.
        :param str user_data: (optional) User data to be made available when
               initializing the bare metal server.
        """
        self.image = image
        self.keys = keys
        self.user_data = user_data

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerInitializationPrototype':
        """Initialize a BareMetalServerInitializationPrototype object from a json dictionary."""
        args = {}
        if 'image' in _dict:
            args['image'] = _dict.get('image')
        else:
            raise ValueError('Required property \'image\' not present in BareMetalServerInitializationPrototype JSON')
        if 'keys' in _dict:
            args['keys'] = _dict.get('keys')
        else:
            raise ValueError('Required property \'keys\' not present in BareMetalServerInitializationPrototype JSON')
        if 'user_data' in _dict:
            args['user_data'] = _dict.get('user_data')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerInitializationPrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'image') and self.image is not None:
            if isinstance(self.image, dict):
                _dict['image'] = self.image
            else:
                _dict['image'] = self.image.to_dict()
        if hasattr(self, 'keys') and self.keys is not None:
            keys_list = []
            for v in self.keys:
                if isinstance(v, dict):
                    keys_list.append(v)
                else:
                    keys_list.append(v.to_dict())
            _dict['keys'] = keys_list
        if hasattr(self, 'user_data') and self.user_data is not None:
            _dict['user_data'] = self.user_data
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerInitializationPrototype object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerInitializationPrototype') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerInitializationPrototype') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerInitializationUserAccount():
    """
    BareMetalServerInitializationUserAccount.

    """

    def __init__(self) -> None:
        """
        Initialize a BareMetalServerInitializationUserAccount object.

        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
                  ", ".join(['BareMetalServerInitializationUserAccountBareMetalServerInitializationHostUserAccount']))
        raise Exception(msg)

class BareMetalServerNetworkInterface():
    """
    BareMetalServerNetworkInterface.

    :attr bool allow_ip_spoofing: Indicates whether source IP spoofing is allowed on
          this interface. If false, source IP spoofing is prevented on this interface. If
          true, source IP spoofing is allowed on this interface.
    :attr datetime created_at: The date and time that the network interface was
          created.
    :attr bool enable_infrastructure_nat: If `true`:
          - The VPC infrastructure performs any needed NAT operations.
          - `floating_ips` must not have more than one floating IP.
          If `false`:
          - Packets are passed unchanged to/from the network interface,
            allowing the workload to perform any needed NAT operations.
          - `allow_ip_spoofing` must be `false`.
          - `interface_type` must not be `hipersocket`.
    :attr List[FloatingIPReference] floating_ips: The floating IPs associated with
          this network interface.
    :attr str href: The URL for this network interface.
    :attr str id: The unique identifier for this network interface.
    :attr str interface_type: The network interface type:
          - `hipersocket`: a virtual network device that provides high-speed TCP/IP
          connectivity
            within a `s390x` based system
          - `pci`: a physical PCI device which can only be created or deleted when the
          bare metal
            server is stopped
            - Has an `allowed_vlans` property which controls the VLANs that will be
          permitted
              to use the PCI interface
            - Cannot directly use an IEEE 802.1q VLAN tag.
          - `vlan`: a virtual device, used through a `pci` device that has the `vlan` in
          its
            array of `allowed_vlans`.
            - Must use an IEEE 802.1q tag.
            - Has its own security groups and does not inherit those of the PCI device
          through
              which traffic flows.
          The enumerated values for this property are expected to expand in the future.
          When processing this property, check for and log unknown values. Optionally halt
          processing and surface the error, or bypass the resource on which the unexpected
          property value was encountered.
    :attr str mac_address: The MAC address of the interface.  If absent, the value
          is not known.
    :attr str name: The name for this network interface.
    :attr int port_speed: The network interface port speed in Mbps.
    :attr ReservedIPReference primary_ip:
    :attr str resource_type: The resource type.
    :attr List[SecurityGroupReference] security_groups: The security groups
          targeting this network interface.
    :attr str status: The status of the network interface.
    :attr SubnetReference subnet: The associated subnet.
    :attr str type: The type of this bare metal server network interface.
    """

    def __init__(self,
                 allow_ip_spoofing: bool,
                 created_at: datetime,
                 enable_infrastructure_nat: bool,
                 floating_ips: List['FloatingIPReference'],
                 href: str,
                 id: str,
                 interface_type: str,
                 mac_address: str,
                 name: str,
                 port_speed: int,
                 primary_ip: 'ReservedIPReference',
                 resource_type: str,
                 security_groups: List['SecurityGroupReference'],
                 status: str,
                 subnet: 'SubnetReference',
                 type: str) -> None:
        """
        Initialize a BareMetalServerNetworkInterface object.

        :param bool allow_ip_spoofing: Indicates whether source IP spoofing is
               allowed on this interface. If false, source IP spoofing is prevented on
               this interface. If true, source IP spoofing is allowed on this interface.
        :param datetime created_at: The date and time that the network interface
               was created.
        :param bool enable_infrastructure_nat: If `true`:
               - The VPC infrastructure performs any needed NAT operations.
               - `floating_ips` must not have more than one floating IP.
               If `false`:
               - Packets are passed unchanged to/from the network interface,
                 allowing the workload to perform any needed NAT operations.
               - `allow_ip_spoofing` must be `false`.
               - `interface_type` must not be `hipersocket`.
        :param List[FloatingIPReference] floating_ips: The floating IPs associated
               with this network interface.
        :param str href: The URL for this network interface.
        :param str id: The unique identifier for this network interface.
        :param str interface_type: The network interface type:
               - `hipersocket`: a virtual network device that provides high-speed TCP/IP
               connectivity
                 within a `s390x` based system
               - `pci`: a physical PCI device which can only be created or deleted when
               the bare metal
                 server is stopped
                 - Has an `allowed_vlans` property which controls the VLANs that will be
               permitted
                   to use the PCI interface
                 - Cannot directly use an IEEE 802.1q VLAN tag.
               - `vlan`: a virtual device, used through a `pci` device that has the `vlan`
               in its
                 array of `allowed_vlans`.
                 - Must use an IEEE 802.1q tag.
                 - Has its own security groups and does not inherit those of the PCI
               device through
                   which traffic flows.
               The enumerated values for this property are expected to expand in the
               future. When processing this property, check for and log unknown values.
               Optionally halt processing and surface the error, or bypass the resource on
               which the unexpected property value was encountered.
        :param str mac_address: The MAC address of the interface.  If absent, the
               value is not known.
        :param str name: The name for this network interface.
        :param int port_speed: The network interface port speed in Mbps.
        :param ReservedIPReference primary_ip:
        :param str resource_type: The resource type.
        :param List[SecurityGroupReference] security_groups: The security groups
               targeting this network interface.
        :param str status: The status of the network interface.
        :param SubnetReference subnet: The associated subnet.
        :param str type: The type of this bare metal server network interface.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
                  ", ".join(['BareMetalServerNetworkInterfaceByHiperSocket', 'BareMetalServerNetworkInterfaceByPCI', 'BareMetalServerNetworkInterfaceByVLAN']))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterface':
        """Initialize a BareMetalServerNetworkInterface object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = ("Cannot convert dictionary into an instance of base class 'BareMetalServerNetworkInterface'. " +
                "The discriminator value should map to a valid subclass: {1}").format(
                  ", ".join(['BareMetalServerNetworkInterfaceByHiperSocket', 'BareMetalServerNetworkInterfaceByPCI', 'BareMetalServerNetworkInterfaceByVLAN']))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a BareMetalServerNetworkInterface object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['hipersocket'] = 'BareMetalServerNetworkInterfaceByHiperSocket'
        mapping['pci'] = 'BareMetalServerNetworkInterfaceByPCI'
        mapping['vlan'] = 'BareMetalServerNetworkInterfaceByVLAN'
        disc_value = _dict.get('interface_type')
        if disc_value is None:
            raise ValueError('Discriminator property \'interface_type\' not found in BareMetalServerNetworkInterface JSON')
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class InterfaceTypeEnum(str, Enum):
        """
        The network interface type:
        - `hipersocket`: a virtual network device that provides high-speed TCP/IP
        connectivity
          within a `s390x` based system
        - `pci`: a physical PCI device which can only be created or deleted when the bare
        metal
          server is stopped
          - Has an `allowed_vlans` property which controls the VLANs that will be
        permitted
            to use the PCI interface
          - Cannot directly use an IEEE 802.1q VLAN tag.
        - `vlan`: a virtual device, used through a `pci` device that has the `vlan` in its
          array of `allowed_vlans`.
          - Must use an IEEE 802.1q tag.
          - Has its own security groups and does not inherit those of the PCI device
        through
            which traffic flows.
        The enumerated values for this property are expected to expand in the future. When
        processing this property, check for and log unknown values. Optionally halt
        processing and surface the error, or bypass the resource on which the unexpected
        property value was encountered.
        """
        HIPERSOCKET = 'hipersocket'
        PCI = 'pci'
        VLAN = 'vlan'


    class ResourceTypeEnum(str, Enum):
        """
        The resource type.
        """
        NETWORK_INTERFACE = 'network_interface'


    class StatusEnum(str, Enum):
        """
        The status of the network interface.
        """
        AVAILABLE = 'available'
        DELETING = 'deleting'
        FAILED = 'failed'
        PENDING = 'pending'


    class TypeEnum(str, Enum):
        """
        The type of this bare metal server network interface.
        """
        PRIMARY = 'primary'
        SECONDARY = 'secondary'


class BareMetalServerNetworkInterfaceCollection():
    """
    BareMetalServerNetworkInterfaceCollection.

    :attr BareMetalServerNetworkInterfaceCollectionFirst first: A link to the first
          page of resources.
    :attr int limit: The maximum number of resources that can be returned by the
          request.
    :attr List[BareMetalServerNetworkInterface] network_interfaces: Collection of
          network interfaces.
    :attr BareMetalServerNetworkInterfaceCollectionNext next: (optional) A link to
          the next page of resources. This property is present for all pages
          except the last page.
    :attr int total_count: The total number of resources across all pages.
    """

    def __init__(self,
                 first: 'BareMetalServerNetworkInterfaceCollectionFirst',
                 limit: int,
                 network_interfaces: List['BareMetalServerNetworkInterface'],
                 total_count: int,
                 *,
                 next: 'BareMetalServerNetworkInterfaceCollectionNext' = None) -> None:
        """
        Initialize a BareMetalServerNetworkInterfaceCollection object.

        :param BareMetalServerNetworkInterfaceCollectionFirst first: A link to the
               first page of resources.
        :param int limit: The maximum number of resources that can be returned by
               the request.
        :param List[BareMetalServerNetworkInterface] network_interfaces: Collection
               of network interfaces.
        :param int total_count: The total number of resources across all pages.
        :param BareMetalServerNetworkInterfaceCollectionNext next: (optional) A
               link to the next page of resources. This property is present for all pages
               except the last page.
        """
        self.first = first
        self.limit = limit
        self.network_interfaces = network_interfaces
        self.next = next
        self.total_count = total_count

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterfaceCollection':
        """Initialize a BareMetalServerNetworkInterfaceCollection object from a json dictionary."""
        args = {}
        if 'first' in _dict:
            args['first'] = BareMetalServerNetworkInterfaceCollectionFirst.from_dict(_dict.get('first'))
        else:
            raise ValueError('Required property \'first\' not present in BareMetalServerNetworkInterfaceCollection JSON')
        if 'limit' in _dict:
            args['limit'] = _dict.get('limit')
        else:
            raise ValueError('Required property \'limit\' not present in BareMetalServerNetworkInterfaceCollection JSON')
        if 'network_interfaces' in _dict:
            args['network_interfaces'] = [BareMetalServerNetworkInterface.from_dict(v) for v in _dict.get('network_interfaces')]
        else:
            raise ValueError('Required property \'network_interfaces\' not present in BareMetalServerNetworkInterfaceCollection JSON')
        if 'next' in _dict:
            args['next'] = BareMetalServerNetworkInterfaceCollectionNext.from_dict(_dict.get('next'))
        if 'total_count' in _dict:
            args['total_count'] = _dict.get('total_count')
        else:
            raise ValueError('Required property \'total_count\' not present in BareMetalServerNetworkInterfaceCollection JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerNetworkInterfaceCollection object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'first') and self.first is not None:
            if isinstance(self.first, dict):
                _dict['first'] = self.first
            else:
                _dict['first'] = self.first.to_dict()
        if hasattr(self, 'limit') and self.limit is not None:
            _dict['limit'] = self.limit
        if hasattr(self, 'network_interfaces') and self.network_interfaces is not None:
            network_interfaces_list = []
            for v in self.network_interfaces:
                if isinstance(v, dict):
                    network_interfaces_list.append(v)
                else:
                    network_interfaces_list.append(v.to_dict())
            _dict['network_interfaces'] = network_interfaces_list
        if hasattr(self, 'next') and self.next is not None:
            if isinstance(self.next, dict):
                _dict['next'] = self.next
            else:
                _dict['next'] = self.next.to_dict()
        if hasattr(self, 'total_count') and self.total_count is not None:
            _dict['total_count'] = self.total_count
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerNetworkInterfaceCollection object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerNetworkInterfaceCollection') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerNetworkInterfaceCollection') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerNetworkInterfaceCollectionFirst():
    """
    A link to the first page of resources.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BareMetalServerNetworkInterfaceCollectionFirst object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterfaceCollectionFirst':
        """Initialize a BareMetalServerNetworkInterfaceCollectionFirst object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerNetworkInterfaceCollectionFirst JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerNetworkInterfaceCollectionFirst object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerNetworkInterfaceCollectionFirst object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerNetworkInterfaceCollectionFirst') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerNetworkInterfaceCollectionFirst') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerNetworkInterfaceCollectionNext():
    """
    A link to the next page of resources. This property is present for all pages except
    the last page.

    :attr str href: The URL for a page of resources.
    """

    def __init__(self,
                 href: str) -> None:
        """
        Initialize a BareMetalServerNetworkInterfaceCollectionNext object.

        :param str href: The URL for a page of resources.
        """
        self.href = href

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterfaceCollectionNext':
        """Initialize a BareMetalServerNetworkInterfaceCollectionNext object from a json dictionary."""
        args = {}
        if 'href' in _dict:
            args['href'] = _dict.get('href')
        else:
            raise ValueError('Required property \'href\' not present in BareMetalServerNetworkInterfaceCollectionNext JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerNetworkInterfaceCollectionNext object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'href') and self.href is not None:
            _dict['href'] = self.href
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerNetworkInterfaceCollectionNext object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerNetworkInterfaceCollectionNext') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerNetworkInterfaceCollectionNext') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerNetworkInterfacePatch():
    """
    BareMetalServerNetworkInterfacePatch.

    :attr bool allow_ip_spoofing: (optional) Indicates whether source IP spoofing is
          allowed on this interface. If false, source IP spoofing is prevented on this
          interface. If true, source IP spoofing is allowed on this interface.
    :attr List[int] allowed_vlans: (optional) Indicates what VLAN IDs (for VLAN type
          only) can use this physical (PCI type) interface.
    :attr bool enable_infrastructure_nat: (optional) If `true`:
          - The VPC infrastructure performs any needed NAT operations.
          - `floating_ips` must not have more than one floating IP.
          If `false`:
          - Packets are passed unchanged to/from the network interface,
            allowing the workload to perform any needed NAT operations.
          - `allow_ip_spoofing` must be `false`.
          - `interface_type` must not be `hipersocket`.
    :attr str name: (optional) The name for this network interface. The name must
          not be used by another network interface on the bare metal server.
    """

    def __init__(self,
                 *,
                 allow_ip_spoofing: bool = None,
                 allowed_vlans: List[int] = None,
                 enable_infrastructure_nat: bool = None,
                 name: str = None) -> None:
        """
        Initialize a BareMetalServerNetworkInterfacePatch object.

        :param bool allow_ip_spoofing: (optional) Indicates whether source IP
               spoofing is allowed on this interface. If false, source IP spoofing is
               prevented on this interface. If true, source IP spoofing is allowed on this
               interface.
        :param List[int] allowed_vlans: (optional) Indicates what VLAN IDs (for
               VLAN type only) can use this physical (PCI type) interface.
        :param bool enable_infrastructure_nat: (optional) If `true`:
               - The VPC infrastructure performs any needed NAT operations.
               - `floating_ips` must not have more than one floating IP.
               If `false`:
               - Packets are passed unchanged to/from the network interface,
                 allowing the workload to perform any needed NAT operations.
               - `allow_ip_spoofing` must be `false`.
               - `interface_type` must not be `hipersocket`.
        :param str name: (optional) The name for this network interface. The name
               must not be used by another network interface on the bare metal server.
        """
        self.allow_ip_spoofing = allow_ip_spoofing
        self.allowed_vlans = allowed_vlans
        self.enable_infrastructure_nat = enable_infrastructure_nat
        self.name = name

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterfacePatch':
        """Initialize a BareMetalServerNetworkInterfacePatch object from a json dictionary."""
        args = {}
        if 'allow_ip_spoofing' in _dict:
            args['allow_ip_spoofing'] = _dict.get('allow_ip_spoofing')
        if 'allowed_vlans' in _dict:
            args['allowed_vlans'] = _dict.get('allowed_vlans')
        if 'enable_infrastructure_nat' in _dict:
            args['enable_infrastructure_nat'] = _dict.get('enable_infrastructure_nat')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerNetworkInterfacePatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'allow_ip_spoofing') and self.allow_ip_spoofing is not None:
            _dict['allow_ip_spoofing'] = self.allow_ip_spoofing
        if hasattr(self, 'allowed_vlans') and self.allowed_vlans is not None:
            _dict['allowed_vlans'] = self.allowed_vlans
        if hasattr(self, 'enable_infrastructure_nat') and self.enable_infrastructure_nat is not None:
            _dict['enable_infrastructure_nat'] = self.enable_infrastructure_nat
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerNetworkInterfacePatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerNetworkInterfacePatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerNetworkInterfacePatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerNetworkInterfacePrototype():
    """
    BareMetalServerNetworkInterfacePrototype.

    :attr bool allow_ip_spoofing: (optional) Indicates whether source IP spoofing is
          allowed on this interface. If false, source IP spoofing is prevented on this
          interface. If true, source IP spoofing is allowed on this interface.
    :attr bool enable_infrastructure_nat: (optional) If `true`:
          - The VPC infrastructure performs any needed NAT operations.
          - `floating_ips` must not have more than one floating IP.
          If `false`:
          - Packets are passed unchanged to/from the network interface,
            allowing the workload to perform any needed NAT operations.
          - `allow_ip_spoofing` must be `false`.
          - `interface_type` must not be `hipersocket`.
    :attr str interface_type: The network interface type:
          - `hipersocket`: a virtual network device that provides high-speed TCP/IP
          connectivity
            within a `s390x` based system
            - Not supported on bare metal servers with a `cpu.architecture` of `amd64`
          - `pci`: a physical PCI device which can only be created or deleted when the
          bare metal
            server is stopped
            - Has an `allowed_vlans` property which controls the VLANs that will be
          permitted
              to use the PCI interface
            - Cannot directly use an IEEE 802.1q VLAN tag.
            - Not supported on bare metal servers with a `cpu.architecture` of `s390x`
          - `vlan`: a virtual device, used through a `pci` device that has the `vlan` in
          its
            array of `allowed_vlans`.
            - Must use an IEEE 802.1q tag.
            - Has its own security groups and does not inherit those of the PCI device
          through
              which traffic flows.
            - Not supported on bare metal servers with a `cpu.architecture` of `s390x`.
    :attr str name: (optional) The name for this network interface. The name must
          not be used by another network interface on the bare metal server. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    :attr NetworkInterfaceIPPrototype primary_ip: (optional) The primary IP address
          to bind to the network interface. This can be specified using
          an existing reserved IP, or a prototype object for a new reserved IP.
          If an existing reserved IP or a prototype object with an address is specified,
          it must
          be available on the network interface's subnet. Otherwise, an available address
          on the
          subnet will be automatically selected and reserved.
    :attr List[SecurityGroupIdentity] security_groups: (optional) The security
          groups to use for this network interface. If unspecified, the VPC's default
          security group is used.
    :attr SubnetIdentity subnet: The associated subnet.
    """

    def __init__(self,
                 interface_type: str,
                 subnet: 'SubnetIdentity',
                 *,
                 allow_ip_spoofing: bool = None,
                 enable_infrastructure_nat: bool = None,
                 name: str = None,
                 primary_ip: 'NetworkInterfaceIPPrototype' = None,
                 security_groups: List['SecurityGroupIdentity'] = None) -> None:
        """
        Initialize a BareMetalServerNetworkInterfacePrototype object.

        :param str interface_type: The network interface type:
               - `hipersocket`: a virtual network device that provides high-speed TCP/IP
               connectivity
                 within a `s390x` based system
                 - Not supported on bare metal servers with a `cpu.architecture` of
               `amd64`
               - `pci`: a physical PCI device which can only be created or deleted when
               the bare metal
                 server is stopped
                 - Has an `allowed_vlans` property which controls the VLANs that will be
               permitted
                   to use the PCI interface
                 - Cannot directly use an IEEE 802.1q VLAN tag.
                 - Not supported on bare metal servers with a `cpu.architecture` of
               `s390x`
               - `vlan`: a virtual device, used through a `pci` device that has the `vlan`
               in its
                 array of `allowed_vlans`.
                 - Must use an IEEE 802.1q tag.
                 - Has its own security groups and does not inherit those of the PCI
               device through
                   which traffic flows.
                 - Not supported on bare metal servers with a `cpu.architecture` of
               `s390x`.
        :param SubnetIdentity subnet: The associated subnet.
        :param bool allow_ip_spoofing: (optional) Indicates whether source IP
               spoofing is allowed on this interface. If false, source IP spoofing is
               prevented on this interface. If true, source IP spoofing is allowed on this
               interface.
        :param bool enable_infrastructure_nat: (optional) If `true`:
               - The VPC infrastructure performs any needed NAT operations.
               - `floating_ips` must not have more than one floating IP.
               If `false`:
               - Packets are passed unchanged to/from the network interface,
                 allowing the workload to perform any needed NAT operations.
               - `allow_ip_spoofing` must be `false`.
               - `interface_type` must not be `hipersocket`.
        :param str name: (optional) The name for this network interface. The name
               must not be used by another network interface on the bare metal server. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param NetworkInterfaceIPPrototype primary_ip: (optional) The primary IP
               address to bind to the network interface. This can be specified using
               an existing reserved IP, or a prototype object for a new reserved IP.
               If an existing reserved IP or a prototype object with an address is
               specified, it must
               be available on the network interface's subnet. Otherwise, an available
               address on the
               subnet will be automatically selected and reserved.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this network interface. If unspecified, the VPC's default
               security group is used.
        """
        msg = "Cannot instantiate base class. Instead, instantiate one of the defined subclasses: {0}".format(
                  ", ".join(['BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByHiperSocketPrototype', 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByPCIPrototype', 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByVLANPrototype']))
        raise Exception(msg)

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerNetworkInterfacePrototype':
        """Initialize a BareMetalServerNetworkInterfacePrototype object from a json dictionary."""
        disc_class = cls._get_class_by_discriminator(_dict)
        if disc_class != cls:
            return disc_class.from_dict(_dict)
        msg = ("Cannot convert dictionary into an instance of base class 'BareMetalServerNetworkInterfacePrototype'. " +
                "The discriminator value should map to a valid subclass: {1}").format(
                  ", ".join(['BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByHiperSocketPrototype', 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByPCIPrototype', 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByVLANPrototype']))
        raise Exception(msg)

    @classmethod
    def _from_dict(cls, _dict: Dict):
        """Initialize a BareMetalServerNetworkInterfacePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    @classmethod
    def _get_class_by_discriminator(cls, _dict: Dict) -> object:
        mapping = {}
        mapping['hipersocket'] = 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByHiperSocketPrototype'
        mapping['pci'] = 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByPCIPrototype'
        mapping['vlan'] = 'BareMetalServerNetworkInterfacePrototypeBareMetalServerNetworkInterfaceByVLANPrototype'
        disc_value = _dict.get('interface_type')
        if disc_value is None:
            raise ValueError('Discriminator property \'interface_type\' not found in BareMetalServerNetworkInterfacePrototype JSON')
        class_name = mapping.get(disc_value, disc_value)
        try:
            disc_class = getattr(sys.modules[__name__], class_name)
        except AttributeError:
            disc_class = cls
        if isinstance(disc_class, object):
            return disc_class
        raise TypeError('%s is not a discriminator class' % class_name)

    class InterfaceTypeEnum(str, Enum):
        """
        The network interface type:
        - `hipersocket`: a virtual network device that provides high-speed TCP/IP
        connectivity
          within a `s390x` based system
          - Not supported on bare metal servers with a `cpu.architecture` of `amd64`
        - `pci`: a physical PCI device which can only be created or deleted when the bare
        metal
          server is stopped
          - Has an `allowed_vlans` property which controls the VLANs that will be
        permitted
            to use the PCI interface
          - Cannot directly use an IEEE 802.1q VLAN tag.
          - Not supported on bare metal servers with a `cpu.architecture` of `s390x`
        - `vlan`: a virtual device, used through a `pci` device that has the `vlan` in its
          array of `allowed_vlans`.
          - Must use an IEEE 802.1q tag.
          - Has its own security groups and does not inherit those of the PCI device
        through
            which traffic flows.
          - Not supported on bare metal servers with a `cpu.architecture` of `s390x`.
        """
        HIPERSOCKET = 'hipersocket'
        PCI = 'pci'
        VLAN = 'vlan'


class BareMetalServerPatch():
    """
    BareMetalServerPatch.

    :attr bool enable_secure_boot: (optional) Indicates whether secure boot is
          enabled. If enabled, the image must support secure boot or the bare metal server
          will fail to boot.
          For `enable_secure_boot` to be changed, the bare metal server `status` must be
          `stopped`.
    :attr str name: (optional) The name for this bare metal server. The name must
          not be used by another bare metal server in the region. Changing the name will
          not affect the system hostname.
    :attr BareMetalServerTrustedPlatformModulePatch trusted_platform_module:
          (optional)
    """

    def __init__(self,
                 *,
                 enable_secure_boot: bool = None,
                 name: str = None,
                 trusted_platform_module: 'BareMetalServerTrustedPlatformModulePatch' = None) -> None:
        """
        Initialize a BareMetalServerPatch object.

        :param bool enable_secure_boot: (optional) Indicates whether secure boot is
               enabled. If enabled, the image must support secure boot or the bare metal
               server will fail to boot.
               For `enable_secure_boot` to be changed, the bare metal server `status` must
               be
               `stopped`.
        :param str name: (optional) The name for this bare metal server. The name
               must not be used by another bare metal server in the region. Changing the
               name will not affect the system hostname.
        :param BareMetalServerTrustedPlatformModulePatch trusted_platform_module:
               (optional)
        """
        self.enable_secure_boot = enable_secure_boot
        self.name = name
        self.trusted_platform_module = trusted_platform_module

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerPatch':
        """Initialize a BareMetalServerPatch object from a json dictionary."""
        args = {}
        if 'enable_secure_boot' in _dict:
            args['enable_secure_boot'] = _dict.get('enable_secure_boot')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'trusted_platform_module' in _dict:
            args['trusted_platform_module'] = BareMetalServerTrustedPlatformModulePatch.from_dict(_dict.get('trusted_platform_module'))
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerPatch object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'enable_secure_boot') and self.enable_secure_boot is not None:
            _dict['enable_secure_boot'] = self.enable_secure_boot
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'trusted_platform_module') and self.trusted_platform_module is not None:
            if isinstance(self.trusted_platform_module, dict):
                _dict['trusted_platform_module'] = self.trusted_platform_module
            else:
                _dict['trusted_platform_module'] = self.trusted_platform_module.to_dict()
        return _dict

    def _to_dict(self):
        """Return a json dictionary representing this model."""
        return self.to_dict()

    def __str__(self) -> str:
        """Return a `str` version of this BareMetalServerPatch object."""
        return json.dumps(self.to_dict(), indent=2)

    def __eq__(self, other: 'BareMetalServerPatch') -> bool:
        """Return `true` when self and other are equal, false otherwise."""
        if not isinstance(other, self.__class__):
            return False
        return self.__dict__ == other.__dict__

    def __ne__(self, other: 'BareMetalServerPatch') -> bool:
        """Return `true` when self and other are not equal, false otherwise."""
        return not self == other

class BareMetalServerPrimaryNetworkInterfacePrototype():
    """
    BareMetalServerPrimaryNetworkInterfacePrototype.

    :attr bool allow_ip_spoofing: (optional) Indicates whether source IP spoofing is
          allowed on this interface. If false, source IP spoofing is prevented on this
          interface. If true, source IP spoofing is allowed on this interface.
    :attr List[int] allowed_vlans: (optional) Indicates what VLAN IDs (for VLAN type
          only) can use this physical (PCI type) interface.
    :attr bool enable_infrastructure_nat: (optional) If `true`:
          - The VPC infrastructure performs any needed NAT operations.
          - `floating_ips` must not have more than one floating IP.
          If `false`:
          - Packets are passed unchanged to/from the network interface,
            allowing the workload to perform any needed NAT operations.
          - `allow_ip_spoofing` must be `false`.
          - `interface_type` must not be `hipersocket`.
    :attr str interface_type: (optional) The network interface type:
          - `hipersocket`: a virtual network device that provides high-speed TCP/IP
          connectivity
            within a `s390x` based system.
            - Not supported on bare metal servers with a `cpu.architecture` of `amd64`
          - `pci`: a physical PCI device which can only be created or deleted when the
          bare metal
            server is stopped
            - Has an `allowed_vlans` property which controls the VLANs that will be
          permitted
              to use the PCI interface
            - Cannot directly use an IEEE 802.1q VLAN tag.
            - Not supported on bare metal servers with a `cpu.architecture` of `s390x`.
    :attr str name: (optional) The name for this network interface. The name must
          not be used by another network interface on the bare metal server. If
          unspecified, the name will be a hyphenated list of randomly-selected words.
    :attr NetworkInterfaceIPPrototype primary_ip: (optional) The primary IP address
          to bind to the network interface. This can be specified using
          an existing reserved IP, or a prototype object for a new reserved IP.
          If an existing reserved IP or a prototype object with an address is specified,
          it must
          be available on the network interface's subnet. Otherwise, an available address
          on the
          subnet will be automatically selected and reserved.
    :attr List[SecurityGroupIdentity] security_groups: (optional) The security
          groups to use for this network interface. If unspecified, the VPC's default
          security group is used.
    :attr SubnetIdentity subnet: The associated subnet.
    """

    def __init__(self,
                 subnet: 'SubnetIdentity',
                 *,
                 allow_ip_spoofing: bool = None,
                 allowed_vlans: List[int] = None,
                 enable_infrastructure_nat: bool = None,
                 interface_type: str = None,
                 name: str = None,
                 primary_ip: 'NetworkInterfaceIPPrototype' = None,
                 security_groups: List['SecurityGroupIdentity'] = None) -> None:
        """
        Initialize a BareMetalServerPrimaryNetworkInterfacePrototype object.

        :param SubnetIdentity subnet: The associated subnet.
        :param bool allow_ip_spoofing: (optional) Indicates whether source IP
               spoofing is allowed on this interface. If false, source IP spoofing is
               prevented on this interface. If true, source IP spoofing is allowed on this
               interface.
        :param List[int] allowed_vlans: (optional) Indicates what VLAN IDs (for
               VLAN type only) can use this physical (PCI type) interface.
        :param bool enable_infrastructure_nat: (optional) If `true`:
               - The VPC infrastructure performs any needed NAT operations.
               - `floating_ips` must not have more than one floating IP.
               If `false`:
               - Packets are passed unchanged to/from the network interface,
                 allowing the workload to perform any needed NAT operations.
               - `allow_ip_spoofing` must be `false`.
               - `interface_type` must not be `hipersocket`.
        :param str interface_type: (optional) The network interface type:
               - `hipersocket`: a virtual network device that provides high-speed TCP/IP
               connectivity
                 within a `s390x` based system.
                 - Not supported on bare metal servers with a `cpu.architecture` of
               `amd64`
               - `pci`: a physical PCI device which can only be created or deleted when
               the bare metal
                 server is stopped
                 - Has an `allowed_vlans` property which controls the VLANs that will be
               permitted
                   to use the PCI interface
                 - Cannot directly use an IEEE 802.1q VLAN tag.
                 - Not supported on bare metal servers with a `cpu.architecture` of
               `s390x`.
        :param str name: (optional) The name for this network interface. The name
               must not be used by another network interface on the bare metal server. If
               unspecified, the name will be a hyphenated list of randomly-selected words.
        :param NetworkInterfaceIPPrototype primary_ip: (optional) The primary IP
               address to bind to the network interface. This can be specified using
               an existing reserved IP, or a prototype object for a new reserved IP.
               If an existing reserved IP or a prototype object with an address is
               specified, it must
               be available on the network interface's subnet. Otherwise, an available
               address on the
               subnet will be automatically selected and reserved.
        :param List[SecurityGroupIdentity] security_groups: (optional) The security
               groups to use for this network interface. If unspecified, the VPC's default
               security group is used.
        """
        self.allow_ip_spoofing = allow_ip_spoofing
        self.allowed_vlans = allowed_vlans
        self.enable_infrastructure_nat = enable_infrastructure_nat
        self.interface_type = interface_type
        self.name = name
        self.primary_ip = primary_ip
        self.security_groups = security_groups
        self.subnet = subnet

    @classmethod
    def from_dict(cls, _dict: Dict) -> 'BareMetalServerPrimaryNetworkInterfacePrototype':
        """Initialize a BareMetalServerPrimaryNetworkInterfacePrototype object from a json dictionary."""
        args = {}
        if 'allow_ip_spoofing' in _dict:
            args['allow_ip_spoofing'] = _dict.get('allow_ip_spoofing')
        if 'allowed_vlans' in _dict:
            args['allowed_vlans'] = _dict.get('allowed_vlans')
        if 'enable_infrastructure_nat' in _dict:
            args['enable_infrastructure_nat'] = _dict.get('enable_infrastructure_nat')
        if 'interface_type' in _dict:
            args['interface_type'] = _dict.get('interface_type')
        if 'name' in _dict:
            args['name'] = _dict.get('name')
        if 'primary_ip' in _dict:
            args['primary_ip'] = _dict.get('primary_ip')
        if 'security_groups' in _dict:
            args['security_groups'] = _dict.get('security_groups')
        if 'subnet' in _dict:
            args['subnet'] = _dict.get('subnet')
        else:
            raise ValueError('Required property \'subnet\' not present in BareMetalServerPrimaryNetworkInterfacePrototype JSON')
        return cls(**args)

    @classmethod
    def _from_dict(cls, _dict):
        """Initialize a BareMetalServerPrimaryNetworkInterfacePrototype object from a json dictionary."""
        return cls.from_dict(_dict)

    def to_dict(self) -> Dict:
        """Return a json dictionary representing this model."""
        _dict = {}
        if hasattr(self, 'allow_ip_spoofing') and self.allow_ip_spoofing is not None:
            _dict['allow_ip_spoofing'] = self.allow_ip_spoofing
        if hasattr(self, 'allowed_vlans') and self.allowed_vlans is not None:
            _dict['allowed_vlans'] = self.allowed_vlans
        if hasattr(self, 'enable_infrastructure_nat') and self.enable_infrastructure_nat is not None:
            _dict['enable_infrastructure_nat'] = self.enable_infrastructure_nat
        if hasattr(self, 'interface_type') and self.interface_type is not None:
            _dict['interface_type'] = self.interface_type
        if hasattr(self, 'name') and self.name is not None:
            _dict['name'] = self.name
        if hasattr(self, 'primary_ip') and self.primary_ip is not None:
            if isinstance(self.primary_ip, dict):
                _dict['primary_ip'] = self.primary_ip
            else:
                _dict['primary_ip'] = self.primary_ip.to_dict()
        if hasattr(self, 'security_groups') and self.security_groups is not None:
            security_groups_list = []
            for v in self.security_groups:
                if isinstance(v, dict):
                    security_groups_list.append(v)
                else:
                    s