#==================================================================================================================

def voigtslow(a, u):
    """ Calculate the voigt function H(a,u).

    Uses numerical integration, so is slow.  Answer is correct to 20
    significant figures.
    """
    try:
        import mpmath as mp
    except ImportError:
        from sympy import mpmath as mp
    with mp.workdps(20):
        z = mp.mpc(u, a)
        result = mp.exp(-z*z) * mp.erfc(-1j*z)

    return result.real

#==================================================================================================================

def voigt(a, u):
    """ Compute the Voigt function.

    a: float
      The lorentzian FWHM divided by the gaussian FWHM of the line.
    u: array of floats
      The frequency offsets from the frequency at the line centre, in
      units of the FWHM frequency of the doppler broadening.

    Returns voigt function H(a,u).

    Notes
    -----
    This is a Taylor approximation to the Voigt function valid for 0 < a <
    0.1. (Harris 1948, ApJ, 108, 112).  Relative error with respect to
    voigt_de is < 10^-4.9 for a < 0.1, gets worse for larger a (since
    it's an expansion in a).

    Examples
    --------
    >>> uvals = np.arange(-25, 25, 1./100)
    >>> a = 0.01
    >>> y1 = voigt(a, uvals)
    >>> y2 = np.fromfile('data/idl_voigt.txt', sep=' ')
    >>> np.allclose(y1, y2, rtol=1e-8)
    True
    >>> uvals = np.linspace(-25, 25, 40)
    >>> a = 1e-5
    >>> y1 = voigt(a, uvals)
    >>> y2 = [voigtslow(a,u) for u in uvals]
    >>> all(mp.almosteq(i, j, 1e-5) for i,j in zip(y1,y2))
    True
    >>> a = 0.1
    >>> y1 = voigt(a, uvals)
    >>> y2 = [voigtslow(a,u) for u in uvals]
    >>> all(mp.almosteq(i, j, 1e-5) for i,j, in zip(y1,y2))
    True
    """

    sqrtpi = sqrt(pi)
    # u values corresponding to H table values below
    uvals = np.arange(0, 20, 0.01)
    
    # Table of h1 and h3 for use in Voigt function
    H1 = np.array( 
        [-1.128379167096,-1.128153506307,-1.127476704444,-1.126349302806,
         -1.124772202831,-1.122746665023,-1.120274307436,-1.117357103734,
         -1.113997380833,-1.110197816115,-1.105961434223,-1.101291603457,
         -1.096192031753,-1.090666762269,-1.084720168573,-1.078356949458,
         -1.071582123367,-1.064401022458,-1.056819286313,-1.048842855295,
         -1.040477963566,-1.031731131783,-1.022609159473,-1.013119117112,
         -1.003268337903,-0.993064409287,-0.982515164176,-0.971628671947,
         -0.960413229188,-0.948877350223,-0.937029757433,-0.924879371370,
         -0.912435300708,-0.899706832010,-0.886703419362,-0.873434673861,
         -0.859910352995,-0.846140349907,-0.832134682585,-0.817903482971,
         -0.803456986018,-0.788805518705,-0.773959489031,-0.758929374994,
         -0.743725713581,-0.728359089780,-0.712840125622,-0.697179469285,
         -0.681387784257,-0.665475738580,-0.649453994194,-0.633333196387,
         -0.617123963365,-0.600836875965,-0.584482467511,-0.568071213835,
         -0.551613523468,-0.535119728024,-0.518600072770,-0.502064707417,
         -0.485523677115,-0.468986913693,-0.452464227121,-0.435965297228,
         -0.419499665677,-0.403076728196,-0.386705727090,-0.370395744018,
         -0.354155693073,-0.337994314132,-0.321920166521,-0.305941622964,
         -0.290066863852,-0.274303871809,-0.258660426580,-0.243144100224,
         -0.227762252632,-0.212522027360,-0.197430347784,-0.182493913569,
         -0.167719197466,-0.153112442425,-0.138679659026,-0.124426623230,
         -0.110358874442,-0.096481713891,-0.082800203315,-0.069319163960,
         -0.056043175874,-0.042976577504,-0.030123465589,-0.017487695339,
         -0.005072880900, 0.007117603905, 0.019080624559, 0.030813284570,
         0.042312923934, 0.053577117337, 0.064603672120, 0.075390626005,
         0.085936244587, 0.096239018606, 0.106297661007, 0.116111103790,
         0.125678494668, 0.134999193535, 0.144072768747, 0.152898993242,
         0.161477840493, 0.169809480309, 0.177894274488, 0.185732772342,
         0.193325706088, 0.200673986130, 0.207778696230, 0.214641088574,
         0.221262578761, 0.227644740700, 0.233789301442, 0.239698135944,
         0.245373261784, 0.250816833826, 0.256031138848, 0.261018590143,
         0.265781722099, 0.270323184764, 0.274645738407, 0.278752248083,
         0.282645678207, 0.286329087142, 0.289805621818, 0.293078512380,
         0.296151066866, 0.299026665943, 0.301708757680, 0.304200852389,
         0.306506517515, 0.308629372606, 0.310573084348, 0.312341361679,
         0.313937950984, 0.315366631379, 0.316631210086, 0.317735517896,
         0.318683404738, 0.319478735348, 0.320125385037, 0.320627235572,
         0.320988171168, 0.321212074587, 0.321302823355, 0.321264286099,
         0.321100318992, 0.320814762332, 0.320411437226, 0.319894142361,
         0.319266651125, 0.318532708393, 0.317696027886, 0.316760289435,
         0.315729136408, 0.314606173263, 0.313394963225, 0.312099026077,
         0.310721836083, 0.309266820022, 0.307737355347, 0.306136768458,
         0.304468333094, 0.302735268836, 0.300940739729, 0.299087853007,
         0.297179657938, 0.295219144761, 0.293209243745, 0.291152824341,
         0.289052694434, 0.286911599696, 0.284732223035, 0.282517184136,
         0.280269039094, 0.277990280130, 0.275683335398, 0.273350568872,
         0.270994280310, 0.268616705302, 0.266220015385, 0.263806318229,
         0.261377657901, 0.258936015180, 0.256483307947, 0.254021391629,
         0.251552059701, 0.249077044244, 0.246598016547, 0.244116587773,
         0.241634309652, 0.239152675232, 0.236673119663, 0.234197021025,
         0.231725701178, 0.229260426666, 0.226802409630, 0.224352808763,
         0.221912730283, 0.219483228934, 0.217065309004, 0.214659925362,
         0.212267984517, 0.209890345679, 0.207527821848, 0.205181180899,
         0.202851146686, 0.200538400145, 0.198243580407, 0.195967285910,
         0.193710075514, 0.191472469618, 0.189254951272, 0.187057967287,
         0.184881929338, 0.182727215069, 0.180594169179, 0.178483104495,
         0.176394303104, 0.174328017332, 0.172284470848, 0.170263859698,
         0.168266353322, 0.166292095577, 0.164341205726, 0.162413779421,
         0.160509889676, 0.158629587805, 0.156772904362, 0.154939850049,
         0.153130416616, 0.151344577731, 0.149582289845, 0.147843493024,
         0.146128111769, 0.144436055819, 0.142767220922, 0.141121489599,
         0.139498731881, 0.137898806029, 0.136321559226, 0.134766828262,
         0.133234440187, 0.131724212947, 0.130235956007, 0.128769470940,
         0.127324552012, 0.125900986736, 0.124498556410, 0.123117036639,
         0.121756197832, 0.120415805687, 0.119095621649, 0.117795403363,
         0.116514905091, 0.115253878132, 0.114012071207, 0.112789230837,
         0.111585101702, 0.110399426986, 0.109231948698, 0.108082407992,
         0.106950545455, 0.105836101395, 0.104738816107, 0.103658430123,
         0.102594684457, 0.101547320830, 0.100516081880, 0.099500711373,
         0.098500954380, 0.097516557468, 0.096547268856, 0.095592838578,
         0.094653018623, 0.093727563075, 0.092816228235, 0.091918772736,
         0.091034957656, 0.090164546608, 0.089307305837, 0.088463004296,
         0.087631413725, 0.086812308713, 0.086005466764, 0.085210668341,
         0.084427696923, 0.083656339036, 0.082896384293, 0.082147625423,
         0.081409858291, 0.080682881921, 0.079966498505, 0.079260513420,
         0.078564735226, 0.077878975668, 0.077203049679, 0.076536775369,
         0.075879974015, 0.075232470052, 0.074594091054, 0.073964667719,
         0.073344033845, 0.072732026308, 0.072128485038, 0.071533252989,
         0.070946176112, 0.070367103323, 0.069795886469, 0.069232380298,
         0.068676442412, 0.068127933258, 0.067586716054, 0.067052656776,
         0.066525624108, 0.066005489409, 0.065492126662, 0.064985412445,
         0.064485225879, 0.063991448591, 0.063503964673, 0.063022660634,
         0.062547425362, 0.062078150081, 0.061614728306, 0.061157055799,
         0.060705030534, 0.060258552644, 0.059817524389, 0.059381850104,
         0.058951436167, 0.058526190951, 0.058106024786, 0.057690849916,
         0.057280580461, 0.056875132379, 0.056474423420, 0.056078373094,
         0.055686902597, 0.055299934908, 0.054917394549, 0.054539207682,
         0.054165302043, 0.053795606904, 0.053430053038, 0.053068572685,
         0.052711099521, 0.052357568621, 0.052007916428, 0.051662080722,
         0.051320000589, 0.050981616389, 0.050646869729, 0.050315703428,
         0.049988061495, 0.049663889096, 0.049343132530, 0.049025739199,
         0.048711657583, 0.048400837216, 0.048093228658, 0.047788783472,
         0.047487454201, 0.047189194340, 0.046893958321, 0.046601701483,
         0.046312380055, 0.046025951131, 0.045742372654, 0.045461603391,
         0.045183602918, 0.044908331595, 0.044635750552, 0.044365821669,
         0.044098507559, 0.043833771549, 0.043571577663, 0.043311890609,
         0.043054675757, 0.042799899130, 0.042547527383, 0.042297527792,
         0.042049868238, 0.041804517192, 0.041561443703, 0.041320617383,
         0.041082008396, 0.040845587444, 0.040611325754, 0.040379195067,
         0.040149167627, 0.039921216168, 0.039695313903, 0.039471434514,
         0.039249552141, 0.039029641375, 0.038811677240, 0.038595635191,
         0.038381491103, 0.038169221257, 0.037958802339, 0.037750211423,
         0.037543425969, 0.037338423811, 0.037135183150, 0.036933682546,
         0.036733900911, 0.036535817501, 0.036339411909, 0.036144664057,
         0.035951554189, 0.035760062868, 0.035570170964, 0.035381859650,
         0.035195110398, 0.035009904968, 0.034826225408, 0.034644054043,
         0.034463373472, 0.034284166563, 0.034106416445, 0.033930106506,
         0.033755220385, 0.033581741970, 0.033409655381, 0.033238945006,
         0.033069595437, 0.032901591502, 0.032734918257, 0.032569560976,
         0.032405505149, 0.032242736480, 0.032081240877, 0.031921004457,
         0.031762013533, 0.031604254614, 0.031447714408, 0.031292379807,
         0.031138237889, 0.030985275916, 0.030833481330, 0.030682841747,
         0.030533344956, 0.030384978917, 0.030237731757, 0.030091591765,
         0.029946547392, 0.029802587246, 0.029659700092, 0.029517874848,
         0.029377100578, 0.029237366499, 0.029098661968, 0.028960976487,
         0.028824299698, 0.028688621380, 0.028553931447, 0.028420219946,
         0.028287477146, 0.028155693172, 0.028024858549, 0.027894963834,
         0.027765999707, 0.027637956969, 0.027510826538, 0.027384599450,
         0.027259266855, 0.027134820015, 0.027011250304, 0.026888549204,
         0.026766708306, 0.026645719305, 0.026525574002, 0.026406264299,
         0.026287782199, 0.026170119804, 0.026053269315, 0.025937223027,
         0.025821973333, 0.025707512715, 0.025593833750, 0.025480929104,
         0.025368791533, 0.025257413880, 0.025146789074, 0.025036910129,
         0.024927770145, 0.024819362301, 0.024711679860, 0.024604716163,
         0.024498464633, 0.024392918767, 0.024288072140, 0.024183918405,
         0.024080451285, 0.023977664578, 0.023875552157, 0.023774107962,
         0.023673326005, 0.023573200367, 0.023473725197, 0.023374894712,
         0.023276703194, 0.023179144990, 0.023082214514, 0.022985906240,
         0.022890214705, 0.022795134511, 0.022700660318, 0.022606786845,
         0.022513508873, 0.022420821238, 0.022328718838, 0.022237196622,
         0.022146249601, 0.022055872835, 0.021966061444, 0.021876810597,
         0.021788115520, 0.021699971488, 0.021612373829, 0.021525317922,
         0.021438799195, 0.021352813126, 0.021267355243, 0.021182421120,
         0.021098006380, 0.021014106692, 0.020930717773, 0.020847835383,
         0.020765455329, 0.020683573461, 0.020602185675, 0.020521287909,
         0.020440876143, 0.020360946400, 0.020281494745, 0.020202517283,
         0.020124010162, 0.020045969566, 0.019968391723, 0.019891272897,
         0.019814609391, 0.019738397547, 0.019662633744, 0.019587314399,
         0.019512435964, 0.019437994927, 0.019363987815, 0.019290411187,
         0.019217261637, 0.019144535794, 0.019072230323, 0.019000341920,
         0.018928867313, 0.018857803268, 0.018787146573, 0.018716894064,
         0.018647042595, 0.018577589056, 0.018508530367, 0.018439863479,
         0.018371585372, 0.018303693057, 0.018236183573, 0.018169053990,
         0.018102301405, 0.018035922942, 0.017969915757, 0.017904277029,
         0.017839003967, 0.017774093806, 0.017709543809, 0.017645351263,
         0.017581513483, 0.017518027809, 0.017454891607, 0.017392102266,
         0.017329657201, 0.017267553854, 0.017205789687, 0.017144362190,
         0.017083268872, 0.017022507270, 0.016962074942, 0.016901969469,
         0.016842188454, 0.016782729524, 0.016723590326, 0.016664768531,
         0.016606261831, 0.016548067937, 0.016490184585, 0.016432609529,
         0.016375340544, 0.016318375428, 0.016261711995, 0.016205348082,
         0.016149281544, 0.016093510257, 0.016038032116, 0.015982845035,
         0.015927946945, 0.015873335798, 0.015819009564, 0.015764966230,
         0.015711203802, 0.015657720305, 0.015604513780, 0.015551582286,
         0.015498923899, 0.015446536712, 0.015394418835, 0.015342568396,
         0.015290983537, 0.015239662419, 0.015188603218, 0.015137804125,
         0.015087263349, 0.015036979113, 0.014986949655, 0.014937173231,
         0.014887648110, 0.014838372576, 0.014789344929, 0.014740563483,
         0.014692026566, 0.014643732522, 0.014595679707, 0.014547866494,
         0.014500291267, 0.014452952426, 0.014405848383, 0.014358977564,
         0.014312338410, 0.014265929373, 0.014219748919, 0.014173795527,
         0.014128067689, 0.014082563909, 0.014037282704, 0.013992222604,
         0.013947382150, 0.013902759897, 0.013858354410, 0.013814164268,
         0.013770188060, 0.013726424388, 0.013682871864, 0.013639529114,
         0.013596394772, 0.013553467487, 0.013510745917, 0.013468228730,
         0.013425914607, 0.013383802238, 0.013341890327, 0.013300177583,
         0.013258662731, 0.013217344504, 0.013176221644, 0.013135292905,
         0.013094557051, 0.013054012855, 0.013013659100, 0.012973494580,
         0.012933518098, 0.012893728465, 0.012854124505, 0.012814705048,
         0.012775468934, 0.012736415014, 0.012697542147, 0.012658849200,
         0.012620335051, 0.012581998585, 0.012543838698, 0.012505854291,
         0.012468044278, 0.012430407578, 0.012392943121, 0.012355649844,
         0.012318526692, 0.012281572619, 0.012244786587, 0.012208167567,
         0.012171714535, 0.012135426478, 0.012099302390, 0.012063341272,
         0.012027542133, 0.011991903989, 0.011956425866, 0.011921106795,
         0.011885945815, 0.011850941973, 0.011816094322, 0.011781401924,
         0.011746863846, 0.011712479164, 0.011678246960, 0.011644166323,
         0.011610236415, 0.011576456207, 0.011542824873, 0.011509341531,
         0.011476005304, 0.011442815321, 0.011409770717, 0.011376870636,
         0.011344114226, 0.011311500642, 0.011279029046, 0.011246698605,
         0.011214508494, 0.011182457891, 0.011150545983, 0.011118771961,
         0.011087135025, 0.011055634376, 0.011024269226, 0.010993038789,
         0.010961942287, 0.010930978946, 0.010900147999, 0.010869448684,
         0.010838880244, 0.010808441929, 0.010778132993, 0.010747952695,
         0.010717900302, 0.010687975084, 0.010658176315, 0.010628503279,
         0.010598955260, 0.010569531549, 0.010540231444, 0.010511054245,
         0.010481999259, 0.010453065798, 0.010424253175, 0.010395560715,
         0.010366987742, 0.010338533587, 0.010310197585, 0.010281979076,
         0.010253877405, 0.010225891921, 0.010198021978, 0.010170266935,
         0.010142626154, 0.010115099002, 0.010087684852, 0.010060383080,
         0.010033193065, 0.010006114194, 0.009979145854, 0.009952287440,
         0.009925538348, 0.009898897981, 0.009872365744, 0.009845941047,
         0.009819623305, 0.009793411934, 0.009767306357, 0.009741306000,
         0.009715410293, 0.009689618670, 0.009663930567, 0.009638345428,
         0.009612862696, 0.009587481821, 0.009562202255, 0.009537023456,
         0.009511944883, 0.009486966000, 0.009462086274, 0.009437305177,
         0.009412622182, 0.009388036769, 0.009363548418, 0.009339156614,
         0.009314860847, 0.009290660607, 0.009266555391, 0.009242544697,
         0.009218628026, 0.009194804885, 0.009171074781, 0.009147437227,
         0.009123891737, 0.009100437830, 0.009077075027, 0.009053802853,
         0.009030620835, 0.009007528505, 0.008984525396, 0.008961611045,
         0.008938784992, 0.008916046779, 0.008893395954, 0.008870832064,
         0.008848354662, 0.008825963303, 0.008803657543, 0.008781436944,
         0.008759301068, 0.008737249482, 0.008715281756, 0.008693397459,
         0.008671596168, 0.008649877459, 0.008628240913, 0.008606686111,
         0.008585212639, 0.008563820085, 0.008542508040, 0.008521276096,
         0.008500123850, 0.008479050900, 0.008458056847, 0.008437141293,
         0.008416303846, 0.008395544113, 0.008374861705, 0.008354256236,
         0.008333727321, 0.008313274579, 0.008292897630, 0.008272596097,
         0.008252369605, 0.008232217783, 0.008212140259, 0.008192136667,
         0.008172206641, 0.008152349817, 0.008132565835, 0.008112854337,
         0.008093214965, 0.008073647366, 0.008054151187, 0.008034726079,
         0.008015371695, 0.007996087687, 0.007976873714, 0.007957729433,
         0.007938654505, 0.007919648594, 0.007900711364, 0.007881842482,
         0.007863041617, 0.007844308441, 0.007825642625, 0.007807043845,
         0.007788511779, 0.007770046104, 0.007751646503, 0.007733312657,
         0.007715044251, 0.007696840973, 0.007678702509, 0.007660628552,
         0.007642618793, 0.007624672926, 0.007606790647, 0.007588971653,
         0.007571215645, 0.007553522324, 0.007535891393, 0.007518322556,
         0.007500815521, 0.007483369996, 0.007465985690, 0.007448662316,
         0.007431399588, 0.007414197219, 0.007397054929, 0.007379972434,
         0.007362949455, 0.007345985715, 0.007329080936, 0.007312234844,
         0.007295447166, 0.007278717629, 0.007262045965, 0.007245431904,
         0.007228875179, 0.007212375525, 0.007195932679, 0.007179546378,
         0.007163216361, 0.007146942369, 0.007130724144, 0.007114561431,
         0.007098453973, 0.007082401519, 0.007066403816, 0.007050460614,
         0.007034571663, 0.007018736717, 0.007002955529, 0.006987227854,
         0.006971553450, 0.006955932074, 0.006940363485, 0.006924847446,
         0.006909383717, 0.006893972062, 0.006878612247, 0.006863304038,
         0.006848047201, 0.006832841507, 0.006817686725, 0.006802582626,
         0.006787528984, 0.006772525572, 0.006757572167, 0.006742668544,
         0.006727814481, 0.006713009757, 0.006698254153, 0.006683547451,
         0.006668889432, 0.006654279882, 0.006639718585, 0.006625205327,
         0.006610739897, 0.006596322083, 0.006581951674, 0.006567628463,
         0.006553352241, 0.006539122801, 0.006524939939, 0.006510803450,
         0.006496713130, 0.006482668778, 0.006468670193, 0.006454717175,
         0.006440809524, 0.006426947044, 0.006413129538, 0.006399356810,
         0.006385628666, 0.006371944913, 0.006358305357, 0.006344709808,
         0.006331158076, 0.006317649971, 0.006304185306, 0.006290763892,
         0.006277385544, 0.006264050077, 0.006250757306, 0.006237507049,
         0.006224299123, 0.006211133347, 0.006198009541, 0.006184927525,
         0.006171887122, 0.006158888154, 0.006145930444, 0.006133013818,
         0.006120138100, 0.006107303117, 0.006094508697, 0.006081754667,
         0.006069040857, 0.006056367097, 0.006043733219, 0.006031139053,
         0.006018584432, 0.006006069192, 0.005993593165, 0.005981156187,
         0.005968758095, 0.005956398726, 0.005944077917, 0.005931795508,
         0.005919551338, 0.005907345249, 0.005895177080, 0.005883046675,
         0.005870953876, 0.005858898528, 0.005846880474, 0.005834899562,
         0.005822955635, 0.005811048543, 0.005799178132, 0.005787344252,
         0.005775546751, 0.005763785481, 0.005752060291, 0.005740371034,
         0.005728717562, 0.005717099730, 0.005705517389, 0.005693970397,
         0.005682458607, 0.005670981876, 0.005659540062, 0.005648133022,
         0.005636760614, 0.005625422698, 0.005614119134, 0.005602849782,
         0.005591614503, 0.005580413160, 0.005569245616, 0.005558111734,
         0.005547011377, 0.005535944412, 0.005524910702, 0.005513910116,
         0.005502942519, 0.005492007778, 0.005481105763, 0.005470236342,
         0.005459399385, 0.005448594761, 0.005437822342, 0.005427081999,
         0.005416373605, 0.005405697031, 0.005395052152, 0.005384438841,
         0.005373856973, 0.005363306424, 0.005352787069, 0.005342298785,
         0.005331841449, 0.005321414939, 0.005311019132, 0.005300653909,
         0.005290319149, 0.005280014731, 0.005269740537, 0.005259496447,
         0.005249282345, 0.005239098112, 0.005228943631, 0.005218818786,
         0.005208723461, 0.005198657585, 0.005188620956, 0.005178613502,
         0.005168635112, 0.005158685671, 0.005148765068, 0.005138873190,
         0.005129009927, 0.005119175167, 0.005109368800, 0.005099590716,
         0.005089840807, 0.005080118963, 0.005070425077, 0.005060759041,
         0.005051120748, 0.005041510090, 0.005031926964, 0.005022371261,
         0.005012842879, 0.005003341711, 0.004993867654, 0.004984420605,
         0.004975000461, 0.004965607118, 0.004956240476, 0.004946900431,
         0.004937586884, 0.004928299734, 0.004919038880, 0.004909804223,
         0.004900595664, 0.004891413104, 0.004882256445, 0.004873125588,
         0.004864020438, 0.004854940896, 0.004845886867, 0.004836858255,
         0.004827854964, 0.004818876899, 0.004809923965, 0.004800996069,
         0.004792093117, 0.004783215015, 0.004774361670, 0.004765532991,
         0.004756728886, 0.004747949262, 0.004739194029, 0.004730463096,
         0.004721756373, 0.004713073770, 0.004704415197, 0.004695780566,
         0.004687169789, 0.004678582776, 0.004670019440, 0.004661479694,
         0.004652963451, 0.004644470625, 0.004636001128, 0.004627554876,
         0.004619131783, 0.004610731765, 0.004602354735, 0.004594000612,
         0.004585669309, 0.004577360745, 0.004569074837, 0.004560811501,
         0.004552570655, 0.004544352217, 0.004536156107, 0.004527982242,
         0.004519830542, 0.004511700927, 0.004503593316, 0.004495507630,
         0.004487443790, 0.004479401716, 0.004471381331, 0.004463382555,
         0.004455405311, 0.004447449521, 0.004439515108, 0.004431601996,
         0.004423710108, 0.004415839367, 0.004407989699, 0.004400161027,
         0.004392353276, 0.004384566371, 0.004376800239, 0.004369054805,
         0.004361329995, 0.004353625735, 0.004345941954, 0.004338278577,
         0.004330635532, 0.004323012748, 0.004315410152, 0.004307827672,
         0.004300265239, 0.004292722780, 0.004285200226, 0.004277697505,
         0.004270214548, 0.004262751286, 0.004255307649, 0.004247883568,
         0.004240478974, 0.004233093799, 0.004225727975, 0.004218381434,
         0.004211054109, 0.004203745931, 0.004196456836, 0.004189186754,
         0.004181935622, 0.004174703371, 0.004167489937, 0.004160295255,
         0.004153119258, 0.004145961882, 0.004138823063, 0.004131702735,
         0.004124600836, 0.004117517301, 0.004110452067, 0.004103405070,
         0.004096376248, 0.004089365537, 0.004082372876, 0.004075398203,
         0.004068441455, 0.004061502571, 0.004054581489, 0.004047678149,
         0.004040792489, 0.004033924449, 0.004027073970, 0.004020240990,
         0.004013425450, 0.004006627290, 0.003999846452, 0.003993082875,
         0.003986336503, 0.003979607275, 0.003972895133, 0.003966200021,
         0.003959521879, 0.003952860651, 0.003946216278, 0.003939588705,
         0.003932977874, 0.003926383729, 0.003919806213, 0.003913245271,
         0.003906700846, 0.003900172883, 0.003893661326, 0.003887166121,
         0.003880687213, 0.003874224546, 0.003867778066, 0.003861347720,
         0.003854933453, 0.003848535212, 0.003842152942, 0.003835786592,
         0.003829436106, 0.003823101434, 0.003816782521, 0.003810479316,
         0.003804191766, 0.003797919820, 0.003791663426, 0.003785422532,
         0.003779197086, 0.003772987038, 0.003766792337, 0.003760612932,
         0.003754448773, 0.003748299808, 0.003742165989, 0.003736047266,
         0.003729943588, 0.003723854906, 0.003717781171, 0.003711722334,
         0.003705678346, 0.003699649159, 0.003693634723, 0.003687634991,
         0.003681649915, 0.003675679447, 0.003669723539, 0.003663782143,
         0.003657855213, 0.003651942701, 0.003646044561, 0.003640160745,
         0.003634291208, 0.003628435902, 0.003622594783, 0.003616767803,
         0.003610954917, 0.003605156080, 0.003599371246, 0.003593600370,
         0.003587843407, 0.003582100311, 0.003576371040, 0.003570655547,
         0.003564953789, 0.003559265722, 0.003553591301, 0.003547930483,
         0.003542283225, 0.003536649482, 0.003531029212, 0.003525422371,
         0.003519828917, 0.003514248807, 0.003508681999, 0.003503128449,
         0.003497588116, 0.003492060959, 0.003486546934, 0.003481046000,
         0.003475558116, 0.003470083240, 0.003464621331, 0.003459172348,
         0.003453736250, 0.003448312997, 0.003442902548, 0.003437504862,
         0.003432119899, 0.003426747620, 0.003421387983, 0.003416040950,
         0.003410706481, 0.003405384537, 0.003400075077, 0.003394778063,
         0.003389493457, 0.003384221218, 0.003378961309, 0.003373713691,
         0.003368478325, 0.003363255173, 0.003358044198, 0.003352845361,
         0.003347658624, 0.003342483951, 0.003337321302, 0.003332170642,
         0.003327031932, 0.003321905136, 0.003316790217, 0.003311687138,
         0.003306595863, 0.003301516354, 0.003296448576, 0.003291392492,
         0.003286348067, 0.003281315264, 0.003276294048, 0.003271284383,
         0.003266286233, 0.003261299564, 0.003256324339, 0.003251360524,
         0.003246408084, 0.003241466984, 0.003236537189, 0.003231618665,
         0.003226711377, 0.003221815291, 0.003216930373, 0.003212056589,
         0.003207193904, 0.003202342286, 0.003197501700, 0.003192672113,
         0.003187853491, 0.003183045802, 0.003178249012, 0.003173463088,
         0.003168687997, 0.003163923706, 0.003159170184, 0.003154427396,
         0.003149695312, 0.003144973898, 0.003140263123, 0.003135562954,
         0.003130873359, 0.003126194308, 0.003121525767, 0.003116867706,
         0.003112220093, 0.003107582897, 0.003102956086, 0.003098339630,
         0.003093733497, 0.003089137657, 0.003084552079, 0.003079976733,
         0.003075411587, 0.003070856612, 0.003066311777, 0.003061777051,
         0.003057252406, 0.003052737811, 0.003048233237, 0.003043738652,
         0.003039254029, 0.003034779337, 0.003030314547, 0.003025859629,
         0.003021414556, 0.003016979297, 0.003012553823, 0.003008138106,
         0.003003732117, 0.002999335828, 0.002994949209, 0.002990572233,
         0.002986204871, 0.002981847094, 0.002977498876, 0.002973160187,
         0.002968831000, 0.002964511287, 0.002960201021, 0.002955900173,
         0.002951608717, 0.002947326624, 0.002943053868, 0.002938790421,
         0.002934536257, 0.002930291348, 0.002926055667, 0.002921829188,
         0.002917611884, 0.002913403728, 0.002909204693, 0.002905014755,
         0.002900833885, 0.002896662058, 0.002892499247, 0.002888345427,
         0.002884200572, 0.002880064656, 0.002875937653, 0.002871819538,
         0.002867710284, 0.002863609866, 0.002859518260, 0.002855435439,
         0.002851361379, 0.002847296054, 0.002843239440, 0.002839191511,
         0.002835152243, 0.002831121611, 0.002827099590, 0.002823086155,
         0.002819081283, 0.002815084948, 0.002811097127, 0.002807117795,
         0.002803146928, 0.002799184502, 0.002795230493, 0.002791284877,
         0.002787347631, 0.002783418730, 0.002779498151, 0.002775585871,
         0.002771681866, 0.002767786112, 0.002763898587, 0.002760019267,
         0.002756148128, 0.002752285149, 0.002748430306, 0.002744583575,
         0.002740744935, 0.002736914363, 0.002733091835, 0.002729277330,
         0.002725470825, 0.002721672297, 0.002717881724, 0.002714099084,
         0.002710324354, 0.002706557514, 0.002702798539, 0.002699047410,
         0.002695304103, 0.002691568597, 0.002687840870, 0.002684120901,
         0.002680408668, 0.002676704149, 0.002673007324, 0.002669318170,
         0.002665636667, 0.002661962793, 0.002658296527, 0.002654637849,
         0.002650986737, 0.002647343170, 0.002643707128, 0.002640078589,
         0.002636457533, 0.002632843940, 0.002629237789, 0.002625639059,
         0.002622047730, 0.002618463782, 0.002614887194, 0.002611317947,
         0.002607756019, 0.002604201392, 0.002600654045, 0.002597113958,
         0.002593581111, 0.002590055485, 0.002586537060, 0.002583025816,
         0.002579521733, 0.002576024793, 0.002572534976, 0.002569052261,
         0.002565576631, 0.002562108065, 0.002558646545, 0.002555192052,
         0.002551744566, 0.002548304068, 0.002544870540, 0.002541443962,
         0.002538024316, 0.002534611584, 0.002531205746, 0.002527806784,
         0.002524414679, 0.002521029413, 0.002517650967, 0.002514279324,
         0.002510914464, 0.002507556370, 0.002504205023, 0.002500860405,
         0.002497522499, 0.002494191286, 0.002490866749, 0.002487548869,
         0.002484237628, 0.002480933010, 0.002477634995, 0.002474343568,
         0.002471058709, 0.002467780402, 0.002464508629, 0.002461243373,
         0.002457984616, 0.002454732341, 0.002451486532, 0.002448247169,
         0.002445014238, 0.002441787720, 0.002438567598, 0.002435353856,
         0.002432146477, 0.002428945444, 0.002425750740, 0.002422562349,
         0.002419380253, 0.002416204437, 0.002413034883, 0.002409871575,
         0.002406714497, 0.002403563633, 0.002400418965, 0.002397280478,
         0.002394148183, 0.002391022009, 0.002387901967, 0.002384788041,
         0.002381680216, 0.002378578474, 0.002375482801, 0.002372393180,
         0.002369309596, 0.002366232033, 0.002363160475, 0.002360094907,
         0.002357035312, 0.002353981676, 0.002350933983, 0.002347892217,
         0.002344856364, 0.002341826407, 0.002338802331, 0.002335784122,
         0.002332771764, 0.002329765241, 0.002326764540, 0.002323769644,
         0.002320780538, 0.002317797209, 0.002314819640, 0.002311847818,
         0.002308881726, 0.002305921351, 0.002302966678, 0.002300017692,
         0.002297074378, 0.002294136722, 0.002291204709, 0.002288278326,
         0.002285357556, 0.002282442387, 0.002279532804, 0.002276628792,
         0.002273730337, 0.002270837425, 0.002267950042, 0.002265068174,
         0.002262191807, 0.002259320926, 0.002256455517, 0.002253595568,
         0.002250741063, 0.002247891990, 0.002245048333, 0.002242210080,
         0.002239377217, 0.002236549730, 0.002233727606, 0.002230910830,
         0.002228099390, 0.002225293271, 0.002222492461, 0.002219696946,
         0.002216906713, 0.002214121748, 0.002211342038, 0.002208567569,
         0.002205798330, 0.002203034306, 0.002200275484, 0.002197521851,
         0.002194773395, 0.002192030102, 0.002189291959, 0.002186558953,
         0.002183831072, 0.002181108303, 0.002178390633, 0.002175678048,
         0.002172970537, 0.002170268087, 0.002167570685, 0.002164878319,
         0.002162190975, 0.002159508643, 0.002156831308, 0.002154158958,
         0.002151491582, 0.002148829167, 0.002146171700, 0.002143519169,
         0.002140871562, 0.002138228868, 0.002135591072, 0.002132958164,
         0.002130330132, 0.002127706963, 0.002125088645, 0.002122475167,
         0.002119866516, 0.002117262680, 0.002114663648, 0.002112069408,
         0.002109479948, 0.002106895257, 0.002104315321, 0.002101740131,
         0.002099169674, 0.002096603938, 0.002094042912, 0.002091486585,
         0.002088934945, 0.002086387980, 0.002083845679, 0.002081308031,
         0.002078775024, 0.002076246648, 0.002073722889, 0.002071203738,
         0.002068689184, 0.002066179214, 0.002063673818, 0.002061172985,
         0.002058676703, 0.002056184961, 0.002053697750, 0.002051215056,
         0.002048736871, 0.002046263181, 0.002043793978, 0.002041329250,
         0.002038868985, 0.002036413174, 0.002033961805, 0.002031514868,
         0.002029072352, 0.002026634246, 0.002024200540, 0.002021771223,
         0.002019346285, 0.002016925715, 0.002014509503, 0.002012097637,
         0.002009690108, 0.002007286906, 0.002004888019, 0.002002493438,
         0.002000103152, 0.001997717151, 0.001995335424, 0.001992957962,
         0.001990584754, 0.001988215791, 0.001985851061, 0.001983490555,
         0.001981134263, 0.001978782174, 0.001976434279, 0.001974090568,
         0.001971751031, 0.001969415658, 0.001967084438, 0.001964757363,
         0.001962434422, 0.001960115605, 0.001957800903, 0.001955490306,
         0.001953183805, 0.001950881388, 0.001948583048, 0.001946288774,
         0.001943998556, 0.001941712386, 0.001939430253, 0.001937152148,
         0.001934878062, 0.001932607985, 0.001930341907, 0.001928079819,
         0.001925821713, 0.001923567577, 0.001921317404, 0.001919071184,
         0.001916828908, 0.001914590565, 0.001912356148, 0.001910125647,
         0.001907899053, 0.001905676356, 0.001903457548, 0.001901242619,
         0.001899031560, 0.001896824363, 0.001894621019, 0.001892421517,
         0.001890225851, 0.001888034009, 0.001885845985, 0.001883661768,
         0.001881481350, 0.001879304722, 0.001877131876, 0.001874962802,
         0.001872797491, 0.001870635936, 0.001868478127, 0.001866324056,
         0.001864173714, 0.001862027093, 0.001859884183, 0.001857744977,
         0.001855609465, 0.001853477639, 0.001851349492, 0.001849225013,
         0.001847104196, 0.001844987031, 0.001842873509, 0.001840763624,
         0.001838657366, 0.001836554726, 0.001834455698, 0.001832360271,
         0.001830268439, 0.001828180193, 0.001826095525, 0.001824014426,
         0.001821936888, 0.001819862904, 0.001817792465, 0.001815725562,
         0.001813662189, 0.001811602337, 0.001809545997, 0.001807493163,
         0.001805443825, 0.001803397977, 0.001801355609, 0.001799316715,
         0.001797281286, 0.001795249314, 0.001793220792, 0.001791195711,
         0.001789174065, 0.001787155845, 0.001785141043, 0.001783129652,
         0.001781121664, 0.001779117071, 0.001777115866, 0.001775118040,
         0.001773123588, 0.001771132500, 0.001769144769, 0.001767160388,
         0.001765179349, 0.001763201645, 0.001761227267, 0.001759256210,
         0.001757288464, 0.001755324024, 0.001753362880, 0.001751405027,
         0.001749450457, 0.001747499161, 0.001745551134, 0.001743606367,
         0.001741664853, 0.001739726586, 0.001737791557, 0.001735859760,
         0.001733931188, 0.001732005833, 0.001730083687, 0.001728164745,
         0.001726248999, 0.001724336441, 0.001722427065, 0.001720520863,
         0.001718617829, 0.001716717955, 0.001714821235, 0.001712927662,
         0.001711037227, 0.001709149926, 0.001707265750, 0.001705384693,
         0.001703506748, 0.001701631907, 0.001699760165, 0.001697891514,
         0.001696025948, 0.001694163459, 0.001692304041, 0.001690447687,
         0.001688594391, 0.001686744146, 0.001684896944, 0.001683052780,
         0.001681211646, 0.001679373537, 0.001677538444, 0.001675706363,
         0.001673877285, 0.001672051206, 0.001670228117, 0.001668408013,
         0.001666590887, 0.001664776732, 0.001662965542, 0.001661157311,
         0.001659352031, 0.001657549698, 0.001655750303, 0.001653953842,
         0.001652160307, 0.001650369692, 0.001648581990, 0.001646797197,
         0.001645015304, 0.001643236306, 0.001641460196, 0.001639686969,
         0.001637916618, 0.001636149137, 0.001634384519, 0.001632622759,
         0.001630863850, 0.001629107786, 0.001627354561, 0.001625604168,
         0.001623856603, 0.001622111858, 0.001620369927, 0.001618630805,
         0.001616894486, 0.001615160962, 0.001613430230, 0.001611702281,
         0.001609977111, 0.001608254714, 0.001606535083, 0.001604818212,
         0.001603104096, 0.001601392729, 0.001599684105, 0.001597978218,
         0.001596275062, 0.001594574631, 0.001592876920, 0.001591181923,
         0.001589489633, 0.001587800046, 0.001586113155, 0.001584428955,
         0.001582747439, 0.001581068603, 0.001579392440, 0.001577718945,
         0.001576048112, 0.001574379936, 0.001572714411, 0.001571051531,
         0.001569391290, 0.001567733684, 0.001566078706, 0.001564426351,
         0.001562776613, 0.001561129487, 0.001559484967, 0.001557843049,
         0.001556203725, 0.001554566992, 0.001552932843, 0.001551301272,
         0.001549672276, 0.001548045847, 0.001546421981, 0.001544800672,
         0.001543181915, 0.001541565705, 0.001539952036, 0.001538340903,
         0.001536732301, 0.001535126223, 0.001533522666, 0.001531921623,
         0.001530323090, 0.001528727061, 0.001527133531, 0.001525542495,
         0.001523953947, 0.001522367883, 0.001520784296, 0.001519203183,
         0.001517624538, 0.001516048355, 0.001514474629, 0.001512903357,
         0.001511334531, 0.001509768148, 0.001508204202, 0.001506642688,
         0.001505083601, 0.001503526936, 0.001501972689, 0.001500420853,
         0.001498871424, 0.001497324398, 0.001495779768, 0.001494237530,
         0.001492697680, 0.001491160212, 0.001489625121, 0.001488092403,
         0.001486562052, 0.001485034064, 0.001483508433, 0.001481985156,
         0.001480464226, 0.001478945640, 0.001477429393, 0.001475915478,
         0.001474403893, 0.001472894632, 0.001471387690, 0.001469883063,
         0.001468380745, 0.001466880732, 0.001465383020, 0.001463887603,
         0.001462394478, 0.001460903638, 0.001459415080, 0.001457928799,
         0.001456444791, 0.001454963050, 0.001453483572, 0.001452006352,
         0.001450531386, 0.001449058669, 0.001447588197, 0.001446119965,
         0.001444653969, 0.001443190203, 0.001441728664, 0.001440269347,
         0.001438812248, 0.001437357361, 0.001435904683, 0.001434454208,
         0.001433005933, 0.001431559854, 0.001430115964, 0.001428674261,
         0.001427234740, 0.001425797396, 0.001424362225, 0.001422929223,
         0.001421498385, 0.001420069706, 0.001418643183, 0.001417218812])
    H3 = np.array(
        [-0.752252778064,-0.751951907041,-0.751049654958,-0.749547104192,
         -0.747446056783,-0.744749031550,-0.741459260066,-0.737580681497,
         -0.733117936310,-0.728076358876,-0.722461968963,-0.716281462157,
         -0.709542199216,-0.702252194382,-0.694420102672,-0.686055206185,
         -0.677167399429,-0.667767173726,-0.657865600697,-0.647474314879,
         -0.636605495506,-0.625271847477,-0.613486581562,-0.601263393883,
         -0.588616444696,-0.575560336534,-0.562110091745,-0.548281129459,
         -0.534089242044,-0.519550571089,-0.504681582955,-0.489499043946,
         -0.474019995148,-0.458261726974,-0.442241753478,-0.425977786464,
         -0.409487709464,-0.392789551607,-0.375901461443,-0.358841680766,
         -0.341628518477,-0.324280324544,-0.306815464089,-0.289252291671,
         -0.271609125783,-0.253904223627,-0.236155756202,-0.218381783743,
         -0.200600231563,-0.182828866326,-0.165085272797,-0.147386831102,
         -0.129750694537,-0.112193767960,-0.094732686795,-0.077383796679,
         -0.060163133797,-0.043086405902,-0.026168974085,-0.009425835284,
          0.007128394424, 0.023479495729, 0.039613661181, 0.055517509450,
          0.071178098729, 0.086582939277, 0.101720005089, 0.116577744673,
          0.131145090944, 0.145411470205, 0.159366810241, 0.173001547492,
          0.186306633327, 0.199273539414, 0.211894262182, 0.224161326392,
          0.236067787813, 0.247607235019, 0.258773790309, 0.269562109768,
          0.279967382485, 0.289985328924, 0.299612198492, 0.308844766297,
          0.317680329127, 0.326116700665, 0.334152205965, 0.341785675206,
          0.349016436756, 0.355844309555, 0.362269594861, 0.368293067366,
          0.373915965728, 0.379139982525, 0.383967253684, 0.388400347386,
          0.392442252501, 0.396096366567, 0.399366483349, 0.402256780005,
          0.404771803894, 0.406916459051, 0.408695992364, 0.410115979481,
          0.411182310478, 0.411901175319, 0.412279049136, 0.412322677365,
          0.412039060758, 0.411435440304, 0.410519282100, 0.409298262172,
          0.407780251309, 0.405973299902, 0.403885622848, 0.401525584513,
          0.398901683806, 0.396022539369, 0.392896874922, 0.389533504769,
          0.385941319503, 0.382129271921, 0.378106363172, 0.373881629157,
          0.369464127198, 0.364862923000, 0.360087077909, 0.355145636493,
          0.350047614456, 0.344801986899, 0.339417676935, 0.333903544682,
          0.328268376626, 0.322520875384, 0.316669649852, 0.310723205768,
          0.304689936683, 0.298578115342, 0.292395885492, 0.286151254111,
          0.279852084063, 0.273506087181, 0.267120817773, 0.260703666564,
          0.254261855060, 0.247802430334, 0.241332260239, 0.234858029039,
          0.228386233449, 0.221923179092, 0.215474977354, 0.209047542641,
          0.202646590024, 0.196277633269, 0.189945983241, 0.183656746711,
          0.177414825371, 0.171224915479, 0.165091507575, 0.159018886653,
          0.153011132637, 0.147072121151, 0.141205524598, 0.135414813519,
          0.129703258229, 0.124073930714, 0.118529706783, 0.113073268455,
          0.107707106577, 0.102433523653, 0.097254636883, 0.092172381387,
          0.087188513608, 0.082304614890, 0.077522095202, 0.072842197009,
          0.068265999280, 0.063794421603, 0.059428228425, 0.055168033374,
          0.051014303682, 0.046967364672, 0.043027404315, 0.039194477841,
          0.035468512397, 0.031849311736, 0.028336560933, 0.024929831117,
          0.021628584210, 0.018432177663, 0.015339869187, 0.012350821460,
          0.009464106811, 0.006678711873, 0.003993542194, 0.001407426800,
         -0.001080877289,-0.003472680608,-0.005769356857,-0.007972338583,
         -0.010083112932,-0.012103217485,-0.014034236175,-0.015877795293,
         -0.017635559582,-0.019309228431,-0.020900532156,-0.022411228396,
         -0.023843098593,-0.025197944596,-0.026477585353,-0.027683853723,
         -0.028818593393,-0.029883655903,-0.030880897783,-0.031812177804,
         -0.032679354333,-0.033484282807,-0.034228813314,-0.034914788284,
         -0.035544040295,-0.036118389978,-0.036639644043,-0.037109593368,
         -0.037530011368,-0.037902652131,-0.038229248990,-0.038511513028,
         -0.038751131720,-0.038949767669,-0.039109057435,-0.039230610461,
         -0.039316008087,-0.039366802653,-0.039384516689,-0.039370642184,
         -0.039326639941,-0.039253939009,-0.039153936188,-0.039027995612,
         -0.038877448393,-0.038703592350,-0.038507691783,-0.038290977323,
         -0.038054645839,-0.037799860397,-0.037527750284,-0.037239411070,
         -0.036935904732,-0.036618259817,-0.036287471656,-0.035944502607,
         -0.035590282354,-0.035225708231,-0.034851645582,-0.034468928160,
         -0.034078358546,-0.033680708602,-0.033276719950,-0.032867104470,
         -0.032452544821,-0.032033694986,-0.031611180823,-0.031185600648,
         -0.030757525815,-0.030327501320,-0.029896046409,-0.029463655198,
         -0.029030797297,-0.028597918444,-0.028165441141,-0.027733765290,
         -0.027303268837,-0.026874308413,-0.026447219972,-0.026022319432,
         -0.025599903313,-0.025180249365,-0.024763617198,-0.024350248907,
         -0.023940369683,-0.023534188425,-0.023131898342,-0.022733677544,
         -0.022339689625,-0.021950084244,-0.021564997680,-0.021184553396,
         -0.020808862576,-0.020438024658,-0.020072127859,-0.019711249681,
         -0.019355457408,-0.019004808592,-0.018659351527,-0.018319125708,
         -0.017984162280,-0.017654484470,-0.017330108015,-0.017011041567,
         -0.016697287096,-0.016388840267,-0.016085690819,-0.015787822923,
         -0.015495215525,-0.015207842688,-0.014925673910,-0.014648674439,
         -0.014376805566,-0.014110024919,-0.013848286738,-0.013591542134,
         -0.013339739353,-0.013092824008,-0.012850739321,-0.012613426340,
         -0.012380824123,-0.012152870052,-0.011929499863,-0.011710647936,
         -0.011496247440,-0.011286230502,-0.011080528362,-0.010879071520,
         -0.010681789878,-0.010488612873,-0.010299469600,-0.010114288929,
         -0.009932999618,-0.009755530417,-0.009581810161,-0.009411767864,
         -0.009245332805,-0.009082434602,-0.008923003292,-0.008766969393,
         -0.008614263971,-0.008464818695,-0.008318565892,-0.008175438598,
         -0.008035370597,-0.007898296464,-0.007764151606,-0.007632872286,
         -0.007504395421,-0.007378659557,-0.007255603473,-0.007135167138,
         -0.007017291500,-0.006901918498,-0.006788991072,-0.006678453177,
         -0.006570249787,-0.006464326900,-0.006360631544,-0.006259111777,
         -0.006159716688,-0.006062396393,-0.005967102034,-0.005873785773,
         -0.005782400788,-0.005692901265,-0.005605242388,-0.005519380334,
         -0.005435272260,-0.005352876292,-0.005272151518,-0.005193057969,
         -0.005115556612,-0.005039609333,-0.004965178925,-0.004892229074,
         -0.004820724345,-0.004750630162,-0.004681912801,-0.004614539369,
         -0.004548477789,-0.004483696787,-0.004420165872,-0.004357855325,
         -0.004296736179,-0.004236780206,-0.004177959899,-0.004120248458,
         -0.004063619772,-0.004008048406,-0.003953509583,-0.003899979171,
         -0.003847433665,-0.003795850176,-0.003745206410,-0.003695480658,
         -0.003646651783,-0.003598699199,-0.003551602863,-0.003505343259,
         -0.003459901384,-0.003415258734,-0.003371397294,-0.003328299521,
         -0.003285948336,-0.003244327106,-0.003203419637,-0.003163210160,
         -0.003123683318,-0.003084824158,-0.003046618118,-0.003009051013,
         -0.002972109033,-0.002935778724,-0.002900046982,-0.002864901043,
         -0.002830328472,-0.002796317156,-0.002762855295,-0.002729931387,
         -0.002697534230,-0.002665652905,-0.002634276769,-0.002603395454,
         -0.002572998849,-0.002543077100,-0.002513620601,-0.002484619985,
         -0.002456066121,-0.002427950100,-0.002400263239,-0.002372997065,
         -0.002346143316,-0.002319693929,-0.002293640920,-0.002267976851,
         -0.002242694116,-0.002217785407,-0.002193243589,-0.002169061700,
         -0.002145232941,-0.002121750674,-0.002098608417,-0.002075799841,
         -0.002053318764,-0.002031159126,-0.002009315067,-0.001987780805,
         -0.001966550708,-0.001945619272,-0.001924981119,-0.001904630990,
         -0.001884563747,-0.001864774365,-0.001845257931,-0.001826009640,
         -0.001807024795,-0.001788298800,-0.001769827160,-0.001751605476,
         -0.001733629448,-0.001715894863,-0.001698397604,-0.001681133636,
         -0.001664099014,-0.001647289874,-0.001630702434,-0.001614332989,
         -0.001598179129,-0.001582234872,-0.001566497956,-0.001550964973,
         -0.001535632585,-0.001520497524,-0.001505556585,-0.001490806630,
         -0.001476244583,-0.001461867430,-0.001447672214,-0.001433656041,
         -0.001419816070,-0.001406149518,-0.001392653656,-0.001379325806,
         -0.001366163345,-0.001353163699,-0.001340324341,-0.001327642797,
         -0.001315116635,-0.001302743474,-0.001290520974,-0.001278446841,
         -0.001266518822,-0.001254734709,-0.001243092331,-0.001231589562,
         -0.001220224309,-0.001208994523,-0.001197898190,-0.001186933332,
         -0.001176098007,-0.001165390310,-0.001154808368,-0.001144350341,
         -0.001134014426,-0.001123798847,-0.001113701861,-0.001103721759,
         -0.001093856857,-0.001084105504,-0.001074466076,-0.001064936978,
         -0.001055516643,-0.001046203530,-0.001036996125,-0.001027892940,
         -0.001018892513,-0.001009993405,-0.001001194204,-0.000992493520,
         -0.000983889986,-0.000975382261,-0.000966969022,-0.000958648972,
         -0.000950420834,-0.000942283351,-0.000934235288,-0.000926275431,
         -0.000918402585,-0.000910615574,-0.000902913243,-0.000895294453,
         -0.000887758087,-0.000880303042,-0.000872928236,-0.000865632604,
         -0.000858415095,-0.000851274680,-0.000844210340,-0.000837221079,
         -0.000830305911,-0.000823463869,-0.000816693999,-0.000809995364,
         -0.000803367040,-0.000796808118,-0.000790317704,-0.000783894917,
         -0.000777538888,-0.000771248764,-0.000765023705,-0.000758862881,
         -0.000752765478,-0.000746730693,-0.000740757735,-0.000734845825,
         -0.000728994196,-0.000723202092,-0.000717468770,-0.000711793497,
         -0.000706175549,-0.000700614217,-0.000695108799,-0.000689658606,
         -0.000684262956,-0.000678921179,-0.000673632543,-0.000668396541,
         -0.000663212461,-0.000658079671,-0.000652997548,-0.000647965479,
         -0.000642982859,-0.000638049094,-0.000633163594,-0.000628325781,
         -0.000623535085,-0.000618790943,-0.000614092801,-0.000609440112,
         -0.000604832337,-0.000600268946,-0.000595749415,-0.000591273227,
         -0.000586839873,-0.000582448852,-0.000578099668,-0.000573791833,
         -0.000569524867,-0.000565298294,-0.000561111648,-0.000556964465,
         -0.000552856292,-0.000548786679,-0.000544755183,-0.000540761368,
         -0.000536804803,-0.000532885063,-0.000529001730,-0.000525154389,
         -0.000521342633,-0.000517566059,-0.000513824271,-0.000510116877,
         -0.000506443491,-0.000502803731,-0.000499197222,-0.000495623592,
         -0.000492082475,-0.000488573510,-0.000485096341,-0.000481650614,
         -0.000478235985,-0.000474852108,-0.000471498648,-0.000468175269,
         -0.000464881625,-0.000461617425,-0.000458382332,-0.000455176027,
         -0.000451998200,-0.000448848541,-0.000445726746,-0.000442632514,
         -0.000439565547,-0.000436525555,-0.000433512245,-0.000430525334,
         -0.000427564539,-0.000424629581,-0.000421720186,-0.000418836083,
         -0.000415977003,-0.000413142681,-0.000410332856,-0.000407547271,
         -0.000404785670,-0.000402047802,-0.000399333418,-0.000396642273,
         -0.000393974124,-0.000391328732,-0.000388705860,-0.000386105276,
         -0.000383526747,-0.000380970048,-0.000378434951,-0.000375921236,
         -0.000373428682,-0.000370957073,-0.000368506194,-0.000366075834,
         -0.000363665783,-0.000361275835,-0.000358905786,-0.000356555434,
         -0.000354224580,-0.000351913028,-0.000349620581,-0.000347347050,
         -0.000345092242,-0.000342855972,-0.000340638053,-0.000338438303,
         -0.000336256539,-0.000334092584,-0.000331946261,-0.000329817395,
         -0.000327705812,-0.000325611343,-0.000323533819,-0.000321473073,
         -0.000319428941,-0.000317401259,-0.000315389867,-0.000313394605,
         -0.000311415316,-0.000309451846,-0.000307504039,-0.000305571745,
         -0.000303654813,-0.000301753094,-0.000299866443,-0.000297994713,
         -0.000296137762,-0.000294295448,-0.000292467630,-0.000290654171,
         -0.000288854932,-0.000287069779,-0.000285298577,-0.000283541195,
         -0.000281797501,-0.000280067365,-0.000278350660,-0.000276647259,
         -0.000274957037,-0.000273279870,-0.000271615635,-0.000269964212,
         -0.000268325481,-0.000266699323,-0.000265085621,-0.000263484260,
         -0.000261895124,-0.000260318101,-0.000258753078,-0.000257199946,
         -0.000255658593,-0.000254128911,-0.000252610794,-0.000251104135,
         -0.000249610990,-0.000248126920,-0.000246653996,-0.000245192119,
         -0.000243741185,-0.000242301097,-0.000240871755,-0.000239453063,
         -0.000238044925,-0.000236647244,-0.000235259927,-0.000233882881,
         -0.000232516014,-0.000231159234,-0.000229812450,-0.000228475574,
         -0.000227148518,-0.000225831193,-0.000224523513,-0.000223225393,
         -0.000221936748,-0.000220657494,-0.000219387548,-0.000218126828,
         -0.000216875252,-0.000215632741,-0.000214399215,-0.000213174594,
         -0.000211958802,-0.000210751761,-0.000209553394,-0.000208363626,
         -0.000207182383,-0.000206009589,-0.000204845173,-0.000203689061,
         -0.000202541182,-0.000201401464,-0.000200269790,-0.000199146185,
         -0.000198030532,-0.000196922764,-0.000195822813,-0.000194730613,
         -0.000193646097,-0.000192569199,-0.000191499856,-0.000190438001,
         -0.000189383574,-0.000188336509,-0.000187296746,-0.000186264221,
         -0.000185238875,-0.000184220647,-0.000183209477,-0.000182205306,
         -0.000181208074,-0.000180217725,-0.000179234200,-0.000178257443,
         -0.000177287397,-0.000176324006,-0.000175367216,-0.000174416971,
         -0.000173473217,-0.000172535901,-0.000171604968,-0.000170680368,
         -0.000169762047,-0.000168849954,-0.000167944038,-0.000167044248,
         -0.000166150534,-0.000165262847,-0.000164381136,-0.000163505354,
         -0.000162635453,-0.000161771383,-0.000160913100,-0.000160060554,
         -0.000159213700,-0.000158372493,-0.000157536885,-0.000156706833,
         -0.000155882291,-0.000155063216,-0.000154249563,-0.000153441289,
         -0.000152638351,-0.000151840707,-0.000151048314,-0.000150261130,
         -0.000149479114,-0.000148702224,-0.000147930421,-0.000147163665,
         -0.000146401914,-0.000145645130,-0.000144893273,-0.000144146305,
         -0.000143404187,-0.000142666880,-0.000141934348,-0.000141206553,
         -0.000140483459,-0.000139765027,-0.000139051222,-0.000138342008,
         -0.000137637349,-0.000136937210,-0.000136241554,-0.000135550350,
         -0.000134863561,-0.000134181153,-0.000133503093,-0.000132829347,
         -0.000132159882,-0.000131494665,-0.000130833663,-0.000130176845,
         -0.000129524178,-0.000128875630,-0.000128231170,-0.000127590768,
         -0.000126954392,-0.000126322010,-0.000125693594,-0.000125069114,
         -0.000124448538,-0.000123831839,-0.000123218986,-0.000122609950,
         -0.000122004703,-0.000121403217,-0.000120805462,-0.000120211398,
         -0.000119621024,-0.000119034299,-0.000118451195,-0.000117871686,
         -0.000117295744,-0.000116723344,-0.000116154459,-0.000115589063,
         -0.000115027129,-0.000114468633,-0.000113913550,-0.000113361853,
         -0.000112813518,-0.000112268520,-0.000111726834,-0.000111188437,
         -0.000110653305,-0.000110121412,-0.000109592736,-0.000109067253,
         -0.000108544940,-0.000108025773,-0.000107509731,-0.000106996789,
         -0.000106486925,-0.000105980118,-0.000105476346,-0.000104975585,
         -0.000104477815,-0.000103983013,-0.000103491159,-0.000103002232,
         -0.000102516209,-0.000102033071,-0.000101552797,-0.000101075366,
         -0.000100600758,-0.000100128953,-0.000099659931,-0.000099193670,
         -0.000098730154,-0.000098269362,-0.000097811273,-0.000097355870,
         -0.000096903134,-0.000096453045,-0.000096005584,-0.000095560734,
         -0.000095118477,-0.000094678791,-0.000094241663,-0.000093807071,
         -0.000093375000,-0.000092945431,-0.000092518346,-0.000092093728,
         -0.000091671561,-0.000091251828,-0.000090834509,-0.000090419590,
         -0.000090007054,-0.000089596884,-0.000089189066,-0.000088783579,
         -0.000088380409,-0.000087979542,-0.000087580961,-0.000087184648,
         -0.000086790590,-0.000086398772,-0.000086009176,-0.000085621788,
         -0.000085236594,-0.000084853579,-0.000084472726,-0.000084094021,
         -0.000083717451,-0.000083343000,-0.000082970655,-0.000082600399,
         -0.000082232220,-0.000081866104,-0.000081502037,-0.000081140003,
         -0.000080779991,-0.000080421987,-0.000080065976,-0.000079711946,
         -0.000079359883,-0.000079009773,-0.000078661606,-0.000078315366,
         -0.000077971040,-0.000077628617,-0.000077288085,-0.000076949428,
         -0.000076612636,-0.000076277697,-0.000075944598,-0.000075613325,
         -0.000075283869,-0.000074956216,-0.000074630355,-0.000074306274,
         -0.000073983961,-0.000073663404,-0.000073344592,-0.000073027514,
         -0.000072712158,-0.000072398512,-0.000072086567,-0.000071776309,
         -0.000071467731,-0.000071160818,-0.000070855559,-0.000070551947,
         -0.000070249969,-0.000069949614,-0.000069650873,-0.000069353734,
         -0.000069058188,-0.000068764223,-0.000068471830,-0.000068181000,
         -0.000067891719,-0.000067603982,-0.000067317776,-0.000067033091,
         -0.000066749920,-0.000066468249,-0.000066188072,-0.000065909379,
         -0.000065632160,-0.000065356404,-0.000065082104,-0.000064809249,
         -0.000064537830,-0.000064267841,-0.000063999270,-0.000063732106,
         -0.000063466345,-0.000063201975,-0.000062938987,-0.000062677374,
         -0.000062417127,-0.000062158236,-0.000061900694,-0.000061644492,
         -0.000061389621,-0.000061136073,-0.000060883842,-0.000060632915,
         -0.000060383289,-0.000060134953,-0.000059887864,-0.000059642083,
         -0.000059397570,-0.000059154315,-0.000058912309,-0.000058671548,
         -0.000058432022,-0.000058193723,-0.000057956645,-0.000057720779,
         -0.000057486117,-0.000057252655,-0.000057020383,-0.000056789292,
         -0.000056559380,-0.000056330634,-0.000056103051,-0.000055876620,
         -0.000055651339,-0.000055427196,-0.000055204187,-0.000054982304,
         -0.000054761541,-0.000054541891,-0.000054323347,-0.000054105902,
         -0.000053889548,-0.000053674282,-0.000053460095,-0.000053246980,
         -0.000053034932,-0.000052823945,-0.000052614010,-0.000052405123,
         -0.000052197277,-0.000051990464,-0.000051784681,-0.000051579921,
         -0.000051376175,-0.000051173441,-0.000050971711,-0.000050770978,
         -0.000050571237,-0.000050372484,-0.000050174711,-0.000049977910,
         -0.000049782080,-0.000049587212,-0.000049393303,-0.000049200344,
         -0.000049008332,-0.000048820422,-0.000048630261,-0.000048441030,
         -0.000048252722,-0.000048065332,-0.000047878856,-0.000047693287,
         -0.000047508621,-0.000047324852,-0.000047141973,-0.000046959983,
         -0.000046778874,-0.000046598641,-0.000046419280,-0.000046240785,
         -0.000046063150,-0.000045886371,-0.000045710445,-0.000045535364,
         -0.000045361124,-0.000045187721,-0.000045015150,-0.000044843405,
         -0.000044672483,-0.000044502378,-0.000044333084,-0.000044164600,
         -0.000043996919,-0.000043830035,-0.000043663946,-0.000043498647,
         -0.000043334133,-0.000043170399,-0.000043007443,-0.000042845255,
         -0.000042683836,-0.000042523182,-0.000042363284,-0.000042204141,
         -0.000042045748,-0.000041888100,-0.000041731194,-0.000041575025,
         -0.000041419591,-0.000041264882,-0.000041110901,-0.000040957641,
         -0.000040805096,-0.000040653264,-0.000040502141,-0.000040351722,
         -0.000040202005,-0.000040052984,-0.000039904655,-0.000039757016,
         -0.000039610063,-0.000039463789,-0.000039318195,-0.000039173273,
         -0.000039029021,-0.000038885436,-0.000038742515,-0.000038600250,
         -0.000038458642,-0.000038317685,-0.000038177376,-0.000038037711,
         -0.000037898687,-0.000037760301,-0.000037622548,-0.000037485426,
         -0.000037348930,-0.000037213058,-0.000037077806,-0.000036943170,
         -0.000036809146,-0.000036675734,-0.000036542929,-0.000036410724,
         -0.000036279120,-0.000036148115,-0.000036017700,-0.000035887877,
         -0.000035758640,-0.000035629986,-0.000035501914,-0.000035374420,
         -0.000035247496,-0.000035121149,-0.000034995368,-0.000034870150,
         -0.000034745493,-0.000034621399,-0.000034497857,-0.000034374867,
         -0.000034252431,-0.000034130540,-0.000034009195,-0.000033888389,
         -0.000033768121,-0.000033648386,-0.000033529187,-0.000033410517,
         -0.000033292372,-0.000033174751,-0.000033057655,-0.000032941073,
         -0.000032825009,-0.000032709445,-0.000032594403,-0.000032479866,
         -0.000032365835,-0.000032252309,-0.000032139280,-0.000032026745,
         -0.000031914707,-0.000031803162,-0.000031692103,-0.000031581532,
         -0.000031471444,-0.000031361837,-0.000031252709,-0.000031144055,
         -0.000031035877,-0.000030928170,-0.000030820929,-0.000030714158,
         -0.000030607851,-0.000030502003,-0.000030396614,-0.000030291683,
         -0.000030187205,-0.000030083179,-0.000029979603,-0.000029876472,
         -0.000029773788,-0.000029671547,-0.000029569744,-0.000029468381,
         -0.000029367450,-0.000029266957,-0.000029166892,-0.000029067256,
         -0.000028968050,-0.000028869268,-0.000028770907,-0.000028672964,
         -0.000028575442,-0.000028478335,-0.000028381639,-0.000028285359,
         -0.000028189486,-0.000028094022,-0.000027998959,-0.000027904304,
         -0.000027810049,-0.000027716193,-0.000027622734,-0.000027529671,
         -0.000027436999,-0.000027344717,-0.000027252827,-0.000027161322,
         -0.000027070206,-0.000026979468,-0.000026889117,-0.000026799141,
         -0.000026709543,-0.000026620320,-0.000026531473,-0.000026442997,
         -0.000026354890,-0.000026267149,-0.000026179778,-0.000026092771,
         -0.000026006124,-0.000025919839,-0.000025833911,-0.000025748342,
         -0.000025663130,-0.000025578271,-0.000025493760,-0.000025409601,
         -0.000025325792,-0.000025242328,-0.000025159208,-0.000025076433,
         -0.000024993995,-0.000024911902,-0.000024830144,-0.000024748726,
         -0.000024667636,-0.000024586884,-0.000024506464,-0.000024426371,
         -0.000024346609,-0.000024267170,-0.000024188056,-0.000024109265,
         -0.000024030799,-0.000023952651,-0.000023874822,-0.000023797310,
         -0.000023720113,-0.000023643230,-0.000023566661,-0.000023490400,
         -0.000023414450,-0.000023338806,-0.000023263470,-0.000023188440,
         -0.000023113711,-0.000023039284,-0.000022965158,-0.000022891329,
         -0.000022817800,-0.000022744566,-0.000022671628,-0.000022598981,
         -0.000022526626,-0.000022454564,-0.000022382789,-0.000022311301,
         -0.000022240098,-0.000022169181,-0.000022098547,-0.000022028196,
         -0.000021958126,-0.000021888335,-0.000021818823,-0.000021749584,
         -0.000021680622,-0.000021611936,-0.000021543520,-0.000021475377,
         -0.000021407504,-0.000021339898,-0.000021272563,-0.000021205493,
         -0.000021138686,-0.000021072146,-0.000021005867,-0.000020939846,
         -0.000020874089,-0.000020808589,-0.000020743350,-0.000020678365,
         -0.000020613631,-0.000020549154,-0.000020484934,-0.000020420957,
         -0.000020357237,-0.000020293765,-0.000020230538,-0.000020167562,
         -0.000020104832,-0.000020042343,-0.000019980097,-0.000019918094,
         -0.000019856335,-0.000019794814,-0.000019733531,-0.000019672485,
         -0.000019611678,-0.000019551105,-0.000019490768,-0.000019430661,
         -0.000019370790,-0.000019311147,-0.000019251735,-0.000019192557,
         -0.000019133600,-0.000019074873,-0.000019016371,-0.000018958095,
         -0.000018900039,-0.000018842210,-0.000018784603,-0.000018727213,
         -0.000018670047,-0.000018613097,-0.000018556365,-0.000018499848,
         -0.000018443548,-0.000018387465,-0.000018331591,-0.000018275933,
         -0.000018220484,-0.000018165250,-0.000018110224,-0.000018055409,
         -0.000018000797,-0.000017946368,-0.000017892173,-0.000017838178,
         -0.000017784394,-0.000017730808,-0.000017677427,-0.000017624243,
         -0.000017571262,-0.000017518482,-0.000017465898,-0.000017413516,
         -0.000017361325,-0.000017309333,-0.000017257536,-0.000017205934,
         -0.000017154522,-0.000017103307,-0.000017052279,-0.000017001446,
         -0.000016950802,-0.000016900344,-0.000016850076,-0.000016799996,
         -0.000016750100,-0.000016700391,-0.000016650869,-0.000016601528,
         -0.000016552372,-0.000016503396,-0.000016454605,-0.000016405991,
         -0.000016357560,-0.000016309309,-0.000016261231,-0.000016213333,
         -0.000016165615,-0.000016118065,-0.000016070700,-0.000016023506,
         -0.000015976480,-0.000015929631,-0.000015882957,-0.000015836450,
         -0.000015790114,-0.000015743950,-0.000015697953,-0.000015652126,
         -0.000015606467,-0.000015560973,-0.000015515645,-0.000015470483,
         -0.000015425484,-0.000015380653,-0.000015335981,-0.000015291478,
         -0.000015247131,-0.000015202943,-0.000015158921,-0.000015115055,
         -0.000015071350,-0.000015027801,-0.000014984411,-0.000014941180,
         -0.000014898105,-0.000014855180,-0.000014812414,-0.000014769805,
         -0.000014727347,-0.000014685040,-0.000014642884,-0.000014600881,
         -0.000014559031,-0.000014517328,-0.000014475782,-0.000014434379,
         -0.000014393121,-0.000014352019,-0.000014311058,-0.000014270245,
         -0.000014229578,-0.000014189055,-0.000014148680,-0.000014108446,
         -0.000014068353,-0.000014028407,-0.000013988601,-0.000013948938,
         -0.000013909417,-0.000013870032,-0.000013830791,-0.000013791682,
         -0.000013752719,-0.000013713888,-0.000013675201,-0.000013636646,
         -0.000013598224,-0.000013559943,-0.000013521794,-0.000013483787,
         -0.000013445904,-0.000013408159,-0.000013370545,-0.000013333065,
         -0.000013295714,-0.000013258497,-0.000013221409,-0.000013184449,
         -0.000013147625,-0.000013110920,-0.000013074350,-0.000013037908,
         -0.000013001586,-0.000012965401,-0.000012929339,-0.000012893399,
         -0.000012857583,-0.000012821894,-0.000012786331,-0.000012750890,
         -0.000012715572,-0.000012680376,-0.000012645302,-0.000012610349,
         -0.000012575520,-0.000012540810,-0.000012506216,-0.000012471746,
         -0.000012437394,-0.000012403161,-0.000012369047,-0.000012335050,
         -0.000012301167,-0.000012267404,-0.000012233750,-0.000012200220,
         -0.000012166803,-0.000012133497,-0.000012100313,-0.000012067236,
         -0.000012034272,-0.000012001421,-0.000011968683,-0.000011936060,
         -0.000011903547,-0.000011871144,-0.000011838848,-0.000011806664,
         -0.000011774589,-0.000011742629,-0.000011710771,-0.000011679025,
         -0.000011647383,-0.000011615852,-0.000011584427,-0.000011553112,
         -0.000011521895,-0.000011490790,-0.000011459786,-0.000011428886,
         -0.000011398096,-0.000011367410,-0.000011336821,-0.000011306340,
         -0.000011275960,-0.000011245680,-0.000011215503,-0.000011185429,
         -0.000011155458,-0.000011125582,-0.000011095810,-0.000011066134,
         -0.000011036565,-0.000011007086,-0.000010977713,-0.000010948432,
         -0.000010919250,-0.000010890170,-0.000010861182,-0.000010832294,
         -0.000010803504,-0.000010774803,-0.000010746203,-0.000010717697,
         -0.000010689281,-0.000010660963,-0.000010632742,-0.000010604605,
         -0.000010576575,-0.000010548631,-0.000010520774,-0.000010493012,
         -0.000010465346,-0.000010437768,-0.000010410283,-0.000010382885,
         -0.000010355578,-0.000010328369,-0.000010301239,-0.000010274208,
         -0.000010251560,-0.000010224675,-0.000010197872,-0.000010171157,
         -0.000010144533,-0.000010117994,-0.000010091545,-0.000010065176,
         -0.000010038899,-0.000010012705,-0.000009986596,-0.000009960573,
         -0.000009934633,-0.000009908783,-0.000009883016,-0.000009857331,
         -0.000009831723,-0.000009806208,-0.000009780772,-0.000009755416,
         -0.000009730149,-0.000009704965,-0.000009679854,-0.000009654825,
         -0.000009629881,-0.000009605015,-0.000009580232,-0.000009555527,
         -0.000009530906,-0.000009506359,-0.000009481894,-0.000009457507,
         -0.000009433201,-0.000009408968,-0.000009384813,-0.000009360741,
         -0.000009336743,-0.000009312822,-0.000009288983,-0.000009265219,
         -0.000009241523,-0.000009217903,-0.000009194367,-0.000009170904,
         -0.000009147513,-0.000009124185,-0.000009100951,-0.000009077784,
         -0.000009054691,-0.000009031668,-0.000009008727,-0.000008985847,
         -0.000008963044,-0.000008940323,-0.000008917665,-0.000008895083,
         -0.000008872572,-0.000008850132,-0.000008827756,-0.000008805460,
         -0.000008783231,-0.000008761077,-0.000008738982,-0.000008716967,
         -0.000008695016,-0.000008673142,-0.000008651317,-0.000008629577,
         -0.000008607915,-0.000008586302,-0.000008564769,-0.000008543299,
         -0.000008521889,-0.000008500552,-0.000008479283,-0.000008458087,
         -0.000008436951,-0.000008415879,-0.000008394877,-0.000008373949,
         -0.000008353056,-0.000008332256,-0.000008311510,-0.000008290834,
         -0.000008270228,-0.000008249672,-0.000008229185,-0.000008208760,
         -0.000008188409,-0.000008168114,-0.000008147887,-0.000008127708,
         -0.000008107605,-0.000008087561,-0.000008067576,-0.000008047655,
         -0.000008027797,-0.000008007992,-0.000007988257,-0.000007968579,
         -0.000007948961,-0.000007929423,-0.000007909923,-0.000007890483,
         -0.000007871118,-0.000007851794,-0.000007832545,-0.000007813350,
         -0.000007794215,-0.000007775134,-0.000007756107,-0.000007737143,
         -0.000007718239,-0.000007699398,-0.000007680602,-0.000007661878,
         -0.000007643192,-0.000007624581,-0.000007606023,-0.000007587516,
         -0.000007569072,-0.000007550684,-0.000007532342,-0.000007514064,
         -0.000007495842,-0.000007477669,-0.000007459564,-0.000007441513,
         -0.000007423500,-0.000007405565,-0.000007387653,-0.000007369819,
         -0.000007352033,-0.000007334302,-0.000007316624,-0.000007298991,
         -0.000007281419,-0.000007263898,-0.000007246430,-0.000007229017,
         -0.000007211665,-0.000007194341,-0.000007177091,-0.000007159891,
         -0.000007142723,-0.000007125624,-0.000007108576,-0.000007091568,
         -0.000007074623,-0.000007057725,-0.000007040876,-0.000007024086,
         -0.000007007336,-0.000006990642,-0.000006973998,-0.000006957391,
         -0.000006940839,-0.000006924342,-0.000006907899,-0.000006891498,
         -0.000006875140,-0.000006858837,-0.000006842582,-0.000006826375,
         -0.000006810219,-0.000006794105,-0.000006778029,-0.000006762023,
         -0.000006746059,-0.000006730128,-0.000006714271,-0.000006698432,
         -0.000006682660,-0.000006666918,-0.000006651234,-0.000006635596,
         -0.000006620005,-0.000006604454,-0.000006588941,-0.000006573490,
         -0.000006558070,-0.000006542702,-0.000006527380,-0.000006512109,
         -0.000006496882,-0.000006481687,-0.000006466546,-0.000006451435,
         -0.000006436391,-0.000006421375,-0.000006406417,-0.000006391483,
         -0.000006376616,-0.000006361774,-0.000006346978,-0.000006332234,
         -0.000006317521,-0.000006302863,-0.000006288236,-0.000006273658,
         -0.000006259130,-0.000006244639,-0.000006230172,-0.000006215769,
         -0.000006201406,-0.000006187056,-0.000006172780,-0.000006158538,
         -0.000006144333,-0.000006130175,-0.000006116049,-0.000006101973,
         -0.000006087918,-0.000006073934,-0.000006059967,-0.000006046060,
         -0.000006032157,-0.000006018329,-0.000006004539,-0.000005990767,
         -0.000005977053,-0.000005963368,-0.000005949736,-0.000005936125,
         -0.000005922564,-0.000005909033,-0.000005895557,-0.000005882108,
         -0.000005868694,-0.000005855330,-0.000005841986,-0.000005828695,
         -0.000005815440,-0.000005802219,-0.000005789042,-0.000005775894,
         -0.000005762793,-0.000005749713,-0.000005736681,-0.000005723693,
         -0.000005710736,-0.000005697801,-0.000005684917,-0.000005672083,
         -0.000005659257,-0.000005646489,-0.000005633742,-0.000005621052,
         -0.000005608376,-0.000005595751,-0.000005583147,-0.000005570596,
         -0.000005558069,-0.000005545574,-0.000005533131,-0.000005520686,
         -0.000005508306,-0.000005495958,-0.000005483642,-0.000005471368,
         -0.000005459117,-0.000005446911,-0.000005434723,-0.000005422582,
         -0.000005410465,-0.000005398405,-0.000005386318,-0.000005374325,
         -0.000005362350,-0.000005350395,-0.000005338497,-0.000005326630,
         -0.000005314785,-0.000005302970,-0.000005291186,-0.000005279453,
         -0.000005267752,-0.000005256061,-0.000005244408,-0.000005232808,
         -0.000005221217,-0.000005209672,-0.000005198143,-0.000005186671,
         -0.000005175216,-0.000005163788,-0.000005152404,-0.000005141034,
         -0.000005129712,-0.000005118411,-0.000005107148,-0.000005095903,
         -0.000005084703,-0.000005073544,-0.000005062386,-0.000005051294,
         -0.000005040207,-0.000005029153,-0.000005018130,-0.000005007138,
         -0.000004996188,-0.000004985255,-0.000004974357,-0.000004963482,
         -0.000004952658,-0.000004941822,-0.000004931049,-0.000004920300,
         -0.000004909585,-0.000004898883,-0.000004888218,-0.000004877591,
         -0.000004866978,-0.000004856408,-0.000004845851,-0.000004835339,
         -0.000004824820,-0.000004814384,-0.000004803951,-0.000004793527,
         -0.000004783167,-0.000004772816,-0.000004762493,-0.000004752199,
         -0.000004741928,-0.000004731692,-0.000004721489,-0.000004711313,
         -0.000004701153,-0.000004691028,-0.000004680906,-0.000004670841,
         -0.000004660800,-0.000004650784,-0.000004640794,-0.000004630823,
         -0.000004620897,-0.000004610976,-0.000004601088,-0.000004591223,
         -0.000004581397,-0.000004571577,-0.000004561807,-0.000004552078,
         -0.000004542317,-0.000004532634,-0.000004522956,-0.000004513308,
         -0.000004503683,-0.000004494084,-0.000004484512,-0.000004474978,
         -0.000004465445,-0.000004455950,-0.000004446495,-0.000004437029,
         -0.000004427618,-0.000004418215,-0.000004408848,-0.000004399505,
         -0.000004390179,-0.000004380881,-0.000004371601,-0.000004362365,
         -0.000004353139,-0.000004343950,-0.000004334749,-0.000004325623,
         -0.000004316509,-0.000004307391,-0.000004298319,-0.000004289266,
         -0.000004280235,-0.000004271233,-0.000004262246,-0.000004253291,
         -0.000004244373,-0.000004235455,-0.000004226566,-0.000004217706,
         -0.000004208841,-0.000004200037,-0.000004191235,-0.000004182468,
         -0.000004173720,-0.000004164993,-0.000004156290,-0.000004147604,
         -0.000004138957,-0.000004130317,-0.000004121718,-0.000004113097,
         -0.000004104542,-0.000004096010,-0.000004087470,-0.000004078981,
         -0.000004070497,-0.000004062048,-0.000004053610,-0.000004045201,
         -0.000004036807,-0.000004028443,-0.000004020089,-0.000004011762,
         -0.000004003482,-0.000003995175,-0.000003986913,-0.000003978681,
         -0.000003970453,-0.000003962262,-0.000003954084,-0.000003945935,
         -0.000003937788,-0.000003929692,-0.000003921595,-0.000003913524,
         -0.000003905457,-0.000003897431,-0.000003889449,-0.000003881430,
         -0.000003873473,-0.000003865523,-0.000003857602,-0.000003849695,
         -0.000003841812,-0.000003833951,-0.000003826106,-0.000003818275,
         -0.000003810468,-0.000003802692,-0.000003794902,-0.000003787173,
         -0.000003779443,-0.000003771744,-0.000003764045,-0.000003756378,
         -0.000003748734,-0.000003741099,-0.000003733489,-0.000003725902,
         -0.000003718344,-0.000003710764,-0.000003703250,-0.000003695750,
         -0.000003688226,-0.000003680780,-0.000003673310,-0.000003665875,
         -0.000003658465,-0.000003651067,-0.000003643680,-0.000003636336,
         -0.000003628987,-0.000003621648,-0.000003614371,-0.000003607042,
         -0.000003599790,-0.000003592538,-0.000003585317,-0.000003578103])

    # Voigt function by power law expansion in a, a >= 0.0
    # Taken from Harris, 1948, ApJ, 108, 112
    # Relative error with respect to voigt_de is < 10^-4.9 for a < 0.1,
    # gets worse for larger a (since it's an expansion in a)
    # Doesn't test for a == 0.0 because it's rare and it'll get it right anyway
    
    a = float(a)
    if not 0 <= a <= 0.1:
        raise ValueError('a must satisfy 0 <= a <= 0.1 (%f)' % a)
    u = np.abs(u)
    out = np.empty_like(u)
    u2 = u*u

    cond = u > 19.99

    if cond.any():
        # Use asymptotic approximation.
        iu2c = 1. / u2[cond]
        iu2c2 = iu2c * iu2c
        iu2c3 = iu2c2 * iu2c
        iu2c4 = iu2c3 * iu2c
        a2 = a**2
        k2 = 1.5 + a2
        k3 = 3.75 + 5 * a2
        k4 = 26.25 * a2
        out[cond] = a / sqrtpi * (iu2c + k2 * iu2c2 + k3 * iu2c3 + k4 * iu2c4)

    # for u values with abs(u) <= 19.99 use lookup tables
    notcond = ~cond
    u = u[notcond]
    u2 = u2[notcond]
    expmu2 = np.exp(-u2)

    out[notcond] = expmu2 + a*(np.interp(u, uvals, H1) + a*(
        (1. - 2.*u2)*expmu2  + a*(np.interp(u, uvals, H3) + a*(
        0.5 - 2.*u2 + 2./3.*u2*u2)*expmu2)))
    return out

# def _time_voigt():
#     import timeit
#     setup = """import numpy as np
# from numarray.convolve.lineshape import voigt as voigt2
# from voigt import voigt,voigtslow
# uvals = np.linspace(-25, 25, 100)
# a = 1e-3
# """
#     time1 = "h1 = voigt(a, uvals)"
#     time2 = "h2 = voigt2(a, uvals)"
#     time3 = "h3 = [voigtslow(a,u) for u in uvals]"
#     t1 = timeit.Timer(time1,setup)
#     t2 = timeit.Timer(time2,setup)
#     t3 = timeit.Timer(time3,setup)
        
#==================================================================================================================

def p_voigt(n,b,wave,lambdao,gamma,f):
    
    c       = 299792458.
    b       = (b+0) * 1000
    nn      = float(10**n)
    u       = (c/b)*(lambdao/wave-1)
    wave    = wave*(10**(-10))
    lambdao = lambdao*10**(-10)
    a       = lambdao*gamma/(4*pi*b)
    H       = voigt(a,u)
    tau     = 0
    wave    = wave*10**(10)    # in Angstrom
    b       = b/1000           # in km/s
    tau     = 1.497*10**(-15)*nn*f*wave*H/b
    lambdao = lambdao*10**(10)
    
    return np.exp(-tau)
    
#==================================================================================================================
