"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ssm = require("./ssm.generated");
const util_1 = require("./util");
/**
 * Basic features shared across all types of SSM Parameters.
 */
class ParameterBase extends core_1.Resource {
    grantRead(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'ssm:DescribeParameters',
                'ssm:GetParameters',
                'ssm:GetParameter',
                'ssm:GetParameterHistory',
            ],
            resourceArns: [this.parameterArn],
        });
    }
    grantWrite(grantee) {
        if (this.encryptionKey) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['ssm:PutParameter'],
            resourceArns: [this.parameterArn],
        });
    }
}
/**
 * SSM parameter type
 */
var ParameterType;
(function (ParameterType) {
    /**
     * String
     */
    ParameterType["STRING"] = "String";
    /**
     * Secure String
     * Parameter Store uses an AWS Key Management Service (KMS) customer master key (CMK) to encrypt the parameter value.
     */
    ParameterType["SECURE_STRING"] = "SecureString";
    /**
     * String List
     */
    ParameterType["STRING_LIST"] = "StringList";
    /**
     * An Amazon EC2 image ID, such as ami-0ff8a91507f77f867
     */
    ParameterType["AWS_EC2_IMAGE_ID"] = "AWS::EC2::Image::Id";
})(ParameterType = exports.ParameterType || (exports.ParameterType = {}));
/**
 * SSM parameter tier
 */
var ParameterTier;
(function (ParameterTier) {
    /**
     * String
     */
    ParameterTier["ADVANCED"] = "Advanced";
    /**
     * String
     */
    ParameterTier["INTELLIGENT_TIERING"] = "Intelligent-Tiering";
    /**
     * String
     */
    ParameterTier["STANDARD"] = "Standard";
})(ParameterTier = exports.ParameterTier || (exports.ParameterTier = {}));
/**
 * Creates a new String SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.allowedPattern) {
            _assertValidValue(props.stringValue, props.allowedPattern);
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: props.type || ParameterType.STRING,
            value: props.stringValue,
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringValue = resource.attrValue;
    }
    /**
     * Imports an external string parameter by name.
     */
    static fromStringParameterName(scope, id, stringParameterName) {
        return this.fromStringParameterAttributes(scope, id, { parameterName: stringParameterName });
    }
    /**
     * Imports an external string parameter with name and optional version.
     */
    static fromStringParameterAttributes(scope, id, attrs) {
        if (!attrs.parameterName) {
            throw new Error('parameterName cannot be an empty string');
        }
        const type = attrs.type || ParameterType.STRING;
        const stringValue = attrs.version
            ? new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, `${attrs.parameterName}:${attrs.version}`).toString()
            : new core_1.CfnParameter(scope, `${id}.Parameter`, { type: `AWS::SSM::Parameter::Value<${type}>`, default: attrs.parameterName }).valueAsString;
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = type;
                this.stringValue = stringValue;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a secure string parameter from the SSM parameter store.
     */
    static fromSecureStringParameterAttributes(scope, id, attrs) {
        const stringValue = new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM_SECURE, `${attrs.parameterName}:${attrs.version}`).toString();
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = attrs.parameterName;
                this.parameterArn = util_1.arnForParameterName(this, attrs.parameterName, { simpleName: attrs.simpleName });
                this.parameterType = ParameterType.SECURE_STRING;
                this.stringValue = stringValue;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Reads the value of an SSM parameter during synthesis through an
     * environmental context provider.
     *
     * Requires that the stack this scope is defined in will have explicit
     * account/region information. Otherwise, it will fail during synthesis.
     */
    static valueFromLookup(scope, parameterName) {
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.SSM_PARAMETER_PROVIDER,
            props: { parameterName },
            dummyValue: `dummy-value-for-${parameterName}`,
        }).value;
        return value;
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForStringParameter(scope, parameterName, version) {
        return StringParameter.valueForTypedStringParameter(scope, parameterName, ParameterType.STRING, version);
    }
    /**
     * Returns a token that will resolve (during deployment) to the string value of an SSM string parameter.
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter.
     * @param type The type of the SSM parameter.
     * @param version The parameter version (recommended in order to ensure that the value won't change during deployment)
     */
    static valueForTypedStringParameter(scope, parameterName, type = ParameterType.STRING, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromStringParameterAttributes(stack, id, { parameterName, version, type }).stringValue;
    }
    /**
     * Returns a token that will resolve (during deployment)
     * @param scope Some scope within a stack
     * @param parameterName The name of the SSM parameter
     * @param version The parameter version (required for secure strings)
     */
    static valueForSecureStringParameter(scope, parameterName, version) {
        const stack = core_1.Stack.of(scope);
        const id = makeIdentityForImportedValue(parameterName);
        const exists = stack.node.tryFindChild(id);
        if (exists) {
            return exists.stringValue;
        }
        return this.fromSecureStringParameterAttributes(stack, id, { parameterName, version }).stringValue;
    }
}
exports.StringParameter = StringParameter;
/**
 * Creates a new StringList SSM Parameter.
 * @resource AWS::SSM::Parameter
 */
class StringListParameter extends ParameterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.parameterName,
        });
        if (props.stringListValue.find(str => !core_1.Token.isUnresolved(str) && str.indexOf(',') !== -1)) {
            throw new Error('Values of a StringList SSM Parameter cannot contain the \',\' character. Use a string parameter instead.');
        }
        if (props.allowedPattern && !core_1.Token.isUnresolved(props.stringListValue)) {
            props.stringListValue.forEach(str => _assertValidValue(str, props.allowedPattern));
        }
        if (this.physicalName.length > 2048) {
            throw new Error('Name cannot be longer than 2048 characters.');
        }
        if (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 1024) {
            throw new Error('Description cannot be longer than 1024 characters.');
        }
        const resource = new ssm.CfnParameter(this, 'Resource', {
            allowedPattern: props.allowedPattern,
            description: props.description,
            name: this.physicalName,
            tier: props.tier,
            type: ParameterType.STRING_LIST,
            value: props.stringListValue.join(','),
        });
        this.parameterName = this.getResourceNameAttribute(resource.ref);
        this.parameterArn = util_1.arnForParameterName(this, this.parameterName, {
            physicalName: props.parameterName || util_1.AUTOGEN_MARKER,
            simpleName: props.simpleName,
        });
        this.parameterType = resource.attrType;
        this.stringListValue = core_1.Fn.split(',', resource.attrValue);
    }
    /**
     * Imports an external parameter of type string list.
     * Returns a token and should not be parsed.
     */
    static fromStringListParameterName(scope, id, stringListParameterName) {
        class Import extends ParameterBase {
            constructor() {
                super(...arguments);
                this.parameterName = stringListParameterName;
                this.parameterArn = util_1.arnForParameterName(this, this.parameterName);
                this.parameterType = ParameterType.STRING_LIST;
                this.stringListValue = core_1.Fn.split(',', new core_1.CfnDynamicReference(core_1.CfnDynamicReferenceService.SSM, stringListParameterName).toString());
            }
        }
        return new Import(scope, id);
    }
}
exports.StringListParameter = StringListParameter;
/**
 * Validates whether a supplied value conforms to the allowedPattern, granted neither is an unresolved token.
 *
 * @param value          the value to be validated.
 * @param allowedPattern the regular expression to use for validation.
 *
 * @throws if the ``value`` does not conform to the ``allowedPattern`` and neither is an unresolved token (per
 *         ``cdk.unresolved``).
 */
function _assertValidValue(value, allowedPattern) {
    if (core_1.Token.isUnresolved(value) || core_1.Token.isUnresolved(allowedPattern)) {
        // Unable to perform validations against unresolved tokens
        return;
    }
    if (!new RegExp(allowedPattern).test(value)) {
        throw new Error(`The supplied value (${value}) does not match the specified allowedPattern (${allowedPattern})`);
    }
}
function makeIdentityForImportedValue(parameterName) {
    return `SsmParameterValue:${parameterName}:C96584B6-F00A-464E-AD19-53AFF4B05118`;
}
//# sourceMappingURL=data:application/json;base64,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