import os
import re
import requests
import urllib3
from pathlib import Path
from requests import Session
from requests.auth import HTTPBasicAuth
from zeep import Client, Settings
from zeep.cache import SqliteCache
from zeep.exceptions import Fault
from zeep.transports import Transport


class axl(object):
    """
    This class creates the SOAP interface to interact with Cisco Unified Communciations Manager.
    The class methods were created with the WSDL file for CUCM release 11.5.
    Tested with Python 3.8.10.
    """

    def __init__(self, username, password, cucm, cucm_version):
        """
        :param username: axl username
        :param password: axl password
        :param cucm: fqdn or IP address of CUCM
        :param cucm_version: CUCM version
        """

        cwd = os.path.dirname(os.path.abspath(__file__))
        if os.name == "posix":
            wsdl = Path(f"{cwd}/schema/{cucm_version}/AXLAPI.wsdl").as_uri()
        else:
            wsdl = str(Path(f"{cwd}/schema/{cucm_version}/AXLAPI.wsdl").absolute())

        # create a session
        session = Session()
        
        # disable certificate verification and insecure request warnings
        session.verify = False
        urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
        requests.packages.urllib3.disable_warnings( )

        # to enable SSL cert verification, copy the CUCM tomcat .pem file and uncomment the following lines
        # CERT = 'cucmtomcat.pem'
        # session.verify = CERT

        session.auth = HTTPBasicAuth(username, password)
        transport = Transport(session=session, timeout=10, cache=SqliteCache())

        settings = Settings(strict=False, xml_huge_tree=True, xsd_ignore_sequence_order=True)
        
        axl_client = Client(wsdl, settings=settings, transport=transport)

        self.wsdl = wsdl
        self.username = username
        self.password = password
        self.cucm = cucm
        self.cucm_version = cucm_version
        self.UUID_PATTERN = re.compile(
            r"^[\da-f]{8}-([\da-f]{4}-){3}[\da-f]{12}$", re.IGNORECASE
        )
        self.client = axl_client.create_service(
            "{http://www.cisco.com/AXLAPIService/}AXLAPIBinding",
            f"https://{cucm}:8443/axl/",
        )

    def addSipProfile(self, name="", description=None, defaultTelephonyEventPayloadType=None, redirectByApplication=None, ringing180=None, timerInvite=None, timerRegisterDelta=None, timerRegister=None, timerT1=None, timerT2=None, retryInvite=None, retryNotInvite=None, startMediaPort=None, stopMediaPort=None, startVideoPort=None, stopVideoPort=None, dscpForAudioCalls=None, dscpForVideoCalls=None, dscpForAudioPortionOfVideoCalls=None, dscpForTelePresenceCalls=None, dscpForAudioPortionOfTelePresenceCalls=None, callpickupListUri=None, callpickupGroupUri=None, meetmeServiceUrl=None, userInfo=None, dtmfDbLevel=None, callHoldRingback=None, anonymousCallBlock=None, callerIdBlock=None, dndControl=None, telnetLevel=None, timerKeepAlive=None, timerSubscribe=None, timerSubscribeDelta=None, maxRedirects=None, timerOffHookToFirstDigit=None, callForwardUri=None, abbreviatedDialUri=None, confJointEnable=None, rfc2543Hold=None, semiAttendedTransfer=None, enableVad=None, stutterMsgWaiting=None, callStats=None, t38Invite=None, faxInvite=None, rerouteIncomingRequest=None, resourcePriorityNamespaceListName=None, enableAnatForEarlyOfferCalls=None, rsvpOverSip=None, fallbackToLocalRsvp=None, sipRe11XxEnabled=None, gClear=None, sendRecvSDPInMidCallInvite=None, enableOutboundOptionsPing=None, optionsPingIntervalWhenStatusOK=None, optionsPingIntervalWhenStatusNotOK=None, deliverConferenceBridgeIdentifier=None, sipOptionsRetryCount=None, sipOptionsRetryTimer=None, sipBandwidthModifier="", enableUriOutdialSupport=None, userAgentServerHeaderInfo="", allowPresentationSharingUsingBfcp=None, scriptParameters=None, isScriptTraceEnabled=None, sipNormalizationScript=None, allowiXApplicationMedia=None, dialStringInterpretation="", acceptAudioCodecPreferences=None, mlppUserAuthorization=None, isAssuredSipServiceEnabled=None, enableExternalQoS=None, resourcePriorityNamespace=None, useCallerIdCallerNameinUriOutgoingRequest=None, callerIdDn=None, callerName=None, callingLineIdentification="", rejectAnonymousIncomingCall=None, callpickupUri=None, rejectAnonymousOutgoingCall=None, videoCallTrafficClass=None, sdpTransparency=None, allowMultipleCodecs=None, sipSessionRefreshMethod="", earlyOfferSuppVoiceCall="", cucmVersionInSipHeader="", confidentialAccessLevelHeaders="", destRouteString=None, inactiveSDPRequired=None, connectCallBeforePlayingAnnouncement=None):
        """
        :param sipProfile: 
            name: String50(value)
            description: String100(value), optional
            defaultTelephonyEventPayloadType: UnionType(value), optional
            redirectByApplication: boolean(value), optional
            ringing180: boolean(value), optional
            timerInvite: UnionType(value), optional
            timerRegisterDelta: UnionType(value), optional
            timerRegister: UnionType(value), optional
            timerT1: UnionType(value), optional
            timerT2: UnionType(value), optional
            retryInvite: UnionType(value), optional
            retryNotInvite: UnionType(value), optional
            startMediaPort: UnionType(value), optional
            stopMediaPort: UnionType(value), optional
            startVideoPort: UnionType(value), optional
            stopVideoPort: UnionType(value), optional
            dscpForAudioCalls: String(value), optional
            dscpForVideoCalls: String(value), optional
            dscpForAudioPortionOfVideoCalls: String(value), optional
            dscpForTelePresenceCalls: String(value), optional
            dscpForAudioPortionOfTelePresenceCalls: String(value), optional
            callpickupListUri: String128(value), optional
            callpickupGroupUri: String128(value), optional
            meetmeServiceUrl: String128(value), optional
            userInfo: UnionType(value), optional
            dtmfDbLevel: UnionType(value), optional
            callHoldRingback: UnionType(value), optional
            anonymousCallBlock: UnionType(value), optional
            callerIdBlock: UnionType(value), optional
            dndControl: UnionType(value), optional
            telnetLevel: UnionType(value), optional
            timerKeepAlive: UnionType(value), optional
            timerSubscribe: UnionType(value), optional
            timerSubscribeDelta: UnionType(value), optional
            maxRedirects: UnionType(value), optional
            timerOffHookToFirstDigit: UnionType(value), optional
            callForwardUri: String128(value), optional
            abbreviatedDialUri: String128(value), optional
            confJointEnable: boolean(value), optional
            rfc2543Hold: boolean(value), optional
            semiAttendedTransfer: boolean(value), optional
            enableVad: boolean(value), optional
            stutterMsgWaiting: boolean(value), optional
            callStats: boolean(value), optional
            t38Invite: boolean(value), optional
            faxInvite: boolean(value), optional
            rerouteIncomingRequest: UnionType(value), optional
            resourcePriorityNamespaceListName: optional
                _value_1: String(value)
            enableAnatForEarlyOfferCalls: boolean(value), optional
            rsvpOverSip: UnionType(value), optional
            fallbackToLocalRsvp: boolean(value), optional
            sipRe11XxEnabled: UnionType(value), optional
            gClear: UnionType(value), optional
            sendRecvSDPInMidCallInvite: boolean(value), optional
            enableOutboundOptionsPing: boolean(value), optional
            optionsPingIntervalWhenStatusOK: UnionType(value), optional
            optionsPingIntervalWhenStatusNotOK: UnionType(value), optional
            deliverConferenceBridgeIdentifier: boolean(value), optional
            sipOptionsRetryCount: UnionType(value), optional
            sipOptionsRetryTimer: UnionType(value), optional
            sipBandwidthModifier: UnionType(value)
            enableUriOutdialSupport: String(value), optional
            userAgentServerHeaderInfo: UnionType(value)
            allowPresentationSharingUsingBfcp: boolean(value), optional
            scriptParameters: String(value), optional
            isScriptTraceEnabled: boolean(value), optional
            sipNormalizationScript: optional
                _value_1: String(value)
            allowiXApplicationMedia: boolean(value), optional
            dialStringInterpretation: UnionType(value)
            acceptAudioCodecPreferences: UnionType(value), optional
            mlppUserAuthorization: boolean(value), optional
            isAssuredSipServiceEnabled: boolean(value), optional
            enableExternalQoS: boolean(value), optional
            resourcePriorityNamespace: optional
                _value_1: String(value)
            useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
            callerIdDn: String50(value), optional
            callerName: String50(value), optional
            callingLineIdentification: UnionType(value)
            rejectAnonymousIncomingCall: boolean(value), optional
            callpickupUri: String128(value), optional
            rejectAnonymousOutgoingCall: boolean(value), optional
            videoCallTrafficClass: UnionType(value), optional
            sdpTransparency: optional
                _value_1: String(value)
            allowMultipleCodecs: boolean(value), optional
            sipSessionRefreshMethod: UnionType(value)
            earlyOfferSuppVoiceCall: UnionType(value)
            cucmVersionInSipHeader: UnionType(value)
            confidentialAccessLevelHeaders: UnionType(value)
            destRouteString: boolean(value), optional
            inactiveSDPRequired: boolean(value), optional
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addSipProfile({"name": name, "description": description, "defaultTelephonyEventPayloadType": defaultTelephonyEventPayloadType, "redirectByApplication": redirectByApplication, "ringing180": ringing180, "timerInvite": timerInvite, "timerRegisterDelta": timerRegisterDelta, "timerRegister": timerRegister, "timerT1": timerT1, "timerT2": timerT2, "retryInvite": retryInvite, "retryNotInvite": retryNotInvite, "startMediaPort": startMediaPort, "stopMediaPort": stopMediaPort, "startVideoPort": startVideoPort, "stopVideoPort": stopVideoPort, "dscpForAudioCalls": dscpForAudioCalls, "dscpForVideoCalls": dscpForVideoCalls, "dscpForAudioPortionOfVideoCalls": dscpForAudioPortionOfVideoCalls, "dscpForTelePresenceCalls": dscpForTelePresenceCalls, "dscpForAudioPortionOfTelePresenceCalls": dscpForAudioPortionOfTelePresenceCalls, "callpickupListUri": callpickupListUri, "callpickupGroupUri": callpickupGroupUri, "meetmeServiceUrl": meetmeServiceUrl, "userInfo": userInfo, "dtmfDbLevel": dtmfDbLevel, "callHoldRingback": callHoldRingback, "anonymousCallBlock": anonymousCallBlock, "callerIdBlock": callerIdBlock, "dndControl": dndControl, "telnetLevel": telnetLevel, "timerKeepAlive": timerKeepAlive, "timerSubscribe": timerSubscribe, "timerSubscribeDelta": timerSubscribeDelta, "maxRedirects": maxRedirects, "timerOffHookToFirstDigit": timerOffHookToFirstDigit, "callForwardUri": callForwardUri, "abbreviatedDialUri": abbreviatedDialUri, "confJointEnable": confJointEnable, "rfc2543Hold": rfc2543Hold, "semiAttendedTransfer": semiAttendedTransfer, "enableVad": enableVad, "stutterMsgWaiting": stutterMsgWaiting, "callStats": callStats, "t38Invite": t38Invite, "faxInvite": faxInvite, "rerouteIncomingRequest": rerouteIncomingRequest, "resourcePriorityNamespaceListName": resourcePriorityNamespaceListName, "enableAnatForEarlyOfferCalls": enableAnatForEarlyOfferCalls, "rsvpOverSip": rsvpOverSip, "fallbackToLocalRsvp": fallbackToLocalRsvp, "sipRe11XxEnabled": sipRe11XxEnabled, "gClear": gClear, "sendRecvSDPInMidCallInvite": sendRecvSDPInMidCallInvite, "enableOutboundOptionsPing": enableOutboundOptionsPing, "optionsPingIntervalWhenStatusOK": optionsPingIntervalWhenStatusOK, "optionsPingIntervalWhenStatusNotOK": optionsPingIntervalWhenStatusNotOK, "deliverConferenceBridgeIdentifier": deliverConferenceBridgeIdentifier, "sipOptionsRetryCount": sipOptionsRetryCount, "sipOptionsRetryTimer": sipOptionsRetryTimer, "sipBandwidthModifier": sipBandwidthModifier, "enableUriOutdialSupport": enableUriOutdialSupport, "userAgentServerHeaderInfo": userAgentServerHeaderInfo, "allowPresentationSharingUsingBfcp": allowPresentationSharingUsingBfcp, "scriptParameters": scriptParameters, "isScriptTraceEnabled": isScriptTraceEnabled, "sipNormalizationScript": sipNormalizationScript, "allowiXApplicationMedia": allowiXApplicationMedia, "dialStringInterpretation": dialStringInterpretation, "acceptAudioCodecPreferences": acceptAudioCodecPreferences, "mlppUserAuthorization": mlppUserAuthorization, "isAssuredSipServiceEnabled": isAssuredSipServiceEnabled, "enableExternalQoS": enableExternalQoS, "resourcePriorityNamespace": resourcePriorityNamespace, "useCallerIdCallerNameinUriOutgoingRequest": useCallerIdCallerNameinUriOutgoingRequest, "callerIdDn": callerIdDn, "callerName": callerName, "callingLineIdentification": callingLineIdentification, "rejectAnonymousIncomingCall": rejectAnonymousIncomingCall, "callpickupUri": callpickupUri, "rejectAnonymousOutgoingCall": rejectAnonymousOutgoingCall, "videoCallTrafficClass": videoCallTrafficClass, "sdpTransparency": sdpTransparency, "allowMultipleCodecs": allowMultipleCodecs, "sipSessionRefreshMethod": sipSessionRefreshMethod, "earlyOfferSuppVoiceCall": earlyOfferSuppVoiceCall, "cucmVersionInSipHeader": cucmVersionInSipHeader, "confidentialAccessLevelHeaders": confidentialAccessLevelHeaders, "destRouteString": destRouteString, "inactiveSDPRequired": inactiveSDPRequired, "connectCallBeforePlayingAnnouncement": connectCallBeforePlayingAnnouncement})["return"]
        except Fault as e:
            return e

    def updateSipProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String100(value), optional
        :param defaultTelephonyEventPayloadType: UnionType(value), optional
        :param redirectByApplication: boolean(value), optional
        :param ringing180: boolean(value), optional
        :param timerInvite: UnionType(value), optional
        :param timerRegisterDelta: UnionType(value), optional
        :param timerRegister: UnionType(value), optional
        :param timerT1: UnionType(value), optional
        :param timerT2: UnionType(value), optional
        :param retryInvite: UnionType(value), optional
        :param retryNotInvite: UnionType(value), optional
        :param startMediaPort: UnionType(value), optional
        :param stopMediaPort: UnionType(value), optional
        :param startVideoPort: UnionType(value), optional
        :param stopVideoPort: UnionType(value), optional
        :param dscpForAudioCalls: String(value), optional
        :param dscpForVideoCalls: String(value), optional
        :param dscpForAudioPortionOfVideoCalls: String(value), optional
        :param dscpForTelePresenceCalls: String(value), optional
        :param dscpForAudioPortionOfTelePresenceCalls: String(value), optional
        :param callpickupListUri: String128(value), optional
        :param callpickupGroupUri: String128(value), optional
        :param meetmeServiceUrl: String128(value), optional
        :param userInfo: UnionType(value), optional
        :param dtmfDbLevel: UnionType(value), optional
        :param callHoldRingback: UnionType(value), optional
        :param anonymousCallBlock: UnionType(value), optional
        :param callerIdBlock: UnionType(value), optional
        :param dndControl: UnionType(value), optional
        :param telnetLevel: UnionType(value), optional
        :param timerKeepAlive: UnionType(value), optional
        :param timerSubscribe: UnionType(value), optional
        :param timerSubscribeDelta: UnionType(value), optional
        :param maxRedirects: UnionType(value), optional
        :param timerOffHookToFirstDigit: UnionType(value), optional
        :param callForwardUri: String128(value), optional
        :param abbreviatedDialUri: String128(value), optional
        :param confJointEnable: boolean(value), optional
        :param rfc2543Hold: boolean(value), optional
        :param semiAttendedTransfer: boolean(value), optional
        :param enableVad: boolean(value), optional
        :param stutterMsgWaiting: boolean(value), optional
        :param callStats: boolean(value), optional
        :param t38Invite: boolean(value), optional
        :param faxInvite: boolean(value), optional
        :param rerouteIncomingRequest: UnionType(value), optional
        :param resourcePriorityNamespaceListName: optional
            _value_1: String(value)
        :param enableAnatForEarlyOfferCalls: boolean(value), optional
        :param rsvpOverSip: UnionType(value), optional
        :param fallbackToLocalRsvp: boolean(value), optional
        :param sipRe11XxEnabled: UnionType(value), optional
        :param gClear: UnionType(value), optional
        :param sendRecvSDPInMidCallInvite: boolean(value), optional
        :param enableOutboundOptionsPing: boolean(value), optional
        :param optionsPingIntervalWhenStatusOK: UnionType(value), optional
        :param optionsPingIntervalWhenStatusNotOK: UnionType(value), optional
        :param deliverConferenceBridgeIdentifier: boolean(value), optional
        :param sipOptionsRetryCount: UnionType(value), optional
        :param sipOptionsRetryTimer: UnionType(value), optional
        :param sipBandwidthModifier: UnionType(value), optional
        :param enableUriOutdialSupport: String(value), optional
        :param userAgentServerHeaderInfo: UnionType(value), optional
        :param allowPresentationSharingUsingBfcp: boolean(value), optional
        :param scriptParameters: String(value), optional
        :param isScriptTraceEnabled: boolean(value), optional
        :param sipNormalizationScript: optional
            _value_1: String(value)
        :param allowiXApplicationMedia: boolean(value), optional
        :param dialStringInterpretation: UnionType(value), optional
        :param acceptAudioCodecPreferences: UnionType(value), optional
        :param mlppUserAuthorization: boolean(value), optional
        :param isAssuredSipServiceEnabled: boolean(value), optional
        :param enableExternalQoS: boolean(value), optional
        :param resourcePriorityNamespace: optional
            _value_1: String(value)
        :param useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
        :param callerIdDn: String50(value), optional
        :param callerName: String50(value), optional
        :param callingLineIdentification: UnionType(value), optional
        :param rejectAnonymousIncomingCall: boolean(value), optional
        :param callpickupUri: String128(value), optional
        :param rejectAnonymousOutgoingCall: boolean(value), optional
        :param videoCallTrafficClass: UnionType(value), optional
        :param sdpTransparency: optional
            _value_1: String(value)
        :param allowMultipleCodecs: boolean(value), optional
        :param sipSessionRefreshMethod: UnionType(value), optional
        :param earlyOfferSuppVoiceCall: UnionType(value), optional
        :param cucmVersionInSipHeader: UnionType(value), optional
        :param confidentialAccessLevelHeaders: UnionType(value), optional
        :param destRouteString: boolean(value), optional
        :param inactiveSDPRequired: boolean(value), optional
        :param connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateSipProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getSipProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String100(value), optional
            defaultTelephonyEventPayloadType: UnionType(value), optional
            redirectByApplication: boolean(value), optional
            ringing180: boolean(value), optional
            timerInvite: UnionType(value), optional
            timerRegisterDelta: UnionType(value), optional
            timerRegister: UnionType(value), optional
            timerT1: UnionType(value), optional
            timerT2: UnionType(value), optional
            retryInvite: UnionType(value), optional
            retryNotInvite: UnionType(value), optional
            startMediaPort: UnionType(value), optional
            stopMediaPort: UnionType(value), optional
            startVideoPort: UnionType(value), optional
            stopVideoPort: UnionType(value), optional
            dscpForAudioCalls: String(value), optional
            dscpForVideoCalls: String(value), optional
            dscpForAudioPortionOfVideoCalls: String(value), optional
            dscpForTelePresenceCalls: String(value), optional
            dscpForAudioPortionOfTelePresenceCalls: String(value), optional
            callpickupListUri: String128(value), optional
            callpickupGroupUri: String128(value), optional
            meetmeServiceUrl: String128(value), optional
            userInfo: UnionType(value), optional
            dtmfDbLevel: UnionType(value), optional
            callHoldRingback: UnionType(value), optional
            anonymousCallBlock: UnionType(value), optional
            callerIdBlock: UnionType(value), optional
            dndControl: UnionType(value), optional
            telnetLevel: UnionType(value), optional
            timerKeepAlive: UnionType(value), optional
            timerSubscribe: UnionType(value), optional
            timerSubscribeDelta: UnionType(value), optional
            maxRedirects: UnionType(value), optional
            timerOffHookToFirstDigit: UnionType(value), optional
            callForwardUri: String128(value), optional
            abbreviatedDialUri: String128(value), optional
            confJointEnable: boolean(value), optional
            rfc2543Hold: boolean(value), optional
            semiAttendedTransfer: boolean(value), optional
            enableVad: boolean(value), optional
            stutterMsgWaiting: boolean(value), optional
            callStats: boolean(value), optional
            t38Invite: boolean(value), optional
            faxInvite: boolean(value), optional
            rerouteIncomingRequest: UnionType(value), optional
            resourcePriorityNamespaceListName: optional
                _value_1: String(value)
            enableAnatForEarlyOfferCalls: boolean(value), optional
            rsvpOverSip: UnionType(value), optional
            fallbackToLocalRsvp: boolean(value), optional
            sipRe11XxEnabled: UnionType(value), optional
            gClear: UnionType(value), optional
            sendRecvSDPInMidCallInvite: boolean(value), optional
            enableOutboundOptionsPing: boolean(value), optional
            optionsPingIntervalWhenStatusOK: UnionType(value), optional
            optionsPingIntervalWhenStatusNotOK: UnionType(value), optional
            deliverConferenceBridgeIdentifier: boolean(value), optional
            sipOptionsRetryCount: UnionType(value), optional
            sipOptionsRetryTimer: UnionType(value), optional
            sipBandwidthModifier: UnionType(value), optional
            enableUriOutdialSupport: String(value), optional
            userAgentServerHeaderInfo: UnionType(value), optional
            allowPresentationSharingUsingBfcp: boolean(value), optional
            scriptParameters: String(value), optional
            isScriptTraceEnabled: boolean(value), optional
            sipNormalizationScript: optional
                _value_1: String(value)
            allowiXApplicationMedia: boolean(value), optional
            dialStringInterpretation: UnionType(value), optional
            acceptAudioCodecPreferences: UnionType(value), optional
            mlppUserAuthorization: boolean(value), optional
            isAssuredSipServiceEnabled: boolean(value), optional
            enableExternalQoS: boolean(value), optional
            resourcePriorityNamespace: optional
                _value_1: String(value)
            useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
            callerIdDn: String50(value), optional
            callerName: String50(value), optional
            callingLineIdentification: UnionType(value), optional
            rejectAnonymousIncomingCall: boolean(value), optional
            callpickupUri: String128(value), optional
            rejectAnonymousOutgoingCall: boolean(value), optional
            videoCallTrafficClass: UnionType(value), optional
            sdpTransparency: optional
                _value_1: String(value)
            allowMultipleCodecs: boolean(value), optional
            sipSessionRefreshMethod: UnionType(value), optional
            earlyOfferSuppVoiceCall: UnionType(value), optional
            cucmVersionInSipHeader: UnionType(value), optional
            confidentialAccessLevelHeaders: UnionType(value), optional
            destRouteString: boolean(value), optional
            inactiveSDPRequired: boolean(value), optional
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getSipProfile(**kwargs)["return"]["sipProfile"]
        except Fault as e:
            return e

    def removeSipProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSipProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSipProfile(self, returnedTags={'name': '', 'description': '', 'defaultTelephonyEventPayloadType': '', 'redirectByApplication': '', 'ringing180': '', 'timerInvite': '', 'timerRegisterDelta': '', 'timerRegister': '', 'timerT1': '', 'timerT2': '', 'retryInvite': '', 'retryNotInvite': '', 'startMediaPort': '', 'stopMediaPort': '', 'callpickupListUri': '', 'callpickupGroupUri': '', 'meetmeServiceUrl': '', 'userInfo': '', 'dtmfDbLevel': '', 'callHoldRingback': '', 'anonymousCallBlock': '', 'callerIdBlock': '', 'dndControl': '', 'telnetLevel': '', 'timerKeepAlive': '', 'timerSubscribe': '', 'timerSubscribeDelta': '', 'maxRedirects': '', 'timerOffHookToFirstDigit': '', 'callForwardUri': '', 'abbreviatedDialUri': '', 'confJointEnable': '', 'rfc2543Hold': '', 'semiAttendedTransfer': '', 'enableVad': '', 'stutterMsgWaiting': '', 'callStats': '', 't38Invite': '', 'faxInvite': '', 'rerouteIncomingRequest': '', 'resourcePriorityNamespaceListName': {'_value_1': ''}, 'enableAnatForEarlyOfferCalls': '', 'rsvpOverSip': '', 'fallbackToLocalRsvp': '', 'sipRe11XxEnabled': '', 'gClear': '', 'sendRecvSDPInMidCallInvite': '', 'enableOutboundOptionsPing': '', 'optionsPingIntervalWhenStatusOK': '', 'optionsPingIntervalWhenStatusNotOK': '', 'deliverConferenceBridgeIdentifier': '', 'sipOptionsRetryCount': '', 'sipOptionsRetryTimer': '', 'sipBandwidthModifier': '', 'enableUriOutdialSupport': '', 'userAgentServerHeaderInfo': '', 'allowPresentationSharingUsingBfcp': '', 'scriptParameters': '', 'isScriptTraceEnabled': '', 'sipNormalizationScript': {'_value_1': ''}, 'allowiXApplicationMedia': '', 'dialStringInterpretation': '', 'acceptAudioCodecPreferences': '', 'mlppUserAuthorization': '', 'isAssuredSipServiceEnabled': '', 'resourcePriorityNamespace': {'_value_1': ''}, 'useCallerIdCallerNameinUriOutgoingRequest': '', 'callerIdDn': '', 'callerName': '', 'callingLineIdentification': '', 'rejectAnonymousIncomingCall': '', 'callpickupUri': '', 'rejectAnonymousOutgoingCall': '', 'videoCallTrafficClass': '', 'sdpTransparency': {'_value_1': ''}, 'allowMultipleCodecs': '', 'sipSessionRefreshMethod': '', 'earlyOfferSuppVoiceCall': '', 'cucmVersionInSipHeader': '', 'confidentialAccessLevelHeaders': '', 'destRouteString': '', 'inactiveSDPRequired': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String100(value), optional
            defaultTelephonyEventPayloadType: UnionType(value), optional
            redirectByApplication: boolean(value), optional
            ringing180: boolean(value), optional
            timerInvite: UnionType(value), optional
            timerRegisterDelta: UnionType(value), optional
            timerRegister: UnionType(value), optional
            timerT1: UnionType(value), optional
            timerT2: UnionType(value), optional
            retryInvite: UnionType(value), optional
            retryNotInvite: UnionType(value), optional
            startMediaPort: UnionType(value), optional
            stopMediaPort: UnionType(value), optional
            callpickupListUri: String128(value), optional
            callpickupGroupUri: String128(value), optional
            meetmeServiceUrl: String128(value), optional
            userInfo: UnionType(value), optional
            dtmfDbLevel: UnionType(value), optional
            callHoldRingback: UnionType(value), optional
            anonymousCallBlock: UnionType(value), optional
            callerIdBlock: UnionType(value), optional
            dndControl: UnionType(value), optional
            telnetLevel: UnionType(value), optional
            timerKeepAlive: UnionType(value), optional
            timerSubscribe: UnionType(value), optional
            timerSubscribeDelta: UnionType(value), optional
            maxRedirects: UnionType(value), optional
            timerOffHookToFirstDigit: UnionType(value), optional
            callForwardUri: String128(value), optional
            abbreviatedDialUri: String128(value), optional
            confJointEnable: boolean(value), optional
            rfc2543Hold: boolean(value), optional
            semiAttendedTransfer: boolean(value), optional
            enableVad: boolean(value), optional
            stutterMsgWaiting: boolean(value), optional
            callStats: boolean(value), optional
            t38Invite: boolean(value), optional
            faxInvite: boolean(value), optional
            rerouteIncomingRequest: UnionType(value), optional
            resourcePriorityNamespaceListName: optional
                _value_1: String(value)
            enableAnatForEarlyOfferCalls: boolean(value), optional
            rsvpOverSip: UnionType(value), optional
            fallbackToLocalRsvp: boolean(value), optional
            sipRe11XxEnabled: UnionType(value), optional
            gClear: UnionType(value), optional
            sendRecvSDPInMidCallInvite: boolean(value), optional
            enableOutboundOptionsPing: boolean(value), optional
            optionsPingIntervalWhenStatusOK: UnionType(value), optional
            optionsPingIntervalWhenStatusNotOK: UnionType(value), optional
            deliverConferenceBridgeIdentifier: boolean(value), optional
            sipOptionsRetryCount: UnionType(value), optional
            sipOptionsRetryTimer: UnionType(value), optional
            sipBandwidthModifier: UnionType(value), optional
            enableUriOutdialSupport: String(value), optional
            userAgentServerHeaderInfo: UnionType(value), optional
            allowPresentationSharingUsingBfcp: boolean(value), optional
            scriptParameters: String(value), optional
            isScriptTraceEnabled: boolean(value), optional
            sipNormalizationScript: optional
                _value_1: String(value)
            allowiXApplicationMedia: boolean(value), optional
            dialStringInterpretation: UnionType(value), optional
            acceptAudioCodecPreferences: UnionType(value), optional
            mlppUserAuthorization: boolean(value), optional
            isAssuredSipServiceEnabled: boolean(value), optional
            resourcePriorityNamespace: optional
                _value_1: String(value)
            useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
            callerIdDn: String50(value), optional
            callerName: String50(value), optional
            callingLineIdentification: UnionType(value), optional
            rejectAnonymousIncomingCall: boolean(value), optional
            callpickupUri: String128(value), optional
            rejectAnonymousOutgoingCall: boolean(value), optional
            videoCallTrafficClass: UnionType(value), optional
            sdpTransparency: optional
                _value_1: String(value)
            allowMultipleCodecs: boolean(value), optional
            sipSessionRefreshMethod: UnionType(value), optional
            earlyOfferSuppVoiceCall: UnionType(value), optional
            cucmVersionInSipHeader: UnionType(value), optional
            confidentialAccessLevelHeaders: UnionType(value), optional
            destRouteString: boolean(value), optional
            inactiveSDPRequired: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSipProfile({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["sipProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def restartSipProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartSipProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applySipProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applySipProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def getSipProfileOptions(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedChoices: optional
            userInfo: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dtmfDbLevel: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callHoldRingback: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            anonymousCallBlock: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callerIdBlock: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dndControl: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            telnetLevel: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            rerouteIncomingRequest: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            resourcePriorityNamespaceListName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            rsvpOverSip: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            sipRe11XxEnabled: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            gClear: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            sipBandwidthModifier: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            userAgentServerHeaderInfo: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            sipNormalizationScript: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dialStringInterpretation: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            acceptAudioCodecPreferences: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            resourcePriorityNamespace: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingLineIdentification: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            videoCallTrafficClass: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            sdpTransparency: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            sipSessionRefreshMethod: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            earlyOfferSuppVoiceCall: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            cucmVersionInSipHeader: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            confidentialAccessLevelHeaders: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
        """
        try:
            return self.client.getSipProfileOptions(**kwargs)["return"]["sipProfileOptions"]
        except Fault as e:
            return e

    def addSipTrunkSecurityProfile(self, name="", description=None, securityMode=None, incomingTransport=None, outgoingTransport=None, digestAuthentication=None, noncePolicyTime=None, x509SubjectName=None, incomingPort=None, applLevelAuthentication=None, acceptPresenceSubscription=None, acceptOutOfDialogRefer=None, acceptUnsolicitedNotification=None, allowReplaceHeader=None, transmitSecurityStatus=None, sipV150OutboundSdpOfferFiltering="", allowChargingHeader=None):
        """
        :param sipTrunkSecurityProfile: 
            name: String75(value)
            description: String100(value), optional
            securityMode: UnionType(value), optional
            incomingTransport: UnionType(value), optional
            outgoingTransport: UnionType(value), optional
            digestAuthentication: boolean(value), optional
            noncePolicyTime: UnionType(value), optional
            x509SubjectName: String4096(value), optional
            incomingPort: UnionType(value), optional
            applLevelAuthentication: boolean(value), optional
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            transmitSecurityStatus: boolean(value), optional
            sipV150OutboundSdpOfferFiltering: UnionType(value)
            allowChargingHeader: boolean(value), optional
        """
        try:
            return self.client.addSipTrunkSecurityProfile({"name": name, "description": description, "securityMode": securityMode, "incomingTransport": incomingTransport, "outgoingTransport": outgoingTransport, "digestAuthentication": digestAuthentication, "noncePolicyTime": noncePolicyTime, "x509SubjectName": x509SubjectName, "incomingPort": incomingPort, "applLevelAuthentication": applLevelAuthentication, "acceptPresenceSubscription": acceptPresenceSubscription, "acceptOutOfDialogRefer": acceptOutOfDialogRefer, "acceptUnsolicitedNotification": acceptUnsolicitedNotification, "allowReplaceHeader": allowReplaceHeader, "transmitSecurityStatus": transmitSecurityStatus, "sipV150OutboundSdpOfferFiltering": sipV150OutboundSdpOfferFiltering, "allowChargingHeader": allowChargingHeader})["return"]
        except Fault as e:
            return e

    def updateSipTrunkSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String75(value), optional
        :param description: String100(value), optional
        :param securityMode: UnionType(value), optional
        :param incomingTransport: UnionType(value), optional
        :param outgoingTransport: UnionType(value), optional
        :param digestAuthentication: boolean(value), optional
        :param noncePolicyTime: UnionType(value), optional
        :param x509SubjectName: String4096(value), optional
        :param incomingPort: UnionType(value), optional
        :param applLevelAuthentication: boolean(value), optional
        :param acceptPresenceSubscription: boolean(value), optional
        :param acceptOutOfDialogRefer: boolean(value), optional
        :param acceptUnsolicitedNotification: boolean(value), optional
        :param allowReplaceHeader: boolean(value), optional
        :param transmitSecurityStatus: boolean(value), optional
        :param sipV150OutboundSdpOfferFiltering: UnionType(value), optional
        :param allowChargingHeader: boolean(value), optional
        """
        try:
            return self.client.updateSipTrunkSecurityProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getSipTrunkSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String75(value), optional
            description: String100(value), optional
            securityMode: UnionType(value), optional
            incomingTransport: UnionType(value), optional
            outgoingTransport: UnionType(value), optional
            digestAuthentication: boolean(value), optional
            noncePolicyTime: UnionType(value), optional
            x509SubjectName: String4096(value), optional
            incomingPort: UnionType(value), optional
            applLevelAuthentication: boolean(value), optional
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            transmitSecurityStatus: boolean(value), optional
            sipV150OutboundSdpOfferFiltering: UnionType(value), optional
            allowChargingHeader: boolean(value), optional
        """
        try:
            return self.client.getSipTrunkSecurityProfile(**kwargs)["return"]["sipTrunkSecurityProfile"]
        except Fault as e:
            return e

    def removeSipTrunkSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSipTrunkSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSipTrunkSecurityProfile(self, returnedTags={'name': '', 'description': '', 'securityMode': '', 'incomingTransport': '', 'outgoingTransport': '', 'digestAuthentication': '', 'noncePolicyTime': '', 'x509SubjectName': '', 'incomingPort': '', 'applLevelAuthentication': '', 'acceptPresenceSubscription': '', 'acceptOutOfDialogRefer': '', 'acceptUnsolicitedNotification': '', 'allowReplaceHeader': '', 'transmitSecurityStatus': '', 'sipV150OutboundSdpOfferFiltering': '', 'allowChargingHeader': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String75(value), optional
            description: String100(value), optional
            securityMode: UnionType(value), optional
            incomingTransport: UnionType(value), optional
            outgoingTransport: UnionType(value), optional
            digestAuthentication: boolean(value), optional
            noncePolicyTime: UnionType(value), optional
            x509SubjectName: String4096(value), optional
            incomingPort: UnionType(value), optional
            applLevelAuthentication: boolean(value), optional
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            transmitSecurityStatus: boolean(value), optional
            sipV150OutboundSdpOfferFiltering: UnionType(value), optional
            allowChargingHeader: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSipTrunkSecurityProfile({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["sipTrunkSecurityProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetSipTrunkSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetSipTrunkSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applySipTrunkSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applySipTrunkSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addTimePeriod(self, name="", startTime=None, endTime=None, startDay=None, endDay=None, monthOfYear=None, dayOfMonth=None, description=None, isPublished=None, todOwnerIdName=None, dayOfMonthEnd=None, monthOfYearEnd=None):
        """
        :param timePeriod: 
            name: String50(value)
            startTime: UnionType(value), optional
            endTime: UnionType(value), optional
            startDay: UnionType(value), optional
            endDay: UnionType(value), optional
            monthOfYear: UnionType(value), optional
            dayOfMonth: UnionType(value), optional
            description: String255(value), optional
            isPublished: boolean(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
            dayOfMonthEnd: UnionType(value), optional
            monthOfYearEnd: UnionType(value), optional
        """
        try:
            return self.client.addTimePeriod({"name": name, "startTime": startTime, "endTime": endTime, "startDay": startDay, "endDay": endDay, "monthOfYear": monthOfYear, "dayOfMonth": dayOfMonth, "description": description, "isPublished": isPublished, "todOwnerIdName": todOwnerIdName, "dayOfMonthEnd": dayOfMonthEnd, "monthOfYearEnd": monthOfYearEnd})["return"]
        except Fault as e:
            return e

    def updateTimePeriod(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param startTime: UnionType(value), optional
        :param endTime: UnionType(value), optional
        :param startDay: UnionType(value), optional
        :param endDay: UnionType(value), optional
        :param monthOfYear: UnionType(value), optional
        :param dayOfMonth: UnionType(value), optional
        :param description: String255(value), optional
        :param dayOfMonthEnd: UnionType(value), optional
        :param monthOfYearEnd: UnionType(value), optional
        """
        try:
            return self.client.updateTimePeriod(**kwargs)["return"]
        except Fault as e:
            return e

    def getTimePeriod(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            startTime: UnionType(value), optional
            endTime: UnionType(value), optional
            startDay: UnionType(value), optional
            endDay: UnionType(value), optional
            monthOfYear: UnionType(value), optional
            dayOfMonth: UnionType(value), optional
            description: String255(value), optional
            isPublished: boolean(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
            dayOfMonthEnd: UnionType(value), optional
            monthOfYearEnd: UnionType(value), optional
        """
        try:
            return self.client.getTimePeriod(**kwargs)["return"]["timePeriod"]
        except Fault as e:
            return e

    def removeTimePeriod(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeTimePeriod(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listTimePeriod(self, returnedTags={'name': '', 'startTime': '', 'endTime': '', 'startDay': '', 'endDay': '', 'monthOfYear': '', 'dayOfMonth': '', 'description': '', 'isPublished': '', 'todOwnerIdName': {'_value_1': ''}, 'dayOfMonthEnd': '', 'monthOfYearEnd': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            startTime: UnionType(value), optional
            endTime: UnionType(value), optional
            startDay: UnionType(value), optional
            endDay: UnionType(value), optional
            monthOfYear: UnionType(value), optional
            dayOfMonth: UnionType(value), optional
            description: String255(value), optional
            isPublished: boolean(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
            dayOfMonthEnd: UnionType(value), optional
            monthOfYearEnd: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listTimePeriod({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["timePeriod"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addTimeSchedule(self, name="", members="", description=None, isPublished=None, timeScheduleCategory=None, todOwnerIdName=None):
        """
        :param timeSchedule: 
            name: String50(value)
            members: 
            description: String255(value), optional
            isPublished: boolean(value), optional
            timeScheduleCategory: UnionType(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addTimeSchedule({"name": name, "members": members, "description": description, "isPublished": isPublished, "timeScheduleCategory": timeScheduleCategory, "todOwnerIdName": todOwnerIdName})["return"]
        except Fault as e:
            return e

    def updateTimeSchedule(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
        :param members: optional
            member: 
                    _value_1: String(value)
        :param description: String255(value), optional
        :param timeScheduleCategory: UnionType(value), optional
        """
        try:
            return self.client.updateTimeSchedule(**kwargs)["return"]
        except Fault as e:
            return e

    def getTimeSchedule(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            members: optional
            description: String255(value), optional
            isPublished: boolean(value), optional
            timeScheduleCategory: UnionType(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getTimeSchedule(**kwargs)["return"]["timeSchedule"]
        except Fault as e:
            return e

    def removeTimeSchedule(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeTimeSchedule(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listTimeSchedule(self, returnedTags={'name': '', 'description': '', 'isPublished': '', 'timeScheduleCategory': '', 'todOwnerIdName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String255(value), optional
            isPublished: boolean(value), optional
            timeScheduleCategory: UnionType(value), optional
            todOwnerIdName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listTimeSchedule({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["timeSchedule"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addTodAccess(self, name="", description=None, ownerIdName="", members=None):
        """
        :param todAccess: 
            name: String128(value)
            description: String255(value), optional
            ownerIdName: 
                _value_1: String(value)
            members: optional
        """
        try:
            return self.client.addTodAccess({"name": name, "description": description, "ownerIdName": ownerIdName, "members": members})["return"]
        except Fault as e:
            return e

    def updateTodAccess(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String128(value), optional
        :param description: String255(value), optional
        :param members: optional
            member: 
                    _value_1: String(value)
                    _value_1: String(value)
        """
        try:
            return self.client.updateTodAccess(**kwargs)["return"]
        except Fault as e:
            return e

    def getTodAccess(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String128(value), optional
            description: String255(value), optional
            ownerIdName: optional
                _value_1: String(value)
            members: optional
            associatedRemoteDestination: optional
                remoteDestination: String(value), optional
        """
        try:
            return self.client.getTodAccess(**kwargs)["return"]["todAccess"]
        except Fault as e:
            return e

    def removeTodAccess(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeTodAccess(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listTodAccess(self, returnedTags={'name': '', 'description': '', 'ownerIdName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            ownerIdName: String(value), optional
        :param returnedTags: 
            name: String128(value), optional
            description: String255(value), optional
            ownerIdName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listTodAccess({'name': '%', 'description': '%', 'ownerIdName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["todAccess"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addVoiceMailPilot(self, dirn="", description=None, cssName=None, isDefault=""):
        """
        :param voiceMailPilot: 
            dirn: String255(value)
            description: String50(value), optional
            cssName: optional
                _value_1: String(value)
            isDefault: boolean(value)
        """
        try:
            return self.client.addVoiceMailPilot({"dirn": dirn, "description": description, "cssName": cssName, "isDefault": isDefault})["return"]
        except Fault as e:
            return e

    def updateVoiceMailPilot(self, dirn="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param dirn: String255(value)
        :param cssName: optional
            _value_1: String(value)
        :param newDirn: String255(value), optional
        :param description: String50(value), optional
        :param newCssName: optional
            _value_1: String(value)
        :param isDefault: boolean(value), optional
        """
        try:
            return self.client.updateVoiceMailPilot(dirn=dirn, **kwargs)["return"]
        except Fault as e:
            return e

    def getVoiceMailPilot(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param dirn: String255(value)
        :param cssName: optional
            _value_1: String(value)
        :param returnedTags: optional
            dirn: String255(value), optional
            description: String50(value), optional
            cssName: optional
                _value_1: String(value)
            isDefault: boolean(value), optional
        """
        try:
            return self.client.getVoiceMailPilot(**kwargs)["return"]["voiceMailPilot"]
        except Fault as e:
            return e

    def removeVoiceMailPilot(self, uuid="", dirn="", cssName=""):
        """
        :param uuid: XUUID(value), optional
        :param dirn: String255(value)
        :param cssName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeVoiceMailPilot(uuid=uuid, dirn=dirn, cssName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listVoiceMailPilot(self, returnedTags={'dirn': '', 'description': '', 'cssName': {'_value_1': ''}, 'isDefault': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            dirn: String(value), optional
            description: String(value), optional
            cssName: String(value), optional
        :param returnedTags: 
            dirn: String255(value), optional
            description: String50(value), optional
            cssName: optional
                _value_1: String(value)
            isDefault: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listVoiceMailPilot({'dirn': '%', 'description': '%', 'cssName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["voiceMailPilot"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addProcessNode(self, name="", description=None, mac=None, ipv6Name=None, lbmHubGroup=None, processNodeRole="", cupDomain=None):
        """
        :param processNode: 
            name: String255(value)
            description: String50(value), optional
            mac: XMacAddress(value), optional
            ipv6Name: String(value), optional
            lbmHubGroup: optional
                _value_1: String(value)
            processNodeRole: UnionType(value)
            cupDomain: String(value), optional
        """
        try:
            return self.client.addProcessNode({"name": name, "description": description, "mac": mac, "ipv6Name": ipv6Name, "lbmHubGroup": lbmHubGroup, "processNodeRole": processNodeRole, "cupDomain": cupDomain})["return"]
        except Fault as e:
            return e

    def updateProcessNode(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String255(value), optional
        :param description: String50(value), optional
        :param mac: XMacAddress(value), optional
        :param ipv6Name: String(value), optional
        :param lbmHubGroup: optional
            _value_1: String(value)
        :param cupDomain: String(value), optional
        """
        try:
            return self.client.updateProcessNode(**kwargs)["return"]
        except Fault as e:
            return e

    def getProcessNode(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String255(value), optional
            description: String50(value), optional
            mac: XMacAddress(value), optional
            ipv6Name: String(value), optional
            nodeUsage: UnionType(value), optional
            lbmHubGroup: optional
                _value_1: String(value)
            processNodeRole: UnionType(value), optional
            cupDomain: String(value), optional
        """
        try:
            return self.client.getProcessNode(**kwargs)["return"]["processNode"]
        except Fault as e:
            return e

    def removeProcessNode(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeProcessNode(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listProcessNode(self, returnedTags={'name': '', 'description': '', 'mac': '', 'ipv6Name': '', 'nodeUsage': '', 'lbmHubGroup': {'_value_1': ''}, 'processNodeRole': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            processNodeRole: String(value), optional
        :param returnedTags: 
            name: String255(value), optional
            description: String50(value), optional
            mac: XMacAddress(value), optional
            ipv6Name: String(value), optional
            nodeUsage: UnionType(value), optional
            lbmHubGroup: optional
                _value_1: String(value)
            processNodeRole: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listProcessNode({'name': '%', 'description': '%', 'processNodeRole': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["processNode"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCallerFilterList(self, name="", description=None, isAllowedType="", endUserIdName=None, members=None):
        """
        :param callerFilterList: 
            name: String50(value)
            description: String128(value), optional
            isAllowedType: boolean(value)
            endUserIdName: optional
                _value_1: String(value)
            members: optional
                    DnMask: String50(value)
                    callerFilterMask: UnionType(value)
        """
        try:
            return self.client.addCallerFilterList({"name": name, "description": description, "isAllowedType": isAllowedType, "endUserIdName": endUserIdName, "members": members})["return"]
        except Fault as e:
            return e

    def updateCallerFilterList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String128(value), optional
        :param removeMembers: optional
            member: 
                DnMask: String50(value)
                callerFilterMask: UnionType(value)
        :param addMembers: optional
            member: 
                DnMask: String50(value)
                callerFilterMask: UnionType(value)
        :param members: optional
            member: 
                DnMask: String50(value)
                callerFilterMask: UnionType(value)
        """
        try:
            return self.client.updateCallerFilterList(**kwargs)["return"]
        except Fault as e:
            return e

    def getCallerFilterList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String128(value), optional
            isAllowedType: boolean(value), optional
            endUserIdName: optional
                _value_1: String(value)
            members: optional
                    DnMask: String50(value), optional
                    callerFilterMask: UnionType(value), optional
        """
        try:
            return self.client.getCallerFilterList(**kwargs)["return"]["callerFilterList"]
        except Fault as e:
            return e

    def removeCallerFilterList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCallerFilterList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCallerFilterList(self, returnedTags={'name': '', 'description': '', 'isAllowedType': '', 'endUserIdName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String128(value), optional
            isAllowedType: boolean(value), optional
            endUserIdName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallerFilterList({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callerFilterList"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRoutePartition(self, name="", description=None, timeScheduleIdName=None, useOriginatingDeviceTimeZone=None, timeZone=None, partitionUsage=None):
        """
        :param routePartition: 
            name: String(value)
            description: String(value), optional
            timeScheduleIdName: optional
                _value_1: String(value)
            useOriginatingDeviceTimeZone: boolean(value), optional
            timeZone: UnionType(value), optional
            partitionUsage: UnionType(value), optional
        """
        try:
            return self.client.addRoutePartition({"name": name, "description": description, "timeScheduleIdName": timeScheduleIdName, "useOriginatingDeviceTimeZone": useOriginatingDeviceTimeZone, "timeZone": timeZone, "partitionUsage": partitionUsage})["return"]
        except Fault as e:
            return e

    def updateRoutePartition(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param timeScheduleIdName: optional
            _value_1: String(value)
        :param useOriginatingDeviceTimeZone: boolean(value), optional
        :param timeZone: UnionType(value), optional
        """
        try:
            return self.client.updateRoutePartition(**kwargs)["return"]
        except Fault as e:
            return e

    def getRoutePartition(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            dialPlanWizardGenId: UnionType(value), optional
            timeScheduleIdName: optional
                _value_1: String(value)
            useOriginatingDeviceTimeZone: boolean(value), optional
            timeZone: UnionType(value), optional
            partitionUsage: UnionType(value), optional
        """
        try:
            return self.client.getRoutePartition(**kwargs)["return"]["routePartition"]
        except Fault as e:
            return e

    def removeRoutePartition(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRoutePartition(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRoutePartition(self, returnedTags={'name': '', 'description': '', 'dialPlanWizardGenId': '', 'timeScheduleIdName': {'_value_1': ''}, 'useOriginatingDeviceTimeZone': '', 'timeZone': '', 'partitionUsage': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            dialPlanWizardGenId: UnionType(value), optional
            timeScheduleIdName: optional
                _value_1: String(value)
            useOriginatingDeviceTimeZone: boolean(value), optional
            timeZone: UnionType(value), optional
            partitionUsage: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRoutePartition({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routePartition"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def restartRoutePartition(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartRoutePartition(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyRoutePartition(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyRoutePartition(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCss(self, description=None, members=None, partitionUsage=None, name=""):
        """
        :param css: 
            description: String50(value), optional
            members: optional
                    index: UnionType(value)
            partitionUsage: UnionType(value), optional
            name: UniqueString50(value)
        """
        try:
            return self.client.addCss({"description": description, "members": members, "partitionUsage": partitionUsage, "name": name})["return"]
        except Fault as e:
            return e

    def updateCss(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param description: String50(value), optional
        :param removeMembers: optional
            member: optional
                    _value_1: String(value)
                index: UnionType(value)
        :param addMembers: optional
            member: optional
                    _value_1: String(value)
                index: UnionType(value)
        :param members: optional
            member: optional
                    _value_1: String(value)
                index: UnionType(value)
        :param newName: UniqueString50(value), optional
        """
        try:
            return self.client.updateCss(**kwargs)["return"]
        except Fault as e:
            return e

    def getCss(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            description: String50(value), optional
            clause: String1024(value), optional
            dialPlanWizardGenId: UnionType(value), optional
            members: optional
                    index: UnionType(value), optional
            partitionUsage: UnionType(value), optional
            name: UniqueString50(value), optional
        """
        try:
            return self.client.getCss(**kwargs)["return"]["css"]
        except Fault as e:
            return e

    def removeCss(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCss(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCss(self, returnedTags={'description': '', 'clause': '', 'dialPlanWizardGenId': '', 'partitionUsage': '', 'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            description: String(value), optional
            partitionUsage: String(value), optional
            name: String(value), optional
        :param returnedTags: 
            description: String50(value), optional
            clause: String1024(value), optional
            dialPlanWizardGenId: UnionType(value), optional
            partitionUsage: UnionType(value), optional
            name: UniqueString50(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCss({'description': '%', 'partitionUsage': '%', 'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["css"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def updateCallManager(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String50(value), optional
        :param autoRegistration: optional
            startDn: String255(value), optional
            endDn: String255(value), optional
            nextDn: String255(value), optional
            routePartitionName: optional
                _value_1: String(value)
            e164Mask: String50(value), optional
            autoRegistrationEnabled: boolean(value), optional
            universalDeviceTemplate: optional
                _value_1: String(value)
            lineTemplate: optional
                _value_1: String(value)
        :param ports: optional
            ethernetPhonePort: UnionType(value)
            mgcpPorts: 
                listen: UnionType(value)
                keepAlive: UnionType(value)
            sipPorts: 
                sipPhonePort: UnionType(value)
                sipPhoneSecurePort: UnionType(value)
        :param lbmGroup: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCallManager(**kwargs)["return"]
        except Fault as e:
            return e

    def getCallManager(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String50(value), optional
            autoRegistration: optional
                startDn: String255(value), optional
                endDn: String255(value), optional
                nextDn: String255(value), optional
                    _value_1: String(value)
                e164Mask: String50(value), optional
                autoRegistrationEnabled: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
            ports: optional
                ethernetPhonePort: UnionType(value), optional
                    listen: UnionType(value), optional
                    keepAlive: UnionType(value), optional
                    sipPhonePort: UnionType(value), optional
                    sipPhoneSecurePort: UnionType(value), optional
            processNodeName: optional
                _value_1: String(value)
            lbmGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCallManager(**kwargs)["return"]["callManager"]
        except Fault as e:
            return e

    def listCallManager(self, returnedTags={'name': '', 'description': '', 'autoRegistration': {'startDn': '', 'endDn': '', 'nextDn': '', 'e164Mask': '', 'autoRegistrationEnabled': ''}, 'processNodeName': {'_value_1': ''}, 'lbmGroup': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String50(value), optional
            autoRegistration: optional
                startDn: String255(value), optional
                endDn: String255(value), optional
                nextDn: String255(value), optional
                    _value_1: String(value)
                e164Mask: String50(value), optional
                autoRegistrationEnabled: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
            processNodeName: optional
                _value_1: String(value)
            lbmGroup: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallManager({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callManager"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCallManager(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCallManager(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCallManager(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCallManager(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCallManager(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCallManager(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addMediaResourceGroup(self, name="", description=None, multicast="", members=""):
        """
        :param mediaResourceGroup: 
            name: String50(value)
            description: String50(value), optional
            multicast: boolean(value)
            members: 
        """
        try:
            return self.client.addMediaResourceGroup({"name": name, "description": description, "multicast": multicast, "members": members})["return"]
        except Fault as e:
            return e

    def updateMediaResourceGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String50(value), optional
        :param multicast: boolean(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
        :param members: optional
            member: 
                    _value_1: String(value)
        """
        try:
            return self.client.updateMediaResourceGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getMediaResourceGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String50(value), optional
            multicast: boolean(value), optional
            members: optional
        """
        try:
            return self.client.getMediaResourceGroup(**kwargs)["return"]["mediaResourceGroup"]
        except Fault as e:
            return e

    def removeMediaResourceGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMediaResourceGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listMediaResourceGroup(self, returnedTags={'name': '', 'description': '', 'multicast': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String50(value), optional
            multicast: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMediaResourceGroup({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mediaResourceGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addMediaResourceList(self, name="", members=""):
        """
        :param mediaResourceList: 
            name: String50(value)
            members: 
                    order: UnionType(value)
        """
        try:
            return self.client.addMediaResourceList({"name": name, "members": members})["return"]
        except Fault as e:
            return e

    def updateMediaResourceList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
                order: UnionType(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
                order: UnionType(value)
        :param members: optional
            member: 
                    _value_1: String(value)
                order: UnionType(value)
        """
        try:
            return self.client.updateMediaResourceList(**kwargs)["return"]
        except Fault as e:
            return e

    def getMediaResourceList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            clause: String1024(value), optional
            members: optional
                    order: UnionType(value), optional
        """
        try:
            return self.client.getMediaResourceList(**kwargs)["return"]["mediaResourceList"]
        except Fault as e:
            return e

    def removeMediaResourceList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMediaResourceList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listMediaResourceList(self, returnedTags={'name': '', 'clause': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            clause: String1024(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMediaResourceList({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mediaResourceList"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRegion(self, name="", relatedRegions=None, defaultCodec=None):
        """
        :param region: 
            name: String50(value)
            relatedRegions: optional
                    bandwidth: String(value)
                    videoBandwidth: UnionType(value)
                    lossyNetwork: UnionType(value), optional
                    immersiveVideoBandwidth: UnionType(value)
            defaultCodec: String(value), optional
        """
        try:
            return self.client.addRegion({"name": name, "relatedRegions": relatedRegions, "defaultCodec": defaultCodec})["return"]
        except Fault as e:
            return e

    def updateRegion(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param relatedRegions: optional
            relatedRegion: 
                    _value_1: String(value)
                bandwidth: String(value)
                videoBandwidth: UnionType(value)
                lossyNetwork: UnionType(value), optional
                    _value_1: String(value)
                immersiveVideoBandwidth: UnionType(value)
        """
        try:
            return self.client.updateRegion(**kwargs)["return"]
        except Fault as e:
            return e

    def getRegion(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            relatedRegions: optional
                    bandwidth: String(value), optional
                    videoBandwidth: UnionType(value), optional
                    lossyNetwork: UnionType(value), optional
                    immersiveVideoBandwidth: UnionType(value), optional
            defaultCodec: String(value), optional
        """
        try:
            return self.client.getRegion(**kwargs)["return"]["region"]
        except Fault as e:
            return e

    def removeRegion(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRegion(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRegion(self, returnedTags={'name': '', 'defaultCodec': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            defaultCodec: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRegion({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["region"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def restartRegion(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartRegion(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyRegion(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyRegion(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addAarGroup(self, name=""):
        """
        :param aarGroup: 
            name: String32(value)
        """
        try:
            return self.client.addAarGroup({"name": name})["return"]
        except Fault as e:
            return e

    def updateAarGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String32(value), optional
        """
        try:
            return self.client.updateAarGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getAarGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String32(value), optional
            relatedGroups: optional
                    prefixDigit: String32(value), optional
        """
        try:
            return self.client.getAarGroup(**kwargs)["return"]["aarGroup"]
        except Fault as e:
            return e

    def removeAarGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeAarGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listAarGroup(self, returnedTags={'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String32(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listAarGroup({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["aarGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addPhysicalLocation(self, name="", description=None):
        """
        :param physicalLocation: 
            name: String(value)
            description: String(value), optional
        """
        try:
            return self.client.addPhysicalLocation({"name": name, "description": description})["return"]
        except Fault as e:
            return e

    def updatePhysicalLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        """
        try:
            return self.client.updatePhysicalLocation(**kwargs)["return"]
        except Fault as e:
            return e

    def getPhysicalLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
        """
        try:
            return self.client.getPhysicalLocation(**kwargs)["return"]["physicalLocation"]
        except Fault as e:
            return e

    def removePhysicalLocation(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePhysicalLocation(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPhysicalLocation(self, returnedTags={'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPhysicalLocation({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["physicalLocation"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRouteGroup(self, distributionAlgorithm="", members="", name=""):
        """
        :param routeGroup: 
            distributionAlgorithm: UnionType(value)
            members: 
                    deviceSelectionOrder: UnionType(value)
                    port: UnionType(value)
            name: String(value)
        """
        try:
            return self.client.addRouteGroup({"distributionAlgorithm": distributionAlgorithm, "members": members, "name": name})["return"]
        except Fault as e:
            return e

    def updateRouteGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param distributionAlgorithm: UnionType(value), optional
        :param removeMembers: optional
            member: 
                deviceSelectionOrder: UnionType(value)
                    _value_1: String(value)
                port: UnionType(value)
        :param addMembers: optional
            member: 
                deviceSelectionOrder: UnionType(value)
                    _value_1: String(value)
                port: UnionType(value)
        :param members: optional
            member: 
                deviceSelectionOrder: UnionType(value)
                    _value_1: String(value)
                port: UnionType(value)
        :param newName: String(value), optional
        """
        try:
            return self.client.updateRouteGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getRouteGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            dialPlanWizardGenld: UnionType(value), optional
            distributionAlgorithm: UnionType(value), optional
            members: optional
                    deviceSelectionOrder: UnionType(value), optional
                    dialPlanWizardGenId: UnionType(value), optional
                    port: UnionType(value), optional
            name: String(value), optional
        """
        try:
            return self.client.getRouteGroup(**kwargs)["return"]["routeGroup"]
        except Fault as e:
            return e

    def removeRouteGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRouteGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRouteGroup(self, returnedTags={'dialPlanWizardGenld': '', 'distributionAlgorithm': '', 'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            dialPlanWizardGenld: UnionType(value), optional
            distributionAlgorithm: UnionType(value), optional
            name: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRouteGroup({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routeGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDevicePool(self, name="", autoSearchSpaceName=None, dateTimeSettingName="", callManagerGroupName="", mediaResourceListName=None, regionName="", networkLocale=None, srstName="", connectionMonitorDuration=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName="", locationName=None, mobilityCssName=None, physicalLocationName=None, deviceMobilityGroupName=None, revertPriority=None, singleButtonBarge=None, joinAcrossLines=None, cgpnTransformationCssName=None, cdpnTransformationCssName=None, localRouteGroupName=None, geoLocationName=None, geoLocationFilterName=None, callingPartyNationalPrefix=None, callingPartyInternationalPrefix=None, callingPartyUnknownPrefix=None, callingPartySubscriberPrefix=None, adjunctCallingSearchSpace=None, callingPartyNationalStripDigits=None, callingPartyInternationalStripDigits=None, callingPartyUnknownStripDigits=None, callingPartySubscriberStripDigits=None, callingPartyNationalTransformationCssName=None, callingPartyInternationalTransformationCssName=None, callingPartyUnknownTransformationCssName=None, callingPartySubscriberTransformationCssName=None, calledPartyNationalPrefix=None, calledPartyInternationalPrefix=None, calledPartyUnknownPrefix=None, calledPartySubscriberPrefix=None, calledPartyNationalStripDigits=None, calledPartyInternationalStripDigits=None, calledPartyUnknownStripDigits=None, calledPartySubscriberStripDigits=None, calledPartyNationalTransformationCssName=None, calledPartyInternationalTransformationCssName=None, calledPartyUnknownTransformationCssName=None, calledPartySubscriberTransformationCssName=None, imeEnrolledPatternGroupName=None, cntdPnTransformationCssName=None, localRouteGroup=None, redirectingPartyTransformationCSS=None, callingPartyTransformationCSS=None, wirelessLanProfileGroup=None, elinGroup=None):
        """
        :param devicePool: 
            name: UniqueString50(value)
            autoSearchSpaceName: optional
                _value_1: String(value)
            dateTimeSettingName: 
                _value_1: String(value)
            callManagerGroupName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            regionName: 
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            srstName: 
                _value_1: String(value)
            connectionMonitorDuration: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: 
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mobilityCssName: optional
                _value_1: String(value)
            physicalLocationName: optional
                _value_1: String(value)
            deviceMobilityGroupName: optional
                _value_1: String(value)
            revertPriority: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            localRouteGroupName: optional
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            adjunctCallingSearchSpace: optional
                _value_1: String(value)
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            imeEnrolledPatternGroupName: optional
                _value_1: String(value)
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            localRouteGroup: optional
                name: String(value), optional
                value: String(value), optional
            redirectingPartyTransformationCSS: optional
                _value_1: String(value)
            callingPartyTransformationCSS: optional
                _value_1: String(value)
            wirelessLanProfileGroup: optional
                _value_1: String(value)
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.addDevicePool({"name": name, "autoSearchSpaceName": autoSearchSpaceName, "dateTimeSettingName": dateTimeSettingName, "callManagerGroupName": callManagerGroupName, "mediaResourceListName": mediaResourceListName, "regionName": regionName, "networkLocale": networkLocale, "srstName": srstName, "connectionMonitorDuration": connectionMonitorDuration, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "locationName": locationName, "mobilityCssName": mobilityCssName, "physicalLocationName": physicalLocationName, "deviceMobilityGroupName": deviceMobilityGroupName, "revertPriority": revertPriority, "singleButtonBarge": singleButtonBarge, "joinAcrossLines": joinAcrossLines, "cgpnTransformationCssName": cgpnTransformationCssName, "cdpnTransformationCssName": cdpnTransformationCssName, "localRouteGroupName": localRouteGroupName, "geoLocationName": geoLocationName, "geoLocationFilterName": geoLocationFilterName, "callingPartyNationalPrefix": callingPartyNationalPrefix, "callingPartyInternationalPrefix": callingPartyInternationalPrefix, "callingPartyUnknownPrefix": callingPartyUnknownPrefix, "callingPartySubscriberPrefix": callingPartySubscriberPrefix, "adjunctCallingSearchSpace": adjunctCallingSearchSpace, "callingPartyNationalStripDigits": callingPartyNationalStripDigits, "callingPartyInternationalStripDigits": callingPartyInternationalStripDigits, "callingPartyUnknownStripDigits": callingPartyUnknownStripDigits, "callingPartySubscriberStripDigits": callingPartySubscriberStripDigits, "callingPartyNationalTransformationCssName": callingPartyNationalTransformationCssName, "callingPartyInternationalTransformationCssName": callingPartyInternationalTransformationCssName, "callingPartyUnknownTransformationCssName": callingPartyUnknownTransformationCssName, "callingPartySubscriberTransformationCssName": callingPartySubscriberTransformationCssName, "calledPartyNationalPrefix": calledPartyNationalPrefix, "calledPartyInternationalPrefix": calledPartyInternationalPrefix, "calledPartyUnknownPrefix": calledPartyUnknownPrefix, "calledPartySubscriberPrefix": calledPartySubscriberPrefix, "calledPartyNationalStripDigits": calledPartyNationalStripDigits, "calledPartyInternationalStripDigits": calledPartyInternationalStripDigits, "calledPartyUnknownStripDigits": calledPartyUnknownStripDigits, "calledPartySubscriberStripDigits": calledPartySubscriberStripDigits, "calledPartyNationalTransformationCssName": calledPartyNationalTransformationCssName, "calledPartyInternationalTransformationCssName": calledPartyInternationalTransformationCssName, "calledPartyUnknownTransformationCssName": calledPartyUnknownTransformationCssName, "calledPartySubscriberTransformationCssName": calledPartySubscriberTransformationCssName, "imeEnrolledPatternGroupName": imeEnrolledPatternGroupName, "cntdPnTransformationCssName": cntdPnTransformationCssName, "localRouteGroup": localRouteGroup, "redirectingPartyTransformationCSS": redirectingPartyTransformationCSS, "callingPartyTransformationCSS": callingPartyTransformationCSS, "wirelessLanProfileGroup": wirelessLanProfileGroup, "elinGroup": elinGroup})["return"]
        except Fault as e:
            return e

    def updateDevicePool(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString50(value), optional
        :param autoSearchSpaceName: optional
            _value_1: String(value)
        :param dateTimeSettingName: optional
            _value_1: String(value)
        :param callManagerGroupName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param regionName: optional
            _value_1: String(value)
        :param networkLocale: UnionType(value), optional
        :param srstName: optional
            _value_1: String(value)
        :param connectionMonitorDuration: UnionType(value), optional
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param mobilityCssName: optional
            _value_1: String(value)
        :param physicalLocationName: optional
            _value_1: String(value)
        :param deviceMobilityGroupName: optional
            _value_1: String(value)
        :param revertPriority: UnionType(value), optional
        :param singleButtonBarge: UnionType(value), optional
        :param joinAcrossLines: UnionType(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param localRouteGroupName: optional
            _value_1: String(value)
        :param geoLocationName: optional
            _value_1: String(value)
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param callingPartyNationalPrefix: String16(value), optional
        :param callingPartyInternationalPrefix: String16(value), optional
        :param callingPartyUnknownPrefix: String16(value), optional
        :param callingPartySubscriberPrefix: String16(value), optional
        :param adjunctCallingSearchSpace: optional
            _value_1: String(value)
        :param callingPartyNationalStripDigits: UnionType(value), optional
        :param callingPartyInternationalStripDigits: UnionType(value), optional
        :param callingPartyUnknownStripDigits: UnionType(value), optional
        :param callingPartySubscriberStripDigits: UnionType(value), optional
        :param callingPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param callingPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyNationalPrefix: String16(value), optional
        :param calledPartyInternationalPrefix: String16(value), optional
        :param calledPartyUnknownPrefix: String16(value), optional
        :param calledPartySubscriberPrefix: String16(value), optional
        :param calledPartyNationalStripDigits: UnionType(value), optional
        :param calledPartyInternationalStripDigits: UnionType(value), optional
        :param calledPartyUnknownStripDigits: UnionType(value), optional
        :param calledPartySubscriberStripDigits: UnionType(value), optional
        :param calledPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param calledPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param imeEnrolledPatternGroupName: optional
            _value_1: String(value)
        :param cntdPnTransformationCssName: optional
            _value_1: String(value)
        :param localRouteGroup: optional
            name: String(value), optional
            value: String(value), optional
        :param redirectingPartyTransformationCSS: optional
            _value_1: String(value)
        :param callingPartyTransformationCSS: optional
            _value_1: String(value)
        :param wirelessLanProfileGroup: optional
            _value_1: String(value)
        :param elinGroup: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateDevicePool(**kwargs)["return"]
        except Fault as e:
            return e

    def getDevicePool(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString50(value), optional
            autoSearchSpaceName: optional
                _value_1: String(value)
            dateTimeSettingName: optional
                _value_1: String(value)
            callManagerGroupName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            regionName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            srstName: optional
                _value_1: String(value)
            connectionMonitorDuration: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mobilityCssName: optional
                _value_1: String(value)
            physicalLocationName: optional
                _value_1: String(value)
            deviceMobilityGroupName: optional
                _value_1: String(value)
            revertPriority: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            localRouteGroupName: optional
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            adjunctCallingSearchSpace: optional
                _value_1: String(value)
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            imeEnrolledPatternGroupName: optional
                _value_1: String(value)
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            localRouteGroup: optional
                name: String(value), optional
                value: String(value), optional
            redirectingPartyTransformationCSS: optional
                _value_1: String(value)
            callingPartyTransformationCSS: optional
                _value_1: String(value)
            wirelessLanProfileGroup: optional
                _value_1: String(value)
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.getDevicePool(**kwargs)["return"]["devicePool"]
        except Fault as e:
            return e

    def removeDevicePool(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDevicePool(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDevicePool(self, returnedTags={'name': '', 'autoSearchSpaceName': {'_value_1': ''}, 'dateTimeSettingName': {'_value_1': ''}, 'callManagerGroupName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'regionName': {'_value_1': ''}, 'networkLocale': '', 'srstName': {'_value_1': ''}, 'connectionMonitorDuration': '', 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'mobilityCssName': {'_value_1': ''}, 'physicalLocationName': {'_value_1': ''}, 'deviceMobilityGroupName': {'_value_1': ''}, 'revertPriority': '', 'singleButtonBarge': '', 'joinAcrossLines': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'cdpnTransformationCssName': {'_value_1': ''}, 'localRouteGroupName': {'_value_1': ''}, 'geoLocationName': {'_value_1': ''}, 'geoLocationFilterName': {'_value_1': ''}, 'callingPartyNationalPrefix': '', 'callingPartyInternationalPrefix': '', 'callingPartyUnknownPrefix': '', 'callingPartySubscriberPrefix': '', 'adjunctCallingSearchSpace': {'_value_1': ''}, 'callingPartyNationalStripDigits': '', 'callingPartyInternationalStripDigits': '', 'callingPartyUnknownStripDigits': '', 'callingPartySubscriberStripDigits': '', 'callingPartyNationalTransformationCssName': {'_value_1': ''}, 'callingPartyInternationalTransformationCssName': {'_value_1': ''}, 'callingPartyUnknownTransformationCssName': {'_value_1': ''}, 'callingPartySubscriberTransformationCssName': {'_value_1': ''}, 'calledPartyNationalPrefix': '', 'calledPartyInternationalPrefix': '', 'calledPartyUnknownPrefix': '', 'calledPartySubscriberPrefix': '', 'calledPartyNationalStripDigits': '', 'calledPartyInternationalStripDigits': '', 'calledPartyUnknownStripDigits': '', 'calledPartySubscriberStripDigits': '', 'calledPartyNationalTransformationCssName': {'_value_1': ''}, 'calledPartyInternationalTransformationCssName': {'_value_1': ''}, 'calledPartyUnknownTransformationCssName': {'_value_1': ''}, 'calledPartySubscriberTransformationCssName': {'_value_1': ''}, 'imeEnrolledPatternGroupName': {'_value_1': ''}, 'localRouteGroup': {'name': '', 'value': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            callManagerGroupName: String(value), optional
            regionName: String(value), optional
        :param returnedTags: 
            name: UniqueString50(value), optional
            autoSearchSpaceName: optional
                _value_1: String(value)
            dateTimeSettingName: optional
                _value_1: String(value)
            callManagerGroupName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            regionName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            srstName: optional
                _value_1: String(value)
            connectionMonitorDuration: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mobilityCssName: optional
                _value_1: String(value)
            physicalLocationName: optional
                _value_1: String(value)
            deviceMobilityGroupName: optional
                _value_1: String(value)
            revertPriority: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            localRouteGroupName: optional
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            adjunctCallingSearchSpace: optional
                _value_1: String(value)
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            imeEnrolledPatternGroupName: optional
                _value_1: String(value)
            localRouteGroup: optional
                name: String(value), optional
                value: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDevicePool({'name': '%', 'callManagerGroupName': '%', 'regionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["devicePool"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetDevicePool(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetDevicePool(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartDevicePool(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartDevicePool(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyDevicePool(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyDevicePool(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addDeviceMobilityGroup(self, name="", description=None):
        """
        :param deviceMobilityGroup: 
            name: String(value)
            description: String(value), optional
        """
        try:
            return self.client.addDeviceMobilityGroup({"name": name, "description": description})["return"]
        except Fault as e:
            return e

    def updateDeviceMobilityGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        """
        try:
            return self.client.updateDeviceMobilityGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getDeviceMobilityGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
        """
        try:
            return self.client.getDeviceMobilityGroup(**kwargs)["return"]["deviceMobilityGroup"]
        except Fault as e:
            return e

    def removeDeviceMobilityGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDeviceMobilityGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDeviceMobilityGroup(self, returnedTags={'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDeviceMobilityGroup({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["deviceMobilityGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addLocation(self, name="", relatedLocations=None, withinAudioBandwidth="", withinVideoBandwidth="", withinImmersiveKbits="", betweenLocations=""):
        """
        :param location: 
            name: String50(value)
            relatedLocations: optional
                    rsvpSetting: UnionType(value)
            withinAudioBandwidth: UnionType(value)
            withinVideoBandwidth: UnionType(value)
            withinImmersiveKbits: UnionType(value)
            betweenLocations: 
                    weight: UnionType(value)
                    audioBandwidth: UnionType(value)
                    videoBandwidth: UnionType(value)
                    immersiveBandwidth: UnionType(value)
        """
        try:
            return self.client.addLocation({"name": name, "relatedLocations": relatedLocations, "withinAudioBandwidth": withinAudioBandwidth, "withinVideoBandwidth": withinVideoBandwidth, "withinImmersiveKbits": withinImmersiveKbits, "betweenLocations": betweenLocations})["return"]
        except Fault as e:
            return e

    def updateLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param relatedLocations: optional
            relatedLocation: 
                    _value_1: String(value)
                rsvpSetting: UnionType(value)
        :param withinAudioBandwidth: UnionType(value), optional
        :param withinVideoBandwidth: UnionType(value), optional
        :param withinImmersiveKbits: UnionType(value), optional
        :param betweenLocations: optional
            betweenLocation: 
                    _value_1: String(value)
                weight: UnionType(value)
                audioBandwidth: UnionType(value)
                videoBandwidth: UnionType(value)
                immersiveBandwidth: UnionType(value)
        """
        try:
            return self.client.updateLocation(**kwargs)["return"]
        except Fault as e:
            return e

    def getLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            id: UnionType(value), optional
            relatedLocations: optional
                    rsvpSetting: UnionType(value), optional
            withinAudioBandwidth: UnionType(value), optional
            withinVideoBandwidth: UnionType(value), optional
            withinImmersiveKbits: UnionType(value), optional
            betweenLocations: optional
                    weight: UnionType(value), optional
                    audioBandwidth: UnionType(value), optional
                    videoBandwidth: UnionType(value), optional
                    immersiveBandwidth: UnionType(value), optional
        """
        try:
            return self.client.getLocation(**kwargs)["return"]["location"]
        except Fault as e:
            return e

    def removeLocation(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeLocation(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listLocation(self, returnedTags={'name': '', 'id': '', 'withinAudioBandwidth': '', 'withinVideoBandwidth': '', 'withinImmersiveKbits': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            id: UnionType(value), optional
            withinAudioBandwidth: UnionType(value), optional
            withinVideoBandwidth: UnionType(value), optional
            withinImmersiveKbits: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listLocation({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["location"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addSoftKeyTemplate(self, name="", description="", baseSoftkeyTemplateName="", isDefault=None):
        """
        :param softKeyTemplate: 
            name: String100(value)
            description: String100(value)
            baseSoftkeyTemplateName: 
                _value_1: String(value)
            isDefault: boolean(value), optional
        """
        try:
            return self.client.addSoftKeyTemplate({"name": name, "description": description, "baseSoftkeyTemplateName": baseSoftkeyTemplateName, "isDefault": isDefault})["return"]
        except Fault as e:
            return e

    def updateSoftKeyTemplate(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String100(value), optional
        :param description: String100(value), optional
        :param isDefault: boolean(value), optional
        """
        try:
            return self.client.updateSoftKeyTemplate(**kwargs)["return"]
        except Fault as e:
            return e

    def getSoftKeyTemplate(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String100(value), optional
            description: String100(value), optional
            baseSoftkeyTemplateName: optional
                _value_1: String(value)
            isDefault: boolean(value), optional
            applications: optional
                application: String50(value), optional
        """
        try:
            return self.client.getSoftKeyTemplate(**kwargs)["return"]["softKeyTemplate"]
        except Fault as e:
            return e

    def removeSoftKeyTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSoftKeyTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSoftKeyTemplate(self, returnedTags={'name': '', 'description': '', 'baseSoftkeyTemplateName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String100(value), optional
            description: String100(value), optional
            baseSoftkeyTemplateName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSoftKeyTemplate({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["softKeyTemplate"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def restartSoftKeyTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartSoftKeyTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applySoftKeyTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applySoftKeyTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addTranscoder(self, name="", description=None, product="", subUnit=None, devicePoolName="", commonDeviceConfigName=None, loadInformation="", vendorConfig=None, isTrustedRelayPoint=None, maximumCapacity=None):
        """
        :param transcoder: 
            name: UniqueString255(value)
            description: String128(value), optional
            product: UnionType(value)
            subUnit: UnionType(value), optional
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            loadInformation: 
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            isTrustedRelayPoint: boolean(value), optional
            maximumCapacity: UnionType(value), optional
        """
        try:
            return self.client.addTranscoder({"name": name, "description": description, "product": product, "subUnit": subUnit, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "isTrustedRelayPoint": isTrustedRelayPoint, "maximumCapacity": maximumCapacity})["return"]
        except Fault as e:
            return e

    def updateTranscoder(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString255(value), optional
        :param description: String128(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param isTrustedRelayPoint: boolean(value), optional
        :param maximumCapacity: UnionType(value), optional
        """
        try:
            return self.client.updateTranscoder(**kwargs)["return"]
        except Fault as e:
            return e

    def getTranscoder(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString255(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            subUnit: UnionType(value), optional
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            isTrustedRelayPoint: boolean(value), optional
            maximumCapacity: UnionType(value), optional
        """
        try:
            return self.client.getTranscoder(**kwargs)["return"]["transcoder"]
        except Fault as e:
            return e

    def removeTranscoder(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeTranscoder(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listTranscoder(self, returnedTags={'name': '', 'description': '', 'product': '', 'subUnit': '', 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'isTrustedRelayPoint': '', 'maximumCapacity': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: UniqueString255(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            subUnit: UnionType(value), optional
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            isTrustedRelayPoint: boolean(value), optional
            maximumCapacity: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listTranscoder({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["transcoder"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetTranscoder(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetTranscoder(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyTranscoder(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyTranscoder(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCommonDeviceConfig(self, name="", softkeyTemplateName=None, userLocale=None, networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, mlppDomainId=None, mlppIndicationStatus=None, useTrustedRelayPoint=None, preemption=None, ipAddressingMode=None, ipAddressingModePreferenceControl=None, allowAutoConfigurationForPhones=None, useImeForOutboundCalls=None, confidentialAccess=None, allowDuplicateAddressDetection=None, acceptRedirectMessages=None, replyMulticastEchoRequest=None):
        """
        :param commonDeviceConfig: 
            name: UniqueString50(value)
            softkeyTemplateName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            useTrustedRelayPoint: boolean(value), optional
            preemption: UnionType(value), optional
            ipAddressingMode: UnionType(value), optional
            ipAddressingModePreferenceControl: UnionType(value), optional
            allowAutoConfigurationForPhones: UnionType(value), optional
            useImeForOutboundCalls: UnionType(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            allowDuplicateAddressDetection: UnionType(value), optional
            acceptRedirectMessages: UnionType(value), optional
            replyMulticastEchoRequest: UnionType(value), optional
        """
        try:
            return self.client.addCommonDeviceConfig({"name": name, "softkeyTemplateName": softkeyTemplateName, "userLocale": userLocale, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "useTrustedRelayPoint": useTrustedRelayPoint, "preemption": preemption, "ipAddressingMode": ipAddressingMode, "ipAddressingModePreferenceControl": ipAddressingModePreferenceControl, "allowAutoConfigurationForPhones": allowAutoConfigurationForPhones, "useImeForOutboundCalls": useImeForOutboundCalls, "confidentialAccess": confidentialAccess, "allowDuplicateAddressDetection": allowDuplicateAddressDetection, "acceptRedirectMessages": acceptRedirectMessages, "replyMulticastEchoRequest": replyMulticastEchoRequest})["return"]
        except Fault as e:
            return e

    def updateCommonDeviceConfig(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString50(value), optional
        :param softkeyTemplateName: optional
            _value_1: String(value)
        :param userLocale: UnionType(value), optional
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param useTrustedRelayPoint: boolean(value), optional
        :param preemption: UnionType(value), optional
        :param ipAddressingMode: UnionType(value), optional
        :param ipAddressingModePreferenceControl: UnionType(value), optional
        :param allowAutoConfigurationForPhones: UnionType(value), optional
        :param useImeForOutboundCalls: UnionType(value), optional
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        :param allowDuplicateAddressDetection: UnionType(value), optional
        :param acceptRedirectMessages: UnionType(value), optional
        :param replyMulticastEchoRequest: UnionType(value), optional
        """
        try:
            return self.client.updateCommonDeviceConfig(**kwargs)["return"]
        except Fault as e:
            return e

    def getCommonDeviceConfig(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString50(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            useTrustedRelayPoint: boolean(value), optional
            preemption: UnionType(value), optional
            ipAddressingMode: UnionType(value), optional
            ipAddressingModePreferenceControl: UnionType(value), optional
            allowAutoConfigurationForPhones: UnionType(value), optional
            useImeForOutboundCalls: UnionType(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            allowDuplicateAddressDetection: UnionType(value), optional
            acceptRedirectMessages: UnionType(value), optional
            replyMulticastEchoRequest: UnionType(value), optional
        """
        try:
            return self.client.getCommonDeviceConfig(**kwargs)["return"]["commonDeviceConfig"]
        except Fault as e:
            return e

    def removeCommonDeviceConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCommonDeviceConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCommonDeviceConfig(self, returnedTags={'name': '', 'softkeyTemplateName': {'_value_1': ''}, 'userLocale': '', 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'mlppIndicationStatus': '', 'useTrustedRelayPoint': '', 'preemption': '', 'ipAddressingMode': '', 'ipAddressingModePreferenceControl': '', 'allowAutoConfigurationForPhones': '', 'useImeForOutboundCalls': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}, 'allowDuplicateAddressDetection': '', 'acceptRedirectMessages': '', 'replyMulticastEchoRequest': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: UniqueString50(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            mlppIndicationStatus: UnionType(value), optional
            useTrustedRelayPoint: boolean(value), optional
            preemption: UnionType(value), optional
            ipAddressingMode: UnionType(value), optional
            ipAddressingModePreferenceControl: UnionType(value), optional
            allowAutoConfigurationForPhones: UnionType(value), optional
            useImeForOutboundCalls: UnionType(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            allowDuplicateAddressDetection: UnionType(value), optional
            acceptRedirectMessages: UnionType(value), optional
            replyMulticastEchoRequest: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCommonDeviceConfig({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["commonDeviceConfig"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCommonDeviceConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCommonDeviceConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCommonDeviceConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCommonDeviceConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addResourcePriorityNamespace(self, namespace="", description=None, isDefault=None):
        """
        :param resourcePriorityNamespace: 
            namespace: String10(value)
            description: String(value), optional
            isDefault: boolean(value), optional
        """
        try:
            return self.client.addResourcePriorityNamespace({"namespace": namespace, "description": description, "isDefault": isDefault})["return"]
        except Fault as e:
            return e

    def updateResourcePriorityNamespace(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param namespace: String10(value), optional
        :param newNamespace: String10(value), optional
        :param description: String(value), optional
        :param isDefault: boolean(value), optional
        """
        try:
            return self.client.updateResourcePriorityNamespace(**kwargs)["return"]
        except Fault as e:
            return e

    def getResourcePriorityNamespace(self, **kwargs):
        """
        :param namespace: String10(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            namespace: String10(value), optional
            description: String(value), optional
            isDefault: boolean(value), optional
        """
        try:
            return self.client.getResourcePriorityNamespace(**kwargs)["return"]["resourcePriorityNamespace"]
        except Fault as e:
            return e

    def removeResourcePriorityNamespace(self, namespace="", uuid=""):
        """
        :param namespace: String10(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeResourcePriorityNamespace(namespace=namespace, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listResourcePriorityNamespace(self, returnedTags={'namespace': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            namespace: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            namespace: String10(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listResourcePriorityNamespace({'namespace': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["resourcePriorityNamespace"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetResourcePriorityNamespace(self, namespace="", uuid=""):
        """
        :param namespace: String10(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetResourcePriorityNamespace(namespace=namespace, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartResourcePriorityNamespace(self, namespace="", uuid=""):
        """
        :param namespace: String10(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartResourcePriorityNamespace(namespace=namespace, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyResourcePriorityNamespace(self, namespace="", uuid=""):
        """
        :param namespace: String10(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyResourcePriorityNamespace(namespace=namespace, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addResourcePriorityNamespaceList(self, name="", description=None, members=None):
        """
        :param resourcePriorityNamespaceList: 
            name: String50(value)
            description: String(value), optional
            members: optional
                    index: UnionType(value)
        """
        try:
            return self.client.addResourcePriorityNamespaceList({"name": name, "description": description, "members": members})["return"]
        except Fault as e:
            return e

    def updateResourcePriorityNamespaceList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
                index: UnionType(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
                index: UnionType(value)
        :param members: optional
            member: 
                    _value_1: String(value)
                index: UnionType(value)
        """
        try:
            return self.client.updateResourcePriorityNamespaceList(**kwargs)["return"]
        except Fault as e:
            return e

    def getResourcePriorityNamespaceList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String(value), optional
            members: optional
                    index: UnionType(value), optional
        """
        try:
            return self.client.getResourcePriorityNamespaceList(**kwargs)["return"]["resourcePriorityNamespaceList"]
        except Fault as e:
            return e

    def removeResourcePriorityNamespaceList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeResourcePriorityNamespaceList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listResourcePriorityNamespaceList(self, returnedTags={'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listResourcePriorityNamespaceList({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["resourcePriorityNamespaceList"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetResourcePriorityNamespaceList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetResourcePriorityNamespaceList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartResourcePriorityNamespaceList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartResourcePriorityNamespaceList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyResourcePriorityNamespaceList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyResourcePriorityNamespaceList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addDeviceMobility(self, name="", subNet="", subNetMaskSz="", members=""):
        """
        :param deviceMobility: 
            name: String(value)
            subNet: String(value)
            subNetMaskSz: UnionType(value)
            members: 
        """
        try:
            return self.client.addDeviceMobility({"name": name, "subNet": subNet, "subNetMaskSz": subNetMaskSz, "members": members})["return"]
        except Fault as e:
            return e

    def updateDeviceMobility(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param subNet: String(value), optional
        :param subNetMaskSz: UnionType(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
        :param members: optional
            member: 
                    _value_1: String(value)
        """
        try:
            return self.client.updateDeviceMobility(**kwargs)["return"]
        except Fault as e:
            return e

    def getDeviceMobility(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            subNet: String(value), optional
            subNetMaskSz: UnionType(value), optional
            members: optional
        """
        try:
            return self.client.getDeviceMobility(**kwargs)["return"]["deviceMobility"]
        except Fault as e:
            return e

    def removeDeviceMobility(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDeviceMobility(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDeviceMobility(self, returnedTags={'name': '', 'subNet': '', 'subNetMaskSz': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            subNet: String(value), optional
            subNetMaskSz: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            subNet: String(value), optional
            subNetMaskSz: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDeviceMobility({'name': '%', 'subNet': '%', 'subNetMaskSz': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["deviceMobility"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCmcInfo(self, code="", description=None):
        """
        :param cmcInfo: 
            code: String16(value)
            description: String50(value), optional
        """
        try:
            return self.client.addCmcInfo({"code": code, "description": description})["return"]
        except Fault as e:
            return e

    def updateCmcInfo(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param code: String16(value), optional
        :param newCode: String16(value), optional
        :param description: String50(value), optional
        """
        try:
            return self.client.updateCmcInfo(**kwargs)["return"]
        except Fault as e:
            return e

    def getCmcInfo(self, **kwargs):
        """
        :param code: String16(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            code: String16(value), optional
            description: String50(value), optional
        """
        try:
            return self.client.getCmcInfo(**kwargs)["return"]["cmcInfo"]
        except Fault as e:
            return e

    def removeCmcInfo(self, code="", uuid=""):
        """
        :param code: String16(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCmcInfo(code=code, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCmcInfo(self, returnedTags={'code': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            code: String(value), optional
        :param returnedTags: 
            code: String16(value), optional
            description: String50(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCmcInfo({'code': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["cmcInfo"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCredentialPolicy(self, name="", failedLogon="", resetFailedLogonAttempts="", lockoutDuration="", credChangeDuration="", credExpiresAfter="", minCredLength="", prevCredStoredNum="", inactiveDaysAllowed="", expiryWarningDays="", trivialCredCheck="", minCharsToChange=""):
        """
        :param credentialPolicy: 
            name: String255(value)
            failedLogon: UnionType(value)
            resetFailedLogonAttempts: UnionType(value)
            lockoutDuration: UnionType(value)
            credChangeDuration: UnionType(value)
            credExpiresAfter: UnionType(value)
            minCredLength: UnionType(value)
            prevCredStoredNum: UnionType(value)
            inactiveDaysAllowed: UnionType(value)
            expiryWarningDays: UnionType(value)
            trivialCredCheck: boolean(value)
            minCharsToChange: UnionType(value)
        """
        try:
            return self.client.addCredentialPolicy({"name": name, "failedLogon": failedLogon, "resetFailedLogonAttempts": resetFailedLogonAttempts, "lockoutDuration": lockoutDuration, "credChangeDuration": credChangeDuration, "credExpiresAfter": credExpiresAfter, "minCredLength": minCredLength, "prevCredStoredNum": prevCredStoredNum, "inactiveDaysAllowed": inactiveDaysAllowed, "expiryWarningDays": expiryWarningDays, "trivialCredCheck": trivialCredCheck, "minCharsToChange": minCharsToChange})["return"]
        except Fault as e:
            return e

    def updateCredentialPolicy(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String255(value), optional
        :param failedLogon: UnionType(value), optional
        :param resetFailedLogonAttempts: UnionType(value), optional
        :param lockoutDuration: UnionType(value), optional
        :param credChangeDuration: UnionType(value), optional
        :param credExpiresAfter: UnionType(value), optional
        :param minCredLength: UnionType(value), optional
        :param prevCredStoredNum: UnionType(value), optional
        :param inactiveDaysAllowed: UnionType(value), optional
        :param expiryWarningDays: UnionType(value), optional
        :param trivialCredCheck: boolean(value), optional
        :param minCharsToChange: UnionType(value), optional
        """
        try:
            return self.client.updateCredentialPolicy(**kwargs)["return"]
        except Fault as e:
            return e

    def getCredentialPolicy(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String255(value), optional
            failedLogon: UnionType(value), optional
            resetFailedLogonAttempts: UnionType(value), optional
            lockoutDuration: UnionType(value), optional
            credChangeDuration: UnionType(value), optional
            credExpiresAfter: UnionType(value), optional
            minCredLength: UnionType(value), optional
            prevCredStoredNum: UnionType(value), optional
            inactiveDaysAllowed: UnionType(value), optional
            expiryWarningDays: UnionType(value), optional
            trivialCredCheck: boolean(value), optional
            minCharsToChange: UnionType(value), optional
        """
        try:
            return self.client.getCredentialPolicy(**kwargs)["return"]["credentialPolicy"]
        except Fault as e:
            return e

    def removeCredentialPolicy(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCredentialPolicy(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCredentialPolicy(self, returnedTags={'name': '', 'failedLogon': '', 'resetFailedLogonAttempts': '', 'lockoutDuration': '', 'credChangeDuration': '', 'credExpiresAfter': '', 'minCredLength': '', 'prevCredStoredNum': '', 'inactiveDaysAllowed': '', 'expiryWarningDays': '', 'trivialCredCheck': '', 'minCharsToChange': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String255(value), optional
            failedLogon: UnionType(value), optional
            resetFailedLogonAttempts: UnionType(value), optional
            lockoutDuration: UnionType(value), optional
            credChangeDuration: UnionType(value), optional
            credExpiresAfter: UnionType(value), optional
            minCredLength: UnionType(value), optional
            prevCredStoredNum: UnionType(value), optional
            inactiveDaysAllowed: UnionType(value), optional
            expiryWarningDays: UnionType(value), optional
            trivialCredCheck: boolean(value), optional
            minCharsToChange: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCredentialPolicy({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["credentialPolicy"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addFacInfo(self, name="", code="", authorizationLevel=""):
        """
        :param facInfo: 
            name: String50(value)
            code: String16(value)
            authorizationLevel: UnionType(value)
        """
        try:
            return self.client.addFacInfo({"name": name, "code": code, "authorizationLevel": authorizationLevel})["return"]
        except Fault as e:
            return e

    def updateFacInfo(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param code: String16(value), optional
        :param authorizationLevel: UnionType(value), optional
        """
        try:
            return self.client.updateFacInfo(**kwargs)["return"]
        except Fault as e:
            return e

    def getFacInfo(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            code: String16(value), optional
            authorizationLevel: UnionType(value), optional
        """
        try:
            return self.client.getFacInfo(**kwargs)["return"]["facInfo"]
        except Fault as e:
            return e

    def removeFacInfo(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeFacInfo(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listFacInfo(self, returnedTags={'name': '', 'code': '', 'authorizationLevel': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            code: String(value), optional
            authorizationLevel: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            code: String16(value), optional
            authorizationLevel: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listFacInfo({'name': '%', 'code': '%', 'authorizationLevel': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["facInfo"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addHuntList(self, description=None, callManagerGroupName="", routeListEnabled=None, voiceMailUsage=None, members=None, name=""):
        """
        :param huntList: 
            description: String(value), optional
            callManagerGroupName: 
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            voiceMailUsage: boolean(value), optional
            members: optional
                    selectionOrder: UnionType(value)
            name: String(value)
        """
        try:
            return self.client.addHuntList({"description": description, "callManagerGroupName": callManagerGroupName, "routeListEnabled": routeListEnabled, "voiceMailUsage": voiceMailUsage, "members": members, "name": name})["return"]
        except Fault as e:
            return e

    def updateHuntList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param description: String(value), optional
        :param callManagerGroupName: optional
            _value_1: String(value)
        :param routeListEnabled: boolean(value), optional
        :param voiceMailUsage: boolean(value), optional
        :param removeMembers: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        :param addMembers: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        :param members: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        :param newName: String(value), optional
        """
        try:
            return self.client.updateHuntList(**kwargs)["return"]
        except Fault as e:
            return e

    def getHuntList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            description: String(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            voiceMailUsage: boolean(value), optional
            members: optional
                    selectionOrder: UnionType(value), optional
            name: String(value), optional
        """
        try:
            return self.client.getHuntList(**kwargs)["return"]["huntList"]
        except Fault as e:
            return e

    def removeHuntList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeHuntList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listHuntList(self, returnedTags={'description': '', 'callManagerGroupName': {'_value_1': ''}, 'routeListEnabled': '', 'voiceMailUsage': '', 'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            description: String(value), optional
            name: String(value), optional
        :param returnedTags: 
            description: String(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            voiceMailUsage: boolean(value), optional
            name: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listHuntList({'description': '%', 'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["huntList"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetHuntList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetHuntList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyHuntList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyHuntList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addIvrUserLocale(self, userLocale="", orderIndex=""):
        """
        :param ivrUserLocale: 
            userLocale: UnionType(value)
            orderIndex: UnionType(value)
        """
        try:
            return self.client.addIvrUserLocale({"userLocale": userLocale, "orderIndex": orderIndex})["return"]
        except Fault as e:
            return e

    def updateIvrUserLocale(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param userLocale: UnionType(value), optional
        :param newUserLocale: UnionType(value), optional
        :param orderIndex: UnionType(value), optional
        """
        try:
            return self.client.updateIvrUserLocale(**kwargs)["return"]
        except Fault as e:
            return e

    def getIvrUserLocale(self, **kwargs):
        """
        :param userLocale: UnionType(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            userLocale: UnionType(value), optional
            orderIndex: UnionType(value), optional
        """
        try:
            return self.client.getIvrUserLocale(**kwargs)["return"]["ivrUserLocale"]
        except Fault as e:
            return e

    def removeIvrUserLocale(self, userLocale="", uuid=""):
        """
        :param userLocale: UnionType(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeIvrUserLocale(userLocale=userLocale, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listIvrUserLocale(self, returnedTags={'userLocale': '', 'orderIndex': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            userLocale: String(value), optional
        :param returnedTags: 
            userLocale: UnionType(value), optional
            orderIndex: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listIvrUserLocale({'userLocale': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ivrUserLocale"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addLineGroup(self, distributionAlgorithm="", rnaReversionTimeOut="", huntAlgorithmNoAnswer="", huntAlgorithmBusy="", huntAlgorithmNotAvailable="", members=None, name="", autoLogOffHunt=None):
        """
        :param lineGroup: 
            distributionAlgorithm: UnionType(value)
            rnaReversionTimeOut: UnionType(value)
            huntAlgorithmNoAnswer: UnionType(value)
            huntAlgorithmBusy: UnionType(value)
            huntAlgorithmNotAvailable: UnionType(value)
            members: optional
                    lineSelectionOrder: UnionType(value)
            name: String(value)
            autoLogOffHunt: boolean(value), optional
        """
        try:
            return self.client.addLineGroup({"distributionAlgorithm": distributionAlgorithm, "rnaReversionTimeOut": rnaReversionTimeOut, "huntAlgorithmNoAnswer": huntAlgorithmNoAnswer, "huntAlgorithmBusy": huntAlgorithmBusy, "huntAlgorithmNotAvailable": huntAlgorithmNotAvailable, "members": members, "name": name, "autoLogOffHunt": autoLogOffHunt})["return"]
        except Fault as e:
            return e

    def updateLineGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param distributionAlgorithm: UnionType(value), optional
        :param rnaReversionTimeOut: UnionType(value), optional
        :param huntAlgorithmNoAnswer: UnionType(value), optional
        :param huntAlgorithmBusy: UnionType(value), optional
        :param huntAlgorithmNotAvailable: UnionType(value), optional
        :param removeMembers: optional
            member: optional
                lineSelectionOrder: UnionType(value)
                    pattern: String(value)
        :param addMembers: optional
            member: optional
                lineSelectionOrder: UnionType(value)
                    pattern: String(value)
        :param members: optional
            member: optional
                lineSelectionOrder: UnionType(value)
                    pattern: String(value)
        :param newName: String(value), optional
        :param autoLogOffHunt: boolean(value), optional
        """
        try:
            return self.client.updateLineGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getLineGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            distributionAlgorithm: UnionType(value), optional
            rnaReversionTimeOut: UnionType(value), optional
            huntAlgorithmNoAnswer: UnionType(value), optional
            huntAlgorithmBusy: UnionType(value), optional
            huntAlgorithmNotAvailable: UnionType(value), optional
            members: optional
                    lineSelectionOrder: UnionType(value), optional
            name: String(value), optional
            autoLogOffHunt: boolean(value), optional
        """
        try:
            return self.client.getLineGroup(**kwargs)["return"]["lineGroup"]
        except Fault as e:
            return e

    def removeLineGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeLineGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listLineGroup(self, returnedTags={'distributionAlgorithm': '', 'rnaReversionTimeOut': '', 'huntAlgorithmNoAnswer': '', 'huntAlgorithmBusy': '', 'huntAlgorithmNotAvailable': '', 'name': '', 'autoLogOffHunt': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            distributionAlgorithm: UnionType(value), optional
            rnaReversionTimeOut: UnionType(value), optional
            huntAlgorithmNoAnswer: UnionType(value), optional
            huntAlgorithmBusy: UnionType(value), optional
            huntAlgorithmNotAvailable: UnionType(value), optional
            name: String(value), optional
            autoLogOffHunt: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listLineGroup({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["lineGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRecordingProfile(self, name="", recordingCssName=None, recorderDestination=""):
        """
        :param recordingProfile: 
            name: String50(value)
            recordingCssName: optional
                _value_1: String(value)
            recorderDestination: String(value)
        """
        try:
            return self.client.addRecordingProfile({"name": name, "recordingCssName": recordingCssName, "recorderDestination": recorderDestination})["return"]
        except Fault as e:
            return e

    def updateRecordingProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param recordingCssName: optional
            _value_1: String(value)
        :param recorderDestination: String(value), optional
        """
        try:
            return self.client.updateRecordingProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getRecordingProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            recordingCssName: optional
                _value_1: String(value)
            recorderDestination: String(value), optional
        """
        try:
            return self.client.getRecordingProfile(**kwargs)["return"]["recordingProfile"]
        except Fault as e:
            return e

    def removeRecordingProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRecordingProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRecordingProfile(self, returnedTags={'name': '', 'recordingCssName': {'_value_1': ''}, 'recorderDestination': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            recordingCssName: String(value), optional
            recorderDestination: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            recordingCssName: optional
                _value_1: String(value)
            recorderDestination: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRecordingProfile({'name': '%', 'recordingCssName': '%', 'recorderDestination': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["recordingProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRouteFilter(self, name="", dialPlanName="", members=""):
        """
        :param routeFilter: 
            name: String(value)
            dialPlanName: 
                _value_1: String(value)
            members: 
                    digits: String50(value), optional
                    operator: UnionType(value)
                    priority: UnionType(value)
        """
        try:
            return self.client.addRouteFilter({"name": name, "dialPlanName": dialPlanName, "members": members})["return"]
        except Fault as e:
            return e

    def updateRouteFilter(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param dialPlanName: optional
            _value_1: String(value)
        :param removeMembers: optional
            member: optional
                    _value_1: String(value)
                digits: String50(value), optional
                operator: UnionType(value)
                priority: UnionType(value)
        :param addMembers: optional
            member: optional
                    _value_1: String(value)
                digits: String50(value), optional
                operator: UnionType(value)
                priority: UnionType(value)
        :param members: optional
            member: optional
                    _value_1: String(value)
                digits: String50(value), optional
                operator: UnionType(value)
                priority: UnionType(value)
        """
        try:
            return self.client.updateRouteFilter(**kwargs)["return"]
        except Fault as e:
            return e

    def getRouteFilter(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            clause: String(value), optional
            dialPlanName: optional
                _value_1: String(value)
            dialPlanWizardGenId: UnionType(value), optional
            members: optional
                    digits: String50(value), optional
                    operator: UnionType(value), optional
                    priority: UnionType(value), optional
        """
        try:
            return self.client.getRouteFilter(**kwargs)["return"]["routeFilter"]
        except Fault as e:
            return e

    def removeRouteFilter(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRouteFilter(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRouteFilter(self, returnedTags={'name': '', 'clause': '', 'dialPlanName': {'_value_1': ''}, 'dialPlanWizardGenId': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            clause: String(value), optional
            dialPlanName: optional
                _value_1: String(value)
            dialPlanWizardGenId: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRouteFilter({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routeFilter"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCallManagerGroup(self, name="", tftpDefault=None, members=""):
        """
        :param callManagerGroup: 
            name: String50(value)
            tftpDefault: boolean(value), optional
            members: 
                    priority: UnionType(value)
        """
        try:
            return self.client.addCallManagerGroup({"name": name, "tftpDefault": tftpDefault, "members": members})["return"]
        except Fault as e:
            return e

    def updateCallManagerGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param tftpDefault: boolean(value), optional
        :param removeMembers: optional
            member: 
                    _value_1: String(value)
                priority: UnionType(value)
        :param addMembers: optional
            member: 
                    _value_1: String(value)
                priority: UnionType(value)
        :param members: optional
            member: 
                    _value_1: String(value)
                priority: UnionType(value)
        """
        try:
            return self.client.updateCallManagerGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getCallManagerGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            tftpDefault: boolean(value), optional
            members: optional
                    priority: UnionType(value), optional
        """
        try:
            return self.client.getCallManagerGroup(**kwargs)["return"]["callManagerGroup"]
        except Fault as e:
            return e

    def removeCallManagerGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCallManagerGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCallManagerGroup(self, returnedTags={'name': '', 'tftpDefault': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            tftpDefault: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallManagerGroup({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callManagerGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCallManagerGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCallManagerGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCallManagerGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCallManagerGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addUserGroup(self, members=None, userRoles=None, name=""):
        """
        :param userGroup: 
            members: optional
                    userId: String(value)
            userRoles: optional
            name: UniqueString128(value)
        """
        try:
            return self.client.addUserGroup({"members": members, "userRoles": userRoles, "name": name})["return"]
        except Fault as e:
            return e

    def updateUserGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param removeMembers: optional
            member: optional
                userId: String(value)
        :param addMembers: optional
            member: optional
                userId: String(value)
        :param members: optional
            member: optional
                userId: String(value)
        :param removeUserRoles: optional
            userRole: optional
                    _value_1: String(value)
        :param addUserRoles: optional
            userRole: optional
                    _value_1: String(value)
        :param userRoles: optional
            userRole: optional
                    _value_1: String(value)
        :param newName: UniqueString128(value), optional
        """
        try:
            return self.client.updateUserGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getUserGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            members: optional
                    userId: String(value), optional
            userRoles: optional
            name: UniqueString128(value), optional
        """
        try:
            return self.client.getUserGroup(**kwargs)["return"]["userGroup"]
        except Fault as e:
            return e

    def removeUserGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeUserGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listUserGroup(self, returnedTags={'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listUserGroup({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["userGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getDialPlan(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String50(value), optional
        """
        try:
            return self.client.getDialPlan(**kwargs)["return"]["dialPlan"]
        except Fault as e:
            return e

    def listDialPlan(self, returnedTags={'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String50(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDialPlan({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["dialPlan"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getDialPlanTag(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param name: String50(value)
        :param dialPlanName: 
            _value_1: String(value)
        :param returnedTags: optional
            name: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            operator: UnionType(value), optional
            suppressFromRouteFilter: boolean(value), optional
        """
        try:
            return self.client.getDialPlanTag(**kwargs)["return"]["dialPlanTag"]
        except Fault as e:
            return e

    def listDialPlanTag(self, returnedTags={'name': '', 'dialPlanName': {'_value_1': ''}, 'operator': '', 'suppressFromRouteFilter': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            operator: UnionType(value), optional
            suppressFromRouteFilter: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDialPlanTag({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["dialPlanTag"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getDdi(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param name: String50(value)
        :param dialPlanName: 
            _value_1: String(value)
        :param returnedTags: optional
            name: String50(value), optional
            clause: String1024(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitAnalysisId: UnionType(value), optional
            members: optional
        """
        try:
            return self.client.getDdi(**kwargs)["return"]["ddi"]
        except Fault as e:
            return e

    def listDdi(self, returnedTags={'name': '', 'clause': '', 'dialPlanName': {'_value_1': ''}, 'digitAnalysisId': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            clause: String1024(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitAnalysisId: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDdi({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ddi"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getMobileSmartClientProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            mobileSmartClient: UnionType(value), optional
            enableSnrUri: String(value), optional
            enableCFAUri: String(value), optional
            handOffUri: String(value), optional
        """
        try:
            return self.client.getMobileSmartClientProfile(**kwargs)["return"]["mobileSmartClientProfile"]
        except Fault as e:
            return e

    def listMobileSmartClientProfile(self, returnedTags={'name': '', 'mobileSmartClient': '', 'enableSnrUri': '', 'enableCFAUri': '', 'handOffUri': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            mobileSmartClient: UnionType(value), optional
            enableSnrUri: String(value), optional
            enableCFAUri: String(value), optional
            handOffUri: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMobileSmartClientProfile({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mobileSmartClientProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def updateProcessNodeService(self, processNodeName="", service="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param processNodeName: 
            _value_1: String(value)
        :param service: UnionType(value)
        :param traceLevel: UnionType(value), optional
        :param userCategories: UnionType(value), optional
        :param enable: boolean(value), optional
        :param numFiles: UnionType(value), optional
        :param maxFileSize: UnionType(value), optional
        """
        try:
            return self.client.updateProcessNodeService(processNodeName=processNodeName, service=service, **kwargs)["return"]
        except Fault as e:
            return e

    def getProcessNodeService(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param processNodeName: 
            _value_1: String(value)
        :param service: UnionType(value)
        :param returnedTags: optional
            processNodeName: optional
                _value_1: String(value)
            service: UnionType(value), optional
            traceLevel: UnionType(value), optional
            userCategories: UnionType(value), optional
            enable: boolean(value), optional
            numFiles: UnionType(value), optional
            maxFileSize: UnionType(value), optional
            isActive: boolean(value), optional
        """
        try:
            return self.client.getProcessNodeService(**kwargs)["return"]["processNodeService"]
        except Fault as e:
            return e

    def listProcessNodeService(self, returnedTags={'processNodeName': {'_value_1': ''}, 'service': '', 'traceLevel': '', 'userCategories': '', 'enable': '', 'numFiles': '', 'maxFileSize': '', 'isActive': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            processNodeName: String(value), optional
        :param returnedTags: 
            processNodeName: optional
                _value_1: String(value)
            service: UnionType(value), optional
            traceLevel: UnionType(value), optional
            userCategories: UnionType(value), optional
            enable: boolean(value), optional
            numFiles: UnionType(value), optional
            maxFileSize: UnionType(value), optional
            isActive: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listProcessNodeService({'processNodeName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["processNodeService"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def updateMohAudioSource(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param sourceId: UnionType(value), optional
        :param newName: String50(value), optional
        :param sourceFile: String255(value), optional
        :param multicast: boolean(value), optional
        :param mohFileStatus: UnionType(value), optional
        :param initialAnnouncement: optional
            _value_1: String(value)
        :param periodicAnnouncement: optional
            _value_1: String(value)
        :param periodicAnnouncementInterval: UnionType(value), optional
        :param localeAnnouncement: UnionType(value), optional
        :param initialAnnouncementPlayed: boolean(value), optional
        :param isExternalSource: boolean(value), optional
        """
        try:
            return self.client.updateMohAudioSource(**kwargs)["return"]
        except Fault as e:
            return e

    def getMohAudioSource(self, **kwargs):
        """
        :param sourceId: UnionType(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            sourceId: UnionType(value), optional
            name: String50(value), optional
            sourceFile: String255(value), optional
            multicast: boolean(value), optional
            mohFileStatus: UnionType(value), optional
            initialAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncementInterval: UnionType(value), optional
            localeAnnouncement: UnionType(value), optional
            initialAnnouncementPlayed: boolean(value), optional
            isExternalSource: boolean(value), optional
        """
        try:
            return self.client.getMohAudioSource(**kwargs)["return"]["mohAudioSource"]
        except Fault as e:
            return e

    def removeMohAudioSource(self, sourceId="", uuid=""):
        """
        :param sourceId: UnionType(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMohAudioSource(sourceId=sourceId, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listMohAudioSource(self, returnedTags={'sourceId': '', 'name': '', 'sourceFile': '', 'multicast': '', 'mohFileStatus': '', 'initialAnnouncement': {'_value_1': ''}, 'periodicAnnouncement': {'_value_1': ''}, 'periodicAnnouncementInterval': '', 'localeAnnouncement': '', 'initialAnnouncementPlayed': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            sourceId: String(value), optional
            name: String(value), optional
        :param returnedTags: 
            sourceId: UnionType(value), optional
            name: String50(value), optional
            sourceFile: String255(value), optional
            multicast: boolean(value), optional
            mohFileStatus: UnionType(value), optional
            initialAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncementInterval: UnionType(value), optional
            localeAnnouncement: UnionType(value), optional
            initialAnnouncementPlayed: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMohAudioSource({'sourceId': '%', 'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mohAudioSource"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDhcpServer(self, processNodeName="", primaryDnsIpAddress=None, secondaryDnsIpAddress=None, primaryTftpServerIpAddress=None, secondaryTftpServerIpAddress=None, bootstrapServerIpAddress=None, domainName=None, tftpServerName=None, arpCacheTimeout="", ipAddressLeaseTime="", renewalTime="", rebindingTime=""):
        """
        :param dhcpServer: 
            processNodeName: 
                _value_1: String(value)
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            domainName: String(value), optional
            tftpServerName: String(value), optional
            arpCacheTimeout: UnionType(value)
            ipAddressLeaseTime: UnionType(value)
            renewalTime: UnionType(value)
            rebindingTime: UnionType(value)
        """
        try:
            return self.client.addDhcpServer({"processNodeName": processNodeName, "primaryDnsIpAddress": primaryDnsIpAddress, "secondaryDnsIpAddress": secondaryDnsIpAddress, "primaryTftpServerIpAddress": primaryTftpServerIpAddress, "secondaryTftpServerIpAddress": secondaryTftpServerIpAddress, "bootstrapServerIpAddress": bootstrapServerIpAddress, "domainName": domainName, "tftpServerName": tftpServerName, "arpCacheTimeout": arpCacheTimeout, "ipAddressLeaseTime": ipAddressLeaseTime, "renewalTime": renewalTime, "rebindingTime": rebindingTime})["return"]
        except Fault as e:
            return e

    def updateDhcpServer(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param processNodeName: optional
            _value_1: String(value)
        :param newProcessNodeName: optional
            _value_1: String(value)
        :param primaryDnsIpAddress: String(value), optional
        :param secondaryDnsIpAddress: String(value), optional
        :param primaryTftpServerIpAddress: String(value), optional
        :param secondaryTftpServerIpAddress: String(value), optional
        :param bootstrapServerIpAddress: String(value), optional
        :param domainName: String(value), optional
        :param tftpServerName: String(value), optional
        :param arpCacheTimeout: UnionType(value), optional
        :param ipAddressLeaseTime: UnionType(value), optional
        :param renewalTime: UnionType(value), optional
        :param rebindingTime: UnionType(value), optional
        """
        try:
            return self.client.updateDhcpServer(**kwargs)["return"]
        except Fault as e:
            return e

    def getDhcpServer(self, **kwargs):
        """
        :param processNodeName: optional
            _value_1: String(value)
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            processNodeName: optional
                _value_1: String(value)
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            domainName: String(value), optional
            tftpServerName: String(value), optional
            arpCacheTimeout: UnionType(value), optional
            ipAddressLeaseTime: UnionType(value), optional
            renewalTime: UnionType(value), optional
            rebindingTime: UnionType(value), optional
        """
        try:
            return self.client.getDhcpServer(**kwargs)["return"]["dhcpServer"]
        except Fault as e:
            return e

    def removeDhcpServer(self, processNodeName="", uuid=""):
        """
        :param processNodeName: optional
            _value_1: String(value)
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDhcpServer(processNodeName={'_value_1': ''}, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDhcpServer(self, returnedTags={'processNodeName': {'_value_1': ''}, 'primaryDnsIpAddress': '', 'secondaryDnsIpAddress': '', 'primaryTftpServerIpAddress': '', 'secondaryTftpServerIpAddress': '', 'bootstrapServerIpAddress': '', 'domainName': '', 'tftpServerName': '', 'arpCacheTimeout': '', 'ipAddressLeaseTime': '', 'renewalTime': '', 'rebindingTime': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            processNodeName: String(value), optional
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            domainName: String(value), optional
        :param returnedTags: 
            processNodeName: optional
                _value_1: String(value)
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            domainName: String(value), optional
            tftpServerName: String(value), optional
            arpCacheTimeout: UnionType(value), optional
            ipAddressLeaseTime: UnionType(value), optional
            renewalTime: UnionType(value), optional
            rebindingTime: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDhcpServer({'processNodeName': '%', 'primaryDnsIpAddress': '%', 'secondaryDnsIpAddress': '%', 'domainName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["dhcpServer"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDhcpSubnet(self, dhcpServerName="", subnetIpAddress="", primaryStartIpAddress="", primaryEndIpAddress="", secondaryStartIpAddress=None, secondaryEndIpAddress=None, primaryRouterIpAddress=None, secondaryRouterIpAddress=None, subnetMask="", domainName=None, primaryDnsIpAddress=None, secondaryDnsIpAddress=None, tftpServerName=None, primaryTftpServerIpAddress=None, secondaryTftpServerIpAddress=None, bootstrapServerIpAddress=None, arpCacheTimeout=None, ipAddressLeaseTime=None, renewalTime=None, rebindingTime=None):
        """
        :param dhcpSubnet: 
            dhcpServerName: 
                _value_1: String(value)
            subnetIpAddress: String(value)
            primaryStartIpAddress: String(value)
            primaryEndIpAddress: String(value)
            secondaryStartIpAddress: String(value), optional
            secondaryEndIpAddress: String(value), optional
            primaryRouterIpAddress: String(value), optional
            secondaryRouterIpAddress: String(value), optional
            subnetMask: String(value)
            domainName: String(value), optional
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            tftpServerName: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            arpCacheTimeout: UnionType(value), optional
            ipAddressLeaseTime: UnionType(value), optional
            renewalTime: UnionType(value), optional
            rebindingTime: UnionType(value), optional
        """
        try:
            return self.client.addDhcpSubnet({"dhcpServerName": dhcpServerName, "subnetIpAddress": subnetIpAddress, "primaryStartIpAddress": primaryStartIpAddress, "primaryEndIpAddress": primaryEndIpAddress, "secondaryStartIpAddress": secondaryStartIpAddress, "secondaryEndIpAddress": secondaryEndIpAddress, "primaryRouterIpAddress": primaryRouterIpAddress, "secondaryRouterIpAddress": secondaryRouterIpAddress, "subnetMask": subnetMask, "domainName": domainName, "primaryDnsIpAddress": primaryDnsIpAddress, "secondaryDnsIpAddress": secondaryDnsIpAddress, "tftpServerName": tftpServerName, "primaryTftpServerIpAddress": primaryTftpServerIpAddress, "secondaryTftpServerIpAddress": secondaryTftpServerIpAddress, "bootstrapServerIpAddress": bootstrapServerIpAddress, "arpCacheTimeout": arpCacheTimeout, "ipAddressLeaseTime": ipAddressLeaseTime, "renewalTime": renewalTime, "rebindingTime": rebindingTime})["return"]
        except Fault as e:
            return e

    def updateDhcpSubnet(self, dhcpServerName="", subnetIpAddress="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param dhcpServerName: 
            _value_1: String(value)
        :param subnetIpAddress: String(value)
        :param newDhcpServerName: optional
            _value_1: String(value)
        :param newSubnetIpAddress: String(value), optional
        :param primaryStartIpAddress: String(value), optional
        :param primaryEndIpAddress: String(value), optional
        :param secondaryStartIpAddress: String(value), optional
        :param secondaryEndIpAddress: String(value), optional
        :param primaryRouterIpAddress: String(value), optional
        :param secondaryRouterIpAddress: String(value), optional
        :param subnetMask: String(value), optional
        :param domainName: String(value), optional
        :param primaryDnsIpAddress: String(value), optional
        :param secondaryDnsIpAddress: String(value), optional
        :param tftpServerName: String(value), optional
        :param primaryTftpServerIpAddress: String(value), optional
        :param secondaryTftpServerIpAddress: String(value), optional
        :param bootstrapServerIpAddress: String(value), optional
        :param arpCacheTimeout: UnionType(value), optional
        :param ipAddressLeaseTime: UnionType(value), optional
        :param renewalTime: UnionType(value), optional
        :param rebindingTime: UnionType(value), optional
        """
        try:
            return self.client.updateDhcpSubnet(dhcpServerName=dhcpServerName, subnetIpAddress=subnetIpAddress, **kwargs)["return"]
        except Fault as e:
            return e

    def getDhcpSubnet(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param dhcpServerName: 
            _value_1: String(value)
        :param subnetIpAddress: String(value)
        :param returnedTags: optional
            dhcpServerName: optional
                _value_1: String(value)
            subnetIpAddress: String(value), optional
            primaryStartIpAddress: String(value), optional
            primaryEndIpAddress: String(value), optional
            secondaryStartIpAddress: String(value), optional
            secondaryEndIpAddress: String(value), optional
            primaryRouterIpAddress: String(value), optional
            secondaryRouterIpAddress: String(value), optional
            subnetMask: String(value), optional
            domainName: String(value), optional
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            tftpServerName: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            arpCacheTimeout: UnionType(value), optional
            ipAddressLeaseTime: UnionType(value), optional
            renewalTime: UnionType(value), optional
            rebindingTime: UnionType(value), optional
        """
        try:
            return self.client.getDhcpSubnet(**kwargs)["return"]["dhcpSubnet"]
        except Fault as e:
            return e

    def removeDhcpSubnet(self, uuid="", dhcpServerName="", subnetIpAddress=""):
        """
        :param uuid: XUUID(value), optional
        :param dhcpServerName: 
            _value_1: String(value)
        :param subnetIpAddress: String(value)
        """
        try:
            return self.client.removeDhcpSubnet(uuid=uuid, dhcpServerName={'_value_1': ''}, subnetIpAddress=subnetIpAddress)["return"]
        except Fault as e:
            return e

    def listDhcpSubnet(self, returnedTags={'subnetIpAddress': '', 'primaryStartIpAddress': '', 'primaryEndIpAddress': '', 'secondaryStartIpAddress': '', 'secondaryEndIpAddress': '', 'primaryRouterIpAddress': '', 'secondaryRouterIpAddress': '', 'subnetMask': '', 'domainName': '', 'primaryDnsIpAddress': '', 'secondaryDnsIpAddress': '', 'tftpServerName': '', 'primaryTftpServerIpAddress': '', 'secondaryTftpServerIpAddress': '', 'bootstrapServerIpAddress': '', 'arpCacheTimeout': '', 'ipAddressLeaseTime': '', 'renewalTime': '', 'rebindingTime': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            dhcpServerName: String(value), optional
            subnetIpAddress: String(value), optional
            primaryStartIpAddress: String(value), optional
            primaryEndIpAddress: String(value), optional
            secondaryStartIpAddress: String(value), optional
            secondaryEndIpAddress: String(value), optional
        :param returnedTags: 
            subnetIpAddress: String(value), optional
            primaryStartIpAddress: String(value), optional
            primaryEndIpAddress: String(value), optional
            secondaryStartIpAddress: String(value), optional
            secondaryEndIpAddress: String(value), optional
            primaryRouterIpAddress: String(value), optional
            secondaryRouterIpAddress: String(value), optional
            subnetMask: String(value), optional
            domainName: String(value), optional
            primaryDnsIpAddress: String(value), optional
            secondaryDnsIpAddress: String(value), optional
            tftpServerName: String(value), optional
            primaryTftpServerIpAddress: String(value), optional
            secondaryTftpServerIpAddress: String(value), optional
            bootstrapServerIpAddress: String(value), optional
            arpCacheTimeout: UnionType(value), optional
            ipAddressLeaseTime: UnionType(value), optional
            renewalTime: UnionType(value), optional
            rebindingTime: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDhcpSubnet({'dhcpServerName': '%', 'subnetIpAddress': '%', 'primaryStartIpAddress': '%', 'primaryEndIpAddress': '%', 'secondaryStartIpAddress': '%', 'secondaryEndIpAddress': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["dhcpSubnet"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCallPark(self, pattern="", description=None, routePartitionName=None, callManagerName=""):
        """
        :param callPark: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            callManagerName: 
                _value_1: String(value)
        """
        try:
            return self.client.addCallPark({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "callManagerName": callManagerName})["return"]
        except Fault as e:
            return e

    def updateCallPark(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param callManagerName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCallPark(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getCallPark(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            callManagerName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCallPark(**kwargs)["return"]["callPark"]
        except Fault as e:
            return e

    def removeCallPark(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeCallPark(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listCallPark(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'callManagerName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            callManagerName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            callManagerName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallPark({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'callManagerName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callPark"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDirectedCallPark(self, pattern="", description=None, routePartitionName=None, retrievalPrefix="", reversionPattern=None, revertCssName=None):
        """
        :param directedCallPark: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            retrievalPrefix: String(value)
            reversionPattern: String(value), optional
            revertCssName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addDirectedCallPark({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "retrievalPrefix": retrievalPrefix, "reversionPattern": reversionPattern, "revertCssName": revertCssName})["return"]
        except Fault as e:
            return e

    def updateDirectedCallPark(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param retrievalPrefix: String(value), optional
        :param reversionPattern: String(value), optional
        :param revertCssName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateDirectedCallPark(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getDirectedCallPark(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            retrievalPrefix: String(value), optional
            reversionPattern: String(value), optional
            revertCssName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getDirectedCallPark(**kwargs)["return"]["directedCallPark"]
        except Fault as e:
            return e

    def removeDirectedCallPark(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeDirectedCallPark(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listDirectedCallPark(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'retrievalPrefix': '', 'reversionPattern': '', 'revertCssName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            reversionPattern: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            retrievalPrefix: String(value), optional
            reversionPattern: String(value), optional
            revertCssName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDirectedCallPark({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'reversionPattern': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["directedCallPark"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetDirectedCallPark(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.resetDirectedCallPark(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def applyDirectedCallPark(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.applyDirectedCallPark(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def addMeetMe(self, pattern="", description=None, routePartitionName=None, minimumSecurityLevel=""):
        """
        :param meetMe: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            minimumSecurityLevel: UnionType(value)
        """
        try:
            return self.client.addMeetMe({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "minimumSecurityLevel": minimumSecurityLevel})["return"]
        except Fault as e:
            return e

    def updateMeetMe(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param minimumSecurityLevel: UnionType(value), optional
        """
        try:
            return self.client.updateMeetMe(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getMeetMe(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            minimumSecurityLevel: UnionType(value), optional
        """
        try:
            return self.client.getMeetMe(**kwargs)["return"]["meetMe"]
        except Fault as e:
            return e

    def removeMeetMe(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeMeetMe(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listMeetMe(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'minimumSecurityLevel': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            minimumSecurityLevel: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMeetMe({'pattern': '%', 'description': '%', 'routePartitionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["meetMe"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addConferenceNow(self, conferenceNowNumber="", routePartitionName=None, description=None, maxWaitTimeForHost="", MohAudioSourceId=None):
        """
        :param conferenceNow: 
            conferenceNowNumber: String(value)
            routePartitionName: optional
                _value_1: String(value)
            description: String(value), optional
            maxWaitTimeForHost: UnionType(value)
            MohAudioSourceId: UnionType(value), optional
        """
        try:
            return self.client.addConferenceNow({"conferenceNowNumber": conferenceNowNumber, "routePartitionName": routePartitionName, "description": description, "maxWaitTimeForHost": maxWaitTimeForHost, "MohAudioSourceId": MohAudioSourceId})["return"]
        except Fault as e:
            return e

    def updateConferenceNow(self, conferenceNowNumber="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param conferenceNowNumber: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newConferenceNowNumber: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param description: String(value), optional
        :param maxWaitTimeForHost: UnionType(value), optional
        :param MohAudioSourceId: UnionType(value), optional
        """
        try:
            return self.client.updateConferenceNow(conferenceNowNumber=conferenceNowNumber, **kwargs)["return"]
        except Fault as e:
            return e

    def getConferenceNow(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param conferenceNowNumber: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            conferenceNowNumber: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            description: String(value), optional
            maxWaitTimeForHost: UnionType(value), optional
            MohAudioSourceId: UnionType(value), optional
        """
        try:
            return self.client.getConferenceNow(**kwargs)["return"]["conferenceNow"]
        except Fault as e:
            return e

    def removeConferenceNow(self, uuid="", conferenceNowNumber="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param conferenceNowNumber: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeConferenceNow(uuid=uuid, conferenceNowNumber=conferenceNowNumber, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listConferenceNow(self, returnedTags={'conferenceNowNumber': '', 'routePartitionName': {'_value_1': ''}, 'description': '', 'maxWaitTimeForHost': '', 'MohAudioSourceId': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            conferenceNowNumber: String(value), optional
            routePartitionName: String(value), optional
        :param returnedTags: 
            conferenceNowNumber: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            description: String(value), optional
            maxWaitTimeForHost: UnionType(value), optional
            MohAudioSourceId: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listConferenceNow({'conferenceNowNumber': '%', 'routePartitionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["conferenceNow"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addMobileVoiceAccess(self, pattern="", routePartitionName=None, locales=""):
        """
        :param mobileVoiceAccess: 
            pattern: String(value)
            routePartitionName: optional
                _value_1: String(value)
            locales: 
                    userLocale: UnionType(value)
                    orderIndex: UnionType(value)
        """
        try:
            return self.client.addMobileVoiceAccess({"pattern": pattern, "routePartitionName": routePartitionName, "locales": locales})["return"]
        except Fault as e:
            return e

    def updateMobileVoiceAccess(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value), optional
        :param newPattern: String(value), optional
        :param routePartitionName: optional
            _value_1: String(value)
        :param removeLocales: optional
            locale: 
                userLocale: UnionType(value)
                orderIndex: UnionType(value)
        :param addLocales: optional
            locale: 
                userLocale: UnionType(value)
                orderIndex: UnionType(value)
        :param locales: optional
            locale: 
                userLocale: UnionType(value)
                orderIndex: UnionType(value)
        """
        try:
            return self.client.updateMobileVoiceAccess(**kwargs)["return"]
        except Fault as e:
            return e

    def getMobileVoiceAccess(self, **kwargs):
        """
        :param pattern: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            pattern: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            locales: optional
                    userLocale: UnionType(value), optional
                    orderIndex: UnionType(value), optional
        """
        try:
            return self.client.getMobileVoiceAccess(**kwargs)["return"]["mobileVoiceAccess"]
        except Fault as e:
            return e

    def removeMobileVoiceAccess(self, pattern="", uuid=""):
        """
        :param pattern: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMobileVoiceAccess(pattern=pattern, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addRouteList(self, name="", description=None, callManagerGroupName="", routeListEnabled=None, members=None, runOnEveryNode=None):
        """
        :param routeList: 
            name: String(value)
            description: String(value), optional
            callManagerGroupName: 
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            members: optional
                    selectionOrder: UnionType(value)
                    calledPartyTransformationMask: String50(value), optional
                    callingPartyTransformationMask: String50(value), optional
                    callingPartyPrefixDigits: String50(value), optional
                    prefixDigitsOut: String50(value), optional
                    useFullyQualifiedCallingPartyNumber: UnionType(value)
                    callingPartyNumberingPlan: UnionType(value), optional
                    callingPartyNumberType: UnionType(value), optional
                    calledPartyNumberingPlan: UnionType(value), optional
                    calledPartyNumberType: UnionType(value), optional
            runOnEveryNode: boolean(value), optional
        """
        try:
            return self.client.addRouteList({"name": name, "description": description, "callManagerGroupName": callManagerGroupName, "routeListEnabled": routeListEnabled, "members": members, "runOnEveryNode": runOnEveryNode})["return"]
        except Fault as e:
            return e

    def updateRouteList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param callManagerGroupName: optional
            _value_1: String(value)
        :param routeListEnabled: boolean(value), optional
        :param removeMembers: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
                calledPartyTransformationMask: String50(value), optional
                callingPartyTransformationMask: String50(value), optional
                    _value_1: String(value)
                callingPartyPrefixDigits: String50(value), optional
                prefixDigitsOut: String50(value), optional
                useFullyQualifiedCallingPartyNumber: UnionType(value)
                callingPartyNumberingPlan: UnionType(value), optional
                callingPartyNumberType: UnionType(value), optional
                calledPartyNumberingPlan: UnionType(value), optional
                calledPartyNumberType: UnionType(value), optional
        :param addMembers: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
                calledPartyTransformationMask: String50(value), optional
                callingPartyTransformationMask: String50(value), optional
                    _value_1: String(value)
                callingPartyPrefixDigits: String50(value), optional
                prefixDigitsOut: String50(value), optional
                useFullyQualifiedCallingPartyNumber: UnionType(value)
                callingPartyNumberingPlan: UnionType(value), optional
                callingPartyNumberType: UnionType(value), optional
                calledPartyNumberingPlan: UnionType(value), optional
                calledPartyNumberType: UnionType(value), optional
        :param members: optional
            member: optional
                    _value_1: String(value)
                selectionOrder: UnionType(value)
                calledPartyTransformationMask: String50(value), optional
                callingPartyTransformationMask: String50(value), optional
                    _value_1: String(value)
                callingPartyPrefixDigits: String50(value), optional
                prefixDigitsOut: String50(value), optional
                useFullyQualifiedCallingPartyNumber: UnionType(value)
                callingPartyNumberingPlan: UnionType(value), optional
                callingPartyNumberType: UnionType(value), optional
                calledPartyNumberingPlan: UnionType(value), optional
                calledPartyNumberType: UnionType(value), optional
        :param runOnEveryNode: boolean(value), optional
        """
        try:
            return self.client.updateRouteList(**kwargs)["return"]
        except Fault as e:
            return e

    def getRouteList(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            members: optional
                    selectionOrder: UnionType(value), optional
                    calledPartyTransformationMask: String50(value), optional
                    callingPartyTransformationMask: String50(value), optional
                    dialPlanWizardGenId: UnionType(value), optional
                    callingPartyPrefixDigits: String50(value), optional
                    prefixDigitsOut: String50(value), optional
                    useFullyQualifiedCallingPartyNumber: UnionType(value), optional
                    callingPartyNumberingPlan: UnionType(value), optional
                    callingPartyNumberType: UnionType(value), optional
                    calledPartyNumberingPlan: UnionType(value), optional
                    calledPartyNumberType: UnionType(value), optional
            runOnEveryNode: boolean(value), optional
        """
        try:
            return self.client.getRouteList(**kwargs)["return"]["routeList"]
        except Fault as e:
            return e

    def removeRouteList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRouteList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRouteList(self, returnedTags={'name': '', 'description': '', 'callManagerGroupName': {'_value_1': ''}, 'routeListEnabled': '', 'runOnEveryNode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            routeListEnabled: boolean(value), optional
            runOnEveryNode: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRouteList({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routeList"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetRouteList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetRouteList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyRouteList(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyRouteList(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addUser(self, firstName=None, displayName=None, middleName=None, lastName="", userid="", password=None, pin=None, mailid=None, department=None, manager=None, userLocale=None, associatedDevices=None, primaryExtension=None, associatedPc=None, associatedGroups=None, enableCti=None, digestCredentials=None, phoneProfiles=None, defaultProfile=None, presenceGroupName="", subscribeCallingSearchSpaceName=None, enableMobility=None, enableMobileVoiceAccess=None, maxDeskPickupWaitTime=None, remoteDestinationLimit=None, passwordCredentials=None, pinCredentials=None, enableEmcc=None, ctiControlledDeviceProfiles=None, patternPrecedence=None, numericUserId=None, mlppPassword=None, customUserFields=None, homeCluster=None, imAndPresenceEnable=None, serviceProfile=None, lineAppearanceAssociationForPresences=None, directoryUri=None, telephoneNumber=None, title=None, mobileNumber=None, homeNumber=None, pagerNumber=None, extensionsInfo=None, selfService=None, userProfile=None, calendarPresence=None, ldapDirectoryName=None, userIdentity=None, nameDialing=None, ipccExtension=None, convertUserAccount=None, accountType=None, authenticationType=None, enableUserToHostConferenceNow=None, attendeesAccessCode=None, zeroHop=None):
        """
        :param user: 
            firstName: String(value), optional
            displayName: String(value), optional
            middleName: String(value), optional
            lastName: String(value)
            userid: String(value)
            password: String(value), optional
            pin: String(value), optional
            mailid: String(value), optional
            department: String(value), optional
            manager: String(value), optional
            userLocale: UnionType(value), optional
            associatedDevices: optional
                device: String50(value), optional
            primaryExtension: optional
                pattern: String255(value)
                routePartitionName: String50(value), optional
            associatedPc: String(value), optional
            associatedGroups: optional
                    name: String100(value)
            enableCti: boolean(value), optional
            digestCredentials: String(value), optional
            phoneProfiles: optional
                    _value_1: String(value)
            defaultProfile: optional
                _value_1: String(value)
            presenceGroupName: 
                _value_1: String(value)
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            enableMobility: boolean(value), optional
            enableMobileVoiceAccess: boolean(value), optional
            maxDeskPickupWaitTime: UnionType(value), optional
            remoteDestinationLimit: UnionType(value), optional
            passwordCredentials: optional
                    _value_1: String(value)
                pwdCredUserCantChange: boolean(value), optional
                pwdCredUserMustChange: boolean(value), optional
                pwdCredDoesNotExpire: boolean(value), optional
                pwdCredTimeChanged: String(value), optional
                pwdCredTimeAdminLockout: String(value), optional
                pwdCredLockedByAdministrator: boolean(value), optional
                pwdResetHackCount: boolean(value), optional
            pinCredentials: optional
                    _value_1: String(value)
                pinCredUserCantChange: boolean(value), optional
                pinCredUserMustChange: boolean(value), optional
                pinCredDoesNotExpire: boolean(value), optional
                pinCredTimeChanged: boolean(value), optional
                pinCredTimeAdminLockout: String(value), optional
                pinCredLockedByAdministrator: boolean(value), optional
                pinResetHackCount: boolean(value), optional
            enableEmcc: boolean(value), optional
            ctiControlledDeviceProfiles: optional
                    _value_1: String(value)
            patternPrecedence: UnionType(value), optional
            numericUserId: String(value), optional
            mlppPassword: String(value), optional
            customUserFields: optional
                    name: String(value), optional
                    value: String(value), optional
            homeCluster: boolean(value), optional
            imAndPresenceEnable: boolean(value), optional
            serviceProfile: optional
                _value_1: String(value)
            lineAppearanceAssociationForPresences: optional
                    laapProductType: String128(value), optional
                    laapDeviceName: String128(value), optional
                    laapDirectory: String128(value), optional
                    laapPartition: String128(value), optional
                    laapDescription: String128(value), optional
            directoryUri: String(value), optional
            telephoneNumber: String(value), optional
            title: String(value), optional
            mobileNumber: String(value), optional
            homeNumber: String(value), optional
            pagerNumber: String(value), optional
            extensionsInfo: optional
                    sortOrder: UnionType(value), optional
                    routePartition: String(value), optional
                    linePrimaryUri: String(value), optional
            selfService: UniqueString50(value), optional
            userProfile: optional
                _value_1: String(value)
            calendarPresence: boolean(value), optional
            ldapDirectoryName: optional
                _value_1: String(value)
            userIdentity: String(value), optional
            nameDialing: String(value), optional
            ipccExtension: optional
                _value_1: String(value)
            convertUserAccount: optional
                _value_1: String(value)
            accountType: UnionType(value), optional
            authenticationType: UnionType(value), optional
            enableUserToHostConferenceNow: boolean(value), optional
            attendeesAccessCode: String(value), optional
            zeroHop: boolean(value), optional
        """
        try:
            return self.client.addUser({"firstName": firstName, "displayName": displayName, "middleName": middleName, "lastName": lastName, "userid": userid, "password": password, "pin": pin, "mailid": mailid, "department": department, "manager": manager, "userLocale": userLocale, "associatedDevices": associatedDevices, "primaryExtension": primaryExtension, "associatedPc": associatedPc, "associatedGroups": associatedGroups, "enableCti": enableCti, "digestCredentials": digestCredentials, "phoneProfiles": phoneProfiles, "defaultProfile": defaultProfile, "presenceGroupName": presenceGroupName, "subscribeCallingSearchSpaceName": subscribeCallingSearchSpaceName, "enableMobility": enableMobility, "enableMobileVoiceAccess": enableMobileVoiceAccess, "maxDeskPickupWaitTime": maxDeskPickupWaitTime, "remoteDestinationLimit": remoteDestinationLimit, "passwordCredentials": passwordCredentials, "pinCredentials": pinCredentials, "enableEmcc": enableEmcc, "ctiControlledDeviceProfiles": ctiControlledDeviceProfiles, "patternPrecedence": patternPrecedence, "numericUserId": numericUserId, "mlppPassword": mlppPassword, "customUserFields": customUserFields, "homeCluster": homeCluster, "imAndPresenceEnable": imAndPresenceEnable, "serviceProfile": serviceProfile, "lineAppearanceAssociationForPresences": lineAppearanceAssociationForPresences, "directoryUri": directoryUri, "telephoneNumber": telephoneNumber, "title": title, "mobileNumber": mobileNumber, "homeNumber": homeNumber, "pagerNumber": pagerNumber, "extensionsInfo": extensionsInfo, "selfService": selfService, "userProfile": userProfile, "calendarPresence": calendarPresence, "ldapDirectoryName": ldapDirectoryName, "userIdentity": userIdentity, "nameDialing": nameDialing, "ipccExtension": ipccExtension, "convertUserAccount": convertUserAccount, "accountType": accountType, "authenticationType": authenticationType, "enableUserToHostConferenceNow": enableUserToHostConferenceNow, "attendeesAccessCode": attendeesAccessCode, "zeroHop": zeroHop})["return"]
        except Fault as e:
            return e

    def updateUser(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param userid: String(value), optional
        :param firstName: String(value), optional
        :param displayName: String(value), optional
        :param middleName: String(value), optional
        :param lastName: String(value), optional
        :param newUserid: String(value), optional
        :param password: String(value), optional
        :param pin: String(value), optional
        :param mailid: String(value), optional
        :param department: String(value), optional
        :param manager: String(value), optional
        :param userLocale: UnionType(value), optional
        :param associatedDevices: optional
            device: String50(value), optional
        :param primaryExtension: optional
            pattern: String255(value)
            routePartitionName: String50(value), optional
        :param associatedPc: String(value), optional
        :param associatedGroups: optional
            userGroup: optional
                name: String100(value)
                    userRole: String100(value), optional
        :param enableCti: boolean(value), optional
        :param digestCredentials: String(value), optional
        :param phoneProfiles: optional
            profileName: 
                _value_1: String(value)
        :param defaultProfile: optional
            _value_1: String(value)
        :param presenceGroupName: optional
            _value_1: String(value)
        :param subscribeCallingSearchSpaceName: optional
            _value_1: String(value)
        :param enableMobility: boolean(value), optional
        :param enableMobileVoiceAccess: boolean(value), optional
        :param maxDeskPickupWaitTime: UnionType(value), optional
        :param remoteDestinationLimit: UnionType(value), optional
        :param passwordCredentials: optional
            pwdCredPolicyName: optional
                _value_1: String(value)
            pwdCredUserCantChange: boolean(value), optional
            pwdCredUserMustChange: boolean(value), optional
            pwdCredDoesNotExpire: boolean(value), optional
            pwdCredTimeChanged: String(value), optional
            pwdCredTimeAdminLockout: String(value), optional
            pwdCredLockedByAdministrator: boolean(value), optional
            pwdResetHackCount: boolean(value), optional
        :param pinCredentials: optional
            pinCredPolicyName: optional
                _value_1: String(value)
            pinCredUserCantChange: boolean(value), optional
            pinCredUserMustChange: boolean(value), optional
            pinCredDoesNotExpire: boolean(value), optional
            pinCredTimeChanged: boolean(value), optional
            pinCredTimeAdminLockout: String(value), optional
            pinCredLockedByAdministrator: boolean(value), optional
            pinResetHackCount: boolean(value), optional
        :param enableEmcc: boolean(value), optional
        :param ctiControlledDeviceProfiles: optional
            profileName: 
                _value_1: String(value)
        :param patternPrecedence: UnionType(value), optional
        :param numericUserId: String(value), optional
        :param mlppPassword: String(value), optional
        :param customUserFields: optional
            customUserField: optional
                name: String(value), optional
                value: String(value), optional
        :param homeCluster: boolean(value), optional
        :param imAndPresenceEnable: boolean(value), optional
        :param serviceProfile: optional
            _value_1: String(value)
        :param lineAppearanceAssociationForPresences: optional
            lineAppearanceAssociationForPresence: optional
                    _value_1: String(value)
                laapProductType: String128(value), optional
                laapDeviceName: String128(value), optional
                laapDirectory: String128(value), optional
                laapPartition: String128(value), optional
                laapDescription: String128(value), optional
        :param directoryUri: String(value), optional
        :param telephoneNumber: String(value), optional
        :param title: String(value), optional
        :param mobileNumber: String(value), optional
        :param homeNumber: String(value), optional
        :param pagerNumber: String(value), optional
        :param removeExtensionsInfo: optional
            extension: optional
                sortOrder: UnionType(value), optional
                    _value_1: String(value)
                routePartition: String(value), optional
                linePrimaryUri: String(value), optional
                    _value_1: String(value)
        :param addExtensionsInfo: optional
            extension: optional
                sortOrder: UnionType(value), optional
                    _value_1: String(value)
                routePartition: String(value), optional
                linePrimaryUri: String(value), optional
                    _value_1: String(value)
        :param extensionsInfo: optional
            extension: optional
                sortOrder: UnionType(value), optional
                    _value_1: String(value)
                routePartition: String(value), optional
                linePrimaryUri: String(value), optional
                    _value_1: String(value)
        :param selfService: UniqueString50(value), optional
        :param userProfile: optional
            _value_1: String(value)
        :param calendarPresence: boolean(value), optional
        :param ldapDirectoryName: optional
            _value_1: String(value)
        :param userIdentity: String(value), optional
        :param nameDialing: String(value), optional
        :param ipccExtension: optional
            _value_1: String(value)
        :param convertUserAccount: optional
            _value_1: String(value)
        :param accountType: UnionType(value), optional
        :param authenticationType: UnionType(value), optional
        :param enableUserToHostConferenceNow: boolean(value), optional
        :param attendeesAccessCode: String(value), optional
        :param zeroHop: boolean(value), optional
        """
        try:
            return self.client.updateUser(**kwargs)["return"]
        except Fault as e:
            return e

    def getUser(self, **kwargs):
        """
        :param userid: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            firstName: String(value), optional
            displayName: String(value), optional
            middleName: String(value), optional
            lastName: String(value), optional
            userid: String(value), optional
            password: String(value), optional
            pin: String(value), optional
            mailid: String(value), optional
            department: String(value), optional
            manager: String(value), optional
            userLocale: UnionType(value), optional
            associatedDevices: optional
                device: String50(value), optional
            primaryExtension: optional
                pattern: String255(value), optional
                routePartitionName: String50(value), optional
            associatedPc: String(value), optional
            associatedGroups: optional
                    name: String100(value), optional
            enableCti: boolean(value), optional
            digestCredentials: String(value), optional
            phoneProfiles: optional
                    _value_1: String(value)
            defaultProfile: optional
                _value_1: String(value)
            presenceGroupName: optional
                _value_1: String(value)
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            enableMobility: boolean(value), optional
            enableMobileVoiceAccess: boolean(value), optional
            maxDeskPickupWaitTime: UnionType(value), optional
            remoteDestinationLimit: UnionType(value), optional
            associatedRemoteDestinationProfiles: optional
                remoteDestinationProfile: String(value), optional
            passwordCredentials: optional
                    _value_1: String(value)
                pwdCredUserCantChange: boolean(value), optional
                pwdCredUserMustChange: boolean(value), optional
                pwdCredDoesNotExpire: boolean(value), optional
                pwdCredTimeChanged: String(value), optional
                pwdCredTimeAdminLockout: String(value), optional
                pwdCredLockedByAdministrator: boolean(value), optional
                pwdResetHackCount: boolean(value), optional
            pinCredentials: optional
                    _value_1: String(value)
                pinCredUserCantChange: boolean(value), optional
                pinCredUserMustChange: boolean(value), optional
                pinCredDoesNotExpire: boolean(value), optional
                pinCredTimeChanged: boolean(value), optional
                pinCredTimeAdminLockout: String(value), optional
                pinCredLockedByAdministrator: boolean(value), optional
                pinResetHackCount: boolean(value), optional
            associatedTodAccess: optional
                todAccess: String(value), optional
            status: UnionType(value), optional
            enableEmcc: boolean(value), optional
            associatedCapfProfiles: optional
                capfProfileInstanceId: String128(value), optional
            ctiControlledDeviceProfiles: optional
                    _value_1: String(value)
            patternPrecedence: UnionType(value), optional
            numericUserId: String(value), optional
            mlppPassword: String(value), optional
            customUserFields: optional
                    name: String(value), optional
                    value: String(value), optional
            homeCluster: boolean(value), optional
            imAndPresenceEnable: boolean(value), optional
            serviceProfile: optional
                _value_1: String(value)
            lineAppearanceAssociationForPresences: optional
                    laapProductType: String128(value), optional
                    laapDeviceName: String128(value), optional
                    laapDirectory: String128(value), optional
                    laapPartition: String128(value), optional
                    laapDescription: String128(value), optional
            directoryUri: String(value), optional
            telephoneNumber: String(value), optional
            title: String(value), optional
            mobileNumber: String(value), optional
            homeNumber: String(value), optional
            pagerNumber: String(value), optional
            extensionsInfo: optional
                    sortOrder: UnionType(value), optional
                    routePartition: String(value), optional
                    linePrimaryUri: String(value), optional
            selfService: UniqueString50(value), optional
            userProfile: optional
                _value_1: String(value)
            calendarPresence: boolean(value), optional
            ldapDirectoryName: optional
                _value_1: String(value)
            userIdentity: String(value), optional
            nameDialing: String(value), optional
            ipccExtension: optional
                _value_1: String(value)
            convertUserAccount: optional
                _value_1: String(value)
            accountType: UnionType(value), optional
            authenticationType: UnionType(value), optional
            enableUserToHostConferenceNow: boolean(value), optional
            attendeesAccessCode: String(value), optional
            zeroHop: boolean(value), optional
        """
        try:
            return self.client.getUser(**kwargs)["return"]["user"]
        except Fault as e:
            return e

    def removeUser(self, userid="", uuid=""):
        """
        :param userid: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeUser(userid=userid, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listUser(self, returnedTags={'firstName': '', 'displayName': '', 'middleName': '', 'lastName': '', 'userid': '', 'mailid': '', 'department': '', 'manager': '', 'userLocale': '', 'primaryExtension': {'pattern': '', 'routePartitionName': ''}, 'associatedPc': '', 'enableCti': '', 'presenceGroupName': {'_value_1': ''}, 'subscribeCallingSearchSpaceName': {'_value_1': ''}, 'enableMobility': '', 'enableMobileVoiceAccess': '', 'maxDeskPickupWaitTime': '', 'remoteDestinationLimit': '', 'status': '', 'enableEmcc': '', 'patternPrecedence': '', 'numericUserId': '', 'mlppPassword': '', 'homeCluster': '', 'imAndPresenceEnable': '', 'serviceProfile': {'_value_1': ''}, 'directoryUri': '', 'telephoneNumber': '', 'title': '', 'mobileNumber': '', 'homeNumber': '', 'pagerNumber': '', 'selfService': '', 'calendarPresence': '', 'ldapDirectoryName': {'_value_1': ''}, 'userIdentity': '', 'accountType': '', 'authenticationType': '', 'enableUserToHostConferenceNow': '', 'attendeesAccessCode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            firstName: String(value), optional
            lastName: String(value), optional
            userid: String(value), optional
            department: String(value), optional
        :param returnedTags: 
            firstName: String(value), optional
            displayName: String(value), optional
            middleName: String(value), optional
            lastName: String(value), optional
            userid: String(value), optional
            mailid: String(value), optional
            department: String(value), optional
            manager: String(value), optional
            userLocale: UnionType(value), optional
            primaryExtension: optional
                pattern: String255(value), optional
                routePartitionName: String50(value), optional
            associatedPc: String(value), optional
            enableCti: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            enableMobility: boolean(value), optional
            enableMobileVoiceAccess: boolean(value), optional
            maxDeskPickupWaitTime: UnionType(value), optional
            remoteDestinationLimit: UnionType(value), optional
            status: UnionType(value), optional
            enableEmcc: boolean(value), optional
            patternPrecedence: UnionType(value), optional
            numericUserId: String(value), optional
            mlppPassword: String(value), optional
            homeCluster: boolean(value), optional
            imAndPresenceEnable: boolean(value), optional
            serviceProfile: optional
                _value_1: String(value)
            directoryUri: String(value), optional
            telephoneNumber: String(value), optional
            title: String(value), optional
            mobileNumber: String(value), optional
            homeNumber: String(value), optional
            pagerNumber: String(value), optional
            selfService: UniqueString50(value), optional
            calendarPresence: boolean(value), optional
            ldapDirectoryName: optional
                _value_1: String(value)
            userIdentity: String(value), optional
            accountType: UnionType(value), optional
            authenticationType: UnionType(value), optional
            enableUserToHostConferenceNow: boolean(value), optional
            attendeesAccessCode: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listUser({'firstName': '%', 'lastName': '%', 'userid': '%', 'department': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["user"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addAppUser(self, userid="", password=None, passwordCredentials=None, digestCredentials=None, presenceGroupName=None, acceptPresenceSubscription=None, acceptOutOfDialogRefer=None, acceptUnsolicitedNotification=None, allowReplaceHeader=None, associatedDevices=None, associatedGroups=None, ctiControlledDeviceProfiles=None):
        """
        :param appUser: 
            userid: String(value)
            password: String(value), optional
            passwordCredentials: optional
                    _value_1: String(value)
                pwdCredUserCantChange: boolean(value), optional
                pwdCredUserMustChange: boolean(value), optional
                pwdCredDoesNotExpire: boolean(value), optional
                pwdCredTimeChanged: String(value), optional
                pwdCredTimeAdminLockout: String(value), optional
                pwdCredLockedByAdministrator: boolean(value), optional
                pwdResetHackCount: boolean(value), optional
            digestCredentials: String(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            associatedDevices: optional
                device: String50(value), optional
            associatedGroups: optional
                    name: String100(value)
            ctiControlledDeviceProfiles: optional
                deviceProfile: String50(value), optional
        """
        try:
            return self.client.addAppUser({"userid": userid, "password": password, "passwordCredentials": passwordCredentials, "digestCredentials": digestCredentials, "presenceGroupName": presenceGroupName, "acceptPresenceSubscription": acceptPresenceSubscription, "acceptOutOfDialogRefer": acceptOutOfDialogRefer, "acceptUnsolicitedNotification": acceptUnsolicitedNotification, "allowReplaceHeader": allowReplaceHeader, "associatedDevices": associatedDevices, "associatedGroups": associatedGroups, "ctiControlledDeviceProfiles": ctiControlledDeviceProfiles})["return"]
        except Fault as e:
            return e

    def updateAppUser(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param userid: String(value), optional
        :param newUserid: String(value), optional
        :param password: String(value), optional
        :param passwordCredentials: optional
            pwdCredPolicyName: optional
                _value_1: String(value)
            pwdCredUserCantChange: boolean(value), optional
            pwdCredUserMustChange: boolean(value), optional
            pwdCredDoesNotExpire: boolean(value), optional
            pwdCredTimeChanged: String(value), optional
            pwdCredTimeAdminLockout: String(value), optional
            pwdCredLockedByAdministrator: boolean(value), optional
            pwdResetHackCount: boolean(value), optional
        :param digestCredentials: String(value), optional
        :param presenceGroupName: optional
            _value_1: String(value)
        :param acceptPresenceSubscription: boolean(value), optional
        :param acceptOutOfDialogRefer: boolean(value), optional
        :param acceptUnsolicitedNotification: boolean(value), optional
        :param allowReplaceHeader: boolean(value), optional
        :param associatedDevices: optional
            device: String50(value), optional
        :param associatedGroups: optional
            userGroup: optional
                name: String100(value)
                    userRole: String100(value), optional
        :param ctiControlledDeviceProfiles: optional
            deviceProfile: String50(value), optional
        """
        try:
            return self.client.updateAppUser(**kwargs)["return"]
        except Fault as e:
            return e

    def getAppUser(self, **kwargs):
        """
        :param userid: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            userid: String(value), optional
            password: String(value), optional
            passwordCredentials: optional
                    _value_1: String(value)
                pwdCredUserCantChange: boolean(value), optional
                pwdCredUserMustChange: boolean(value), optional
                pwdCredDoesNotExpire: boolean(value), optional
                pwdCredTimeChanged: String(value), optional
                pwdCredTimeAdminLockout: String(value), optional
                pwdCredLockedByAdministrator: boolean(value), optional
                pwdResetHackCount: boolean(value), optional
            digestCredentials: String(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            isStandard: boolean(value), optional
            associatedDevices: optional
                device: String50(value), optional
            associatedGroups: optional
                    name: String100(value), optional
            associatedCapfProfiles: optional
                capfProfileInstanceId: String128(value), optional
            ctiControlledDeviceProfiles: optional
                deviceProfile: String50(value), optional
        """
        try:
            return self.client.getAppUser(**kwargs)["return"]["appUser"]
        except Fault as e:
            return e

    def removeAppUser(self, userid="", uuid=""):
        """
        :param userid: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeAppUser(userid=userid, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listAppUser(self, returnedTags={'userid': '', 'presenceGroupName': {'_value_1': ''}, 'acceptPresenceSubscription': '', 'acceptOutOfDialogRefer': '', 'acceptUnsolicitedNotification': '', 'allowReplaceHeader': '', 'isStandard': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            userid: String(value), optional
        :param returnedTags: 
            userid: String(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            acceptPresenceSubscription: boolean(value), optional
            acceptOutOfDialogRefer: boolean(value), optional
            acceptUnsolicitedNotification: boolean(value), optional
            allowReplaceHeader: boolean(value), optional
            isStandard: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listAppUser({'userid': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["appUser"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addSipRealm(self, realm="", userid="", digestCredentials=""):
        """
        :param sipRealm: 
            realm: String(value)
            userid: String(value)
            digestCredentials: String(value)
        """
        try:
            return self.client.addSipRealm({"realm": realm, "userid": userid, "digestCredentials": digestCredentials})["return"]
        except Fault as e:
            return e

    def updateSipRealm(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param realm: String(value), optional
        :param newRealm: String(value), optional
        :param userid: String(value), optional
        :param digestCredentials: String(value), optional
        """
        try:
            return self.client.updateSipRealm(**kwargs)["return"]
        except Fault as e:
            return e

    def getSipRealm(self, **kwargs):
        """
        :param realm: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            realm: String(value), optional
            userid: String(value), optional
            digestCredentials: String(value), optional
        """
        try:
            return self.client.getSipRealm(**kwargs)["return"]["sipRealm"]
        except Fault as e:
            return e

    def removeSipRealm(self, realm="", uuid=""):
        """
        :param realm: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSipRealm(realm=realm, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSipRealm(self, returnedTags={'realm': '', 'userid': '', 'digestCredentials': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            realm: String(value), optional
            userid: String(value), optional
        :param returnedTags: 
            realm: String(value), optional
            userid: String(value), optional
            digestCredentials: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSipRealm({'realm': '%', 'userid': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["sipRealm"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addPhoneNtp(self, ipAddress="", description=None, mode=""):
        """
        :param phoneNtp: 
            ipAddress: String(value)
            description: String(value), optional
            mode: UnionType(value)
        """
        try:
            return self.client.addPhoneNtp({"ipAddress": ipAddress, "description": description, "mode": mode})["return"]
        except Fault as e:
            return e

    def updatePhoneNtp(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param ipAddress: String(value), optional
        :param newIpAddress: String(value), optional
        :param description: String(value), optional
        :param mode: UnionType(value), optional
        """
        try:
            return self.client.updatePhoneNtp(**kwargs)["return"]
        except Fault as e:
            return e

    def getPhoneNtp(self, **kwargs):
        """
        :param ipAddress: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            ipAddress: String(value), optional
            description: String(value), optional
            mode: UnionType(value), optional
        """
        try:
            return self.client.getPhoneNtp(**kwargs)["return"]["phoneNtp"]
        except Fault as e:
            return e

    def removePhoneNtp(self, ipAddress="", uuid=""):
        """
        :param ipAddress: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePhoneNtp(ipAddress=ipAddress, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPhoneNtp(self, returnedTags={'ipAddress': '', 'description': '', 'mode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            ipAddress: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            ipAddress: String(value), optional
            description: String(value), optional
            mode: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPhoneNtp({'ipAddress': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["phoneNtp"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDateTimeGroup(self, name="", timeZone="", separator="", dateformat="", timeFormat="", phoneNtpReferences=None):
        """
        :param dateTimeGroup: 
            name: String(value)
            timeZone: UnionType(value)
            separator: String(value)
            dateformat: String(value)
            timeFormat: String(value)
            phoneNtpReferences: optional
                    selectionOrder: UnionType(value)
        """
        try:
            return self.client.addDateTimeGroup({"name": name, "timeZone": timeZone, "separator": separator, "dateformat": dateformat, "timeFormat": timeFormat, "phoneNtpReferences": phoneNtpReferences})["return"]
        except Fault as e:
            return e

    def updateDateTimeGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param timeZone: UnionType(value), optional
        :param separator: String(value), optional
        :param dateformat: String(value), optional
        :param timeFormat: String(value), optional
        :param removePhoneNtpReferences: optional
            selectedPhoneNtpReference: 
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        :param addPhoneNtpReferences: optional
            selectedPhoneNtpReference: 
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        :param phoneNtpReferences: optional
            selectedPhoneNtpReference: 
                    _value_1: String(value)
                selectionOrder: UnionType(value)
        """
        try:
            return self.client.updateDateTimeGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getDateTimeGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            timeZone: UnionType(value), optional
            separator: String(value), optional
            dateformat: String(value), optional
            timeFormat: String(value), optional
            phoneNtpReferences: optional
                    selectionOrder: UnionType(value), optional
        """
        try:
            return self.client.getDateTimeGroup(**kwargs)["return"]["dateTimeGroup"]
        except Fault as e:
            return e

    def removeDateTimeGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDateTimeGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDateTimeGroup(self, returnedTags={'name': '', 'timeZone': '', 'dateformat': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            timeZone: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            timeZone: UnionType(value), optional
            dateformat: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDateTimeGroup({'name': '%', 'timeZone': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["dateTimeGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetDateTimeGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetDateTimeGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyDateTimeGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyDateTimeGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addPresenceGroup(self, name="", description=None, presenceGroups=None):
        """
        :param presenceGroup: 
            name: String(value)
            description: String(value), optional
            presenceGroups: optional
                    subscriptionPermission: UnionType(value)
        """
        try:
            return self.client.addPresenceGroup({"name": name, "description": description, "presenceGroups": presenceGroups})["return"]
        except Fault as e:
            return e

    def updatePresenceGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param presenceGroups: optional
            presenceGroup: optional
                    _value_1: String(value)
                subscriptionPermission: UnionType(value)
        """
        try:
            return self.client.updatePresenceGroup(**kwargs)["return"]
        except Fault as e:
            return e

    def getPresenceGroup(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            presenceGroups: optional
                    subscriptionPermission: UnionType(value), optional
        """
        try:
            return self.client.getPresenceGroup(**kwargs)["return"]["presenceGroup"]
        except Fault as e:
            return e

    def removePresenceGroup(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePresenceGroup(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPresenceGroup(self, returnedTags={'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPresenceGroup({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["presenceGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addGeoLocation(self, name="", country=None, description=None, nationalSubDivision=None, district=None, communityName=None, cityDivision=None, neighbourhood=None, street=None, leadingStreetDirection=None, trailingStreetSuffix=None, streetSuffix=None, houseNumber=None, houseNumberSuffix=None, landmark=None, location=None, floor=None, occupantName=None, postalCode=None):
        """
        :param geoLocation: 
            name: String50(value)
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
        """
        try:
            return self.client.addGeoLocation({"name": name, "country": country, "description": description, "nationalSubDivision": nationalSubDivision, "district": district, "communityName": communityName, "cityDivision": cityDivision, "neighbourhood": neighbourhood, "street": street, "leadingStreetDirection": leadingStreetDirection, "trailingStreetSuffix": trailingStreetSuffix, "streetSuffix": streetSuffix, "houseNumber": houseNumber, "houseNumberSuffix": houseNumberSuffix, "landmark": landmark, "location": location, "floor": floor, "occupantName": occupantName, "postalCode": postalCode})["return"]
        except Fault as e:
            return e

    def updateGeoLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param country: String(value), optional
        :param description: String255(value), optional
        :param nationalSubDivision: String100(value), optional
        :param district: String255(value), optional
        :param communityName: String255(value), optional
        :param cityDivision: String255(value), optional
        :param neighbourhood: String255(value), optional
        :param street: String255(value), optional
        :param leadingStreetDirection: String50(value), optional
        :param trailingStreetSuffix: String50(value), optional
        :param streetSuffix: String255(value), optional
        :param houseNumber: String16(value), optional
        :param houseNumberSuffix: String100(value), optional
        :param landmark: String255(value), optional
        :param location: String255(value), optional
        :param floor: String16(value), optional
        :param occupantName: String255(value), optional
        :param postalCode: String100(value), optional
        """
        try:
            return self.client.updateGeoLocation(**kwargs)["return"]
        except Fault as e:
            return e

    def getGeoLocation(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
        """
        try:
            return self.client.getGeoLocation(**kwargs)["return"]["geoLocation"]
        except Fault as e:
            return e

    def removeGeoLocation(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGeoLocation(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listGeoLocation(self, returnedTags={'name': '', 'country': '', 'description': '', 'nationalSubDivision': '', 'district': '', 'communityName': '', 'cityDivision': '', 'neighbourhood': '', 'street': '', 'leadingStreetDirection': '', 'trailingStreetSuffix': '', 'streetSuffix': '', 'houseNumber': '', 'houseNumberSuffix': '', 'landmark': '', 'location': '', 'floor': '', 'occupantName': '', 'postalCode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listGeoLocation({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["geoLocation"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addSrst(self, name="", port="", ipAddress="", SipNetwork=None, SipPort="", isSecure=None):
        """
        :param srst: 
            name: String(value)
            port: UnionType(value)
            ipAddress: String(value)
            SipNetwork: String(value), optional
            SipPort: UnionType(value)
            isSecure: boolean(value), optional
        """
        try:
            return self.client.addSrst({"name": name, "port": port, "ipAddress": ipAddress, "SipNetwork": SipNetwork, "SipPort": SipPort, "isSecure": isSecure})["return"]
        except Fault as e:
            return e

    def updateSrst(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param port: UnionType(value), optional
        :param ipAddress: String(value), optional
        :param SipNetwork: String(value), optional
        :param SipPort: UnionType(value), optional
        :param isSecure: boolean(value), optional
        """
        try:
            return self.client.updateSrst(**kwargs)["return"]
        except Fault as e:
            return e

    def getSrst(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            port: UnionType(value), optional
            ipAddress: String(value), optional
            SipNetwork: String(value), optional
            SipPort: UnionType(value), optional
            srstCertificatePort: UnionType(value), optional
            isSecure: boolean(value), optional
        """
        try:
            return self.client.getSrst(**kwargs)["return"]["srst"]
        except Fault as e:
            return e

    def removeSrst(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSrst(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSrst(self, returnedTags={'name': '', 'port': '', 'ipAddress': '', 'SipNetwork': '', 'SipPort': '', 'srstCertificatePort': '', 'isSecure': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            ipAddress: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            port: UnionType(value), optional
            ipAddress: String(value), optional
            SipNetwork: String(value), optional
            SipPort: UnionType(value), optional
            srstCertificatePort: UnionType(value), optional
            isSecure: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSrst({'name': '%', 'ipAddress': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["srst"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetSrst(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetSrst(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartSrst(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartSrst(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applySrst(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applySrst(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addMlppDomain(self, domainName="", domainId=""):
        """
        :param mlppDomain: 
            domainName: String(value)
            domainId: String(value)
        """
        try:
            return self.client.addMlppDomain({"domainName": domainName, "domainId": domainId})["return"]
        except Fault as e:
            return e

    def updateMlppDomain(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param domainName: String(value), optional
        :param newDomainName: String(value), optional
        :param domainId: String(value), optional
        """
        try:
            return self.client.updateMlppDomain(**kwargs)["return"]
        except Fault as e:
            return e

    def getMlppDomain(self, **kwargs):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            domainId: String(value), optional
        """
        try:
            return self.client.getMlppDomain(**kwargs)["return"]["mlppDomain"]
        except Fault as e:
            return e

    def removeMlppDomain(self, domainName="", uuid=""):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMlppDomain(domainName=domainName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listMlppDomain(self, returnedTags={'domainName': '', 'domainId': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            domainName: String(value), optional
        :param returnedTags: 
            domainName: String(value), optional
            domainId: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMlppDomain({'domainName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mlppDomain"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCumaServerSecurityProfile(self, name="", description=None, securityMode="", transportType="", x509SubjectName=None, serverIpHostName=""):
        """
        :param cumaServerSecurityProfile: 
            name: String(value)
            description: String(value), optional
            securityMode: UnionType(value)
            transportType: UnionType(value)
            x509SubjectName: String(value), optional
            serverIpHostName: String255(value)
        """
        try:
            return self.client.addCumaServerSecurityProfile({"name": name, "description": description, "securityMode": securityMode, "transportType": transportType, "x509SubjectName": x509SubjectName, "serverIpHostName": serverIpHostName})["return"]
        except Fault as e:
            return e

    def updateCumaServerSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param securityMode: UnionType(value), optional
        :param transportType: UnionType(value), optional
        :param x509SubjectName: String(value), optional
        :param serverIpHostName: String255(value), optional
        """
        try:
            return self.client.updateCumaServerSecurityProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getCumaServerSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            securityMode: UnionType(value), optional
            transportType: UnionType(value), optional
            x509SubjectName: String(value), optional
            serverIpHostName: String255(value), optional
        """
        try:
            return self.client.getCumaServerSecurityProfile(**kwargs)["return"]["cumaServerSecurityProfile"]
        except Fault as e:
            return e

    def removeCumaServerSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCumaServerSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCumaServerSecurityProfile(self, returnedTags={'name': '', 'description': '', 'securityMode': '', 'transportType': '', 'x509SubjectName': '', 'serverIpHostName': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            securityMode: UnionType(value), optional
            transportType: UnionType(value), optional
            x509SubjectName: String(value), optional
            serverIpHostName: String255(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCumaServerSecurityProfile({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["cumaServerSecurityProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addApplicationServer(self, appServerType="", name="", ipAddress=None, appUsers=None, url=None, endUserUrl=None, processNodeName=None, endUsers=None):
        """
        :param applicationServer: 
            appServerType: UnionType(value)
            name: String(value)
            ipAddress: String(value), optional
            appUsers: optional
            url: String(value), optional
            endUserUrl: String(value), optional
            processNodeName: optional
                _value_1: String(value)
            endUsers: optional
        """
        try:
            return self.client.addApplicationServer({"appServerType": appServerType, "name": name, "ipAddress": ipAddress, "appUsers": appUsers, "url": url, "endUserUrl": endUserUrl, "processNodeName": processNodeName, "endUsers": endUsers})["return"]
        except Fault as e:
            return e

    def updateApplicationServer(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param ipAddress: String(value), optional
        :param removeAppUsers: optional
            selectedAppUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        :param addAppUsers: optional
            selectedAppUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        :param appUsers: optional
            selectedAppUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        :param url: String(value), optional
        :param endUserUrl: String(value), optional
        :param processNodeName: optional
            _value_1: String(value)
        :param removeEndUsers: optional
            selectedEndUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        :param addEndUsers: optional
            selectedEndUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        :param endUsers: optional
            selectedEndUser: 
                    _value_1: String(value)
                    _value_1: AnyType, optional
        """
        try:
            return self.client.updateApplicationServer(**kwargs)["return"]
        except Fault as e:
            return e

    def getApplicationServer(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            appServerType: UnionType(value), optional
            name: String(value), optional
            ipAddress: String(value), optional
            appUsers: optional
            url: String(value), optional
            endUserUrl: String(value), optional
            processNodeName: optional
                _value_1: String(value)
            endUsers: optional
        """
        try:
            return self.client.getApplicationServer(**kwargs)["return"]["applicationServer"]
        except Fault as e:
            return e

    def removeApplicationServer(self, uuid=""):
        """
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeApplicationServer(uuid=uuid)["return"]
        except Fault as e:
            return e

    def listApplicationServer(self, returnedTags={'appServerType': '', 'name': '', 'ipAddress': '', 'url': '', 'endUserUrl': '', 'processNodeName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            ipAddress: String(value), optional
        :param returnedTags: 
            appServerType: UnionType(value), optional
            name: String(value), optional
            ipAddress: String(value), optional
            url: String(value), optional
            endUserUrl: String(value), optional
            processNodeName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listApplicationServer({'name': '%', 'ipAddress': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["applicationServer"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addApplicationUserCapfProfile(self, applicationUser="", instanceId="", certificateOperation="", authenticationMode="", authenticationString=None, keySize="", keyOrder=None, ecKeySize=None, operationCompletion=None):
        """
        :param applicationUserCapfProfile: 
            applicationUser: 
                _value_1: String(value)
            instanceId: String(value)
            certificateOperation: UnionType(value)
            authenticationMode: UnionType(value)
            authenticationString: String(value), optional
            keySize: UnionType(value)
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
        """
        try:
            return self.client.addApplicationUserCapfProfile({"applicationUser": applicationUser, "instanceId": instanceId, "certificateOperation": certificateOperation, "authenticationMode": authenticationMode, "authenticationString": authenticationString, "keySize": keySize, "keyOrder": keyOrder, "ecKeySize": ecKeySize, "operationCompletion": operationCompletion})["return"]
        except Fault as e:
            return e

    def updateApplicationUserCapfProfile(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param instanceId: String(value), optional
        :param certificateOperation: UnionType(value), optional
        :param authenticationMode: UnionType(value), optional
        :param authenticationString: String(value), optional
        :param keySize: UnionType(value), optional
        :param keyOrder: UnionType(value), optional
        :param ecKeySize: UnionType(value), optional
        :param operationCompletion: String(value), optional
        """
        try:
            return self.client.updateApplicationUserCapfProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getApplicationUserCapfProfile(self, **kwargs):
        """
        :param instanceId: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            applicationUser: optional
                _value_1: String(value)
            instanceId: String(value), optional
            certificateOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            authenticationString: String(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
            certificationOperationStatus: UnionType(value), optional
        """
        try:
            return self.client.getApplicationUserCapfProfile(**kwargs)["return"]["applicationUserCapfProfile"]
        except Fault as e:
            return e

    def removeApplicationUserCapfProfile(self, instanceId="", uuid=""):
        """
        :param instanceId: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeApplicationUserCapfProfile(instanceId=instanceId, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listApplicationUserCapfProfile(self, returnedTags={'applicationUser': {'_value_1': ''}, 'instanceId': '', 'certificateOperation': '', 'authenticationMode': '', 'authenticationString': '', 'keySize': '', 'keyOrder': '', 'ecKeySize': '', 'operationCompletion': '', 'certificationOperationStatus': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            applicationUser: String(value), optional
            instanceId: String(value), optional
        :param returnedTags: 
            applicationUser: optional
                _value_1: String(value)
            instanceId: String(value), optional
            certificateOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            authenticationString: String(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
            certificationOperationStatus: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listApplicationUserCapfProfile({'applicationUser': '%', 'instanceId': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["applicationUserCapfProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addEndUserCapfProfile(self, endUserId="", instanceId="", certificationOperation="", authenticationMode="", authenticationString=None, keySize="", keyOrder=None, ecKeySize=None, operationCompletion=None):
        """
        :param endUserCapfProfile: 
            endUserId: 
                _value_1: String(value)
            instanceId: String(value)
            certificationOperation: UnionType(value)
            authenticationMode: UnionType(value)
            authenticationString: String(value), optional
            keySize: UnionType(value)
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
        """
        try:
            return self.client.addEndUserCapfProfile({"endUserId": endUserId, "instanceId": instanceId, "certificationOperation": certificationOperation, "authenticationMode": authenticationMode, "authenticationString": authenticationString, "keySize": keySize, "keyOrder": keyOrder, "ecKeySize": ecKeySize, "operationCompletion": operationCompletion})["return"]
        except Fault as e:
            return e

    def updateEndUserCapfProfile(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param instanceId: String(value), optional
        :param certificationOperation: UnionType(value), optional
        :param authenticationMode: UnionType(value), optional
        :param authenticationString: String(value), optional
        :param keySize: UnionType(value), optional
        :param keyOrder: UnionType(value), optional
        :param ecKeySize: UnionType(value), optional
        :param operationCompletion: String(value), optional
        """
        try:
            return self.client.updateEndUserCapfProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getEndUserCapfProfile(self, **kwargs):
        """
        :param instanceId: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            endUserId: optional
                _value_1: String(value)
            instanceId: String(value), optional
            certificationOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            authenticationString: String(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
            certificationOperationStatus: UnionType(value), optional
        """
        try:
            return self.client.getEndUserCapfProfile(**kwargs)["return"]["endUserCapfProfile"]
        except Fault as e:
            return e

    def removeEndUserCapfProfile(self, instanceId="", uuid=""):
        """
        :param instanceId: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeEndUserCapfProfile(instanceId=instanceId, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listEndUserCapfProfile(self, returnedTags={'endUserId': {'_value_1': ''}, 'instanceId': '', 'certificationOperation': '', 'authenticationMode': '', 'authenticationString': '', 'keySize': '', 'keyOrder': '', 'ecKeySize': '', 'operationCompletion': '', 'certificationOperationStatus': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            endUserId: String(value), optional
            instanceId: String(value), optional
        :param returnedTags: 
            endUserId: optional
                _value_1: String(value)
            instanceId: String(value), optional
            certificationOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            authenticationString: String(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            operationCompletion: String(value), optional
            certificationOperationStatus: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listEndUserCapfProfile({'endUserId': '%', 'instanceId': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["endUserCapfProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def updateServiceParameter(self, processNodeName="", name="", service="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param processNodeName: 
            _value_1: String(value)
        :param name: String(value)
        :param service: UnionType(value)
        :param value: String(value), optional
        """
        try:
            return self.client.updateServiceParameter(processNodeName=processNodeName, name=name, service=service, **kwargs)["return"]
        except Fault as e:
            return e

    def getServiceParameter(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param processNodeName: 
            _value_1: String(value)
        :param name: String(value)
        :param service: UnionType(value)
        :param returnedTags: optional
            processNodeName: optional
                _value_1: String(value)
            name: String(value), optional
            service: UnionType(value), optional
            value: String(value), optional
            valueType: UnionType(value), optional
        """
        try:
            return self.client.getServiceParameter(**kwargs)["return"]["serviceParameter"]
        except Fault as e:
            return e

    def listServiceParameter(self, returnedTags={'processNodeName': {'_value_1': ''}, 'name': '', 'service': '', 'value': '', 'valueType': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            processNodeName: String(value), optional
            service: String(value), optional
        :param returnedTags: 
            processNodeName: optional
                _value_1: String(value)
            name: String(value), optional
            service: UnionType(value), optional
            value: String(value), optional
            valueType: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listServiceParameter({'processNodeName': '%', 'service': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["serviceParameter"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addUserPhoneAssociation(self, userId="", password=None, pin=None, lastName="", middleName=None, firstName=None, productType="", name="", dnCssName=None, phoneCssName=None, e164Mask="", extension="", routePartitionName=None, voiceMailProfileName=None, enableExtensionMobility=None, DirectoryURI=None, DirectoryNumberURIPartition=None):
        """
        :param userPhoneAssociation: 
            userId: String(value)
            password: String(value), optional
            pin: UnionType(value), optional
            lastName: String(value)
            middleName: String(value), optional
            firstName: String(value), optional
            productType: UnionType(value)
            name: UniqueString128(value)
            dnCssName: optional
                _value_1: String(value)
            phoneCssName: optional
                _value_1: String(value)
            e164Mask: String(value)
            extension: String(value)
            routePartitionName: optional
                _value_1: String(value)
            voiceMailProfileName: optional
                _value_1: String(value)
            enableExtensionMobility: boolean(value), optional
            DirectoryURI: String(value), optional
            DirectoryNumberURIPartition: optional
                _value_1: String(value)
        """
        try:
            return self.client.addUserPhoneAssociation({"userId": userId, "password": password, "pin": pin, "lastName": lastName, "middleName": middleName, "firstName": firstName, "productType": productType, "name": name, "dnCssName": dnCssName, "phoneCssName": phoneCssName, "e164Mask": e164Mask, "extension": extension, "routePartitionName": routePartitionName, "voiceMailProfileName": voiceMailProfileName, "enableExtensionMobility": enableExtensionMobility, "DirectoryURI": DirectoryURI, "DirectoryNumberURIPartition": DirectoryNumberURIPartition})["return"]
        except Fault as e:
            return e

    def addGeoLocationFilter(self, name="", description=None, useCountry=None, useNationalSubDivision=None, useDistrict=None, useCommunityName=None, useCityDivision=None, useNeighbourhood=None, useStreet=None, useLeadingStreetDirection=None, useTrailingStreetSuffix=None, useStreetSuffix=None, useHouseNumber=None, useHouseNumberSuffix=None, useLandmark=None, useLocation=None, useFloor=None, useOccupantName=None, usePostalCode=None):
        """
        :param geoLocationFilter: 
            name: String50(value)
            description: String255(value), optional
            useCountry: boolean(value), optional
            useNationalSubDivision: boolean(value), optional
            useDistrict: boolean(value), optional
            useCommunityName: boolean(value), optional
            useCityDivision: boolean(value), optional
            useNeighbourhood: boolean(value), optional
            useStreet: boolean(value), optional
            useLeadingStreetDirection: boolean(value), optional
            useTrailingStreetSuffix: boolean(value), optional
            useStreetSuffix: boolean(value), optional
            useHouseNumber: boolean(value), optional
            useHouseNumberSuffix: boolean(value), optional
            useLandmark: boolean(value), optional
            useLocation: boolean(value), optional
            useFloor: boolean(value), optional
            useOccupantName: boolean(value), optional
            usePostalCode: boolean(value), optional
        """
        try:
            return self.client.addGeoLocationFilter({"name": name, "description": description, "useCountry": useCountry, "useNationalSubDivision": useNationalSubDivision, "useDistrict": useDistrict, "useCommunityName": useCommunityName, "useCityDivision": useCityDivision, "useNeighbourhood": useNeighbourhood, "useStreet": useStreet, "useLeadingStreetDirection": useLeadingStreetDirection, "useTrailingStreetSuffix": useTrailingStreetSuffix, "useStreetSuffix": useStreetSuffix, "useHouseNumber": useHouseNumber, "useHouseNumberSuffix": useHouseNumberSuffix, "useLandmark": useLandmark, "useLocation": useLocation, "useFloor": useFloor, "useOccupantName": useOccupantName, "usePostalCode": usePostalCode})["return"]
        except Fault as e:
            return e

    def updateGeoLocationFilter(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String255(value), optional
        :param useCountry: boolean(value), optional
        :param useNationalSubDivision: boolean(value), optional
        :param useDistrict: boolean(value), optional
        :param useCommunityName: boolean(value), optional
        :param useCityDivision: boolean(value), optional
        :param useNeighbourhood: boolean(value), optional
        :param useStreet: boolean(value), optional
        :param useLeadingStreetDirection: boolean(value), optional
        :param useTrailingStreetSuffix: boolean(value), optional
        :param useStreetSuffix: boolean(value), optional
        :param useHouseNumber: boolean(value), optional
        :param useHouseNumberSuffix: boolean(value), optional
        :param useLandmark: boolean(value), optional
        :param useLocation: boolean(value), optional
        :param useFloor: boolean(value), optional
        :param useOccupantName: boolean(value), optional
        :param usePostalCode: boolean(value), optional
        """
        try:
            return self.client.updateGeoLocationFilter(**kwargs)["return"]
        except Fault as e:
            return e

    def getGeoLocationFilter(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String255(value), optional
            useCountry: boolean(value), optional
            useNationalSubDivision: boolean(value), optional
            useDistrict: boolean(value), optional
            useCommunityName: boolean(value), optional
            useCityDivision: boolean(value), optional
            useNeighbourhood: boolean(value), optional
            useStreet: boolean(value), optional
            useLeadingStreetDirection: boolean(value), optional
            useTrailingStreetSuffix: boolean(value), optional
            useStreetSuffix: boolean(value), optional
            useHouseNumber: boolean(value), optional
            useHouseNumberSuffix: boolean(value), optional
            useLandmark: boolean(value), optional
            useLocation: boolean(value), optional
            useFloor: boolean(value), optional
            useOccupantName: boolean(value), optional
            usePostalCode: boolean(value), optional
        """
        try:
            return self.client.getGeoLocationFilter(**kwargs)["return"]["geoLocationFilter"]
        except Fault as e:
            return e

    def removeGeoLocationFilter(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGeoLocationFilter(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listGeoLocationFilter(self, returnedTags={'name': '', 'description': '', 'useCountry': '', 'useNationalSubDivision': '', 'useDistrict': '', 'useCommunityName': '', 'useCityDivision': '', 'useNeighbourhood': '', 'useStreet': '', 'useLeadingStreetDirection': '', 'useTrailingStreetSuffix': '', 'useStreetSuffix': '', 'useHouseNumber': '', 'useHouseNumberSuffix': '', 'useLandmark': '', 'useLocation': '', 'useFloor': '', 'useOccupantName': '', 'usePostalCode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String255(value), optional
            useCountry: boolean(value), optional
            useNationalSubDivision: boolean(value), optional
            useDistrict: boolean(value), optional
            useCommunityName: boolean(value), optional
            useCityDivision: boolean(value), optional
            useNeighbourhood: boolean(value), optional
            useStreet: boolean(value), optional
            useLeadingStreetDirection: boolean(value), optional
            useTrailingStreetSuffix: boolean(value), optional
            useStreetSuffix: boolean(value), optional
            useHouseNumber: boolean(value), optional
            useHouseNumberSuffix: boolean(value), optional
            useLandmark: boolean(value), optional
            useLocation: boolean(value), optional
            useFloor: boolean(value), optional
            useOccupantName: boolean(value), optional
            usePostalCode: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listGeoLocationFilter({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["geoLocationFilter"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addVoiceMailProfile(self, name="", description=None, isDefault=None, voiceMailboxMask=None, voiceMailPilot=""):
        """
        :param voiceMailProfile: 
            name: String50(value)
            description: String50(value), optional
            isDefault: boolean(value), optional
            voiceMailboxMask: String(value), optional
            voiceMailPilot: 
                dirn: String50(value)
                    _value_1: String(value)
        """
        try:
            return self.client.addVoiceMailProfile({"name": name, "description": description, "isDefault": isDefault, "voiceMailboxMask": voiceMailboxMask, "voiceMailPilot": voiceMailPilot})["return"]
        except Fault as e:
            return e

    def updateVoiceMailProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String50(value), optional
        :param isDefault: boolean(value), optional
        :param voiceMailboxMask: String(value), optional
        :param voiceMailPilot: optional
            dirn: String50(value)
            cssName: optional
                _value_1: String(value)
        """
        try:
            return self.client.updateVoiceMailProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getVoiceMailProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            description: String50(value), optional
            isDefault: boolean(value), optional
            voiceMailboxMask: String(value), optional
            voiceMailPilot: optional
                dirn: String50(value), optional
                    _value_1: String(value)
        """
        try:
            return self.client.getVoiceMailProfile(**kwargs)["return"]["voiceMailProfile"]
        except Fault as e:
            return e

    def removeVoiceMailProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeVoiceMailProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listVoiceMailProfile(self, returnedTags={'name': '', 'description': '', 'isDefault': '', 'voiceMailboxMask': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            description: String50(value), optional
            isDefault: boolean(value), optional
            voiceMailboxMask: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listVoiceMailProfile({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["voiceMailProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetVoiceMailProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetVoiceMailProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartVoiceMailProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartVoiceMailProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyVoiceMailProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyVoiceMailProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addVoiceMailPort(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, locationName="", preemption=None, useTrustedRelayPoint="", securityProfileName="", geoLocationName=None, automatedAlternateRoutingCssName=None, dnPattern="", routePartition=None, dnCallingSearchSpace=None, aarNeighborhoodName=None, callerIdDisplay=None, callerIdDisplayAscii=None, externalMask=None):
        """
        :param voiceMailPort: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: 
                _value_1: String(value)
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value)
            securityProfileName: 
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            dnPattern: String(value)
            routePartition: optional
                _value_1: String(value)
            dnCallingSearchSpace: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            callerIdDisplay: String128(value), optional
            callerIdDisplayAscii: String50(value), optional
            externalMask: String(value), optional
        """
        try:
            return self.client.addVoiceMailPort({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "locationName": locationName, "preemption": preemption, "useTrustedRelayPoint": useTrustedRelayPoint, "securityProfileName": securityProfileName, "geoLocationName": geoLocationName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "dnPattern": dnPattern, "routePartition": routePartition, "dnCallingSearchSpace": dnCallingSearchSpace, "aarNeighborhoodName": aarNeighborhoodName, "callerIdDisplay": callerIdDisplay, "callerIdDisplayAscii": callerIdDisplayAscii, "externalMask": externalMask})["return"]
        except Fault as e:
            return e

    def updateVoiceMailPort(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param useTrustedRelayPoint: UnionType(value), optional
        :param securityProfileName: optional
            _value_1: String(value)
        :param geoLocationName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param dnPattern: String(value), optional
        :param routePartition: optional
            _value_1: String(value)
        :param dnCallingSearchSpace: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param callerIdDisplay: String128(value), optional
        :param callerIdDisplayAscii: String50(value), optional
        :param externalMask: String(value), optional
        """
        try:
            return self.client.updateVoiceMailPort(**kwargs)["return"]
        except Fault as e:
            return e

    def getVoiceMailPort(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            securityProfileName: optional
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            dnPattern: String(value), optional
            routePartition: optional
                _value_1: String(value)
            dnCallingSearchSpace: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            callerIdDisplay: String128(value), optional
            callerIdDisplayAscii: String50(value), optional
            externalMask: String(value), optional
        """
        try:
            return self.client.getVoiceMailPort(**kwargs)["return"]["voiceMailPort"]
        except Fault as e:
            return e

    def removeVoiceMailPort(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeVoiceMailPort(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listVoiceMailPort(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'preemption': '', 'useTrustedRelayPoint': '', 'securityProfileName': {'_value_1': ''}, 'geoLocationName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'dnPattern': '', 'callerIdDisplay': '', 'callerIdDisplayAscii': '', 'externalMask': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
            securityProfileName: String(value), optional
            dnPattern: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            securityProfileName: optional
                _value_1: String(value)
            geoLocationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            dnPattern: String(value), optional
            callerIdDisplay: String128(value), optional
            callerIdDisplayAscii: String50(value), optional
            externalMask: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listVoiceMailPort({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%', 'securityProfileName': '%', 'dnPattern': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["voiceMailPort"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetVoiceMailPort(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetVoiceMailPort(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartVoiceMailPort(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartVoiceMailPort(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyVoiceMailPort(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyVoiceMailPort(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGatekeeper(self, name="", description=None, rrqTimeToLive="", retryTimeout="", enableDevice=None):
        """
        :param gatekeeper: 
            name: UniqueString128(value)
            description: String128(value), optional
            rrqTimeToLive: UnionType(value)
            retryTimeout: UnionType(value)
            enableDevice: boolean(value), optional
        """
        try:
            return self.client.addGatekeeper({"name": name, "description": description, "rrqTimeToLive": rrqTimeToLive, "retryTimeout": retryTimeout, "enableDevice": enableDevice})["return"]
        except Fault as e:
            return e

    def updateGatekeeper(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param rrqTimeToLive: UnionType(value), optional
        :param retryTimeout: UnionType(value), optional
        :param enableDevice: boolean(value), optional
        """
        try:
            return self.client.updateGatekeeper(**kwargs)["return"]
        except Fault as e:
            return e

    def getGatekeeper(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            rrqTimeToLive: UnionType(value), optional
            retryTimeout: UnionType(value), optional
            enableDevice: boolean(value), optional
        """
        try:
            return self.client.getGatekeeper(**kwargs)["return"]["gatekeeper"]
        except Fault as e:
            return e

    def removeGatekeeper(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGatekeeper(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listGatekeeper(self, returnedTags={'name': '', 'description': '', 'rrqTimeToLive': '', 'retryTimeout': '', 'enableDevice': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            rrqTimeToLive: UnionType(value), optional
            retryTimeout: UnionType(value), optional
            enableDevice: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listGatekeeper({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["gatekeeper"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetGatekeeper(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetGatekeeper(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartGatekeeper(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartGatekeeper(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyGatekeeper(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyGatekeeper(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addPhoneButtonTemplate(self, name="", basePhoneTemplateName="", buttons=None):
        """
        :param phoneButtonTemplate: 
            name: String(value)
            basePhoneTemplateName: 
                _value_1: String(value)
            buttons: optional
                    feature: UnionType(value)
                    label: String50(value), optional
                    buttonNumber: UnionType(value)
        """
        try:
            return self.client.addPhoneButtonTemplate({"name": name, "basePhoneTemplateName": basePhoneTemplateName, "buttons": buttons})["return"]
        except Fault as e:
            return e

    def updatePhoneButtonTemplate(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param buttons: optional
            button: optional
                feature: UnionType(value)
                label: String50(value), optional
                buttonNumber: UnionType(value)
        """
        try:
            return self.client.updatePhoneButtonTemplate(**kwargs)["return"]
        except Fault as e:
            return e

    def getPhoneButtonTemplate(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            isUserModifiable: boolean(value), optional
            buttons: optional
                    feature: UnionType(value), optional
                    label: String50(value), optional
                    buttonNumber: UnionType(value), optional
                    isFixedFeature: boolean(value), optional
        """
        try:
            return self.client.getPhoneButtonTemplate(**kwargs)["return"]["phoneButtonTemplate"]
        except Fault as e:
            return e

    def removePhoneButtonTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePhoneButtonTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPhoneButtonTemplate(self, returnedTags={'name': '', 'isUserModifiable': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            isUserModifiable: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPhoneButtonTemplate({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["phoneButtonTemplate"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def restartPhoneButtonTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartPhoneButtonTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyPhoneButtonTemplate(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyPhoneButtonTemplate(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCommonPhoneConfig(self, name="", description=None, unlockPwd=None, dndOption=None, dndAlertingType=None, backgroundImage=None, phonePersonalization=None, phoneServiceDisplay=None, sshUserId=None, sshPwd=None, vendorConfig=None, alwaysUsePrimeLine=None, alwaysUsePrimeLineForVoiceMessage=None, vpnGroupName=None, vpnProfileName=None, featureControlPolicy=None, wifiHotspotProfile=None):
        """
        :param commonPhoneConfig: 
            name: UniqueString50(value)
            description: String128(value), optional
            unlockPwd: String50(value), optional
            dndOption: UnionType(value), optional
            dndAlertingType: UnionType(value), optional
            backgroundImage: boolean(value), optional
            phonePersonalization: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            sshUserId: String50(value), optional
            sshPwd: String255(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            vpnGroupName: optional
                _value_1: String(value)
            vpnProfileName: optional
                _value_1: String(value)
            featureControlPolicy: optional
                _value_1: String(value)
            wifiHotspotProfile: optional
                _value_1: String(value)
        """
        try:
            return self.client.addCommonPhoneConfig({"name": name, "description": description, "unlockPwd": unlockPwd, "dndOption": dndOption, "dndAlertingType": dndAlertingType, "backgroundImage": backgroundImage, "phonePersonalization": phonePersonalization, "phoneServiceDisplay": phoneServiceDisplay, "sshUserId": sshUserId, "sshPwd": sshPwd, "vendorConfig": vendorConfig, "alwaysUsePrimeLine": alwaysUsePrimeLine, "alwaysUsePrimeLineForVoiceMessage": alwaysUsePrimeLineForVoiceMessage, "vpnGroupName": vpnGroupName, "vpnProfileName": vpnProfileName, "featureControlPolicy": featureControlPolicy, "wifiHotspotProfile": wifiHotspotProfile})["return"]
        except Fault as e:
            return e

    def updateCommonPhoneConfig(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString50(value), optional
        :param description: String128(value), optional
        :param unlockPwd: String50(value), optional
        :param dndOption: UnionType(value), optional
        :param dndAlertingType: UnionType(value), optional
        :param backgroundImage: boolean(value), optional
        :param phonePersonalization: UnionType(value), optional
        :param phoneServiceDisplay: UnionType(value), optional
        :param sshUserId: String50(value), optional
        :param sshPwd: String255(value), optional
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param alwaysUsePrimeLine: UnionType(value), optional
        :param alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param vpnGroupName: optional
            _value_1: String(value)
        :param vpnProfileName: optional
            _value_1: String(value)
        :param featureControlPolicy: optional
            _value_1: String(value)
        :param wifiHotspotProfile: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCommonPhoneConfig(**kwargs)["return"]
        except Fault as e:
            return e

    def getCommonPhoneConfig(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString50(value), optional
            description: String128(value), optional
            unlockPwd: String50(value), optional
            dndOption: UnionType(value), optional
            dndAlertingType: UnionType(value), optional
            backgroundImage: boolean(value), optional
            phonePersonalization: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            sshUserId: String50(value), optional
            sshPwd: String255(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            vpnGroupName: optional
                _value_1: String(value)
            vpnProfileName: optional
                _value_1: String(value)
            featureControlPolicy: optional
                _value_1: String(value)
            wifiHotspotProfile: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCommonPhoneConfig(**kwargs)["return"]["commonPhoneConfig"]
        except Fault as e:
            return e

    def removeCommonPhoneConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCommonPhoneConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCommonPhoneConfig(self, returnedTags={'name': '', 'description': '', 'dndOption': '', 'dndAlertingType': '', 'backgroundImage': '', 'phonePersonalization': '', 'phoneServiceDisplay': '', 'sshUserId': '', 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'vpnGroupName': {'_value_1': ''}, 'vpnProfileName': {'_value_1': ''}, 'featureControlPolicy': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: UniqueString50(value), optional
            description: String128(value), optional
            dndOption: UnionType(value), optional
            dndAlertingType: UnionType(value), optional
            backgroundImage: boolean(value), optional
            phonePersonalization: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            sshUserId: String50(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            vpnGroupName: optional
                _value_1: String(value)
            vpnProfileName: optional
                _value_1: String(value)
            featureControlPolicy: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCommonPhoneConfig({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["commonPhoneConfig"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCommonPhoneConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCommonPhoneConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCommonPhoneConfig(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCommonPhoneConfig(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addMessageWaiting(self, pattern="", routePartitionName="", description=None, messageWaitingIndicator="", callingSearchSpaceName=None):
        """
        :param messageWaiting: 
            pattern: String(value)
            routePartitionName: 
                _value_1: String(value)
            description: String128(value), optional
            messageWaitingIndicator: boolean(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addMessageWaiting({"pattern": pattern, "routePartitionName": routePartitionName, "description": description, "messageWaitingIndicator": messageWaitingIndicator, "callingSearchSpaceName": callingSearchSpaceName})["return"]
        except Fault as e:
            return e

    def updateMessageWaiting(self, pattern="", routePartitionName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param description: String128(value), optional
        :param messageWaitingIndicator: boolean(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateMessageWaiting(pattern=pattern, routePartitionName=routePartitionName, **kwargs)["return"]
        except Fault as e:
            return e

    def getMessageWaiting(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            description: String128(value), optional
            messageWaitingIndicator: boolean(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getMessageWaiting(**kwargs)["return"]["messageWaiting"]
        except Fault as e:
            return e

    def removeMessageWaiting(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        """
        try:
            return self.client.removeMessageWaiting(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listMessageWaiting(self, returnedTags={'pattern': '', 'routePartitionName': {'_value_1': ''}, 'description': '', 'messageWaitingIndicator': '', 'callingSearchSpaceName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            routePartitionName: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            description: String128(value), optional
            messageWaitingIndicator: boolean(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMessageWaiting({'pattern': '%', 'routePartitionName': '%', 'description': '%', 'callingSearchSpaceName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["messageWaiting"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addIpPhoneServices(self, serviceName="", asciiServiceName="", serviceDescription=None, serviceUrl="", secureServiceUrl=None, serviceCategory="", serviceType="", serviceVendor=None, serviceVersion=None, enabled=None, enterpriseSubscription=None, parameters=None):
        """
        :param ipPhoneServices: 
            serviceName: UniqueString128(value)
            asciiServiceName: UniqueString128(value)
            serviceDescription: String128(value), optional
            serviceUrl: String(value)
            secureServiceUrl: String(value), optional
            serviceCategory: UnionType(value)
            serviceType: UnionType(value)
            serviceVendor: String(value), optional
            serviceVersion: String(value), optional
            enabled: boolean(value), optional
            enterpriseSubscription: boolean(value), optional
            parameters: optional
                    name: UniqueString50(value)
                    displayName: String128(value)
                    default: String255(value), optional
                    description: String255(value)
                    paramRequired: boolean(value), optional
                    paramPassword: boolean(value), optional
        """
        try:
            return self.client.addIpPhoneServices({"serviceName": serviceName, "asciiServiceName": asciiServiceName, "serviceDescription": serviceDescription, "serviceUrl": serviceUrl, "secureServiceUrl": secureServiceUrl, "serviceCategory": serviceCategory, "serviceType": serviceType, "serviceVendor": serviceVendor, "serviceVersion": serviceVersion, "enabled": enabled, "enterpriseSubscription": enterpriseSubscription, "parameters": parameters})["return"]
        except Fault as e:
            return e

    def updateIpPhoneServices(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param serviceName: UniqueString128(value), optional
        :param newServiceName: UniqueString128(value), optional
        :param asciiServiceName: UniqueString128(value), optional
        :param serviceDescription: String128(value), optional
        :param serviceUrl: String(value), optional
        :param secureServiceUrl: String(value), optional
        :param serviceCategory: UnionType(value), optional
        :param serviceType: UnionType(value), optional
        :param serviceVendor: String(value), optional
        :param serviceVersion: String(value), optional
        :param enabled: boolean(value), optional
        :param removeParameters: optional
            parameter: 
                name: UniqueString50(value)
                displayName: String128(value)
                default: String255(value), optional
                description: String255(value)
                paramRequired: boolean(value), optional
                paramPassword: boolean(value), optional
        :param addParameters: optional
            parameter: 
                name: UniqueString50(value)
                displayName: String128(value)
                default: String255(value), optional
                description: String255(value)
                paramRequired: boolean(value), optional
                paramPassword: boolean(value), optional
        :param parameters: optional
            parameter: 
                name: UniqueString50(value)
                displayName: String128(value)
                default: String255(value), optional
                description: String255(value)
                paramRequired: boolean(value), optional
                paramPassword: boolean(value), optional
        """
        try:
            return self.client.updateIpPhoneServices(**kwargs)["return"]
        except Fault as e:
            return e

    def getIpPhoneServices(self, **kwargs):
        """
        :param serviceName: UniqueString128(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            serviceName: UniqueString128(value), optional
            asciiServiceName: UniqueString128(value), optional
            serviceDescription: String128(value), optional
            serviceUrl: String(value), optional
            secureServiceUrl: String(value), optional
            serviceCategory: UnionType(value), optional
            serviceType: UnionType(value), optional
            serviceVendor: String(value), optional
            serviceVersion: String(value), optional
            enabled: boolean(value), optional
            enterpriseSubscription: boolean(value), optional
            parameters: optional
                    name: UniqueString50(value), optional
                    displayName: String128(value), optional
                    default: String255(value), optional
                    description: String255(value), optional
                    paramRequired: boolean(value), optional
                    paramPassword: boolean(value), optional
        """
        try:
            return self.client.getIpPhoneServices(**kwargs)["return"]["ipPhoneServices"]
        except Fault as e:
            return e

    def removeIpPhoneServices(self, serviceName="", uuid=""):
        """
        :param serviceName: UniqueString128(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeIpPhoneServices(serviceName=serviceName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listIpPhoneServices(self, returnedTags={'serviceName': '', 'asciiServiceName': '', 'serviceDescription': '', 'serviceUrl': '', 'secureServiceUrl': '', 'serviceCategory': '', 'serviceType': '', 'serviceVendor': '', 'serviceVersion': '', 'enabled': '', 'enterpriseSubscription': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            serviceName: String(value), optional
            serviceDescription: String(value), optional
        :param returnedTags: 
            serviceName: UniqueString128(value), optional
            asciiServiceName: UniqueString128(value), optional
            serviceDescription: String128(value), optional
            serviceUrl: String(value), optional
            secureServiceUrl: String(value), optional
            serviceCategory: UnionType(value), optional
            serviceType: UnionType(value), optional
            serviceVendor: String(value), optional
            serviceVersion: String(value), optional
            enabled: boolean(value), optional
            enterpriseSubscription: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listIpPhoneServices({'serviceName': '%', 'serviceDescription': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ipPhoneServices"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addCtiRoutePoint(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, locationName="", mediaResourceListName=None, networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, useTrustedRelayPoint="", cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, userLocale=None, lines=None):
        """
        :param ctiRoutePoint: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            lines: optional
                    index: UnionType(value)
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value)
                    routePartitionName: String50(value)
        """
        try:
            return self.client.addCtiRoutePoint({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "useTrustedRelayPoint": useTrustedRelayPoint, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "userLocale": userLocale, "lines": lines})["return"]
        except Fault as e:
            return e

    def updateCtiRoutePoint(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param protocol: UnionType(value), optional
        :param protocolSide: UnionType(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param userLocale: UnionType(value), optional
        :param lines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        """
        try:
            return self.client.updateCtiRoutePoint(**kwargs)["return"]
        except Fault as e:
            return e

    def getCtiRoutePoint(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            lines: optional
                    index: UnionType(value), optional
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    dialPlanWizardId: UnionType(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value), optional
                    routePartitionName: String50(value), optional
        """
        try:
            return self.client.getCtiRoutePoint(**kwargs)["return"]["ctiRoutePoint"]
        except Fault as e:
            return e

    def removeCtiRoutePoint(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCtiRoutePoint(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCtiRoutePoint(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'useTrustedRelayPoint': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'userLocale': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCtiRoutePoint({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ctiRoutePoint"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCtiRoutePoint(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCtiRoutePoint(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCtiRoutePoint(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCtiRoutePoint(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCtiRoutePoint(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCtiRoutePoint(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addTransPattern(self, pattern="", description=None, usage="", routePartitionName="", blockEnable=None, calledPartyTransformationMask=None, callingPartyTransformationMask=None, useCallingPartyPhoneMask=None, callingPartyPrefixDigits=None, dialPlanName=None, digitDiscardInstructionName=None, patternUrgency=None, prefixDigitsOut=None, routeFilterName=None, callingLinePresentationBit=None, callingNamePresentationBit=None, connectedLinePresentationBit=None, connectedNamePresentationBit=None, patternPrecedence=None, provideOutsideDialtone=None, callingPartyNumberingPlan=None, callingPartyNumberType=None, calledPartyNumberingPlan=None, calledPartyNumberType=None, callingSearchSpaceName=None, resourcePriorityNamespaceName=None, routeNextHopByCgpn=None, routeClass=None, callInterceptProfileName=None, releaseClause=None, useOriginatorCss=None, dontWaitForIDTOnSubsequentHops=None, isEmergencyServiceNumber=None):
        """
        :param transPattern: 
            pattern: String(value)
            description: String(value), optional
            usage: UnionType(value)
            routePartitionName: 
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeNextHopByCgpn: boolean(value), optional
            routeClass: UnionType(value), optional
            callInterceptProfileName: optional
                _value_1: String(value)
            releaseClause: UnionType(value), optional
            useOriginatorCss: boolean(value), optional
            dontWaitForIDTOnSubsequentHops: boolean(value), optional
            isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.addTransPattern({"pattern": pattern, "description": description, "usage": usage, "routePartitionName": routePartitionName, "blockEnable": blockEnable, "calledPartyTransformationMask": calledPartyTransformationMask, "callingPartyTransformationMask": callingPartyTransformationMask, "useCallingPartyPhoneMask": useCallingPartyPhoneMask, "callingPartyPrefixDigits": callingPartyPrefixDigits, "dialPlanName": dialPlanName, "digitDiscardInstructionName": digitDiscardInstructionName, "patternUrgency": patternUrgency, "prefixDigitsOut": prefixDigitsOut, "routeFilterName": routeFilterName, "callingLinePresentationBit": callingLinePresentationBit, "callingNamePresentationBit": callingNamePresentationBit, "connectedLinePresentationBit": connectedLinePresentationBit, "connectedNamePresentationBit": connectedNamePresentationBit, "patternPrecedence": patternPrecedence, "provideOutsideDialtone": provideOutsideDialtone, "callingPartyNumberingPlan": callingPartyNumberingPlan, "callingPartyNumberType": callingPartyNumberType, "calledPartyNumberingPlan": calledPartyNumberingPlan, "calledPartyNumberType": calledPartyNumberType, "callingSearchSpaceName": callingSearchSpaceName, "resourcePriorityNamespaceName": resourcePriorityNamespaceName, "routeNextHopByCgpn": routeNextHopByCgpn, "routeClass": routeClass, "callInterceptProfileName": callInterceptProfileName, "releaseClause": releaseClause, "useOriginatorCss": useOriginatorCss, "dontWaitForIDTOnSubsequentHops": dontWaitForIDTOnSubsequentHops, "isEmergencyServiceNumber": isEmergencyServiceNumber})["return"]
        except Fault as e:
            return e

    def updateTransPattern(self, pattern="", routePartitionName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param blockEnable: boolean(value), optional
        :param calledPartyTransformationMask: String(value), optional
        :param callingPartyTransformationMask: String(value), optional
        :param useCallingPartyPhoneMask: UnionType(value), optional
        :param callingPartyPrefixDigits: String50(value), optional
        :param newDialPlanName: optional
            _value_1: String(value)
        :param digitDiscardInstructionName: optional
            _value_1: String(value)
        :param patternUrgency: boolean(value), optional
        :param prefixDigitsOut: String(value), optional
        :param newRouteFilterName: optional
            _value_1: String(value)
        :param callingLinePresentationBit: UnionType(value), optional
        :param callingNamePresentationBit: UnionType(value), optional
        :param connectedLinePresentationBit: UnionType(value), optional
        :param connectedNamePresentationBit: UnionType(value), optional
        :param patternPrecedence: UnionType(value), optional
        :param provideOutsideDialtone: boolean(value), optional
        :param callingPartyNumberingPlan: UnionType(value), optional
        :param callingPartyNumberType: UnionType(value), optional
        :param calledPartyNumberingPlan: UnionType(value), optional
        :param calledPartyNumberType: UnionType(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param resourcePriorityNamespaceName: optional
            _value_1: String(value)
        :param routeNextHopByCgpn: boolean(value), optional
        :param routeClass: UnionType(value), optional
        :param callInterceptProfileName: optional
            _value_1: String(value)
        :param releaseClause: UnionType(value), optional
        :param useOriginatorCss: boolean(value), optional
        :param dontWaitForIDTOnSubsequentHops: boolean(value), optional
        :param isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.updateTransPattern(pattern=pattern, routePartitionName=routePartitionName, **kwargs)["return"]
        except Fault as e:
            return e

    def getTransPattern(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeNextHopByCgpn: boolean(value), optional
            routeClass: UnionType(value), optional
            callInterceptProfileName: optional
                _value_1: String(value)
            releaseClause: UnionType(value), optional
            useOriginatorCss: boolean(value), optional
            dontWaitForIDTOnSubsequentHops: boolean(value), optional
            isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.getTransPattern(**kwargs)["return"]["transPattern"]
        except Fault as e:
            return e

    def removeTransPattern(self, uuid="", pattern="", routePartitionName="", dialPlanName="", routeFilterName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeTransPattern(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''}, dialPlanName={'_value_1': ''}, routeFilterName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listTransPattern(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'blockEnable': '', 'calledPartyTransformationMask': '', 'callingPartyTransformationMask': '', 'useCallingPartyPhoneMask': '', 'callingPartyPrefixDigits': '', 'dialPlanName': {'_value_1': ''}, 'digitDiscardInstructionName': {'_value_1': ''}, 'patternUrgency': '', 'prefixDigitsOut': '', 'routeFilterName': {'_value_1': ''}, 'callingLinePresentationBit': '', 'callingNamePresentationBit': '', 'connectedLinePresentationBit': '', 'connectedNamePresentationBit': '', 'patternPrecedence': '', 'provideOutsideDialtone': '', 'callingPartyNumberingPlan': '', 'callingPartyNumberType': '', 'calledPartyNumberingPlan': '', 'calledPartyNumberType': '', 'callingSearchSpaceName': {'_value_1': ''}, 'resourcePriorityNamespaceName': {'_value_1': ''}, 'routeNextHopByCgpn': '', 'routeClass': '', 'callInterceptProfileName': {'_value_1': ''}, 'releaseClause': '', 'useOriginatorCss': '', 'dontWaitForIDTOnSubsequentHops': '', 'isEmergencyServiceNumber': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            isEmergencyServiceNumber: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeNextHopByCgpn: boolean(value), optional
            routeClass: UnionType(value), optional
            callInterceptProfileName: optional
                _value_1: String(value)
            releaseClause: UnionType(value), optional
            useOriginatorCss: boolean(value), optional
            dontWaitForIDTOnSubsequentHops: boolean(value), optional
            isEmergencyServiceNumber: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listTransPattern({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'isEmergencyServiceNumber': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["transPattern"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getTransPatternOptions(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedChoices: optional
            usage: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            routePartitionName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            useCallingPartyPhoneMask: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dialPlanName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            digitDiscardInstructionName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            routeFilterName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingLinePresentationBit: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingNamePresentationBit: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            connectedLinePresentationBit: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            connectedNamePresentationBit: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            patternPrecedence: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingPartyNumberingPlan: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingPartyNumberType: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            calledPartyNumberingPlan: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            calledPartyNumberType: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingSearchSpaceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            resourcePriorityNamespaceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            routeClass: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callInterceptProfileName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            releaseClause: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
        """
        try:
            return self.client.getTransPatternOptions(**kwargs)["return"]["transPatternOptions"]
        except Fault as e:
            return e

    def addCallingPartyTransformationPattern(self, pattern="", description=None, routePartitionName="", callingPartyTransformationMask=None, useCallingPartyPhoneMask=None, dialPlanName=None, digitDiscardInstructionName=None, callingPartyPrefixDigits=None, routeFilterName=None, callingLinePresentationBit=None, callingPartyNumberingPlan=None, callingPartyNumberType=None, mlppPreemptionDisabled=None):
        """
        :param callingPartyTransformationPattern: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: 
                _value_1: String(value)
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            callingPartyPrefixDigits: String50(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            mlppPreemptionDisabled: boolean(value), optional
        """
        try:
            return self.client.addCallingPartyTransformationPattern({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "callingPartyTransformationMask": callingPartyTransformationMask, "useCallingPartyPhoneMask": useCallingPartyPhoneMask, "dialPlanName": dialPlanName, "digitDiscardInstructionName": digitDiscardInstructionName, "callingPartyPrefixDigits": callingPartyPrefixDigits, "routeFilterName": routeFilterName, "callingLinePresentationBit": callingLinePresentationBit, "callingPartyNumberingPlan": callingPartyNumberingPlan, "callingPartyNumberType": callingPartyNumberType, "mlppPreemptionDisabled": mlppPreemptionDisabled})["return"]
        except Fault as e:
            return e

    def updateCallingPartyTransformationPattern(self, pattern="", routePartitionName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param callingPartyTransformationMask: String(value), optional
        :param useCallingPartyPhoneMask: UnionType(value), optional
        :param newDialPlanName: optional
            _value_1: String(value)
        :param digitDiscardInstructionName: optional
            _value_1: String(value)
        :param callingPartyPrefixDigits: String50(value), optional
        :param newRouteFilterName: optional
            _value_1: String(value)
        :param callingLinePresentationBit: UnionType(value), optional
        :param callingPartyNumberingPlan: UnionType(value), optional
        :param callingPartyNumberType: UnionType(value), optional
        :param mlppPreemptionDisabled: boolean(value), optional
        """
        try:
            return self.client.updateCallingPartyTransformationPattern(pattern=pattern, routePartitionName=routePartitionName, **kwargs)["return"]
        except Fault as e:
            return e

    def getCallingPartyTransformationPattern(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            callingPartyPrefixDigits: String50(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            mlppPreemptionDisabled: boolean(value), optional
        """
        try:
            return self.client.getCallingPartyTransformationPattern(**kwargs)["return"]["callingPartyTransformationPattern"]
        except Fault as e:
            return e

    def removeCallingPartyTransformationPattern(self, uuid="", pattern="", routePartitionName="", dialPlanName="", routeFilterName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: optional
            _value_1: String(value)
        :param routeFilterName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeCallingPartyTransformationPattern(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''}, dialPlanName={'_value_1': ''}, routeFilterName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listCallingPartyTransformationPattern(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'callingPartyTransformationMask': '', 'useCallingPartyPhoneMask': '', 'dialPlanName': {'_value_1': ''}, 'digitDiscardInstructionName': {'_value_1': ''}, 'patternUrgency': '', 'callingPartyPrefixDigits': '', 'routeFilterName': {'_value_1': ''}, 'callingLinePresentationBit': '', 'callingPartyNumberingPlan': '', 'callingPartyNumberType': '', 'mlppPreemptionDisabled': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            dialPlanName: String(value), optional
            routeFilterName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            callingPartyPrefixDigits: String50(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            mlppPreemptionDisabled: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallingPartyTransformationPattern({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'dialPlanName': '%', 'routeFilterName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callingPartyTransformationPattern"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addSipRoutePattern(self, pattern="", description=None, usage="", routePartitionName="", blockEnable=None, callingPartyTransformationMask=None, useCallingPartyPhoneMask="", callingPartyPrefixDigits=None, callingLinePresentationBit=None, callingNamePresentationBit=None, connectedLinePresentationBit=None, connectedNamePresentationBit=None, sipTrunkName="", dnOrPatternIpv6=None, routeOnUserPart=None, useCallerCss=None, domainRoutingCssName=None):
        """
        :param sipRoutePattern: 
            pattern: String(value)
            description: String(value), optional
            usage: UnionType(value)
            routePartitionName: 
                _value_1: String(value)
            blockEnable: boolean(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value)
            callingPartyPrefixDigits: String50(value), optional
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            sipTrunkName: 
                _value_1: String(value)
            dnOrPatternIpv6: String(value), optional
            routeOnUserPart: boolean(value), optional
            useCallerCss: boolean(value), optional
            domainRoutingCssName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addSipRoutePattern({"pattern": pattern, "description": description, "usage": usage, "routePartitionName": routePartitionName, "blockEnable": blockEnable, "callingPartyTransformationMask": callingPartyTransformationMask, "useCallingPartyPhoneMask": useCallingPartyPhoneMask, "callingPartyPrefixDigits": callingPartyPrefixDigits, "callingLinePresentationBit": callingLinePresentationBit, "callingNamePresentationBit": callingNamePresentationBit, "connectedLinePresentationBit": connectedLinePresentationBit, "connectedNamePresentationBit": connectedNamePresentationBit, "sipTrunkName": sipTrunkName, "dnOrPatternIpv6": dnOrPatternIpv6, "routeOnUserPart": routeOnUserPart, "useCallerCss": useCallerCss, "domainRoutingCssName": domainRoutingCssName})["return"]
        except Fault as e:
            return e

    def updateSipRoutePattern(self, pattern="", routePartitionName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param blockEnable: boolean(value), optional
        :param callingPartyTransformationMask: String(value), optional
        :param useCallingPartyPhoneMask: UnionType(value), optional
        :param callingPartyPrefixDigits: String50(value), optional
        :param callingLinePresentationBit: UnionType(value), optional
        :param callingNamePresentationBit: UnionType(value), optional
        :param connectedLinePresentationBit: UnionType(value), optional
        :param connectedNamePresentationBit: UnionType(value), optional
        :param sipTrunkName: optional
            _value_1: String(value)
        :param dnOrPatternIpv6: String(value), optional
        :param routeOnUserPart: boolean(value), optional
        :param useCallerCss: boolean(value), optional
        :param domainRoutingCssName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateSipRoutePattern(pattern=pattern, routePartitionName=routePartitionName, **kwargs)["return"]
        except Fault as e:
            return e

    def getSipRoutePattern(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            sipTrunkName: optional
                _value_1: String(value)
            dnOrPatternIpv6: String(value), optional
            routeOnUserPart: boolean(value), optional
            useCallerCss: boolean(value), optional
            domainRoutingCssName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getSipRoutePattern(**kwargs)["return"]["sipRoutePattern"]
        except Fault as e:
            return e

    def removeSipRoutePattern(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        """
        try:
            return self.client.removeSipRoutePattern(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listSipRoutePattern(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'blockEnable': '', 'callingPartyTransformationMask': '', 'useCallingPartyPhoneMask': '', 'callingPartyPrefixDigits': '', 'callingLinePresentationBit': '', 'callingNamePresentationBit': '', 'connectedLinePresentationBit': '', 'connectedNamePresentationBit': '', 'sipTrunkName': {'_value_1': ''}, 'dnOrPatternIpv6': '', 'routeOnUserPart': '', 'useCallerCss': '', 'domainRoutingCssName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            dnOrPatternIpv6: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            sipTrunkName: optional
                _value_1: String(value)
            dnOrPatternIpv6: String(value), optional
            routeOnUserPart: boolean(value), optional
            useCallerCss: boolean(value), optional
            domainRoutingCssName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSipRoutePattern({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'dnOrPatternIpv6': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["sipRoutePattern"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addHuntPilot(self, pattern="", description=None, routePartitionName=None, blockEnable="", calledPartyTransformationMask=None, callingPartyTransformationMask=None, useCallingPartyPhoneMask="", callingPartyPrefixDigits=None, dialPlanName=None, digitDiscardInstructionName=None, patternUrgency=None, prefixDigitsOut=None, routeFilterName=None, callingLinePresentationBit=None, callingNamePresentationBit=None, connectedLinePresentationBit=None, connectedNamePresentationBit=None, patternPrecedence=None, provideOutsideDialtone=None, callingPartyNumberingPlan=None, callingPartyNumberType=None, calledPartyNumberingPlan=None, calledPartyNumberType=None, huntListName="", parkMonForwardNoRetrieve=None, alertingName=None, asciiAlertingName=None, e164Mask=None, aarNeighborhoodName=None, forwardHuntNoAnswer=None, forwardHuntBusy=None, callPickupGroupName=None, maxHuntduration=None, releaseClause=None, displayConnectedNumber=None, queueCalls=None):
        """
        :param huntPilot: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value)
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value)
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            huntListName: 
                _value_1: String(value)
            parkMonForwardNoRetrieve: optional
                usePersonalPreferences: boolean(value), optional
                destination: String50(value), optional
                    _value_1: String(value)
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            e164Mask: String(value), optional
            aarNeighborhoodName: optional
                _value_1: String(value)
            forwardHuntNoAnswer: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            forwardHuntBusy: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            callPickupGroupName: optional
                _value_1: String(value)
            maxHuntduration: UnionType(value), optional
            releaseClause: UnionType(value), optional
            displayConnectedNumber: boolean(value), optional
            queueCalls: optional
                maxCallersInQueue: UnionType(value)
                queueFullDestination: String50(value), optional
                    _value_1: String(value)
                maxWaitTimeInQueue: UnionType(value)
                maxWaitTimeDestination: String50(value), optional
                    _value_1: String(value)
                noAgentDestination: String50(value), optional
                    _value_1: String(value)
                networkHoldMohAudioSourceID: UnionType(value), optional
        """
        try:
            return self.client.addHuntPilot({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "blockEnable": blockEnable, "calledPartyTransformationMask": calledPartyTransformationMask, "callingPartyTransformationMask": callingPartyTransformationMask, "useCallingPartyPhoneMask": useCallingPartyPhoneMask, "callingPartyPrefixDigits": callingPartyPrefixDigits, "dialPlanName": dialPlanName, "digitDiscardInstructionName": digitDiscardInstructionName, "patternUrgency": patternUrgency, "prefixDigitsOut": prefixDigitsOut, "routeFilterName": routeFilterName, "callingLinePresentationBit": callingLinePresentationBit, "callingNamePresentationBit": callingNamePresentationBit, "connectedLinePresentationBit": connectedLinePresentationBit, "connectedNamePresentationBit": connectedNamePresentationBit, "patternPrecedence": patternPrecedence, "provideOutsideDialtone": provideOutsideDialtone, "callingPartyNumberingPlan": callingPartyNumberingPlan, "callingPartyNumberType": callingPartyNumberType, "calledPartyNumberingPlan": calledPartyNumberingPlan, "calledPartyNumberType": calledPartyNumberType, "huntListName": huntListName, "parkMonForwardNoRetrieve": parkMonForwardNoRetrieve, "alertingName": alertingName, "asciiAlertingName": asciiAlertingName, "e164Mask": e164Mask, "aarNeighborhoodName": aarNeighborhoodName, "forwardHuntNoAnswer": forwardHuntNoAnswer, "forwardHuntBusy": forwardHuntBusy, "callPickupGroupName": callPickupGroupName, "maxHuntduration": maxHuntduration, "releaseClause": releaseClause, "displayConnectedNumber": displayConnectedNumber, "queueCalls": queueCalls})["return"]
        except Fault as e:
            return e

    def updateHuntPilot(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param blockEnable: boolean(value), optional
        :param calledPartyTransformationMask: String(value), optional
        :param callingPartyTransformationMask: String(value), optional
        :param useCallingPartyPhoneMask: UnionType(value), optional
        :param callingPartyPrefixDigits: String50(value), optional
        :param dialPlanName: optional
            _value_1: String(value)
        :param digitDiscardInstructionName: optional
            _value_1: String(value)
        :param patternUrgency: boolean(value), optional
        :param prefixDigitsOut: String(value), optional
        :param routeFilterName: optional
            _value_1: String(value)
        :param callingLinePresentationBit: UnionType(value), optional
        :param callingNamePresentationBit: UnionType(value), optional
        :param connectedLinePresentationBit: UnionType(value), optional
        :param connectedNamePresentationBit: UnionType(value), optional
        :param patternPrecedence: UnionType(value), optional
        :param provideOutsideDialtone: boolean(value), optional
        :param callingPartyNumberingPlan: UnionType(value), optional
        :param callingPartyNumberType: UnionType(value), optional
        :param calledPartyNumberingPlan: UnionType(value), optional
        :param calledPartyNumberType: UnionType(value), optional
        :param huntListName: optional
            _value_1: String(value)
        :param parkMonForwardNoRetrieve: optional
            usePersonalPreferences: boolean(value), optional
            destination: String50(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
        :param alertingName: String50(value), optional
        :param asciiAlertingName: String32(value), optional
        :param e164Mask: String(value), optional
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param forwardHuntNoAnswer: optional
            usePersonalPreferences: boolean(value), optional
            destination: String(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
        :param forwardHuntBusy: optional
            usePersonalPreferences: boolean(value), optional
            destination: String(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
        :param callPickupGroupName: optional
            _value_1: String(value)
        :param maxHuntduration: UnionType(value), optional
        :param releaseClause: UnionType(value), optional
        :param displayConnectedNumber: boolean(value), optional
        :param queueCalls: optional
            maxCallersInQueue: UnionType(value)
            queueFullDestination: String50(value), optional
            callingSearchSpacePilotQueueFull: optional
                _value_1: String(value)
            maxWaitTimeInQueue: UnionType(value)
            maxWaitTimeDestination: String50(value), optional
            callingSearchSpaceMaxWaitTime: optional
                _value_1: String(value)
            noAgentDestination: String50(value), optional
            callingSearchSpaceNoAgent: optional
                _value_1: String(value)
            networkHoldMohAudioSourceID: UnionType(value), optional
        """
        try:
            return self.client.updateHuntPilot(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getHuntPilot(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            huntListName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieve: optional
                usePersonalPreferences: boolean(value), optional
                destination: String50(value), optional
                    _value_1: String(value)
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            e164Mask: String(value), optional
            aarNeighborhoodName: optional
                _value_1: String(value)
            forwardHuntNoAnswer: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            forwardHuntBusy: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            callPickupGroupName: optional
                _value_1: String(value)
            maxHuntduration: UnionType(value), optional
            releaseClause: UnionType(value), optional
            displayConnectedNumber: boolean(value), optional
            queueCalls: optional
                maxCallersInQueue: UnionType(value), optional
                queueFullDestination: String50(value), optional
                    _value_1: String(value)
                maxWaitTimeInQueue: UnionType(value), optional
                maxWaitTimeDestination: String50(value), optional
                    _value_1: String(value)
                noAgentDestination: String50(value), optional
                    _value_1: String(value)
                networkHoldMohAudioSourceID: UnionType(value), optional
        """
        try:
            return self.client.getHuntPilot(**kwargs)["return"]["huntPilot"]
        except Fault as e:
            return e

    def removeHuntPilot(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeHuntPilot(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listHuntPilot(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'blockEnable': '', 'calledPartyTransformationMask': '', 'callingPartyTransformationMask': '', 'useCallingPartyPhoneMask': '', 'callingPartyPrefixDigits': '', 'dialPlanName': {'_value_1': ''}, 'digitDiscardInstructionName': {'_value_1': ''}, 'patternUrgency': '', 'prefixDigitsOut': '', 'routeFilterName': {'_value_1': ''}, 'callingLinePresentationBit': '', 'callingNamePresentationBit': '', 'connectedLinePresentationBit': '', 'connectedNamePresentationBit': '', 'patternPrecedence': '', 'provideOutsideDialtone': '', 'callingPartyNumberingPlan': '', 'callingPartyNumberType': '', 'calledPartyNumberingPlan': '', 'calledPartyNumberType': '', 'huntListName': {'_value_1': ''}, 'parkMonForwardNoRetrieve': {'usePersonalPreferences': '', 'destination': ''}, 'alertingName': '', 'asciiAlertingName': '', 'aarNeighborhoodName': {'_value_1': ''}, 'forwardHuntNoAnswer': {'usePersonalPreferences': '', 'destination': ''}, 'forwardHuntBusy': {'usePersonalPreferences': '', 'destination': ''}, 'callPickupGroupName': {'_value_1': ''}, 'maxHuntduration': '', 'releaseClause': '', 'displayConnectedNumber': '', 'queueCalls': {'maxCallersInQueue': '', 'queueFullDestination': '', 'maxWaitTimeInQueue': '', 'maxWaitTimeDestination': '', 'noAgentDestination': '', 'networkHoldMohAudioSourceID': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            digitDiscardInstructionName: optional
                _value_1: String(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            patternPrecedence: UnionType(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            huntListName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieve: optional
                usePersonalPreferences: boolean(value), optional
                destination: String50(value), optional
                    _value_1: String(value)
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            aarNeighborhoodName: optional
                _value_1: String(value)
            forwardHuntNoAnswer: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            forwardHuntBusy: optional
                usePersonalPreferences: boolean(value), optional
                destination: String(value), optional
                    _value_1: String(value)
            callPickupGroupName: optional
                _value_1: String(value)
            maxHuntduration: UnionType(value), optional
            releaseClause: UnionType(value), optional
            displayConnectedNumber: boolean(value), optional
            queueCalls: optional
                maxCallersInQueue: UnionType(value), optional
                queueFullDestination: String50(value), optional
                    _value_1: String(value)
                maxWaitTimeInQueue: UnionType(value), optional
                maxWaitTimeDestination: String50(value), optional
                    _value_1: String(value)
                noAgentDestination: String50(value), optional
                    _value_1: String(value)
                networkHoldMohAudioSourceID: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listHuntPilot({'pattern': '%', 'description': '%', 'routePartitionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["huntPilot"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addRoutePattern(self, pattern="", description=None, routePartitionName="", blockEnable="", calledPartyTransformationMask=None, callingPartyTransformationMask=None, useCallingPartyPhoneMask="", callingPartyPrefixDigits=None, dialPlanName="", digitDiscardInstructionName="", networkLocation="", patternUrgency=None, prefixDigitsOut="", routeFilterName="", callingLinePresentationBit=None, callingNamePresentationBit=None, connectedLinePresentationBit=None, connectedNamePresentationBit=None, supportOverlapSending=None, patternPrecedence=None, releaseClause=None, allowDeviceOverride=None, provideOutsideDialtone=None, callingPartyNumberingPlan=None, callingPartyNumberType=None, calledPartyNumberingPlan=None, calledPartyNumberType=None, destination="", authorizationCodeRequired=None, authorizationLevelRequired=None, clientCodeRequired=None, isdnNsfInfoElement=None, resourcePriorityNamespaceName=None, routeClass=None, enableDccEnforcement=None, blockedCallPercentage=None, externalCallControl=None, isEmergencyServiceNumber=None):
        """
        :param routePattern: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: 
                _value_1: String(value)
            blockEnable: boolean(value)
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value)
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: 
                _value_1: String(value)
            digitDiscardInstructionName: 
                _value_1: String(value)
            networkLocation: UnionType(value)
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value)
            routeFilterName: 
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            supportOverlapSending: boolean(value), optional
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            allowDeviceOverride: boolean(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            destination: 
                    _value_1: String(value)
                    _value_1: String(value)
            authorizationCodeRequired: boolean(value), optional
            authorizationLevelRequired: UnionType(value), optional
            clientCodeRequired: boolean(value), optional
            isdnNsfInfoElement: optional
                cic: String(value), optional
                networkServiceProtocol: UnionType(value)
                networkService: UnionType(value), optional
                paramValue: String(value), optional
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeClass: UnionType(value), optional
            enableDccEnforcement: boolean(value), optional
            blockedCallPercentage: String(value), optional
            externalCallControl: optional
                _value_1: String(value)
            isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.addRoutePattern({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "blockEnable": blockEnable, "calledPartyTransformationMask": calledPartyTransformationMask, "callingPartyTransformationMask": callingPartyTransformationMask, "useCallingPartyPhoneMask": useCallingPartyPhoneMask, "callingPartyPrefixDigits": callingPartyPrefixDigits, "dialPlanName": dialPlanName, "digitDiscardInstructionName": digitDiscardInstructionName, "networkLocation": networkLocation, "patternUrgency": patternUrgency, "prefixDigitsOut": prefixDigitsOut, "routeFilterName": routeFilterName, "callingLinePresentationBit": callingLinePresentationBit, "callingNamePresentationBit": callingNamePresentationBit, "connectedLinePresentationBit": connectedLinePresentationBit, "connectedNamePresentationBit": connectedNamePresentationBit, "supportOverlapSending": supportOverlapSending, "patternPrecedence": patternPrecedence, "releaseClause": releaseClause, "allowDeviceOverride": allowDeviceOverride, "provideOutsideDialtone": provideOutsideDialtone, "callingPartyNumberingPlan": callingPartyNumberingPlan, "callingPartyNumberType": callingPartyNumberType, "calledPartyNumberingPlan": calledPartyNumberingPlan, "calledPartyNumberType": calledPartyNumberType, "destination": destination, "authorizationCodeRequired": authorizationCodeRequired, "authorizationLevelRequired": authorizationLevelRequired, "clientCodeRequired": clientCodeRequired, "isdnNsfInfoElement": isdnNsfInfoElement, "resourcePriorityNamespaceName": resourcePriorityNamespaceName, "routeClass": routeClass, "enableDccEnforcement": enableDccEnforcement, "blockedCallPercentage": blockedCallPercentage, "externalCallControl": externalCallControl, "isEmergencyServiceNumber": isEmergencyServiceNumber})["return"]
        except Fault as e:
            return e

    def updateRoutePattern(self, pattern="", routePartitionName="", dialPlanName="", routeFilterName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: 
            _value_1: String(value)
        :param routeFilterName: 
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param blockEnable: boolean(value), optional
        :param calledPartyTransformationMask: String(value), optional
        :param callingPartyTransformationMask: String(value), optional
        :param useCallingPartyPhoneMask: UnionType(value), optional
        :param callingPartyPrefixDigits: String50(value), optional
        :param newDialPlanName: optional
            _value_1: String(value)
        :param digitDiscardInstructionName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param patternUrgency: boolean(value), optional
        :param prefixDigitsOut: String(value), optional
        :param newRouteFilterName: optional
            _value_1: String(value)
        :param callingLinePresentationBit: UnionType(value), optional
        :param callingNamePresentationBit: UnionType(value), optional
        :param connectedLinePresentationBit: UnionType(value), optional
        :param connectedNamePresentationBit: UnionType(value), optional
        :param supportOverlapSending: boolean(value), optional
        :param patternPrecedence: UnionType(value), optional
        :param releaseClause: UnionType(value), optional
        :param allowDeviceOverride: boolean(value), optional
        :param provideOutsideDialtone: boolean(value), optional
        :param callingPartyNumberingPlan: UnionType(value), optional
        :param callingPartyNumberType: UnionType(value), optional
        :param calledPartyNumberingPlan: UnionType(value), optional
        :param calledPartyNumberType: UnionType(value), optional
        :param destination: optional
            gatewayName: optional
                _value_1: String(value)
            routeListName: optional
                _value_1: String(value)
        :param authorizationCodeRequired: boolean(value), optional
        :param authorizationLevelRequired: UnionType(value), optional
        :param clientCodeRequired: boolean(value), optional
        :param isdnNsfInfoElement: optional
            cic: String(value), optional
            networkServiceProtocol: UnionType(value)
            networkService: UnionType(value), optional
            paramValue: String(value), optional
        :param resourcePriorityNamespaceName: optional
            _value_1: String(value)
        :param routeClass: UnionType(value), optional
        :param enableDccEnforcement: boolean(value), optional
        :param blockedCallPercentage: String(value), optional
        :param externalCallControl: optional
            _value_1: String(value)
        :param isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.updateRoutePattern(pattern=pattern, routePartitionName=routePartitionName, dialPlanName=dialPlanName, routeFilterName=routeFilterName, **kwargs)["return"]
        except Fault as e:
            return e

    def getRoutePattern(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: 
            _value_1: String(value)
        :param routeFilterName: 
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            dialPlanWizardGenId: UnionType(value), optional
            digitDiscardInstructionName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            supportOverlapSending: boolean(value), optional
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            allowDeviceOverride: boolean(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            destination: optional
                    _value_1: String(value)
                    _value_1: String(value)
            authorizationCodeRequired: boolean(value), optional
            authorizationLevelRequired: UnionType(value), optional
            clientCodeRequired: boolean(value), optional
            withTag: String50(value), optional
            withValueClause: String255(value), optional
            isdnNsfInfoElement: optional
                cic: String(value), optional
                networkServiceProtocol: UnionType(value), optional
                networkService: UnionType(value), optional
                paramValue: String(value), optional
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeClass: UnionType(value), optional
            enableDccEnforcement: boolean(value), optional
            blockedCallPercentage: String(value), optional
            externalCallControl: optional
                _value_1: String(value)
            isEmergencyServiceNumber: boolean(value), optional
        """
        try:
            return self.client.getRoutePattern(**kwargs)["return"]["routePattern"]
        except Fault as e:
            return e

    def removeRoutePattern(self, uuid="", pattern="", routePartitionName="", dialPlanName="", routeFilterName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: 
            _value_1: String(value)
        :param dialPlanName: 
            _value_1: String(value)
        :param routeFilterName: 
            _value_1: String(value)
        """
        try:
            return self.client.removeRoutePattern(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''}, dialPlanName={'_value_1': ''}, routeFilterName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listRoutePattern(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'blockEnable': '', 'calledPartyTransformationMask': '', 'callingPartyTransformationMask': '', 'useCallingPartyPhoneMask': '', 'callingPartyPrefixDigits': '', 'dialPlanName': {'_value_1': ''}, 'dialPlanWizardGenId': '', 'digitDiscardInstructionName': {'_value_1': ''}, 'networkLocation': '', 'patternUrgency': '', 'prefixDigitsOut': '', 'routeFilterName': {'_value_1': ''}, 'callingLinePresentationBit': '', 'callingNamePresentationBit': '', 'connectedLinePresentationBit': '', 'connectedNamePresentationBit': '', 'supportOverlapSending': '', 'patternPrecedence': '', 'releaseClause': '', 'allowDeviceOverride': '', 'provideOutsideDialtone': '', 'callingPartyNumberingPlan': '', 'callingPartyNumberType': '', 'calledPartyNumberingPlan': '', 'calledPartyNumberType': '', 'authorizationCodeRequired': '', 'authorizationLevelRequired': '', 'clientCodeRequired': '', 'withTag': '', 'withValueClause': '', 'resourcePriorityNamespaceName': {'_value_1': ''}, 'routeClass': '', 'externalCallControl': {'_value_1': ''}, 'isEmergencyServiceNumber': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
            isEmergencyServiceNumber: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            blockEnable: boolean(value), optional
            calledPartyTransformationMask: String(value), optional
            callingPartyTransformationMask: String(value), optional
            useCallingPartyPhoneMask: UnionType(value), optional
            callingPartyPrefixDigits: String50(value), optional
            dialPlanName: optional
                _value_1: String(value)
            dialPlanWizardGenId: UnionType(value), optional
            digitDiscardInstructionName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            patternUrgency: boolean(value), optional
            prefixDigitsOut: String(value), optional
            routeFilterName: optional
                _value_1: String(value)
            callingLinePresentationBit: UnionType(value), optional
            callingNamePresentationBit: UnionType(value), optional
            connectedLinePresentationBit: UnionType(value), optional
            connectedNamePresentationBit: UnionType(value), optional
            supportOverlapSending: boolean(value), optional
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            allowDeviceOverride: boolean(value), optional
            provideOutsideDialtone: boolean(value), optional
            callingPartyNumberingPlan: UnionType(value), optional
            callingPartyNumberType: UnionType(value), optional
            calledPartyNumberingPlan: UnionType(value), optional
            calledPartyNumberType: UnionType(value), optional
            authorizationCodeRequired: boolean(value), optional
            authorizationLevelRequired: UnionType(value), optional
            clientCodeRequired: boolean(value), optional
            withTag: String50(value), optional
            withValueClause: String255(value), optional
            resourcePriorityNamespaceName: optional
                _value_1: String(value)
            routeClass: UnionType(value), optional
            externalCallControl: optional
                _value_1: String(value)
            isEmergencyServiceNumber: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRoutePattern({'pattern': '%', 'description': '%', 'routePartitionName': '%', 'isEmergencyServiceNumber': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routePattern"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addApplicationDialRules(self, name="", description=None, numberBeginWith=None, numberOfDigits="", digitsToBeRemoved="", prefixPattern=None, priority=""):
        """
        :param applicationDialRules: 
            name: String(value)
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value)
            digitsToBeRemoved: UnionType(value)
            prefixPattern: String(value), optional
            priority: UnionType(value)
        """
        try:
            return self.client.addApplicationDialRules({"name": name, "description": description, "numberBeginWith": numberBeginWith, "numberOfDigits": numberOfDigits, "digitsToBeRemoved": digitsToBeRemoved, "prefixPattern": prefixPattern, "priority": priority})["return"]
        except Fault as e:
            return e

    def updateApplicationDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param numberBeginWith: String(value), optional
        :param numberOfDigits: UnionType(value), optional
        :param digitsToBeRemoved: UnionType(value), optional
        :param prefixPattern: String(value), optional
        :param priority: UnionType(value), optional
        """
        try:
            return self.client.updateApplicationDialRules(**kwargs)["return"]
        except Fault as e:
            return e

    def getApplicationDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value), optional
            digitsToBeRemoved: UnionType(value), optional
            prefixPattern: String(value), optional
            priority: UnionType(value), optional
        """
        try:
            return self.client.getApplicationDialRules(**kwargs)["return"]["applicationDialRules"]
        except Fault as e:
            return e

    def removeApplicationDialRules(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeApplicationDialRules(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listApplicationDialRules(self, returnedTags={'name': '', 'description': '', 'numberBeginWith': '', 'numberOfDigits': '', 'digitsToBeRemoved': '', 'prefixPattern': '', 'priority': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value), optional
            digitsToBeRemoved: UnionType(value), optional
            prefixPattern: String(value), optional
            priority: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listApplicationDialRules({'name': '%', 'description': '%', 'numberBeginWith': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["applicationDialRules"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addDirectoryLookupDialRules(self, name="", description=None, numberBeginWith=None, numberOfDigits="", digitsToBeRemoved="", prefixPattern=None, priority=""):
        """
        :param directoryLookupDialRules: 
            name: String(value)
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value)
            digitsToBeRemoved: UnionType(value)
            prefixPattern: String(value), optional
            priority: UnionType(value)
        """
        try:
            return self.client.addDirectoryLookupDialRules({"name": name, "description": description, "numberBeginWith": numberBeginWith, "numberOfDigits": numberOfDigits, "digitsToBeRemoved": digitsToBeRemoved, "prefixPattern": prefixPattern, "priority": priority})["return"]
        except Fault as e:
            return e

    def updateDirectoryLookupDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param numberBeginWith: String(value), optional
        :param numberOfDigits: UnionType(value), optional
        :param digitsToBeRemoved: UnionType(value), optional
        :param prefixPattern: String(value), optional
        :param priority: UnionType(value), optional
        """
        try:
            return self.client.updateDirectoryLookupDialRules(**kwargs)["return"]
        except Fault as e:
            return e

    def getDirectoryLookupDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value), optional
            digitsToBeRemoved: UnionType(value), optional
            prefixPattern: String(value), optional
            priority: UnionType(value), optional
        """
        try:
            return self.client.getDirectoryLookupDialRules(**kwargs)["return"]["directoryLookupDialRules"]
        except Fault as e:
            return e

    def removeDirectoryLookupDialRules(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDirectoryLookupDialRules(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDirectoryLookupDialRules(self, returnedTags={'name': '', 'description': '', 'numberBeginWith': '', 'numberOfDigits': '', 'digitsToBeRemoved': '', 'prefixPattern': '', 'priority': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            numberBeginWith: String(value), optional
            numberOfDigits: UnionType(value), optional
            digitsToBeRemoved: UnionType(value), optional
            prefixPattern: String(value), optional
            priority: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDirectoryLookupDialRules({'name': '%', 'description': '%', 'numberBeginWith': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["directoryLookupDialRules"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addPhoneSecurityProfile(self, phoneType="", protocol="", name="", description=None, deviceSecurityMode=None, authenticationMode=None, keySize=None, keyOrder=None, ecKeySize=None, tftpEncryptedConfig=None, nonceValidityTime=None, transportType=None, sipPhonePort=None, enableDigestAuthentication=None, excludeDigestCredentials=None):
        """
        :param phoneSecurityProfile: 
            phoneType: UnionType(value)
            protocol: UnionType(value)
            name: String(value)
            description: String(value), optional
            deviceSecurityMode: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            tftpEncryptedConfig: boolean(value), optional
            nonceValidityTime: UnionType(value), optional
            transportType: UnionType(value), optional
            sipPhonePort: UnionType(value), optional
            enableDigestAuthentication: boolean(value), optional
            excludeDigestCredentials: boolean(value), optional
        """
        try:
            return self.client.addPhoneSecurityProfile({"phoneType": phoneType, "protocol": protocol, "name": name, "description": description, "deviceSecurityMode": deviceSecurityMode, "authenticationMode": authenticationMode, "keySize": keySize, "keyOrder": keyOrder, "ecKeySize": ecKeySize, "tftpEncryptedConfig": tftpEncryptedConfig, "nonceValidityTime": nonceValidityTime, "transportType": transportType, "sipPhonePort": sipPhonePort, "enableDigestAuthentication": enableDigestAuthentication, "excludeDigestCredentials": excludeDigestCredentials})["return"]
        except Fault as e:
            return e

    def updatePhoneSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param deviceSecurityMode: UnionType(value), optional
        :param authenticationMode: UnionType(value), optional
        :param keySize: UnionType(value), optional
        :param keyOrder: UnionType(value), optional
        :param ecKeySize: UnionType(value), optional
        :param tftpEncryptedConfig: boolean(value), optional
        :param nonceValidityTime: UnionType(value), optional
        :param transportType: UnionType(value), optional
        :param sipPhonePort: UnionType(value), optional
        :param enableDigestAuthentication: boolean(value), optional
        :param excludeDigestCredentials: boolean(value), optional
        """
        try:
            return self.client.updatePhoneSecurityProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getPhoneSecurityProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            phoneType: UnionType(value), optional
            protocol: UnionType(value), optional
            name: String(value), optional
            description: String(value), optional
            deviceSecurityMode: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            tftpEncryptedConfig: boolean(value), optional
            nonceValidityTime: UnionType(value), optional
            transportType: UnionType(value), optional
            sipPhonePort: UnionType(value), optional
            enableDigestAuthentication: boolean(value), optional
            excludeDigestCredentials: boolean(value), optional
        """
        try:
            return self.client.getPhoneSecurityProfile(**kwargs)["return"]["phoneSecurityProfile"]
        except Fault as e:
            return e

    def removePhoneSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePhoneSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPhoneSecurityProfile(self, returnedTags={'phoneType': '', 'protocol': '', 'name': '', 'description': '', 'deviceSecurityMode': '', 'authenticationMode': '', 'keySize': '', 'keyOrder': '', 'ecKeySize': '', 'tftpEncryptedConfig': '', 'nonceValidityTime': '', 'transportType': '', 'sipPhonePort': '', 'enableDigestAuthentication': '', 'excludeDigestCredentials': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            phoneType: UnionType(value), optional
            protocol: UnionType(value), optional
            name: String(value), optional
            description: String(value), optional
            deviceSecurityMode: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            tftpEncryptedConfig: boolean(value), optional
            nonceValidityTime: UnionType(value), optional
            transportType: UnionType(value), optional
            sipPhonePort: UnionType(value), optional
            enableDigestAuthentication: boolean(value), optional
            excludeDigestCredentials: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPhoneSecurityProfile({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["phoneSecurityProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetPhoneSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetPhoneSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyPhoneSecurityProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyPhoneSecurityProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addSipDialRules(self, dialPattern="", name="", description=None, patterns=None, plars=None):
        """
        :param sipDialRules: 
            dialPattern: UnionType(value)
            name: String50(value)
            description: String(value), optional
            patterns: optional
                    description: String(value), optional
                    selectionOrder: UnionType(value)
            plars: optional
                    description: String(value), optional
                    selectionOrder: UnionType(value)
        """
        try:
            return self.client.addSipDialRules({"dialPattern": dialPattern, "name": name, "description": description, "patterns": patterns, "plars": plars})["return"]
        except Fault as e:
            return e

    def updateSipDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param description: String(value), optional
        :param removePatterns: optional
            pattern: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        :param addPatterns: optional
            pattern: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        :param patterns: optional
            pattern: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        :param removePlars: optional
            plar: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        :param addPlars: optional
            plar: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        :param plars: optional
            plar: 
                description: String(value), optional
                selectionOrder: UnionType(value)
        """
        try:
            return self.client.updateSipDialRules(**kwargs)["return"]
        except Fault as e:
            return e

    def getSipDialRules(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            dialPattern: UnionType(value), optional
            name: String50(value), optional
            description: String(value), optional
            patterns: optional
                    description: String(value), optional
                    selectionOrder: UnionType(value), optional
            plars: optional
                    description: String(value), optional
                    selectionOrder: UnionType(value), optional
        """
        try:
            return self.client.getSipDialRules(**kwargs)["return"]["sipDialRules"]
        except Fault as e:
            return e

    def removeSipDialRules(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSipDialRules(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSipDialRules(self, returnedTags={'dialPattern': '', 'name': '', 'description': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            dialPattern: String(value), optional
            name: String(value), optional
        :param returnedTags: 
            dialPattern: UnionType(value), optional
            name: String50(value), optional
            description: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listSipDialRules({'dialPattern': '%', 'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["sipDialRules"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addConferenceBridge(self, name="", description=None, product="", devicePoolName="", commonDeviceConfigName=None, locationName="", subUnit=None, loadInformation=None, vendorConfig=None, maximumCapacity=None, useTrustedRelayPoint=None, securityProfileName=None, destinationAddress=None, mcuConferenceBridgeSipPort=None, sipProfile=None, srtpAllowed=None, normalizationScript=None, enableTrace=None, normalizationScriptInfos=None, userName=None, password=None, httpPort=None, useHttps=None, addresses=None, conferenceBridgePrefix=None, allowCFBControlOfCallSecurityIcon=None, overrideSIPTrunkAddress=None, sipTrunkName=None):
        """
        :param conferenceBridge: 
            name: UniqueName128(value)
            description: String128(value), optional
            product: UnionType(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: 
                _value_1: String(value)
            subUnit: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            maximumCapacity: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            securityProfileName: optional
                _value_1: String(value)
            destinationAddress: String(value), optional
            mcuConferenceBridgeSipPort: UnionType(value), optional
            sipProfile: optional
                _value_1: String(value)
            srtpAllowed: boolean(value), optional
            normalizationScript: optional
                _value_1: String(value)
            enableTrace: boolean(value), optional
            normalizationScriptInfos: optional
                    parameterName: String128(value), optional
                    parameterValue: String128(value), optional
            userName: String(value), optional
            password: String(value), optional
            httpPort: UnionType(value), optional
            useHttps: boolean(value), optional
            addresses: optional
                address: String(value), optional
            conferenceBridgePrefix: String(value), optional
            allowCFBControlOfCallSecurityIcon: boolean(value), optional
            overrideSIPTrunkAddress: boolean(value), optional
            sipTrunkName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addConferenceBridge({"name": name, "description": description, "product": product, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "locationName": locationName, "subUnit": subUnit, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "maximumCapacity": maximumCapacity, "useTrustedRelayPoint": useTrustedRelayPoint, "securityProfileName": securityProfileName, "destinationAddress": destinationAddress, "mcuConferenceBridgeSipPort": mcuConferenceBridgeSipPort, "sipProfile": sipProfile, "srtpAllowed": srtpAllowed, "normalizationScript": normalizationScript, "enableTrace": enableTrace, "normalizationScriptInfos": normalizationScriptInfos, "userName": userName, "password": password, "httpPort": httpPort, "useHttps": useHttps, "addresses": addresses, "conferenceBridgePrefix": conferenceBridgePrefix, "allowCFBControlOfCallSecurityIcon": allowCFBControlOfCallSecurityIcon, "overrideSIPTrunkAddress": overrideSIPTrunkAddress, "sipTrunkName": sipTrunkName})["return"]
        except Fault as e:
            return e

    def updateConferenceBridge(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueName128(value), optional
        :param description: String128(value), optional
        :param product: UnionType(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param maximumCapacity: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param securityProfileName: optional
            _value_1: String(value)
        :param destinationAddress: String(value), optional
        :param mcuConferenceBridgeSipPort: UnionType(value), optional
        :param sipProfile: optional
            _value_1: String(value)
        :param srtpAllowed: boolean(value), optional
        :param normalizationScript: optional
            _value_1: String(value)
        :param enableTrace: boolean(value), optional
        :param normalizationScriptInfos: optional
            normalizationScriptInfo: optional
                parameterName: String128(value), optional
                parameterValue: String128(value), optional
        :param userName: String(value), optional
        :param password: String(value), optional
        :param httpPort: UnionType(value), optional
        :param useHttps: boolean(value), optional
        :param addresses: optional
            address: String(value), optional
        :param conferenceBridgePrefix: String(value), optional
        :param allowCFBControlOfCallSecurityIcon: boolean(value), optional
        :param overrideSIPTrunkAddress: boolean(value), optional
        :param sipTrunkName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateConferenceBridge(**kwargs)["return"]
        except Fault as e:
            return e

    def getConferenceBridge(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueName128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            subUnit: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            maximumCapacity: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            securityProfileName: optional
                _value_1: String(value)
            destinationAddress: String(value), optional
            mcuConferenceBridgeSipPort: UnionType(value), optional
            sipProfile: optional
                _value_1: String(value)
            srtpAllowed: boolean(value), optional
            normalizationScript: optional
                _value_1: String(value)
            enableTrace: boolean(value), optional
            normalizationScriptInfos: optional
                    parameterName: String128(value), optional
                    parameterValue: String128(value), optional
            userName: String(value), optional
            password: String(value), optional
            httpPort: UnionType(value), optional
            useHttps: boolean(value), optional
            addresses: optional
                address: String(value), optional
            conferenceBridgePrefix: String(value), optional
            allowCFBControlOfCallSecurityIcon: boolean(value), optional
            overrideSIPTrunkAddress: boolean(value), optional
            sipTrunkName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getConferenceBridge(**kwargs)["return"]["conferenceBridge"]
        except Fault as e:
            return e

    def removeConferenceBridge(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeConferenceBridge(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listConferenceBridge(self, returnedTags={'name': '', 'description': '', 'product': '', 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'subUnit': '', 'loadInformation': {'_value_1': ''}, 'maximumCapacity': '', 'useTrustedRelayPoint': '', 'securityProfileName': {'_value_1': ''}, 'destinationAddress': '', 'mcuConferenceBridgeSipPort': '', 'sipProfile': {'_value_1': ''}, 'srtpAllowed': '', 'normalizationScript': {'_value_1': ''}, 'enableTrace': '', 'userName': '', 'password': '', 'httpPort': '', 'useHttps': '', 'conferenceBridgePrefix': '', 'allowCFBControlOfCallSecurityIcon': '', 'overrideSIPTrunkAddress': '', 'sipTrunkName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueName128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            subUnit: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            maximumCapacity: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            securityProfileName: optional
                _value_1: String(value)
            destinationAddress: String(value), optional
            mcuConferenceBridgeSipPort: UnionType(value), optional
            sipProfile: optional
                _value_1: String(value)
            srtpAllowed: boolean(value), optional
            normalizationScript: optional
                _value_1: String(value)
            enableTrace: boolean(value), optional
            userName: String(value), optional
            password: String(value), optional
            httpPort: UnionType(value), optional
            useHttps: boolean(value), optional
            conferenceBridgePrefix: String(value), optional
            allowCFBControlOfCallSecurityIcon: boolean(value), optional
            overrideSIPTrunkAddress: boolean(value), optional
            sipTrunkName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listConferenceBridge({'name': '%', 'description': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["conferenceBridge"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetConferenceBridge(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetConferenceBridge(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartConferenceBridge(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartConferenceBridge(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyConferenceBridge(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyConferenceBridge(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def updateAnnunciator(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param useTrustedRelayPoint: UnionType(value), optional
        """
        try:
            return self.client.updateAnnunciator(**kwargs)["return"]
        except Fault as e:
            return e

    def getAnnunciator(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            useTrustedRelayPoint: UnionType(value), optional
            serverName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getAnnunciator(**kwargs)["return"]["annunciator"]
        except Fault as e:
            return e

    def listAnnunciator(self, returnedTags={'name': '', 'description': '', 'devicePoolName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'useTrustedRelayPoint': '', 'serverName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            useTrustedRelayPoint: UnionType(value), optional
            serverName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listAnnunciator({'name': '%', 'description': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["annunciator"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def updateInteractiveVoiceResponse(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param useTrustedRelayPoint: UnionType(value), optional
        """
        try:
            return self.client.updateInteractiveVoiceResponse(**kwargs)["return"]
        except Fault as e:
            return e

    def getInteractiveVoiceResponse(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            useTrustedRelayPoint: UnionType(value), optional
            serverName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getInteractiveVoiceResponse(**kwargs)["return"]["interactiveVoiceResponse"]
        except Fault as e:
            return e

    def listInteractiveVoiceResponse(self, returnedTags={'name': '', 'description': '', 'devicePoolName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'useTrustedRelayPoint': '', 'serverName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            useTrustedRelayPoint: UnionType(value), optional
            serverName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listInteractiveVoiceResponse({'name': '%', 'description': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["interactiveVoiceResponse"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addMtp(self, mtpType="", name="", description=None, devicePoolName="", trustedRelayPoint=None):
        """
        :param mtp: 
            mtpType: UnionType(value)
            name: String(value)
            description: String(value), optional
            devicePoolName: 
                _value_1: String(value)
            trustedRelayPoint: boolean(value), optional
        """
        try:
            return self.client.addMtp({"mtpType": mtpType, "name": name, "description": description, "devicePoolName": devicePoolName, "trustedRelayPoint": trustedRelayPoint})["return"]
        except Fault as e:
            return e

    def updateMtp(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param trustedRelayPoint: boolean(value), optional
        """
        try:
            return self.client.updateMtp(**kwargs)["return"]
        except Fault as e:
            return e

    def getMtp(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            mtpType: UnionType(value), optional
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            trustedRelayPoint: boolean(value), optional
        """
        try:
            return self.client.getMtp(**kwargs)["return"]["mtp"]
        except Fault as e:
            return e

    def removeMtp(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeMtp(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listMtp(self, returnedTags={'mtpType': '', 'name': '', 'description': '', 'devicePoolName': {'_value_1': ''}, 'trustedRelayPoint': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            mtpType: UnionType(value), optional
            name: String(value), optional
            description: String(value), optional
            devicePoolName: optional
                _value_1: String(value)
            trustedRelayPoint: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMtp({'name': '%', 'description': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mtp"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetMtp(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetMtp(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartMtp(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartMtp(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyMtp(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyMtp(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def updateFixedMohAudioSource(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param multicast: boolean(value), optional
        :param enable: String(value), optional
        :param initialAnnouncement: optional
            _value_1: String(value)
        :param periodicAnnouncement: optional
            _value_1: String(value)
        :param periodicAnnouncementInterval: UnionType(value), optional
        :param localeAnnouncement: UnionType(value), optional
        :param initialAnnouncementPlayed: boolean(value), optional
        """
        try:
            return self.client.updateFixedMohAudioSource(**kwargs)["return"]
        except Fault as e:
            return e

    def getFixedMohAudioSource(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            sourceId: UnionType(value), optional
            name: String(value), optional
            multicast: boolean(value), optional
            enable: String(value), optional
            initialAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncement: optional
                _value_1: String(value)
            periodicAnnouncementInterval: UnionType(value), optional
            localeAnnouncement: UnionType(value), optional
            initialAnnouncementPlayed: boolean(value), optional
        """
        try:
            return self.client.getFixedMohAudioSource(**kwargs)["return"]["fixedMohAudioSource"]
        except Fault as e:
            return e

    def updateAarGroupMatrix(self, aarGroupFromName="", aarGroupToName="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param aarGroupFromName: 
            _value_1: String(value)
        :param aarGroupToName: 
            _value_1: String(value)
        :param prefixDigit: String255(value), optional
        """
        try:
            return self.client.updateAarGroupMatrix(aarGroupFromName=aarGroupFromName, aarGroupToName=aarGroupToName, **kwargs)["return"]
        except Fault as e:
            return e

    def addRemoteDestinationProfile(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, lines=None, callInfoPrivacyStatus="", userId="", ignorePresentationIndicators=None, rerouteCallingSearchSpaceName=None, cgpnTransformationCssName=None, automatedAlternateRoutingCssName=None, useDevicePoolCgpnTransformCss=None, userLocale=None, networkLocale=None, primaryPhoneName=None, dndOption=None, dndStatus=None, mobileSmartClientProfileName=None):
        """
        :param remoteDestinationProfile: 
            name: String(value)
            description: String(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            lines: optional
                    index: UnionType(value)
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value)
                    routePartitionName: String50(value)
            callInfoPrivacyStatus: UnionType(value)
            userId: String(value)
            ignorePresentationIndicators: boolean(value), optional
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            primaryPhoneName: optional
                _value_1: String(value)
            dndOption: UnionType(value), optional
            dndStatus: boolean(value), optional
            mobileSmartClientProfileName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addRemoteDestinationProfile({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "lines": lines, "callInfoPrivacyStatus": callInfoPrivacyStatus, "userId": userId, "ignorePresentationIndicators": ignorePresentationIndicators, "rerouteCallingSearchSpaceName": rerouteCallingSearchSpaceName, "cgpnTransformationCssName": cgpnTransformationCssName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "userLocale": userLocale, "networkLocale": networkLocale, "primaryPhoneName": primaryPhoneName, "dndOption": dndOption, "dndStatus": dndStatus, "mobileSmartClientProfileName": mobileSmartClientProfileName})["return"]
        except Fault as e:
            return e

    def updateRemoteDestinationProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param lines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param callInfoPrivacyStatus: UnionType(value), optional
        :param userId: String(value), optional
        :param ignorePresentationIndicators: boolean(value), optional
        :param rerouteCallingSearchSpaceName: optional
            _value_1: String(value)
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param userLocale: UnionType(value), optional
        :param networkLocale: UnionType(value), optional
        :param primaryPhoneName: optional
            _value_1: String(value)
        :param dndOption: UnionType(value), optional
        :param dndStatus: boolean(value), optional
        :param mobileSmartClientProfileName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateRemoteDestinationProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getRemoteDestinationProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            lines: optional
                    index: UnionType(value), optional
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    dialPlanWizardId: UnionType(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value), optional
                    routePartitionName: String50(value), optional
            callInfoPrivacyStatus: UnionType(value), optional
            userId: String(value), optional
            ignorePresentationIndicators: boolean(value), optional
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            primaryPhoneName: optional
                _value_1: String(value)
            dndOption: UnionType(value), optional
            dndStatus: boolean(value), optional
            mobileSmartClientProfileName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getRemoteDestinationProfile(**kwargs)["return"]["remoteDestinationProfile"]
        except Fault as e:
            return e

    def removeRemoteDestinationProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRemoteDestinationProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRemoteDestinationProfile(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'callInfoPrivacyStatus': '', 'userId': '', 'ignorePresentationIndicators': '', 'rerouteCallingSearchSpaceName': {'_value_1': ''}, 'cgpnTransformationCssName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'userLocale': '', 'networkLocale': '', 'primaryPhoneName': {'_value_1': ''}, 'dndOption': '', 'dndStatus': '', 'mobileSmartClientProfileName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            callInfoPrivacyStatus: UnionType(value), optional
            userId: String(value), optional
            ignorePresentationIndicators: boolean(value), optional
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            primaryPhoneName: optional
                _value_1: String(value)
            dndOption: UnionType(value), optional
            dndStatus: boolean(value), optional
            mobileSmartClientProfileName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRemoteDestinationProfile({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["remoteDestinationProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addLine(self, pattern="", description=None, usage="", routePartitionName=None, aarNeighborhoodName=None, aarDestinationMask=None, aarKeepCallHistory=None, aarVoiceMailEnabled=None, callForwardAll=None, callForwardBusy=None, callForwardBusyInt=None, callForwardNoAnswer=None, callForwardNoAnswerInt=None, callForwardNoCoverage=None, callForwardNoCoverageInt=None, callForwardOnFailure=None, callForwardAlternateParty=None, callForwardNotRegistered=None, callForwardNotRegisteredInt=None, callPickupGroupName=None, autoAnswer=None, networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, alertingName=None, asciiAlertingName=None, presenceGroupName=None, shareLineAppearanceCssName=None, voiceMailProfileName=None, patternPrecedence=None, releaseClause=None, hrDuration=None, hrInterval=None, cfaCssPolicy=None, defaultActivatedDeviceName=None, parkMonForwardNoRetrieveDn=None, parkMonForwardNoRetrieveIntDn=None, parkMonForwardNoRetrieveVmEnabled=None, parkMonForwardNoRetrieveIntVmEnabled=None, parkMonForwardNoRetrieveCssName=None, parkMonForwardNoRetrieveIntCssName=None, parkMonReversionTimer=None, partyEntranceTone=None, directoryURIs=None, allowCtiControlFlag=None, rejectAnonymousCall=None, patternUrgency=None, confidentialAccess=None, externalCallControlProfile=None, enterpriseAltNum=None, e164AltNum=None, pstnFailover=None, callControlAgentProfile=None, useEnterpriseAltNum=None, useE164AltNum=None, active=None):
        """
        :param line: 
            pattern: String(value)
            description: String(value), optional
            usage: UnionType(value)
            routePartitionName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            aarDestinationMask: String128(value), optional
            aarKeepCallHistory: boolean(value), optional
            aarVoiceMailEnabled: boolean(value), optional
            callForwardAll: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardBusy: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardBusyInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNoAnswer: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNoAnswerInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNoCoverage: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNoCoverageInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardOnFailure: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardAlternateParty: optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNotRegistered: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNotRegisteredInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callPickupGroupName: optional
                _value_1: String(value)
            autoAnswer: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            shareLineAppearanceCssName: optional
                _value_1: String(value)
            voiceMailProfileName: optional
                _value_1: String(value)
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            hrDuration: UnionType(value), optional
            hrInterval: UnionType(value), optional
            cfaCssPolicy: UnionType(value), optional
            defaultActivatedDeviceName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveDn: String50(value), optional
            parkMonForwardNoRetrieveIntDn: String50(value), optional
            parkMonForwardNoRetrieveVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveIntVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveCssName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveIntCssName: optional
                _value_1: String(value)
            parkMonReversionTimer: UnionType(value), optional
            partyEntranceTone: UnionType(value), optional
            directoryURIs: optional
                    isPrimary: boolean(value)
                    uri: String128(value)
                    advertiseGloballyViaIls: boolean(value), optional
            allowCtiControlFlag: boolean(value), optional
            rejectAnonymousCall: boolean(value), optional
            patternUrgency: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            externalCallControlProfile: optional
                _value_1: String(value)
            enterpriseAltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            e164AltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            pstnFailover: String(value), optional
            callControlAgentProfile: String(value), optional
            useEnterpriseAltNum: boolean(value), optional
            useE164AltNum: boolean(value), optional
            active: boolean(value), optional
        """
        try:
            return self.client.addLine({"pattern": pattern, "description": description, "usage": usage, "routePartitionName": routePartitionName, "aarNeighborhoodName": aarNeighborhoodName, "aarDestinationMask": aarDestinationMask, "aarKeepCallHistory": aarKeepCallHistory, "aarVoiceMailEnabled": aarVoiceMailEnabled, "callForwardAll": callForwardAll, "callForwardBusy": callForwardBusy, "callForwardBusyInt": callForwardBusyInt, "callForwardNoAnswer": callForwardNoAnswer, "callForwardNoAnswerInt": callForwardNoAnswerInt, "callForwardNoCoverage": callForwardNoCoverage, "callForwardNoCoverageInt": callForwardNoCoverageInt, "callForwardOnFailure": callForwardOnFailure, "callForwardAlternateParty": callForwardAlternateParty, "callForwardNotRegistered": callForwardNotRegistered, "callForwardNotRegisteredInt": callForwardNotRegisteredInt, "callPickupGroupName": callPickupGroupName, "autoAnswer": autoAnswer, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "alertingName": alertingName, "asciiAlertingName": asciiAlertingName, "presenceGroupName": presenceGroupName, "shareLineAppearanceCssName": shareLineAppearanceCssName, "voiceMailProfileName": voiceMailProfileName, "patternPrecedence": patternPrecedence, "releaseClause": releaseClause, "hrDuration": hrDuration, "hrInterval": hrInterval, "cfaCssPolicy": cfaCssPolicy, "defaultActivatedDeviceName": defaultActivatedDeviceName, "parkMonForwardNoRetrieveDn": parkMonForwardNoRetrieveDn, "parkMonForwardNoRetrieveIntDn": parkMonForwardNoRetrieveIntDn, "parkMonForwardNoRetrieveVmEnabled": parkMonForwardNoRetrieveVmEnabled, "parkMonForwardNoRetrieveIntVmEnabled": parkMonForwardNoRetrieveIntVmEnabled, "parkMonForwardNoRetrieveCssName": parkMonForwardNoRetrieveCssName, "parkMonForwardNoRetrieveIntCssName": parkMonForwardNoRetrieveIntCssName, "parkMonReversionTimer": parkMonReversionTimer, "partyEntranceTone": partyEntranceTone, "directoryURIs": directoryURIs, "allowCtiControlFlag": allowCtiControlFlag, "rejectAnonymousCall": rejectAnonymousCall, "patternUrgency": patternUrgency, "confidentialAccess": confidentialAccess, "externalCallControlProfile": externalCallControlProfile, "enterpriseAltNum": enterpriseAltNum, "e164AltNum": e164AltNum, "pstnFailover": pstnFailover, "callControlAgentProfile": callControlAgentProfile, "useEnterpriseAltNum": useEnterpriseAltNum, "useE164AltNum": useE164AltNum, "active": active})["return"]
        except Fault as e:
            return e

    def updateLine(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param aarDestinationMask: String128(value), optional
        :param aarKeepCallHistory: boolean(value), optional
        :param aarVoiceMailEnabled: boolean(value), optional
        :param callForwardAll: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            secondaryCallingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardBusy: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardBusyInt: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardNoAnswer: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
            duration: UnionType(value), optional
        :param callForwardNoAnswerInt: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
            duration: UnionType(value), optional
        :param callForwardNoCoverage: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardNoCoverageInt: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardOnFailure: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardAlternateParty: optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
            duration: UnionType(value), optional
        :param callForwardNotRegistered: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callForwardNotRegisteredInt: optional
            forwardToVoiceMail: boolean(value), optional
            callingSearchSpaceName: 
                _value_1: String(value)
            destination: String50(value), optional
        :param callPickupGroupName: optional
            _value_1: String(value)
        :param autoAnswer: UnionType(value), optional
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param alertingName: String50(value), optional
        :param asciiAlertingName: String32(value), optional
        :param presenceGroupName: optional
            _value_1: String(value)
        :param shareLineAppearanceCssName: optional
            _value_1: String(value)
        :param voiceMailProfileName: optional
            _value_1: String(value)
        :param patternPrecedence: UnionType(value), optional
        :param releaseClause: UnionType(value), optional
        :param hrDuration: UnionType(value), optional
        :param hrInterval: UnionType(value), optional
        :param cfaCssPolicy: UnionType(value), optional
        :param defaultActivatedDeviceName: optional
            _value_1: String(value)
        :param parkMonForwardNoRetrieveDn: String50(value), optional
        :param parkMonForwardNoRetrieveIntDn: String50(value), optional
        :param parkMonForwardNoRetrieveVmEnabled: boolean(value), optional
        :param parkMonForwardNoRetrieveIntVmEnabled: boolean(value), optional
        :param parkMonForwardNoRetrieveCssName: optional
            _value_1: String(value)
        :param parkMonForwardNoRetrieveIntCssName: optional
            _value_1: String(value)
        :param parkMonReversionTimer: UnionType(value), optional
        :param partyEntranceTone: UnionType(value), optional
        :param directoryURIs: optional
            directoryUri: optional
                isPrimary: boolean(value)
                uri: String128(value)
                    _value_1: String(value)
                advertiseGloballyViaIls: boolean(value), optional
        :param allowCtiControlFlag: boolean(value), optional
        :param rejectAnonymousCall: boolean(value), optional
        :param patternUrgency: boolean(value), optional
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        :param externalCallControlProfile: optional
            _value_1: String(value)
        :param enterpriseAltNum: optional
            numMask: String(value), optional
            isUrgent: boolean(value), optional
            addLocalRoutePartition: boolean(value), optional
            routePartition: optional
                _value_1: String(value)
            advertiseGloballyIls: boolean(value), optional
        :param e164AltNum: optional
            numMask: String(value), optional
            isUrgent: boolean(value), optional
            addLocalRoutePartition: boolean(value), optional
            routePartition: optional
                _value_1: String(value)
            advertiseGloballyIls: boolean(value), optional
        :param pstnFailover: String(value), optional
        :param callControlAgentProfile: String(value), optional
        :param useEnterpriseAltNum: boolean(value), optional
        :param useE164AltNum: boolean(value), optional
        :param active: boolean(value), optional
        """
        try:
            return self.client.updateLine(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getLine(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            aarDestinationMask: String128(value), optional
            aarKeepCallHistory: boolean(value), optional
            aarVoiceMailEnabled: boolean(value), optional
            callForwardAll: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardBusy: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardBusyInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNoAnswer: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNoAnswerInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNoCoverage: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNoCoverageInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardOnFailure: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardAlternateParty: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
                duration: UnionType(value), optional
            callForwardNotRegistered: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callForwardNotRegisteredInt: optional
                forwardToVoiceMail: boolean(value), optional
                    _value_1: String(value)
                destination: String50(value), optional
            callPickupGroupName: optional
                _value_1: String(value)
            autoAnswer: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            shareLineAppearanceCssName: optional
                _value_1: String(value)
            voiceMailProfileName: optional
                _value_1: String(value)
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            hrDuration: UnionType(value), optional
            hrInterval: UnionType(value), optional
            cfaCssPolicy: UnionType(value), optional
            defaultActivatedDeviceName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveDn: String50(value), optional
            parkMonForwardNoRetrieveIntDn: String50(value), optional
            parkMonForwardNoRetrieveVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveIntVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveCssName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveIntCssName: optional
                _value_1: String(value)
            parkMonReversionTimer: UnionType(value), optional
            partyEntranceTone: UnionType(value), optional
            directoryURIs: optional
                    isPrimary: boolean(value), optional
                    uri: String128(value), optional
                    advertiseGloballyViaIls: boolean(value), optional
            allowCtiControlFlag: boolean(value), optional
            rejectAnonymousCall: boolean(value), optional
            patternUrgency: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            externalCallControlProfile: optional
                _value_1: String(value)
            enterpriseAltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            e164AltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            pstnFailover: String(value), optional
            callControlAgentProfile: String(value), optional
            associatedDevices: optional
                device: String(value), optional
            useEnterpriseAltNum: boolean(value), optional
            useE164AltNum: boolean(value), optional
            active: boolean(value), optional
        """
        try:
            return self.client.getLine(**kwargs)["return"]["line"]
        except Fault as e:
            return e

    def removeLine(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeLine(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listLine(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'aarDestinationMask': '', 'aarKeepCallHistory': '', 'aarVoiceMailEnabled': '', 'callPickupGroupName': {'_value_1': ''}, 'autoAnswer': '', 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'alertingName': '', 'asciiAlertingName': '', 'presenceGroupName': {'_value_1': ''}, 'shareLineAppearanceCssName': {'_value_1': ''}, 'voiceMailProfileName': {'_value_1': ''}, 'patternPrecedence': '', 'releaseClause': '', 'hrDuration': '', 'hrInterval': '', 'cfaCssPolicy': '', 'defaultActivatedDeviceName': {'_value_1': ''}, 'parkMonForwardNoRetrieveDn': '', 'parkMonForwardNoRetrieveIntDn': '', 'parkMonForwardNoRetrieveVmEnabled': '', 'parkMonForwardNoRetrieveIntVmEnabled': '', 'parkMonForwardNoRetrieveCssName': {'_value_1': ''}, 'parkMonForwardNoRetrieveIntCssName': {'_value_1': ''}, 'parkMonReversionTimer': '', 'partyEntranceTone': '', 'allowCtiControlFlag': '', 'rejectAnonymousCall': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}, 'externalCallControlProfile': {'_value_1': ''}, 'enterpriseAltNum': {'numMask': '', 'isUrgent': '', 'addLocalRoutePartition': '', 'advertiseGloballyIls': ''}, 'e164AltNum': {'numMask': '', 'isUrgent': '', 'addLocalRoutePartition': '', 'advertiseGloballyIls': ''}, 'pstnFailover': '', 'associatedDevices': {'device': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            usage: String(value), optional
            routePartitionName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            aarDestinationMask: String128(value), optional
            aarKeepCallHistory: boolean(value), optional
            aarVoiceMailEnabled: boolean(value), optional
            callPickupGroupName: optional
                _value_1: String(value)
            autoAnswer: UnionType(value), optional
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            alertingName: String50(value), optional
            asciiAlertingName: String32(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            shareLineAppearanceCssName: optional
                _value_1: String(value)
            voiceMailProfileName: optional
                _value_1: String(value)
            patternPrecedence: UnionType(value), optional
            releaseClause: UnionType(value), optional
            hrDuration: UnionType(value), optional
            hrInterval: UnionType(value), optional
            cfaCssPolicy: UnionType(value), optional
            defaultActivatedDeviceName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveDn: String50(value), optional
            parkMonForwardNoRetrieveIntDn: String50(value), optional
            parkMonForwardNoRetrieveVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveIntVmEnabled: boolean(value), optional
            parkMonForwardNoRetrieveCssName: optional
                _value_1: String(value)
            parkMonForwardNoRetrieveIntCssName: optional
                _value_1: String(value)
            parkMonReversionTimer: UnionType(value), optional
            partyEntranceTone: UnionType(value), optional
            allowCtiControlFlag: boolean(value), optional
            rejectAnonymousCall: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            externalCallControlProfile: optional
                _value_1: String(value)
            enterpriseAltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            e164AltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    _value_1: String(value)
                advertiseGloballyIls: boolean(value), optional
            pstnFailover: String(value), optional
            associatedDevices: optional
                device: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listLine({'pattern': '%', 'description': '%', 'usage': '%', 'routePartitionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["line"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetLine(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.resetLine(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def restartLine(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.restartLine(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def applyLine(self, uuid="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.applyLine(uuid=uuid, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def getLineOptions(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedChoices: optional
            usage: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            routePartitionName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            aarNeighborhoodName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callForwardAll: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardBusy: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardBusyInt: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNoAnswer: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNoAnswerInt: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNoCoverage: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNoCoverageInt: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardOnFailure: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardAlternateParty: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNotRegistered: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callForwardNotRegisteredInt: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            callPickupGroupName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            autoAnswer: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            networkHoldMohAudioSourceId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            userHoldMohAudioSourceId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            presenceGroupName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            shareLineAppearanceCssName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            voiceMailProfileName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            patternPrecedence: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            releaseClause: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            cfaCssPolicy: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            defaultActivatedDeviceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            parkMonForwardNoRetrieveCssName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            parkMonForwardNoRetrieveIntCssName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            partyEntranceTone: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            directoryURIs: optional
            confidentialAccess: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
                confidentialAccessLevel: UnionType(value), optional
            externalCallControlProfile: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            enterpriseAltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
                advertiseGloballyIls: boolean(value), optional
            e164AltNum: optional
                numMask: String(value), optional
                isUrgent: boolean(value), optional
                addLocalRoutePartition: boolean(value), optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
                advertiseGloballyIls: boolean(value), optional
        """
        try:
            return self.client.getLineOptions(**kwargs)["return"]["lineOptions"]
        except Fault as e:
            return e

    def addDefaultDeviceProfile(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", userHoldMohAudioSourceId=None, userLocale=None, phoneButtonTemplate="", softkeyTemplate="", privacy="", singleButtonBarge=None, joinAcrossLines=None, ignorePi=None, dndStatus=None, dndRingSetting=None, dndOption=None, mlppDomainId=None, mlppIndication="", preemption="", alwaysUsePrimeLine=None, alwaysUsePrimeLineForVoiceMessage=None, emccCallingSearchSpace=None):
        """
        :param defaultDeviceProfile: 
            name: UniqueString128(value)
            description: String255(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            userHoldMohAudioSourceId: UnionType(value), optional
            userLocale: UnionType(value), optional
            phoneButtonTemplate: 
                _value_1: String(value)
            softkeyTemplate: 
                _value_1: String(value)
            privacy: UnionType(value)
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            ignorePi: boolean(value), optional
            dndStatus: boolean(value), optional
            dndRingSetting: UnionType(value), optional
            dndOption: UnionType(value), optional
            mlppDomainId: String128(value), optional
            mlppIndication: UnionType(value)
            preemption: UnionType(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            emccCallingSearchSpace: optional
                _value_1: String(value)
        """
        try:
            return self.client.addDefaultDeviceProfile({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "userLocale": userLocale, "phoneButtonTemplate": phoneButtonTemplate, "softkeyTemplate": softkeyTemplate, "privacy": privacy, "singleButtonBarge": singleButtonBarge, "joinAcrossLines": joinAcrossLines, "ignorePi": ignorePi, "dndStatus": dndStatus, "dndRingSetting": dndRingSetting, "dndOption": dndOption, "mlppDomainId": mlppDomainId, "mlppIndication": mlppIndication, "preemption": preemption, "alwaysUsePrimeLine": alwaysUsePrimeLine, "alwaysUsePrimeLineForVoiceMessage": alwaysUsePrimeLineForVoiceMessage, "emccCallingSearchSpace": emccCallingSearchSpace})["return"]
        except Fault as e:
            return e

    def updateDefaultDeviceProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param description: String255(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param userLocale: UnionType(value), optional
        :param phoneButtonTemplate: optional
            _value_1: String(value)
        :param softkeyTemplate: optional
            _value_1: String(value)
        :param privacy: UnionType(value), optional
        :param singleButtonBarge: UnionType(value), optional
        :param joinAcrossLines: UnionType(value), optional
        :param ignorePi: boolean(value), optional
        :param dndStatus: boolean(value), optional
        :param dndRingSetting: UnionType(value), optional
        :param dndOption: UnionType(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndication: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param alwaysUsePrimeLine: UnionType(value), optional
        :param alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param emccCallingSearchSpace: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateDefaultDeviceProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getDefaultDeviceProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String255(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            userLocale: UnionType(value), optional
            phoneButtonTemplate: optional
                _value_1: String(value)
            softkeyTemplate: optional
                _value_1: String(value)
            privacy: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            ignorePi: boolean(value), optional
            dndStatus: boolean(value), optional
            dndRingSetting: UnionType(value), optional
            dndOption: UnionType(value), optional
            mlppDomainId: String128(value), optional
            mlppIndication: UnionType(value), optional
            preemption: UnionType(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            emccCallingSearchSpace: optional
                _value_1: String(value)
        """
        try:
            return self.client.getDefaultDeviceProfile(**kwargs)["return"]["defaultDeviceProfile"]
        except Fault as e:
            return e

    def removeDefaultDeviceProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDefaultDeviceProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDefaultDeviceProfile(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'userHoldMohAudioSourceId': '', 'userLocale': '', 'phoneButtonTemplate': {'_value_1': ''}, 'softkeyTemplate': {'_value_1': ''}, 'privacy': '', 'singleButtonBarge': '', 'joinAcrossLines': '', 'ignorePi': '', 'dndStatus': '', 'dndRingSetting': '', 'dndOption': '', 'mlppIndication': '', 'preemption': '', 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String255(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            userLocale: UnionType(value), optional
            phoneButtonTemplate: optional
                _value_1: String(value)
            softkeyTemplate: optional
                _value_1: String(value)
            privacy: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            ignorePi: boolean(value), optional
            dndStatus: boolean(value), optional
            dndRingSetting: UnionType(value), optional
            dndOption: UnionType(value), optional
            mlppIndication: UnionType(value), optional
            preemption: UnionType(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDefaultDeviceProfile({'name': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["defaultDeviceProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addH323Phone(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, commonPhoneConfigName="", locationName="", mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, traceFlag=None, mlppDomainId=None, useTrustedRelayPoint="", retryVideoCallAsAudio=None, remoteDevice=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, alwaysUsePrimeLine=None, alwaysUsePrimeLineForVoiceMessage=None, srtpAllowed=None, unattendedPort=None, subscribeCallingSearchSpaceName=None, waitForFarEndH245TerminalSet=None, mtpRequired=None, mtpPreferredCodec=None, callerIdDn=None, callingPartySelection=None, callingLineIdPresentation=None, displayIEDelivery=None, redirectOutboundNumberIe=None, redirectInboundNumberIe=None, presenceGroupName=None, hlogStatus=None, ownerUserName=None, signalingPort="", gateKeeperInfo=None, lines=None, ignorePresentationIndicators=None, elinGroup=None):
        """
        :param h323Phone: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: 
                _value_1: String(value)
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value)
            retryVideoCallAsAudio: boolean(value), optional
            remoteDevice: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            mtpPreferredCodec: UnionType(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            hlogStatus: boolean(value), optional
            ownerUserName: optional
                _value_1: String(value)
            signalingPort: UnionType(value)
            gateKeeperInfo: optional
                    _value_1: String(value)
                e164: String50(value)
                technologyPrefix: String50(value)
                zone: String50(value)
            lines: optional
                    index: UnionType(value)
                    label: String50(value), optional
                    asciiLabel: String50(value), optional
                    display: String50(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    directoryNumber: String255(value)
                    routePartitionName: String50(value)
            ignorePresentationIndicators: boolean(value), optional
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.addH323Phone({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "commonPhoneConfigName": commonPhoneConfigName, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "remoteDevice": remoteDevice, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "alwaysUsePrimeLine": alwaysUsePrimeLine, "alwaysUsePrimeLineForVoiceMessage": alwaysUsePrimeLineForVoiceMessage, "srtpAllowed": srtpAllowed, "unattendedPort": unattendedPort, "subscribeCallingSearchSpaceName": subscribeCallingSearchSpaceName, "waitForFarEndH245TerminalSet": waitForFarEndH245TerminalSet, "mtpRequired": mtpRequired, "mtpPreferredCodec": mtpPreferredCodec, "callerIdDn": callerIdDn, "callingPartySelection": callingPartySelection, "callingLineIdPresentation": callingLineIdPresentation, "displayIEDelivery": displayIEDelivery, "redirectOutboundNumberIe": redirectOutboundNumberIe, "redirectInboundNumberIe": redirectInboundNumberIe, "presenceGroupName": presenceGroupName, "hlogStatus": hlogStatus, "ownerUserName": ownerUserName, "signalingPort": signalingPort, "gateKeeperInfo": gateKeeperInfo, "lines": lines, "ignorePresentationIndicators": ignorePresentationIndicators, "elinGroup": elinGroup})["return"]
        except Fault as e:
            return e

    def updateH323Phone(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param commonPhoneConfigName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param remoteDevice: boolean(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param alwaysUsePrimeLine: UnionType(value), optional
        :param alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param srtpAllowed: boolean(value), optional
        :param unattendedPort: boolean(value), optional
        :param subscribeCallingSearchSpaceName: optional
            _value_1: String(value)
        :param waitForFarEndH245TerminalSet: boolean(value), optional
        :param mtpRequired: boolean(value), optional
        :param mtpPreferredCodec: UnionType(value), optional
        :param callerIdDn: String50(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingLineIdPresentation: UnionType(value), optional
        :param displayIEDelivery: boolean(value), optional
        :param redirectOutboundNumberIe: boolean(value), optional
        :param redirectInboundNumberIe: boolean(value), optional
        :param presenceGroupName: optional
            _value_1: String(value)
        :param hlogStatus: boolean(value), optional
        :param ownerUserName: optional
            _value_1: String(value)
        :param signalingPort: UnionType(value), optional
        :param gateKeeperInfo: optional
            gateKeeperName: 
                _value_1: String(value)
            e164: String50(value)
            technologyPrefix: String50(value)
            zone: String50(value)
        :param lines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                asciiLabel: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param ignorePresentationIndicators: boolean(value), optional
        :param elinGroup: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateH323Phone(**kwargs)["return"]
        except Fault as e:
            return e

    def getH323Phone(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            remoteDevice: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            mtpPreferredCodec: UnionType(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            hlogStatus: boolean(value), optional
            ownerUserName: optional
                _value_1: String(value)
            signalingPort: UnionType(value), optional
            gateKeeperInfo: optional
                    _value_1: String(value)
                e164: String50(value), optional
                technologyPrefix: String50(value), optional
                zone: String50(value), optional
            lines: optional
                    index: UnionType(value), optional
                    label: String50(value), optional
                    asciiLabel: String50(value), optional
                    display: String50(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    dialPlanWizardId: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    directoryNumber: String255(value), optional
                    routePartitionName: String50(value), optional
            ignorePresentationIndicators: boolean(value), optional
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.getH323Phone(**kwargs)["return"]["h323Phone"]
        except Fault as e:
            return e

    def removeH323Phone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeH323Phone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listH323Phone(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'commonPhoneConfigName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'traceFlag': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'remoteDevice': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'srtpAllowed': '', 'unattendedPort': '', 'subscribeCallingSearchSpaceName': {'_value_1': ''}, 'mtpRequired': '', 'mtpPreferredCodec': '', 'callerIdDn': '', 'callingPartySelection': '', 'callingLineIdPresentation': '', 'displayIEDelivery': '', 'redirectOutboundNumberIe': '', 'redirectInboundNumberIe': '', 'presenceGroupName': {'_value_1': ''}, 'hlogStatus': '', 'ownerUserName': {'_value_1': ''}, 'signalingPort': '', 'ignorePresentationIndicators': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            remoteDevice: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            mtpRequired: boolean(value), optional
            mtpPreferredCodec: UnionType(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            hlogStatus: boolean(value), optional
            ownerUserName: optional
                _value_1: String(value)
            signalingPort: UnionType(value), optional
            ignorePresentationIndicators: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listH323Phone({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["h323Phone"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetH323Phone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetH323Phone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartH323Phone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartH323Phone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyH323Phone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyH323Phone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def updateMohServer(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String(value), optional
        :param description: String(value), optional
        :param devicePoolName: optional
            _value_1: String(value)
        :param locationName: optional
            _value_1: String(value)
        :param maxUnicastConnections: UnionType(value), optional
        :param maxMulticastConnections: UnionType(value), optional
        :param fixedAudioSourceDevice: String(value), optional
        :param runFlag: boolean(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param isMultiCastEnabled: boolean(value), optional
        :param baseMulticastIpaddress: String(value), optional
        :param baseMulticastPort: UnionType(value), optional
        :param multicastIncrementOnIp: boolean(value), optional
        :param audioSources: optional
            audioSource: optional
                sourceId: UnionType(value)
                maxHops: UnionType(value), optional
        """
        try:
            return self.client.updateMohServer(**kwargs)["return"]
        except Fault as e:
            return e

    def getMohServer(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            description: String(value), optional
            processNodeName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            maxUnicastConnections: UnionType(value), optional
            maxMulticastConnections: UnionType(value), optional
            fixedAudioSourceDevice: String(value), optional
            runFlag: boolean(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            isMultiCastEnabled: boolean(value), optional
            baseMulticastIpaddress: String(value), optional
            baseMulticastPort: UnionType(value), optional
            multicastIncrementOnIp: boolean(value), optional
            audioSources: optional
                    sourceId: UnionType(value), optional
                    maxHops: UnionType(value), optional
        """
        try:
            return self.client.getMohServer(**kwargs)["return"]["mohServer"]
        except Fault as e:
            return e

    def listMohServer(self, returnedTags={'name': '', 'description': '', 'processNodeName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'locationName': {'_value_1': ''}, 'maxUnicastConnections': '', 'maxMulticastConnections': '', 'fixedAudioSourceDevice': '', 'runFlag': '', 'useTrustedRelayPoint': '', 'isMultiCastEnabled': '', 'baseMulticastIpaddress': '', 'baseMulticastPort': '', 'multicastIncrementOnIp': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            description: String(value), optional
            processNodeName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            locationName: optional
                _value_1: String(value)
            maxUnicastConnections: UnionType(value), optional
            maxMulticastConnections: UnionType(value), optional
            fixedAudioSourceDevice: String(value), optional
            runFlag: boolean(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            isMultiCastEnabled: boolean(value), optional
            baseMulticastIpaddress: String(value), optional
            baseMulticastPort: UnionType(value), optional
            multicastIncrementOnIp: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listMohServer({'name': '%', 'description': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["mohServer"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addH323Trunk(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", mediaResourceListName=None, aarNeighborhoodName=None, traceFlag=None, mlppDomainId=None, mlppIndicationStatus=None, preemption=None, useTrustedRelayPoint=None, retryVideoCallAsAudio=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, geoLocationFilterName=None, sendGeoLocation=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, packetCaptureMode=None, packetCaptureDuration=None, srtpAllowed=None, unattendedPort=None, waitForFarEndH245TerminalSet=None, mtpRequired=None, callerIdDn=None, callingPartySelection=None, callingLineIdPresentation=None, displayIEDelivery=None, redirectOutboundNumberIe=None, redirectInboundNumberIe=None, enableInboundFaststart=None, enableOutboundFaststart=None, codecForOutboundFaststart=None, allowH235PassThrough=None, tunneledProtocol=None, asn1RoseOidEncoding=None, qsigVariant=None, transmitUtf8=None, signalingPort=None, nationalPrefix=None, internationalPrefix=None, unknownPrefix=None, subscriberPrefix=None, sigDigits=None, prefixDn=None, calledPartyIeNumberType=None, callingPartyIeNumberType=None, calledNumberingPlan=None, callingNumberingPlan=None, pathReplacementSupport=None, gateKeeperInfo=None, ictPassingPrecedenceLevelThroughUuie=None, ictSecurityAccessLevel=None, isSafEnabled=None, callingPartyNationalStripDigits=None, callingPartyInternationalStripDigits=None, callingPartyUnknownStripDigits=None, callingPartySubscriberStripDigits=None, callingPartyNationalTransformationCssName=None, callingPartyInternationalTransformationCssName=None, callingPartyUnknownTransformationCssName=None, callingPartySubscriberTransformationCssName=None, calledPartyNationalPrefix=None, calledPartyInternationalPrefix=None, calledPartyUnknownPrefix=None, calledPartySubscriberPrefix=None, pstnAccess=None, imeE164TransformationName=None, automatedAlternateRoutingCssName=None, useDevicePoolCgpnTransformCssNatl=None, useDevicePoolCgpnTransformCssIntl=None, useDevicePoolCgpnTransformCssUnkn=None, useDevicePoolCgpnTransformCssSubs=None, useDevicePoolCalledCssNatl=None, useDevicePoolCalledCssIntl=None, useDevicePoolCalledCssUnkn=None, useDevicePoolCalledCssSubs=None, calledPartyNationalStripDigits=None, calledPartyInternationalStripDigits=None, calledPartyUnknownStripDigits=None, calledPartySubscriberStripDigits=None, calledPartyNationalTransformationCssName=None, calledPartyInternationalTransformationCssName=None, calledPartyUnknownTransformationCssName=None, calledPartySubscriberTransformationCssName=None, runOnEveryNode=None, destinations=None, useDevicePoolCntdPnTransformationCss=None, cntdPnTransformationCssName=None, confidentialAccess=None, connectCallBeforePlayingAnnouncement=None):
        """
        :param h323Trunk: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            allowH235PassThrough: boolean(value), optional
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value), optional
            callingPartyIeNumberType: UnionType(value), optional
            calledNumberingPlan: UnionType(value), optional
            callingNumberingPlan: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            gateKeeperInfo: optional
                    _value_1: String(value)
                terminalType: UnionType(value)
                technologyPrefix: String50(value), optional
                zone: String50(value), optional
            ictPassingPrecedenceLevelThroughUuie: boolean(value), optional
            ictSecurityAccessLevel: UnionType(value), optional
            isSafEnabled: boolean(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            runOnEveryNode: boolean(value), optional
            destinations: optional
                    addressIpv4: String255(value), optional
                    sortOrder: UnionType(value)
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addH323Trunk({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "aarNeighborhoodName": aarNeighborhoodName, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "preemption": preemption, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "geoLocationFilterName": geoLocationFilterName, "sendGeoLocation": sendGeoLocation, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "srtpAllowed": srtpAllowed, "unattendedPort": unattendedPort, "waitForFarEndH245TerminalSet": waitForFarEndH245TerminalSet, "mtpRequired": mtpRequired, "callerIdDn": callerIdDn, "callingPartySelection": callingPartySelection, "callingLineIdPresentation": callingLineIdPresentation, "displayIEDelivery": displayIEDelivery, "redirectOutboundNumberIe": redirectOutboundNumberIe, "redirectInboundNumberIe": redirectInboundNumberIe, "enableInboundFaststart": enableInboundFaststart, "enableOutboundFaststart": enableOutboundFaststart, "codecForOutboundFaststart": codecForOutboundFaststart, "allowH235PassThrough": allowH235PassThrough, "tunneledProtocol": tunneledProtocol, "asn1RoseOidEncoding": asn1RoseOidEncoding, "qsigVariant": qsigVariant, "transmitUtf8": transmitUtf8, "signalingPort": signalingPort, "nationalPrefix": nationalPrefix, "internationalPrefix": internationalPrefix, "unknownPrefix": unknownPrefix, "subscriberPrefix": subscriberPrefix, "sigDigits": sigDigits, "prefixDn": prefixDn, "calledPartyIeNumberType": calledPartyIeNumberType, "callingPartyIeNumberType": callingPartyIeNumberType, "calledNumberingPlan": calledNumberingPlan, "callingNumberingPlan": callingNumberingPlan, "pathReplacementSupport": pathReplacementSupport, "gateKeeperInfo": gateKeeperInfo, "ictPassingPrecedenceLevelThroughUuie": ictPassingPrecedenceLevelThroughUuie, "ictSecurityAccessLevel": ictSecurityAccessLevel, "isSafEnabled": isSafEnabled, "callingPartyNationalStripDigits": callingPartyNationalStripDigits, "callingPartyInternationalStripDigits": callingPartyInternationalStripDigits, "callingPartyUnknownStripDigits": callingPartyUnknownStripDigits, "callingPartySubscriberStripDigits": callingPartySubscriberStripDigits, "callingPartyNationalTransformationCssName": callingPartyNationalTransformationCssName, "callingPartyInternationalTransformationCssName": callingPartyInternationalTransformationCssName, "callingPartyUnknownTransformationCssName": callingPartyUnknownTransformationCssName, "callingPartySubscriberTransformationCssName": callingPartySubscriberTransformationCssName, "calledPartyNationalPrefix": calledPartyNationalPrefix, "calledPartyInternationalPrefix": calledPartyInternationalPrefix, "calledPartyUnknownPrefix": calledPartyUnknownPrefix, "calledPartySubscriberPrefix": calledPartySubscriberPrefix, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "useDevicePoolCgpnTransformCssNatl": useDevicePoolCgpnTransformCssNatl, "useDevicePoolCgpnTransformCssIntl": useDevicePoolCgpnTransformCssIntl, "useDevicePoolCgpnTransformCssUnkn": useDevicePoolCgpnTransformCssUnkn, "useDevicePoolCgpnTransformCssSubs": useDevicePoolCgpnTransformCssSubs, "useDevicePoolCalledCssNatl": useDevicePoolCalledCssNatl, "useDevicePoolCalledCssIntl": useDevicePoolCalledCssIntl, "useDevicePoolCalledCssUnkn": useDevicePoolCalledCssUnkn, "useDevicePoolCalledCssSubs": useDevicePoolCalledCssSubs, "calledPartyNationalStripDigits": calledPartyNationalStripDigits, "calledPartyInternationalStripDigits": calledPartyInternationalStripDigits, "calledPartyUnknownStripDigits": calledPartyUnknownStripDigits, "calledPartySubscriberStripDigits": calledPartySubscriberStripDigits, "calledPartyNationalTransformationCssName": calledPartyNationalTransformationCssName, "calledPartyInternationalTransformationCssName": calledPartyInternationalTransformationCssName, "calledPartyUnknownTransformationCssName": calledPartyUnknownTransformationCssName, "calledPartySubscriberTransformationCssName": calledPartySubscriberTransformationCssName, "runOnEveryNode": runOnEveryNode, "destinations": destinations, "useDevicePoolCntdPnTransformationCss": useDevicePoolCntdPnTransformationCss, "cntdPnTransformationCssName": cntdPnTransformationCssName, "confidentialAccess": confidentialAccess, "connectCallBeforePlayingAnnouncement": connectCallBeforePlayingAnnouncement})["return"]
        except Fault as e:
            return e

    def updateH323Trunk(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param sendGeoLocation: boolean(value), optional
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param srtpAllowed: boolean(value), optional
        :param unattendedPort: boolean(value), optional
        :param waitForFarEndH245TerminalSet: boolean(value), optional
        :param mtpRequired: boolean(value), optional
        :param callerIdDn: String50(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingLineIdPresentation: UnionType(value), optional
        :param displayIEDelivery: boolean(value), optional
        :param redirectOutboundNumberIe: boolean(value), optional
        :param redirectInboundNumberIe: boolean(value), optional
        :param enableInboundFaststart: boolean(value), optional
        :param enableOutboundFaststart: boolean(value), optional
        :param codecForOutboundFaststart: UnionType(value), optional
        :param allowH235PassThrough: boolean(value), optional
        :param tunneledProtocol: UnionType(value), optional
        :param asn1RoseOidEncoding: UnionType(value), optional
        :param qsigVariant: UnionType(value), optional
        :param transmitUtf8: boolean(value), optional
        :param signalingPort: UnionType(value), optional
        :param nationalPrefix: String16(value), optional
        :param internationalPrefix: String16(value), optional
        :param unknownPrefix: String16(value), optional
        :param subscriberPrefix: String16(value), optional
        :param sigDigits: optional
            _value_1: UnionType(value)
        :param prefixDn: String50(value), optional
        :param calledPartyIeNumberType: UnionType(value), optional
        :param callingPartyIeNumberType: UnionType(value), optional
        :param calledNumberingPlan: UnionType(value), optional
        :param callingNumberingPlan: UnionType(value), optional
        :param pathReplacementSupport: boolean(value), optional
        :param gateKeeperInfo: optional
            gateKeeper: 
                _value_1: String(value)
            terminalType: UnionType(value)
            technologyPrefix: String50(value), optional
            zone: String50(value), optional
        :param ictPassingPrecedenceLevelThroughUuie: boolean(value), optional
        :param ictSecurityAccessLevel: UnionType(value), optional
        :param isSafEnabled: boolean(value), optional
        :param callingPartyNationalStripDigits: UnionType(value), optional
        :param callingPartyInternationalStripDigits: UnionType(value), optional
        :param callingPartyUnknownStripDigits: UnionType(value), optional
        :param callingPartySubscriberStripDigits: UnionType(value), optional
        :param callingPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param callingPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyNationalPrefix: String16(value), optional
        :param calledPartyInternationalPrefix: String16(value), optional
        :param calledPartyUnknownPrefix: String16(value), optional
        :param calledPartySubscriberPrefix: String16(value), optional
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCssNatl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssIntl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
        :param useDevicePoolCgpnTransformCssSubs: boolean(value), optional
        :param useDevicePoolCalledCssNatl: boolean(value), optional
        :param useDevicePoolCalledCssIntl: boolean(value), optional
        :param useDevicePoolCalledCssUnkn: boolean(value), optional
        :param useDevicePoolCalledCssSubs: boolean(value), optional
        :param calledPartyNationalStripDigits: UnionType(value), optional
        :param calledPartyInternationalStripDigits: UnionType(value), optional
        :param calledPartyUnknownStripDigits: UnionType(value), optional
        :param calledPartySubscriberStripDigits: UnionType(value), optional
        :param calledPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param calledPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param runOnEveryNode: boolean(value), optional
        :param removeDestinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                sortOrder: UnionType(value)
        :param addDestinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                sortOrder: UnionType(value)
        :param destinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                sortOrder: UnionType(value)
        :param useDevicePoolCntdPnTransformationCss: boolean(value), optional
        :param cntdPnTransformationCssName: optional
            _value_1: String(value)
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        :param connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateH323Trunk(**kwargs)["return"]
        except Fault as e:
            return e

    def getH323Trunk(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            allowH235PassThrough: boolean(value), optional
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value), optional
            callingPartyIeNumberType: UnionType(value), optional
            calledNumberingPlan: UnionType(value), optional
            callingNumberingPlan: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            gateKeeperInfo: optional
                    _value_1: String(value)
                terminalType: UnionType(value), optional
                technologyPrefix: String50(value), optional
                zone: String50(value), optional
            ictPassingPrecedenceLevelThroughUuie: boolean(value), optional
            ictSecurityAccessLevel: UnionType(value), optional
            isSafEnabled: boolean(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            runOnEveryNode: boolean(value), optional
            destinations: optional
                    addressIpv4: String255(value), optional
                    sortOrder: UnionType(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getH323Trunk(**kwargs)["return"]["h323Trunk"]
        except Fault as e:
            return e

    def removeH323Trunk(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeH323Trunk(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listH323Trunk(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'traceFlag': '', 'mlppIndicationStatus': '', 'preemption': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'geoLocationFilterName': {'_value_1': ''}, 'sendGeoLocation': '', 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'packetCaptureMode': '', 'packetCaptureDuration': '', 'srtpAllowed': '', 'unattendedPort': '', 'mtpRequired': '', 'callerIdDn': '', 'callingPartySelection': '', 'callingLineIdPresentation': '', 'displayIEDelivery': '', 'redirectOutboundNumberIe': '', 'redirectInboundNumberIe': '', 'enableInboundFaststart': '', 'enableOutboundFaststart': '', 'codecForOutboundFaststart': '', 'allowH235PassThrough': '', 'tunneledProtocol': '', 'asn1RoseOidEncoding': '', 'qsigVariant': '', 'transmitUtf8': '', 'signalingPort': '', 'nationalPrefix': '', 'internationalPrefix': '', 'unknownPrefix': '', 'subscriberPrefix': '', 'sigDigits': {'_value_1': ''}, 'prefixDn': '', 'calledPartyIeNumberType': '', 'callingPartyIeNumberType': '', 'calledNumberingPlan': '', 'callingNumberingPlan': '', 'pathReplacementSupport': '', 'gateKeeperInfo': {'terminalType': '', 'technologyPrefix': '', 'zone': ''}, 'ictPassingPrecedenceLevelThroughUuie': '', 'ictSecurityAccessLevel': '', 'isSafEnabled': '', 'callingPartyNationalStripDigits': '', 'callingPartyInternationalStripDigits': '', 'callingPartyUnknownStripDigits': '', 'callingPartySubscriberStripDigits': '', 'callingPartyNationalTransformationCssName': {'_value_1': ''}, 'callingPartyInternationalTransformationCssName': {'_value_1': ''}, 'callingPartyUnknownTransformationCssName': {'_value_1': ''}, 'callingPartySubscriberTransformationCssName': {'_value_1': ''}, 'calledPartyNationalPrefix': '', 'calledPartyInternationalPrefix': '', 'calledPartyUnknownPrefix': '', 'calledPartySubscriberPrefix': '', 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCssNatl': '', 'useDevicePoolCgpnTransformCssIntl': '', 'useDevicePoolCgpnTransformCssUnkn': '', 'useDevicePoolCgpnTransformCssSubs': '', 'useDevicePoolCalledCssNatl': '', 'useDevicePoolCalledCssIntl': '', 'useDevicePoolCalledCssUnkn': '', 'useDevicePoolCalledCssSubs': '', 'calledPartyNationalStripDigits': '', 'calledPartyInternationalStripDigits': '', 'calledPartyUnknownStripDigits': '', 'calledPartySubscriberStripDigits': '', 'calledPartyNationalTransformationCssName': {'_value_1': ''}, 'calledPartyInternationalTransformationCssName': {'_value_1': ''}, 'calledPartyUnknownTransformationCssName': {'_value_1': ''}, 'calledPartySubscriberTransformationCssName': {'_value_1': ''}, 'runOnEveryNode': '', 'useDevicePoolCntdPnTransformationCss': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            unattendedPort: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            displayIEDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            allowH235PassThrough: boolean(value), optional
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value), optional
            callingPartyIeNumberType: UnionType(value), optional
            calledNumberingPlan: UnionType(value), optional
            callingNumberingPlan: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            gateKeeperInfo: optional
                    _value_1: String(value)
                terminalType: UnionType(value), optional
                technologyPrefix: String50(value), optional
                zone: String50(value), optional
            ictPassingPrecedenceLevelThroughUuie: boolean(value), optional
            ictSecurityAccessLevel: UnionType(value), optional
            isSafEnabled: boolean(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            runOnEveryNode: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listH323Trunk({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["h323Trunk"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetH323Trunk(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetH323Trunk(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartH323Trunk(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartH323Trunk(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addPhone(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, commonPhoneConfigName="", networkLocation=None, locationName="", mediaResourceListName=None, networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, vendorConfig=None, versionStamp=None, traceFlag=None, mlppDomainId=None, mlppIndicationStatus=None, preemption=None, useTrustedRelayPoint="", retryVideoCallAsAudio=None, securityProfileName=None, sipProfileName=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, geoLocationFilterName=None, sendGeoLocation=None, lines=None, phoneTemplateName="", speeddials=None, busyLampFields=None, primaryPhoneName="", ringSettingIdleBlfAudibleAlert=None, ringSettingBusyBlfAudibleAlert=None, blfDirectedCallParks=None, addOnModules=None, userLocale=None, networkLocale=None, idleTimeout=None, authenticationUrl=None, directoryUrl=None, idleUrl=None, informationUrl=None, messagesUrl=None, proxyServerUrl=None, servicesUrl=None, services=None, softkeyTemplateName=None, defaultProfileName=None, enableExtensionMobility=None, singleButtonBarge=None, joinAcrossLines=None, builtInBridgeStatus="", callInfoPrivacyStatus=None, hlogStatus=None, ownerUserName=None, ignorePresentationIndicators=None, packetCaptureMode="", packetCaptureDuration=None, subscribeCallingSearchSpaceName=None, rerouteCallingSearchSpaceName=None, allowCtiControlFlag=None, presenceGroupName=None, unattendedPort=None, requireDtmfReception=None, rfc2833Disabled=None, certificateOperation="", authenticationMode=None, keySize=None, keyOrder=None, ecKeySize=None, authenticationString=None, upgradeFinishTime=None, deviceMobilityMode="", remoteDevice=None, dndOption=None, dndRingSetting=None, dndStatus=None, isActive=None, isDualMode=None, mobilityUserIdName=None, phoneSuite=None, phoneServiceDisplay=None, isProtected=None, mtpRequired=None, mtpPreferedCodec=None, dialRulesName=None, sshUserId=None, sshPwd=None, digestUser=None, outboundCallRollover=None, hotlineDevice=None, secureInformationUrl=None, secureDirectoryUrl=None, secureMessageUrl=None, secureServicesUrl=None, secureAuthenticationUrl=None, secureIdleUrl=None, alwaysUsePrimeLine=None, alwaysUsePrimeLineForVoiceMessage=None, featureControlPolicy=None, deviceTrustMode=None, earlyOfferSupportForVoiceCall=None, requireThirdPartyRegistration=None, blockIncomingCallsWhenRoaming=None, homeNetworkId=None, AllowPresentationSharingUsingBfcp=None, confidentialAccess=None, requireOffPremiseLocation=None, allowiXApplicableMedia=None, cgpnIngressDN=None, useDevicePoolCgpnIngressDN=None, msisdn=None, enableCallRoutingToRdWhenNoneIsActive=None, wifiHotspotProfile=None, wirelessLanProfileGroup=None, elinGroup=None):
        """
        :param phone: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: 
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value)
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: optional
                _value_1: String(value)
            sipProfileName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            lines: optional
                    index: UnionType(value)
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value)
                    routePartitionName: String50(value)
            phoneTemplateName: 
                _value_1: String(value)
            speeddials: optional
                    dirn: String255(value)
                    label: String(value)
                    index: UnionType(value)
            busyLampFields: optional
                    blfDest: String(value)
                    blfDirn: String255(value), optional
                    routePartition: String(value), optional
                    label: String(value), optional
                    index: UnionType(value)
            primaryPhoneName: 
                _value_1: String(value)
            ringSettingIdleBlfAudibleAlert: UnionType(value), optional
            ringSettingBusyBlfAudibleAlert: UnionType(value), optional
            blfDirectedCallParks: optional
                    label: String50(value), optional
                    directedCallParkId: XUUID(value), optional
                    index: UnionType(value)
            addOnModules: optional
                    model: UnionType(value)
                    index: UnionType(value)
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            idleTimeout: UnionType(value), optional
            authenticationUrl: String(value), optional
            directoryUrl: String(value), optional
            idleUrl: String(value), optional
            informationUrl: String(value), optional
            messagesUrl: String(value), optional
            proxyServerUrl: String(value), optional
            servicesUrl: String(value), optional
            services: optional
                    name: String100(value)
                    url: String(value), optional
                    urlButtonIndex: UnionType(value), optional
                    urlLabel: String50(value), optional
                    serviceNameAscii: String50(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            defaultProfileName: optional
                _value_1: String(value)
            enableExtensionMobility: boolean(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            builtInBridgeStatus: UnionType(value)
            callInfoPrivacyStatus: UnionType(value), optional
            hlogStatus: UnionType(value), optional
            ownerUserName: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            packetCaptureMode: UnionType(value)
            packetCaptureDuration: UnionType(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            allowCtiControlFlag: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            unattendedPort: boolean(value), optional
            requireDtmfReception: boolean(value), optional
            rfc2833Disabled: boolean(value), optional
            certificateOperation: UnionType(value)
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            authenticationString: String128(value), optional
            upgradeFinishTime: String(value), optional
            deviceMobilityMode: UnionType(value)
            remoteDevice: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            isActive: boolean(value), optional
            isDualMode: boolean(value), optional
            mobilityUserIdName: optional
                _value_1: String(value)
            phoneSuite: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            isProtected: boolean(value), optional
            mtpRequired: boolean(value), optional
            mtpPreferedCodec: UnionType(value), optional
            dialRulesName: optional
                _value_1: String(value)
            sshUserId: String50(value), optional
            sshPwd: String255(value), optional
            digestUser: String255(value), optional
            outboundCallRollover: UnionType(value), optional
            hotlineDevice: boolean(value), optional
            secureInformationUrl: String255(value), optional
            secureDirectoryUrl: String255(value), optional
            secureMessageUrl: String255(value), optional
            secureServicesUrl: String255(value), optional
            secureAuthenticationUrl: String255(value), optional
            secureIdleUrl: String255(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
            deviceTrustMode: UnionType(value), optional
            earlyOfferSupportForVoiceCall: boolean(value), optional
            requireThirdPartyRegistration: boolean(value), optional
            blockIncomingCallsWhenRoaming: boolean(value), optional
            homeNetworkId: String(value), optional
            AllowPresentationSharingUsingBfcp: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            requireOffPremiseLocation: boolean(value), optional
            allowiXApplicableMedia: boolean(value), optional
            cgpnIngressDN: optional
                _value_1: String(value)
            useDevicePoolCgpnIngressDN: boolean(value), optional
            msisdn: String128(value), optional
            enableCallRoutingToRdWhenNoneIsActive: boolean(value), optional
            wifiHotspotProfile: optional
                _value_1: String(value)
            wirelessLanProfileGroup: optional
                _value_1: String(value)
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.addPhone({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "commonPhoneConfigName": commonPhoneConfigName, "networkLocation": networkLocation, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "versionStamp": versionStamp, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "preemption": preemption, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "securityProfileName": securityProfileName, "sipProfileName": sipProfileName, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "geoLocationFilterName": geoLocationFilterName, "sendGeoLocation": sendGeoLocation, "lines": lines, "phoneTemplateName": phoneTemplateName, "speeddials": speeddials, "busyLampFields": busyLampFields, "primaryPhoneName": primaryPhoneName, "ringSettingIdleBlfAudibleAlert": ringSettingIdleBlfAudibleAlert, "ringSettingBusyBlfAudibleAlert": ringSettingBusyBlfAudibleAlert, "blfDirectedCallParks": blfDirectedCallParks, "addOnModules": addOnModules, "userLocale": userLocale, "networkLocale": networkLocale, "idleTimeout": idleTimeout, "authenticationUrl": authenticationUrl, "directoryUrl": directoryUrl, "idleUrl": idleUrl, "informationUrl": informationUrl, "messagesUrl": messagesUrl, "proxyServerUrl": proxyServerUrl, "servicesUrl": servicesUrl, "services": services, "softkeyTemplateName": softkeyTemplateName, "defaultProfileName": defaultProfileName, "enableExtensionMobility": enableExtensionMobility, "singleButtonBarge": singleButtonBarge, "joinAcrossLines": joinAcrossLines, "builtInBridgeStatus": builtInBridgeStatus, "callInfoPrivacyStatus": callInfoPrivacyStatus, "hlogStatus": hlogStatus, "ownerUserName": ownerUserName, "ignorePresentationIndicators": ignorePresentationIndicators, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "subscribeCallingSearchSpaceName": subscribeCallingSearchSpaceName, "rerouteCallingSearchSpaceName": rerouteCallingSearchSpaceName, "allowCtiControlFlag": allowCtiControlFlag, "presenceGroupName": presenceGroupName, "unattendedPort": unattendedPort, "requireDtmfReception": requireDtmfReception, "rfc2833Disabled": rfc2833Disabled, "certificateOperation": certificateOperation, "authenticationMode": authenticationMode, "keySize": keySize, "keyOrder": keyOrder, "ecKeySize": ecKeySize, "authenticationString": authenticationString, "upgradeFinishTime": upgradeFinishTime, "deviceMobilityMode": deviceMobilityMode, "remoteDevice": remoteDevice, "dndOption": dndOption, "dndRingSetting": dndRingSetting, "dndStatus": dndStatus, "isActive": isActive, "isDualMode": isDualMode, "mobilityUserIdName": mobilityUserIdName, "phoneSuite": phoneSuite, "phoneServiceDisplay": phoneServiceDisplay, "isProtected": isProtected, "mtpRequired": mtpRequired, "mtpPreferedCodec": mtpPreferedCodec, "dialRulesName": dialRulesName, "sshUserId": sshUserId, "sshPwd": sshPwd, "digestUser": digestUser, "outboundCallRollover": outboundCallRollover, "hotlineDevice": hotlineDevice, "secureInformationUrl": secureInformationUrl, "secureDirectoryUrl": secureDirectoryUrl, "secureMessageUrl": secureMessageUrl, "secureServicesUrl": secureServicesUrl, "secureAuthenticationUrl": secureAuthenticationUrl, "secureIdleUrl": secureIdleUrl, "alwaysUsePrimeLine": alwaysUsePrimeLine, "alwaysUsePrimeLineForVoiceMessage": alwaysUsePrimeLineForVoiceMessage, "featureControlPolicy": featureControlPolicy, "deviceTrustMode": deviceTrustMode, "earlyOfferSupportForVoiceCall": earlyOfferSupportForVoiceCall, "requireThirdPartyRegistration": requireThirdPartyRegistration, "blockIncomingCallsWhenRoaming": blockIncomingCallsWhenRoaming, "homeNetworkId": homeNetworkId, "AllowPresentationSharingUsingBfcp": AllowPresentationSharingUsingBfcp, "confidentialAccess": confidentialAccess, "requireOffPremiseLocation": requireOffPremiseLocation, "allowiXApplicableMedia": allowiXApplicableMedia, "cgpnIngressDN": cgpnIngressDN, "useDevicePoolCgpnIngressDN": useDevicePoolCgpnIngressDN, "msisdn": msisdn, "enableCallRoutingToRdWhenNoneIsActive": enableCallRoutingToRdWhenNoneIsActive, "wifiHotspotProfile": wifiHotspotProfile, "wirelessLanProfileGroup": wirelessLanProfileGroup, "elinGroup": elinGroup})["return"]
        except Fault as e:
            return e

    def updatePhone(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param commonPhoneConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param versionStamp: String128(value), optional
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param securityProfileName: optional
            _value_1: String(value)
        :param sipProfileName: optional
            _value_1: String(value)
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param sendGeoLocation: boolean(value), optional
        :param removeLines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param addLines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param lines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param phoneTemplateName: optional
            _value_1: String(value)
        :param speeddials: optional
            speeddial: optional
                dirn: String255(value)
                label: String(value)
                index: UnionType(value)
        :param busyLampFields: optional
            busyLampField: optional
                blfDest: String(value)
                blfDirn: String255(value), optional
                routePartition: String(value), optional
                label: String(value), optional
                    feature: UnionType(value), optional
                index: UnionType(value)
        :param primaryPhoneName: optional
            _value_1: String(value)
        :param ringSettingIdleBlfAudibleAlert: UnionType(value), optional
        :param ringSettingBusyBlfAudibleAlert: UnionType(value), optional
        :param blfDirectedCallParks: optional
            blfDirectedCallPark: optional
                label: String50(value), optional
                directedCallParkId: XUUID(value), optional
                    dnPattern: String255(value)
                index: UnionType(value)
        :param addOnModules: optional
            addOnModule: optional
                    _value_1: String(value)
                model: UnionType(value)
                index: UnionType(value)
        :param userLocale: UnionType(value), optional
        :param networkLocale: UnionType(value), optional
        :param idleTimeout: UnionType(value), optional
        :param authenticationUrl: String(value), optional
        :param directoryUrl: String(value), optional
        :param idleUrl: String(value), optional
        :param informationUrl: String(value), optional
        :param messagesUrl: String(value), optional
        :param proxyServerUrl: String(value), optional
        :param servicesUrl: String(value), optional
        :param services: optional
            service: optional
                    _value_1: String(value)
                name: String100(value)
                url: String(value), optional
                urlButtonIndex: UnionType(value), optional
                urlLabel: String50(value), optional
                serviceNameAscii: String50(value), optional
        :param softkeyTemplateName: optional
            _value_1: String(value)
        :param defaultProfileName: optional
            _value_1: String(value)
        :param enableExtensionMobility: boolean(value), optional
        :param singleButtonBarge: UnionType(value), optional
        :param joinAcrossLines: UnionType(value), optional
        :param builtInBridgeStatus: UnionType(value), optional
        :param callInfoPrivacyStatus: UnionType(value), optional
        :param hlogStatus: UnionType(value), optional
        :param ownerUserName: optional
            _value_1: String(value)
        :param ignorePresentationIndicators: boolean(value), optional
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param subscribeCallingSearchSpaceName: optional
            _value_1: String(value)
        :param rerouteCallingSearchSpaceName: optional
            _value_1: String(value)
        :param allowCtiControlFlag: boolean(value), optional
        :param presenceGroupName: optional
            _value_1: String(value)
        :param unattendedPort: boolean(value), optional
        :param requireDtmfReception: boolean(value), optional
        :param rfc2833Disabled: boolean(value), optional
        :param certificateOperation: UnionType(value), optional
        :param authenticationMode: UnionType(value), optional
        :param keySize: UnionType(value), optional
        :param keyOrder: UnionType(value), optional
        :param ecKeySize: UnionType(value), optional
        :param authenticationString: String128(value), optional
        :param upgradeFinishTime: String(value), optional
        :param deviceMobilityMode: UnionType(value), optional
        :param remoteDevice: boolean(value), optional
        :param dndOption: UnionType(value), optional
        :param dndRingSetting: UnionType(value), optional
        :param dndStatus: boolean(value), optional
        :param isActive: boolean(value), optional
        :param mobilityUserIdName: optional
            _value_1: String(value)
        :param phoneSuite: UnionType(value), optional
        :param phoneServiceDisplay: UnionType(value), optional
        :param isProtected: boolean(value), optional
        :param mtpRequired: boolean(value), optional
        :param mtpPreferedCodec: UnionType(value), optional
        :param dialRulesName: optional
            _value_1: String(value)
        :param sshUserId: String50(value), optional
        :param sshPwd: String255(value), optional
        :param digestUser: String255(value), optional
        :param outboundCallRollover: UnionType(value), optional
        :param hotlineDevice: boolean(value), optional
        :param secureInformationUrl: String255(value), optional
        :param secureDirectoryUrl: String255(value), optional
        :param secureMessageUrl: String255(value), optional
        :param secureServicesUrl: String255(value), optional
        :param secureAuthenticationUrl: String255(value), optional
        :param secureIdleUrl: String255(value), optional
        :param alwaysUsePrimeLine: UnionType(value), optional
        :param alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param featureControlPolicy: optional
            _value_1: String(value)
        :param deviceTrustMode: UnionType(value), optional
        :param earlyOfferSupportForVoiceCall: boolean(value), optional
        :param requireThirdPartyRegistration: boolean(value), optional
        :param blockIncomingCallsWhenRoaming: boolean(value), optional
        :param homeNetworkId: String(value), optional
        :param AllowPresentationSharingUsingBfcp: boolean(value), optional
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        :param requireOffPremiseLocation: boolean(value), optional
        :param allowiXApplicableMedia: boolean(value), optional
        :param cgpnIngressDN: optional
            _value_1: String(value)
        :param useDevicePoolCgpnIngressDN: boolean(value), optional
        :param msisdn: String128(value), optional
        :param enableCallRoutingToRdWhenNoneIsActive: boolean(value), optional
        :param wifiHotspotProfile: optional
            _value_1: String(value)
        :param wirelessLanProfileGroup: optional
            _value_1: String(value)
        :param elinGroup: optional
            _value_1: String(value)
        """
        try:
            return self.client.updatePhone(**kwargs)["return"]
        except Fault as e:
            return e

    def getPhone(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: optional
                _value_1: String(value)
            sipProfileName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            lines: optional
                    index: UnionType(value), optional
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    dialPlanWizardId: UnionType(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value), optional
                    routePartitionName: String50(value), optional
            numberOfButtons: UnionType(value), optional
            phoneTemplateName: optional
                _value_1: String(value)
            speeddials: optional
                    dirn: String255(value), optional
                    label: String(value), optional
                    index: UnionType(value), optional
            busyLampFields: optional
                    blfDest: String(value), optional
                    blfDirn: String255(value), optional
                    routePartition: String(value), optional
                    label: String(value), optional
                    index: UnionType(value), optional
            primaryPhoneName: optional
                _value_1: String(value)
            ringSettingIdleBlfAudibleAlert: UnionType(value), optional
            ringSettingBusyBlfAudibleAlert: UnionType(value), optional
            blfDirectedCallParks: optional
                    label: String50(value), optional
                    directedCallParkId: XUUID(value), optional
                    index: UnionType(value), optional
            addOnModules: optional
                    model: UnionType(value), optional
                    index: UnionType(value), optional
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            idleTimeout: UnionType(value), optional
            authenticationUrl: String(value), optional
            directoryUrl: String(value), optional
            idleUrl: String(value), optional
            informationUrl: String(value), optional
            messagesUrl: String(value), optional
            proxyServerUrl: String(value), optional
            servicesUrl: String(value), optional
            services: optional
                    name: String100(value), optional
                    url: String(value), optional
                    urlButtonIndex: UnionType(value), optional
                    urlLabel: String50(value), optional
                    serviceNameAscii: String50(value), optional
                    phoneService: UnionType(value), optional
                    phoneServiceCategory: UnionType(value), optional
                    vendor: String(value), optional
                    version: String(value), optional
                    priority: Int(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            loginUserId: String255(value), optional
            defaultProfileName: optional
                _value_1: String(value)
            enableExtensionMobility: boolean(value), optional
            currentProfileName: optional
                _value_1: String(value)
            loginTime: UnionType(value), optional
            loginDuration: UnionType(value), optional
            currentConfig: optional
                userHoldMohAudioSourceId: UnionType(value), optional
                    _value_1: String(value)
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                    _value_1: String(value)
                ignorePresentationIndicators: boolean(value), optional
                singleButtonBarge: UnionType(value), optional
                joinAcrossLines: UnionType(value), optional
                callInfoPrivacyStatus: UnionType(value), optional
                dndStatus: boolean(value), optional
                dndRingSetting: UnionType(value), optional
                dndOption: UnionType(value), optional
                alwaysUsePrimeLine: UnionType(value), optional
                alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
                    _value_1: String(value)
                deviceName: String255(value), optional
                model: UnionType(value), optional
                product: UnionType(value), optional
                deviceProtocol: UnionType(value), optional
                class: UnionType(value), optional
                addressMode: UnionType(value), optional
                allowAutoConfig: UnionType(value), optional
                remoteSrstOption: String50(value), optional
                remoteSrstIp: String15(value), optional
                remoteSrstPort: UnionType(value), optional
                remoteSipSrstIp: String15(value), optional
                remoteSipSrstPort: UnionType(value), optional
                geolocationInfo: String2048(value), optional
                remoteLocationName: String50(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            builtInBridgeStatus: UnionType(value), optional
            callInfoPrivacyStatus: UnionType(value), optional
            hlogStatus: UnionType(value), optional
            ownerUserName: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            allowCtiControlFlag: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            unattendedPort: boolean(value), optional
            requireDtmfReception: boolean(value), optional
            rfc2833Disabled: boolean(value), optional
            certificateOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            authenticationString: String128(value), optional
            certificateStatus: UnionType(value), optional
            upgradeFinishTime: String(value), optional
            deviceMobilityMode: UnionType(value), optional
            roamingDevicePoolName: optional
                _value_1: String(value)
            remoteDevice: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            isActive: boolean(value), optional
            isDualMode: boolean(value), optional
            mobilityUserIdName: optional
                _value_1: String(value)
            phoneSuite: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            isProtected: boolean(value), optional
            mtpRequired: boolean(value), optional
            mtpPreferedCodec: UnionType(value), optional
            dialRulesName: optional
                _value_1: String(value)
            sshUserId: String50(value), optional
            sshPwd: String255(value), optional
            digestUser: String255(value), optional
            outboundCallRollover: UnionType(value), optional
            hotlineDevice: boolean(value), optional
            secureInformationUrl: String255(value), optional
            secureDirectoryUrl: String255(value), optional
            secureMessageUrl: String255(value), optional
            secureServicesUrl: String255(value), optional
            secureAuthenticationUrl: String255(value), optional
            secureIdleUrl: String255(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
            deviceTrustMode: UnionType(value), optional
            earlyOfferSupportForVoiceCall: boolean(value), optional
            requireThirdPartyRegistration: boolean(value), optional
            blockIncomingCallsWhenRoaming: boolean(value), optional
            homeNetworkId: String(value), optional
            AllowPresentationSharingUsingBfcp: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            requireOffPremiseLocation: boolean(value), optional
            allowiXApplicableMedia: boolean(value), optional
            cgpnIngressDN: optional
                _value_1: String(value)
            useDevicePoolCgpnIngressDN: boolean(value), optional
            msisdn: String128(value), optional
            enableCallRoutingToRdWhenNoneIsActive: boolean(value), optional
            wifiHotspotProfile: optional
                _value_1: String(value)
            wirelessLanProfileGroup: optional
                _value_1: String(value)
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.getPhone(**kwargs)["return"]["phone"]
        except Fault as e:
            return e

    def removePhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removePhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listPhone(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'commonPhoneConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'traceFlag': '', 'mlppIndicationStatus': '', 'preemption': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'securityProfileName': {'_value_1': ''}, 'sipProfileName': {'_value_1': ''}, 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'geoLocationFilterName': {'_value_1': ''}, 'sendGeoLocation': '', 'numberOfButtons': '', 'phoneTemplateName': {'_value_1': ''}, 'primaryPhoneName': {'_value_1': ''}, 'ringSettingIdleBlfAudibleAlert': '', 'ringSettingBusyBlfAudibleAlert': '', 'userLocale': '', 'networkLocale': '', 'idleTimeout': '', 'authenticationUrl': '', 'directoryUrl': '', 'idleUrl': '', 'informationUrl': '', 'messagesUrl': '', 'proxyServerUrl': '', 'servicesUrl': '', 'softkeyTemplateName': {'_value_1': ''}, 'loginUserId': '', 'defaultProfileName': {'_value_1': ''}, 'enableExtensionMobility': '', 'currentProfileName': {'_value_1': ''}, 'loginTime': '', 'loginDuration': '', 'currentConfig': {'userHoldMohAudioSourceId': '', 'mlppDomainId': '', 'mlppIndicationStatus': '', 'preemption': '', 'ignorePresentationIndicators': '', 'singleButtonBarge': '', 'joinAcrossLines': '', 'callInfoPrivacyStatus': '', 'dndStatus': '', 'dndRingSetting': '', 'dndOption': '', 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'deviceName': '', 'model': '', 'product': '', 'deviceProtocol': '', 'class': '', 'addressMode': '', 'allowAutoConfig': '', 'remoteSrstOption': '', 'remoteSrstIp': '', 'remoteSrstPort': '', 'remoteSipSrstIp': '', 'remoteSipSrstPort': '', 'geolocationInfo': '', 'remoteLocationName': ''}, 'singleButtonBarge': '', 'joinAcrossLines': '', 'builtInBridgeStatus': '', 'callInfoPrivacyStatus': '', 'hlogStatus': '', 'ownerUserName': {'_value_1': ''}, 'ignorePresentationIndicators': '', 'packetCaptureMode': '', 'packetCaptureDuration': '', 'subscribeCallingSearchSpaceName': {'_value_1': ''}, 'rerouteCallingSearchSpaceName': {'_value_1': ''}, 'allowCtiControlFlag': '', 'presenceGroupName': {'_value_1': ''}, 'unattendedPort': '', 'requireDtmfReception': '', 'rfc2833Disabled': '', 'certificateOperation': '', 'authenticationMode': '', 'keySize': '', 'keyOrder': '', 'ecKeySize': '', 'authenticationString': '', 'certificateStatus': '', 'upgradeFinishTime': '', 'deviceMobilityMode': '', 'roamingDevicePoolName': {'_value_1': ''}, 'remoteDevice': '', 'dndOption': '', 'dndRingSetting': '', 'dndStatus': '', 'isActive': '', 'isDualMode': '', 'mobilityUserIdName': {'_value_1': ''}, 'phoneSuite': '', 'phoneServiceDisplay': '', 'isProtected': '', 'mtpRequired': '', 'mtpPreferedCodec': '', 'dialRulesName': {'_value_1': ''}, 'sshUserId': '', 'digestUser': '', 'outboundCallRollover': '', 'hotlineDevice': '', 'secureInformationUrl': '', 'secureDirectoryUrl': '', 'secureMessageUrl': '', 'secureServicesUrl': '', 'secureAuthenticationUrl': '', 'secureIdleUrl': '', 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'featureControlPolicy': {'_value_1': ''}, 'deviceTrustMode': '', 'earlyOfferSupportForVoiceCall': '', 'requireThirdPartyRegistration': '', 'blockIncomingCallsWhenRoaming': '', 'homeNetworkId': '', 'AllowPresentationSharingUsingBfcp': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}, 'requireOffPremiseLocation': '', 'allowiXApplicableMedia': '', 'enableCallRoutingToRdWhenNoneIsActive': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            protocol: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
            securityProfileName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            commonPhoneConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: optional
                _value_1: String(value)
            sipProfileName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            numberOfButtons: UnionType(value), optional
            phoneTemplateName: optional
                _value_1: String(value)
            primaryPhoneName: optional
                _value_1: String(value)
            ringSettingIdleBlfAudibleAlert: UnionType(value), optional
            ringSettingBusyBlfAudibleAlert: UnionType(value), optional
            userLocale: UnionType(value), optional
            networkLocale: UnionType(value), optional
            idleTimeout: UnionType(value), optional
            authenticationUrl: String(value), optional
            directoryUrl: String(value), optional
            idleUrl: String(value), optional
            informationUrl: String(value), optional
            messagesUrl: String(value), optional
            proxyServerUrl: String(value), optional
            servicesUrl: String(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            loginUserId: String255(value), optional
            defaultProfileName: optional
                _value_1: String(value)
            enableExtensionMobility: boolean(value), optional
            currentProfileName: optional
                _value_1: String(value)
            loginTime: UnionType(value), optional
            loginDuration: UnionType(value), optional
            currentConfig: optional
                userHoldMohAudioSourceId: UnionType(value), optional
                    _value_1: String(value)
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                    _value_1: String(value)
                ignorePresentationIndicators: boolean(value), optional
                singleButtonBarge: UnionType(value), optional
                joinAcrossLines: UnionType(value), optional
                callInfoPrivacyStatus: UnionType(value), optional
                dndStatus: boolean(value), optional
                dndRingSetting: UnionType(value), optional
                dndOption: UnionType(value), optional
                alwaysUsePrimeLine: UnionType(value), optional
                alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
                    _value_1: String(value)
                deviceName: String255(value), optional
                model: UnionType(value), optional
                product: UnionType(value), optional
                deviceProtocol: UnionType(value), optional
                class: UnionType(value), optional
                addressMode: UnionType(value), optional
                allowAutoConfig: UnionType(value), optional
                remoteSrstOption: String50(value), optional
                remoteSrstIp: String15(value), optional
                remoteSrstPort: UnionType(value), optional
                remoteSipSrstIp: String15(value), optional
                remoteSipSrstPort: UnionType(value), optional
                geolocationInfo: String2048(value), optional
                remoteLocationName: String50(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            builtInBridgeStatus: UnionType(value), optional
            callInfoPrivacyStatus: UnionType(value), optional
            hlogStatus: UnionType(value), optional
            ownerUserName: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            allowCtiControlFlag: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            unattendedPort: boolean(value), optional
            requireDtmfReception: boolean(value), optional
            rfc2833Disabled: boolean(value), optional
            certificateOperation: UnionType(value), optional
            authenticationMode: UnionType(value), optional
            keySize: UnionType(value), optional
            keyOrder: UnionType(value), optional
            ecKeySize: UnionType(value), optional
            authenticationString: String128(value), optional
            certificateStatus: UnionType(value), optional
            upgradeFinishTime: String(value), optional
            deviceMobilityMode: UnionType(value), optional
            roamingDevicePoolName: optional
                _value_1: String(value)
            remoteDevice: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            isActive: boolean(value), optional
            isDualMode: boolean(value), optional
            mobilityUserIdName: optional
                _value_1: String(value)
            phoneSuite: UnionType(value), optional
            phoneServiceDisplay: UnionType(value), optional
            isProtected: boolean(value), optional
            mtpRequired: boolean(value), optional
            mtpPreferedCodec: UnionType(value), optional
            dialRulesName: optional
                _value_1: String(value)
            sshUserId: String50(value), optional
            digestUser: String255(value), optional
            outboundCallRollover: UnionType(value), optional
            hotlineDevice: boolean(value), optional
            secureInformationUrl: String255(value), optional
            secureDirectoryUrl: String255(value), optional
            secureMessageUrl: String255(value), optional
            secureServicesUrl: String255(value), optional
            secureAuthenticationUrl: String255(value), optional
            secureIdleUrl: String255(value), optional
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
            deviceTrustMode: UnionType(value), optional
            earlyOfferSupportForVoiceCall: boolean(value), optional
            requireThirdPartyRegistration: boolean(value), optional
            blockIncomingCallsWhenRoaming: boolean(value), optional
            homeNetworkId: String(value), optional
            AllowPresentationSharingUsingBfcp: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            requireOffPremiseLocation: boolean(value), optional
            allowiXApplicableMedia: boolean(value), optional
            enableCallRoutingToRdWhenNoneIsActive: boolean(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listPhone({'name': '%', 'description': '%', 'protocol': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%', 'securityProfileName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["phone"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetPhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetPhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartPhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartPhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyPhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyPhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def wipePhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.wipePhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def lockPhone(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.lockPhone(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def getPhoneOptions(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedChoices: optional
            product: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            class: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            protocol: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
            protocolSide: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callingSearchSpaceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            devicePoolName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            commonDeviceConfigName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            commonPhoneConfigName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            networkLocation: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            locationName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            mediaResourceListName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            networkHoldMohAudioSourceId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            userHoldMohAudioSourceId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            automatedAlternateRoutingCssName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            aarNeighborhoodName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            mlppIndicationStatus: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            preemption: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            useTrustedRelayPoint: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            securityProfileName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
                protocol: String(value)
            sipProfileName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            cgpnTransformationCssName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            geoLocationName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            geoLocationFilterName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            lines: optional
            phoneTemplateName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            primaryPhoneName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            ringSettingIdleBlfAudibleAlert: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            ringSettingBusyBlfAudibleAlert: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            blfDirectedCallParks: optional
                    directedCallParkId: XUUID(value), optional
            addOnModules: optional
            userLocale: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            networkLocale: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
            services: optional
            softkeyTemplateName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            defaultProfileName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
                protocol: String(value)
            singleButtonBarge: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            joinAcrossLines: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            builtInBridgeStatus: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callInfoPrivacyStatus: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            ownerUserName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
            packetCaptureMode: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            subscribeCallingSearchSpaceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            rerouteCallingSearchSpaceName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            presenceGroupName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            certificateOperation: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            authenticationMode: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            keySize: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            keyOrder: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            ecKeySize: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            deviceMobilityMode: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dndOption: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
                protocol: String(value)
            dndRingSetting: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            mobilityUserIdName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            phoneSuite: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            phoneServiceDisplay: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            mtpPreferedCodec: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dialRulesName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
            outboundCallRollover: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            alwaysUsePrimeLine: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            alwaysUsePrimeLineForVoiceMessage: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            featureControlPolicy: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            deviceTrustMode: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            confidentialAccess: optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
                    skip: UnsignedLong(value), optional
                    first: UnsignedLong(value), optional
            cgpnIngressDN: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            wifiHotspotProfile: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            wirelessLanProfileGroup: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            elinGroup: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
        """
        try:
            return self.client.getPhoneOptions(**kwargs)["return"]["phoneOptions"]
        except Fault as e:
            return e

    def addH323Gateway(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, tunneledProtocol="", asn1RoseOidEncoding=None, qsigVariant=None, vendorConfig=None, pathReplacementSupport=None, traceFlag=None, mlppDomainId=None, useTrustedRelayPoint="", retryVideoCallAsAudio=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, geoLocationFilterName=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, packetCaptureMode="", packetCaptureDuration=None, srtpAllowed=None, waitForFarEndH245TerminalSet=None, mtpRequired=None, callerIdDn=None, callingPartySelection="", callingLineIdPresentation="", enableInboundFaststart=None, enableOutboundFaststart=None, codecForOutboundFaststart=None, transmitUtf8=None, signalingPort="", allowH235PassThrough=None, sigDigits=None, prefixDn=None, calledPartyIeNumberType="", callingPartyIeNumberType="", calledNumberingPlan="", callingNumberingPlan="", callingPartyNationalPrefix=None, callingPartyInternationalPrefix=None, callingPartyUnknownPrefix=None, callingPartySubscriberPrefix=None, callingPartyNationalStripDigits=None, callingPartyInternationalStripDigits=None, callingPartyUnknownStripDigits=None, callingPartySubscriberStripDigits=None, callingPartyNationalTransformationCssName=None, callingPartyInternationalTransformationCssName=None, callingPartyUnknownTransformationCssName=None, callingPartySubscriberTransformationCssName=None, calledPartyNationalPrefix=None, calledPartyInternationalPrefix=None, calledPartyUnknownPrefix=None, calledPartySubscriberPrefix=None, calledPartyNationalStripDigits=None, calledPartyInternationalStripDigits=None, calledPartyUnknownStripDigits=None, calledPartySubscriberStripDigits=None, calledPartyNationalTransformationCssName=None, calledPartyInternationalTransformationCssName=None, calledPartyUnknownTransformationCssName=None, calledPartySubscriberTransformationCssName=None, pstnAccess=None, imeE164TransformationName=None, displayIeDelivery=None, redirectOutboundNumberIe=None, redirectInboundNumberIe=None, useDevicePoolCgpnTransformCssNatl=None, useDevicePoolCgpnTransformCssIntl=None, useDevicePoolCgpnTransformCssUnkn=None, useDevicePoolCgpnTransformCssSubs=None, useDevicePoolCalledCssNatl=None, useDevicePoolCalledCssIntl=None, useDevicePoolCalledCssUnkn=None, useDevicePoolCalledCssSubs=None, useDevicePoolCntdPnTransformationCss=None, cntdPnTransformationCssName=None, confidentialAccess=None, redirectingPartyTransformationCSS=None, connectCallBeforePlayingAnnouncement=None):
        """
        :param h323Gateway: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            tunneledProtocol: UnionType(value)
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            pathReplacementSupport: boolean(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value)
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value)
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value)
            callingLineIdPresentation: UnionType(value)
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value)
            allowH235PassThrough: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value)
            callingPartyIeNumberType: UnionType(value)
            calledNumberingPlan: UnionType(value)
            callingNumberingPlan: UnionType(value)
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            displayIeDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            redirectingPartyTransformationCSS: optional
                _value_1: String(value)
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addH323Gateway({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "tunneledProtocol": tunneledProtocol, "asn1RoseOidEncoding": asn1RoseOidEncoding, "qsigVariant": qsigVariant, "vendorConfig": vendorConfig, "pathReplacementSupport": pathReplacementSupport, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "geoLocationFilterName": geoLocationFilterName, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "srtpAllowed": srtpAllowed, "waitForFarEndH245TerminalSet": waitForFarEndH245TerminalSet, "mtpRequired": mtpRequired, "callerIdDn": callerIdDn, "callingPartySelection": callingPartySelection, "callingLineIdPresentation": callingLineIdPresentation, "enableInboundFaststart": enableInboundFaststart, "enableOutboundFaststart": enableOutboundFaststart, "codecForOutboundFaststart": codecForOutboundFaststart, "transmitUtf8": transmitUtf8, "signalingPort": signalingPort, "allowH235PassThrough": allowH235PassThrough, "sigDigits": sigDigits, "prefixDn": prefixDn, "calledPartyIeNumberType": calledPartyIeNumberType, "callingPartyIeNumberType": callingPartyIeNumberType, "calledNumberingPlan": calledNumberingPlan, "callingNumberingPlan": callingNumberingPlan, "callingPartyNationalPrefix": callingPartyNationalPrefix, "callingPartyInternationalPrefix": callingPartyInternationalPrefix, "callingPartyUnknownPrefix": callingPartyUnknownPrefix, "callingPartySubscriberPrefix": callingPartySubscriberPrefix, "callingPartyNationalStripDigits": callingPartyNationalStripDigits, "callingPartyInternationalStripDigits": callingPartyInternationalStripDigits, "callingPartyUnknownStripDigits": callingPartyUnknownStripDigits, "callingPartySubscriberStripDigits": callingPartySubscriberStripDigits, "callingPartyNationalTransformationCssName": callingPartyNationalTransformationCssName, "callingPartyInternationalTransformationCssName": callingPartyInternationalTransformationCssName, "callingPartyUnknownTransformationCssName": callingPartyUnknownTransformationCssName, "callingPartySubscriberTransformationCssName": callingPartySubscriberTransformationCssName, "calledPartyNationalPrefix": calledPartyNationalPrefix, "calledPartyInternationalPrefix": calledPartyInternationalPrefix, "calledPartyUnknownPrefix": calledPartyUnknownPrefix, "calledPartySubscriberPrefix": calledPartySubscriberPrefix, "calledPartyNationalStripDigits": calledPartyNationalStripDigits, "calledPartyInternationalStripDigits": calledPartyInternationalStripDigits, "calledPartyUnknownStripDigits": calledPartyUnknownStripDigits, "calledPartySubscriberStripDigits": calledPartySubscriberStripDigits, "calledPartyNationalTransformationCssName": calledPartyNationalTransformationCssName, "calledPartyInternationalTransformationCssName": calledPartyInternationalTransformationCssName, "calledPartyUnknownTransformationCssName": calledPartyUnknownTransformationCssName, "calledPartySubscriberTransformationCssName": calledPartySubscriberTransformationCssName, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName, "displayIeDelivery": displayIeDelivery, "redirectOutboundNumberIe": redirectOutboundNumberIe, "redirectInboundNumberIe": redirectInboundNumberIe, "useDevicePoolCgpnTransformCssNatl": useDevicePoolCgpnTransformCssNatl, "useDevicePoolCgpnTransformCssIntl": useDevicePoolCgpnTransformCssIntl, "useDevicePoolCgpnTransformCssUnkn": useDevicePoolCgpnTransformCssUnkn, "useDevicePoolCgpnTransformCssSubs": useDevicePoolCgpnTransformCssSubs, "useDevicePoolCalledCssNatl": useDevicePoolCalledCssNatl, "useDevicePoolCalledCssIntl": useDevicePoolCalledCssIntl, "useDevicePoolCalledCssUnkn": useDevicePoolCalledCssUnkn, "useDevicePoolCalledCssSubs": useDevicePoolCalledCssSubs, "useDevicePoolCntdPnTransformationCss": useDevicePoolCntdPnTransformationCss, "cntdPnTransformationCssName": cntdPnTransformationCssName, "confidentialAccess": confidentialAccess, "redirectingPartyTransformationCSS": redirectingPartyTransformationCSS, "connectCallBeforePlayingAnnouncement": connectCallBeforePlayingAnnouncement})["return"]
        except Fault as e:
            return e

    def updateH323Gateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param tunneledProtocol: UnionType(value), optional
        :param asn1RoseOidEncoding: UnionType(value), optional
        :param qsigVariant: UnionType(value), optional
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param pathReplacementSupport: boolean(value), optional
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param srtpAllowed: boolean(value), optional
        :param waitForFarEndH245TerminalSet: boolean(value), optional
        :param mtpRequired: boolean(value), optional
        :param callerIdDn: String50(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingLineIdPresentation: UnionType(value), optional
        :param enableInboundFaststart: boolean(value), optional
        :param enableOutboundFaststart: boolean(value), optional
        :param codecForOutboundFaststart: UnionType(value), optional
        :param transmitUtf8: boolean(value), optional
        :param signalingPort: UnionType(value), optional
        :param allowH235PassThrough: boolean(value), optional
        :param sigDigits: optional
            _value_1: UnionType(value)
        :param prefixDn: String50(value), optional
        :param calledPartyIeNumberType: UnionType(value), optional
        :param callingPartyIeNumberType: UnionType(value), optional
        :param calledNumberingPlan: UnionType(value), optional
        :param callingNumberingPlan: UnionType(value), optional
        :param callingPartyNationalPrefix: String16(value), optional
        :param callingPartyInternationalPrefix: String16(value), optional
        :param callingPartyUnknownPrefix: String16(value), optional
        :param callingPartySubscriberPrefix: String16(value), optional
        :param callingPartyNationalStripDigits: UnionType(value), optional
        :param callingPartyInternationalStripDigits: UnionType(value), optional
        :param callingPartyUnknownStripDigits: UnionType(value), optional
        :param callingPartySubscriberStripDigits: UnionType(value), optional
        :param callingPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param callingPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param callingPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyNationalPrefix: String16(value), optional
        :param calledPartyInternationalPrefix: String16(value), optional
        :param calledPartyUnknownPrefix: String16(value), optional
        :param calledPartySubscriberPrefix: String16(value), optional
        :param calledPartyNationalStripDigits: UnionType(value), optional
        :param calledPartyInternationalStripDigits: UnionType(value), optional
        :param calledPartyUnknownStripDigits: UnionType(value), optional
        :param calledPartySubscriberStripDigits: UnionType(value), optional
        :param calledPartyNationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyInternationalTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param calledPartySubscriberTransformationCssName: optional
            _value_1: String(value)
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        :param displayIeDelivery: boolean(value), optional
        :param redirectOutboundNumberIe: boolean(value), optional
        :param redirectInboundNumberIe: boolean(value), optional
        :param useDevicePoolCgpnTransformCssNatl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssIntl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
        :param useDevicePoolCgpnTransformCssSubs: boolean(value), optional
        :param useDevicePoolCalledCssNatl: boolean(value), optional
        :param useDevicePoolCalledCssIntl: boolean(value), optional
        :param useDevicePoolCalledCssUnkn: boolean(value), optional
        :param useDevicePoolCalledCssSubs: boolean(value), optional
        :param useDevicePoolCntdPnTransformationCss: boolean(value), optional
        :param cntdPnTransformationCssName: optional
            _value_1: String(value)
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        :param redirectingPartyTransformationCSS: optional
            _value_1: String(value)
        :param connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateH323Gateway(**kwargs)["return"]
        except Fault as e:
            return e

    def getH323Gateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            pathReplacementSupport: boolean(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            waitForFarEndH245TerminalSet: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value), optional
            allowH235PassThrough: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value), optional
            callingPartyIeNumberType: UnionType(value), optional
            calledNumberingPlan: UnionType(value), optional
            callingNumberingPlan: UnionType(value), optional
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            displayIeDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            redirectingPartyTransformationCSS: optional
                _value_1: String(value)
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getH323Gateway(**kwargs)["return"]["h323Gateway"]
        except Fault as e:
            return e

    def removeH323Gateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeH323Gateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listH323Gateway(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'tunneledProtocol': '', 'asn1RoseOidEncoding': '', 'qsigVariant': '', 'pathReplacementSupport': '', 'traceFlag': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'geoLocationFilterName': {'_value_1': ''}, 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'packetCaptureMode': '', 'packetCaptureDuration': '', 'srtpAllowed': '', 'mtpRequired': '', 'callerIdDn': '', 'callingPartySelection': '', 'callingLineIdPresentation': '', 'enableInboundFaststart': '', 'enableOutboundFaststart': '', 'codecForOutboundFaststart': '', 'transmitUtf8': '', 'signalingPort': '', 'allowH235PassThrough': '', 'sigDigits': {'_value_1': ''}, 'prefixDn': '', 'calledPartyIeNumberType': '', 'callingPartyIeNumberType': '', 'calledNumberingPlan': '', 'callingNumberingPlan': '', 'callingPartyNationalPrefix': '', 'callingPartyInternationalPrefix': '', 'callingPartyUnknownPrefix': '', 'callingPartySubscriberPrefix': '', 'callingPartyNationalStripDigits': '', 'callingPartyInternationalStripDigits': '', 'callingPartyUnknownStripDigits': '', 'callingPartySubscriberStripDigits': '', 'callingPartyNationalTransformationCssName': {'_value_1': ''}, 'callingPartyInternationalTransformationCssName': {'_value_1': ''}, 'callingPartyUnknownTransformationCssName': {'_value_1': ''}, 'callingPartySubscriberTransformationCssName': {'_value_1': ''}, 'calledPartyNationalPrefix': '', 'calledPartyInternationalPrefix': '', 'calledPartyUnknownPrefix': '', 'calledPartySubscriberPrefix': '', 'calledPartyNationalStripDigits': '', 'calledPartyInternationalStripDigits': '', 'calledPartyUnknownStripDigits': '', 'calledPartySubscriberStripDigits': '', 'calledPartyNationalTransformationCssName': {'_value_1': ''}, 'calledPartyInternationalTransformationCssName': {'_value_1': ''}, 'calledPartyUnknownTransformationCssName': {'_value_1': ''}, 'calledPartySubscriberTransformationCssName': {'_value_1': ''}, 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}, 'displayIeDelivery': '', 'redirectOutboundNumberIe': '', 'redirectInboundNumberIe': '', 'useDevicePoolCgpnTransformCssNatl': '', 'useDevicePoolCgpnTransformCssIntl': '', 'useDevicePoolCgpnTransformCssUnkn': '', 'useDevicePoolCgpnTransformCssSubs': '', 'useDevicePoolCalledCssNatl': '', 'useDevicePoolCalledCssIntl': '', 'useDevicePoolCalledCssUnkn': '', 'useDevicePoolCalledCssSubs': '', 'useDevicePoolCntdPnTransformationCss': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            protocol: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            traceFlag: boolean(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            srtpAllowed: boolean(value), optional
            mtpRequired: boolean(value), optional
            callerIdDn: String50(value), optional
            callingPartySelection: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            enableInboundFaststart: boolean(value), optional
            enableOutboundFaststart: boolean(value), optional
            codecForOutboundFaststart: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            signalingPort: UnionType(value), optional
            allowH235PassThrough: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            prefixDn: String50(value), optional
            calledPartyIeNumberType: UnionType(value), optional
            callingPartyIeNumberType: UnionType(value), optional
            calledNumberingPlan: UnionType(value), optional
            callingNumberingPlan: UnionType(value), optional
            callingPartyNationalPrefix: String16(value), optional
            callingPartyInternationalPrefix: String16(value), optional
            callingPartyUnknownPrefix: String16(value), optional
            callingPartySubscriberPrefix: String16(value), optional
            callingPartyNationalStripDigits: UnionType(value), optional
            callingPartyInternationalStripDigits: UnionType(value), optional
            callingPartyUnknownStripDigits: UnionType(value), optional
            callingPartySubscriberStripDigits: UnionType(value), optional
            callingPartyNationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            callingPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            displayIeDelivery: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            redirectInboundNumberIe: boolean(value), optional
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listH323Gateway({'name': '%', 'description': '%', 'protocol': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["h323Gateway"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetH323Gateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetH323Gateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartH323Gateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartH323Gateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyH323Gateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyH323Gateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addDeviceProfile(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", userHoldMohAudioSourceId=None, vendorConfig=None, traceFlag=None, mlppDomainId=None, mlppIndicationStatus=None, preemption=None, lines=None, phoneTemplateName="", speeddials=None, busyLampFields=None, blfDirectedCallParks=None, addOnModules=None, userLocale=None, singleButtonBarge=None, joinAcrossLines=None, loginUserId=None, ignorePresentationIndicators=None, dndOption=None, dndRingSetting=None, dndStatus=None, emccCallingSearchSpace=None, alwaysUsePrimeLine=None, alwaysUsePrimeLineForVoiceMessage=None, softkeyTemplateName=None, callInfoPrivacyStatus=None, services=None, featureControlPolicy=None):
        """
        :param deviceProfile: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            userHoldMohAudioSourceId: UnionType(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            lines: optional
                    index: UnionType(value)
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value)
                    routePartitionName: String50(value)
            phoneTemplateName: 
                _value_1: String(value)
            speeddials: optional
                    dirn: String255(value)
                    label: String(value)
                    index: UnionType(value)
            busyLampFields: optional
                    blfDest: String(value)
                    blfDirn: String255(value), optional
                    routePartition: String(value), optional
                    label: String(value), optional
                    index: UnionType(value)
            blfDirectedCallParks: optional
                    label: String50(value), optional
                    directedCallParkId: XUUID(value), optional
                    index: UnionType(value)
            addOnModules: optional
                    model: UnionType(value)
                    index: UnionType(value)
            userLocale: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            loginUserId: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            emccCallingSearchSpace: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            callInfoPrivacyStatus: UnionType(value), optional
            services: optional
                    name: String100(value)
                    url: String(value), optional
                    urlButtonIndex: UnionType(value), optional
                    urlLabel: String50(value), optional
                    serviceNameAscii: String50(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
        """
        try:
            return self.client.addDeviceProfile({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "vendorConfig": vendorConfig, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "preemption": preemption, "lines": lines, "phoneTemplateName": phoneTemplateName, "speeddials": speeddials, "busyLampFields": busyLampFields, "blfDirectedCallParks": blfDirectedCallParks, "addOnModules": addOnModules, "userLocale": userLocale, "singleButtonBarge": singleButtonBarge, "joinAcrossLines": joinAcrossLines, "loginUserId": loginUserId, "ignorePresentationIndicators": ignorePresentationIndicators, "dndOption": dndOption, "dndRingSetting": dndRingSetting, "dndStatus": dndStatus, "emccCallingSearchSpace": emccCallingSearchSpace, "alwaysUsePrimeLine": alwaysUsePrimeLine, "alwaysUsePrimeLineForVoiceMessage": alwaysUsePrimeLineForVoiceMessage, "softkeyTemplateName": softkeyTemplateName, "callInfoPrivacyStatus": callInfoPrivacyStatus, "services": services, "featureControlPolicy": featureControlPolicy})["return"]
        except Fault as e:
            return e

    def updateDeviceProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param lines: optional
            line: optional
                index: UnionType(value)
                label: String50(value), optional
                display: String50(value), optional
                    pattern: String(value)
                ringSetting: UnionType(value), optional
                consecutiveRingSetting: UnionType(value), optional
                ringSettingIdlePickupAlert: UnionType(value), optional
                ringSettingActivePickupAlert: UnionType(value), optional
                displayAscii: String50(value), optional
                e164Mask: String255(value), optional
                mwlPolicy: UnionType(value), optional
                maxNumCalls: UnionType(value), optional
                busyTrigger: UnionType(value), optional
                    callerName: boolean(value), optional
                    callerNumber: boolean(value), optional
                    redirectedNumber: boolean(value), optional
                    dialedNumber: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                recordingFlag: UnionType(value), optional
                audibleMwi: UnionType(value), optional
                speedDial: String(value), optional
                partitionUsage: UnionType(value), optional
                missedCallLogging: boolean(value), optional
                recordingMediaSource: UnionType(value), optional
            lineIdentifier: optional
                directoryNumber: String255(value)
                routePartitionName: String50(value)
        :param phoneTemplateName: optional
            _value_1: String(value)
        :param speeddials: optional
            speeddial: optional
                dirn: String255(value)
                label: String(value)
                index: UnionType(value)
        :param busyLampFields: optional
            busyLampField: optional
                blfDest: String(value)
                blfDirn: String255(value), optional
                routePartition: String(value), optional
                label: String(value), optional
                    feature: UnionType(value), optional
                index: UnionType(value)
        :param blfDirectedCallParks: optional
            blfDirectedCallPark: optional
                label: String50(value), optional
                directedCallParkId: XUUID(value), optional
                    dnPattern: String255(value)
                index: UnionType(value)
        :param addOnModules: optional
            addOnModule: optional
                    _value_1: String(value)
                model: UnionType(value)
                index: UnionType(value)
        :param userLocale: UnionType(value), optional
        :param singleButtonBarge: UnionType(value), optional
        :param joinAcrossLines: UnionType(value), optional
        :param loginUserId: optional
            _value_1: String(value)
        :param ignorePresentationIndicators: boolean(value), optional
        :param dndOption: UnionType(value), optional
        :param dndRingSetting: UnionType(value), optional
        :param dndStatus: boolean(value), optional
        :param emccCallingSearchSpace: optional
            _value_1: String(value)
        :param alwaysUsePrimeLine: UnionType(value), optional
        :param alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
        :param softkeyTemplateName: optional
            _value_1: String(value)
        :param callInfoPrivacyStatus: UnionType(value), optional
        :param services: optional
            service: optional
                    _value_1: String(value)
                name: String100(value)
                url: String(value), optional
                urlButtonIndex: UnionType(value), optional
                urlLabel: String50(value), optional
                serviceNameAscii: String50(value), optional
        :param featureControlPolicy: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateDeviceProfile(**kwargs)["return"]
        except Fault as e:
            return e

    def getDeviceProfile(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            type: String(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            lines: optional
                    index: UnionType(value), optional
                    label: String50(value), optional
                    display: String50(value), optional
                    ringSetting: UnionType(value), optional
                    consecutiveRingSetting: UnionType(value), optional
                    ringSettingIdlePickupAlert: UnionType(value), optional
                    ringSettingActivePickupAlert: UnionType(value), optional
                    displayAscii: String50(value), optional
                    e164Mask: String255(value), optional
                    dialPlanWizardId: UnionType(value), optional
                    mwlPolicy: UnionType(value), optional
                    maxNumCalls: UnionType(value), optional
                    busyTrigger: UnionType(value), optional
                    recordingFlag: UnionType(value), optional
                    audibleMwi: UnionType(value), optional
                    speedDial: String(value), optional
                    partitionUsage: UnionType(value), optional
                    missedCallLogging: boolean(value), optional
                    recordingMediaSource: UnionType(value), optional
                    directoryNumber: String255(value), optional
                    routePartitionName: String50(value), optional
            numberOfButtons: UnionType(value), optional
            phoneTemplateName: optional
                _value_1: String(value)
            speeddials: optional
                    dirn: String255(value), optional
                    label: String(value), optional
                    index: UnionType(value), optional
            busyLampFields: optional
                    blfDest: String(value), optional
                    blfDirn: String255(value), optional
                    routePartition: String(value), optional
                    label: String(value), optional
                    index: UnionType(value), optional
            blfDirectedCallParks: optional
                    label: String50(value), optional
                    directedCallParkId: XUUID(value), optional
                    index: UnionType(value), optional
            addOnModules: optional
                    model: UnionType(value), optional
                    index: UnionType(value), optional
            userLocale: UnionType(value), optional
            defaultProfileName: String50(value), optional
            currentProfileName: optional
                _value_1: String(value)
            loginTime: UnionType(value), optional
            loginDuration: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            loginUserId: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            emccCallingSearchSpace: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            callInfoPrivacyStatus: UnionType(value), optional
            currentConfig: optional
                userHoldMohAudioSourceId: UnionType(value), optional
                    _value_1: String(value)
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                    _value_1: String(value)
                ignorePresentationIndicators: boolean(value), optional
                singleButtonBarge: UnionType(value), optional
                joinAcrossLines: UnionType(value), optional
                callInfoPrivacyStatus: UnionType(value), optional
                dndStatus: boolean(value), optional
                dndRingSetting: UnionType(value), optional
                dndOption: UnionType(value), optional
                alwaysUsePrimeLine: UnionType(value), optional
                alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
                    _value_1: String(value)
                deviceName: String255(value), optional
                model: UnionType(value), optional
                product: UnionType(value), optional
                deviceProtocol: UnionType(value), optional
                class: UnionType(value), optional
                addressMode: UnionType(value), optional
                allowAutoConfig: UnionType(value), optional
                remoteSrstOption: String50(value), optional
                remoteSrstIp: String15(value), optional
                remoteSrstPort: UnionType(value), optional
                remoteSipSrstIp: String15(value), optional
                remoteSipSrstPort: UnionType(value), optional
                geolocationInfo: String2048(value), optional
                remoteLocationName: String50(value), optional
            services: optional
                    name: String100(value), optional
                    url: String(value), optional
                    urlButtonIndex: UnionType(value), optional
                    urlLabel: String50(value), optional
                    serviceNameAscii: String50(value), optional
                    phoneService: UnionType(value), optional
                    phoneServiceCategory: UnionType(value), optional
                    vendor: String(value), optional
                    version: String(value), optional
                    priority: Int(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
        """
        try:
            return self.client.getDeviceProfile(**kwargs)["return"]["deviceProfile"]
        except Fault as e:
            return e

    def removeDeviceProfile(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeDeviceProfile(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listDeviceProfile(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'userHoldMohAudioSourceId': '', 'traceFlag': '', 'mlppIndicationStatus': '', 'preemption': '', 'phoneTemplateName': {'_value_1': ''}, 'userLocale': '', 'defaultProfileName': '', 'currentProfileName': {'_value_1': ''}, 'loginTime': '', 'loginDuration': '', 'singleButtonBarge': '', 'joinAcrossLines': '', 'loginUserId': {'_value_1': ''}, 'ignorePresentationIndicators': '', 'dndOption': '', 'dndRingSetting': '', 'dndStatus': '', 'emccCallingSearchSpace': {'_value_1': ''}, 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'softkeyTemplateName': {'_value_1': ''}, 'callInfoPrivacyStatus': '', 'currentConfig': {'userHoldMohAudioSourceId': '', 'mlppDomainId': '', 'mlppIndicationStatus': '', 'preemption': '', 'ignorePresentationIndicators': '', 'singleButtonBarge': '', 'joinAcrossLines': '', 'callInfoPrivacyStatus': '', 'dndStatus': '', 'dndRingSetting': '', 'dndOption': '', 'alwaysUsePrimeLine': '', 'alwaysUsePrimeLineForVoiceMessage': '', 'deviceName': '', 'model': '', 'product': '', 'deviceProtocol': '', 'class': '', 'addressMode': '', 'allowAutoConfig': '', 'remoteSrstOption': '', 'remoteSrstIp': '', 'remoteSrstPort': '', 'remoteSipSrstIp': '', 'remoteSipSrstPort': '', 'geolocationInfo': '', 'remoteLocationName': ''}, 'featureControlPolicy': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            traceFlag: boolean(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            phoneTemplateName: optional
                _value_1: String(value)
            userLocale: UnionType(value), optional
            defaultProfileName: String50(value), optional
            currentProfileName: optional
                _value_1: String(value)
            loginTime: UnionType(value), optional
            loginDuration: UnionType(value), optional
            singleButtonBarge: UnionType(value), optional
            joinAcrossLines: UnionType(value), optional
            loginUserId: optional
                _value_1: String(value)
            ignorePresentationIndicators: boolean(value), optional
            dndOption: UnionType(value), optional
            dndRingSetting: UnionType(value), optional
            dndStatus: boolean(value), optional
            emccCallingSearchSpace: optional
                _value_1: String(value)
            alwaysUsePrimeLine: UnionType(value), optional
            alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
            softkeyTemplateName: optional
                _value_1: String(value)
            callInfoPrivacyStatus: UnionType(value), optional
            currentConfig: optional
                userHoldMohAudioSourceId: UnionType(value), optional
                    _value_1: String(value)
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                    _value_1: String(value)
                ignorePresentationIndicators: boolean(value), optional
                singleButtonBarge: UnionType(value), optional
                joinAcrossLines: UnionType(value), optional
                callInfoPrivacyStatus: UnionType(value), optional
                dndStatus: boolean(value), optional
                dndRingSetting: UnionType(value), optional
                dndOption: UnionType(value), optional
                alwaysUsePrimeLine: UnionType(value), optional
                alwaysUsePrimeLineForVoiceMessage: UnionType(value), optional
                    _value_1: String(value)
                deviceName: String255(value), optional
                model: UnionType(value), optional
                product: UnionType(value), optional
                deviceProtocol: UnionType(value), optional
                class: UnionType(value), optional
                addressMode: UnionType(value), optional
                allowAutoConfig: UnionType(value), optional
                remoteSrstOption: String50(value), optional
                remoteSrstIp: String15(value), optional
                remoteSrstPort: UnionType(value), optional
                remoteSipSrstIp: String15(value), optional
                remoteSipSrstPort: UnionType(value), optional
                geolocationInfo: String2048(value), optional
                remoteLocationName: String50(value), optional
            featureControlPolicy: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listDeviceProfile({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["deviceProfile"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def getDeviceProfileOptions(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param returnedChoices: optional
            product: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            class: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            protocol: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
            protocolSide: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            userHoldMohAudioSourceId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            mlppIndicationStatus: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            preemption: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            lines: optional
            phoneTemplateName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
                product: String(value)
                protocol: String(value)
            blfDirectedCallParks: optional
                    directedCallParkId: XUUID(value), optional
            addOnModules: optional
            userLocale: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            singleButtonBarge: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            joinAcrossLines: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            loginUserId: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dndOption: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            dndRingSetting: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            emccCallingSearchSpace: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            alwaysUsePrimeLine: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            alwaysUsePrimeLineForVoiceMessage: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            softkeyTemplateName: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            callInfoPrivacyStatus: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
            services: optional
            featureControlPolicy: optional
                skip: UnsignedLong(value), optional
                first: UnsignedLong(value), optional
        """
        try:
            return self.client.getDeviceProfileOptions(**kwargs)["return"]["deviceProfileOptions"]
        except Fault as e:
            return e

    def addRemoteDestination(self, name=None, destination="", answerTooSoonTimer="", answerTooLateTimer="", delayBeforeRingingCell="", ownerUserId="", enableUnifiedMobility=None, remoteDestinationProfileName="", enableExtendAndConnect=None, ctiRemoteDeviceName=None, dualModeDeviceName="", isMobilePhone=None, enableMobileConnect=None, lineAssociations=None, timeZone=None, todAccessName=None, mobileSmartClientName=None, mobilityProfileName=None, singleNumberReachVoicemail=None, dialViaOfficeReverseVoicemail=None, ringSchedule=None, accessListName=None):
        """
        :param remoteDestination: 
            name: String(value), optional
            destination: String(value)
            answerTooSoonTimer: UnionType(value)
            answerTooLateTimer: UnionType(value)
            delayBeforeRingingCell: UnionType(value)
            ownerUserId: 
                _value_1: String(value)
            enableUnifiedMobility: boolean(value), optional
            remoteDestinationProfileName: 
                _value_1: String(value)
            enableExtendAndConnect: boolean(value), optional
            ctiRemoteDeviceName: optional
                _value_1: String(value)
            dualModeDeviceName: 
                _value_1: String(value)
            isMobilePhone: boolean(value), optional
            enableMobileConnect: boolean(value), optional
            lineAssociations: optional
                    uuid: XUUID(value), optional
                    pattern: String(value)
            timeZone: UnionType(value), optional
            todAccessName: optional
                _value_1: String(value)
            mobileSmartClientName: optional
                _value_1: String(value)
            mobilityProfileName: optional
                _value_1: String(value)
            singleNumberReachVoicemail: UnionType(value), optional
            dialViaOfficeReverseVoicemail: UnionType(value), optional
            ringSchedule: optional
                    startTime: UnionType(value), optional
                    endTime: UnionType(value), optional
                    startDay: UnionType(value), optional
                    endDay: UnionType(value), optional
            accessListName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addRemoteDestination({"name": name, "destination": destination, "answerTooSoonTimer": answerTooSoonTimer, "answerTooLateTimer": answerTooLateTimer, "delayBeforeRingingCell": delayBeforeRingingCell, "ownerUserId": ownerUserId, "enableUnifiedMobility": enableUnifiedMobility, "remoteDestinationProfileName": remoteDestinationProfileName, "enableExtendAndConnect": enableExtendAndConnect, "ctiRemoteDeviceName": ctiRemoteDeviceName, "dualModeDeviceName": dualModeDeviceName, "isMobilePhone": isMobilePhone, "enableMobileConnect": enableMobileConnect, "lineAssociations": lineAssociations, "timeZone": timeZone, "todAccessName": todAccessName, "mobileSmartClientName": mobileSmartClientName, "mobilityProfileName": mobilityProfileName, "singleNumberReachVoicemail": singleNumberReachVoicemail, "dialViaOfficeReverseVoicemail": dialViaOfficeReverseVoicemail, "ringSchedule": ringSchedule, "accessListName": accessListName})["return"]
        except Fault as e:
            return e

    def updateRemoteDestination(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param destination: String(value), optional
        :param newName: String(value), optional
        :param newDestination: String(value), optional
        :param answerTooSoonTimer: UnionType(value), optional
        :param answerTooLateTimer: UnionType(value), optional
        :param delayBeforeRingingCell: UnionType(value), optional
        :param ownerUserId: optional
            _value_1: String(value)
        :param enableUnifiedMobility: boolean(value), optional
        :param remoteDestinationProfileName: optional
            _value_1: String(value)
        :param enableExtendAndConnect: boolean(value), optional
        :param ctiRemoteDeviceName: optional
            _value_1: String(value)
        :param dualModeDeviceName: optional
            _value_1: String(value)
        :param isMobilePhone: boolean(value), optional
        :param enableMobileConnect: boolean(value), optional
        :param lineAssociations: optional
            lineAssociation: 
                uuid: XUUID(value), optional
                pattern: String(value)
                    _value_1: String(value)
        :param timeZone: UnionType(value), optional
        :param todAccessName: optional
            _value_1: String(value)
        :param mobileSmartClientName: optional
            _value_1: String(value)
        :param mobilityProfileName: optional
            _value_1: String(value)
        :param singleNumberReachVoicemail: UnionType(value), optional
        :param dialViaOfficeReverseVoicemail: UnionType(value), optional
        :param removeRingSchedule: optional
            member: 
                startTime: UnionType(value), optional
                endTime: UnionType(value), optional
                startDay: UnionType(value), optional
                endDay: UnionType(value), optional
        :param addRingSchedule: optional
            member: 
                startTime: UnionType(value), optional
                endTime: UnionType(value), optional
                startDay: UnionType(value), optional
                endDay: UnionType(value), optional
        :param ringSchedule: optional
            member: 
                startTime: UnionType(value), optional
                endTime: UnionType(value), optional
                startDay: UnionType(value), optional
                endDay: UnionType(value), optional
        :param accessListName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateRemoteDestination(**kwargs)["return"]
        except Fault as e:
            return e

    def getRemoteDestination(self, **kwargs):
        """
        :param destination: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String(value), optional
            destination: String(value), optional
            answerTooSoonTimer: UnionType(value), optional
            answerTooLateTimer: UnionType(value), optional
            delayBeforeRingingCell: UnionType(value), optional
            ownerUserId: optional
                _value_1: String(value)
            remoteDestinationProfileName: optional
                _value_1: String(value)
            ctiRemoteDeviceName: optional
                _value_1: String(value)
            dualModeDeviceName: optional
                _value_1: String(value)
            isMobilePhone: boolean(value), optional
            enableMobileConnect: boolean(value), optional
            lineAssociations: optional
                    uuid: XUUID(value), optional
                    pattern: String(value), optional
                    index: UnionType(value), optional
            timeZone: UnionType(value), optional
            todAccessName: optional
                _value_1: String(value)
            mobileSmartClientName: optional
                _value_1: String(value)
            mobilityProfileName: optional
                _value_1: String(value)
            singleNumberReachVoicemail: UnionType(value), optional
            dialViaOfficeReverseVoicemail: UnionType(value), optional
            ringSchedule: optional
                    startTime: UnionType(value), optional
                    endTime: UnionType(value), optional
                    startDay: UnionType(value), optional
                    endDay: UnionType(value), optional
            accessListName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getRemoteDestination(**kwargs)["return"]["remoteDestination"]
        except Fault as e:
            return e

    def removeRemoteDestination(self, destination="", uuid=""):
        """
        :param destination: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeRemoteDestination(destination=destination, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listRemoteDestination(self, returnedTags={'name': '', 'destination': '', 'answerTooSoonTimer': '', 'answerTooLateTimer': '', 'delayBeforeRingingCell': '', 'remoteDestinationProfileName': {'_value_1': ''}, 'ctiRemoteDeviceName': {'_value_1': ''}, 'dualModeDeviceName': {'_value_1': ''}, 'isMobilePhone': '', 'enableMobileConnect': '', 'timeZone': '', 'todAccessName': {'_value_1': ''}, 'mobileSmartClientName': {'_value_1': ''}, 'mobilityProfileName': {'_value_1': ''}, 'singleNumberReachVoicemail': '', 'dialViaOfficeReverseVoicemail': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            remoteDestinationProfileName: String(value), optional
            ctiRemoteDeviceName: String(value), optional
            dualModeDeviceName: String(value), optional
        :param returnedTags: 
            name: String(value), optional
            destination: String(value), optional
            answerTooSoonTimer: UnionType(value), optional
            answerTooLateTimer: UnionType(value), optional
            delayBeforeRingingCell: UnionType(value), optional
            remoteDestinationProfileName: optional
                _value_1: String(value)
            ctiRemoteDeviceName: optional
                _value_1: String(value)
            dualModeDeviceName: optional
                _value_1: String(value)
            isMobilePhone: boolean(value), optional
            enableMobileConnect: boolean(value), optional
            timeZone: UnionType(value), optional
            todAccessName: optional
                _value_1: String(value)
            mobileSmartClientName: optional
                _value_1: String(value)
            mobilityProfileName: optional
                _value_1: String(value)
            singleNumberReachVoicemail: UnionType(value), optional
            dialViaOfficeReverseVoicemail: UnionType(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRemoteDestination({'name': '%', 'remoteDestinationProfileName': '%', 'ctiRemoteDeviceName': '%', 'dualModeDeviceName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["remoteDestination"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addVg224(self, domainName="", description=None, product="", protocol="", callManagerGroupName="", units=None, vendorConfig=None, versionStamp=None):
        """
        :param vg224: 
            domainName: String(value)
            description: String(value), optional
            product: UnionType(value)
            protocol: UnionType(value)
            callManagerGroupName: 
                _value_1: String(value)
            units: optional
                    index: UnionType(value)
                    product: UnionType(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
        """
        try:
            return self.client.addVg224({"domainName": domainName, "description": description, "product": product, "protocol": protocol, "callManagerGroupName": callManagerGroupName, "units": units, "vendorConfig": vendorConfig, "versionStamp": versionStamp})["return"]
        except Fault as e:
            return e

    def updateVg224(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param domainName: String(value), optional
        :param newDomainName: String(value), optional
        :param description: String(value), optional
        :param callManagerGroupName: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        """
        try:
            return self.client.updateVg224(**kwargs)["return"]
        except Fault as e:
            return e

    def getVg224(self, **kwargs):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            description: String(value), optional
            product: UnionType(value), optional
            protocol: UnionType(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            units: optional
                    index: UnionType(value), optional
                    product: UnionType(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
        """
        try:
            return self.client.getVg224(**kwargs)["return"]["vg224"]
        except Fault as e:
            return e

    def removeVg224(self, domainName="", uuid=""):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeVg224(domainName=domainName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGateway(self, domainName="", description=None, product="", protocol="", callManagerGroupName="", units=None, vendorConfig=None):
        """
        :param gateway: 
            domainName: String(value)
            description: String(value), optional
            product: UnionType(value)
            protocol: UnionType(value)
            callManagerGroupName: 
                _value_1: String(value)
            units: optional
                    index: UnionType(value)
                    product: UnionType(value)
            vendorConfig: optional
                _value_1: AnyType, optional
        """
        try:
            return self.client.addGateway({"domainName": domainName, "description": description, "product": product, "protocol": protocol, "callManagerGroupName": callManagerGroupName, "units": units, "vendorConfig": vendorConfig})["return"]
        except Fault as e:
            return e

    def updateGateway(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param domainName: String(value), optional
        :param newDomainName: String(value), optional
        :param description: String(value), optional
        :param product: UnionType(value), optional
        :param protocol: UnionType(value), optional
        :param callManagerGroupName: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        """
        try:
            return self.client.updateGateway(**kwargs)["return"]
        except Fault as e:
            return e

    def getGateway(self, **kwargs):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            description: String(value), optional
            product: UnionType(value), optional
            protocol: UnionType(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            units: optional
                    index: UnionType(value), optional
                    product: UnionType(value), optional
            scratch: String(value), optional
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String(value), optional
            loadInformation: String(value), optional
        """
        try:
            return self.client.getGateway(**kwargs)["return"]["gateway"]
        except Fault as e:
            return e

    def removeGateway(self, domainName="", uuid=""):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGateway(domainName=domainName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listGateway(self, returnedTags={'domainName': '', 'description': '', 'product': '', 'protocol': '', 'callManagerGroupName': {'_value_1': ''}, 'scratch': '', 'loadInformation': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            domainName: String(value), optional
            description: String(value), optional
            product: String(value), optional
            protocol: String(value), optional
        :param returnedTags: 
            domainName: String(value), optional
            description: String(value), optional
            product: UnionType(value), optional
            protocol: UnionType(value), optional
            callManagerGroupName: optional
                _value_1: String(value)
            scratch: String(value), optional
            loadInformation: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listGateway({'domainName': '%', 'description': '%', 'product': '%', 'protocol': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["gateway"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetGateway(self, domainName="", uuid=""):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetGateway(domainName=domainName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyGateway(self, domainName="", uuid=""):
        """
        :param domainName: String(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyGateway(domainName=domainName, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGatewayEndpointAnalogAccess(self, domainName=None, gatewayUuid=None, unit="", subunit="", endpoint=""):
        """
        :param gatewayEndpointAnalogAccess: 
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value)
            subunit: UnionType(value)
            endpoint: 
                index: UnionType(value)
                name: UniqueString128(value)
                description: String128(value), optional
                product: UnionType(value)
                model: UnionType(value), optional
                class: UnionType(value)
                protocol: UnionType(value)
                protocolSide: UnionType(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                mlppDomainId: String128(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                retryVideoCallAsAudio: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    portNumber: UnionType(value)
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value)
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value)
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    smdiPortNumber: UnionType(value)
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value)
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    timer1: UnionType(value)
                    timer2: UnionType(value)
                    timer3: UnionType(value)
                    timer4: UnionType(value)
                    timer5: UnionType(value)
                    timer6: UnionType(value)
                trunkSelectionOrder: UnionType(value)
                transmitUtf8: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                callingPartyNumberPrefix: String16(value), optional
                callingPartyStripDigits: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssUnknown: boolean(value), optional
                hotlineDevice: boolean(value), optional
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                imeE164DirectoryNumber: String16(value), optional
                    confidentialAccessMode: UnionType(value)
                    confidentialAccessLevel: UnionType(value)
                    _value_1: String(value)
        """
        try:
            return self.client.addGatewayEndpointAnalogAccess({"domainName": domainName, "gatewayUuid": gatewayUuid, "unit": unit, "subunit": subunit, "endpoint": endpoint})["return"]
        except Fault as e:
            return e

    def updateGatewayEndpointAnalogAccess(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param endpoint: optional
            index: UnionType(value)
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            model: UnionType(value), optional
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            port: 
                portNumber: UnionType(value)
                attendantDn: String255(value), optional
                unattendedPort: boolean(value), optional
                callerIdDn: String255(value), optional
                callerIdEnable: boolean(value)
                callingPartySelection: UnionType(value), optional
                digitSending: UnionType(value), optional
                expectedDigits: UnionType(value)
                    _value_1: UnionType(value)
                prefixDn: String255(value), optional
                presentationBit: UnionType(value), optional
                silenceSuppressionThreshold: UnionType(value), optional
                smdiPortNumber: UnionType(value)
                startDialProtocol: UnionType(value), optional
                trunk: UnionType(value), optional
                trunkDirection: UnionType(value)
                trunkLevel: UnionType(value), optional
                trunkPadRx: UnionType(value), optional
                trunkPadTx: UnionType(value), optional
                    _value_1: AnyType, optional
                timer1: UnionType(value)
                timer2: UnionType(value)
                timer3: UnionType(value)
                timer4: UnionType(value)
                timer5: UnionType(value)
                timer6: UnionType(value)
            trunkSelectionOrder: UnionType(value)
            transmitUtf8: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            callingPartyNumberPrefix: String16(value), optional
            callingPartyStripDigits: UnionType(value), optional
            callingPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssUnknown: boolean(value), optional
            hotlineDevice: boolean(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            imeE164DirectoryNumber: String16(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            elinGroup: optional
                _value_1: String(value)
        """
        try:
            return self.client.updateGatewayEndpointAnalogAccess(**kwargs)["return"]
        except Fault as e:
            return e

    def getGatewayEndpointAnalogAccess(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value), optional
            subunit: UnionType(value), optional
            endpoint: optional
                index: UnionType(value), optional
                name: UniqueString128(value), optional
                description: String128(value), optional
                product: UnionType(value), optional
                model: UnionType(value), optional
                class: UnionType(value), optional
                protocol: UnionType(value), optional
                protocolSide: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                mlppDomainId: String128(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                retryVideoCallAsAudio: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    portNumber: UnionType(value), optional
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value), optional
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value), optional
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    smdiPortNumber: UnionType(value), optional
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value), optional
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    timer1: UnionType(value), optional
                    timer2: UnionType(value), optional
                    timer3: UnionType(value), optional
                    timer4: UnionType(value), optional
                    timer5: UnionType(value), optional
                    timer6: UnionType(value), optional
                trunkSelectionOrder: UnionType(value), optional
                transmitUtf8: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                callingPartyNumberPrefix: String16(value), optional
                callingPartyStripDigits: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssUnknown: boolean(value), optional
                hotlineDevice: boolean(value), optional
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                imeE164DirectoryNumber: String16(value), optional
                    confidentialAccessMode: UnionType(value), optional
                    confidentialAccessLevel: UnionType(value), optional
                    _value_1: String(value)
        """
        try:
            return self.client.getGatewayEndpointAnalogAccess(**kwargs)["return"]["gatewayEndpointAnalogAccess"]
        except Fault as e:
            return e

    def removeGatewayEndpointAnalogAccess(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGatewayEndpointAnalogAccess(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGatewayEndpointDigitalAccessPri(self, domainName=None, gatewayUuid=None, unit="", subunit="", endpoint=""):
        """
        :param gatewayEndpointDigitalAccessPri: 
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value)
            subunit: UnionType(value)
            endpoint: 
                index: UnionType(value)
                name: UniqueString128(value)
                description: String128(value), optional
                product: UnionType(value)
                class: UnionType(value)
                protocol: UnionType(value)
                protocolSide: UnionType(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                mlppPreemption: UnionType(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                redirectInboundNumberIe: boolean(value)
                calledPlan: UnionType(value)
                calledPri: UnionType(value)
                callerIdDn: String255(value)
                callingPartySelection: UnionType(value)
                callingPlan: UnionType(value)
                callingPri: UnionType(value)
                chanIE: UnionType(value)
                clockReference: UnionType(value)
                dChannelEnable: boolean(value)
                channelSelectionOrder: UnionType(value)
                displayIe: boolean(value)
                pcmType: UnionType(value)
                csuParam: UnionType(value)
                firstDelay: UnionType(value)
                interfaceIdPresent: boolean(value)
                interfaceId: UnionType(value)
                intraDelay: UnionType(value)
                mcdnEnable: boolean(value)
                redirectOutboundNumberIe: boolean(value)
                numDigitsToStrip: UnionType(value)
                passingPrecedenceLevelThrough: boolean(value)
                prefix: String(value)
                callingLinePresentationBit: UnionType(value)
                connectedLineIdPresentation: UnionType(value)
                priProtocol: UnionType(value)
                securityAccessLevel: UnionType(value)
                sendCallingNameInFacilityIe: boolean(value)
                sendExLeadingCharInDispIe: boolean(value)
                sendRestart: boolean(value)
                setupNonIsdnPi: boolean(value)
                    _value_1: NonNegativeInteger(value)
                span: UnionType(value)
                statusPoll: boolean(value)
                smdiBasePort: UnionType(value)
                GClearEnable: boolean(value)
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                transmitUtf8: boolean(value), optional
                v150: boolean(value), optional
                asn1RoseOidEncoding: UnionType(value), optional
                qsigVariant: UnionType(value), optional
                unattendedPort: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                nationalPrefix: String16(value), optional
                internationalPrefix: String16(value), optional
                unknownPrefix: String16(value), optional
                subscriberPrefix: String16(value), optional
                    _value_1: String(value)
                routeClassSignalling: UnionType(value), optional
                nationalStripDigits: UnionType(value), optional
                internationalStripDigits: UnionType(value), optional
                unknownStripDigits: UnionType(value), optional
                subscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssNatl: boolean(value), optional
                useDevicePoolCgpnTransformCssIntl: boolean(value), optional
                useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
                useDevicePoolCgpnTransformCssSubs: boolean(value), optional
                calledPartyNationalPrefix: String16(value), optional
                calledPartyInternationalPrefix: String16(value), optional
                calledPartyUnknownPrefix: String16(value), optional
                calledPartySubscriberPrefix: String16(value), optional
                calledPartyNationalStripDigits: UnionType(value), optional
                calledPartyInternationalStripDigits: UnionType(value), optional
                calledPartyUnknownStripDigits: UnionType(value), optional
                calledPartySubscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                useDevicePoolCalledCssNatl: boolean(value), optional
                useDevicePoolCalledCssIntl: boolean(value), optional
                useDevicePoolCalledCssUnkn: boolean(value), optional
                useDevicePoolCalledCssSubs: boolean(value), optional
                useDevicePoolCntdPartyTransformationCss: boolean(value), optional
                    _value_1: String(value)
                    confidentialAccessMode: UnionType(value)
                    confidentialAccessLevel: UnionType(value)
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addGatewayEndpointDigitalAccessPri({"domainName": domainName, "gatewayUuid": gatewayUuid, "unit": unit, "subunit": subunit, "endpoint": endpoint})["return"]
        except Fault as e:
            return e

    def updateGatewayEndpointDigitalAccessPri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param endpoint: optional
            index: UnionType(value)
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: 
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: 
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: 
                _value_1: String(value)
            automatedAlternateRoutingCssName: 
                _value_1: String(value)
            aarNeighborhoodName: 
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            mlppPreemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: 
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: 
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value)
            calledPlan: UnionType(value)
            calledPri: UnionType(value)
            callerIdDn: String255(value)
            callingPartySelection: UnionType(value)
            callingPlan: UnionType(value)
            callingPri: UnionType(value)
            chanIE: UnionType(value)
            clockReference: UnionType(value)
            dChannelEnable: boolean(value)
            channelSelectionOrder: UnionType(value)
            displayIe: boolean(value)
            pcmType: UnionType(value)
            csuParam: UnionType(value)
            firstDelay: UnionType(value)
            interfaceIdPresent: boolean(value)
            interfaceId: UnionType(value)
            intraDelay: UnionType(value)
            mcdnEnable: boolean(value)
            redirectOutboundNumberIe: boolean(value)
            numDigitsToStrip: UnionType(value)
            passingPrecedenceLevelThrough: boolean(value)
            prefix: String(value)
            callingLinePresentationBit: UnionType(value)
            connectedLineIdPresentation: UnionType(value)
            priProtocol: UnionType(value)
            securityAccessLevel: UnionType(value)
            sendCallingNameInFacilityIe: boolean(value)
            sendExLeadingCharInDispIe: boolean(value)
            sendRestart: boolean(value)
            setupNonIsdnPi: boolean(value)
            sigDigits: 
                _value_1: NonNegativeInteger(value)
            span: UnionType(value)
            statusPoll: boolean(value)
            smdiBasePort: UnionType(value)
            GClearEnable: boolean(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            routeClassSignalling: UnionType(value), optional
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            useDevicePoolCntdPartyTransformationCss: boolean(value), optional
            cntdPartyTransformationCssName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateGatewayEndpointDigitalAccessPri(**kwargs)["return"]
        except Fault as e:
            return e

    def getGatewayEndpointDigitalAccessPri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value), optional
            subunit: UnionType(value), optional
            endpoint: optional
                index: UnionType(value), optional
                name: UniqueString128(value), optional
                description: String128(value), optional
                product: UnionType(value), optional
                model: UnionType(value), optional
                class: UnionType(value), optional
                protocol: UnionType(value), optional
                protocolSide: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                mlppPreemption: UnionType(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                redirectInboundNumberIe: boolean(value), optional
                calledPlan: UnionType(value), optional
                calledPri: UnionType(value), optional
                callerIdDn: String255(value), optional
                callingPartySelection: UnionType(value), optional
                callingPlan: UnionType(value), optional
                callingPri: UnionType(value), optional
                chanIE: UnionType(value), optional
                clockReference: UnionType(value), optional
                dChannelEnable: boolean(value), optional
                channelSelectionOrder: UnionType(value), optional
                displayIe: boolean(value), optional
                pcmType: UnionType(value), optional
                csuParam: UnionType(value), optional
                firstDelay: UnionType(value), optional
                interfaceIdPresent: boolean(value), optional
                interfaceId: UnionType(value), optional
                intraDelay: UnionType(value), optional
                mcdnEnable: boolean(value), optional
                redirectOutboundNumberIe: boolean(value), optional
                numDigitsToStrip: UnionType(value), optional
                passingPrecedenceLevelThrough: boolean(value), optional
                prefix: String(value), optional
                callingLinePresentationBit: UnionType(value), optional
                connectedLineIdPresentation: UnionType(value), optional
                priProtocol: UnionType(value), optional
                securityAccessLevel: UnionType(value), optional
                sendCallingNameInFacilityIe: boolean(value), optional
                sendExLeadingCharInDispIe: boolean(value), optional
                sendRestart: boolean(value), optional
                setupNonIsdnPi: boolean(value), optional
                    _value_1: NonNegativeInteger(value)
                span: UnionType(value), optional
                statusPoll: boolean(value), optional
                smdiBasePort: UnionType(value), optional
                GClearEnable: boolean(value), optional
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                transmitUtf8: boolean(value), optional
                v150: boolean(value), optional
                asn1RoseOidEncoding: UnionType(value), optional
                qsigVariant: UnionType(value), optional
                unattendedPort: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                nationalPrefix: String16(value), optional
                internationalPrefix: String16(value), optional
                unknownPrefix: String16(value), optional
                subscriberPrefix: String16(value), optional
                    _value_1: String(value)
                routeClassSignalling: UnionType(value), optional
                nationalStripDigits: UnionType(value), optional
                internationalStripDigits: UnionType(value), optional
                unknownStripDigits: UnionType(value), optional
                subscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssNatl: boolean(value), optional
                useDevicePoolCgpnTransformCssIntl: boolean(value), optional
                useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
                useDevicePoolCgpnTransformCssSubs: boolean(value), optional
                calledPartyNationalPrefix: String16(value), optional
                calledPartyInternationalPrefix: String16(value), optional
                calledPartyUnknownPrefix: String16(value), optional
                calledPartySubscriberPrefix: String16(value), optional
                calledPartyNationalStripDigits: UnionType(value), optional
                calledPartyInternationalStripDigits: UnionType(value), optional
                calledPartyUnknownStripDigits: UnionType(value), optional
                calledPartySubscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                useDevicePoolCalledCssNatl: boolean(value), optional
                useDevicePoolCalledCssIntl: boolean(value), optional
                useDevicePoolCalledCssUnkn: boolean(value), optional
                useDevicePoolCalledCssSubs: boolean(value), optional
                useDevicePoolCntdPartyTransformationCss: boolean(value), optional
                    _value_1: String(value)
                    confidentialAccessMode: UnionType(value), optional
                    confidentialAccessLevel: UnionType(value), optional
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getGatewayEndpointDigitalAccessPri(**kwargs)["return"]["gatewayEndpointDigitalAccessPri"]
        except Fault as e:
            return e

    def removeGatewayEndpointDigitalAccessPri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGatewayEndpointDigitalAccessPri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGatewayEndpointDigitalAccessBri(self, domainName=None, gatewayUuid=None, unit="", subunit="", endpoint=""):
        """
        :param gatewayEndpointDigitalAccessBri: 
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value)
            subunit: UnionType(value)
            endpoint: 
                index: UnionType(value)
                name: UniqueString128(value)
                description: String128(value), optional
                product: UnionType(value)
                class: UnionType(value)
                protocol: UnionType(value)
                protocolSide: UnionType(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                redirectInboundNumberIe: boolean(value)
                briProtocol: UnionType(value)
                calledPlan: UnionType(value)
                calledPri: UnionType(value)
                callerIdDn: String255(value)
                callingPartySelection: UnionType(value)
                callingPlan: UnionType(value)
                callingPri: UnionType(value)
                clockReference: UnionType(value)
                csuParam: UnionType(value)
                dChannelEnable: boolean(value)
                channelSelectionOrder: UnionType(value)
                pcmType: UnionType(value)
                firstDelay: UnionType(value)
                intraDelay: UnionType(value)
                redirectOutboundNumberIe: boolean(value)
                numDigitsToStrip: UnionType(value)
                prefix: String(value)
                presentationBit: UnionType(value)
                sendRestart: boolean(value)
                setupNonIsdnPi: boolean(value)
                    _value_1: NonNegativeInteger(value)
                statusPoll: boolean(value)
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                nationalPrefix: String16(value), optional
                internationalPrefix: String16(value), optional
                unknownPrefix: String16(value), optional
                subscriberPrefix: String16(value), optional
                nationalStripDigits: UnionType(value), optional
                internationalStripDigits: UnionType(value), optional
                unknownStripDigits: UnionType(value), optional
                subscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssNatl: boolean(value), optional
                useDevicePoolCgpnTransformCssIntl: boolean(value), optional
                useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
                useDevicePoolCgpnTransformCssSubs: boolean(value), optional
                GClearEnable: boolean(value)
                calledPartyNationalPrefix: String16(value), optional
                calledPartyInternationalPrefix: String16(value), optional
                calledPartyUnknownPrefix: String16(value), optional
                calledPartySubscriberPrefix: String16(value), optional
                calledPartyNationalStripDigits: UnionType(value), optional
                calledPartyInternationalStripDigits: UnionType(value), optional
                calledPartyUnknownStripDigits: UnionType(value), optional
                calledPartySubscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                useDevicePoolCalledCssNatl: boolean(value), optional
                useDevicePoolCalledCssIntl: boolean(value), optional
                useDevicePoolCalledCssUnkn: boolean(value), optional
                useDevicePoolCalledCssSubs: boolean(value), optional
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addGatewayEndpointDigitalAccessBri({"domainName": domainName, "gatewayUuid": gatewayUuid, "unit": unit, "subunit": subunit, "endpoint": endpoint})["return"]
        except Fault as e:
            return e

    def updateGatewayEndpointDigitalAccessBri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param endpoint: optional
            index: UnionType(value)
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: 
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: 
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: 
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            automatedAlternateRoutingCssName: 
                _value_1: String(value)
            aarNeighborhoodName: 
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            cgpnTransformationCssName: 
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: 
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value)
            briProtocol: UnionType(value)
            calledPlan: UnionType(value)
            calledPri: UnionType(value)
            callerIdDn: String255(value)
            callingPartySelection: UnionType(value)
            callingPlan: UnionType(value)
            callingPri: UnionType(value)
            clockReference: UnionType(value)
            csuParam: UnionType(value)
            dChannelEnable: boolean(value)
            channelSelectionOrder: UnionType(value)
            pcmType: UnionType(value)
            firstDelay: UnionType(value)
            intraDelay: UnionType(value)
            redirectOutboundNumberIe: boolean(value)
            numDigitsToStrip: UnionType(value)
            prefix: String(value)
            presentationBit: UnionType(value)
            sendRestart: boolean(value)
            setupNonIsdnPi: boolean(value)
            sigDigits: 
                _value_1: NonNegativeInteger(value)
            statusPoll: boolean(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            GClearEnable: boolean(value)
            calledPartyNationalPrefix: String16(value), optional
            calledPartyInternationalPrefix: String16(value), optional
            calledPartyUnknownPrefix: String16(value), optional
            calledPartySubscriberPrefix: String16(value), optional
            calledPartyNationalStripDigits: UnionType(value), optional
            calledPartyInternationalStripDigits: UnionType(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            calledPartySubscriberStripDigits: UnionType(value), optional
            calledPartyNationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyInternationalTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartySubscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCalledCssNatl: boolean(value), optional
            useDevicePoolCalledCssIntl: boolean(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            useDevicePoolCalledCssSubs: boolean(value), optional
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateGatewayEndpointDigitalAccessBri(**kwargs)["return"]
        except Fault as e:
            return e

    def getGatewayEndpointDigitalAccessBri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value), optional
            subunit: UnionType(value), optional
            endpoint: optional
                index: UnionType(value), optional
                name: UniqueString128(value), optional
                description: String128(value), optional
                product: UnionType(value), optional
                model: UnionType(value), optional
                class: UnionType(value), optional
                protocol: UnionType(value), optional
                protocolSide: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocale: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                redirectInboundNumberIe: boolean(value), optional
                briProtocol: UnionType(value), optional
                calledPlan: UnionType(value), optional
                calledPri: UnionType(value), optional
                callerIdDn: String255(value), optional
                callingPartySelection: UnionType(value), optional
                callingPlan: UnionType(value), optional
                callingPri: UnionType(value), optional
                clockReference: UnionType(value), optional
                csuParam: UnionType(value), optional
                dChannelEnable: boolean(value), optional
                channelSelectionOrder: UnionType(value), optional
                pcmType: UnionType(value), optional
                firstDelay: UnionType(value), optional
                intraDelay: UnionType(value), optional
                redirectOutboundNumberIe: boolean(value), optional
                numDigitsToStrip: UnionType(value), optional
                prefix: String(value), optional
                presentationBit: UnionType(value), optional
                sendRestart: boolean(value), optional
                setupNonIsdnPi: boolean(value), optional
                    _value_1: NonNegativeInteger(value)
                statusPoll: boolean(value), optional
                packetCaptureMode: UnionType(value), optional
                packetCaptureDuration: UnionType(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                nationalPrefix: String16(value), optional
                internationalPrefix: String16(value), optional
                unknownPrefix: String16(value), optional
                subscriberPrefix: String16(value), optional
                nationalStripDigits: UnionType(value), optional
                internationalStripDigits: UnionType(value), optional
                unknownStripDigits: UnionType(value), optional
                subscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCssNatl: boolean(value), optional
                useDevicePoolCgpnTransformCssIntl: boolean(value), optional
                useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
                useDevicePoolCgpnTransformCssSubs: boolean(value), optional
                GClearEnable: boolean(value), optional
                calledPartyNationalPrefix: String16(value), optional
                calledPartyInternationalPrefix: String16(value), optional
                calledPartyUnknownPrefix: String16(value), optional
                calledPartySubscriberPrefix: String16(value), optional
                calledPartyNationalStripDigits: UnionType(value), optional
                calledPartyInternationalStripDigits: UnionType(value), optional
                calledPartyUnknownStripDigits: UnionType(value), optional
                calledPartySubscriberStripDigits: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                useDevicePoolCalledCssNatl: boolean(value), optional
                useDevicePoolCalledCssIntl: boolean(value), optional
                useDevicePoolCalledCssUnkn: boolean(value), optional
                useDevicePoolCalledCssSubs: boolean(value), optional
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getGatewayEndpointDigitalAccessBri(**kwargs)["return"]["gatewayEndpointDigitalAccessBri"]
        except Fault as e:
            return e

    def removeGatewayEndpointDigitalAccessBri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGatewayEndpointDigitalAccessBri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addGatewayEndpointDigitalAccessT1(self, domainName=None, gatewayUuid=None, unit="", subunit="", endpoint=""):
        """
        :param gatewayEndpointDigitalAccessT1: 
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value)
            subunit: UnionType(value)
            endpoint: 
                index: UnionType(value)
                name: UniqueString128(value)
                description: String128(value), optional
                product: UnionType(value)
                class: UnionType(value)
                protocol: UnionType(value)
                protocolSide: UnionType(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                traceFlag: boolean(value), optional
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                retryVideoCallAsAudio: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                sendGeoLocation: boolean(value)
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                v150: boolean(value), optional
                    _value_1: String(value)
                trunkSelectionOrder: UnionType(value)
                clockReference: UnionType(value)
                csuParam: UnionType(value)
                digitSending: UnionType(value)
                pcmType: UnionType(value)
                fdlChannel: UnionType(value)
                yellowAlarm: UnionType(value)
                zeroSupression: UnionType(value)
                smdiBasePort: UnionType(value)
                handleDtmfPrecedenceSignals: boolean(value), optional
                encodeOutboundVoiceRouteClass: boolean(value), optional
                routeClassSignalling: UnionType(value)
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                    confidentialAccessMode: UnionType(value)
                    confidentialAccessLevel: UnionType(value)
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.addGatewayEndpointDigitalAccessT1({"domainName": domainName, "gatewayUuid": gatewayUuid, "unit": unit, "subunit": subunit, "endpoint": endpoint})["return"]
        except Fault as e:
            return e

    def updateGatewayEndpointDigitalAccessT1(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param endpoint: optional
            index: UnionType(value), optional
            description: String128(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            v150: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            ports: optional
                    portNumber: UnionType(value), optional
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value), optional
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value), optional
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value), optional
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    callerId: UnionType(value), optional
                    endpointId: String128(value), optional
                    timer1: UnionType(value), optional
                    timer2: UnionType(value), optional
                    timer3: UnionType(value), optional
                    timer4: UnionType(value), optional
                    timer5: UnionType(value), optional
                    timer6: UnionType(value), optional
            trunkSelectionOrder: UnionType(value), optional
            clockReference: UnionType(value), optional
            csuParam: UnionType(value), optional
            digitSending: UnionType(value), optional
            pcmType: UnionType(value), optional
            fdlChannel: UnionType(value), optional
            yellowAlarm: UnionType(value), optional
            zeroSupression: UnionType(value), optional
            smdiBasePort: UnionType(value), optional
            handleDtmfPrecedenceSignals: boolean(value), optional
            encodeOutboundVoiceRouteClass: boolean(value), optional
            routeClassSignalling: UnionType(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
            connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.updateGatewayEndpointDigitalAccessT1(**kwargs)["return"]
        except Fault as e:
            return e

    def getGatewayEndpointDigitalAccessT1(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            domainName: String(value), optional
            gatewayUuid: XUUID(value), optional
            unit: UnionType(value), optional
            subunit: UnionType(value), optional
            endpoint: optional
                index: UnionType(value), optional
                name: UniqueString128(value), optional
                description: String128(value), optional
                product: UnionType(value), optional
                model: UnionType(value), optional
                class: UnionType(value), optional
                protocol: UnionType(value), optional
                protocolSide: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                networkLocation: UnionType(value), optional
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: String(value)
                    _value_1: AnyType, optional
                traceFlag: boolean(value), optional
                mlppDomainId: String128(value), optional
                mlppIndicationStatus: UnionType(value), optional
                preemption: UnionType(value), optional
                useTrustedRelayPoint: UnionType(value), optional
                retryVideoCallAsAudio: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCgpnTransformCss: boolean(value), optional
                    _value_1: String(value)
                sendGeoLocation: boolean(value), optional
                    _value_1: String(value)
                useDevicePoolCdpnTransformCss: boolean(value), optional
                v150: boolean(value), optional
                    _value_1: String(value)
                trunkSelectionOrder: UnionType(value), optional
                clockReference: UnionType(value), optional
                csuParam: UnionType(value), optional
                digitSending: UnionType(value), optional
                pcmType: UnionType(value), optional
                fdlChannel: UnionType(value), optional
                yellowAlarm: UnionType(value), optional
                zeroSupression: UnionType(value), optional
                smdiBasePort: UnionType(value), optional
                handleDtmfPrecedenceSignals: boolean(value), optional
                encodeOutboundVoiceRouteClass: boolean(value), optional
                routeClassSignalling: UnionType(value), optional
                pstnAccess: boolean(value), optional
                    _value_1: String(value)
                    confidentialAccessMode: UnionType(value), optional
                    confidentialAccessLevel: UnionType(value), optional
                connectCallBeforePlayingAnnouncement: boolean(value), optional
        """
        try:
            return self.client.getGatewayEndpointDigitalAccessT1(**kwargs)["return"]["gatewayEndpointDigitalAccessT1"]
        except Fault as e:
            return e

    def removeGatewayEndpointDigitalAccessT1(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGatewayEndpointDigitalAccessT1(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCiscoCatalyst600024PortFXSGateway(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocale=None, locationName="", mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, vendorConfig=None, traceFlag=None, mlppDomainId=None, useTrustedRelayPoint=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, ports=None, portSelectionOrder="", transmitUtf8=None, geoLocationFilterName=None):
        """
        :param ciscoCatalyst600024PortFXSGateway: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            ports: optional
                    portNumber: UnionType(value)
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value)
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value)
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    smdiPortNumber: UnionType(value)
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value)
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    timer1: UnionType(value)
                    timer2: UnionType(value)
                    timer3: UnionType(value)
                    timer4: UnionType(value)
                    timer5: UnionType(value)
                    timer6: UnionType(value)
            portSelectionOrder: UnionType(value)
            transmitUtf8: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addCiscoCatalyst600024PortFXSGateway({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocale": networkLocale, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "useTrustedRelayPoint": useTrustedRelayPoint, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "ports": ports, "portSelectionOrder": portSelectionOrder, "transmitUtf8": transmitUtf8, "geoLocationFilterName": geoLocationFilterName})["return"]
        except Fault as e:
            return e

    def updateCiscoCatalyst600024PortFXSGateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocale: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param ports: optional
            port: 
                portNumber: UnionType(value)
                attendantDn: String255(value), optional
                unattendedPort: boolean(value), optional
                callerIdDn: String255(value), optional
                callerIdEnable: boolean(value)
                callingPartySelection: UnionType(value), optional
                digitSending: UnionType(value), optional
                expectedDigits: UnionType(value)
                    _value_1: UnionType(value)
                prefixDn: String255(value), optional
                presentationBit: UnionType(value), optional
                silenceSuppressionThreshold: UnionType(value), optional
                smdiPortNumber: UnionType(value)
                startDialProtocol: UnionType(value), optional
                trunk: UnionType(value), optional
                trunkDirection: UnionType(value)
                trunkLevel: UnionType(value), optional
                trunkPadRx: UnionType(value), optional
                trunkPadTx: UnionType(value), optional
                    _value_1: AnyType, optional
                timer1: UnionType(value)
                timer2: UnionType(value)
                timer3: UnionType(value)
                timer4: UnionType(value)
                timer5: UnionType(value)
                timer6: UnionType(value)
        :param portSelectionOrder: UnionType(value), optional
        :param transmitUtf8: boolean(value), optional
        :param geoLocationFilterName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCiscoCatalyst600024PortFXSGateway(**kwargs)["return"]
        except Fault as e:
            return e

    def getCiscoCatalyst600024PortFXSGateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            versionStamp: String128(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            ports: optional
                    portNumber: UnionType(value), optional
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value), optional
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value), optional
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    smdiPortNumber: UnionType(value), optional
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value), optional
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    timer1: UnionType(value), optional
                    timer2: UnionType(value), optional
                    timer3: UnionType(value), optional
                    timer4: UnionType(value), optional
                    timer5: UnionType(value), optional
                    timer6: UnionType(value), optional
            portSelectionOrder: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCiscoCatalyst600024PortFXSGateway(**kwargs)["return"]["ciscoCatalyst600024PortFXSGateway"]
        except Fault as e:
            return e

    def removeCiscoCatalyst600024PortFXSGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCiscoCatalyst600024PortFXSGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCiscoCatalyst600024PortFXSGateway(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocale': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'traceFlag': '', 'useTrustedRelayPoint': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'portSelectionOrder': '', 'transmitUtf8': '', 'geoLocationFilterName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            portSelectionOrder: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCiscoCatalyst600024PortFXSGateway({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ciscoCatalyst600024PortFXSGateway"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCiscoCatalyst600024PortFXSGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCiscoCatalyst600024PortFXSGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCiscoCatalyst600024PortFXSGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCiscoCatalyst600024PortFXSGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCiscoCatalyst600024PortFXSGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCiscoCatalyst600024PortFXSGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCiscoCatalyst6000E1VoIPGateway(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", networkLocale=None, mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, vendorConfig=None, mlppDomainId=None, useTrustedRelayPoint=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, redirectInboundNumberIe="", calledPlan="", calledPri="", callerIdDn=None, callingPartySelection="", callingPlan="", callingPri="", chanIe="", clockReference="", dChannelEnable="", channelSelectionOrder="", displayIE="", pcmType="", csuParam="", firstDelay="", interfaceIdPresent="", interfaceId="", intraDelay="", mcdnEnable="", redirectOutboundNumberIe="", numDigitsToStrip="", passingPrecedenceLevelThrough="", prefix=None, callingLinePresentationBit="", connectedLineIdPresentation="", priProtocol="", securityAccessLevel="", sendCallingNameInFacilityIe="", sendExLeadingCharInDispIe="", sendRestart="", setupNonIsdnPi="", sigDigits="", span="", statusPoll="", smdiBasePort="", packetCaptureMode=None, packetCaptureDuration=None, transmitUtf8=None, v150=None, asn1RoseOidEncoding=None, QSIGVariant=None, unattendedPort=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, nationalPrefix=None, internationalPrefix=None, unknownPrefix=None, subscriberPrefix=None, geoLocationFilterName=None, nationalStripDigits=None, internationalStripDigits=None, unknownStripDigits=None, subscriberStripDigits=None, nationalTransformationCssName=None, internationalTransformationCssName=None, unknownTransformationCssName=None, subscriberTransformationCssName=None, useDevicePoolCgpnTransformCssNatl=None, useDevicePoolCgpnTransformCssIntl=None, useDevicePoolCgpnTransformCssUnkn=None, useDevicePoolCgpnTransformCssSubs=None, pstnAccess=None, imeE164TransformationName=None):
        """
        :param ciscoCatalyst6000E1VoIPGateway: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value)
            calledPlan: UnionType(value)
            calledPri: UnionType(value)
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value)
            callingPlan: UnionType(value)
            callingPri: UnionType(value)
            chanIe: UnionType(value)
            clockReference: UnionType(value)
            dChannelEnable: boolean(value)
            channelSelectionOrder: UnionType(value)
            displayIE: boolean(value)
            pcmType: UnionType(value)
            csuParam: UnionType(value)
            firstDelay: UnionType(value)
            interfaceIdPresent: boolean(value)
            interfaceId: UnionType(value)
            intraDelay: UnionType(value)
            mcdnEnable: boolean(value)
            redirectOutboundNumberIe: boolean(value)
            numDigitsToStrip: UnionType(value)
            passingPrecedenceLevelThrough: boolean(value)
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value)
            connectedLineIdPresentation: UnionType(value)
            priProtocol: UnionType(value)
            securityAccessLevel: UnionType(value)
            sendCallingNameInFacilityIe: boolean(value)
            sendExLeadingCharInDispIe: boolean(value)
            sendRestart: boolean(value)
            setupNonIsdnPi: boolean(value)
            sigDigits: 
                _value_1: UnionType(value)
            span: UnionType(value)
            statusPoll: boolean(value)
            smdiBasePort: UnionType(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addCiscoCatalyst6000E1VoIPGateway({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "networkLocale": networkLocale, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "mlppDomainId": mlppDomainId, "useTrustedRelayPoint": useTrustedRelayPoint, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "redirectInboundNumberIe": redirectInboundNumberIe, "calledPlan": calledPlan, "calledPri": calledPri, "callerIdDn": callerIdDn, "callingPartySelection": callingPartySelection, "callingPlan": callingPlan, "callingPri": callingPri, "chanIe": chanIe, "clockReference": clockReference, "dChannelEnable": dChannelEnable, "channelSelectionOrder": channelSelectionOrder, "displayIE": displayIE, "pcmType": pcmType, "csuParam": csuParam, "firstDelay": firstDelay, "interfaceIdPresent": interfaceIdPresent, "interfaceId": interfaceId, "intraDelay": intraDelay, "mcdnEnable": mcdnEnable, "redirectOutboundNumberIe": redirectOutboundNumberIe, "numDigitsToStrip": numDigitsToStrip, "passingPrecedenceLevelThrough": passingPrecedenceLevelThrough, "prefix": prefix, "callingLinePresentationBit": callingLinePresentationBit, "connectedLineIdPresentation": connectedLineIdPresentation, "priProtocol": priProtocol, "securityAccessLevel": securityAccessLevel, "sendCallingNameInFacilityIe": sendCallingNameInFacilityIe, "sendExLeadingCharInDispIe": sendExLeadingCharInDispIe, "sendRestart": sendRestart, "setupNonIsdnPi": setupNonIsdnPi, "sigDigits": sigDigits, "span": span, "statusPoll": statusPoll, "smdiBasePort": smdiBasePort, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "transmitUtf8": transmitUtf8, "v150": v150, "asn1RoseOidEncoding": asn1RoseOidEncoding, "QSIGVariant": QSIGVariant, "unattendedPort": unattendedPort, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "nationalPrefix": nationalPrefix, "internationalPrefix": internationalPrefix, "unknownPrefix": unknownPrefix, "subscriberPrefix": subscriberPrefix, "geoLocationFilterName": geoLocationFilterName, "nationalStripDigits": nationalStripDigits, "internationalStripDigits": internationalStripDigits, "unknownStripDigits": unknownStripDigits, "subscriberStripDigits": subscriberStripDigits, "nationalTransformationCssName": nationalTransformationCssName, "internationalTransformationCssName": internationalTransformationCssName, "unknownTransformationCssName": unknownTransformationCssName, "subscriberTransformationCssName": subscriberTransformationCssName, "useDevicePoolCgpnTransformCssNatl": useDevicePoolCgpnTransformCssNatl, "useDevicePoolCgpnTransformCssIntl": useDevicePoolCgpnTransformCssIntl, "useDevicePoolCgpnTransformCssUnkn": useDevicePoolCgpnTransformCssUnkn, "useDevicePoolCgpnTransformCssSubs": useDevicePoolCgpnTransformCssSubs, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName})["return"]
        except Fault as e:
            return e

    def updateCiscoCatalyst6000E1VoIPGateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param networkLocale: UnionType(value), optional
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param mlppDomainId: String128(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param redirectInboundNumberIe: boolean(value), optional
        :param calledPlan: UnionType(value), optional
        :param calledPri: UnionType(value), optional
        :param callerIdDn: String255(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingPlan: UnionType(value), optional
        :param callingPri: UnionType(value), optional
        :param chanIe: UnionType(value), optional
        :param clockReference: UnionType(value), optional
        :param dChannelEnable: boolean(value), optional
        :param channelSelectionOrder: UnionType(value), optional
        :param displayIE: boolean(value), optional
        :param pcmType: UnionType(value), optional
        :param csuParam: UnionType(value), optional
        :param firstDelay: UnionType(value), optional
        :param interfaceIdPresent: boolean(value), optional
        :param interfaceId: UnionType(value), optional
        :param intraDelay: UnionType(value), optional
        :param mcdnEnable: boolean(value), optional
        :param redirectOutboundNumberIe: boolean(value), optional
        :param numDigitsToStrip: UnionType(value), optional
        :param passingPrecedenceLevelThrough: boolean(value), optional
        :param prefix: String(value), optional
        :param callingLinePresentationBit: UnionType(value), optional
        :param connectedLineIdPresentation: UnionType(value), optional
        :param priProtocol: UnionType(value), optional
        :param securityAccessLevel: UnionType(value), optional
        :param sendCallingNameInFacilityIe: boolean(value), optional
        :param sendExLeadingCharInDispIe: boolean(value), optional
        :param sendRestart: boolean(value), optional
        :param setupNonIsdnPi: boolean(value), optional
        :param sigDigits: optional
            _value_1: UnionType(value)
        :param span: UnionType(value), optional
        :param statusPoll: boolean(value), optional
        :param smdiBasePort: UnionType(value), optional
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param transmitUtf8: boolean(value), optional
        :param v150: boolean(value), optional
        :param asn1RoseOidEncoding: UnionType(value), optional
        :param QSIGVariant: UnionType(value), optional
        :param unattendedPort: boolean(value), optional
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param nationalPrefix: String16(value), optional
        :param internationalPrefix: String16(value), optional
        :param unknownPrefix: String16(value), optional
        :param subscriberPrefix: String16(value), optional
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param nationalStripDigits: UnionType(value), optional
        :param internationalStripDigits: UnionType(value), optional
        :param unknownStripDigits: UnionType(value), optional
        :param subscriberStripDigits: UnionType(value), optional
        :param nationalTransformationCssName: optional
            _value_1: String(value)
        :param internationalTransformationCssName: optional
            _value_1: String(value)
        :param unknownTransformationCssName: optional
            _value_1: String(value)
        :param subscriberTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCssNatl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssIntl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
        :param useDevicePoolCgpnTransformCssSubs: boolean(value), optional
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCiscoCatalyst6000E1VoIPGateway(**kwargs)["return"]
        except Fault as e:
            return e

    def getCiscoCatalyst6000E1VoIPGateway(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value), optional
            calledPlan: UnionType(value), optional
            calledPri: UnionType(value), optional
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value), optional
            callingPlan: UnionType(value), optional
            callingPri: UnionType(value), optional
            chanIe: UnionType(value), optional
            clockReference: UnionType(value), optional
            dChannelEnable: boolean(value), optional
            channelSelectionOrder: UnionType(value), optional
            displayIE: boolean(value), optional
            pcmType: UnionType(value), optional
            csuParam: UnionType(value), optional
            firstDelay: UnionType(value), optional
            interfaceIdPresent: boolean(value), optional
            interfaceId: UnionType(value), optional
            intraDelay: UnionType(value), optional
            mcdnEnable: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            numDigitsToStrip: UnionType(value), optional
            passingPrecedenceLevelThrough: boolean(value), optional
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value), optional
            connectedLineIdPresentation: UnionType(value), optional
            priProtocol: UnionType(value), optional
            securityAccessLevel: UnionType(value), optional
            sendCallingNameInFacilityIe: boolean(value), optional
            sendExLeadingCharInDispIe: boolean(value), optional
            sendRestart: boolean(value), optional
            setupNonIsdnPi: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            span: UnionType(value), optional
            statusPoll: boolean(value), optional
            smdiBasePort: UnionType(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCiscoCatalyst6000E1VoIPGateway(**kwargs)["return"]["ciscoCatalyst6000E1VoIPGateway"]
        except Fault as e:
            return e

    def removeCiscoCatalyst6000E1VoIPGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCiscoCatalyst6000E1VoIPGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCiscoCatalyst6000E1VoIPGateway(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'networkLocale': '', 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'useTrustedRelayPoint': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'redirectInboundNumberIe': '', 'calledPlan': '', 'calledPri': '', 'callerIdDn': '', 'callingPartySelection': '', 'callingPlan': '', 'callingPri': '', 'chanIe': '', 'clockReference': '', 'dChannelEnable': '', 'channelSelectionOrder': '', 'displayIE': '', 'pcmType': '', 'csuParam': '', 'firstDelay': '', 'interfaceIdPresent': '', 'interfaceId': '', 'intraDelay': '', 'mcdnEnable': '', 'redirectOutboundNumberIe': '', 'numDigitsToStrip': '', 'passingPrecedenceLevelThrough': '', 'prefix': '', 'callingLinePresentationBit': '', 'connectedLineIdPresentation': '', 'priProtocol': '', 'securityAccessLevel': '', 'sendCallingNameInFacilityIe': '', 'sendExLeadingCharInDispIe': '', 'sendRestart': '', 'setupNonIsdnPi': '', 'sigDigits': {'_value_1': ''}, 'span': '', 'statusPoll': '', 'smdiBasePort': '', 'packetCaptureMode': '', 'packetCaptureDuration': '', 'transmitUtf8': '', 'v150': '', 'asn1RoseOidEncoding': '', 'QSIGVariant': '', 'unattendedPort': '', 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'nationalPrefix': '', 'internationalPrefix': '', 'unknownPrefix': '', 'subscriberPrefix': '', 'geoLocationFilterName': {'_value_1': ''}, 'nationalStripDigits': '', 'internationalStripDigits': '', 'unknownStripDigits': '', 'subscriberStripDigits': '', 'nationalTransformationCssName': {'_value_1': ''}, 'internationalTransformationCssName': {'_value_1': ''}, 'unknownTransformationCssName': {'_value_1': ''}, 'subscriberTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCssNatl': '', 'useDevicePoolCgpnTransformCssIntl': '', 'useDevicePoolCgpnTransformCssUnkn': '', 'useDevicePoolCgpnTransformCssSubs': '', 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value), optional
            calledPlan: UnionType(value), optional
            calledPri: UnionType(value), optional
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value), optional
            callingPlan: UnionType(value), optional
            callingPri: UnionType(value), optional
            chanIe: UnionType(value), optional
            clockReference: UnionType(value), optional
            dChannelEnable: boolean(value), optional
            channelSelectionOrder: UnionType(value), optional
            displayIE: boolean(value), optional
            pcmType: UnionType(value), optional
            csuParam: UnionType(value), optional
            firstDelay: UnionType(value), optional
            interfaceIdPresent: boolean(value), optional
            interfaceId: UnionType(value), optional
            intraDelay: UnionType(value), optional
            mcdnEnable: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            numDigitsToStrip: UnionType(value), optional
            passingPrecedenceLevelThrough: boolean(value), optional
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value), optional
            connectedLineIdPresentation: UnionType(value), optional
            priProtocol: UnionType(value), optional
            securityAccessLevel: UnionType(value), optional
            sendCallingNameInFacilityIe: boolean(value), optional
            sendExLeadingCharInDispIe: boolean(value), optional
            sendRestart: boolean(value), optional
            setupNonIsdnPi: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            span: UnionType(value), optional
            statusPoll: boolean(value), optional
            smdiBasePort: UnionType(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCiscoCatalyst6000E1VoIPGateway({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ciscoCatalyst6000E1VoIPGateway"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCiscoCatalyst6000E1VoIPGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCiscoCatalyst6000E1VoIPGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCiscoCatalyst6000E1VoIPGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCiscoCatalyst6000E1VoIPGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCiscoCatalyst6000E1VoIPGateway(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCiscoCatalyst6000E1VoIPGateway(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCiscoCatalyst6000T1VoIPGatewayPri(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", networkLocale=None, mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, vendorConfig=None, mlppDomainId=None, mlppIndicationStatus=None, mlppPreemption=None, useTrustedRelayPoint=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, redirectInboundNumberIe="", calledPlan="", calledPri="", callerIdDn=None, callingPartySelection="", callingPlan="", callingPri="", chanIe="", clockReference="", dChannelEnable="", channelSelectionOrder="", displayIE="", pcmType="", csuParam="", firstDelay="", interfaceIdPresent="", interfaceId="", intraDelay="", mcdnEnable="", redirectOutboundNumberIe="", numDigitsToStrip="", passingPrecedenceLevelThrough="", prefix=None, callingLinePresentationBit="", connectedLineIdPresentation="", priProtocol="", securityAccessLevel="", sendCallingNameInFacilityIe="", sendExLeadingCharInDispIe="", sendRestart="", setupNonIsdnPi="", sigDigits="", span="", statusPoll="", smdiBasePort="", packetCaptureMode=None, packetCaptureDuration=None, transmitUtf8=None, v150=None, asn1RoseOidEncoding=None, QSIGVariant=None, unattendedPort=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, nationalPrefix=None, internationalPrefix=None, unknownPrefix=None, subscriberPrefix=None, geoLocationFilterName=None, nationalStripDigits=None, internationalStripDigits=None, unknownStripDigits=None, subscriberStripDigits=None, nationalTransformationCssName=None, internationalTransformationCssName=None, unknownTransformationCssName=None, subscriberTransformationCssName=None, useDevicePoolCgpnTransformCssNatl=None, useDevicePoolCgpnTransformCssIntl=None, useDevicePoolCgpnTransformCssUnkn=None, useDevicePoolCgpnTransformCssSubs=None, pstnAccess=None, imeE164TransformationName=None):
        """
        :param ciscoCatalyst6000T1VoIPGatewayPri: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            mlppPreemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value)
            calledPlan: UnionType(value)
            calledPri: UnionType(value)
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value)
            callingPlan: UnionType(value)
            callingPri: UnionType(value)
            chanIe: UnionType(value)
            clockReference: UnionType(value)
            dChannelEnable: boolean(value)
            channelSelectionOrder: UnionType(value)
            displayIE: boolean(value)
            pcmType: UnionType(value)
            csuParam: UnionType(value)
            firstDelay: UnionType(value)
            interfaceIdPresent: boolean(value)
            interfaceId: UnionType(value)
            intraDelay: UnionType(value)
            mcdnEnable: boolean(value)
            redirectOutboundNumberIe: boolean(value)
            numDigitsToStrip: UnionType(value)
            passingPrecedenceLevelThrough: boolean(value)
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value)
            connectedLineIdPresentation: UnionType(value)
            priProtocol: UnionType(value)
            securityAccessLevel: UnionType(value)
            sendCallingNameInFacilityIe: boolean(value)
            sendExLeadingCharInDispIe: boolean(value)
            sendRestart: boolean(value)
            setupNonIsdnPi: boolean(value)
            sigDigits: 
                _value_1: UnionType(value)
            span: UnionType(value)
            statusPoll: boolean(value)
            smdiBasePort: UnionType(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addCiscoCatalyst6000T1VoIPGatewayPri({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "networkLocale": networkLocale, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "mlppPreemption": mlppPreemption, "useTrustedRelayPoint": useTrustedRelayPoint, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "redirectInboundNumberIe": redirectInboundNumberIe, "calledPlan": calledPlan, "calledPri": calledPri, "callerIdDn": callerIdDn, "callingPartySelection": callingPartySelection, "callingPlan": callingPlan, "callingPri": callingPri, "chanIe": chanIe, "clockReference": clockReference, "dChannelEnable": dChannelEnable, "channelSelectionOrder": channelSelectionOrder, "displayIE": displayIE, "pcmType": pcmType, "csuParam": csuParam, "firstDelay": firstDelay, "interfaceIdPresent": interfaceIdPresent, "interfaceId": interfaceId, "intraDelay": intraDelay, "mcdnEnable": mcdnEnable, "redirectOutboundNumberIe": redirectOutboundNumberIe, "numDigitsToStrip": numDigitsToStrip, "passingPrecedenceLevelThrough": passingPrecedenceLevelThrough, "prefix": prefix, "callingLinePresentationBit": callingLinePresentationBit, "connectedLineIdPresentation": connectedLineIdPresentation, "priProtocol": priProtocol, "securityAccessLevel": securityAccessLevel, "sendCallingNameInFacilityIe": sendCallingNameInFacilityIe, "sendExLeadingCharInDispIe": sendExLeadingCharInDispIe, "sendRestart": sendRestart, "setupNonIsdnPi": setupNonIsdnPi, "sigDigits": sigDigits, "span": span, "statusPoll": statusPoll, "smdiBasePort": smdiBasePort, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "transmitUtf8": transmitUtf8, "v150": v150, "asn1RoseOidEncoding": asn1RoseOidEncoding, "QSIGVariant": QSIGVariant, "unattendedPort": unattendedPort, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "nationalPrefix": nationalPrefix, "internationalPrefix": internationalPrefix, "unknownPrefix": unknownPrefix, "subscriberPrefix": subscriberPrefix, "geoLocationFilterName": geoLocationFilterName, "nationalStripDigits": nationalStripDigits, "internationalStripDigits": internationalStripDigits, "unknownStripDigits": unknownStripDigits, "subscriberStripDigits": subscriberStripDigits, "nationalTransformationCssName": nationalTransformationCssName, "internationalTransformationCssName": internationalTransformationCssName, "unknownTransformationCssName": unknownTransformationCssName, "subscriberTransformationCssName": subscriberTransformationCssName, "useDevicePoolCgpnTransformCssNatl": useDevicePoolCgpnTransformCssNatl, "useDevicePoolCgpnTransformCssIntl": useDevicePoolCgpnTransformCssIntl, "useDevicePoolCgpnTransformCssUnkn": useDevicePoolCgpnTransformCssUnkn, "useDevicePoolCgpnTransformCssSubs": useDevicePoolCgpnTransformCssSubs, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName})["return"]
        except Fault as e:
            return e

    def updateCiscoCatalyst6000T1VoIPGatewayPri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param networkLocale: UnionType(value), optional
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param mlppPreemption: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param redirectInboundNumberIe: boolean(value), optional
        :param calledPlan: UnionType(value), optional
        :param calledPri: UnionType(value), optional
        :param callerIdDn: String255(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingPlan: UnionType(value), optional
        :param callingPri: UnionType(value), optional
        :param chanIe: UnionType(value), optional
        :param clockReference: UnionType(value), optional
        :param dChannelEnable: boolean(value), optional
        :param channelSelectionOrder: UnionType(value), optional
        :param displayIE: boolean(value), optional
        :param pcmType: UnionType(value), optional
        :param csuParam: UnionType(value), optional
        :param firstDelay: UnionType(value), optional
        :param interfaceIdPresent: boolean(value), optional
        :param interfaceId: UnionType(value), optional
        :param intraDelay: UnionType(value), optional
        :param mcdnEnable: boolean(value), optional
        :param redirectOutboundNumberIe: boolean(value), optional
        :param numDigitsToStrip: UnionType(value), optional
        :param passingPrecedenceLevelThrough: boolean(value), optional
        :param prefix: String(value), optional
        :param callingLinePresentationBit: UnionType(value), optional
        :param connectedLineIdPresentation: UnionType(value), optional
        :param priProtocol: UnionType(value), optional
        :param securityAccessLevel: UnionType(value), optional
        :param sendCallingNameInFacilityIe: boolean(value), optional
        :param sendExLeadingCharInDispIe: boolean(value), optional
        :param sendRestart: boolean(value), optional
        :param setupNonIsdnPi: boolean(value), optional
        :param sigDigits: optional
            _value_1: UnionType(value)
        :param span: UnionType(value), optional
        :param statusPoll: boolean(value), optional
        :param smdiBasePort: UnionType(value), optional
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param transmitUtf8: boolean(value), optional
        :param v150: boolean(value), optional
        :param asn1RoseOidEncoding: UnionType(value), optional
        :param QSIGVariant: UnionType(value), optional
        :param unattendedPort: boolean(value), optional
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param nationalPrefix: String16(value), optional
        :param internationalPrefix: String16(value), optional
        :param unknownPrefix: String16(value), optional
        :param subscriberPrefix: String16(value), optional
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param nationalStripDigits: UnionType(value), optional
        :param internationalStripDigits: UnionType(value), optional
        :param unknownStripDigits: UnionType(value), optional
        :param subscriberStripDigits: UnionType(value), optional
        :param nationalTransformationCssName: optional
            _value_1: String(value)
        :param internationalTransformationCssName: optional
            _value_1: String(value)
        :param unknownTransformationCssName: optional
            _value_1: String(value)
        :param subscriberTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCssNatl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssIntl: boolean(value), optional
        :param useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
        :param useDevicePoolCgpnTransformCssSubs: boolean(value), optional
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCiscoCatalyst6000T1VoIPGatewayPri(**kwargs)["return"]
        except Fault as e:
            return e

    def getCiscoCatalyst6000T1VoIPGatewayPri(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            mlppPreemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value), optional
            calledPlan: UnionType(value), optional
            calledPri: UnionType(value), optional
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value), optional
            callingPlan: UnionType(value), optional
            callingPri: UnionType(value), optional
            chanIe: UnionType(value), optional
            clockReference: UnionType(value), optional
            dChannelEnable: boolean(value), optional
            channelSelectionOrder: UnionType(value), optional
            displayIE: boolean(value), optional
            pcmType: UnionType(value), optional
            csuParam: UnionType(value), optional
            firstDelay: UnionType(value), optional
            interfaceIdPresent: boolean(value), optional
            interfaceId: UnionType(value), optional
            intraDelay: UnionType(value), optional
            mcdnEnable: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            numDigitsToStrip: UnionType(value), optional
            passingPrecedenceLevelThrough: boolean(value), optional
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value), optional
            connectedLineIdPresentation: UnionType(value), optional
            priProtocol: UnionType(value), optional
            securityAccessLevel: UnionType(value), optional
            sendCallingNameInFacilityIe: boolean(value), optional
            sendExLeadingCharInDispIe: boolean(value), optional
            sendRestart: boolean(value), optional
            setupNonIsdnPi: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            span: UnionType(value), optional
            statusPoll: boolean(value), optional
            smdiBasePort: UnionType(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCiscoCatalyst6000T1VoIPGatewayPri(**kwargs)["return"]["ciscoCatalyst6000T1VoIPGatewayPri"]
        except Fault as e:
            return e

    def removeCiscoCatalyst6000T1VoIPGatewayPri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCiscoCatalyst6000T1VoIPGatewayPri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCiscoCatalyst6000T1VoIPGatewayPri(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'networkLocale': '', 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'mlppIndicationStatus': '', 'mlppPreemption': '', 'useTrustedRelayPoint': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'redirectInboundNumberIe': '', 'calledPlan': '', 'calledPri': '', 'callerIdDn': '', 'callingPartySelection': '', 'callingPlan': '', 'callingPri': '', 'chanIe': '', 'clockReference': '', 'dChannelEnable': '', 'channelSelectionOrder': '', 'displayIE': '', 'pcmType': '', 'csuParam': '', 'firstDelay': '', 'interfaceIdPresent': '', 'interfaceId': '', 'intraDelay': '', 'mcdnEnable': '', 'redirectOutboundNumberIe': '', 'numDigitsToStrip': '', 'passingPrecedenceLevelThrough': '', 'prefix': '', 'callingLinePresentationBit': '', 'connectedLineIdPresentation': '', 'priProtocol': '', 'securityAccessLevel': '', 'sendCallingNameInFacilityIe': '', 'sendExLeadingCharInDispIe': '', 'sendRestart': '', 'setupNonIsdnPi': '', 'sigDigits': {'_value_1': ''}, 'span': '', 'statusPoll': '', 'smdiBasePort': '', 'packetCaptureMode': '', 'packetCaptureDuration': '', 'transmitUtf8': '', 'v150': '', 'asn1RoseOidEncoding': '', 'QSIGVariant': '', 'unattendedPort': '', 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'nationalPrefix': '', 'internationalPrefix': '', 'unknownPrefix': '', 'subscriberPrefix': '', 'geoLocationFilterName': {'_value_1': ''}, 'nationalStripDigits': '', 'internationalStripDigits': '', 'unknownStripDigits': '', 'subscriberStripDigits': '', 'nationalTransformationCssName': {'_value_1': ''}, 'internationalTransformationCssName': {'_value_1': ''}, 'unknownTransformationCssName': {'_value_1': ''}, 'subscriberTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCssNatl': '', 'useDevicePoolCgpnTransformCssIntl': '', 'useDevicePoolCgpnTransformCssUnkn': '', 'useDevicePoolCgpnTransformCssSubs': '', 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            networkLocale: UnionType(value), optional
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            mlppIndicationStatus: UnionType(value), optional
            mlppPreemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            redirectInboundNumberIe: boolean(value), optional
            calledPlan: UnionType(value), optional
            calledPri: UnionType(value), optional
            callerIdDn: String255(value), optional
            callingPartySelection: UnionType(value), optional
            callingPlan: UnionType(value), optional
            callingPri: UnionType(value), optional
            chanIe: UnionType(value), optional
            clockReference: UnionType(value), optional
            dChannelEnable: boolean(value), optional
            channelSelectionOrder: UnionType(value), optional
            displayIE: boolean(value), optional
            pcmType: UnionType(value), optional
            csuParam: UnionType(value), optional
            firstDelay: UnionType(value), optional
            interfaceIdPresent: boolean(value), optional
            interfaceId: UnionType(value), optional
            intraDelay: UnionType(value), optional
            mcdnEnable: boolean(value), optional
            redirectOutboundNumberIe: boolean(value), optional
            numDigitsToStrip: UnionType(value), optional
            passingPrecedenceLevelThrough: boolean(value), optional
            prefix: String(value), optional
            callingLinePresentationBit: UnionType(value), optional
            connectedLineIdPresentation: UnionType(value), optional
            priProtocol: UnionType(value), optional
            securityAccessLevel: UnionType(value), optional
            sendCallingNameInFacilityIe: boolean(value), optional
            sendExLeadingCharInDispIe: boolean(value), optional
            sendRestart: boolean(value), optional
            setupNonIsdnPi: boolean(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            span: UnionType(value), optional
            statusPoll: boolean(value), optional
            smdiBasePort: UnionType(value), optional
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            transmitUtf8: boolean(value), optional
            v150: boolean(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            QSIGVariant: UnionType(value), optional
            unattendedPort: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            nationalPrefix: String16(value), optional
            internationalPrefix: String16(value), optional
            unknownPrefix: String16(value), optional
            subscriberPrefix: String16(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            nationalStripDigits: UnionType(value), optional
            internationalStripDigits: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            subscriberStripDigits: UnionType(value), optional
            nationalTransformationCssName: optional
                _value_1: String(value)
            internationalTransformationCssName: optional
                _value_1: String(value)
            unknownTransformationCssName: optional
                _value_1: String(value)
            subscriberTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssNatl: boolean(value), optional
            useDevicePoolCgpnTransformCssIntl: boolean(value), optional
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            useDevicePoolCgpnTransformCssSubs: boolean(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCiscoCatalyst6000T1VoIPGatewayPri({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ciscoCatalyst6000T1VoIPGatewayPri"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCiscoCatalyst6000T1VoIPGatewayPri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCiscoCatalyst6000T1VoIPGatewayPri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCiscoCatalyst6000T1VoIPGatewayPri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCiscoCatalyst6000T1VoIPGatewayPri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCiscoCatalyst6000T1VoIPGatewayPri(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCiscoCatalyst6000T1VoIPGatewayPri(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCiscoCatalyst6000T1VoIPGatewayT1(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", mediaResourceListName=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, loadInformation=None, vendorConfig=None, traceFlag=None, mlppDomainId=None, mlppIndicationStatus=None, preemption=None, useTrustedRelayPoint=None, retryVideoCallAsAudio=None, cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, sendGeoLocation="", ports=None, trunkSelectionOrder="", clockReference="", csuParam="", digitSending="", pcmType="", fdlChannel="", yellowAlarm="", zeroSupression="", smdiBasePort="", handleDtmfPrecedenceSignals=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, geoLocationFilterName=None, pstnAccess=None, imeE164TransformationName=None):
        """
        :param ciscoCatalyst6000T1VoIPGatewayT1: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value)
            ports: optional
                    portNumber: UnionType(value)
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value)
                    callingPartySelection: UnionType(value)
                    digitSending: UnionType(value)
                    expectedDigits: UnionType(value)
                    prefixDn: String255(value)
                    presentationBit: UnionType(value)
                    silenceSuppressionThreshold: UnionType(value)
                    startDialProtocol: UnionType(value)
                    trunk: UnionType(value)
                    trunkDirection: UnionType(value)
                    trunkLevel: UnionType(value)
                    trunkPadRx: UnionType(value)
                    trunkPadTx: UnionType(value)
                    callerId: UnionType(value)
                    endpointId: String128(value), optional
                    timer1: UnionType(value), optional
                    timer2: UnionType(value), optional
                    timer3: UnionType(value), optional
                    timer4: UnionType(value), optional
                    timer5: UnionType(value), optional
                    timer6: UnionType(value), optional
            trunkSelectionOrder: UnionType(value)
            clockReference: UnionType(value)
            csuParam: UnionType(value)
            digitSending: UnionType(value)
            pcmType: UnionType(value)
            fdlChannel: UnionType(value)
            yellowAlarm: UnionType(value)
            zeroSupression: UnionType(value)
            smdiBasePort: UnionType(value)
            handleDtmfPrecedenceSignals: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.addCiscoCatalyst6000T1VoIPGatewayT1({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "loadInformation": loadInformation, "vendorConfig": vendorConfig, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "preemption": preemption, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "sendGeoLocation": sendGeoLocation, "ports": ports, "trunkSelectionOrder": trunkSelectionOrder, "clockReference": clockReference, "csuParam": csuParam, "digitSending": digitSending, "pcmType": pcmType, "fdlChannel": fdlChannel, "yellowAlarm": yellowAlarm, "zeroSupression": zeroSupression, "smdiBasePort": smdiBasePort, "handleDtmfPrecedenceSignals": handleDtmfPrecedenceSignals, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "geoLocationFilterName": geoLocationFilterName, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName})["return"]
        except Fault as e:
            return e

    def updateCiscoCatalyst6000T1VoIPGatewayT1(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param loadInformation: optional
            _value_1: String(value)
        :param vendorConfig: optional
            _value_1: AnyType, optional
        :param traceFlag: boolean(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param sendGeoLocation: boolean(value), optional
        :param ports: optional
            port: 
                portNumber: UnionType(value)
                attendantDn: String255(value), optional
                unattendedPort: boolean(value), optional
                callerIdDn: String255(value), optional
                callerIdEnable: boolean(value)
                callingPartySelection: UnionType(value)
                digitSending: UnionType(value)
                expectedDigits: UnionType(value)
                    _value_1: UnionType(value)
                prefixDn: String255(value)
                presentationBit: UnionType(value)
                silenceSuppressionThreshold: UnionType(value)
                startDialProtocol: UnionType(value)
                trunk: UnionType(value)
                trunkDirection: UnionType(value)
                trunkLevel: UnionType(value)
                trunkPadRx: UnionType(value)
                trunkPadTx: UnionType(value)
                    _value_1: AnyType, optional
                callerId: UnionType(value)
                endpointId: String128(value), optional
                timer1: UnionType(value), optional
                timer2: UnionType(value), optional
                timer3: UnionType(value), optional
                timer4: UnionType(value), optional
                timer5: UnionType(value), optional
                timer6: UnionType(value), optional
        :param trunkSelectionOrder: UnionType(value), optional
        :param clockReference: UnionType(value), optional
        :param csuParam: UnionType(value), optional
        :param digitSending: UnionType(value), optional
        :param pcmType: UnionType(value), optional
        :param fdlChannel: UnionType(value), optional
        :param yellowAlarm: UnionType(value), optional
        :param zeroSupression: UnionType(value), optional
        :param smdiBasePort: UnionType(value), optional
        :param handleDtmfPrecedenceSignals: boolean(value), optional
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        """
        try:
            return self.client.updateCiscoCatalyst6000T1VoIPGatewayT1(**kwargs)["return"]
        except Fault as e:
            return e

    def getCiscoCatalyst6000T1VoIPGatewayT1(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            vendorConfig: optional
                _value_1: AnyType, optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            ports: optional
                    portNumber: UnionType(value), optional
                    attendantDn: String255(value), optional
                    unattendedPort: boolean(value), optional
                    callerIdDn: String255(value), optional
                    callerIdEnable: boolean(value), optional
                    callingPartySelection: UnionType(value), optional
                    digitSending: UnionType(value), optional
                    expectedDigits: UnionType(value), optional
                    prefixDn: String255(value), optional
                    presentationBit: UnionType(value), optional
                    silenceSuppressionThreshold: UnionType(value), optional
                    startDialProtocol: UnionType(value), optional
                    trunk: UnionType(value), optional
                    trunkDirection: UnionType(value), optional
                    trunkLevel: UnionType(value), optional
                    trunkPadRx: UnionType(value), optional
                    trunkPadTx: UnionType(value), optional
                    callerId: UnionType(value), optional
                    endpointId: String128(value), optional
                    timer1: UnionType(value), optional
                    timer2: UnionType(value), optional
                    timer3: UnionType(value), optional
                    timer4: UnionType(value), optional
                    timer5: UnionType(value), optional
                    timer6: UnionType(value), optional
            trunkSelectionOrder: UnionType(value), optional
            clockReference: UnionType(value), optional
            csuParam: UnionType(value), optional
            digitSending: UnionType(value), optional
            pcmType: UnionType(value), optional
            fdlChannel: UnionType(value), optional
            yellowAlarm: UnionType(value), optional
            zeroSupression: UnionType(value), optional
            smdiBasePort: UnionType(value), optional
            handleDtmfPrecedenceSignals: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        """
        try:
            return self.client.getCiscoCatalyst6000T1VoIPGatewayT1(**kwargs)["return"]["ciscoCatalyst6000T1VoIPGatewayT1"]
        except Fault as e:
            return e

    def removeCiscoCatalyst6000T1VoIPGatewayT1(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeCiscoCatalyst6000T1VoIPGatewayT1(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listCiscoCatalyst6000T1VoIPGatewayT1(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'loadInformation': {'_value_1': ''}, 'traceFlag': '', 'mlppIndicationStatus': '', 'preemption': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'sendGeoLocation': '', 'trunkSelectionOrder': '', 'clockReference': '', 'csuParam': '', 'digitSending': '', 'pcmType': '', 'fdlChannel': '', 'yellowAlarm': '', 'zeroSupression': '', 'smdiBasePort': '', 'handleDtmfPrecedenceSignals': '', 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'geoLocationFilterName': {'_value_1': ''}, 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            loadInformation: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            trunkSelectionOrder: UnionType(value), optional
            clockReference: UnionType(value), optional
            csuParam: UnionType(value), optional
            digitSending: UnionType(value), optional
            pcmType: UnionType(value), optional
            fdlChannel: UnionType(value), optional
            yellowAlarm: UnionType(value), optional
            zeroSupression: UnionType(value), optional
            smdiBasePort: UnionType(value), optional
            handleDtmfPrecedenceSignals: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            geoLocationFilterName: optional
                _value_1: String(value)
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCiscoCatalyst6000T1VoIPGatewayT1({'name': '%', 'description': '%', 'callingSearchSpaceName': '%', 'devicePoolName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["ciscoCatalyst6000T1VoIPGatewayT1"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def resetCiscoCatalyst6000T1VoIPGatewayT1(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.resetCiscoCatalyst6000T1VoIPGatewayT1(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def restartCiscoCatalyst6000T1VoIPGatewayT1(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.restartCiscoCatalyst6000T1VoIPGatewayT1(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def applyCiscoCatalyst6000T1VoIPGatewayT1(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.applyCiscoCatalyst6000T1VoIPGatewayT1(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def addCallPickupGroup(self, pattern="", description=None, routePartitionName=None, members=None, pickupNotification=None, pickupNotificationTimer=None, callInfoForPickupNotification=None, name=""):
        """
        :param callPickupGroup: 
            pattern: String(value)
            description: String(value), optional
            routePartitionName: optional
                _value_1: String(value)
            members: optional
                    priority: UnionType(value)
            pickupNotification: UnionType(value), optional
            pickupNotificationTimer: UnionType(value), optional
            callInfoForPickupNotification: optional
                callingPartyInfo: boolean(value), optional
                calledPartyInfo: boolean(value), optional
            name: String(value)
        """
        try:
            return self.client.addCallPickupGroup({"pattern": pattern, "description": description, "routePartitionName": routePartitionName, "members": members, "pickupNotification": pickupNotification, "pickupNotificationTimer": pickupNotificationTimer, "callInfoForPickupNotification": callInfoForPickupNotification, "name": name})["return"]
        except Fault as e:
            return e

    def updateCallPickupGroup(self, pattern="", **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param name: Name(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param newPattern: String(value), optional
        :param description: String(value), optional
        :param newRoutePartitionName: optional
            _value_1: String(value)
        :param removeMembers: optional
            member: optional
                priority: UnionType(value)
                    _value_1: String(value)
                    dnPattern: String255(value)
        :param addMembers: optional
            member: optional
                priority: UnionType(value)
                    _value_1: String(value)
                    dnPattern: String255(value)
        :param members: optional
            member: optional
                priority: UnionType(value)
                    _value_1: String(value)
                    dnPattern: String255(value)
        :param pickupNotification: UnionType(value), optional
        :param pickupNotificationTimer: UnionType(value), optional
        :param callInfoForPickupNotification: optional
            callingPartyInfo: boolean(value), optional
            calledPartyInfo: boolean(value), optional
        :param newName: String(value), optional
        """
        try:
            return self.client.updateCallPickupGroup(pattern=pattern, **kwargs)["return"]
        except Fault as e:
            return e

    def getCallPickupGroup(self, **kwargs):
        """
        :param uuid: XUUID(value), optional
        :param name: String(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        :param returnedTags: optional
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            members: optional
                    priority: UnionType(value), optional
            pickupNotification: UnionType(value), optional
            pickupNotificationTimer: UnionType(value), optional
            callInfoForPickupNotification: optional
                callingPartyInfo: boolean(value), optional
                calledPartyInfo: boolean(value), optional
            name: String(value), optional
        """
        try:
            return self.client.getCallPickupGroup(**kwargs)["return"]["callPickupGroup"]
        except Fault as e:
            return e

    def removeCallPickupGroup(self, uuid="", name="", pattern="", routePartitionName=""):
        """
        :param uuid: XUUID(value), optional
        :param name: String(value), optional
        :param pattern: String(value)
        :param routePartitionName: optional
            _value_1: String(value)
        """
        try:
            return self.client.removeCallPickupGroup(uuid=uuid, name=name, pattern=pattern, routePartitionName={'_value_1': ''})["return"]
        except Fault as e:
            return e

    def listCallPickupGroup(self, returnedTags={'pattern': '', 'description': '', 'usage': '', 'routePartitionName': {'_value_1': ''}, 'pickupNotification': '', 'pickupNotificationTimer': '', 'callInfoForPickupNotification': {'callingPartyInfo': '', 'calledPartyInfo': ''}, 'name': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            pattern: String(value), optional
            description: String(value), optional
            routePartitionName: String(value), optional
        :param returnedTags: 
            pattern: String(value), optional
            description: String(value), optional
            usage: UnionType(value), optional
            routePartitionName: optional
                _value_1: String(value)
            pickupNotification: UnionType(value), optional
            pickupNotificationTimer: UnionType(value), optional
            callInfoForPickupNotification: optional
                callingPartyInfo: boolean(value), optional
                calledPartyInfo: boolean(value), optional
            name: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listCallPickupGroup({'pattern': '%', 'description': '%', 'routePartitionName': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["callPickupGroup"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def listRoutePlan(self, returnedTags={'dnOrPattern': '', 'partition': {'_value_1': ''}, 'type': '', 'routeDetail': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            dnOrPattern: String(value), optional
            partition: String(value), optional
            type: String(value), optional
        :param returnedTags: 
            dnOrPattern: String255(value), optional
            partition: optional
                _value_1: String(value)
            type: UnionType(value), optional
            routeDetail: String(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listRoutePlan({'dnOrPattern': '%', 'partition': '%', 'type': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["routePlan"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addGeoLocationPolicy(self, name="", country=None, description=None, nationalSubDivision=None, district=None, communityName=None, cityDivision=None, neighbourhood=None, street=None, leadingStreetDirection=None, trailingStreetSuffix=None, streetSuffix=None, houseNumber=None, houseNumberSuffix=None, landmark=None, location=None, floor=None, occupantName=None, postalCode=None, relatedPolicies=None):
        """
        :param geoLocationPolicy: 
            name: String50(value)
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
            relatedPolicies: optional
                    geoLocationDeviceA: UnionType(value)
                    geoLocationDeviceB: UnionType(value)
                    logicalPartitionPolicy: UnionType(value)
        """
        try:
            return self.client.addGeoLocationPolicy({"name": name, "country": country, "description": description, "nationalSubDivision": nationalSubDivision, "district": district, "communityName": communityName, "cityDivision": cityDivision, "neighbourhood": neighbourhood, "street": street, "leadingStreetDirection": leadingStreetDirection, "trailingStreetSuffix": trailingStreetSuffix, "streetSuffix": streetSuffix, "houseNumber": houseNumber, "houseNumberSuffix": houseNumberSuffix, "landmark": landmark, "location": location, "floor": floor, "occupantName": occupantName, "postalCode": postalCode, "relatedPolicies": relatedPolicies})["return"]
        except Fault as e:
            return e

    def updateGeoLocationPolicy(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: String50(value), optional
        :param country: String(value), optional
        :param description: String255(value), optional
        :param nationalSubDivision: String100(value), optional
        :param district: String255(value), optional
        :param communityName: String255(value), optional
        :param cityDivision: String255(value), optional
        :param neighbourhood: String255(value), optional
        :param street: String255(value), optional
        :param leadingStreetDirection: String50(value), optional
        :param trailingStreetSuffix: String50(value), optional
        :param streetSuffix: String255(value), optional
        :param houseNumber: String16(value), optional
        :param houseNumberSuffix: String100(value), optional
        :param landmark: String255(value), optional
        :param location: String255(value), optional
        :param floor: String16(value), optional
        :param occupantName: String255(value), optional
        :param postalCode: String100(value), optional
        :param removeRelatedPolicies: optional
            relatedPolicy: 
                    _value_1: String(value)
                geoLocationDeviceA: UnionType(value)
                    _value_1: String(value)
                geoLocationDeviceB: UnionType(value)
                logicalPartitionPolicy: UnionType(value)
        :param addRelatedPolicies: optional
            relatedPolicy: 
                    _value_1: String(value)
                geoLocationDeviceA: UnionType(value)
                    _value_1: String(value)
                geoLocationDeviceB: UnionType(value)
                logicalPartitionPolicy: UnionType(value)
        :param relatedPolicies: optional
            relatedPolicy: 
                    _value_1: String(value)
                geoLocationDeviceA: UnionType(value)
                    _value_1: String(value)
                geoLocationDeviceB: UnionType(value)
                logicalPartitionPolicy: UnionType(value)
        """
        try:
            return self.client.updateGeoLocationPolicy(**kwargs)["return"]
        except Fault as e:
            return e

    def getGeoLocationPolicy(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: String50(value), optional
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
            relatedPolicies: optional
                    geoLocationDeviceA: UnionType(value), optional
                    geoLocationDeviceB: UnionType(value), optional
                    logicalPartitionPolicy: UnionType(value), optional
        """
        try:
            return self.client.getGeoLocationPolicy(**kwargs)["return"]["geoLocationPolicy"]
        except Fault as e:
            return e

    def removeGeoLocationPolicy(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeGeoLocationPolicy(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listGeoLocationPolicy(self, returnedTags={'name': '', 'country': '', 'description': '', 'nationalSubDivision': '', 'district': '', 'communityName': '', 'cityDivision': '', 'neighbourhood': '', 'street': '', 'leadingStreetDirection': '', 'trailingStreetSuffix': '', 'streetSuffix': '', 'houseNumber': '', 'houseNumberSuffix': '', 'landmark': '', 'location': '', 'floor': '', 'occupantName': '', 'postalCode': ''}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
        :param returnedTags: 
            name: String50(value), optional
            country: String(value), optional
            description: String255(value), optional
            nationalSubDivision: String100(value), optional
            district: String255(value), optional
            communityName: String255(value), optional
            cityDivision: String255(value), optional
            neighbourhood: String255(value), optional
            street: String255(value), optional
            leadingStreetDirection: String50(value), optional
            trailingStreetSuffix: String50(value), optional
            streetSuffix: String255(value), optional
            houseNumber: String16(value), optional
            houseNumberSuffix: String100(value), optional
            landmark: String255(value), optional
            location: String255(value), optional
            floor: String16(value), optional
            occupantName: String255(value), optional
            postalCode: String100(value), optional
        :param skip: UnsignedLong(value), optional
        :param first: UnsignedLong(value), optional
        """
        try:
            returnvalue = self.client.listGeoLocationPolicy({'name': '%', 'description': '%'}, returnedTags=returnedTags, skip=skip, first=first)
            return returnvalue["return"]["geoLocationPolicy"] if returnvalue["return"] else None
        except Fault as e:
            return e

    def addSipTrunk(self, name="", description=None, product="", classvalue="", protocol="", protocolSide="", callingSearchSpaceName=None, devicePoolName="", commonDeviceConfigName=None, networkLocation=None, locationName="", mediaResourceListName=None, networkHoldMohAudioSourceId=None, userHoldMohAudioSourceId=None, automatedAlternateRoutingCssName=None, aarNeighborhoodName=None, packetCaptureMode=None, packetCaptureDuration=None, loadInformation=None, traceFlag=None, mlppDomainId=None, mlppIndicationStatus=None, preemption=None, useTrustedRelayPoint=None, retryVideoCallAsAudio=None, securityProfileName="", sipProfileName="", cgpnTransformationCssName=None, useDevicePoolCgpnTransformCss=None, geoLocationName=None, geoLocationFilterName=None, sendGeoLocation=None, cdpnTransformationCssName=None, useDevicePoolCdpnTransformCss=None, unattendedPort=None, transmitUtf8=None, subscribeCallingSearchSpaceName=None, rerouteCallingSearchSpaceName=None, referCallingSearchSpaceName=None, mtpRequired=None, presenceGroupName="", unknownPrefix=None, destAddrIsSrv=None, tkSipCodec=None, sigDigits=None, connectedNamePresentation=None, connectedPartyIdPresentation=None, callingPartySelection=None, callingname=None, callingLineIdPresentation=None, prefixDn=None, callerName=None, callerIdDn=None, acceptInboundRdnis=None, acceptOutboundRdnis=None, srtpAllowed=None, srtpFallbackAllowed=None, isPaiEnabled=None, sipPrivacy=None, isRpidEnabled=None, sipAssertedType=None, dtmfSignalingMethod=None, routeClassSignalling=None, sipTrunkType=None, pstnAccess=None, imeE164TransformationName=None, useImePublicIpPort=None, useDevicePoolCntdPnTransformationCss=None, cntdPnTransformationCssName=None, useDevicePoolCgpnTransformCssUnkn=None, rdnTransformationCssName=None, useDevicePoolRdnTransformCss=None, sipNormalizationScriptName=None, runOnEveryNode=None, destinations=None, unknownStripDigits=None, cgpnTransformationUnknownCssName=None, tunneledProtocol=None, asn1RoseOidEncoding=None, qsigVariant=None, pathReplacementSupport=None, enableQsigUtf8=None, scriptParameters=None, scriptTraceEnabled=None, trunkTrafficSecure=None, callingAndCalledPartyInfoFormat="", useCallerIdCallerNameinUriOutgoingRequest=None, requestUriDomainName=None, enableCiscoRecordingQsigTunneling=None, recordingInformation=None, calledPartyUnknownTransformationCssName=None, calledPartyUnknownPrefix=None, calledPartyUnknownStripDigits=None, useDevicePoolCalledCssUnkn=None, confidentialAccess=None):
        """
        :param sipTrunk: 
            name: UniqueString128(value)
            description: String128(value), optional
            product: UnionType(value)
            class: UnionType(value)
            protocol: UnionType(value)
            protocolSide: UnionType(value)
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: 
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: 
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: 
                _value_1: String(value)
            sipProfileName: 
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            unattendedPort: boolean(value), optional
            transmitUtf8: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            referCallingSearchSpaceName: optional
                _value_1: String(value)
            mtpRequired: boolean(value), optional
            presenceGroupName: 
                _value_1: String(value)
            unknownPrefix: String16(value), optional
            destAddrIsSrv: boolean(value), optional
            tkSipCodec: UnionType(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            connectedNamePresentation: UnionType(value), optional
            connectedPartyIdPresentation: UnionType(value), optional
            callingPartySelection: UnionType(value), optional
            callingname: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            prefixDn: String50(value), optional
            callerName: String50(value), optional
            callerIdDn: String50(value), optional
            acceptInboundRdnis: boolean(value), optional
            acceptOutboundRdnis: boolean(value), optional
            srtpAllowed: boolean(value), optional
            srtpFallbackAllowed: boolean(value), optional
            isPaiEnabled: boolean(value), optional
            sipPrivacy: UnionType(value), optional
            isRpidEnabled: boolean(value), optional
            sipAssertedType: UnionType(value), optional
            dtmfSignalingMethod: UnionType(value), optional
            routeClassSignalling: UnionType(value), optional
            sipTrunkType: UnionType(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            useImePublicIpPort: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            rdnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolRdnTransformCss: boolean(value), optional
            sipNormalizationScriptName: optional
                _value_1: String(value)
            runOnEveryNode: boolean(value), optional
            destinations: optional
                    addressIpv4: String255(value), optional
                    addressIpv6: String255(value), optional
                    port: UnionType(value), optional
                    sortOrder: UnionType(value)
            unknownStripDigits: UnionType(value), optional
            cgpnTransformationUnknownCssName: optional
                _value_1: String(value)
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            enableQsigUtf8: boolean(value), optional
            scriptParameters: String(value), optional
            scriptTraceEnabled: boolean(value), optional
            trunkTrafficSecure: UnionType(value), optional
            callingAndCalledPartyInfoFormat: UnionType(value)
            useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
            requestUriDomainName: String(value), optional
            enableCiscoRecordingQsigTunneling: boolean(value), optional
            recordingInformation: String(value), optional
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownPrefix: String16(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value)
                confidentialAccessLevel: UnionType(value)
        """
        try:
            return self.client.addSipTrunk({"name": name, "description": description, "product": product, "class": classvalue, "protocol": protocol, "protocolSide": protocolSide, "callingSearchSpaceName": callingSearchSpaceName, "devicePoolName": devicePoolName, "commonDeviceConfigName": commonDeviceConfigName, "networkLocation": networkLocation, "locationName": locationName, "mediaResourceListName": mediaResourceListName, "networkHoldMohAudioSourceId": networkHoldMohAudioSourceId, "userHoldMohAudioSourceId": userHoldMohAudioSourceId, "automatedAlternateRoutingCssName": automatedAlternateRoutingCssName, "aarNeighborhoodName": aarNeighborhoodName, "packetCaptureMode": packetCaptureMode, "packetCaptureDuration": packetCaptureDuration, "loadInformation": loadInformation, "traceFlag": traceFlag, "mlppDomainId": mlppDomainId, "mlppIndicationStatus": mlppIndicationStatus, "preemption": preemption, "useTrustedRelayPoint": useTrustedRelayPoint, "retryVideoCallAsAudio": retryVideoCallAsAudio, "securityProfileName": securityProfileName, "sipProfileName": sipProfileName, "cgpnTransformationCssName": cgpnTransformationCssName, "useDevicePoolCgpnTransformCss": useDevicePoolCgpnTransformCss, "geoLocationName": geoLocationName, "geoLocationFilterName": geoLocationFilterName, "sendGeoLocation": sendGeoLocation, "cdpnTransformationCssName": cdpnTransformationCssName, "useDevicePoolCdpnTransformCss": useDevicePoolCdpnTransformCss, "unattendedPort": unattendedPort, "transmitUtf8": transmitUtf8, "subscribeCallingSearchSpaceName": subscribeCallingSearchSpaceName, "rerouteCallingSearchSpaceName": rerouteCallingSearchSpaceName, "referCallingSearchSpaceName": referCallingSearchSpaceName, "mtpRequired": mtpRequired, "presenceGroupName": presenceGroupName, "unknownPrefix": unknownPrefix, "destAddrIsSrv": destAddrIsSrv, "tkSipCodec": tkSipCodec, "sigDigits": sigDigits, "connectedNamePresentation": connectedNamePresentation, "connectedPartyIdPresentation": connectedPartyIdPresentation, "callingPartySelection": callingPartySelection, "callingname": callingname, "callingLineIdPresentation": callingLineIdPresentation, "prefixDn": prefixDn, "callerName": callerName, "callerIdDn": callerIdDn, "acceptInboundRdnis": acceptInboundRdnis, "acceptOutboundRdnis": acceptOutboundRdnis, "srtpAllowed": srtpAllowed, "srtpFallbackAllowed": srtpFallbackAllowed, "isPaiEnabled": isPaiEnabled, "sipPrivacy": sipPrivacy, "isRpidEnabled": isRpidEnabled, "sipAssertedType": sipAssertedType, "dtmfSignalingMethod": dtmfSignalingMethod, "routeClassSignalling": routeClassSignalling, "sipTrunkType": sipTrunkType, "pstnAccess": pstnAccess, "imeE164TransformationName": imeE164TransformationName, "useImePublicIpPort": useImePublicIpPort, "useDevicePoolCntdPnTransformationCss": useDevicePoolCntdPnTransformationCss, "cntdPnTransformationCssName": cntdPnTransformationCssName, "useDevicePoolCgpnTransformCssUnkn": useDevicePoolCgpnTransformCssUnkn, "rdnTransformationCssName": rdnTransformationCssName, "useDevicePoolRdnTransformCss": useDevicePoolRdnTransformCss, "sipNormalizationScriptName": sipNormalizationScriptName, "runOnEveryNode": runOnEveryNode, "destinations": destinations, "unknownStripDigits": unknownStripDigits, "cgpnTransformationUnknownCssName": cgpnTransformationUnknownCssName, "tunneledProtocol": tunneledProtocol, "asn1RoseOidEncoding": asn1RoseOidEncoding, "qsigVariant": qsigVariant, "pathReplacementSupport": pathReplacementSupport, "enableQsigUtf8": enableQsigUtf8, "scriptParameters": scriptParameters, "scriptTraceEnabled": scriptTraceEnabled, "trunkTrafficSecure": trunkTrafficSecure, "callingAndCalledPartyInfoFormat": callingAndCalledPartyInfoFormat, "useCallerIdCallerNameinUriOutgoingRequest": useCallerIdCallerNameinUriOutgoingRequest, "requestUriDomainName": requestUriDomainName, "enableCiscoRecordingQsigTunneling": enableCiscoRecordingQsigTunneling, "recordingInformation": recordingInformation, "calledPartyUnknownTransformationCssName": calledPartyUnknownTransformationCssName, "calledPartyUnknownPrefix": calledPartyUnknownPrefix, "calledPartyUnknownStripDigits": calledPartyUnknownStripDigits, "useDevicePoolCalledCssUnkn": useDevicePoolCalledCssUnkn, "confidentialAccess": confidentialAccess})["return"]
        except Fault as e:
            return e

    def updateSipTrunk(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param newName: UniqueString128(value), optional
        :param description: String128(value), optional
        :param callingSearchSpaceName: optional
            _value_1: String(value)
        :param devicePoolName: optional
            _value_1: String(value)
        :param commonDeviceConfigName: optional
            _value_1: String(value)
        :param networkLocation: UnionType(value), optional
        :param locationName: optional
            _value_1: String(value)
        :param mediaResourceListName: optional
            _value_1: String(value)
        :param networkHoldMohAudioSourceId: UnionType(value), optional
        :param userHoldMohAudioSourceId: UnionType(value), optional
        :param automatedAlternateRoutingCssName: optional
            _value_1: String(value)
        :param aarNeighborhoodName: optional
            _value_1: String(value)
        :param packetCaptureMode: UnionType(value), optional
        :param packetCaptureDuration: UnionType(value), optional
        :param mlppDomainId: String128(value), optional
        :param mlppIndicationStatus: UnionType(value), optional
        :param preemption: UnionType(value), optional
        :param useTrustedRelayPoint: UnionType(value), optional
        :param retryVideoCallAsAudio: boolean(value), optional
        :param securityProfileName: optional
            _value_1: String(value)
        :param sipProfileName: optional
            _value_1: String(value)
        :param cgpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCss: boolean(value), optional
        :param geoLocationName: optional
            _value_1: String(value)
        :param geoLocationFilterName: optional
            _value_1: String(value)
        :param sendGeoLocation: boolean(value), optional
        :param cdpnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCdpnTransformCss: boolean(value), optional
        :param unattendedPort: boolean(value), optional
        :param transmitUtf8: boolean(value), optional
        :param subscribeCallingSearchSpaceName: optional
            _value_1: String(value)
        :param rerouteCallingSearchSpaceName: optional
            _value_1: String(value)
        :param referCallingSearchSpaceName: optional
            _value_1: String(value)
        :param mtpRequired: boolean(value), optional
        :param presenceGroupName: optional
            _value_1: String(value)
        :param unknownPrefix: String16(value), optional
        :param destAddrIsSrv: boolean(value), optional
        :param tkSipCodec: UnionType(value), optional
        :param sigDigits: optional
            _value_1: UnionType(value)
        :param connectedNamePresentation: UnionType(value), optional
        :param connectedPartyIdPresentation: UnionType(value), optional
        :param callingPartySelection: UnionType(value), optional
        :param callingname: UnionType(value), optional
        :param callingLineIdPresentation: UnionType(value), optional
        :param prefixDn: String50(value), optional
        :param callerName: String50(value), optional
        :param callerIdDn: String50(value), optional
        :param acceptInboundRdnis: boolean(value), optional
        :param acceptOutboundRdnis: boolean(value), optional
        :param srtpAllowed: boolean(value), optional
        :param srtpFallbackAllowed: boolean(value), optional
        :param isPaiEnabled: boolean(value), optional
        :param sipPrivacy: UnionType(value), optional
        :param isRpidEnabled: boolean(value), optional
        :param sipAssertedType: UnionType(value), optional
        :param dtmfSignalingMethod: UnionType(value), optional
        :param routeClassSignalling: UnionType(value), optional
        :param sipTrunkType: UnionType(value), optional
        :param pstnAccess: boolean(value), optional
        :param imeE164TransformationName: optional
            _value_1: String(value)
        :param useImePublicIpPort: boolean(value), optional
        :param useDevicePoolCntdPnTransformationCss: boolean(value), optional
        :param cntdPnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
        :param rdnTransformationCssName: optional
            _value_1: String(value)
        :param useDevicePoolRdnTransformCss: boolean(value), optional
        :param sipNormalizationScriptName: optional
            _value_1: String(value)
        :param runOnEveryNode: boolean(value), optional
        :param removeDestinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                addressIpv6: String255(value), optional
                port: UnionType(value), optional
                sortOrder: UnionType(value)
        :param addDestinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                addressIpv6: String255(value), optional
                port: UnionType(value), optional
                sortOrder: UnionType(value)
        :param destinations: optional
            destination: optional
                addressIpv4: String255(value), optional
                addressIpv6: String255(value), optional
                port: UnionType(value), optional
                sortOrder: UnionType(value)
        :param unknownStripDigits: UnionType(value), optional
        :param cgpnTransformationUnknownCssName: optional
            _value_1: String(value)
        :param tunneledProtocol: UnionType(value), optional
        :param asn1RoseOidEncoding: UnionType(value), optional
        :param qsigVariant: UnionType(value), optional
        :param pathReplacementSupport: boolean(value), optional
        :param enableQsigUtf8: boolean(value), optional
        :param scriptParameters: String(value), optional
        :param scriptTraceEnabled: boolean(value), optional
        :param trunkTrafficSecure: UnionType(value), optional
        :param callingAndCalledPartyInfoFormat: UnionType(value), optional
        :param useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
        :param requestUriDomainName: String(value), optional
        :param enableCiscoRecordingQsigTunneling: boolean(value), optional
        :param recordingInformation: String(value), optional
        :param calledPartyUnknownTransformationCssName: optional
            _value_1: String(value)
        :param calledPartyUnknownPrefix: String16(value), optional
        :param calledPartyUnknownStripDigits: UnionType(value), optional
        :param useDevicePoolCalledCssUnkn: boolean(value), optional
        :param confidentialAccess: optional
            confidentialAccessMode: UnionType(value)
            confidentialAccessLevel: UnionType(value)
        """
        try:
            return self.client.updateSipTrunk(**kwargs)["return"]
        except Fault as e:
            return e

    def getSipTrunk(self, **kwargs):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        :param returnedTags: optional
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            versionStamp: String128(value), optional
            traceFlag: boolean(value), optional
            mlppDomainId: String128(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: optional
                _value_1: String(value)
            sipProfileName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            unattendedPort: boolean(value), optional
            transmitUtf8: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            referCallingSearchSpaceName: optional
                _value_1: String(value)
            mtpRequired: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            unknownPrefix: String16(value), optional
            destAddrIsSrv: boolean(value), optional
            tkSipCodec: UnionType(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            connectedNamePresentation: UnionType(value), optional
            connectedPartyIdPresentation: UnionType(value), optional
            callingPartySelection: UnionType(value), optional
            callingname: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            prefixDn: String50(value), optional
            callerName: String50(value), optional
            callerIdDn: String50(value), optional
            acceptInboundRdnis: boolean(value), optional
            acceptOutboundRdnis: boolean(value), optional
            srtpAllowed: boolean(value), optional
            srtpFallbackAllowed: boolean(value), optional
            isPaiEnabled: boolean(value), optional
            sipPrivacy: UnionType(value), optional
            isRpidEnabled: boolean(value), optional
            sipAssertedType: UnionType(value), optional
            dtmfSignalingMethod: UnionType(value), optional
            routeClassSignalling: UnionType(value), optional
            sipTrunkType: UnionType(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            useImePublicIpPort: boolean(value), optional
            useDevicePoolCntdPnTransformationCss: boolean(value), optional
            cntdPnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCssUnkn: boolean(value), optional
            rdnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolRdnTransformCss: boolean(value), optional
            sipNormalizationScriptName: optional
                _value_1: String(value)
            runOnEveryNode: boolean(value), optional
            destinations: optional
                    addressIpv4: String255(value), optional
                    addressIpv6: String255(value), optional
                    port: UnionType(value), optional
                    sortOrder: UnionType(value), optional
            unknownStripDigits: UnionType(value), optional
            cgpnTransformationUnknownCssName: optional
                _value_1: String(value)
            tunneledProtocol: UnionType(value), optional
            asn1RoseOidEncoding: UnionType(value), optional
            qsigVariant: UnionType(value), optional
            pathReplacementSupport: boolean(value), optional
            enableQsigUtf8: boolean(value), optional
            scriptParameters: String(value), optional
            scriptTraceEnabled: boolean(value), optional
            trunkTrafficSecure: UnionType(value), optional
            callingAndCalledPartyInfoFormat: UnionType(value), optional
            useCallerIdCallerNameinUriOutgoingRequest: boolean(value), optional
            requestUriDomainName: String(value), optional
            enableCiscoRecordingQsigTunneling: boolean(value), optional
            recordingInformation: String(value), optional
            calledPartyUnknownTransformationCssName: optional
                _value_1: String(value)
            calledPartyUnknownPrefix: String16(value), optional
            calledPartyUnknownStripDigits: UnionType(value), optional
            useDevicePoolCalledCssUnkn: boolean(value), optional
            confidentialAccess: optional
                confidentialAccessMode: UnionType(value), optional
                confidentialAccessLevel: UnionType(value), optional
        """
        try:
            return self.client.getSipTrunk(**kwargs)["return"]["sipTrunk"]
        except Fault as e:
            return e

    def removeSipTrunk(self, name="", uuid=""):
        """
        :param name: String100(value), optional
        :param uuid: XUUID(value), optional
        """
        try:
            return self.client.removeSipTrunk(name=name, uuid=uuid)["return"]
        except Fault as e:
            return e

    def listSipTrunk(self, returnedTags={'name': '', 'description': '', 'product': '', 'model': '', 'class': '', 'protocol': '', 'protocolSide': '', 'callingSearchSpaceName': {'_value_1': ''}, 'devicePoolName': {'_value_1': ''}, 'commonDeviceConfigName': {'_value_1': ''}, 'networkLocation': '', 'locationName': {'_value_1': ''}, 'mediaResourceListName': {'_value_1': ''}, 'networkHoldMohAudioSourceId': '', 'userHoldMohAudioSourceId': '', 'automatedAlternateRoutingCssName': {'_value_1': ''}, 'aarNeighborhoodName': {'_value_1': ''}, 'packetCaptureMode': '', 'packetCaptureDuration': '', 'loadInformation': {'_value_1': ''}, 'traceFlag': '', 'mlppIndicationStatus': '', 'preemption': '', 'useTrustedRelayPoint': '', 'retryVideoCallAsAudio': '', 'securityProfileName': {'_value_1': ''}, 'sipProfileName': {'_value_1': ''}, 'cgpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCgpnTransformCss': '', 'geoLocationName': {'_value_1': ''}, 'geoLocationFilterName': {'_value_1': ''}, 'sendGeoLocation': '', 'cdpnTransformationCssName': {'_value_1': ''}, 'useDevicePoolCdpnTransformCss': '', 'unattendedPort': '', 'transmitUtf8': '', 'subscribeCallingSearchSpaceName': {'_value_1': ''}, 'rerouteCallingSearchSpaceName': {'_value_1': ''}, 'referCallingSearchSpaceName': {'_value_1': ''}, 'mtpRequired': '', 'presenceGroupName': {'_value_1': ''}, 'unknownPrefix': '', 'destAddrIsSrv': '', 'tkSipCodec': '', 'sigDigits': {'_value_1': ''}, 'connectedNamePresentation': '', 'connectedPartyIdPresentation': '', 'callingPartySelection': '', 'callingname': '', 'callingLineIdPresentation': '', 'prefixDn': '', 'callerName': '', 'callerIdDn': '', 'acceptInboundRdnis': '', 'acceptOutboundRdnis': '', 'srtpAllowed': '', 'srtpFallbackAllowed': '', 'isPaiEnabled': '', 'sipPrivacy': '', 'isRpidEnabled': '', 'sipAssertedType': '', 'dtmfSignalingMethod': '', 'routeClassSignalling': '', 'sipTrunkType': '', 'pstnAccess': '', 'imeE164TransformationName': {'_value_1': ''}, 'useImePublicIpPort': '', 'useDevicePoolCntdPnTransformationCss': '', 'useDevicePoolCgpnTransformCssUnkn': '', 'rdnTransformationCssName': {'_value_1': ''}, 'useDevicePoolRdnTransformCss': '', 'sipNormalizationScriptName': {'_value_1': ''}, 'runOnEveryNode': '', 'unknownStripDigits': '', 'cgpnTransformationUnknownCssName': {'_value_1': ''}, 'tunneledProtocol': '', 'asn1RoseOidEncoding': '', 'qsigVariant': '', 'pathReplacementSupport': '', 'enableQsigUtf8': '', 'scriptParameters': '', 'scriptTraceEnabled': '', 'trunkTrafficSecure': '', 'callingAndCalledPartyInfoFormat': '', 'useCallerIdCallerNameinUriOutgoingRequest': '', 'requestUriDomainName': '', 'enableCiscoRecordingQsigTunneling': '', 'calledPartyUnknownTransformationCssName': {'_value_1': ''}, 'calledPartyUnknownPrefix': '', 'calledPartyUnknownStripDigits': '', 'useDevicePoolCalledCssUnkn': '', 'confidentialAccess': {'confidentialAccessMode': '', 'confidentialAccessLevel': ''}}, skip=None, first=None):
        """
        :param searchCriteria: 
            name: String(value), optional
            description: String(value), optional
            callingSearchSpaceName: String(value), optional
            devicePoolName: String(value), optional
        :param returnedTags: 
            name: UniqueString128(value), optional
            description: String128(value), optional
            product: UnionType(value), optional
            model: UnionType(value), optional
            class: UnionType(value), optional
            protocol: UnionType(value), optional
            protocolSide: UnionType(value), optional
            callingSearchSpaceName: optional
                _value_1: String(value)
            devicePoolName: optional
                _value_1: String(value)
            commonDeviceConfigName: optional
                _value_1: String(value)
            networkLocation: UnionType(value), optional
            locationName: optional
                _value_1: String(value)
            mediaResourceListName: optional
                _value_1: String(value)
            networkHoldMohAudioSourceId: UnionType(value), optional
            userHoldMohAudioSourceId: UnionType(value), optional
            automatedAlternateRoutingCssName: optional
                _value_1: String(value)
            aarNeighborhoodName: optional
                _value_1: String(value)
            packetCaptureMode: UnionType(value), optional
            packetCaptureDuration: UnionType(value), optional
            loadInformation: optional
                _value_1: String(value)
            traceFlag: boolean(value), optional
            mlppIndicationStatus: UnionType(value), optional
            preemption: UnionType(value), optional
            useTrustedRelayPoint: UnionType(value), optional
            retryVideoCallAsAudio: boolean(value), optional
            securityProfileName: optional
                _value_1: String(value)
            sipProfileName: optional
                _value_1: String(value)
            cgpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCgpnTransformCss: boolean(value), optional
            geoLocationName: optional
                _value_1: String(value)
            geoLocationFilterName: optional
                _value_1: String(value)
            sendGeoLocation: boolean(value), optional
            cdpnTransformationCssName: optional
                _value_1: String(value)
            useDevicePoolCdpnTransformCss: boolean(value), optional
            unattendedPort: boolean(value), optional
            transmitUtf8: boolean(value), optional
            subscribeCallingSearchSpaceName: optional
                _value_1: String(value)
            rerouteCallingSearchSpaceName: optional
                _value_1: String(value)
            referCallingSearchSpaceName: optional
                _value_1: String(value)
            mtpRequired: boolean(value), optional
            presenceGroupName: optional
                _value_1: String(value)
            unknownPrefix: String16(value), optional
            destAddrIsSrv: boolean(value), optional
            tkSipCodec: UnionType(value), optional
            sigDigits: optional
                _value_1: UnionType(value)
            connectedNamePresentation: UnionType(value), optional
            connectedPartyIdPresentation: UnionType(value), optional
            callingPartySelection: UnionType(value), optional
            callingname: UnionType(value), optional
            callingLineIdPresentation: UnionType(value), optional
            prefixDn: String50(value), optional
            callerName: String50(value), optional
            callerIdDn: String50(value), optional
            acceptInboundRdnis: boolean(value), optional
            acceptOutboundRdnis: boolean(value), optional
            srtpAllowed: boolean(value), optional
            srtpFallbackAllowed: boolean(value), optional
            isPaiEnabled: boolean(value), optional
            sipPrivacy: UnionType(value), optional
            isRpidEnabled: boolean(value), optional
            sipAssertedType: UnionType(value), optional
            dtmfSignalingMethod: UnionType(value), optional
            routeClassSignalling: UnionType(value), optional
            sipTrunkType: UnionType(value), optional
            pstnAccess: boolean(value), optional
            imeE164TransformationName: optional
                _value_1: String(value)
            useImePublicIpPort: b