"use strict";
const assert_1 = require("@aws-cdk/assert");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const path = require("path");
const s3deploy = require("../lib");
module.exports = {
    'deploy from local directory asset'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            'ServiceToken': {
                'Fn::GetAtt': [
                    'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                    'Arn',
                ],
            },
            'SourceBucketNames': [{
                    'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
                }],
            'SourceObjectKeys': [{
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                            },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                            },
                                        ],
                                    },
                                ],
                            },
                        ],
                    ],
                }],
            'DestinationBucketName': {
                'Ref': 'DestC383B82A',
            },
        }));
        test.done();
    },
    'deploy from local directory assets'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [
                s3deploy.Source.asset(path.join(__dirname, 'my-website')),
                s3deploy.Source.asset(path.join(__dirname, 'my-website-second')),
            ],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            'ServiceToken': {
                'Fn::GetAtt': [
                    'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                    'Arn',
                ],
            },
            'SourceBucketNames': [
                {
                    'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3Bucket9CD8B20A',
                },
                {
                    'Ref': 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3Bucket99793559',
                },
            ],
            'SourceObjectKeys': [
                {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                            },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersfc4481abf279255619ff7418faa5d24456fef3432ea0da59c95542578ff0222eS3VersionKeyA58D380C',
                                            },
                                        ],
                                    },
                                ],
                            },
                        ],
                    ],
                },
                {
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                            },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParametersa94977ede0211fd3b45efa33d6d8d1d7bbe0c5a96d977139d8b16abfa96fe9cbS3VersionKeyD9ACE665',
                                            },
                                        ],
                                    },
                                ],
                            },
                        ],
                    ],
                },
            ],
            'DestinationBucketName': {
                'Ref': 'DestC383B82A',
            },
        }));
        test.done();
    },
    'fails if local asset is a non-zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // THEN
        test.throws(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
            destinationBucket: bucket,
        }), /Asset path must be either a \.zip file or a directory/);
        test.done();
    },
    'deploy from a local .zip file'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
        });
        test.done();
    },
    'honors passed asset options'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'), {
                    exclude: ['*', '!index.html'],
                })],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            'ServiceToken': {
                'Fn::GetAtt': [
                    'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756C81C01536',
                    'Arn',
                ],
            },
            'SourceBucketNames': [{
                    'Ref': 'AssetParameterse9b696b2a8a1f93ea8b8a9ce1e4dd4727f9243eba984e50411ca95c6b03d26b6S3Bucket1A1EC3E9',
                }],
            'SourceObjectKeys': [{
                    'Fn::Join': [
                        '',
                        [
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParameterse9b696b2a8a1f93ea8b8a9ce1e4dd4727f9243eba984e50411ca95c6b03d26b6S3VersionKeyE46A4824',
                                            },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            {
                                                'Ref': 'AssetParameterse9b696b2a8a1f93ea8b8a9ce1e4dd4727f9243eba984e50411ca95c6b03d26b6S3VersionKeyE46A4824',
                                            },
                                        ],
                                    },
                                ],
                            },
                        ],
                    ],
                }],
            'DestinationBucketName': {
                'Ref': 'DestC383B82A',
            },
        }));
        test.done();
    },
    'retainOnDelete can be used to retain files when resource is deleted'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            retainOnDelete: true,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            RetainOnDelete: true,
        }));
        test.done();
    },
    'user metadata is correctly transformed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            metadata: {
                A: '1',
                B: '2',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            UserMetadata: { 'x-amzn-meta-a': '1', 'x-amzn-meta-b': '2' },
        }));
        test.done();
    },
    'system metadata is correctly transformed'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            contentType: 'text/html',
            contentLanguage: 'en',
            storageClass: s3deploy.StorageClass.INTELLIGENT_TIERING,
            contentDisposition: 'inline',
            serverSideEncryption: s3deploy.ServerSideEncryption.AWS_KMS,
            serverSideEncryptionAwsKmsKeyId: 'mykey',
            serverSideEncryptionCustomerAlgorithm: 'rot13',
            websiteRedirectLocation: 'example',
            cacheControl: [s3deploy.CacheControl.setPublic(), s3deploy.CacheControl.maxAge(cdk.Duration.hours(1))],
            expires: s3deploy.Expires.after(cdk.Duration.hours(12)),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            SystemMetadata: {
                'content-type': 'text/html',
                'content-language': 'en',
                'content-disposition': 'inline',
                'storage-class': 'INTELLIGENT_TIERING',
                'sse': 'aws:kms',
                'sse-kms-key-id': 'mykey',
                'cache-control': 'public, max-age=3600',
                'expires': s3deploy.Expires.after(cdk.Duration.hours(12)).value,
                'sse-c-copy-source': 'rot13',
                'website-redirect': 'example',
            },
        }));
        test.done();
    },
    'expires type has correct values'(test) {
        test.equal(s3deploy.Expires.atDate(new Date('Sun, 26 Jan 2020 00:53:20 GMT')).value, 'Sun, 26 Jan 2020 00:53:20 GMT');
        test.equal(s3deploy.Expires.atTimestamp(1580000000000).value, 'Sun, 26 Jan 2020 00:53:20 GMT');
        test.ok(Math.abs(new Date(s3deploy.Expires.after(cdk.Duration.minutes(10)).value).getTime() - (Date.now() + 600000)) < 15000, 'Expires.after accurate to within 15 seconds');
        test.equal(s3deploy.Expires.fromString('Tue, 04 Feb 2020 08:45:33 GMT').value, 'Tue, 04 Feb 2020 08:45:33 GMT');
        test.done();
    },
    'cache control type has correct values'(test) {
        test.equal(s3deploy.CacheControl.mustRevalidate().value, 'must-revalidate');
        test.equal(s3deploy.CacheControl.noCache().value, 'no-cache');
        test.equal(s3deploy.CacheControl.noTransform().value, 'no-transform');
        test.equal(s3deploy.CacheControl.setPublic().value, 'public');
        test.equal(s3deploy.CacheControl.setPrivate().value, 'private');
        test.equal(s3deploy.CacheControl.proxyRevalidate().value, 'proxy-revalidate');
        test.equal(s3deploy.CacheControl.maxAge(cdk.Duration.minutes(1)).value, 'max-age=60');
        test.equal(s3deploy.CacheControl.sMaxAge(cdk.Duration.minutes(1)).value, 's-max-age=60');
        test.equal(s3deploy.CacheControl.fromString('only-if-cached').value, 'only-if-cached');
        test.done();
    },
    'storage class type has correct values'(test) {
        test.equal(s3deploy.StorageClass.STANDARD, 'STANDARD');
        test.equal(s3deploy.StorageClass.REDUCED_REDUNDANCY, 'REDUCED_REDUNDANCY');
        test.equal(s3deploy.StorageClass.STANDARD_IA, 'STANDARD_IA');
        test.equal(s3deploy.StorageClass.ONEZONE_IA, 'ONEZONE_IA');
        test.equal(s3deploy.StorageClass.INTELLIGENT_TIERING, 'INTELLIGENT_TIERING');
        test.equal(s3deploy.StorageClass.GLACIER, 'GLACIER');
        test.equal(s3deploy.StorageClass.DEEP_ARCHIVE, 'DEEP_ARCHIVE');
        test.done();
    },
    'server side encryption type has correct values'(test) {
        test.equal(s3deploy.ServerSideEncryption.AES_256, 'AES256');
        test.equal(s3deploy.ServerSideEncryption.AWS_KMS, 'aws:kms');
        test.done();
    },
    'distribution can be used to provide a CloudFront distribution for invalidation'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: bucket,
                    },
                    behaviors: [{ isDefaultBehavior: true }],
                },
            ],
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            distribution,
            distributionPaths: ['/images/*'],
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            DistributionId: {
                'Ref': 'DistributionCFDistribution882A7313',
            },
            DistributionPaths: ['/images/*'],
        }));
        test.done();
    },
    'invalidation can happen without distributionPaths provided'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'Distribution', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: bucket,
                    },
                    behaviors: [{ isDefaultBehavior: true }],
                },
            ],
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website.zip'))],
            destinationBucket: bucket,
            distribution,
        });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::CDKBucketDeployment', {
            DistributionId: {
                'Ref': 'DistributionCFDistribution882A7313',
            },
        }));
        test.done();
    },
    'fails if distribution paths provided but not distribution ID'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // THEN
        test.throws(() => new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website', 'index.html'))],
            destinationBucket: bucket,
            distributionPaths: ['/images/*'],
        }), /Distribution must be specified if distribution paths are specified/);
        test.done();
    },
    'lambda execution role gets permissions to read from the source bucket and read/write in destination'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const source = new s3.Bucket(stack, 'Source');
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        new s3deploy.BucketDeployment(stack, 'Deploy', {
            sources: [s3deploy.Source.bucket(source, 'file.zip')],
            destinationBucket: bucket,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'Source71E471F1',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'Source71E471F1',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                    {
                        'Action': [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                            's3:DeleteObject*',
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        'Effect': 'Allow',
                        'Resource': [
                            {
                                'Fn::GetAtt': [
                                    'DestC383B82A',
                                    'Arn',
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        {
                                            'Fn::GetAtt': [
                                                'DestC383B82A',
                                                'Arn',
                                            ],
                                        },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                'Version': '2012-10-17',
            },
            'PolicyName': 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRoleDefaultPolicy88902FDF',
            'Roles': [
                {
                    'Ref': 'CustomCDKBucketDeployment8693BB64968944B69AAFB0CC9EB8756CServiceRole89A01265',
                },
            ],
        }));
        test.done();
    },
    'memoryLimit can be used to specify the memory limit for the deployment resource handler'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        // WHEN
        // we define 3 deployments with 2 different memory configurations
        new s3deploy.BucketDeployment(stack, 'Deploy256-1', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 256,
        });
        new s3deploy.BucketDeployment(stack, 'Deploy256-2', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 256,
        });
        new s3deploy.BucketDeployment(stack, 'Deploy1024', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            memoryLimit: 1024,
        });
        // THEN
        // we expect to find only two handlers, one for each configuration
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { MemorySize: 256 }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', { MemorySize: 1024 }));
        test.done();
    },
    'deployment allows custom role to be supplied'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'Dest');
        const existingRole = new iam.Role(stack, 'Role', {
            assumedBy: new iam.ServicePrincipal('lambda.amazon.com'),
        });
        // WHEN
        new s3deploy.BucketDeployment(stack, 'DeployWithRole', {
            sources: [s3deploy.Source.asset(path.join(__dirname, 'my-website'))],
            destinationBucket: bucket,
            role: existingRole,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::IAM::Role', 1));
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 1));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            'Role': {
                'Fn::GetAtt': [
                    'Role1ABCC5F0',
                    'Arn',
                ],
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5idWNrZXQtZGVwbG95bWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInRlc3QuYnVja2V0LWRlcGxveW1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBLDRDQUF1RTtBQUN2RSxzREFBc0Q7QUFDdEQsd0NBQXdDO0FBQ3hDLHNDQUFzQztBQUN0QyxxQ0FBcUM7QUFFckMsNkJBQTZCO0FBQzdCLG1DQUFtQztBQUtuQyxpQkFBUztJQUNQLG1DQUFtQyxDQUFDLElBQVU7UUFDNUMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFNUMsT0FBTztRQUNQLElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDN0MsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNwRSxpQkFBaUIsRUFBRSxNQUFNO1NBQzFCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7WUFDM0QsY0FBYyxFQUFFO2dCQUNkLFlBQVksRUFBRTtvQkFDWixtRUFBbUU7b0JBQ25FLEtBQUs7aUJBQ047YUFDRjtZQUNELG1CQUFtQixFQUFFLENBQUM7b0JBQ3BCLEtBQUssRUFBRSxpR0FBaUc7aUJBQ3pHLENBQUM7WUFDRixrQkFBa0IsRUFBRSxDQUFDO29CQUNuQixVQUFVLEVBQUU7d0JBQ1YsRUFBRTt3QkFDRjs0QkFDRTtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSjtnREFDRSxLQUFLLEVBQUUscUdBQXFHOzZDQUM3Rzt5Q0FDRjtxQ0FDRjtpQ0FDRjs2QkFDRjs0QkFDRDtnQ0FDRSxZQUFZLEVBQUU7b0NBQ1osQ0FBQztvQ0FDRDt3Q0FDRSxXQUFXLEVBQUU7NENBQ1gsSUFBSTs0Q0FDSjtnREFDRSxLQUFLLEVBQUUscUdBQXFHOzZDQUM3Rzt5Q0FDRjtxQ0FDRjtpQ0FDRjs2QkFDRjt5QkFDRjtxQkFDRjtpQkFDRixDQUFDO1lBQ0YsdUJBQXVCLEVBQUU7Z0JBQ3ZCLEtBQUssRUFBRSxjQUFjO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsb0NBQW9DLENBQUMsSUFBVTtRQUM3QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUU7Z0JBQ1AsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLENBQUM7Z0JBQ3pELFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLG1CQUFtQixDQUFDLENBQUM7YUFDakU7WUFDRCxpQkFBaUIsRUFBRSxNQUFNO1NBQzFCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7WUFDM0QsY0FBYyxFQUFFO2dCQUNkLFlBQVksRUFBRTtvQkFDWixtRUFBbUU7b0JBQ25FLEtBQUs7aUJBQ047YUFDRjtZQUNELG1CQUFtQixFQUFFO2dCQUNuQjtvQkFDRSxLQUFLLEVBQUUsaUdBQWlHO2lCQUN6RztnQkFDRDtvQkFDRSxLQUFLLEVBQUUsaUdBQWlHO2lCQUN6RzthQUNGO1lBQ0Qsa0JBQWtCLEVBQUU7Z0JBQ2xCO29CQUNFLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKO2dEQUNFLEtBQUssRUFBRSxxR0FBcUc7NkNBQzdHO3lDQUNGO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKO2dEQUNFLEtBQUssRUFBRSxxR0FBcUc7NkNBQzdHO3lDQUNGO3FDQUNGO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2dCQUNEO29CQUNFLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKO2dEQUNFLEtBQUssRUFBRSxxR0FBcUc7NkNBQzdHO3lDQUNGO3FDQUNGO2lDQUNGOzZCQUNGOzRCQUNEO2dDQUNFLFlBQVksRUFBRTtvQ0FDWixDQUFDO29DQUNEO3dDQUNFLFdBQVcsRUFBRTs0Q0FDWCxJQUFJOzRDQUNKO2dEQUNFLEtBQUssRUFBRSxxR0FBcUc7NkNBQzdHO3lDQUNGO3FDQUNGO2lDQUNGOzZCQUNGO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCx1QkFBdUIsRUFBRTtnQkFDdkIsS0FBSyxFQUFFLGNBQWM7YUFDdEI7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRTVDLE9BQU87UUFDUCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDL0QsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUM7WUFDbEYsaUJBQWlCLEVBQUUsTUFBTTtTQUMxQixDQUFDLEVBQUUsdURBQXVELENBQUMsQ0FBQztRQUU3RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsK0JBQStCLENBQUMsSUFBVTtRQUN4QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7WUFDeEUsaUJBQWlCLEVBQUUsTUFBTTtTQUMxQixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNkJBQTZCLENBQUMsSUFBVTtRQUN0QyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUMsRUFBRTtvQkFDbEUsT0FBTyxFQUFFLENBQUMsR0FBRyxFQUFFLGFBQWEsQ0FBQztpQkFDOUIsQ0FBQyxDQUFDO1lBQ0gsaUJBQWlCLEVBQUUsTUFBTTtTQUMxQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUU7b0JBQ1osbUVBQW1FO29CQUNuRSxLQUFLO2lCQUNOO2FBQ0Y7WUFDRCxtQkFBbUIsRUFBRSxDQUFDO29CQUNwQixLQUFLLEVBQUUsaUdBQWlHO2lCQUN6RyxDQUFDO1lBQ0Ysa0JBQWtCLEVBQUUsQ0FBQztvQkFDbkIsVUFBVSxFQUFFO3dCQUNWLEVBQUU7d0JBQ0Y7NEJBQ0U7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0o7Z0RBQ0UsS0FBSyxFQUFFLHFHQUFxRzs2Q0FDN0c7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7NEJBQ0Q7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLENBQUM7b0NBQ0Q7d0NBQ0UsV0FBVyxFQUFFOzRDQUNYLElBQUk7NENBQ0o7Z0RBQ0UsS0FBSyxFQUFFLHFHQUFxRzs2Q0FDN0c7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0YsQ0FBQztZQUNGLHVCQUF1QixFQUFFO2dCQUN2QixLQUFLLEVBQUUsY0FBYzthQUN0QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBQ0osSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELHFFQUFxRSxDQUFDLElBQVU7UUFDOUUsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFFNUMsT0FBTztRQUNQLElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDN0MsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDO1lBQ3hFLGlCQUFpQixFQUFFLE1BQU07WUFDekIsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxjQUFjLEVBQUUsSUFBSTtTQUNyQixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCx3Q0FBd0MsQ0FBQyxJQUFVO1FBQ2pELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRTVDLE9BQU87UUFDUCxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQzdDLE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLGdCQUFnQixDQUFDLENBQUMsQ0FBQztZQUN4RSxpQkFBaUIsRUFBRSxNQUFNO1lBQ3pCLFFBQVEsRUFBRTtnQkFDUixDQUFDLEVBQUUsR0FBRztnQkFDTixDQUFDLEVBQUUsR0FBRzthQUNQO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLGVBQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMscUJBQVksQ0FBQyw2QkFBNkIsRUFBRTtZQUMzRCxZQUFZLEVBQUUsRUFBRSxlQUFlLEVBQUUsR0FBRyxFQUFFLGVBQWUsRUFBRSxHQUFHLEVBQUU7U0FDN0QsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7WUFDeEUsaUJBQWlCLEVBQUUsTUFBTTtZQUN6QixXQUFXLEVBQUUsV0FBVztZQUN4QixlQUFlLEVBQUUsSUFBSTtZQUNyQixZQUFZLEVBQUUsUUFBUSxDQUFDLFlBQVksQ0FBQyxtQkFBbUI7WUFDdkQsa0JBQWtCLEVBQUUsUUFBUTtZQUM1QixvQkFBb0IsRUFBRSxRQUFRLENBQUMsb0JBQW9CLENBQUMsT0FBTztZQUMzRCwrQkFBK0IsRUFBRSxPQUFPO1lBQ3hDLHFDQUFxQyxFQUFFLE9BQU87WUFDOUMsdUJBQXVCLEVBQUUsU0FBUztZQUNsQyxZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRSxFQUFFLFFBQVEsQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7WUFDdEcsT0FBTyxFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDO1NBQ3hELENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsNkJBQTZCLEVBQUU7WUFDM0QsY0FBYyxFQUFFO2dCQUNkLGNBQWMsRUFBRSxXQUFXO2dCQUMzQixrQkFBa0IsRUFBRSxJQUFJO2dCQUN4QixxQkFBcUIsRUFBRSxRQUFRO2dCQUMvQixlQUFlLEVBQUUscUJBQXFCO2dCQUN0QyxLQUFLLEVBQUUsU0FBUztnQkFDaEIsZ0JBQWdCLEVBQUUsT0FBTztnQkFDekIsZUFBZSxFQUFFLHNCQUFzQjtnQkFDdkMsU0FBUyxFQUFFLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsS0FBSztnQkFDL0QsbUJBQW1CLEVBQUUsT0FBTztnQkFDNUIsa0JBQWtCLEVBQUUsU0FBUzthQUM5QjtTQUNGLENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGlDQUFpQyxDQUFDLElBQVU7UUFDMUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQywrQkFBK0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLCtCQUErQixDQUFDLENBQUM7UUFDdEgsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsQ0FBQyxLQUFLLEVBQUUsK0JBQStCLENBQUMsQ0FBQztRQUMvRixJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsR0FBRyxNQUFNLENBQUMsQ0FBQyxHQUFHLEtBQUssRUFBRSw2Q0FBNkMsQ0FBQyxDQUFDO1FBQzdLLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUMsK0JBQStCLENBQUMsQ0FBQyxLQUFLLEVBQUUsK0JBQStCLENBQUMsQ0FBQztRQUVoSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUNBQXVDLENBQUMsSUFBVTtRQUNoRCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7UUFDNUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLE9BQU8sRUFBRSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztRQUM5RCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsV0FBVyxFQUFFLENBQUMsS0FBSyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxTQUFTLEVBQUUsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNoRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsZUFBZSxFQUFFLENBQUMsS0FBSyxFQUFFLGtCQUFrQixDQUFDLENBQUM7UUFDOUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztRQUN0RixJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ3pGLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsQ0FBQztRQUV2RixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsdUNBQXVDLENBQUMsSUFBVTtRQUNoRCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ3ZELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxrQkFBa0IsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDO1FBQzNFLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxXQUFXLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDN0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDLFVBQVUsRUFBRSxZQUFZLENBQUMsQ0FBQztRQUMzRCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsbUJBQW1CLEVBQUUscUJBQXFCLENBQUMsQ0FBQztRQUM3RSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsT0FBTyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ3JELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLFlBQVksQ0FBQyxZQUFZLEVBQUUsY0FBYyxDQUFDLENBQUM7UUFFL0QsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELGdEQUFnRCxDQUFDLElBQVU7UUFDekQsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsT0FBTyxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQzVELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sRUFBRSxTQUFTLENBQUMsQ0FBQztRQUU3RCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsZ0ZBQWdGLENBQUMsSUFBVTtRQUN6RixRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUM1QyxNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ25GLGFBQWEsRUFBRTtnQkFDYjtvQkFDRSxjQUFjLEVBQUU7d0JBQ2QsY0FBYyxFQUFFLE1BQU07cUJBQ3ZCO29CQUNELFNBQVMsRUFBRSxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ3pDO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7WUFDeEUsaUJBQWlCLEVBQUUsTUFBTTtZQUN6QixZQUFZO1lBQ1osaUJBQWlCLEVBQUUsQ0FBQyxXQUFXLENBQUM7U0FDakMsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsb0NBQW9DO2FBQzVDO1lBQ0QsaUJBQWlCLEVBQUUsQ0FBQyxXQUFXLENBQUM7U0FDakMsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsNERBQTRELENBQUMsSUFBVTtRQUNyRSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUM1QyxNQUFNLFlBQVksR0FBRyxJQUFJLFVBQVUsQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ25GLGFBQWEsRUFBRTtnQkFDYjtvQkFDRSxjQUFjLEVBQUU7d0JBQ2QsY0FBYyxFQUFFLE1BQU07cUJBQ3ZCO29CQUNELFNBQVMsRUFBRSxDQUFDLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFFLENBQUM7aUJBQ3pDO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7WUFDeEUsaUJBQWlCLEVBQUUsTUFBTTtZQUN6QixZQUFZO1NBQ2IsQ0FBQyxDQUFDO1FBRUgsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLDZCQUE2QixFQUFFO1lBQzNELGNBQWMsRUFBRTtnQkFDZCxLQUFLLEVBQUUsb0NBQW9DO2FBQzVDO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQsOERBQThELENBQUMsSUFBVTtRQUN2RSxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQy9ELE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFlBQVksRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDO1lBQ2xGLGlCQUFpQixFQUFFLE1BQU07WUFDekIsaUJBQWlCLEVBQUUsQ0FBQyxXQUFXLENBQUM7U0FDakMsQ0FBQyxFQUFFLG9FQUFvRSxDQUFDLENBQUM7UUFFMUUsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUVELHFHQUFxRyxDQUFDLElBQVU7UUFDOUcsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBQzlCLE1BQU0sTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7UUFDOUMsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUM3QyxPQUFPLEVBQUUsQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDckQsaUJBQWlCLEVBQUUsTUFBTTtTQUMxQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLGtCQUFrQixFQUFFO1lBQ2hELGdCQUFnQixFQUFFO2dCQUNoQixXQUFXLEVBQUU7b0JBQ1g7d0JBQ0UsUUFBUSxFQUFFOzRCQUNSLGVBQWU7NEJBQ2YsZUFBZTs0QkFDZixVQUFVO3lCQUNYO3dCQUNELFFBQVEsRUFBRSxPQUFPO3dCQUNqQixVQUFVLEVBQUU7NEJBQ1Y7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLGdCQUFnQjtvQ0FDaEIsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRDtnQ0FDRSxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRTs0Q0FDRSxZQUFZLEVBQUU7Z0RBQ1osZ0JBQWdCO2dEQUNoQixLQUFLOzZDQUNOO3lDQUNGO3dDQUNELElBQUk7cUNBQ0w7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7b0JBQ0Q7d0JBQ0UsUUFBUSxFQUFFOzRCQUNSLGVBQWU7NEJBQ2YsZUFBZTs0QkFDZixVQUFVOzRCQUNWLGtCQUFrQjs0QkFDbEIsZUFBZTs0QkFDZixXQUFXO3lCQUNaO3dCQUNELFFBQVEsRUFBRSxPQUFPO3dCQUNqQixVQUFVLEVBQUU7NEJBQ1Y7Z0NBQ0UsWUFBWSxFQUFFO29DQUNaLGNBQWM7b0NBQ2QsS0FBSztpQ0FDTjs2QkFDRjs0QkFDRDtnQ0FDRSxVQUFVLEVBQUU7b0NBQ1YsRUFBRTtvQ0FDRjt3Q0FDRTs0Q0FDRSxZQUFZLEVBQUU7Z0RBQ1osY0FBYztnREFDZCxLQUFLOzZDQUNOO3lDQUNGO3dDQUNELElBQUk7cUNBQ0w7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsU0FBUyxFQUFFLFlBQVk7YUFDeEI7WUFDRCxZQUFZLEVBQUUsMkZBQTJGO1lBQ3pHLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxLQUFLLEVBQUUsOEVBQThFO2lCQUN0RjthQUNGO1NBQ0YsQ0FBQyxDQUFDLENBQUM7UUFDSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBRUQseUZBQXlGLENBQUMsSUFBVTtRQUNsRyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFDOUIsTUFBTSxNQUFNLEdBQUcsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztRQUU1QyxPQUFPO1FBRVAsaUVBQWlFO1FBRWpFLElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDbEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNwRSxpQkFBaUIsRUFBRSxNQUFNO1lBQ3pCLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUVILElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUU7WUFDbEQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNwRSxpQkFBaUIsRUFBRSxNQUFNO1lBQ3pCLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUVILElBQUksUUFBUSxDQUFDLGdCQUFnQixDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUU7WUFDakQsT0FBTyxFQUFFLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNwRSxpQkFBaUIsRUFBRSxNQUFNO1lBQ3pCLFdBQVcsRUFBRSxJQUFJO1NBQ2xCLENBQUMsQ0FBQztRQUVILE9BQU87UUFFUCxrRUFBa0U7UUFFbEUsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0QsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFLEVBQUUsVUFBVSxFQUFFLEdBQUcsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUM3RSxlQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLHFCQUFZLENBQUMsdUJBQXVCLEVBQUUsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQzlFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFFRCw4Q0FBOEMsQ0FBQyxJQUFVO1FBRXZELFFBQVE7UUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM5QixNQUFNLE1BQU0sR0FBRyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQzVDLE1BQU0sWUFBWSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFO1lBQy9DLFNBQVMsRUFBRSxJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztTQUN6RCxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsSUFBSSxRQUFRLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1lBQ3JELE9BQU8sRUFBRSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFlBQVksQ0FBQyxDQUFDLENBQUM7WUFDcEUsaUJBQWlCLEVBQUUsTUFBTTtZQUN6QixJQUFJLEVBQUUsWUFBWTtTQUNuQixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEQsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyx1QkFBYyxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDN0QsZUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxxQkFBWSxDQUFDLHVCQUF1QixFQUFFO1lBQ3JELE1BQU0sRUFBRTtnQkFDTixZQUFZLEVBQUU7b0JBQ1osY0FBYztvQkFDZCxLQUFLO2lCQUNOO2FBQ0Y7U0FDRixDQUFDLENBQUMsQ0FBQztRQUNKLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY291bnRSZXNvdXJjZXMsIGV4cGVjdCwgaGF2ZVJlc291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFRlc3QgfSBmcm9tICdub2RldW5pdCc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgczNkZXBsb3kgZnJvbSAnLi4vbGliJztcblxuLy8gdHNsaW50OmRpc2FibGU6bWF4LWxpbmUtbGVuZ3RoXG4vLyB0c2xpbnQ6ZGlzYWJsZTpvYmplY3QtbGl0ZXJhbC1rZXktcXVvdGVzXG5cbmV4cG9ydCA9IHtcbiAgJ2RlcGxveSBmcm9tIGxvY2FsIGRpcmVjdG9yeSBhc3NldCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUnKSldLFxuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IGJ1Y2tldCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgJ1NlcnZpY2VUb2tlbic6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0N1c3RvbUNES0J1Y2tldERlcGxveW1lbnQ4NjkzQkI2NDk2ODk0NEI2OUFBRkIwQ0M5RUI4NzU2QzgxQzAxNTM2JyxcbiAgICAgICAgICAnQXJuJyxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICAnU291cmNlQnVja2V0TmFtZXMnOiBbe1xuICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2ZjNDQ4MWFiZjI3OTI1NTYxOWZmNzQxOGZhYTVkMjQ0NTZmZWYzNDMyZWEwZGE1OWM5NTU0MjU3OGZmMDIyMmVTM0J1Y2tldDlDRDhCMjBBJyxcbiAgICAgIH1dLFxuICAgICAgJ1NvdXJjZU9iamVjdEtleXMnOiBbe1xuICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgJycsXG4gICAgICAgICAgW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2ZjNDQ4MWFiZjI3OTI1NTYxOWZmNzQxOGZhYTVkMjQ0NTZmZWYzNDMyZWEwZGE1OWM5NTU0MjU3OGZmMDIyMmVTM1ZlcnNpb25LZXlBNThEMzgwQycsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAxLFxuICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICd8fCcsXG4gICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2ZjNDQ4MWFiZjI3OTI1NTYxOWZmNzQxOGZhYTVkMjQ0NTZmZWYzNDMyZWEwZGE1OWM5NTU0MjU3OGZmMDIyMmVTM1ZlcnNpb25LZXlBNThEMzgwQycsXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgIF0sXG4gICAgICB9XSxcbiAgICAgICdEZXN0aW5hdGlvbkJ1Y2tldE5hbWUnOiB7XG4gICAgICAgICdSZWYnOiAnRGVzdEMzODNCODJBJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZXBsb3kgZnJvbSBsb2NhbCBkaXJlY3RvcnkgYXNzZXRzJyh0ZXN0OiBUZXN0KSB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnRGVzdCcpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHN0YWNrLCAnRGVwbG95Jywge1xuICAgICAgc291cmNlczogW1xuICAgICAgICBzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUnKSksXG4gICAgICAgIHMzZGVwbG95LlNvdXJjZS5hc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbXktd2Vic2l0ZS1zZWNvbmQnKSksXG4gICAgICBdLFxuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IGJ1Y2tldCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgJ1NlcnZpY2VUb2tlbic6IHtcbiAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgJ0N1c3RvbUNES0J1Y2tldERlcGxveW1lbnQ4NjkzQkI2NDk2ODk0NEI2OUFBRkIwQ0M5RUI4NzU2QzgxQzAxNTM2JyxcbiAgICAgICAgICAnQXJuJyxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICAnU291cmNlQnVja2V0TmFtZXMnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2ZjNDQ4MWFiZjI3OTI1NTYxOWZmNzQxOGZhYTVkMjQ0NTZmZWYzNDMyZWEwZGE1OWM5NTU0MjU3OGZmMDIyMmVTM0J1Y2tldDlDRDhCMjBBJyxcbiAgICAgICAgfSxcbiAgICAgICAge1xuICAgICAgICAgICdSZWYnOiAnQXNzZXRQYXJhbWV0ZXJzYTk0OTc3ZWRlMDIxMWZkM2I0NWVmYTMzZDZkOGQxZDdiYmUwYzVhOTZkOTc3MTM5ZDhiMTZhYmZhOTZmZTljYlMzQnVja2V0OTk3OTM1NTknLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgICdTb3VyY2VPYmplY3RLZXlzJzogW1xuICAgICAgICB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2ZjNDQ4MWFiZjI3OTI1NTYxOWZmNzQxOGZhYTVkMjQ0NTZmZWYzNDMyZWEwZGE1OWM5NTU0MjU3OGZmMDIyMmVTM1ZlcnNpb25LZXlBNThEMzgwQycsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdSZWYnOiAnQXNzZXRQYXJhbWV0ZXJzZmM0NDgxYWJmMjc5MjU1NjE5ZmY3NDE4ZmFhNWQyNDQ1NmZlZjM0MzJlYTBkYTU5Yzk1NTQyNTc4ZmYwMjIyZVMzVmVyc2lvbktleUE1OEQzODBDJyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgICB7XG4gICAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OlNlbGVjdCc6IFtcbiAgICAgICAgICAgICAgICAgIDAsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICdGbjo6U3BsaXQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgJ3x8JyxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAnUmVmJzogJ0Fzc2V0UGFyYW1ldGVyc2E5NDk3N2VkZTAyMTFmZDNiNDVlZmEzM2Q2ZDhkMWQ3YmJlMGM1YTk2ZDk3NzEzOWQ4YjE2YWJmYTk2ZmU5Y2JTM1ZlcnNpb25LZXlEOUFDRTY2NScsXG4gICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6U2VsZWN0JzogW1xuICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgJ0ZuOjpTcGxpdCc6IFtcbiAgICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICdSZWYnOiAnQXNzZXRQYXJhbWV0ZXJzYTk0OTc3ZWRlMDIxMWZkM2I0NWVmYTMzZDZkOGQxZDdiYmUwYzVhOTZkOTc3MTM5ZDhiMTZhYmZhOTZmZTljYlMzVmVyc2lvbktleUQ5QUNFNjY1JyxcbiAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICBdLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgICdEZXN0aW5hdGlvbkJ1Y2tldE5hbWUnOiB7XG4gICAgICAgICdSZWYnOiAnRGVzdEMzODNCODJBJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdmYWlscyBpZiBsb2NhbCBhc3NldCBpcyBhIG5vbi16aXAgZmlsZScodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUnLCAnaW5kZXguaHRtbCcpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgIH0pLCAvQXNzZXQgcGF0aCBtdXN0IGJlIGVpdGhlciBhIFxcLnppcCBmaWxlIG9yIGEgZGlyZWN0b3J5Lyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnZGVwbG95IGZyb20gYSBsb2NhbCAuemlwIGZpbGUnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdEZXN0Jyk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHMzZGVwbG95LkJ1Y2tldERlcGxveW1lbnQoc3RhY2ssICdEZXBsb3knLCB7XG4gICAgICBzb3VyY2VzOiBbczNkZXBsb3kuU291cmNlLmFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS13ZWJzaXRlLnppcCcpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgIH0pO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2hvbm9ycyBwYXNzZWQgYXNzZXQgb3B0aW9ucycodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUnKSwge1xuICAgICAgICBleGNsdWRlOiBbJyonLCAnIWluZGV4Lmh0bWwnXSxcbiAgICAgIH0pXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBidWNrZXQsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6Q0RLQnVja2V0RGVwbG95bWVudCcsIHtcbiAgICAgICdTZXJ2aWNlVG9rZW4nOiB7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdDdXN0b21DREtCdWNrZXREZXBsb3ltZW50ODY5M0JCNjQ5Njg5NDRCNjlBQUZCMENDOUVCODc1NkM4MUMwMTUzNicsXG4gICAgICAgICAgJ0FybicsXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgJ1NvdXJjZUJ1Y2tldE5hbWVzJzogW3tcbiAgICAgICAgJ1JlZic6ICdBc3NldFBhcmFtZXRlcnNlOWI2OTZiMmE4YTFmOTNlYThiOGE5Y2UxZTRkZDQ3MjdmOTI0M2ViYTk4NGU1MDQxMWNhOTVjNmIwM2QyNmI2UzNCdWNrZXQxQTFFQzNFOScsXG4gICAgICB9XSxcbiAgICAgICdTb3VyY2VPYmplY3RLZXlzJzogW3tcbiAgICAgICAgJ0ZuOjpKb2luJzogW1xuICAgICAgICAgICcnLFxuICAgICAgICAgIFtcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgMCxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgJ1JlZic6ICdBc3NldFBhcmFtZXRlcnNlOWI2OTZiMmE4YTFmOTNlYThiOGE5Y2UxZTRkZDQ3MjdmOTI0M2ViYTk4NGU1MDQxMWNhOTVjNmIwM2QyNmI2UzNWZXJzaW9uS2V5RTQ2QTQ4MjQnLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgJ0ZuOjpTZWxlY3QnOiBbXG4gICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAnRm46OlNwbGl0JzogW1xuICAgICAgICAgICAgICAgICAgICAnfHwnLFxuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgJ1JlZic6ICdBc3NldFBhcmFtZXRlcnNlOWI2OTZiMmE4YTFmOTNlYThiOGE5Y2UxZTRkZDQ3MjdmOTI0M2ViYTk4NGU1MDQxMWNhOTVjNmIwM2QyNmI2UzNWZXJzaW9uS2V5RTQ2QTQ4MjQnLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICBdLFxuICAgICAgICBdLFxuICAgICAgfV0sXG4gICAgICAnRGVzdGluYXRpb25CdWNrZXROYW1lJzoge1xuICAgICAgICAnUmVmJzogJ0Rlc3RDMzgzQjgyQScsXG4gICAgICB9LFxuICAgIH0pKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3JldGFpbk9uRGVsZXRlIGNhbiBiZSB1c2VkIHRvIHJldGFpbiBmaWxlcyB3aGVuIHJlc291cmNlIGlzIGRlbGV0ZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdEZXN0Jyk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHMzZGVwbG95LkJ1Y2tldERlcGxveW1lbnQoc3RhY2ssICdEZXBsb3knLCB7XG4gICAgICBzb3VyY2VzOiBbczNkZXBsb3kuU291cmNlLmFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS13ZWJzaXRlLnppcCcpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgcmV0YWluT25EZWxldGU6IHRydWUsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6Q0RLQnVja2V0RGVwbG95bWVudCcsIHtcbiAgICAgIFJldGFpbk9uRGVsZXRlOiB0cnVlLFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICd1c2VyIG1ldGFkYXRhIGlzIGNvcnJlY3RseSB0cmFuc2Zvcm1lZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUuemlwJykpXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBidWNrZXQsXG4gICAgICBtZXRhZGF0YToge1xuICAgICAgICBBOiAnMScsXG4gICAgICAgIEI6ICcyJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50byhoYXZlUmVzb3VyY2UoJ0N1c3RvbTo6Q0RLQnVja2V0RGVwbG95bWVudCcsIHtcbiAgICAgIFVzZXJNZXRhZGF0YTogeyAneC1hbXpuLW1ldGEtYSc6ICcxJywgJ3gtYW16bi1tZXRhLWInOiAnMicgfSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3lzdGVtIG1ldGFkYXRhIGlzIGNvcnJlY3RseSB0cmFuc2Zvcm1lZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUuemlwJykpXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBidWNrZXQsXG4gICAgICBjb250ZW50VHlwZTogJ3RleHQvaHRtbCcsXG4gICAgICBjb250ZW50TGFuZ3VhZ2U6ICdlbicsXG4gICAgICBzdG9yYWdlQ2xhc3M6IHMzZGVwbG95LlN0b3JhZ2VDbGFzcy5JTlRFTExJR0VOVF9USUVSSU5HLFxuICAgICAgY29udGVudERpc3Bvc2l0aW9uOiAnaW5saW5lJyxcbiAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uOiBzM2RlcGxveS5TZXJ2ZXJTaWRlRW5jcnlwdGlvbi5BV1NfS01TLFxuICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25Bd3NLbXNLZXlJZDogJ215a2V5JyxcbiAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uQ3VzdG9tZXJBbGdvcml0aG06ICdyb3QxMycsXG4gICAgICB3ZWJzaXRlUmVkaXJlY3RMb2NhdGlvbjogJ2V4YW1wbGUnLFxuICAgICAgY2FjaGVDb250cm9sOiBbczNkZXBsb3kuQ2FjaGVDb250cm9sLnNldFB1YmxpYygpLCBzM2RlcGxveS5DYWNoZUNvbnRyb2wubWF4QWdlKGNkay5EdXJhdGlvbi5ob3VycygxKSldLFxuICAgICAgZXhwaXJlczogczNkZXBsb3kuRXhwaXJlcy5hZnRlcihjZGsuRHVyYXRpb24uaG91cnMoMTIpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgU3lzdGVtTWV0YWRhdGE6IHtcbiAgICAgICAgJ2NvbnRlbnQtdHlwZSc6ICd0ZXh0L2h0bWwnLFxuICAgICAgICAnY29udGVudC1sYW5ndWFnZSc6ICdlbicsXG4gICAgICAgICdjb250ZW50LWRpc3Bvc2l0aW9uJzogJ2lubGluZScsXG4gICAgICAgICdzdG9yYWdlLWNsYXNzJzogJ0lOVEVMTElHRU5UX1RJRVJJTkcnLFxuICAgICAgICAnc3NlJzogJ2F3czprbXMnLFxuICAgICAgICAnc3NlLWttcy1rZXktaWQnOiAnbXlrZXknLFxuICAgICAgICAnY2FjaGUtY29udHJvbCc6ICdwdWJsaWMsIG1heC1hZ2U9MzYwMCcsXG4gICAgICAgICdleHBpcmVzJzogczNkZXBsb3kuRXhwaXJlcy5hZnRlcihjZGsuRHVyYXRpb24uaG91cnMoMTIpKS52YWx1ZSxcbiAgICAgICAgJ3NzZS1jLWNvcHktc291cmNlJzogJ3JvdDEzJyxcbiAgICAgICAgJ3dlYnNpdGUtcmVkaXJlY3QnOiAnZXhhbXBsZScsXG4gICAgICB9LFxuICAgIH0pKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdleHBpcmVzIHR5cGUgaGFzIGNvcnJlY3QgdmFsdWVzJyh0ZXN0OiBUZXN0KSB7XG4gICAgdGVzdC5lcXVhbChzM2RlcGxveS5FeHBpcmVzLmF0RGF0ZShuZXcgRGF0ZSgnU3VuLCAyNiBKYW4gMjAyMCAwMDo1MzoyMCBHTVQnKSkudmFsdWUsICdTdW4sIDI2IEphbiAyMDIwIDAwOjUzOjIwIEdNVCcpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuRXhwaXJlcy5hdFRpbWVzdGFtcCgxNTgwMDAwMDAwMDAwKS52YWx1ZSwgJ1N1biwgMjYgSmFuIDIwMjAgMDA6NTM6MjAgR01UJyk7XG4gICAgdGVzdC5vayhNYXRoLmFicyhuZXcgRGF0ZShzM2RlcGxveS5FeHBpcmVzLmFmdGVyKGNkay5EdXJhdGlvbi5taW51dGVzKDEwKSkudmFsdWUpLmdldFRpbWUoKSAtIChEYXRlLm5vdygpICsgNjAwMDAwKSkgPCAxNTAwMCwgJ0V4cGlyZXMuYWZ0ZXIgYWNjdXJhdGUgdG8gd2l0aGluIDE1IHNlY29uZHMnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LkV4cGlyZXMuZnJvbVN0cmluZygnVHVlLCAwNCBGZWIgMjAyMCAwODo0NTozMyBHTVQnKS52YWx1ZSwgJ1R1ZSwgMDQgRmViIDIwMjAgMDg6NDU6MzMgR01UJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnY2FjaGUgY29udHJvbCB0eXBlIGhhcyBjb3JyZWN0IHZhbHVlcycodGVzdDogVGVzdCkge1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuQ2FjaGVDb250cm9sLm11c3RSZXZhbGlkYXRlKCkudmFsdWUsICdtdXN0LXJldmFsaWRhdGUnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LkNhY2hlQ29udHJvbC5ub0NhY2hlKCkudmFsdWUsICduby1jYWNoZScpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuQ2FjaGVDb250cm9sLm5vVHJhbnNmb3JtKCkudmFsdWUsICduby10cmFuc2Zvcm0nKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LkNhY2hlQ29udHJvbC5zZXRQdWJsaWMoKS52YWx1ZSwgJ3B1YmxpYycpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuQ2FjaGVDb250cm9sLnNldFByaXZhdGUoKS52YWx1ZSwgJ3ByaXZhdGUnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LkNhY2hlQ29udHJvbC5wcm94eVJldmFsaWRhdGUoKS52YWx1ZSwgJ3Byb3h5LXJldmFsaWRhdGUnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LkNhY2hlQ29udHJvbC5tYXhBZ2UoY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMSkpLnZhbHVlLCAnbWF4LWFnZT02MCcpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuQ2FjaGVDb250cm9sLnNNYXhBZ2UoY2RrLkR1cmF0aW9uLm1pbnV0ZXMoMSkpLnZhbHVlLCAncy1tYXgtYWdlPTYwJyk7XG4gICAgdGVzdC5lcXVhbChzM2RlcGxveS5DYWNoZUNvbnRyb2wuZnJvbVN0cmluZygnb25seS1pZi1jYWNoZWQnKS52YWx1ZSwgJ29ubHktaWYtY2FjaGVkJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc3RvcmFnZSBjbGFzcyB0eXBlIGhhcyBjb3JyZWN0IHZhbHVlcycodGVzdDogVGVzdCkge1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuU3RvcmFnZUNsYXNzLlNUQU5EQVJELCAnU1RBTkRBUkQnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LlN0b3JhZ2VDbGFzcy5SRURVQ0VEX1JFRFVOREFOQ1ksICdSRURVQ0VEX1JFRFVOREFOQ1knKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LlN0b3JhZ2VDbGFzcy5TVEFOREFSRF9JQSwgJ1NUQU5EQVJEX0lBJyk7XG4gICAgdGVzdC5lcXVhbChzM2RlcGxveS5TdG9yYWdlQ2xhc3MuT05FWk9ORV9JQSwgJ09ORVpPTkVfSUEnKTtcbiAgICB0ZXN0LmVxdWFsKHMzZGVwbG95LlN0b3JhZ2VDbGFzcy5JTlRFTExJR0VOVF9USUVSSU5HLCAnSU5URUxMSUdFTlRfVElFUklORycpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuU3RvcmFnZUNsYXNzLkdMQUNJRVIsICdHTEFDSUVSJyk7XG4gICAgdGVzdC5lcXVhbChzM2RlcGxveS5TdG9yYWdlQ2xhc3MuREVFUF9BUkNISVZFLCAnREVFUF9BUkNISVZFJyk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnc2VydmVyIHNpZGUgZW5jcnlwdGlvbiB0eXBlIGhhcyBjb3JyZWN0IHZhbHVlcycodGVzdDogVGVzdCkge1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuU2VydmVyU2lkZUVuY3J5cHRpb24uQUVTXzI1NiwgJ0FFUzI1NicpO1xuICAgIHRlc3QuZXF1YWwoczNkZXBsb3kuU2VydmVyU2lkZUVuY3J5cHRpb24uQVdTX0tNUywgJ2F3czprbXMnKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkaXN0cmlidXRpb24gY2FuIGJlIHVzZWQgdG8gcHJvdmlkZSBhIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIGZvciBpbnZhbGlkYXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IGJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdEZXN0Jyk7XG4gICAgY29uc3QgZGlzdHJpYnV0aW9uID0gbmV3IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvbihzdGFjaywgJ0Rpc3RyaWJ1dGlvbicsIHtcbiAgICAgIG9yaWdpbkNvbmZpZ3M6IFtcbiAgICAgICAge1xuICAgICAgICAgIHMzT3JpZ2luU291cmNlOiB7XG4gICAgICAgICAgICBzM0J1Y2tldFNvdXJjZTogYnVja2V0LFxuICAgICAgICAgIH0sXG4gICAgICAgICAgYmVoYXZpb3JzOiBbeyBpc0RlZmF1bHRCZWhhdmlvcjogdHJ1ZSB9XSxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHMzZGVwbG95LkJ1Y2tldERlcGxveW1lbnQoc3RhY2ssICdEZXBsb3knLCB7XG4gICAgICBzb3VyY2VzOiBbczNkZXBsb3kuU291cmNlLmFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS13ZWJzaXRlLnppcCcpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgZGlzdHJpYnV0aW9uLFxuICAgICAgZGlzdHJpYnV0aW9uUGF0aHM6IFsnL2ltYWdlcy8qJ10sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgRGlzdHJpYnV0aW9uSWQ6IHtcbiAgICAgICAgJ1JlZic6ICdEaXN0cmlidXRpb25DRkRpc3RyaWJ1dGlvbjg4MkE3MzEzJyxcbiAgICAgIH0sXG4gICAgICBEaXN0cmlidXRpb25QYXRoczogWycvaW1hZ2VzLyonXSxcbiAgICB9KSk7XG5cbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcblxuICAnaW52YWxpZGF0aW9uIGNhbiBoYXBwZW4gd2l0aG91dCBkaXN0cmlidXRpb25QYXRocyBwcm92aWRlZCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcbiAgICBjb25zdCBkaXN0cmlidXRpb24gPSBuZXcgY2xvdWRmcm9udC5DbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uKHN0YWNrLCAnRGlzdHJpYnV0aW9uJywge1xuICAgICAgb3JpZ2luQ29uZmlnczogW1xuICAgICAgICB7XG4gICAgICAgICAgczNPcmlnaW5Tb3VyY2U6IHtcbiAgICAgICAgICAgIHMzQnVja2V0U291cmNlOiBidWNrZXQsXG4gICAgICAgICAgfSxcbiAgICAgICAgICBiZWhhdmlvcnM6IFt7IGlzRGVmYXVsdEJlaGF2aW9yOiB0cnVlIH1dLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUuemlwJykpXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBidWNrZXQsXG4gICAgICBkaXN0cmlidXRpb24sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50Jywge1xuICAgICAgRGlzdHJpYnV0aW9uSWQ6IHtcbiAgICAgICAgJ1JlZic6ICdEaXN0cmlidXRpb25DRkRpc3RyaWJ1dGlvbjg4MkE3MzEzJyxcbiAgICAgIH0sXG4gICAgfSkpO1xuXG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ2ZhaWxzIGlmIGRpc3RyaWJ1dGlvbiBwYXRocyBwcm92aWRlZCBidXQgbm90IGRpc3RyaWJ1dGlvbiBJRCcodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFRIRU5cbiAgICB0ZXN0LnRocm93cygoKSA9PiBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveScsIHtcbiAgICAgIHNvdXJjZXM6IFtzM2RlcGxveS5Tb3VyY2UuYXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ215LXdlYnNpdGUnLCAnaW5kZXguaHRtbCcpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgZGlzdHJpYnV0aW9uUGF0aHM6IFsnL2ltYWdlcy8qJ10sXG4gICAgfSksIC9EaXN0cmlidXRpb24gbXVzdCBiZSBzcGVjaWZpZWQgaWYgZGlzdHJpYnV0aW9uIHBhdGhzIGFyZSBzcGVjaWZpZWQvKTtcblxuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdsYW1iZGEgZXhlY3V0aW9uIHJvbGUgZ2V0cyBwZXJtaXNzaW9ucyB0byByZWFkIGZyb20gdGhlIHNvdXJjZSBidWNrZXQgYW5kIHJlYWQvd3JpdGUgaW4gZGVzdGluYXRpb24nKHRlc3Q6IFRlc3QpIHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgIGNvbnN0IHNvdXJjZSA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdTb3VyY2UnKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnRGVzdCcpO1xuXG4gICAgLy8gV0hFTlxuICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHN0YWNrLCAnRGVwbG95Jywge1xuICAgICAgc291cmNlczogW3MzZGVwbG95LlNvdXJjZS5idWNrZXQoc291cmNlLCAnZmlsZS56aXAnKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgJ1BvbGljeURvY3VtZW50Jzoge1xuICAgICAgICAnU3RhdGVtZW50JzogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgICdBY3Rpb24nOiBbXG4gICAgICAgICAgICAgICdzMzpHZXRPYmplY3QqJyxcbiAgICAgICAgICAgICAgJ3MzOkdldEJ1Y2tldConLFxuICAgICAgICAgICAgICAnczM6TGlzdConLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgJ1Jlc291cmNlJzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAnU291cmNlNzFFNDcxRjEnLFxuICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICdGbjo6Sm9pbic6IFtcbiAgICAgICAgICAgICAgICAgICcnLFxuICAgICAgICAgICAgICAgICAgW1xuICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAnU291cmNlNzFFNDcxRjEnLFxuICAgICAgICAgICAgICAgICAgICAgICAgJ0FybicsXG4gICAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgJy8qJyxcbiAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICB7XG4gICAgICAgICAgICAnQWN0aW9uJzogW1xuICAgICAgICAgICAgICAnczM6R2V0T2JqZWN0KicsXG4gICAgICAgICAgICAgICdzMzpHZXRCdWNrZXQqJyxcbiAgICAgICAgICAgICAgJ3MzOkxpc3QqJyxcbiAgICAgICAgICAgICAgJ3MzOkRlbGV0ZU9iamVjdConLFxuICAgICAgICAgICAgICAnczM6UHV0T2JqZWN0KicsXG4gICAgICAgICAgICAgICdzMzpBYm9ydConLFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICdFZmZlY3QnOiAnQWxsb3cnLFxuICAgICAgICAgICAgJ1Jlc291cmNlJzogW1xuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgJ0ZuOjpHZXRBdHQnOiBbXG4gICAgICAgICAgICAgICAgICAnRGVzdEMzODNCODJBJyxcbiAgICAgICAgICAgICAgICAgICdBcm4nLFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAnRm46OkpvaW4nOiBbXG4gICAgICAgICAgICAgICAgICAnJyxcbiAgICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICAgICAgICAgICAgICAgJ0Rlc3RDMzgzQjgyQScsXG4gICAgICAgICAgICAgICAgICAgICAgICAnQXJuJyxcbiAgICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAnLyonLFxuICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICB9LFxuICAgICAgICBdLFxuICAgICAgICAnVmVyc2lvbic6ICcyMDEyLTEwLTE3JyxcbiAgICAgIH0sXG4gICAgICAnUG9saWN5TmFtZSc6ICdDdXN0b21DREtCdWNrZXREZXBsb3ltZW50ODY5M0JCNjQ5Njg5NDRCNjlBQUZCMENDOUVCODc1NkNTZXJ2aWNlUm9sZURlZmF1bHRQb2xpY3k4ODkwMkZERicsXG4gICAgICAnUm9sZXMnOiBbXG4gICAgICAgIHtcbiAgICAgICAgICAnUmVmJzogJ0N1c3RvbUNES0J1Y2tldERlcGxveW1lbnQ4NjkzQkI2NDk2ODk0NEI2OUFBRkIwQ0M5RUI4NzU2Q1NlcnZpY2VSb2xlODlBMDEyNjUnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG5cbiAgJ21lbW9yeUxpbWl0IGNhbiBiZSB1c2VkIHRvIHNwZWNpZnkgdGhlIG1lbW9yeSBsaW1pdCBmb3IgdGhlIGRlcGxveW1lbnQgcmVzb3VyY2UgaGFuZGxlcicodGVzdDogVGVzdCkge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgY29uc3QgYnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ0Rlc3QnKTtcblxuICAgIC8vIFdIRU5cblxuICAgIC8vIHdlIGRlZmluZSAzIGRlcGxveW1lbnRzIHdpdGggMiBkaWZmZXJlbnQgbWVtb3J5IGNvbmZpZ3VyYXRpb25zXG5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveTI1Ni0xJywge1xuICAgICAgc291cmNlczogW3MzZGVwbG95LlNvdXJjZS5hc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbXktd2Vic2l0ZScpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgbWVtb3J5TGltaXQ6IDI1NixcbiAgICB9KTtcblxuICAgIG5ldyBzM2RlcGxveS5CdWNrZXREZXBsb3ltZW50KHN0YWNrLCAnRGVwbG95MjU2LTInLCB7XG4gICAgICBzb3VyY2VzOiBbczNkZXBsb3kuU291cmNlLmFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdteS13ZWJzaXRlJykpXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBidWNrZXQsXG4gICAgICBtZW1vcnlMaW1pdDogMjU2LFxuICAgIH0pO1xuXG4gICAgbmV3IHMzZGVwbG95LkJ1Y2tldERlcGxveW1lbnQoc3RhY2ssICdEZXBsb3kxMDI0Jywge1xuICAgICAgc291cmNlczogW3MzZGVwbG95LlNvdXJjZS5hc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbXktd2Vic2l0ZScpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgbWVtb3J5TGltaXQ6IDEwMjQsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG5cbiAgICAvLyB3ZSBleHBlY3QgdG8gZmluZCBvbmx5IHR3byBoYW5kbGVycywgb25lIGZvciBlYWNoIGNvbmZpZ3VyYXRpb25cblxuICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIDIpKTtcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgeyBNZW1vcnlTaXplOiAyNTYgfSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oaGF2ZVJlc291cmNlKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7IE1lbW9yeVNpemU6IDEwMjQgfSkpO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuXG4gICdkZXBsb3ltZW50IGFsbG93cyBjdXN0b20gcm9sZSB0byBiZSBzdXBwbGllZCcodGVzdDogVGVzdCkge1xuXG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICBjb25zdCBidWNrZXQgPSBuZXcgczMuQnVja2V0KHN0YWNrLCAnRGVzdCcpO1xuICAgIGNvbnN0IGV4aXN0aW5nUm9sZSA9IG5ldyBpYW0uUm9sZShzdGFjaywgJ1JvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbGFtYmRhLmFtYXpvbi5jb20nKSxcbiAgICB9KTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgczNkZXBsb3kuQnVja2V0RGVwbG95bWVudChzdGFjaywgJ0RlcGxveVdpdGhSb2xlJywge1xuICAgICAgc291cmNlczogW3MzZGVwbG95LlNvdXJjZS5hc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnbXktd2Vic2l0ZScpKV0sXG4gICAgICBkZXN0aW5hdGlvbkJ1Y2tldDogYnVja2V0LFxuICAgICAgcm9sZTogZXhpc3RpbmdSb2xlLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6SUFNOjpSb2xlJywgMSkpO1xuICAgIGV4cGVjdChzdGFjaykudG8oY291bnRSZXNvdXJjZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIDEpKTtcbiAgICBleHBlY3Qoc3RhY2spLnRvKGhhdmVSZXNvdXJjZSgnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywge1xuICAgICAgJ1JvbGUnOiB7XG4gICAgICAgICdGbjo6R2V0QXR0JzogW1xuICAgICAgICAgICdSb2xlMUFCQ0M1RjAnLFxuICAgICAgICAgICdBcm4nLFxuICAgICAgICBdLFxuICAgICAgfSxcbiAgICB9KSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG59O1xuIl19