"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const s3 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
test('check for default props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'solutions/constructs'",
                actions: []
            }
        }
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    const template = assertions_1.Template.fromStack(stack);
    // Check whether construct has two s3 buckets for storing msgs and logging
    template.resourceCountIs('AWS::S3::Bucket', 2);
    // Check for IoT Topic Rule Definition
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "testiots3integrationS3Bucket9B8B180C"
                        },
                        Key: "${topic()}/${timestamp()}",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'solutions/constructs'"
        }
    });
    // Check for IAM policy to have access to s3 bucket
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    template.resourceCountIs('AWS::IAM::Policy', 1);
    // Check for properties
    expect(construct.s3Bucket).toBeDefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for overriden props', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: true,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        s3Key: 'test/key',
        bucketProps: {
            encryption: s3.BucketEncryption.KMS
        },
        loggingBucketProps: {
            encryption: s3.BucketEncryption.KMS_MANAGED
        }
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    // Check whether construct has two s3 buckets for storing msgs and logging
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::S3::Bucket', 2);
    // Check logging bucket encryption type to be KMS_Managed
    template.hasResourceProperties('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        }
    });
    // Check for bucket to have KMS CMK Encryption
    template.hasResourceProperties('AWS::S3::Bucket', {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            "Fn::GetAtt": [
                                "testiots3integrationS3BucketKey127368C9",
                                "Arn"
                            ]
                        },
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        },
    });
    // Check for IoT Topic Rule Definition
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "testiots3integrationS3Bucket9B8B180C"
                        },
                        Key: "test/key",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: true,
            Sql: "SELECT * FROM 'test/constructs'"
        }
    });
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    // Check for automatically created CMK KMS Key
    template.resourceCountIs('AWS::KMS::Key', 1);
    // Check for IoT Topic Rule permissions to KMS key to store msgs to S3 Bucket and access to put data to s3 bucket
    template.resourceCountIs('AWS::IAM::Policy', 1);
    // Check for properties
    expect(construct.s3Bucket).toBeDefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeDefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for existing bucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = new s3.Bucket(stack, `existingBucket`);
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        s3Key: 'existingtest/key',
        existingBucketInterface: existingBucket
    };
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    // Check whether construct has a single s3 bucket, no logging bucket should exist since existing bucket is supplied
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::S3::Bucket', 1);
    // Check for IoT Topic Rule Definition with existing Bucket Ref
    template.hasResourceProperties('AWS::IoT::TopicRule', {
        TopicRulePayload: {
            Actions: [
                {
                    S3: {
                        BucketName: {
                            Ref: "existingBucket9529822F"
                        },
                        Key: "existingtest/key",
                        RoleArn: {
                            "Fn::GetAtt": [
                                "testiots3integrationiotactionsrole04473665",
                                "Arn"
                            ]
                        }
                    }
                }
            ],
            Description: "process solutions constructs messages",
            RuleDisabled: false,
            Sql: "SELECT * FROM 'test/constructs'"
        }
    });
    /**
     * Due to difference in CDK V1 and V2 Synth, the policy documents doesn't match, hence checking only for number of policies
     */
    // Check for IAM policy to have access to s3 bucket
    template.resourceCountIs('AWS::IAM::Policy', 1);
    // since existing bucket is supplied, no key should exist
    defaults.expectNonexistence(stack, 'AWS::KMS::Key', {});
    // Check for IoT Topic Rule permissions to KMS key to store msgs to S3 Bucket
    template.resourceCountIs("AWS::IAM::Policy", 1);
    // Check for properties
    expect(construct.s3Bucket).toBeUndefined();
    expect(construct.s3BucketInterface).toBeDefined();
    expect(construct.s3LoggingBucket).toBeUndefined();
    expect(construct.iotActionsRole).toBeDefined();
    expect(construct.iotTopicRule).toBeDefined();
});
test('check for both bucketProps and existingBucket', () => {
    const stack = new cdk.Stack();
    const existingBucket = new s3.Bucket(stack, `existingBucket`, { encryption: s3.BucketEncryption.KMS });
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        bucketProps: {
            encryption: s3.BucketEncryption.KMS_MANAGED
        },
        existingBucketInterface: existingBucket
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    try {
        new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check for name collision', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        bucketProps: {
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
        }
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    new lib_1.IotToS3(stack, 'test-iot-s3-integration1', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::IoT::TopicRule', 2);
    template.resourceCountIs('AWS::S3::Bucket', 4);
});
test('check for chaining of resource', () => {
    const stack = new cdk.Stack();
    const props = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        }
    };
    // since bucketprops and existing bucket is supplied, this should result in error
    const construct = new lib_1.IotToS3(stack, 'test-iot-s3-integration', props);
    const props1 = {
        iotTopicRuleProps: {
            topicRulePayload: {
                ruleDisabled: false,
                description: "process solutions constructs messages",
                sql: "SELECT * FROM 'test/constructs'",
                actions: []
            }
        },
        existingBucketInterface: construct.s3Bucket
    };
    new lib_1.IotToS3(stack, 'test-iot-s3-integration1', props1);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::IoT::TopicRule', 2);
    template.resourceCountIs('AWS::S3::Bucket', 2);
});
//# sourceMappingURL=data:application/json;base64,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