# Keeping Records of Interactions
When a clemgame is run, the `GameMaster` class (and the `DialogueGameMaster` class inheriting from it) produces a record 
of interactions. This record is stored as a JSON file named `interactions.json` in a `records/<model>/<gamename>` 
subdirectory corresponding to the experiment and episode. For example, the interactions recorded for the openchat-3.5 
model as both players of the `taboo` clemgame experiment `0_high_en` `episode_0` (corresponding to the first instance of 
this experiment) are stored in `records/openchat_3.5-t0.0--openchat_3.5-t0.0/taboo/0_high_en/episode_0/`.

The game master should record every interaction that is necessary to score the game, to generate the dialogue 
transcripts and other relevant information for posterior inspection of the interactions.

This is taken care of by the `GameRecorder` class, which has methods to log various types of information:

- `log_players`: Must be called once, in the game setup, to log the description of the agents playing each role in 
the game.
- `log_event`: Must be called to log every action; see [Logging Interaction](#logging-interaction).
- `log_next_turn`: Must be called at the beginning of every game turn; what constitutes a turns is a decision of the 
game designer.
- `log_key`: Can be used to log game-specific episode-level keys and values.

The `GameMaster` class inherits these methods from `GameRecorder`.

The `DialogueGameMaster` class inherits these methods from `GameMaster`, and has additional logging methods:

- `log_message_to`: Logs a GM->Player `send_message` action.
- `log_message_to_self`: Logs a GM->GM `metadata` action.
- `log_to_self`: Logs a GM->GM action of a type passed as `type_` argument. This method is useful to record both 
standard and custom scoring-relevant actions.
## Logging Players
The GameMaster `setup` method must call `log_players` passing a dictionary that maps player identifier strings to 
a description of the player (e.g. is it a pretrained model, a human, or a program).

Use `GM` for the GameMaster and `Player i`, where i is an integer, for the other players. These identifiers for 
players are also used in the `interaction` dictionaries.

Example:
```python
self.log_players(
    {
        "GM": "Game master for mygame",
        "Player 1": self.player_models[0].get_name(),
        "Player 2": self.player_models[1].get_name()
    }
)
```
This example uses the `GameMaster.player_models` attribute list of `Model`s to get the names for the two players. 
## Logging Interaction
The interactions record for an episode is a dictionary containing at least the `players` and `turns` keys.

The `players` key contains a dictionary with the identifiers described in [Logging Players](logging-players).

The `turns` key contains a list of lists, of the actions logged by the game master, in chronological order. The 
`log_event` method is to be used to log these actions, including automatic timestamps. 

An event can be only an action or an action and a corresponding API call (in which case they are both identifiable by 
the shared timestamp). See [Logging Calls](#logging-calls).

`log_event` requires a `from` and a `to` value (both are Player identifiers, see [Logging Players](#logging-players)). It also 
requires an action dictionary that must contain at least the two following keys:

- `type`: The action type (see basic types below).
- `content`: A string with the action message.

The basic action types are:

- `send message`: An utterance that is sent from GM to a player (its content is only the last utterance, not the whole 
dialogue context).
- `get messsage`: An utterance generated by a player.
- `metadata`: Any relevant information the GameMaster logs (not visible to players, but that should be in the 
transcript).
- `parse`: A manipulation of a response string, when needed (e.g. to extract a grid from a response).
- `error`: An error message emitted by the GameMaster.
- `invalid format`: An invalid answer that causes the game to be aborted.

If a game needs more custom actions, they should be documented in the game directory. See [Custom Actions](#custom-actions).

Use `from: "GM"` and `to: "GM"` for messages that the Game Master emits to itself (not the players).
The `DialogueGameMaster` methods `log_message_to_self` and `log_to_self` call `log_event` with `from: "GM"` and 
`to: "GM"`.

Here is an example of what the `interactions.json` file of an episode will look like:

```json
{
  "some_other_key_1": "some_other_value",
  "some_other_key_2": "some_other_value",
  "players": {
    "GM": "Game Master for privateshared",
    "Player 1": "gpt-3.5",
    "Player 2": "human"
  },
  "turns": [
    [
      {
        "timestamp": "timestamp_1",
        "from": "GM",
        "to": "Player 1",
        "action": {
          "type": "send message",
          "content": "this is a message from GM to Player 1."
        }
      },
      {
        "timestamp": "timestamp_2",
        "from": "Player 1",
        "to": "GM",
        "action": {
          "type": "get message",
          "content": "this is a message from Player 1 to GM."
        }
      },
      {
        "timestamp": "timestamp_3",
        "from": "GM",
        "to": "GM",
        "action": {
          "type": "parse",
          "content": "this is a parsed response from GM to GM.",
          "other_key": "other_value"
        }
      }
    ],
    [
      {
        "timestamp": "timestamp_4",
        "from": "GM",
        "to": "GM",
        "action": {
          "type": "metadata",
          "content": "metadata not visible to players but added to transcript"
        }
      },
      {
        "timestamp": "timestamp_5",
        "from": "GM",
        "to": "Player 2",
        "action": {
          "type": "send message",
          "content": "this is a message from GM to Player 2."
        }
      },
      {
        "timestamp": "timestamp_6",
        "from": "Player 2",
        "to": "GM",
        "action": {
          "type": "get message",
          "content": "this is a message from Player 2 to GM."
        }
      }
    ]
  ]
}
```

## Logging Calls
We also want to log the exact input and output from an backend/API, to make sure that there are no issues with model 
outputs and their processing by backends and the game master.

The backend/API call returns the manipulated prompt (including the applied chat template) and the raw response, which 
should simply be logged by the game master without any modifications.

For that, use the optional `call` argument in `log_event` to log the call with the same timestamp and an action.

Example:
```python
# generate a model response:
player_1_prompt, player_1_response, player_1_response_text = player_1(prompt)
# create the action dict:
action = {'type': 'get message', 'content': player_1_response_text}
# log the action event with log_event along with the backend/API call:
self.log_event(from_="Player 1", to="GM", action=action, call=(player_1_prompt, player_1_response))
```
`DialogueGameMaster`'s default `prompt` method contains `log_event` with a `get message` action and `call` argument as 
in the example.

The backend/API calls will be stored in a `requests.json` file that contains the raw inputs and outputs of calls made 
to backends/APIs. It is stored at the same location as `interactions.json`.

Here is an example of how a basic requests file of an episode will look like:
```json
[
    {
        "timestamp": "timestamp_1",
        "manipulated_prompt_obj": "the whole prompt passed with the backend/API call, as processed by the backend",
        "raw_response_obj": "the whole response object received from the backend/API call"
    },
    {
        "timestamp": "timestamp_2",
        "manipulated_prompt_obj": "the whole prompt passed with the backend/API call, as processed by the backend",
        "raw_response_obj": "the whole response object received from the backend/API call"
    }
]
```
Depending on the backend/API `raw_response_obj` is likely to be more extensive.
## Scoring & Logging Scores
Scores are calculated using the `GameScorer` class, preferably a game-specific child class of it. Game-specific child 
classes of `GameScorer` allow for the implementation of custom scores.  
Scoring is to be done after a clembench run has finished, meaning that all results are recorded in an 
`interactions.json` for each episode.

Default `GameScorer` iterates over the `turns` list in an episode's `interactions.json` and assigns scores based on the 
recorded actions in each turn.  
This is implemented in the `score_turns` method, which receives the entire episode interactions dict as 
`episode_interactions` argument. See the `GameScorer` base class in `clemgame.py` 
[here](../clemgame/clemgame.py) for a base scoring procedure.

`GameScorer` has the following logging methods:
- `log_turn_score`: Should be called in the scoring method to log turn-level scores. It takes a score name and a value 
for a given turn index.
- `log_episode_score`: Should be called in the scoring method to log episode-level scores.  It takes a score name and a 
value for the whole episode.
- `log_main_score`: Should be called once per episode to record the main score/`BENCH_SCORE` of your clemgame.

Games can have multiple turn-level scores and episode-level scores, but only one main score/`BENCH_SCORE` that reflects 
the core quality of a played episode. Episode scores are usually measures of game success.

You can log as many scores as you wish. The minimal requirements is to log the default episode-level scores defined in 
`clemgame/metrics.py` [here](../clemgame/metrics.py) (see the [paper](https://doi.org/10.48550/arXiv.2305.13455)'s 
appendix for details). For custom, game-specific scores see [Custom Scores](#custom-scores).

**Important**: If the game was aborted, all default episode-level scores should be `np.nan` and turn-level scores can be 
logged up to the turn when the game was aborted. If the game was won or lost, all metrics should be a numerical value. 
This is specially relevant for the main score of each game, so that the evaluation script correctly distinguishes 
%played and computes the main score only for actually played games.

The `GameScorer` method `score_turns` is intended to hold all `log_turn_score` calls, iterating over actions in each 
turn. Complex turn-level score calculations are best implemented inside this method as well.  
The `score_game` method is intended for episode-level scores, specially those that rely on turn-level or other 
episode-level scores.  
The `score_game_end` and `score_requests` methods are used to conveniently calculate the minimally required scores 
defined in `clemgame/metrics.py`.

A clemgame's `GameBenchmark` will call the `compute_scores` method of the clemgame's `GameScorer`, which calls 
`score_turns` and then `score_game` (unless modified for game-specific demands). This is followed by its `store_scores` 
method to store the score results.  
The CLI command to score all present results is
```shell
python3 scripts/cli.py score 
```
Or to only score a specific game, inn this example 'taboo':
```shell
python3 scripts/cli.py score -g taboo
```

The score results will be stored to `scores.json` which contains:
- `turn scores`: The turn-level scores for each game turn.
- `episode scores`: The episode-level scores for the episode.

Here is an example of how the `scores.json` file of an episode will look like:
```json
{
    "turn scores": {
        "1": {"accuracy": 0.8, "f1": 0.7},
        "2": {"accuracy": 0.5, "f1": 0.8},
        "3": {"accuracy": 0.4, "f1": 0.9}
    },
    "episode scores": {
        "accuracy": 0.8,
        "f1": 0.7
    }
}
```
### IMPORTANT: Inspecting the game records
During development, always check the generated `interactions.json` and `requests.json` to make sure that the API calls 
are passing the correct structure and that the records are being correctly saved.

`interactions.json` is built by the game master as a way to represent the actual interaction (with all its meta-events 
like parsing messages or checking game rules). This is used to create the transcripts, which are a user-friendly 
visualisation of the interaction. But remember that this does not reflect the actual API calls, this only reflects what 
the game master makes of the game!

The actual prompts and responses from the model are saved into `requests.json`, when an action is logged with its 
corresponding prompt and response object (see below how to do it). This file will reflect what was actually passed to 
and from the LLM. Remember that LLMS do not keep a internal state, so every call to a model must contain its full 
dialogue history. Also remember that when there are two LLMs playing at once, each will have its own dialogue history, 
which may be different! That's why, for debugging purposes, only looking at `interactions.json` is not enough, because 
it may not reflect exactly what the LLMs consumed and output.

See [Transcripts](#transcripts) for a more accessible way to inspect episode interactions.
## Custom Actions
You can record any kind of action that is specific to your game, to show it in the interaction transcripts and/or to use 
it for game-specific scoring.  
To do so, simply create an action dict with a new `type` key and the `content` you want to record and pass it to 
`log_event` (or one of the `DialogueGameMaster` logging methods).  
`content` type is hinted to be string, but any python data type that can be converted into JSON string can be used. 
JSON convertibility can be checked by passing the `content` object to `json.dump` - if there are no exceptions and the 
resulting string adequately contains the information you want to record, it can be used as `content`.

Example:
```python
# create the action dict:
action = {
    'type': 'world_state_requirement_unfulfilled', 
    'content': 
        {'action_type': "take", 'action_target': "apple1", 'fact_requirement': "open(refrigerator1)"}
    }
# log the action event with log_event along with the backend/API call:
self.log_event(from_="GM", to="GM", action=action)
```
Both the action's type and its content will be shown in the interaction transcript for convenient inspection.
## Custom Scores
If you record [Custom Actions](#custom-actions), they can be used to implement custom scores specific to your game.

Example `GameScorer` method implementation:
```python
# in custom GameScorer child class
def score_turns(self, episode_interactions: Dict) -> None:
    # Loop over turns, calculate and log turn-specific scores
    for turn_idx, turn in enumerate(episode_interactions["turns"]):
        state_requirement_fail_count_turn = 0
        for event in turn:
            action = event["action"]
            if action["type"] == "world_state_requirement_unfulfilled":
                state_requirement_fail_count_turn += 1
        self.log_turn_score(turn_idx, "state_requirement_fail", state_requirement_fail_count_turn)
```
## Transcripts
Episode transcripts allow convenient inspection of interactions, by presenting them in a visual format similar 
to common chat applications.

The transcripts are generated as HTML files for quick inspection and in LaTeX for convenient insertion into written 
works.  
Given existing game records, the following CLI command will generate transcripts in the episode directories:  
```shell
python3 scripts/cli.py transcribe
```

Messages are shown as 'chat bubbles', with orange background for player messages and grey background for game master 
messages. Each 'chat bubble' has the sender, receiver and the message direction annotated above ("Player 1 to Game 
Master", for example).

GM-to-GM actions are in boxes with light-grey background to show any processing or intermediate information relevant for 
inspection. Only GM-to-GM actions with string `content` are inserted.

![image](transcript_example.png)  
This example image shows a player message from a 'get message' type action in the interaction log with `content` "> take 
pillow from bed", a GM->GM 'metadata' type action with game-specific information `content`, and a game master message 
from a 'send message' type action with `content` "You take the pillow. In your inventory you have a pillow." in the 
interaction log.
## Overall model comparison tables
To generate overview tables listing all models and aggregate main default scores for all present results, use the 
following CLI command:
```shell
python3 evaluation/bencheval.py
```
Alternatively, if results are stored in a different directory, this can be specified:
```shell
python3 evaluation/bencheval.py --results_path <PATH>
```
See the [benchmark workflow howto](howto_benchmark_workflow.md) for further information and contributing your 
experimental results.