#! /usr/bin/env python

"""Test the locations of sources in various apertures using RADec_To_XY
Note that the truth values in this case were calculated using the
distortion reference files, which are not part of the repo and not
available to Github Actions CI
Values were calculated using run_all_apertures.py in
the test_mirage_subarray_source_location subdirectory
"""

import numpy as np
import pysiaf

from mirage.seed_image import catalog_seed_image
from mirage.utils import siaf_interface

reference_file_values = {
'nircam_NRCA1_FULL': ([1023.4999999998163, 1013.6655281785538, 1003.8299720772586, 984.1556194677335, 949.715171199927, 846.3161706399252, 530.511072051233, 35.229179419690695], [1023.4999999998024, 1033.3109865349104, 1043.122689869189, 1062.7482342576648, 1097.0997294094025, 1200.2048432008278, 1514.8651233861085, 2007.624512937195]),
'nircam_NRCA1_SUB160': ([79.47575618678025, 69.66838521645195, 59.859819900987986, 40.23911767044774, 5.891489647969138, -97.2373108295451, -412.2985763208228, -906.6825875959386], [79.49770829260206, 89.42717067086005, 99.3572632124251, 119.21932657806428, 153.9838998863188, 258.32194748632446, 576.6820423169911, 1075.0470216746535]),
'nircam_NRCA1_SUB320': ([159.47558091216615, 149.66550269545337, 139.8542386707577, 120.22816466854283, 85.87121852763767, -17.284902266516838, -332.42341956509244, -826.9087414221374], [159.49570717060442, 169.41549949944988, 179.33592894473213, 199.1786869908267, 233.90954302268483, 338.1469524512713, 656.2051752443804, 1154.113986493605]),
'nircam_NRCA1_SUB640': ([319.47667565289953, 309.66140058191604, 299.84495704503433, 280.2085761428766, 245.83376069688575, 142.62532411204575, -172.6603187612393, -667.3354208960037], [319.4926777609121, 329.3929293893109, 339.2938320873743, 359.0975784916104, 393.7602984194343, 497.7943256295423, 815.2423013308726, 1312.2279086000126]),
'nircam_NRCA1_GRISMTS': ([1979.2313427254185, 1969.4657029403556, 1959.6987854642452, 1940.1611289202515, 1905.9580287794251, 1803.2566759785423, 1489.4402307546106, 996.8101661301396], [1081.6230605981543, 1091.4652484083779, 1101.308246675403, 1120.9966617315342, 1155.4590807960228, 1258.90383914193, 1574.6635579960184, 2069.348907430881]),
'nircam_NRCA1_GRISMTS256': ([1980.593969382693, 1970.8751396619045, 1961.1549394565927, 1941.710440362619, 1907.6694832139522, 1805.4479302391449, 1493.034502532663, 1002.4207430035899], [166.8560247850687, 176.5931842308039, 186.33137545302748, 205.81083726974032, 239.90969257662925, 342.27954897633697, 654.9032345150479, 1145.0875774252333]),
'nircam_NRCA1_GRISMTS128': ([1980.593969382693, 1970.8751396619045, 1961.1549394565927, 1941.710440362619, 1907.6694832139522, 1805.4479302391449, 1493.034502532663, 1002.4207430035899], [64.85602478506871, 74.5931842308039, 84.33137545302748, 103.81083726974032, 137.90969257662925, 240.27954897633697, 552.9032345150479, 1043.0875774252333]),
'nircam_NRCA1_GRISMTS64': ([1980.593969382693, 1970.8751396619045, 1961.1549394565927, 1941.710440362619, 1907.6694832139522, 1805.4479302391449, 1493.034502532663, 1002.4207430035899], [32.85602478506871, 42.5931842308039, 52.331375453027476, 71.81083726974032, 105.90969257662925, 208.27954897633697, 520.9032345150479, 1011.0875774252333]),
'nircam_NRCA1_FP3MIMF': ([230.01329645230203, 220.16578016329447, 210.3173027215032, 190.61747864833296, 156.1336559827463, 52.61388742745237, -263.46740835808714, -758.867577539334], [228.035810836402, 237.73856759049477, 247.44213669726514, 266.8516978803584, 300.8261303287875, 402.8068571504516, 714.1001899332017, 1201.7638443199696]),
'nircam_NRCA2_FULL': ([1023.5000000000364, 1013.5963968852795, 1003.6919523863716, 983.8805562714804, 949.2026590967788, 845.1099529897176, 527.3780604464274, 29.763924173508826], [1023.4999999996849, 1033.5071104502902, 1043.5144501536506, 1063.529799543284, 1098.5587169468986, 1203.6593720689932, 1524.0438340337425, 2024.5270119055504]),
'nircam_NRCA2_SUB160': ([79.54427460488432, 69.73797864713833, 59.9305132426025, 40.31208572779815, 5.968679873873953, -97.14558613356894, -412.1443826645177, -906.3676673394888], [79.46916662128297, 89.43758570674981, 99.40657975828083, 119.34628025662391, 154.24618466127345, 258.9861122762421, 578.531581590951, 1078.617115924541]),
'nircam_NRCA2_SUB320': ([159.54188494413756, 149.72616576465452, 139.9093008986756, 120.27214595588703, 85.89619764879808, -17.313910895855315, -332.58746370339577, -827.1806668062044], [159.47092108037793, 169.44382519784892, 179.41727889898016, 199.3658223513538, 234.28095589522366, 339.06466926056737, 658.7251985490946, 1158.9281354936581]),
'nircam_NRCA2_SUB640': ([319.5357597688949, 309.7018014528446, 299.86674641473746, 280.19335858571776, 245.75447886593497, 142.35927399288312, -173.4421695224578, -668.7346287976625], [319.4754272890099, 329.4566775209988, 339.4384252119089, 359.4033997331219, 394.3467813240138, 499.21131299218337, 819.0798866266343, 1319.4764456759651]),
'nircam_NRCA2_FP4MIMF': ([227.99187975603672, 218.03633113619844, 208.08036588496793, 188.16721191969032, 153.31540642792322, 48.73375731775832, -270.1578387442628, -768.4024322602579], [1819.001677607011, 1829.0107218027817, 1839.0195568345182, 1859.0365722450479, 1894.0641058389797, 1999.1268245350198, 2319.0559617225326, 2817.6175196634968]),
'nircam_NRCA2_MASK210R': ([348.38977658825, 338.4375984775261, 328.48503203060795, 308.57876202638204, 273.7392882071024, 169.19695528496936, -149.54927867999515, -647.4631253158907], [1643.6490057786837, 1653.7154160297741, 1663.7812953780008, 1683.911427903734, 1719.1337663903437, 1824.756211563918, 2146.1451148257684, 2646.128780715589]),
'nircam_NRCA2_TAMASK210R': ([89.96039966596823, 80.00350722329688, 70.0462642018341, 50.13075486441028, 15.275480236944873, -89.31138945905604, -408.16526728290626, -906.151812849799], [1386.964094673866, 1397.0109836222855, 1407.0574016806372, 1427.1487927146118, 1462.3039200894627, 1567.7293156128726, 1888.562993008456, 2387.834585944479]),
'nircam_NRCA2_FSTAMASK210R': ([90.8414318960771, 80.88864732874731, 70.93545102356427, 51.02785038685602, 16.18581670684648, -88.36603298283956, -407.1602456062069, -905.219261106582], [1386.8110226814147, 1396.8670031784586, 1406.9225538767187, 1427.032334381539, 1462.2200465789065, 1567.7463680086057, 1888.9195494795258, 2388.8350579999774]),
'nircam_NRCA2_FULL_MASK210R': ([771.38977658825, 761.4375984775261, 751.485032030608, 731.578762026382, 696.7392882071024, 592.1969552849694, 273.45072132000485, -224.4631253158907], [2800.6490057786837, 2810.715416029774, 2820.781295378001, 2840.911427903734, 2876.1337663903437, 2981.756211563918, 3303.1451148257684, 3803.128780715589]),
'nircam_NRCA3_FULL': ([1023.4999999999632, 1013.8545390985643, 1004.2075267718307, 984.9088620995956, 951.121385898854, 849.6471994917788, 539.3930480288933, 51.80116507356729], [1023.5000000000216, 1033.3612380824643, 1043.2234600972768, 1062.9508405146105, 1097.483097468811, 1201.1497303171354, 1517.6951539208148, 2013.9237011053606]),
'nircam_NRCA3_SUB160': ([79.5338418935496, 69.75304155742208, 59.9709852161335, 40.40311577494606, 6.147350037185902, -96.71043319395221, -410.98910346476873, -904.2926377520157], [79.46730385721298, 89.40630741138466, 99.34597657502536, 119.22729957101228, 154.0259188862533, 258.4687236020691, 577.1728651885951, 1076.1569949077625]),
'nircam_NRCA3_SUB320': ([159.53350680468827, 149.76305670091506, 139.99132768970298, 120.44404420391788, 86.22408341853009, -16.52797168555162, -330.50028911360914, -823.3767125699183], [159.46769402075347, 169.40126159211718, 179.33551941151973, 199.20609361247898, 233.98613956261102, 338.3750356150822, 656.9318609746533, 1155.7404964444077]),
'nircam_NRCA3_SUB640': ([319.53099854977563, 309.7818385813846, 300.03135356116695, 280.5264197692136, 246.38012791058497, 143.8457036816933, -169.49475211008212, -661.4838227522762], [319.4700735691497, 329.39216120214724, 339.3149888055443, 359.16285156005097, 393.9036296356353, 498.1783548942544, 816.4209115858021, 1314.844842721146]),
'nircam_NRCA3_GRISMTS': ([1045.0010127124685, 1035.3540796463683, 1025.7055980699247, 1006.4040035537652, 972.6114274296342, 871.1221414679509, 560.8236368821802, 73.16672746439508], [1089.9986569174116, 1099.867841526581, 1109.7379968433038, 1129.4812043926459, 1164.041033237564, 1267.7894414018463, 1584.5758110494012, 2081.1576079700494]),
'nircam_NRCA3_GRISMTS256': ([1044.9908464362495, 1035.398090575211, 1025.803657123932, 1006.609774654094, 973.0044775299864, 872.0680159654762, 563.3775709436893, 78.03195682720093], [175.01721784844858, 184.77332815292277, 194.530668767493, 214.04902086367815, 248.21781083593567, 350.81142214039187, 664.2371413845646, 1155.9964686548549]),
'nircam_NRCA3_GRISMTS128': ([1044.9908464362495, 1035.398090575211, 1025.803657123932, 1006.609774654094, 973.0044775299864, 872.0680159654762, 563.3775709436893, 78.03195682720093], [64.01721784844858, 73.77332815292277, 83.53066876749301, 103.04902086367815, 137.21781083593567, 239.81142214039187, 553.2371413845646, 1044.9964686548549]),
'nircam_NRCA3_GRISMTS64': ([1044.9908464362495, 1035.398090575211, 1025.803657123932, 1006.609774654094, 973.0044775299864, 872.0680159654762, 563.3775709436893, 78.03195682720093], [32.01721784844858, 41.77332815292277, 51.53066876749301, 71.04902086367815, 105.21781083593567, 207.81142214039187, 521.2371413845646, 1012.9964686548549]),
'nircam_NRCA3_DHSPIL': ([1014.001585272133, 1004.352044284634, 994.7009609729531, 975.3941814523668, 941.5925903448107, 840.0766998059205, 529.700995041399, 41.93435895381242], [1078.9981737336839, 1088.8656158973927, 1098.7340269562935, 1118.4737405954004, 1153.0274380126616, 1256.7573254465017, 1573.4861748654569, 2069.9752624690586]),
'nircam_NRCA3_DHSPIL_SUB96': ([47.00158527213296, 37.35204428463396, 27.700960972953112, 8.394181452366752, -25.407409655189326, -126.9233001940795, -437.299004958601, -925.0656410461876], [47.99817373368387, 57.86561589739267, 67.73402695629352, 87.47374059540039, 122.0274380126616, 225.75732544650168, 542.4861748654569, 1038.9752624690586]),
'nircam_NRCA3_DHSPIL_WEDGES': ([1021.5079426491114, 1011.788675215786, 1002.0680061394694, 982.6224750308199, 948.5794053915724, 846.3491051526423, 533.885314252245, 43.11817044266081], [2454.491780199024, 2464.4997516250005, 2474.5083962830568, 2494.5276926728575, 2529.5678349642226, 2634.7356979518304, 2955.6531471779217, 3458.1031260959085]),
'nircam_NRCA3_FP1': ([460.0234470283126, 450.3024288404314, 440.5800270499893, 421.13108454264477, 387.0822177466363, 284.83583231727744, -27.665458638789914, -518.4624565800366], [1409.9764905883535, 1419.8695812097847, 1429.7634933031618, 1449.5537689057956, 1484.1945487290184, 1588.175178244104, 1905.575201115541, 2402.846542540273]),
'nircam_NRCA3_FP1_SUB8': ([3.023447028312603, -6.697571159568611, -16.419972950010674, -35.86891545735523, -69.91778225336373, -172.16416768272256, -484.6654586387899, -975.4624565800366], [3.9764905883534993, 13.869581209784656, 23.763493303161795, 43.55376890579555, 78.19454872901838, 182.17517824410402, 499.57520111554095, 996.8465425402728]),
'nircam_NRCA3_FP1_SUB64': ([31.023447028312603, 21.30242884043139, 11.580027049989326, -7.868915457355229, -41.917782253363725, -144.16416768272256, -456.6654586387899, -947.4624565800366], [31.9764905883535, 41.869581209784656, 51.763493303161795, 71.55376890579555, 106.19454872901838, 210.17517824410402, 527.575201115541, 1024.8465425402728]),
'nircam_NRCA3_FP2MIMF': ([1818.9873320643433, 1809.4840422052039, 1799.9788350385938, 1780.962690724537, 1747.6661676853284, 1647.6391568382692, 1341.5751425429567, 859.9494226710383], [228.02363813289878, 237.79042583948285, 247.55858723885126, 267.09900775628205, 301.3077699692599, 404.03131546121074, 717.9380727311263, 1210.6668558086637]),
'nircam_NRCA3_SUB64P': ([31.523787951745817, 21.830938232639255, 12.1366580185869, -7.256180511679759, -41.207304222835546, -143.1636760595734, -454.807853110387, -944.3351916827589], [31.50000796450513, 41.23782523731029, 50.97672033081153, 70.45772718648345, 104.5597225304557, 206.94225365030616, 519.6329271182226, 1010.0002780505613]),
'nircam_NRCA3_SUB160P': ([79.49562954061662, 70.0228297124504, 60.54802346550173, 41.59241563150772, 8.400990251983785, -91.31696426243434, -396.49215841395153, -876.8650791275685], [79.51960073223972, 89.26471844872367, 99.01130392038999, 118.50885274527195, 152.6434778912294, 255.1512042653676, 568.4564497266822, 1060.3901277143418]),
'nircam_NRCA3_SUB400P': ([199.4885828804363, 189.99103157857644, 180.49154647548266, 161.48679717255345, 128.21005970786518, 28.241251201632167, -277.65520997402314, -759.0443534775466], [199.52314394752113, 209.285888005973, 219.05002318844186, 238.5824431932259, 272.77736914949344, 375.46062759072856, 689.2550286917876, 1181.8351424301518]),
'nircam_NRCA4_FULL': ([1023.5000000000055, 1013.6945302466736, 1003.887723099901, 984.2701087010541, 949.926512782387, 846.7993873575252, 531.647688499598, 36.83329082916873], [1023.4999999997228, 1033.520336044648, 1043.5411782478134, 1063.5843676323186, 1098.6647068976233, 1203.9407171664652, 1525.0688322212357, 2027.4542323738174]),
'nircam_NRCA4_SUB160': ([79.46748477099038, 69.62126700445378, 59.773817532090106, 40.0752349004955, 5.590956500590437, -97.95054476824816, -414.29522501557153, -910.7649360198161], [79.48185103377159, 89.39647205461836, 99.3116931837992, 119.14392350388482, 153.8559991948341, 258.03433334353485, 575.886822110811, 1073.392535063897]),
'nircam_NRCA4_SUB320': ([159.46968884007606, 149.6265277851046, 139.78212654010088, 120.08961491625905, 85.61587916607846, -17.894617350746103, -334.1506049927748, -830.4998152686298], [159.4845318437632, 169.40849981090332, 179.3330608038757, 199.1839495487867, 233.92860884238678, 338.2041636354503, 656.3478442746559, 1154.29156377562]),
'nircam_NRCA4_SUB640': ([319.47500881991334, 309.63817631698214, 299.80008655492145, 280.12014673120314, 245.6682476644637, 142.22201178680598, -173.8496878129322, -669.9465548917888], [319.48970825378353, 329.4321656478478, 339.37520158365885, 359.26299662578356, 394.0721013597798, 498.5399042287405, 817.258953321547, 1316.0662811184793]),
'nircam_NRCA4_FP5MIMF': ([1702.99845265554, 1693.2293173109229, 1683.4587508789102, 1663.9133380479407, 1629.6952032539637, 1526.9377667378963, 1212.8535714102095, 719.5422631230435], [1702.9814049070237, 1713.0705689434035, 1723.1601466244308, 1743.340527726753, 1778.6600476700446, 1884.6465182007232, 2207.8694373013354, 2713.282306308629]),
'nircam_NRCA4_MASKSWB': ([364.3839339648887, 354.47018936922586, 344.55542867207737, 324.7228757412762, 290.0062700449299, 185.78454061345747, -132.46146190467505, -631.25981411217], [1630.8219655978428, 1640.9269198245279, 1651.0317596922814, 1671.2410724970014, 1706.606067163656, 1812.6886935426246, 2135.805722981411, 2639.659907438592]),
'nircam_NRCA4_TAMASKSWB': ([106.96601764234856, 97.03680463009584, 87.1066218928471, 67.2433641435955, 32.473472565148654, -71.90468013294424, -390.5949873484219, -889.9836028883692], [1373.878420804067, 1383.9612689060568, 1394.044051841624, 1414.2093991860538, 1449.4979347526314, 1555.3549347044013, 1877.8212701534244, 2380.78415996269]),
'nircam_NRCA4_TAMASKSWBS': ([110.01021279512713, 100.13037488500345, 90.2493995252197, 70.48405147691142, 35.883871421010554, -67.99779815848467, -385.29825894749627, -882.9347681455181], [1372.686210152287, 1382.7955882650895, 1392.9050045301715, 1413.1239306182397, 1448.507241522464, 1554.65651914222, 1878.0935947419553, 2382.8604134498373]),
'nircam_NRCA4_FSTAMASKSWB': ([108.00445290939956, 98.09891682925934, 88.19232576205536, 68.37599447454704, 33.68739423435886, -70.45333747922427, -388.48326803691066, -887.0569166306293], [1373.8386951444763, 1383.9352411967257, 1394.0317759784248, 1414.224789845378, 1449.5622702851142, 1555.570257859731, 1878.5385855662257, 2382.43622414741]),
'nircam_NRCA4_MASKSWB_F182M': ([307.04432419658144, 297.12646758182177, 287.2076099431787, 267.3669085454253, 232.6361897307272, 128.37326074578277, -189.9871635503457, -688.9262286035364], [1631.0100203930856, 1641.1125168206004, 1651.2148896748413, 1671.419240610433, 1706.7754616250959, 1812.8310563883829, 2135.858535233938, 2639.54698986551]),
'nircam_NRCA4_MASKSWB_F187N': ([315.99716245197135, 306.0799484079548, 296.1617309548851, 276.3223027424657, 241.5937887873588, 137.33729392909822, -181.00528714774646, -679.9225181353536], [1630.9803621878955, 1641.083239465297, 1651.1859946846557, 1671.3911149295973, 1706.748697098863, 1812.808492186693, 2135.8499503249036, 2639.5644844842527]),
'nircam_NRCA4_MASKSWB_F210M': ([363.07458299361383, 353.16077300588404, 343.2459471090492, 323.4132643546069, 288.6964333175007, 184.47404204013299, -133.77384444871666, -632.5747177344767], [1630.8229782620037, 1640.9278468827665, 1651.0326013529807, 1671.2417439917526, 1706.6064429052694, 1812.6881979187237, 2135.802728108374, 2639.6535372841013]),
'nircam_NRCA4_MASKSWB_F212N': ([370.01544274515834, 360.1021382108204, 350.18781593031645, 330.35613487543446, 295.6410389396152, 191.42371412609396, -126.81008794555669, -625.5935871602642], [1630.7995724751781, 1640.9047330908998, 1651.009780713479, 1671.2195131532803, 1706.585255567853, 1812.670230296765, 2135.7954721976544, 2639.666251401196]),
'nircam_NRCA4_MASKSWB_F200W': ([394.5597464691514, 384.64823669080295, 374.735702681358, 354.9075786364033, 320.1986443861614, 215.99931468093166, -102.18442890626272, -600.906040863307], [1630.7163584594232, 1640.8225483563556, 1650.9286293342102, 1671.140440796666, 1706.509861459068, 1812.606185025134, 2135.7691795320898, 2639.7103278738464]),
'nircam_NRCA4_MASKSWB_NARROW': ([111.38413698015358, 101.45262068698742, 91.52015622937279, 71.65240048651128, 36.874851429290175, -67.52456184270329, -386.2616287911851, -885.6540117667219], [1631.6359443552137, 1641.7299419645478, 1651.8237819238739, 1672.010964096287, 1707.3368054726125, 1813.298632286083, 2136.0138203123024, 2639.1189427071545]),
'nircam_NRCA4_FULL_MASKSWB': ([523.3839339648887, 513.4701893692259, 503.55542867207737, 483.7228757412762, 449.0062700449299, 344.78454061345747, 26.538538095324952, -472.25981411217003], [2817.821965597843, 2827.926919824528, 2838.0317596922814, 2858.2410724970014, 2893.606067163656, 2999.6886935426246, 3322.805722981411, 3826.659907438592]),
'nircam_NRCA4_FULL_MASKSWB_F182M': ([466.04432419658144, 456.12646758182177, 446.2076099431787, 426.3669085454253, 391.6361897307272, 287.3732607457828, -30.987163550345713, -529.9262286035364], [2817.0100203930856, 2827.1125168206004, 2837.2148896748413, 2857.419240610433, 2892.775461625096, 2998.831056388383, 3321.858535233938, 3825.54698986551]),
'nircam_NRCA4_FULL_MASKSWB_F187N': ([474.99716245197135, 465.0799484079548, 455.1617309548851, 435.3223027424657, 400.5937887873588, 296.3372939290982, -22.005287147746458, -520.9225181353536], [2816.9803621878955, 2827.083239465297, 2837.1859946846557, 2857.3911149295973, 2892.748697098863, 2998.808492186693, 3321.8499503249036, 3825.5644844842527]),
'nircam_NRCA4_FULL_MASKSWB_F210M': ([522.0745829936138, 512.160773005884, 502.2459471090492, 482.4132643546069, 447.6964333175007, 343.474042040133, 25.22615555128334, -473.57471773447674], [2816.8229782620037, 2826.9278468827665, 2837.0326013529807, 2857.2417439917526, 2892.6064429052694, 2998.6881979187237, 3321.802728108374, 3825.6535372841013]),
'nircam_NRCA4_FULL_MASKSWB_F212N': ([529.0154427451583, 519.1021382108204, 509.18781593031645, 489.35613487543446, 454.6410389396152, 350.42371412609396, 32.18991205444331, -466.5935871602642], [2816.799572475178, 2826.9047330908998, 2837.009780713479, 2857.2195131532803, 2892.585255567853, 2998.670230296765, 3321.7954721976544, 3825.666251401196]),
'nircam_NRCA4_FULL_MASKSWB_F200W': ([542.293521573094, 532.3811865472749, 522.4678302649484, 502.6380706329328, 467.92630303810586, 363.7186979985105, 45.51192782048088, -453.2381781676627], [2816.754629817443, 2826.8603478901073, 2836.9659551769573, 2857.1768136167616, 2892.544548181224, 2998.6356695467102, 3321.78135938727, 3825.690255288797]),
'nircam_NRCA5_FULL': ([1023.4999999999161, 1018.6634417331632, 1013.8263230330962, 1004.1504123432998, 987.2122420045221, 936.3578450335696, 781.0253855124345, 537.4066415581242], [1023.499999999897, 1028.402184573697, 1033.304636356964, 1043.1103329363598, 1060.2728012218322, 1111.7784427554927, 1268.8938350062474, 1514.6812247905796]),
'nircam_NRCA5_SUB160': ([79.4999999999161, 74.66344173316315, 69.82632303309617, 60.15041234329976, 43.21224200452207, -7.642154966430439, -162.97461448756553, -406.5933584418758], [79.499999999897, 84.40218457369701, 89.30463635696401, 99.11033293635978, 116.2728012218322, 167.77844275549273, 324.8938350062474, 570.6812247905796]),
'nircam_NRCA5_SUB320': ([159.4999999999161, 154.66344173316315, 149.82632303309617, 140.15041234329976, 123.21224200452207, 72.35784503356956, -82.97461448756553, -326.5933584418758], [159.499999999897, 164.402184573697, 169.304636356964, 179.11033293635978, 196.2728012218322, 247.77844275549273, 404.8938350062474, 650.6812247905796]),
'nircam_NRCA5_SUB640': ([319.4999999999161, 314.66344173316315, 309.8263230330962, 300.15041234329976, 283.21224200452207, 232.35784503356956, 77.02538551243447, -166.5933584418758], [319.499999999897, 324.402184573697, 329.304636356964, 339.1103329363598, 356.2728012218322, 407.77844275549273, 564.8938350062474, 810.6812247905796]),
'nircam_NRCA5_GRISM256_F322W2': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM128_F322W2': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM64_F322W2': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM256_F277W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM128_F277W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM64_F277W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM256_F356W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM128_F356W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM64_F356W': ([1579.9842856588516, 1575.2529881667379, 1570.5208172401606, 1561.0538653482295, 1544.4783788176424, 1494.689363577391, 1342.3978285613234, 1102.9157438676239], [34.01474027230006, 38.81582781050099, 43.617592301416494, 53.22314041205641, 70.0392669150774, 120.53552724624888, 274.84319288493646, 517.0366819055427]),
'nircam_NRCA5_GRISM256_F444W': ([951.0015137353776, 946.206500932404, 941.4108121673298, 931.817415217698, 915.0225355876272, 864.5895789014268, 710.463679338653, 468.4885539504486], [34.00372057311972, 38.79928797808441, 43.595409789362066, 53.18930663255651, 69.98387833581717, 120.40673754596003, 274.4136490874413, 515.9218624295254]),
'nircam_NRCA5_GRISM128_F444W': ([951.0015137353776, 946.206500932404, 941.4108121673298, 931.817415217698, 915.0225355876272, 864.5895789014268, 710.463679338653, 468.4885539504486], [34.00372057311972, 38.79928797808441, 43.595409789362066, 53.18930663255651, 69.98387833581717, 120.40673754596003, 274.4136490874413, 515.9218624295254]),
'nircam_NRCA5_GRISM64_F444W': ([951.0015137353776, 946.206500932404, 941.4108121673298, 931.817415217698, 915.0225355876272, 864.5895789014268, 710.463679338653, 468.4885539504486], [34.00372057311972, 38.79928797808441, 43.595409789362066, 53.18930663255651, 69.98387833581717, 120.40673754596003, 274.4136490874413, 515.9218624295254]),
'nircam_NRCA5_GRISM_F322W2': ([1579.9861485000279, 1575.2391070213175, 1570.4912366578787, 1560.9930188174474, 1544.3632396280436, 1494.4145073411833, 1341.6638575994934, 1101.540323777638], [279.01126042148235, 283.8456151181497, 288.68055579789893, 298.35218456730547, 315.28308657711216, 366.1171742985156, 521.3969598129861, 764.9490290948479]),
'nircam_NRCA5_GRISM_F277W': ([1579.9861485000279, 1575.2391070213175, 1570.4912366578787, 1560.9930188174474, 1544.3632396280436, 1494.4145073411833, 1341.6638575994934, 1101.540323777638], [279.01126042148235, 283.8456151181497, 288.68055579789893, 298.35218456730547, 315.28308657711216, 366.1171742985156, 521.3969598129861, 764.9490290948479]),
'nircam_NRCA5_GRISM_F356W': ([1579.9861485000279, 1575.2391070213175, 1570.4912366578787, 1560.9930188174474, 1544.3632396280436, 1494.4145073411833, 1341.6638575994934, 1101.540323777638], [279.01126042148235, 283.8456151181497, 288.68055579789893, 298.35218456730547, 315.28308657711216, 366.1171742985156, 521.3969598129861, 764.9490290948479]),
'nircam_NRCA5_GRISM_F444W': ([951.0016883130542, 946.1936345359767, 941.3849400852921, 931.765637305811, 914.9257594555919, 864.3603532546782, 709.8537820211749, 467.35199368590486], [278.997589500715, 283.82351801477125, 288.6499198095444, 298.3041340313206, 315.2034868665663, 365.9348487165379, 520.829556985495, 763.5735168944049]),
'nircam_NRCA5_TAGRISMTS32': ([14.993578700049738, 10.219691664707625, 5.445058367348565, -4.10643817515097, -20.828665678571724, -71.04877824054643, -224.57272759085959, -465.74776276799764], [15.006311598870298, 19.815318216115884, 24.62490244761068, 34.24579345116638, 51.087825787715246, 101.65472604295246, 256.1153707899059, 498.3749454020233]),
'nircam_NRCA5_TAGRISMTS_SCI_F322W2': ([14.984288619293466, 10.251490851736662, 5.517823955591439, -3.9521070312157462, -20.532766145832966, -70.33699416603758, -222.6722348708986, -462.2153848536941], [15.51457008880027, 20.318865446063455, 25.12382892835194, 34.73574866162471, 51.56294156180422, 102.09177155907696, 256.49315278747827, 498.8177409545185]),
'nircam_NRCA5_TAGRISMTS_SCI_F444W': ([15.001589413173292, 10.205336390938783, 5.40841074808111, -4.187449969033992, -20.986609329596035, -71.43216510933439, -225.59430897565494, -467.6197822704355], [15.503193911379412, 20.30168996321629, 25.100732609669535, 34.70044777326268, 51.5051270584695, 101.95774812008892, 256.050356456318, 497.678179884172]),
'nircam_NRCA5_SUB64P': ([31.497726593372818, 26.713610986142612, 21.92878318241037, 12.356999639545165, -4.400403746191159, -54.72356742731358, -208.5389654433135, -450.1016048944191], [39.005148775825205, 43.80311341625895, 48.60165288343603, 58.20044605769772, 75.00378087769332, 125.45439304088109, 279.5589338277665, 521.2571632433176]),
'nircam_NRCA5_SUB160P': ([79.48210808961926, 74.79724228544183, 70.11136494100356, 60.73658781926383, 44.32109393803876, -4.997790788020666, -155.9432570771753, -393.5618256304997], [79.52000074932471, 84.3268626914172, 89.13447568638526, 98.75194181323002, 115.58962725404331, 166.15580639192638, 320.72206249438534, 563.4416523191047]),
'nircam_NRCA5_SUB400P': ([199.48113348694392, 194.77152122236316, 190.06097098462487, 180.63706760731748, 164.13630107866106, 114.56682063013432, -37.0967528241772, -275.7134259803281], [199.5183701685478, 204.34300618262432, 209.16831500936814, 218.8209392769529, 235.7194084141537, 286.4623773274849, 441.51874106604805, 684.8713636219036]),
'nircam_NRCA5_GRISMC_WFSS': ([1023.4999999999161, 1018.6634417331632, 1013.8263230330962, 1004.1504123432998, 987.2122420045221, 936.3578450335696, 781.0253855124345, 537.4066415581242], [1023.499999999897, 1028.402184573697, 1033.304636356964, 1043.1103329363598, 1060.2728012218322, 1111.7784427554927, 1268.8938350062474, 1514.6812247905796]),
'nircam_NRCA5_GRISMR_WFSS': ([1023.4999999999161, 1018.6634417331632, 1013.8263230330962, 1004.1504123432998, 987.2122420045221, 936.3578450335696, 781.0253855124345, 537.4066415581242], [1023.499999999897, 1028.402184573697, 1033.304636356964, 1043.1103329363598, 1060.2728012218322, 1111.7784427554927, 1268.8938350062474, 1514.6812247905796]),
'nircam_NRCA5_MASK335R': ([193.36201407986073, 188.46956468033545, 183.57692001488567, 173.79105802104186, 156.6640312386296, 105.2707674503099, -51.43212813316529, -296.26112075331275], [922.6927481390226, 927.6418166771791, 932.5905507275129, 942.486999277538, 959.8024286819518, 1011.7214767040136, 1169.6537996880988, 1415.2252371777422]),
'nircam_NRCA5_MASK430R': ([195.40537234444798, 190.52637246754455, 185.64705292162864, 175.88746630208175, 158.8052099876627, 107.53683744813986, -48.87744846240503, -293.56668104033724], [921.8622978158928, 926.8304852730125, 931.7984166355191, 941.7334963309377, 959.1172961463485, 1011.2474401955797, 1169.8823316379712, 1416.7563874194843]),
'nircam_NRCA5_MASKLWB': ([198.7303920622021, 193.89456044489316, 189.05818788270608, 179.38382955304974, 162.44857715843227, 111.6046366546891, -43.67545285848746, -287.1174815006166], [909.2985738898296, 914.2969129692683, 919.2951348748238, 929.2912144659531, 946.7831186082826, 999.2481293925221, 1159.0072996119488, 1407.9937662347374]),
'nircam_NRCA5_TAMASK335R': ([63.879441949551165, 58.98314653907585, 54.08669338584036, 44.29332741065559, 27.153533427116372, -24.275203074769024, -181.05878578857426, -425.92248629548385], [794.714154190704, 799.6532442849812, 804.5920181149727, 814.4686010978867, 831.749442872001, 883.5661004900003, 1041.1996119992773, 1286.3473873180073]),
'nircam_NRCA5_TAMASK430R': ([66.00409961229445, 61.11931251844749, 56.23423833327752, 46.46324040431432, 29.361329878637434, -21.963570489303038, -178.52685334414923, -423.3730716867708], [794.0295745055741, 798.9881896269771, 803.9465716826626, 813.8626221429013, 831.2133449105645, 883.245998070618, 1041.6006545461396, 1288.0926138001455]),
'nircam_NRCA5_TAMASKLWB': ([71.3470706916778, 66.5253402523756, 61.70301138540026, 52.056567649125554, 35.169609522771225, -15.533734355700744, -170.42623223121927, -413.39611066938187], [780.4331650484128, 785.4339358121115, 790.4346595817788, 800.4359542993525, 817.937667692493, 870.4374084454548, 1030.3540395466812, 1279.7602906322177]),
'nircam_NRCA5_TAMASKLWBL': ([69.66857918961523, 64.8230647340331, 59.977043232946016, 50.28348867202135, 33.314968752042205, -17.626288987922408, -173.1799598665898, -416.9788808600607], [781.829029608095, 786.8188177088332, 791.8085085292728, 801.7875859261203, 819.2499301936232, 871.6277715421775, 1031.13602723626, 1279.7795945753287]),
'nircam_NRCA5_FSTAMASKLWB': ([70.5201207454179, 65.68598946156317, 60.85130555905971, 51.18028927865794, 34.25077099835107, -16.576866845795394, -171.81633701501755, -415.2274747489637], [781.6626805149881, 786.6581492276086, 791.653546327304, 801.6441135935011, 819.1268148022191, 871.5676656702544, 1031.286847825239, 1280.3240796514788]),
'nircam_NRCA5_FSTAMASK335R': ([64.34985440269406, 59.45559378497137, 54.56114507043162, 44.77169645512163, 27.638461838469084, -23.77290312720538, -180.526190643644, -425.4197300437313], [794.5896057663167, 799.5332825093819, 804.476662556538, 814.3625170215869, 831.6597751304816, 883.527176669842, 1041.330140201736, 1286.7938805451613]),
'nircam_NRCA5_FSTAMASK430R': ([66.47684591506527, 61.59540455576348, 56.71364958900995, 46.94921020914683, 29.858519043374372, -21.434729011726404, -177.92111095569476, -422.71322792119827], [793.83443439435, 798.7970174168122, 803.7593842743927, 813.6834553218114, 831.0483791686602, 883.1249245210465, 1041.6260869909302, 1288.3909663051672]),
'nircam_NRCA5_MASKLWB_F250M': ([304.712925632899, 299.8861584078127, 295.05881540681867, 285.4024116039468, 268.4982434902904, 217.74497673416363, 62.71692317908196, -180.40867234082066], [908.7426001179911, 913.7450965977678, 918.7474946621569, 928.7519830629458, 946.258785021826, 998.7699048698337, 1158.6833789232232, 1407.9575978751536]),
'nircam_NRCA5_MASKLWB_F300M': ([279.87787696358964, 275.0489332151692, 270.21942187418927, 260.558705961108, 243.64707058896397, 192.87200720971805, 37.78321543419224, -205.41924578326916], [908.8762699481881, 913.8778112283021, 918.8792497512459, 928.8818059957339, 946.385184479012, 998.8857065505376, 1158.7637003438067, 1407.9717310091673]),
'nircam_NRCA5_MASKLWB_F277W': ([279.7773245139033, 274.9483720188771, 270.11885196429694, 260.4581187236763, 243.54645334731117, 192.77130240441784, 37.68226680645648, -205.5205022493683], [908.8768067351862, 913.8783441245068, 918.8797787387361, 928.8823271129413, 946.3856916519635, 998.886170559419, 1158.7640198639383, 1407.9717810178909]),
'nircam_NRCA5_MASKLWB_F335M': ([271.8339361582971, 267.00429437092225, 262.174087638301, 252.51198889026546, 235.59795909545414, 184.81590830449568, 29.70766467082285, -213.51933678195905], [908.9191010423774, 913.9203304590856, 918.9214557055684, 928.9233811417544, 946.4256419726339, 998.9227045020907, 1158.7891181799514, 1407.9755495114796]),
'nircam_NRCA5_MASKLWB_F360M': ([248.20432949090127, 243.3726566695036, 238.54042667069325, 228.87430471525977, 211.95330926369866, 161.1509365217994, 5.986175323032512, -237.31190657751654], [909.0437143124896, 914.0440205620034, 919.0442184700787, 929.0442766681163, 946.5432292771875, 999.0300524243339, 1158.8621974496727, 1407.984719357457]),
'nircam_NRCA5_MASKLWB_F356W': ([239.255022161773, 234.42258769305067, 229.58959898592957, 219.9219684401046, 202.99836132190717, 152.18837126118842, -2.9975524731696623, -246.32216199238633], [909.0904235415228, 914.0903774014578, 919.090221332247, 929.0895667944883, 946.5872566779062, 999.0701718156711, 1158.8892398327134, 1407.987377022289]),
'nircam_NRCA5_MASKLWB_F410M': ([224.47335071253724, 219.63966729861886, 214.80543449795005, 205.13533033507952, 188.2074412192212, 137.38496463346019, -17.83562222894261, -261.2035747485929], [909.1669814038728, 914.1663498883313, 919.1656058117073, 929.163767332464, 946.6593597216961, 999.1357836685529, 1158.9331332838815, 1407.9907758972]),
'nircam_NRCA5_MASKLWB_F430M': ([214.81983996761278, 209.9853469890436, 205.150307790454, 195.47860034247833, 178.54793605066607, 127.71736866154333, -27.52566011858835, -270.92160057684305], [909.2165866090118, 914.2155705547175, 919.2144402140111, 929.2118240097407, 946.7060385980412, 999.1781988402536, 1158.9612844516428, 1407.9923338545232]),
'nircam_NRCA5_MASKLWB_F460M': ([197.22198329940534, 192.38602686432841, 187.54952997889995, 177.87492448762077, 160.93924433559778, 110.09405723444729, -45.18948480956806, -288.6357984340964], [909.306221286176, 914.3044996542203, 919.3026605770056, 929.2986173848208, 946.7903040154306, 999.2546417540057, 1159.011560763266, 1407.993830954325]),
'nircam_NRCA5_MASKLWB_F480M': ([189.1771146936578, 184.34049453317698, 179.50333655890245, 169.827416810228, 152.8894621724612, 102.03764722226651, -53.26424478707213, -296.7333078264828], [909.3468536033079, 914.3448074382968, 919.3426423778128, 929.3379428512812, 946.8284667900721, 999.2892068863407, 1159.0340929609592, 1407.993933557927]),
'nircam_NRCA5_MASKLWB_F444W': ([189.3782286329199, 184.54162502456848, 179.70448353679194, 170.0285965626765, 153.09069864402363, 102.23904896091244, -53.0623856175298, -296.53088207232213], [909.3458351297395, 914.3437970927134, 919.3416401966006, 929.3369571083326, 946.8275101669451, 999.2883403655246, 1159.0335277956824, 1407.9939301157592]),
'nircam_NRCA5_MASKLWB_NARROW': ([326.4302428374958, 321.60540550168844, 316.77998521865834, 307.12740532452835, 290.2298599024423, 239.4959313385807, 84.5218377947474, -158.53518027242558], [908.6240188994125, 913.6273412331211, 918.6305689223054, 928.6367279270885, 946.1464900885658, 998.6667747066149, 1158.6109395994054, 1407.9424254073424]),
'nircam_NRCA5_FULL_MASK335R': ([678.3620140798607, 673.4695646803355, 668.5769200148857, 658.7910580210419, 641.6640312386296, 590.2707674503099, 433.5678718668347, 188.73887924668725], [2429.6927481390226, 2434.641816677179, 2439.590550727513, 2449.486999277538, 2466.802428681952, 2518.7214767040136, 2676.653799688099, 2922.225237177742]),
'nircam_NRCA5_FULL_MASK430R': ([1005.405372344448, 1000.5263724675445, 995.6470529216286, 985.8874663020817, 968.8052099876627, 917.5368374481399, 761.122551537595, 516.4333189596628], [2434.862297815893, 2439.8304852730125, 2444.798416635519, 2454.7334963309377, 2472.1172961463485, 2524.2474401955797, 2682.882331637971, 2929.7563874194843]),
'nircam_NRCA5_FULL_MASKLWB': ([1649.730392062202, 1644.8945604448932, 1640.058187882706, 1630.3838295530497, 1613.4485771584323, 1562.604636654689, 1407.3245471415125, 1163.8825184993834], [2416.2985738898296, 2421.2969129692683, 2426.295134874824, 2436.291214465953, 2453.7831186082826, 2506.248129392522, 2666.007299611949, 2914.9937662347374]),
'nircam_NRCA5_FULL_MASKLWB_F250M': ([1755.6123706570975, 1750.7855945536394, 1745.9582427075352, 1736.3018213140303, 1719.3976227365727, 1668.6442670411666, 1513.6159654702833, 1270.4900553672699], [2415.743159464365, 2420.7456521002246, 2425.748046302898, 2435.7525269280413, 2453.2593151087513, 2505.770392301906, 2665.683723526274, 2914.95767598329]),
'nircam_NRCA5_FULL_MASKLWB_F300M': ([1730.8778769635896, 1726.0489332151692, 1721.2194218741893, 1711.558705961108, 1694.647070588964, 1643.872007209718, 1488.7832154341922, 1245.5807542167308], [2415.876269948188, 2420.877811228302, 2425.879249751246, 2435.881805995734, 2453.385184479012, 2505.8857065505376, 2665.7637003438067, 2914.9717310091673]),
'nircam_NRCA5_FULL_MASKLWB_F277W': ([1730.7773245139033, 1725.948372018877, 1721.118851964297, 1711.4581187236763, 1694.5464533473112, 1643.7713024044178, 1488.6822668064565, 1245.4794977506317], [2415.876806735186, 2420.878344124507, 2425.879778738736, 2435.8823271129413, 2453.3856916519635, 2505.886170559419, 2665.7640198639383, 2914.971781017891]),
'nircam_NRCA5_FULL_MASKLWB_F335M': ([1722.833936158297, 1718.0042943709223, 1713.174087638301, 1703.5119888902655, 1686.5979590954541, 1635.8159083044957, 1480.7076646708229, 1237.480663218041], [2415.9191010423774, 2420.9203304590856, 2425.9214557055684, 2435.9233811417544, 2453.425641972634, 2505.9227045020907, 2665.7891181799514, 2914.9755495114796]),
'nircam_NRCA5_FULL_MASKLWB_F360M': ([1699.2043294909013, 1694.3726566695036, 1689.5404266706933, 1679.8743047152598, 1662.9533092636987, 1612.1509365217994, 1456.9861753230325, 1213.6880934224835], [2416.0437143124896, 2421.0440205620034, 2426.0442184700787, 2436.0442766681163, 2453.5432292771875, 2506.030052424334, 2665.8621974496727, 2914.984719357457]),
'nircam_NRCA5_FULL_MASKLWB_F356W': ([1690.255022161773, 1685.4225876930507, 1680.5895989859296, 1670.9219684401046, 1653.9983613219072, 1603.1883712611884, 1448.0024475268303, 1204.6778380076137], [2416.090423541523, 2421.0903774014578, 2426.090221332247, 2436.0895667944883, 2453.587256677906, 2506.070171815671, 2665.8892398327134, 2914.987377022289]),
'nircam_NRCA5_FULL_MASKLWB_F410M': ([1675.4733507125372, 1670.6396672986189, 1665.80543449795, 1656.1353303350795, 1639.2074412192212, 1588.3849646334602, 1433.1643777710574, 1189.796425251407], [2416.166981403873, 2421.1663498883313, 2426.1656058117073, 2436.163767332464, 2453.659359721696, 2506.135783668553, 2665.9331332838815, 2914.9907758972]),
'nircam_NRCA5_FULL_MASKLWB_F430M': ([1665.8198399676128, 1660.9853469890436, 1656.150307790454, 1646.4786003424783, 1629.547936050666, 1578.7173686615433, 1423.4743398814117, 1180.078399423157], [2416.2165866090118, 2421.2155705547175, 2426.214440214011, 2436.2118240097407, 2453.7060385980412, 2506.1781988402536, 2665.9612844516428, 2914.992333854523]),
'nircam_NRCA5_FULL_MASKLWB_F460M': ([1648.2219832994053, 1643.3860268643284, 1638.5495299789, 1628.8749244876208, 1611.9392443355978, 1561.0940572344473, 1405.810515190432, 1162.3642015659036], [2416.306221286176, 2421.3044996542203, 2426.3026605770056, 2436.298617384821, 2453.7903040154306, 2506.2546417540057, 2666.011560763266, 2914.993830954325]),
'nircam_NRCA5_FULL_MASKLWB_F480M': ([1640.1771146936578, 1635.340494533177, 1630.5033365589024, 1620.827416810228, 1603.8894621724612, 1553.0376472222665, 1397.7357552129279, 1154.2666921735172], [2416.346853603308, 2421.344807438297, 2426.342642377813, 2436.3379428512812, 2453.828466790072, 2506.2892068863407, 2666.0340929609592, 2914.993933557927]),
'nircam_NRCA5_FULL_MASKLWB_F444W': ([1640.37822863292, 1635.5416250245685, 1630.704483536792, 1621.0285965626765, 1604.0906986440236, 1553.2390489609124, 1397.9376143824702, 1154.4691179276779], [2416.3458351297395, 2421.3437970927134, 2426.3416401966006, 2436.3369571083326, 2453.827510166945, 2506.2883403655246, 2666.0335277956824, 2914.993930115759]),
'nircam_NRCB1_FULL': ([1023.4999999999695, 1013.6948850433738, 1003.8885247270867, 984.2720805122318, 949.9314217803774, 846.8199409470665, 531.7802843441067, 37.34285966406321], [1023.499999999792, 1033.6276952353464, 1043.7551424249125, 1064.0092784575052, 1099.451509148447, 1205.7576444618703, 1529.5094156205514, 2034.4277410420382]),
'nircam_NRCB1_FULLP': ([1429.9519704083186, 1420.1428402338292, 1410.3324005366824, 1390.707605771994, 1356.3517218502834, 1253.1899586418467, 937.9559365775901, 443.09723873845155], [1372.0139278999948, 1382.1817985561445, 1392.3493778968682, 1412.6836475847108, 1448.265688610221, 1554.987936237401, 1879.9748063147347, 2386.7077624408194]),
'nircam_NRCB1_SUB160': ([79.45326208972347, 69.67493772395369, 59.89548648271534, 40.333215020009675, 6.088491982612368, -96.72660267225797, -410.78524590946927, -903.4606741512657], [79.46871417018895, 89.49034522439115, 99.51183764043469, 119.55439372640444, 154.6274243030158, 259.8342339617394, 580.316495562017, 1080.4010546935895]),
'nircam_NRCB1_SUB320': ([159.45588722633033, 149.6749195835215, 139.89281553455862, 120.32520988831448, 86.07106048843832, -16.773010581644485, -330.9264018357512, -823.7696102769139], [159.47058712322587, 169.5015925374065, 179.53245095088675, 199.5937138777416, 234.69940037512413, 340.00355178658504, 660.776456708609, 1161.295184540314]),
'nircam_NRCB1_SUB640': ([319.462571689642, 309.6765189893757, 299.8893106911594, 280.31143904598855, 246.0391400360869, 143.13922025567308, -171.19723251816276, -664.3654525769036], [319.4753202450481, 329.52487119686646, 339.57425808355094, 359.6725266050091, 394.84280728878457, 500.3394596589618, 821.6866504909555, 1323.0612660512238]),
'nircam_NRCB1_FP4MIMF': ([1817.0072950727222, 1807.1878791029708, 1797.3670954493487, 1777.7214392448545, 1743.3284960331034, 1640.0514334070622, 1324.4302427134605, 828.8718302593838], [1819.0002685699587, 1829.2080074439523, 1839.415373523953, 1859.8289708947723, 1895.5490395977386, 2002.6791550158096, 2328.847596625401, 2837.214261436579]),
'nircam_NRCB1_SUB64P': ([29.4988225048603, 19.678334551866328, 9.856442039449348, -9.791541976554527, -44.188909330523074, -147.48187028873008, -463.1745571312415, -958.9076808522093], [17.99383091553068, 28.219402355422517, 38.444567528908465, 58.893662031301346, 94.67552314631348, 201.9884504210895, 528.6885233984617, 1037.797292431364]),
'nircam_NRCB1_SUB160P': ([79.5009370140715, 69.68002051968824, 59.85770908455993, 40.20891595680882, 5.810224211904824, -97.48602227399306, -413.18263650177505, -908.9048667631184], [79.49547590393786, 89.71713908864831, 99.93840153640167, 120.3797072974337, 156.14799236671797, 263.4206179244693, 590.0021399640141, 1098.940613172943]),
'nircam_NRCB1_SUB400P': ([199.50623126060214, 189.68660072806597, 179.86559695866504, 160.2194839423114, 125.82568852434952, 22.54567368535845, -293.08798878806215, -788.6754101584962], [199.49937608033133, 209.70992690557796, 219.92009989146004, 240.3392958400609, 276.0691126758479, 383.2280857833898, 709.4806290089446, 1217.9653433496887]),
'nircam_NRCB1_MASK210R': ([267.4913112752124, 257.5684600544075, 247.64450365831203, 227.79329143487348, 193.04316873465382, 88.71421593362743, -229.92316016548625, -729.5535982465427], [1629.4449557780254, 1639.6241281495068, 1649.8024955293881, 1670.1567924207757, 1705.7688701196794, 1812.5421495269156, 2137.2990931344557, 2642.338056421541]),
'nircam_NRCB1_TAMASK210R': ([13.864384630637915, 3.970125727441882, -5.925335023403932, -25.719846905777104, -60.37168250309833, -164.41304312115972, -482.2445106452899, -980.855623744898], [1372.925347180727, 1383.1229371766303, 1393.3198550611032, 1413.711653883594, 1449.3906583408366, 1556.3748679690484, 1881.8753931262358, 2388.428560278039]),
'nircam_NRCB2_FULL': ([1023.5000000000703, 1013.9079041179562, 1004.3143570627115, 985.1229240773988, 951.524071470462, 850.6231841973535, 542.1857747587015, 57.6461552126068], [1023.4999999996576, 1033.538429727385, 1043.5771197022311, 1063.6552640146124, 1098.79438422909, 1204.22826342966, 1525.663837394558, 2028.0271050930853]),
'nircam_NRCB2_SUB160': ([79.52011851472446, 69.80409300503845, 60.08691794438539, 40.64913062424682, 6.622034884416507, -95.54188148477306, -407.63265149315293, -897.2923539805242], [79.50257639633264, 89.54708862479629, 99.59152767123578, 119.68017349248521, 154.83449213938684, 260.29000303472776, 581.5758552079806, 1083.0639732446107]),
'nircam_NRCB2_SUB320': ([159.52065408344959, 149.81402590862047, 140.10622484218334, 120.68711548961232, 86.69248148225722, -15.375769321429516, -327.19118536617293, -816.473550464392], [159.50015181698973, 169.54530393023538, 179.59040852933185, 199.68046242332957, 234.83749199633667, 340.30302033236376, 661.6370326336732, 1163.257083490179]),
'nircam_NRCB2_SUB640': ([319.5203260378456, 309.8330746010872, 300.1446032623435, 280.76401248357854, 246.8363346460053, 144.96551309940844, -166.28022139781797, -654.7747403651372], [319.4962019480199, 329.5420255130216, 339.5878534356152, 359.6795093641483, 394.83983997729274, 500.3190556651507, 821.729789489671, 1323.5799449807773]),
'nircam_NRCB2_FP3MIMF': ([1818.9846722006555, 1809.524973143914, 1800.0634456732782, 1781.1349283207187, 1747.9926154639659, 1648.4349073046326, 1343.864808508496, 864.7743476197944], [228.0379724918248, 238.0422480511844, 248.04719477835283, 268.0590767720313, 303.0861213429872, 408.2126247428255, 728.9589156262715, 1230.896995649998]),
'nircam_NRCB3_FULL': ([1023.500000000141, 1013.6005948622081, 1003.7004710909277, 983.8980855516143, 949.2371457421468, 845.2044803715917, 527.7430824674034, 30.835706174167626], [1023.4999999998247, 1033.4444492531209, 1043.3883640586964, 1063.2745718859462, 1098.0701352940937, 1202.4144852899165, 1519.9716911934356, 2014.4481486470022]),
'nircam_NRCB3_SUB160': ([79.53071096510439, 69.71738822816064, 59.90300532421679, 40.271070990710996, 5.9050829096916, -97.26884074629925, -412.373577901923, -906.5256268604702], [79.48294600897805, 89.4605926888238, 99.43806334494491, 119.39246363772281, 154.31086065639408, 259.0509961303143, 578.0845834057224, 1075.817542315366]),
'nircam_NRCB3_SUB320': ([159.5283447317131, 149.70653701675155, 139.8836935940917, 120.23491183728538, 85.83967823115017, -17.420141865645064, -332.76873053463555, -827.2399437908836], [159.48583354201378, 169.46188869788853, 179.43774157277312, 199.38882733363926, 234.30116947273427, 339.02117418910905, 657.9744147914666, 1155.51734292054]),
'nircam_NRCB3_SUB640': ([319.52287722850406, 309.68470739048826, 299.845552375906, 280.1642996474866, 245.71273353829838, 142.2877387993808, -173.52667676315787, -668.5945502069346], [319.49128545589826, 329.46353018305354, 339.4355187116407, 359.3787134815615, 394.2767227248744, 498.9496691790333, 817.7199361472972, 1314.8409152737163]),
'nircam_NRCB3_FP5MIMF': ([228.01116337310032, 218.0703964426432, 208.12936305439916, 188.246525125375, 153.4492333238088, 49.042462836049936, -269.20593987324696, -766.0749269573212], [1818.9721823057976, 1828.8569145521692, 1838.7406524720323, 1858.5051168958598, 1893.083118218819, 1996.7393167824669, 2311.8422318379744, 2801.208660519568]),
'nircam_NRCB3_MASKSWB': ([255.3713236314859, 245.37963359761102, 235.38769574570256, 215.4031064023427, 180.42794249335157, 75.48918741348268, -244.36102874486198, -743.6457986911316], [1620.4309399374752, 1630.3580058688704, 1640.283727071067, 1660.13110016462, 1694.8507624825106, 1798.9050863471257, 2114.9541483864446, 2604.8696738912863]),
'nircam_NRCB3_TAMASKSWB': ([-1.776703145160127, -11.755036392589432, -21.733580040696552, -41.69126825282433, -76.6189910336883, -181.41259368338854, -500.792878873762, -999.2506120654577], [1362.5758594968638, 1372.4846980776438, 1382.3922592478616, 1402.2035153583106, 1436.8606303982915, 1540.7324451632599, 1856.2758730240798, 2345.5739714119672]),
'nircam_NRCB3_TAMASKSWBS': ([1.4764258208199408, -8.507060901215482, -18.490998748209677, -38.460202465379325, -73.41043615572835, -178.29008299438465, -498.117937958285, -997.9288440095174], [1364.9639855339083, 1374.929536697393, 1384.8939660470414, 1404.8194289430603, 1439.677934261158, 1544.165958134874, 1861.7090455467187, 2354.5677512985126]),
'nircam_NRCB4_FULL': ([1023.4999999998074, 1013.749198748202, 1003.9974323063026, 984.4910163704407, 950.3456042462423, 847.8404798345202, 534.8409924487244, 44.19131090030453], [1023.4999999998263, 1033.3460900550003, 1043.1921658129147, 1062.8842613944125, 1097.3451812765295, 1200.724775790512, 1515.7230132056343, 2007.5023162620191]),
'nircam_NRCB4_SUB160': ([79.464686133952, 69.67420159753283, 59.882629841574726, 40.296236353097584, 6.009681818370609, -96.9279832733232, -411.332791409714, -904.4626556996992], [79.46783592597149, 89.4323267533282, 99.3967037780917, 119.32510376434038, 154.19860171368146, 258.8086696299956, 577.4900881012709, 1074.81968020891]),
'nircam_NRCB4_SUB320': ([159.46493445943725, 149.67742577473155, 139.88883950822583, 120.30844593331085, 86.03248419687225, -16.872690023613586, -331.1717817837464, -824.1152304035625], [159.46810086328583, 169.42291726860185, 179.37762795755725, 199.28671954666788, 234.1265058024976, 338.63603695027905, 657.0167009924439, 1153.894479482653]),
'nircam_NRCB4_SUB640': ([319.46729985203, 309.6859488661314, 299.9035398001679, 280.3355591997606, 246.0815084728829, 143.24351329784213, -170.83718213708744, -663.3956737144721], [319.470224951383, 329.40549774722876, 339.3406810191441, 359.21076636468456, 393.9824476664626, 498.28885751499683, 816.061667921892, 1312.0251326853527]),
'nircam_NRCB4_DHSPIL': ([1034.9981191303127, 1025.2435529340971, 1015.4880237656682, 995.9740890450153, 961.8155405359003, 859.2711733032263, 546.1537260767877, 55.325734908287586], [1078.9981313689373, 1088.848529862669, 1098.6989029290128, 1118.3995597867854, 1152.87535515671, 1256.298764276925, 1571.4230395372017, 2063.3759666212472]),
'nircam_NRCB4_DHSPIL_SUB96': ([46.998119130312716, 37.243552934097124, 27.48802376566823, 7.974089045015262, -26.184459464099746, -128.7288266967737, -441.84627392321227, -932.6742650917124], [47.998131368937266, 57.84852986266901, 67.69890292901277, 87.3995597867854, 121.87535515670993, 225.2987642769251, 540.4230395372017, 1032.3759666212472]),
'nircam_NRCB4_DHSPIL_WEDGES': ([1026.4906505703855, 1016.6401779940123, 1006.7888832394027, 987.0838423750638, 952.5922359753673, 849.0595025927595, 533.0388486642339, 38.07405355910248], [2454.9729612183132, 2464.894397512401, 2474.815500417172, 2494.6566911055806, 2529.375439420531, 2633.504711276755, 2950.5593260547153, 3444.789331969499]),
'nircam_NRCB4_FP1': ([1818.9656633499046, 1809.180904019394, 1799.395075237696, 1779.8202210479585, 1745.5540316986653, 1642.678776637821, 1328.4765387107132, 835.702468725207], [1816.968737234975, 1826.9151071665399, 1836.861378724439, 1856.7536140925854, 1891.5639727369078, 1995.9857631750606, 2314.104120994748, 2810.5882968922115]),
'nircam_NRCB4_FP1_SUB8': ([2.9656633499046166, -6.819095980606107, -16.60492476230411, -36.17977895204149, -70.44596830133469, -173.3212233621789, -487.52346128928684, -980.297531274793], [3.968737234974924, 13.915107166539883, 23.861378724438964, 43.753614092585394, 78.56397273690777, 182.98576317506058, 501.104120994748, 997.5882968922115]),
'nircam_NRCB4_FP1_SUB64': ([30.965663349904617, 21.180904019393893, 11.395075237695892, -8.179778952041488, -42.44596830133469, -145.3212233621789, -459.52346128928684, -952.297531274793], [31.968737234974924, 41.91510716653988, 51.861378724438964, 71.7536140925854, 106.56397273690777, 210.98576317506058, 529.104120994748, 1025.5882968922115]),
'nircam_NRCB4_FP2MIMF': ([230.0182899090728, 220.31004901356357, 210.60097389983252, 191.18033526489285, 157.18630868193713, 55.145255201905115, -256.34317087440627, -744.3054808800339], [228.02645449295676, 237.7646625401145, 247.50296214874822, 266.97982162016524, 301.065085616685, 403.32527550767315, 714.9788972246961, 1201.7197825309204]),
'nircam_NRCB5_FULL': ([1023.4999999999802, 1018.6774324794693, 1013.854332574907, 1004.2065435659715, 987.3178556331189, 936.6139569836082, 781.7589292430619, 538.9414559465642], [1023.4999999999875, 1028.41390345414, 1033.3277287486342, 1043.1551361836425, 1060.352273811675, 1111.93653070778, 1269.0565328528533, 1514.1428671872418]),
'nircam_NRCB5_FULLP': ([1727.0239671373756, 1722.185935584941, 1717.3472530449392, 1707.6679436978154, 1690.7229597933906, 1639.841570833131, 1484.3675707528982, 1240.3537579706776], [1728.0146270938676, 1733.0003563559299, 1737.985891693363, 1747.9563708473934, 1765.4027606313919, 1817.7260853331668, 1977.0148888566828, 2225.215148074108]),
'nircam_NRCB5_SUB160': ([79.49895642472939, 74.67516141585122, 69.8508448876778, 60.20065524810582, 43.307870986685884, -7.407513464929593, -162.28998565577933, -405.12619639417426], [79.50031301068191, 84.41141524357681, 89.32243455587809, 99.14421574109792, 116.33146158094269, 167.88569221758962, 324.9108892412746, 569.8384858788261]),
'nircam_NRCB5_SUB320': ([159.49895642472939, 154.67516141585122, 149.8508448876778, 140.20065524810582, 123.30787098668588, 72.5924865350704, -82.28998565577933, -325.12619639417426], [159.5003130106819, 164.4114152435768, 169.3224345558781, 179.14421574109792, 196.3314615809427, 247.88569221758962, 404.9108892412746, 649.8384858788261]),
'nircam_NRCB5_SUB640': ([319.4989564247294, 314.6751614158512, 309.8508448876778, 300.2006552481058, 283.3078709866859, 232.5924865350704, 77.71001434422067, -165.12619639417426], [319.5003130106819, 324.4114152435768, 329.3224345558781, 339.1442157410979, 356.3314615809427, 407.8856922175896, 564.9108892412746, 809.8384858788261]),
'nircam_NRCB5_SUB64P': ([27.974320751310188, 23.133001464763765, 18.290976210312238, 8.604817167721194, -8.352677235709507, -59.27553109951964, -214.91066067469455, -459.27217048527837], [53.11518238778581, 58.125197085457785, 63.134948946593795, 73.15365353612833, 90.68376713053976, 143.25305291005725, 303.24074682289756, 552.3632650128993]),
'nircam_NRCB5_SUB160P': ([101.49537376755939, 96.65366377953069, 91.81125345561873, 82.12434111497032, 65.16558218878208, 14.239340073623907, -141.4024889950374, -385.7641211047887], [132.19614779575886, 137.20448817241595, 142.2125687849341, 152.22794014783722, 169.7522506211817, 222.30436162918613, 382.242026959294, 631.2942425106467]),
'nircam_NRCB5_SUB400P': ([282.91826525939814, 278.0766926908768, 273.2344322618046, 263.54785698783644, 246.58980675724683, 195.66658399578614, 40.04186593730083, -204.27107908936114], [313.35868536822045, 318.3623899285276, 323.3658480393806, 333.37201452665977, 350.8803461711807, 403.3855295551075, 563.1900239930346, 812.0672406678768]),
'nircam_NRCB5_TAPSIMG32': ([15.518761014687698, 10.693369368162848, 5.867241323878716, -3.787214657154891, -20.68952223180804, -71.44906758364368, -226.60780346697106, -470.2944241979458], [15.504655607587665, 20.511820248528693, 25.5187996280074, 35.53219254440705, 53.053762213301525, 105.60321547025069, 265.5859974269979, 514.8828708850824]),
'nircam_NRCB5_GRISMC_WFSS': ([1023.4999999999802, 1018.6774324794693, 1013.854332574907, 1004.2065435659715, 987.3178556331189, 936.6139569836082, 781.7589292430619, 538.9414559465642], [1023.4999999999875, 1028.41390345414, 1033.3277287486342, 1043.1551361836425, 1060.352273811675, 1111.93653070778, 1269.0565328528533, 1514.1428671872418]),
'nircam_NRCB5_GRISMR_WFSS': ([1023.4999999999802, 1018.6774324794693, 1013.854332574907, 1004.2065435659715, 987.3178556331189, 936.6139569836082, 781.7589292430619, 538.9414559465642], [1023.4999999999875, 1028.41390345414, 1033.3277287486342, 1043.1551361836425, 1060.352273811675, 1111.93653070778, 1269.0565328528533, 1514.1428671872418]),
'nircam_NRCB5_MASK335R': ([122.72082399540022, 117.81813608201946, 112.91504300584143, 103.10765209780061, 85.94090904257723, 34.41266629739857, -122.85519628559473, -369.08009017187305], [918.9710827542058, 923.9253523074103, 928.8790617250997, 938.7847861492119, 956.1142941213084, 1008.05933853634, 1165.9266147774506, 1411.001653344908]),
'nircam_NRCB5_MASK430R': ([120.76705598088552, 115.85788740921816, 110.94843181805243, 101.12867156704328, 83.94143000460065, 32.36058806079859, -124.9813816266942, -371.03372416607203], [918.0890833259541, 923.0148553276335, 927.9399985180626, 937.7883832708039, 955.0168757054607, 1006.6536266707153, 1163.5292766887392, 1406.8732021063415]),
'nircam_NRCB5_MASKLWB': ([118.88347197401094, 113.9734767794663, 109.06332331886381, 99.2425553781652, 82.05480480104768, 30.482198158359324, -126.73908880067995, -372.27793393888805], [916.8888222782871, 921.7841893762075, 926.678849457142, 936.4660319089912, 953.5866551708723, 1004.8938069583528, 1160.7053004141108, 1402.177631547047]),
'nircam_NRCB5_TAMASK335R': ([-3.8090515900735227, -8.699441415248202, -13.59031361319012, -23.373495209864814, -40.49862207976639, -91.9078301971117, -248.86929903538612, -494.80483950634334], [791.0580948670531, 796.0283708724719, 800.9981754494274, 810.9363576027963, 828.3235313576815, 880.4483086487953, 1038.9294560942863, 1285.1895216172375]),
'nircam_NRCB5_TAMASK430R': ([-5.870245290761204, -10.770637390299044, -15.67140138040213, -25.474034247206646, -42.632130019377655, -94.13197494991891, -251.2902891358035, -497.2717858386238], [790.3432743806243, 795.287013467916, 800.2302196879155, 810.1150198715895, 827.4081758399743, 879.2462257022748, 1036.8074274167507, 1281.4703122281999]),
'nircam_NRCB5_TAMASKLWB': ([-7.911905623820758, -12.816902351214367, -17.722152387142273, -27.53340017975688, -44.70541862877974, -96.23807646557293, -253.40922758474244, -499.1169576710357], [789.3059381536968, 794.2210628352223, 799.1355850760174, 808.962807353611, 826.1545219213453, 877.6829287791429, 1034.2476304695056, 1277.168925024867]),
'nircam_NRCB5_TAMASKLWBL': ([-9.889873593673883, -14.793779122937508, -19.697803906747936, -29.506197044411124, -46.67191270710691, -98.17547778826327, -255.15794789739812, -500.22929803030206], [787.9480343735431, 792.8320731387116, 797.7154281745507, 807.4800706435226, 824.5614751391468, 875.75272206912, 1031.2278320905398, 1272.2293612578233]),
'niriss_NIS_CEN': ([1023.4999999999874, 1018.9088930953566, 1014.3177691298857, 1005.1354699099895, 989.0662812851969, 940.8574524783501, 793.9239331008537, 564.3126252415295], [1023.4999999998655, 1028.2283385271237, 1032.9567315171885, 1042.4136822999405, 1058.9638820923024, 1108.6187006171883, 1259.989449601091, 1496.641253217673]),
'niriss_NIS_AMI1': ([45.00720718135949, 40.416066158526746, 35.82492952999496, 26.642668092404392, 10.573741506461374, -37.63292825843121, -184.5504423395364, -414.128189602216], [39.99845586323488, 44.733976859967925, 49.46941777484892, 58.94006404815241, 75.51296353267128, 125.22652354861373, 276.69825535514804, 513.3243206435903]),
'niriss_NIS_AMI2': ([17.00711829790407, 12.41597190975267, 7.824828288564049, -1.3574519548244552, -17.426426592765665, -65.63335005793533, -212.5524889957078, -442.13436183658905], [56.998128646000396, 61.73343893597996, 66.46867379901698, 75.93892184979097, 92.51116854832549, 142.22309885284176, 293.6926879056682, 530.3226058187541]),
'niriss_NIS_AMI3': ([17.00771149404909, 12.416513145510862, 7.825318721225813, -1.3570596952035885, -17.426195323876073, -65.63352300202143, -212.5532663627357, -442.1348054439594], [23.99858586739174, 28.734646608007893, 33.470625079485785, 42.94233992858699, 59.5170885738404, 109.2360392740876, 260.7228129149554, 497.3685862885593]),
'niriss_NIS_AMI4': ([59.007423839113926, 54.41626726451864, 49.825116240520174, 40.642829439487286, 24.573869250690223, -23.632823000236613, -170.5497879577972, -400.1254635784859], [22.99874633782042, 27.73457920059741, 32.470327833202646, 41.94157716521056, 58.515492656779315, 108.23180701587967, 259.7093964564616, 496.33806136689213]),
'niriss_NIS_AMITA': ([32.006064968208875, 27.41503005718505, 22.823996881497578, 13.64193446280342, -2.426668670016852, -50.63254879459657, -197.54909181647156, -427.12821566874027], [31.997607471081437, 36.731607651451895, 41.465542514086565, 50.933220736430826, 67.50106622668386, 117.20051069145609, 268.63828299312615, 505.2348272486878]),
'niriss_NIS_WFSS_OFFSET': ([1021.5000270905317, 1016.9089161911488, 1012.317788245434, 1003.1354811072239, 987.0662787560752, 938.8574096676024, 791.9237732186542, 562.3122837410754], [1020.099946265382, 1024.8282944714213, 1029.5566969596382, 1039.0136662054224, 1055.5638966460647, 1105.2187953154457, 1256.5897031704453, 1493.241641860147]),
'niriss_NIS_WFSS64': ([2015.4961485292135, 2010.906989282463, 2006.317813496251, 1997.1394118760707, 1981.0770463545157, 1932.8886728850098, 1786.0165248317835, 1556.4931005028407], [2015.5111131408703, 2020.2332362730644, 2024.9553991949615, 2034.399843565157, 2050.9279969003837, 2100.5152312591053, 2251.660759573061, 2487.866958307759]),
'niriss_NIS_WFSS64R': ([31.498014871190207, 26.908672575436412, 22.319347156971162, 13.140745084091577, -2.921661709368891, -51.10793638549012, -197.9568755358721, -427.41608548742715], [795.0043057970163, 799.725302114086, 804.4463011297355, 813.8883103992773, 830.4118788698848, 879.9832793092559, 1031.0717032898424, 1267.2249804321784]),
'niriss_NIS_WFSS64R3': ([31.495044698902348, 26.906208412085107, 22.317359158388854, 13.139621006227117, -2.921551713662211, -51.106135354815706, -197.96559923597647, -427.4707669977556], [1748.0106384563303, 1752.7325520320026, 1757.4545161244598, 1766.8985958903688, 1783.426221771741, 1833.012803976082, 1984.1672275506482, 2220.4339360963922]),
'niriss_NIS_WFSS64C': ([794.9945975915822, 790.4024998518081, 785.810403209003, 776.6262146199454, 760.5539071004162, 712.3373015931452, 565.3986400028587, 335.8591332219878], [31.498930370720245, 36.23291924814794, 40.966947656249886, 50.43512101510714, 67.0047874229358, 116.71634606242515, 268.2356327078951, 504.9912757202055]),
'niriss_NIS_WFSS64C3': ([1747.9941613502156, 1743.4042786083346, 1738.8143772105304, 1729.6345184205504, 1713.5695858709137, 1665.3734210741038, 1518.4777418626356, 1288.924552167759], [31.508825676076867, 36.23354495786202, 40.9583074738091, 50.407961100454486, 66.94526155307722, 116.56014737107398, 267.7911990341054, 504.1306102951021]),
'niriss_NIS_WFSS128': ([1967.995396248094, 1963.406185892808, 1958.8169587805223, 1949.638453871584, 1933.5759056323668, 1885.3869705427935, 1738.5130266393453, 1508.9867868998224], [1968.010853369844, 1972.7334498473692, 1977.4560896055232, 1986.901498229857, 2003.4313735210171, 2053.0240478677747, 2204.188972673742, 2440.435678237672]),
'niriss_NIS_WFSS128R': ([63.49803222240689, 58.908714708249136, 54.31941231498104, 45.14085107215624, 29.078499232703507, -19.10773185056314, -165.95753458488798, -395.42033703467473], [795.0041432935668, 799.7253059367506, 804.4464727972911, 813.8888222813423, 830.4130004786462, 879.98633657907, 1031.0815620468009, 1267.2476848331978]),
'niriss_NIS_WFSS128R3': ([63.494874659899324, 58.90596756132436, 54.31704677477683, 45.13916344247173, 29.07773012337384, -19.107680490096755, -165.96995711422915, -395.4796406597684], [1748.0105293156312, 1752.7327287351172, 1757.4549794068084, 1766.899634505868, 1783.4282740318872, 1833.0179460738045, 1984.1821559225882, 2220.464767800651]),
'niriss_NIS_WFSS128C': ([794.9948206009941, 790.4027030729646, 785.8105856351979, 776.6263523983109, 760.553956738458, 712.3370066988854, 565.3964556963342, 335.85087056431917], [63.49974338956417, 68.23373914156696, 72.96777726008554, 82.43597865166043, 99.00572229102681, 148.71773801092877, 300.2407751420151, 537.0110360260142]),
'niriss_NIS_WFSS128C3': ([1747.9940400974078, 1743.4042037834147, 1738.8143488397006, 1729.6345829984539, 1713.5698130431374, 1665.3741318276593, 1518.4797892254753, 1288.9276786212372], [63.50908869920886, 68.23381245881615, 72.95858151259017, 82.40825449875138, 98.94560953685686, 148.5608274710612, 299.79472125927305, 536.1457412055715]),
'niriss_NIS_SUB64': ([31.496148529213542, 26.906989282462973, 22.317813496250892, 13.139411876070653, -2.922953645484313, -51.11132711499022, -197.9834751682165, -427.50689949715934], [31.511113140870293, 36.23323627306445, 40.9553991949615, 50.39984356515697, 66.92799690038373, 116.51523125910535, 267.6607595730611, 503.866958307759]),
'niriss_NIS_SUB128': ([63.49563052206281, 58.906437501136224, 54.31722779209667, 45.13875789080839, 29.07627153495514, -19.11247366640964, -165.98581281872976, -395.51110892016504], [63.510937949264644, 68.23338405506502, 72.95587233350261, 82.40097463746088, 98.93030290303432, 148.5212489874857, 299.6800082650907, 535.9138149645019]),
'niriss_NIS_SUB256': ([127.4948204394068, 122.90555185386529, 118.31626630929304, 109.13764394556074, 93.07488834942956, 44.88531890733293, -101.99063070515876, -331.51997311937953], [127.51060664795386, 132.23364939840826, 136.95673870526298, 146.40305636306607, 162.9345549178172, 212.53235769331877, 363.71559218350103, 600.0007285649372]),
'niriss_NIS_SUBAMPCAL': ([1023.4999999999874, 1018.9088930953566, 1014.3177691298857, 1005.1354699099895, 989.0662812851969, 940.8574524783501, 793.9239331008537, 564.3126252415295], [1023.4999999998655, 1028.2283385271237, 1032.9567315171885, 1042.4136822999405, 1058.9638820923024, 1108.6187006171883, 1259.989449601091, 1496.641253217673]),
'niriss_NIS_SUBSTRIP96': ([1953.9963901635958, 1949.4076052300134, 1944.818818800381, 1935.6412401527464, 1919.5804522902406, 1871.3977761015974, 1724.5496419849055, 1495.0700248962116], [1198.0063806230603, 1202.7273640092906, 1207.4483889696244, 1216.890565378847, 1233.4147889253584, 1282.990788447082, 1434.1140067578392, 1670.3625287800257]),
'niriss_NIS_SUBSTRIP256': ([1953.9963901635958, 1949.4076052300134, 1944.818818800381, 1935.6412401527464, 1919.5804522902406, 1871.3977761015974, 1724.5496419849055, 1495.0700248962116], [1198.0063806230603, 1202.7273640092906, 1207.4483889696244, 1216.890565378847, 1233.4147889253584, 1282.990788447082, 1434.1140067578392, 1670.3625287800257]),
'niriss_NIS_FP1MIMF': ([1024.0000598519996, 1019.4089631401012, 1014.8178494273761, 1005.6355708780177, 989.5664188824393, 941.3577020389527, 794.4245058052485, 564.8134223198135], [1011.9998729320552, 1016.728178118034, 1021.4565370424514, 1030.9134175591196, 1047.4634877360565, 1097.1178703585251, 1248.4869547054454, 1485.1357450733155]),
'niriss_NIS_FP2MIMF': ([1717.9989992878504, 1713.4094301197667, 1708.8198685430225, 1699.640766580425, 1683.577397993723, 1635.3876014866673, 1488.5227804134124, 1259.0277160384974], [656.0019635977321, 660.725328291801, 665.4486915668522, 674.8954171610637, 691.4272009050769, 741.0229702883553, 892.1835136991907, 1128.445510607468]),
'niriss_NIS_FP3MIMF': ([83.01472177426228, 78.42139375715806, 73.8280560578612, 64.64135194302253, 48.56452945015974, 0.3334122590642892, -146.66122188827444, -376.34058551033877], [195.99661275833216, 200.73732894444072, 205.47801940395846, 214.95932633107964, 231.5513924879706, 281.32620400780434, 433.0154021934894, 670.0503778554576]),
'niriss_NIS_FP4MIMF': ([353.9987782924886, 349.4065461466804, 344.8143320338197, 335.62995979984214, 319.5574980182645, 271.3417536961774, 124.41923730306473, -105.04657329749557], [1867.9994137878416, 1872.7372226995185, 1877.4750744681755, 1886.950904574678, 1903.5340027067227, 1953.2861087092408, 2104.931425153031, 2341.8921462770104]),
'niriss_NIS_FP5MIMF': ([1947.9951454208335, 1943.405911777686, 1938.8166612918812, 1929.6381093851855, 1913.575478129435, 1865.3862887458495, 1718.5115383151224, 1488.9840462631046], [1948.0107593410048, 1952.733544209353, 1957.4563737444375, 1966.9021661265697, 1983.432726717722, 2033.0275662215045, 2184.200218901138, 2420.463122093128]),
'niriss_NIS_AMIFULL': ([1089.0072071813595, 1084.4160661585267, 1079.824929529995, 1070.6426680924044, 1054.5737415064614, 1006.3670717415688, 859.4495576604636, 629.871810397784], [39.99845586323488, 44.733976859967925, 49.46941777484892, 58.94006404815241, 75.51296353267128, 125.22652354861373, 276.69825535514804, 513.3243206435903]),
'fgs_FGS1_FULL': ([1023.5000000000341, 1018.9766405629556, 1014.4532613210031, 1005.4064491793056, 989.5743857912652, 942.0776743440555, 797.3359749693304, 571.2987407328635], [1023.4999999999592, 1027.9608216163967, 1032.4221497134724, 1041.346319553317, 1056.9684428371513, 1103.8710983597032, 1247.1331297833137, 1471.8551277443587]),
'fgs_FGS1_SUB128LL': ([63.5004616982701, 58.94852467859255, 54.396465971702355, 45.29198925391893, 29.358032244916558, -18.45184831593133, -164.2172452801601, -392.06284823604847], [63.487039914943125, 68.01998379508677, 72.55332549005198, 81.62119654949765, 97.49375124578728, 145.13971769431464, 290.5932769681149, 518.5174045903623]),
'fgs_FGS1_SUB128DIAG': ([63.50186035865431, 58.96132090862079, 54.420710613866504, 45.33928324758813, 29.446151961231863, -18.237514554133895, -163.583687521932, -390.67003448053265], [63.501137721873874, 68.00117461448099, 72.50166312594979, 81.50398922074532, 97.2623580017871, 144.5697195164812, 289.0278811945559, 515.5066854906363]),
'fgs_FGS1_SUB128CNTR': ([63.500000000034106, 58.976640562955595, 54.45326132100308, 45.40644917930558, 29.574385791265172, -17.92232565594452, -162.66402503066956, -388.7012592671365], [63.499999999959186, 67.96082161639674, 72.42214971347244, 81.34631955331702, 96.96844283715131, 143.87109835970318, 287.13312978331373, 511.8551277443587]),
'fgs_FGS1_SUB32LL': ([15.500461698270101, 10.948524678592548, 6.396465971702355, -2.7080107460810723, -18.641967755083442, -66.45184831593133, -212.2172452801601, -440.06284823604847], [15.487039914943125, 20.019983795086773, 24.553325490051975, 33.621196549497654, 49.49375124578728, 97.13971769431464, 242.59327696811488, 470.5174045903623]),
'fgs_FGS1_SUB32DIAG': ([15.501860358654312, 10.961320908620792, 6.420710613866504, -2.660716752411872, -18.553848038768137, -66.2375145541339, -211.583687521932, -438.67003448053265], [15.501137721873874, 20.001174614480988, 24.50166312594979, 33.503989220745325, 49.262358001787106, 96.5697195164812, 241.0278811945559, 467.5066854906363]),
'fgs_FGS1_SUB32CNTR': ([15.500000000034106, 10.976640562955595, 6.453261321003083, -2.593550820694418, -18.425614208734828, -65.92232565594452, -210.66402503066956, -436.7012592671365], [15.499999999959186, 19.960821616396743, 24.422149713472436, 33.34631955331702, 48.968442837151315, 95.87109835970318, 239.13312978331373, 463.8551277443587]),
'fgs_FGS1_SUB8LL': ([3.500461698270101, -1.0514753214074517, -5.603534028297645, -14.708010746081072, -30.641967755083442, -78.45184831593133, -224.2172452801601, -452.06284823604847], [3.487039914943125, 8.019983795086773, 12.553325490051975, 21.621196549497654, 37.49375124578728, 85.13971769431464, 230.59327696811488, 458.5174045903623]),
'fgs_FGS1_SUB8DIAG': ([3.501860358654312, -1.0386790913792083, -5.579289386133496, -14.660716752411872, -30.553848038768137, -78.2375145541339, -223.583687521932, -450.67003448053265], [3.5011377218738744, 8.001174614480988, 12.50166312594979, 21.503989220745325, 37.262358001787106, 84.5697195164812, 229.0278811945559, 455.5066854906363]),
'fgs_FGS1_SUB8CNTR': ([3.500000000034106, -1.0233594370444052, -5.546738678996917, -14.593550820694418, -30.425614208734828, -77.92232565594452, -222.66402503066956, -448.7012592671365], [3.4999999999591864, 7.960821616396743, 12.422149713472436, 21.346319553317016, 36.968442837151315, 83.87109835970318, 227.13312978331373, 451.8551277443587]),
'fgs_FGS1_FP1MIMF': ([1012.0001800831649, 1007.477732813455, 1002.9552650485198, 993.9102737893475, 978.0813902995271, 930.5941677920623, 785.8809174143516, 559.8867181796014], [994.0002605345572, 998.4582284211134, 1002.9167080713455, 1011.8351968657213, 1027.447429161056, 1074.3207998739417, 1217.497171113488, 1442.096129925519]),
'fgs_FGS1_FP2MIMF': ([1947.9931600692744, 1943.505556932988, 1939.017636676333, 1930.0408505563198, 1914.3284710217654, 1867.1689580121142, 1723.2527254831907, 1497.8957109564615], [100.00023810052437, 104.32327917394343, 108.64712979327794, 117.29725383075731, 132.44271360674998, 177.93764703918305, 317.1153885924912, 536.0757766610534]),
'fgs_FGS1_FP3MIMF': ([265.9979542287168, 261.50990718881815, 257.02194140009067, 248.04625933370312, 232.3396462418183, 185.22671328737601, 41.723342066251234, -182.17184411184758], [449.0007419995261, 453.4073924809636, 457.81459787063375, 466.63066756616706, 482.0640816109627, 528.4041668906549, 669.9845799296465, 892.1779873158505]),
'fgs_FGS1_FP4MIMF': ([88.00922028601985, 83.5122017511477, 79.01546182710024, 70.02282357098784, 54.28843675561302, 7.106689011746312, -136.47218735414617, -360.0806832069652], [1958.0041318584877, 1962.5390970417316, 1967.0742692250722, 1976.1452288374394, 1992.021352075703, 2039.6640024481349, 2184.977781873362, 2412.2928838337257]),
'fgs_FGS1_FP5MIMF': ([1958.002728021962, 1953.4516182798843, 1948.9003878625595, 1939.7975707598353, 1923.8665278173826, 1876.0654638306908, 1730.3276244012318, 1502.5271534383237], [1941.9912290670766, 1946.5213589477958, 1951.051892914597, 1960.114167331624, 1975.9769882574997, 2023.594214168415, 2168.9644432368978, 2396.771192266453]),
'fgs_FGS2_FULL': ([1023.4999999999363, 1019.0552710766841, 1014.6099539879934, 1005.7175610433771, 990.1502605483066, 943.4060588965484, 800.5711324574082, 576.3474447442138], [1023.499999999794, 1027.8832869037265, 1032.267367235244, 1041.037902260744, 1056.3939254394895, 1102.5193510916909, 1243.6072929394272, 1465.5169623303518]),
'fgs_FGS2_SUB128LL': ([63.503680923660795, 59.22478469022417, 54.94498447497472, 46.382677789907575, 31.389988192188866, -13.653599521036085, -151.52061589516074, -368.6304534137075], [63.512776032965576, 67.68218346847516, 71.85271472258398, 80.19714273736395, 94.81065842090595, 138.73284072045328, 273.3306856381722, 485.77608177216337]),
'fgs_FGS2_SUB128DIAG': ([63.4984807190217, 59.127043474078846, 54.75486155036606, 46.00826937749093, 30.69461363327855, -15.300160575457994, -155.95762486555896, -377.100416815186], [63.50067208791154, 67.7874601392798, 72.07520472012902, 80.65355753564336, 95.67483158428229, 140.807995562217, 278.9827537075997, 496.674538751794]),
'fgs_FGS2_SUB128CNTR': ([63.499999999936335, 59.05527107668411, 54.609953987993435, 45.717561043377145, 30.15026054830662, -16.593941103451584, -159.42886754259177, -383.6525552557862], [63.499999999794, 67.88328690372646, 72.26736723524391, 81.03790226074398, 96.39392543948952, 142.51935109169085, 283.6072929394272, 505.5169623303518]),
'fgs_FGS2_SUB32LL': ([15.503680923660795, 11.224784690224169, 6.944984474974717, -1.617322210092425, -16.610011807811134, -61.653599521036085, -199.52061589516074, -416.6304534137075], [15.512776032965576, 19.68218346847516, 23.852714722583983, 32.19714273736395, 46.81065842090595, 90.73284072045328, 225.3306856381722, 437.77608177216337]),
'fgs_FGS2_SUB32DIAG': ([15.4984807190217, 11.127043474078846, 6.7548615503660585, -1.991730622509067, -17.30538636672145, -63.300160575457994, -203.95762486555896, -425.100416815186], [15.50067208791154, 19.7874601392798, 24.075204720129022, 32.65355753564336, 47.67483158428229, 92.807995562217, 230.98275370759973, 448.674538751794]),
'fgs_FGS2_SUB32CNTR': ([15.499999999936335, 11.055271076684107, 6.6099539879934355, -2.2824389566228547, -17.84973945169338, -64.59394110345158, -207.42886754259177, -431.6525552557862], [15.499999999794, 19.883286903726457, 24.26736723524391, 33.03790226074398, 48.39392543948952, 94.51935109169085, 235.60729293942723, 457.5169623303518]),
'fgs_FGS2_SUB8LL': ([3.503680923660795, -0.775215309775831, -5.055015525025283, -13.617322210092425, -28.610011807811134, -73.65359952103609, -211.52061589516074, -428.6304534137075], [3.5127760329655757, 7.682183468475159, 11.852714722583983, 20.197142737363947, 34.81065842090595, 78.73284072045328, 213.3306856381722, 425.77608177216337]),
'fgs_FGS2_SUB8DIAG': ([3.4984807190216998, -0.872956525921154, -5.2451384496339415, -13.991730622509067, -29.30538636672145, -75.300160575458, -215.95762486555896, -437.100416815186], [3.500672087911539, 7.787460139279801, 12.075204720129022, 20.65355753564336, 35.67483158428229, 80.807995562217, 218.98275370759973, 436.674538751794]),
'fgs_FGS2_SUB8CNTR': ([3.4999999999363354, -0.9447289233158926, -5.3900460120065645, -14.282438956622855, -29.84973945169338, -76.59394110345158, -219.42886754259177, -443.6525552557862], [3.4999999997939995, 7.883286903726457, 12.26736723524391, 21.037902260743977, 36.39392543948952, 82.51935109169085, 223.60729293942723, 445.5169623303518]),
'fgs_FGS2_FP1MIMF': ([1010.0001598632385, 1005.5539706420911, 1001.1071969629212, 992.211901960755, 976.6395585535008, 929.8805007358789, 787.0028211507273, 562.7199434404216], [1030.999993723277, 1035.3849113336078, 1039.770619178724, 1048.5443996538504, 1063.9060716407605, 1110.0482090709081, 1251.1849114237189, 1473.164226800142]),
'fgs_FGS2_FP2MIMF': ([1721.0024921104991, 1716.6666768650832, 1712.330050873989, 1703.65437235823, 1688.4641795552775, 1642.834937012381, 1503.2465170851362, 1283.638454287057], [376.00039124022544, 380.2453874341011, 384.4914037537769, 392.9864908325559, 407.8626619244865, 452.5651907301018, 589.4707662522569, 805.3097145675641]),
'fgs_FGS2_FP3MIMF': ([100.00314044746267, 95.54955634915598, 91.09547775100805, 82.18584278057881, 66.58927023891192, 19.764136233030626, -123.25376474028303, -347.57144663947656], [99.0044175272426, 103.29432543662267, 107.58513296663148, 116.16944090345987, 131.20058824067314, 176.35918107755572, 314.5725309776475, 532.2065275784817]),
'fgs_FGS2_FP4MIMF': ([100.00236800226298, 95.4663660248458, 90.93007190075991, 81.85661297771401, 65.97529650231229, 18.310810099828927, -127.12735385735573, -354.81075280149435], [1946.997779255109, 1951.5137505785374, 1956.030210124378, 1965.0645879890933, 1980.879398756028, 2028.3587588196615, 2173.365544437291, 2400.7734707303885]),
'fgs_FGS2_FP5MIMF': ([1947.9978931028852, 1943.5826792855096, 1939.166780267125, 1930.3329323610747, 1914.8671517475218, 1868.4203782084874, 1726.4259296423133, 1503.3166644025673], [1947.001791135268, 1951.4536917316716, 1955.9062845517615, 1964.8135410061823, 1980.4078531299479, 2027.2407173797942, 2170.416103266566, 2395.3817151395424]),
}


nircam_apertures = []
niriss_apertures = []
fgs_apertures = []
for key in reference_file_values:
    instrument = key.split('_')[0]
    detector = key.split('_')[1]
    aperture = key.split('{}_'.format(instrument))[1]
    if instrument == 'nircam':
        nircam_apertures.append(aperture)
    elif instrument == 'niriss':
        niriss_apertures.append(aperture)
    elif instrument == 'fgs':
        fgs_apertures.append(aperture)

def find_distortion_file(inst, aper):
    """Identify the appropriate distortion reference file for a given
    instrument/aperture combination
    Parameters
    ----------
    inst : str
        Instrument name
    aper : str
        Aperture name (e.g NRCA1_FULL)
    """



def test_locations():
    """Test RA, Dec to x,y translation. This function does the
    translation using pysiaf. This is the truth case we will
    compare against the version translated via distortion
    reference file, which is in ``reference_file_values``. We have to
    test like this because Github Actions CI does not have access to the
    distortion reference file itself.
    """
    # Apertures to skip for this testing. These apertures don't use the
    # reference files in the intended way.
    apertures_to_skip = ['NRCA5_GRISMC_WFSS', 'NRCA5_GRISMR_WFSS', 'NRCB5_GRISMC_WFSS',
                         'NRCB5_GRISMR_WFSS', 'NIS_SUBSTRIP96', 'NIS_SUBSTRIP256']

    # RA, Dec of pointing
    c = catalog_seed_image.Catalog_seed()

    pointing_ra = 12.0
    pointing_dec = 12.0
    rotation = 0.0

    ra_list = []
    dec_list = []
    delta_pix = np.array([0, 10, 20, 40, 75, 180, 500, 1000])
    delta_deg = delta_pix * 0.031 / 3600.
    for delta in delta_deg:
        ra_entry = pointing_ra + delta
        dec_entry = pointing_dec + delta
        ra_list.append(ra_entry)
        dec_list.append(dec_entry)

    xy_vals = {}
    instrument_list = ['niriss', 'fgs']

    for instrument in instrument_list:
        siaf = siaf_interface.get_instance(instrument)
        for aperture in siaf.apernames:
            if 'WEDGE' in aperture or 'MASK' in aperture or aperture in apertures_to_skip:
                continue

            key_value = '{}_{}'.format(instrument, aperture)
            if key_value not in reference_file_values.keys():
                continue

            c.local_roll, c.attitude_matrix, c.ffsize, \
                c.subarray_bounds = siaf_interface.get_siaf_information(siaf,
                                                                        aperture,
                                                                        pointing_ra, pointing_dec,
                                                                        rotation)
            c.coord_transform = None
            c.siaf = siaf[aperture]

            xvals_from_siaf = []
            yvals_from_siaf = []
            for ra, dec in zip(ra_list, dec_list):
                x, y = c.RADecToXY_astrometric(ra, dec)
                ra_check, dec_check, ra_str, dec_str = c.XYToRADec(x, y)
                assert np.isclose(ra, ra_check, rtol=0, atol=8e-6)
                assert np.isclose(dec, dec_check, rtol=0, atol=8e-6)
                xvals_from_siaf.append(x)
                yvals_from_siaf.append(y)

            # Set the tolerance to be pretty high, so that filter-specific
            # differences hopefully won't be caught
            assert np.allclose(xvals_from_siaf, reference_file_values['{}_{}'.format(instrument, aperture)][0], atol=0.5)
            assert np.allclose(yvals_from_siaf, reference_file_values['{}_{}'.format(instrument, aperture)][1], atol=0.5)
