# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pydna']

package_data = \
{'': ['*']}

install_requires = \
['appdirs>=1.4.4',
 'biopython>=1.80',
 'networkx>=2.8.8',
 'prettytable>=3.5.0',
 'pyfiglet>=0.8.post1',
 'pyperclip>=1.8.2']

extras_require = \
{'download': ['pyparsing>=2.4.7', 'requests>=2.26.0'],
 'gel': ['scipy>=1.8.0', 'matplotlib>=3.4.3', 'pillow>=8.4.0']}

setup_kwargs = {
    'name': 'pydna',
    'version': '5.2.0a14',
    'description': 'Representing double stranded DNA and functions for simulating cloning and homologous recombination between DNA molecules.',
    'long_description': '# ![icon](https://raw.githubusercontent.com/bjornFJohansson/pydna/master/docs/pics/pydna.resized.png) pydna\n\n| [![Tests & Coverage](https://github.com/BjornFJohansson/pydna/workflows/Tests%20&%20Coverage/badge.svg)](https://github.com/BjornFJohansson/pydna/actions?query=workflow%3A%22Tests+%26+Coverage%22)                |[![codecov](https://codecov.io/gh/BjornFJohansson/pydna/branch/master/graph/badge.svg)](https://codecov.io/gh/BjornFJohansson/pydna/branch/master)    | [![PyPI version](https://badge.fury.io/py/pydna.svg)](https://badge.fury.io/py/pydna)                                                   |[![Anaconda-Server Badge](https://anaconda.org/bjornfjohansson/pydna/badges/version.svg)](https://anaconda.org/bjornfjohansson/pydna)   | [![Google group : pydna](https://img.shields.io/badge/Google%20Group-pydna-blue.svg)](https://groups.google.com/g/pydna)        |\n|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------|\n| [![Documentation Status](https://readthedocs.org/projects/pydna/badge/?version=latest)](http://pydna.readthedocs.io/?badge=latest)                                                                                  |[![GitHub issues](https://img.shields.io/github/issues/BjornFJohansson/pydna.svg)](https://github.com/BjornFJohansson/pydna/issues)                   | [![Anaconda-Server Badge2](https://anaconda.org/bjornfjohansson/pydna/badges/license.svg)](https://anaconda.org/bjornfjohansson/pydna)  |[![GitHub stars](https://img.shields.io/github/stars/BjornFJohansson/pydna.svg)](https://github.com/BjornFJohansson/pydna/stargazers)   |                                                                                                                                 |\n\n\nPlanning genetic constructs with many parts and assembly steps, such as recombinant\nmetabolic pathways :petri_dish:, are often difficult to **properly** document as is evident from the\nstate of such documentation in the scientific literature :radioactive:.\n\n\nThe pydna python package provide a human-readable formal descriptions of :dna: cloning and genetic assembly\nstrategies in Python :snake: which allow for simulation and verification.\n\n\nPydna can perhaps be thought of as [executable documentation](https://en.wikipedia.org/wiki/Literate_programming) for cloning.\n\n\nA cloning strategy expressed in pydna is **complete**, **unambiguous** and **stable**.\n\n\nPydna provides simulation of:\n\n- Restriction digestion\n- Ligation\n- PCR\n- Primer design\n- Gibson assembly\n- Golden gate assembly\n- Homologous recombination\n- Gel electrophoresis of DNA with generation of gel images\n\nVirtually any sub-cloning experiment can be described in pydna, and its execution yield\nthe sequences of intermediate and final DNA molecules.\n\nPydna has been designed to be understandable for biologists with only some basic understanding of Python.\n\nPydna can formalize planning and sharing of cloning strategies and is especially useful for complex or combinatorial\nDNA molecule constructions.\n\n\nTo get started, we have compiled some [simple examples](https://github.com/MetabolicEngineeringGroupCBMA/pydna-examples#pydna-examples).\nFor more elaborate use, look at some assembly strategies of D-xylose metabolic pathways [MetabolicEngineeringGroupCBMA/ypk-xylose-pathways](https://github.com/MetabolicEngineeringGroupCBMA/ypk-xylose-pathways#pereira-et-al-2016).\n\n\nThere is an open access paper in BMC Bioinformatics describing pydna:\n\n[![abstr](https://raw.githubusercontent.com/bjornFJohansson/pydna/master/docs/pics/BMC_resized.png)](http://www.biomedcentral.com/1471-2105/16/142/abstract)\n\nPlease reference the above paper:\n\n\nPereira, F., Azevedo, F., Carvalho, Â., Ribeiro, G. F., Budde, M. W., & Johansson, B. (2015). Pydna: a simulation and documentation tool for DNA assembly strategies using python. BMC Bioinformatics, 16(142), 142.\n\n\nif using pydna in a scientific publication.\n\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n\n\n## Usage\n\nMost pydna functionality is implemented as methods for the double stranded DNA sequence record\nclasses Dseq and Dseqrecord, which are subclasses of the [Biopython](http://biopython.org/wiki/Main_Page) [Seq](http://biopython.org/wiki/Seq) and [SeqRecord](http://biopython.org/wiki/SeqRecord) classes.\n\nThese classes make cut and paste cloning and PCR very simple:\n\n\n    >>> from pydna.dseq import Dseq\n    >>> seq = Dseq("GGATCCAAA","TTTGGATCC",ovhg=0)\n    >>> seq\n    Dseq(-9)\n    GGATCCAAA\n    CCTAGGTTT\n    >>> from Bio.Restriction import BamHI\n    >>> a,b = seq.cut(BamHI)\n    >>> a\n    Dseq(-5)\n    G\n    CCTAG\n    >>> b\n    Dseq(-8)\n    GATCCAAA\n        GTTT\n    >>> a+b\n    Dseq(-9)\n    GGATCCAAA\n    CCTAGGTTT\n    >>> b+a\n    Dseq(-13)\n    GATCCAAAG\n        GTTTCCTAG\n    >>> b+a+b\n    Dseq(-17)\n    GATCCAAAGGATCCAAA\n        GTTTCCTAGGTTT\n    >>> b+a+a\n    Traceback (most recent call last):\n      File "<stdin>", line 1, in <module>\n      File "/usr/local/lib/python2.7/dist-packages/pydna/dsdna.py", line 217, in __add__\n        raise TypeError("sticky ends not compatible!")\n    TypeError: sticky ends not compatible!\n    >>>\n\nAs the example above shows, pydna keeps track of sticky ends.\n\nNotably, homologous recombination and Gibson assembly between linear DNA fragments\ncan be easily simulated without any additional information besides the primary sequence of the fragments.\n\nGel electrophoresis of DNA fragments can be simulated using the included gel module\n\n\n    Jupyter QtConsole 4.7.7\n    Python 3.8.5 | packaged by conda-forge | (default, Aug 29 2020, 01:22:49)\n    Type \'copyright\', \'credits\' or \'license\' for more information\n    IPython 7.18.1 -- An enhanced Interactive Python. Type \'?\' for help.\n\n    In [1]: from pydna.gel import gel\n\n    In [2]: from pydna.ladders import PennStateLadder\n\n    In [3]: from pydna.dseqrecord import Dseqrecord\n\n    In [4]: gel([PennStateLadder,[Dseqrecord("A"*2000)]])\n    Out[4]:\n\n\n\n![](https://raw.githubusercontent.com/BjornFJohansson/pydna/master/scripts/pydna_gel.png)\n\n\nPydna can be very compact. The eleven lines of Python below simulates the construction of a recombinant plasmid.\nDNA sequences are downloaded from Genbank by accession numbers that are guaranteed to be stable over time.\n\n    from pydna.genbank import Genbank\n    gb = Genbank("myself@email.com") # Tell Genbank who you are!\n    gene = gb.nucleotide("X06997") # Kluyveromyces lactis LAC12 gene for lactose permease.\n    from pydna.parsers import parse_primers\n    primer_f,primer_r = parse_primers(\'\'\' >760_KlLAC12_rv (20-mer)\n                                          ttaaacagattctgcctctg\n\n                                          >759_KlLAC12_fw (19-mer)\n                                          aaatggcagatcattcgag \'\'\')\n    from pydna.amplify import pcr\n    pcr_prod = pcr(primer_f,primer_r, gene)\n    vector = gb.nucleotide("AJ001614") # pCAPs cloning vector\n    from Bio.Restriction import EcoRV\n    lin_vector = vector.linearize(EcoRV)\n    rec_vec =  ( lin_vector + pcr_prod ).looped()\n\nPydna can automate the simulation of [sub cloning](http://en.wikipedia.org/wiki/Subcloning) experiments using\npython. This is helpful to generate examples for teaching purposes.\n\nRead the documentation (below) or the [cookbook](https://github.com/BjornFJohansson/pydna/blob/master/docs/cookbook/cookbook.ipynb) with example files\nfor further information.\n\nPlease post a message in the [google group](https://groups.google.com/d/forum/pydna)\nfor pydna if you need help or have problems, questions or comments :sos:.\n\nFeedback & suggestions are very welcome!\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n\n## Who is using pydna?\n\nTaylor, L. J., & Strebel, K. (2017).\nPyviko: an automated Python tool to design gene knockouts in complex viruses with overlapping genes.\nBMC Microbiology, 17(1), 12.\n[PubMed](https://www.ncbi.nlm.nih.gov/pubmed/28061810)\n\n\nWang, Y., Xue, H., Pourcel, C., Du, Y., & Gautheret, D. (2021).\n2-kupl: mapping-free variant detection from DNA-seq data of matched samples.\nIn Cold Spring Harbor Laboratory (p. 2021.01.17.427048). [DOI](https://doi.org/10.1101/2021.01.17.427048)\n[PubMed](https://www.ncbi.nlm.nih.gov/pmc/articles/PMC8180056)\n\n\n[An Automated Protein Synthesis Pipeline with Transcriptic and Snakemake](http://blog.booleanbiotech.com/transcriptic_protein_synthesis_pipeline.html)\n\n\nand other projects on [github](https://github.com/BjornFJohansson/pydna/network/dependents?package_id=UGFja2FnZS01MjQ2MjYzNQ%3D%3D)\n\n\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Documentation\n\nDocumentation is built using [Sphinx](http://www.sphinx-doc.org/) from [docstrings](https://www.python.org/dev/peps/pep-0257/)\nin the code and displayed at readthedocs [![Documentation Status](https://readthedocs.org/projects/pydna/badge/?version=latest)](http://pydna.readthedocs.io/?badge=latest)\n\nThe [numpy](www.numpy.org) [docstring format](https://github.com/numpy/numpy/blob/release/doc/HOWTO_DOCUMENT.rst.txt) is used.\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Installation using pip\n\nPip is included in recent Python versions and is the\nofficially [recommended](http://python-packaging-user-guide.readthedocs.org/en/latest) tool.\n\nPip installs the minimal installation requirements automatically, but not the optional requirements (see below).\n\n    pip install pydna\n\nor use the --pre switch to get the latest version of pydna.\n\n    pip install pydna --pre\n\n### Windows:\n\nYou should be able to pip install pydna from the Windows terminal as biopython now can be installed with pip as well.\n\n    C:\\> pip install pydna\n\nBy default python and pip are not on the PATH. You can re-install Python and select this option during installation,\nor give the full path for pip. Try something like this, depending on where your copy of Python is installed:\n\n    C:\\Python37\\Scripts\\pip install pydna\n\n### Installing requirements\n\nIf you want to install requirements before installing pydna, you can do:\n\n\tpip install -r requirements.txt\n\nAnd for the optional requirements:\n\n\tpip install -r requirements_optional.txt\n\nFor testing:\n\n\tpip install -r requirements_test.txt\n\nor\n\n\tconda install --file requirements.txt\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Source Code\n\nPydna is developed on [Github](https://github.com/BjornFJohansson/pydna) :octocat:.\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Minimal installation dependencies\n\nPydna versions before 1.0.0 were compatible with python 2.7 only.\nThe list below is the minimal requirements for installing pydna.\nBiopython has c-extensions, but the other modules are pure python.\n\n- [Python 3.8, 3.9, 3.10 or 3.11](http://www.python.org)\n- [appdirs >= 1.3.0](https://pypi.python.org/pypi/appdirs)\n- [biopython >= 1.80](http://pypi.python.org/pypi/biopython)\n- [networkx >= 1.8.1](http://pypi.python.org/pypi/networkx)\n- [prettytable >= 0.7.2](https://pypi.python.org/pypi/PrettyTable)\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Optional dependencies\n\nIf the modules listed below in the first column are installed, they will provide the functionality\nlisted in the second column.\n\n| Dependency                                          | Function in pydna                                      |\n|-----------------------------------------------------|--------------------------------------------------------|\n| [pyparsing](https://pypi.python.org/pypi/pyparsing) | fix corrupt Genbank files with pydna.genbankfixer      |\n| [requests](https://pypi.org/project/requests)       | download sequences with pydna.download                 |\n| [CAI](https://pypi.org/project/CAI)                 | codon adaptation index calculations in several modules |\n| [numpy](http://www.numpy.org)                       | gel simulation with pydna.gel                          |\n| [scipy](https://www.scipy.org)                      | “                                                      |\n| [matplotlib](http://matplotlib.org)                 | “                                                      |\n| [pillow](https://github.com/python-pillow/Pillow)   | “                                                      |\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Requirements for running tests and analyzing code coverage\n\n- [pytest](https://pypi.org/project/pytest)\n- [pytest-cov](https://pypi.org/project/pytest-cov)\n- [pytest-mock](https://pypi.org/project/pytest-mock)\n- [pytest-doctestplus](https://pypi.org/project/pytest-doctestplus)\n- [coverage](https://pypi.org/project/coverage)\n- [nbval](https://pypi.org/project/nbval)\n- [requests-mock](https://pypi.org/project/requests-mock)\n\n\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Changelog\n\nSee the [change log](docs/CHANGELOG.md) for recent changes.\n\n![-----------------------------------------------------](https://raw.githubusercontent.com/andreasbm/readme/master/assets/lines/colored.png)\n\n## Automatic testing & Release process\n\nThere are three github actions associated with this package:\n\n- `pydna_test_and_coverage_workflow.yml`\n\nThe `pydna_test_and_coverage_workflow.yml` is triggered on all pushed non-tagged commits.\nThis workflow run tests, doctests and a series of Jupyter notebooks using pytest.\n\nThe two other workflows build a setuptools wheel and packages for different Python versions\non Linux, Windows and macOS.\n\nThese are triggered by publishing a github release manually from the github interface.\n\n\n## Building a PyPI package\n\n\n\tpoetry build # run this command in the root directory where the pyproject.toml file is located\n\n\nPydna uses the poetry plugin []()\n\n\n\n\n\n\n### History\n\nPydna was made public in 2012 on [Google code](https://code.google.com/archive/p/pydna).\n\n\n:microbe:\n\n\n:portugal:\n',
    'author': 'BjornFJohansson',
    'author_email': 'bjornjobb@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'None',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8',
}


setup(**setup_kwargs)
