import argparse
import logging

import numpy as np
import pandas as pd

import mip.utils as utils
from mip.ome import generate_ome_from_tifs, generate_ome_from_qptiff
from mip.spatial_features import save_spatial_features
from mip.region_analysis import generate_region_metrics

logging.basicConfig(format='%(asctime)s - %(message)s', level=logging.INFO)

parser = argparse.ArgumentParser()

parser.add_argument('mode', type=str,
    choices=['make-ome', 'generate-spatial-features', 'generate-region-features', 'show-channels', 'crop-ome'],
    help='Which task mip is to execute.')


###################
## show-channels ##
###################
## --ome-tiff

parser.add_argument('--sep', type=str, default='\n',
    help='Seperator between channel names. Defaults to newline character (i.e. channels are displayed on seperate lines)')

##############
## make-ome ##
##############
parser.add_argument('--input-tif', type=str,
    help='Used in make-ome mode. Either directory of stitched tif files that will be combined into a single ome.tiff file (for codex platform), or filepath of a .qptiff (phenocycler platform).')

parser.add_argument('--output-filepath', type=str,
    help='Location to write ome.tiff file')

parser.add_argument('--platform', type=str,
    choices=['codex', 'phenocycler', 'raw'], default='phenocycler',
    help='Which platform produced the input images. Raw will stitch a directory of .tifs together without and channel name parsing.')

parser.add_argument('--bbox', type=str,
    help='If desired, bbox in to crop image with. Must be the following format: "top,bottom,left,right"')

###############################
## generate-spatial-features ##
###############################
parser.add_argument('--label-image', type=str,
    help='Filepath of labeled cell segmentation tif.')

parser.add_argument('--ome-tiff', type=str,
    help='Filepath of ome.tiff')

parser.add_argument('--output-prefix', type=str, default='output',
    help='Output prefix for files generated by generate-spatial-features.')

###############################
## generate-region-features ##
###############################
parser.add_argument('--spatial-features', type=str,
    help='Filepath of a tab-seperated .txt file with columns specifying coordinates cell annotations in slide. First column is cell ID, second and third columns are treated as "x" and "y" coordinates respectively. All following columns are treated as cell metadata features and seperate fractions/metrics will be generated for each feature.')

parser.add_argument('--regions-mask', type=str,
    help='Filepath of region mask that will be used when calculating metrics. Mask should be a .tif file and the same height and width as --ome-tiff.')

parser.add_argument('--channel-thresholds-grid', type=str,
    help='Filepath of tab-seperated .txt file where the first column is a channel name in the --ome-tiff and the second column is threshold values to use when determining positive grid polygons.')

parser.add_argument('--channel-thresholds-pixel', type=str,
    help='Filepath of tab-seperated .txt file where the first column is a channel name in the --ome-tiff and the second column is threshold values to use when determining positive pixels when generating region metrics.')

parser.add_argument('--output-dir', type=str, default='output',
    help='Location to write generate-region-features output files.')

parser.add_argument('--boundary-dist', type=int, default=150,
    help='Distance (in pixels) to draw boundary around each region.')

parser.add_argument('--perp-steps', type=int, default=10,
    help='Number of arcs to generate for region grid when drawing grid polygons.')

parser.add_argument('--expansion', type=int, default=40,
    help='Distance (in pixels) of innermost arc to outermost arc.')

parser.add_argument('--parallel-step', type=int, default=50,
    help='Step size (in pixels) of steps along arcs to use when drawing grid polygons.')

parser.add_argument('--breakage-dist', type=int, default=10,
    help='Distance (in pixels) along innermost arc to use when drawing breakage lines.')

parser.add_argument('--area_thresh', type=int, default=2000,
    help='Filter out grid polygons with area greater than area-thresh.')

parser.add_argument('--breakage-line-thresh', type=int, default=100,
    help='Filter out grid polygons with area greater than area-thresh.')

parser.add_argument('--min-region-size', type=int,
    help='Skip regions below --min-region-size when calculating metrics.')

parser.add_argument('--max-region-size', type=int,
    help='Skip regions over --max-region-size when calculating metrics. Helps speed up runs for debugging purposes since there are non-linear increases in runtime with region size.')

parser.add_argument('--skip-grid-metrics', action='store_true',
    help='If --slip-grid-metrics, then do not calculate polygon-mesh related metrics and only calculate basic region metrics.')


args = parser.parse_args()



def run_show_channels(ome_tiff_fp, sep):
    channels = utils.get_ome_tiff_channels(ome_tiff_fp)
    print(sep.join(channels))
    

def run_make_ome(input_tif, output_fp, platform='phenocycler', bbox=None):
    if bbox is not None and isinstance(bbox, str):
        bbox = [int(x) for x in bbox.split(',')]
    if platform in ['codex', 'raw']:
        fps = sorted(utils.listfiles(input_tif, regex=r'.tif[f]*$'))
        generate_ome_from_tifs(fps, output_fp, platform=platform, bbox=bbox)
    elif platform == 'phenocycler':
        generate_ome_from_qptiff(input_tif, output_fp, bbox=bbox)
    logging.info(f'ome.tiff written to {output_fp}')


def run_generate_spatial_features(label_fp, ome_fp, output_prefix='output'):
    save_spatial_features(label_fp, ome_fp, output_prefix=output_prefix)
    logging.info(f'spatial features written to {output_prefix}')


def run_generate_region_features():
    df = pd.read_csv(args.spatial_features, sep='\t', index_col=0)
    metadata_cols = list(df.columns[2:])
    cols = ['x', 'y']
    cols += metadata_cols
    df.columns = cols

    channel_df = pd.read_csv(args.channel_thresholds_grid, sep='\t')
    channel_to_thresh_grid = {c:t for c, t in zip(channel_df.iloc[:, 0], channel_df.iloc[:, 1])}

    channel_df = pd.read_csv(args.channel_thresholds_pixel, sep='\t')
    channel_to_thresh_pixel = {c:t for c, t in zip(channel_df.iloc[:, 0], channel_df.iloc[:, 1])}
 
    generate_region_metrics(
        df, args.ome_tiff, args.regions_mask, args.output_dir,
        y_col='y', x_col='x', cell_metadata_cols=metadata_cols,
        boundary_dist=args.boundary_dist,
        parallel_step=args.parallel_step, perp_steps=args.perp_steps,
        expansion=args.expansion, grouping_dist=args.breakage_dist,
        area_thresh=args.area_thresh, group_line_thresh=args.breakage_line_thresh,
        channel_to_thresh_grid=channel_to_thresh_grid, channel_to_thresh_pixel=channel_to_thresh_pixel,
        min_region_size=args.min_region_size,
        max_region_size=args.max_region_size, calculate_grid_metrics=not args.skip_grid_metrics
    )

def main():
    if args.mode == 'make-ome':
        run_make_ome(args.input_tif, args.output_filepath, platform=args.platform, bbox=args.bbox)
    elif args.mode == 'generate-spatial-features':
        run_generate_spatial_features(
            args.label_image, args.ome_tiff, output_prefix=args.output_prefix)
    elif args.mode == 'generate-region-features':
        run_generate_region_features()
    elif args.mode == 'show-channels':
        run_show_channels(args.ome_tiff, args.sep)
    else:
        raise RuntimeError(f'{args.mode} is not a valid mode.')
    


if __name__ == '__main__':
    main()
