"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
            if (typeof config.blockPublicAccess !== 'undefined') {
                bucketConfig.blockPublicAccess = config.blockPublicAccess;
            }
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: config.cfBehaviors ? config.cfBehaviors : [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.sslMethod !== 'undefined') {
            cfConfig.aliasConfiguration.sslMethod = config.sslMethod;
        }
        if (typeof config.securityPolicy !== 'undefined') {
            cfConfig.aliasConfiguration.securityPolicy = config.securityPolicy;
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website.
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution It will also setup error forwarding and unauth forwarding back to indexDoc.
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.distributionDomainName,
        });
        return { websiteBucket, distribution };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto configured by using the details in the hosted zone provided.
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: [`www.${config.zoneName}`],
            targetDomain: config.zoneName,
        });
        return { websiteBucket, distribution };
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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