# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['tfimm',
 'tfimm.architectures',
 'tfimm.architectures.timm',
 'tfimm.layers',
 'tfimm.models',
 'tfimm.train',
 'tfimm.train.datasets',
 'tfimm.train.examples',
 'tfimm.train.optimizers',
 'tfimm.train.problems',
 'tfimm.utils']

package_data = \
{'': ['*']}

install_requires = \
['markdown>=3.3.6,<4.0.0', 'numpy', 'tensorflow>=2.5,<3.0']

extras_require = \
{'docs': ['sphinx', 'sphinx-rtd-theme'],
 'timm': ['pyyaml', 'timm'],
 'train': ['pandas', 'pyyaml', 'tensorflow-datasets', 'tqdm', 'wandb']}

setup_kwargs = {
    'name': 'tfimm',
    'version': '0.2.5',
    'description': 'TensorFlow port of PyTorch Image Models (timm) - image models with pretrained weights',
    'long_description': '# TensorFlow Image Models\n\n![Test Status](https://github.com/martinsbruveris/tensorflow-image-models/actions/workflows/tests.yml/badge.svg)\n[![Documentation Status](https://readthedocs.org/projects/tfimm/badge/?version=latest)](https://tfimm.readthedocs.io/en/latest/?badge=latest)\n[![License](https://img.shields.io/badge/License-Apache_2.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Slack](https://img.shields.io/badge/Slack-4A154B?style=for-the-badge&logo=slack&logoColor=white)](https://join.slack.com/t/tfimm/shared_invite/zt-13dnaf3qo-5JJaCBFIQhugeBXBT3NK8A)\n\n- [Introduction](#introduction)\n- [Usage](#usage)\n- [Models](#models)\n- [Profiling](#profiling)\n- [License](#license)\n- [Contact](#contact)\n\n## Introduction\n\nTensorFlow Image Models (`tfimm`) is a collection of image models with pretrained\nweights, obtained by porting architectures from \n[timm](https://github.com/rwightman/pytorch-image-models) to TensorFlow. The hope is\nthat the number of available architectures will grow over time. For now, it contains\nvision transformers (ViT, DeiT, CaiT, PVT and Swin Transformers), MLP-Mixer models \n(MLP-Mixer, ResMLP, gMLP, PoolFormer and ConvMixer) and various ResNet flavours (ResNet,\nResNeXt, ECA-ResNet, SE-ResNet) as well as the recent ConvNeXt.\n\nThis work would not have been possible wihout Ross Wightman\'s `timm` library and the\nwork on PyTorch/TensorFlow interoperability in HuggingFace\'s `transformer` repository.\nI tried to make sure all source material is acknowledged. Please let me know if I have\nmissed something.\n\n## Usage\n\n### Installation \n\nThe package can be installed via `pip`,\n\n```shell\npip install tfimm\n```\n\nTo load pretrained weights, `timm` needs to be installed separately.\n\n### Creating models\n\nTo load pretrained models use\n\n```python\nimport tfimm\n\nmodel = tfimm.create_model("vit_tiny_patch16_224", pretrained="timm")\n```\n\nWe can list available models with pretrained weights via\n\n```python\nimport tfimm\n\nprint(tfimm.list_models(pretrained="timm"))\n```\n\nMost models are pretrained on ImageNet or ImageNet-21k. If we want to use them for other\ntasks we need to change the number of classes in the classifier or remove the \nclassifier altogether. We can do this by setting the `nb_classes` parameter in \n`create_model`. If `nb_classes=0`, the model will have no classification layer. If\n`nb_classes` is set to a value different from the default model config, the \nclassification layer will be randomly initialized, while all other weights will be\ncopied from the pretrained model.\n\nThe preprocessing function for each model can be created via\n```python\nimport tensorflow as tf\nimport tfimm\n\npreprocess = tfimm.create_preprocessing("vit_tiny_patch16_224", dtype="float32")\nimg = tf.ones((1, 224, 224, 3), dtype="uint8")\nimg_preprocessed = preprocess(img)\n```\n\n### Saving and loading models\n\nAll models are subclassed from `tf.keras.Model` (they are _not_ functional models).\nThey can still be saved and loaded using the `SavedModel` format.\n\n```\n>>> import tesnorflow as tf\n>>> import tfimm\n>>> model = tfimm.create_model("vit_tiny_patch16_224")\n>>> type(model)\n<class \'tfimm.architectures.vit.ViT\'>\n>>> model.save("/tmp/my_model")\n>>> loaded_model = tf.keras.models.load_model("/tmp/my_model")\n>>> type(loaded_model)\n<class \'tfimm.architectures.vit.ViT\'>\n```\n\nFor this to work, the `tfimm` library needs to be imported before the model is loaded,\nsince during the import process, `tfimm` is registering custom models with Keras.\nOtherwise, we obtain the following output\n\n```\n>>> import tensorflow as tf\n>>> loaded_model = tf.keras.models.load_model("/tmp/my_model")\n>>> type(loaded_model)\n<class \'keras.saving.saved_model.load.Custom>ViT\'>\n```\n\n## Models\n\nThe following architectures are currently available:\n\n- CaiT (vision transformer) \n  [\\[github\\]](https://github.com/facebookresearch/deit/blob/main/README_cait.md)\n  - Going deeper with Image Transformers \n    [\\[arXiv:2103.17239\\]](https://arxiv.org/abs/2103.17239)\n- DeiT (vision transformer) \n  [\\[github\\]](https://github.com/facebookresearch/deit)\n  - Training data-efficient image transformers & distillation through attention. \n    [\\[arXiv:2012.12877\\]](https://arxiv.org/abs/2012.12877) \n- ViT (vision transformer) \n  [\\[github\\]](https://github.com/google-research/vision_transformer)\n  - An Image is Worth 16x16 Words: Transformers for Image Recognition at Scale.\n    [\\[arXiv:2010.11929\\]](https://arxiv.org/abs/2010.11929)\n  - How to train your ViT? Data, Augmentation, and Regularization in Vision \n    Transformers. [\\[arXiv:2106.10270\\]](https://arxiv.org/abs/2106.10270)\n  - Includes models trained with the SAM optimizer: Sharpness-Aware Minimization for \n    Efficiently Improving Generalization. \n    [\\[arXiv:2010.01412\\]](https://arxiv.org/abs/2010.01412)\n  - Includes models from: ImageNet-21K Pretraining for the Masses\n    [\\[arXiv:2104.10972\\]](https://arxiv.org/abs/2104.10972) \n    [\\[github\\]](https://github.com/Alibaba-MIIL/ImageNet21K)\n- Swin Transformer \n  [\\[github\\]](https://github.com/microsoft/Swin-Transformer)\n  - Swin Transformer: Hierarchical Vision Transformer using Shifted Windows. \n    [\\[arXiv:2103.14030\\]](https://arxiv.org/abs/2103.14030)\n  - Tensorflow code adapted from \n    [Swin-Transformer-TF](https://github.com/rishigami/Swin-Transformer-TF)\n- MLP-Mixer and friends\n  - MLP-Mixer: An all-MLP Architecture for Vision \n    [\\[arXiv:2105.01601\\]](https://arxiv.org/abs/2105.01601)\n  - ResMLP: Feedforward networks for image classification... \n    [\\[arXiv:2105.03404\\]](https://arxiv.org/abs/2105.03404)\n  - Pay Attention to MLPs (gMLP)\n    [\\[arXiv:2105.08050\\]](https://arxiv.org/abs/2105.08050)\n- ConvMixer \n  [\\[github\\]](https://github.com/tmp-iclr/convmixer)\n  - Patches Are All You Need? \n    [\\[ICLR 2022 submission\\]](https://openreview.net/forum?id=TVHS5Y4dNvM)\n- Pyramid Vision Transformer \n  [\\[github\\]](https://github.com/whai362/PVT)\n  - Pyramid Vision Transformer: A Versatile Backbone for Dense Prediction without\n    Convolutions. [\\[arXiv:2102.12122\\]](https://arxiv.org/abs/2102.12122)\n  - PVTv2: Improved Baselines with Pyramid Vision Transformer \n    [\\[arXiv:2106.13797\\]](https://arxiv.org/abs/2106.13797)\n- ConvNeXt\n  [\\[github\\]](https://github.com/facebookresearch/ConvNeXt)\n  - A ConvNet for the 2020s. [\\[arXiv:2201.03545\\]](https://arxiv.org/abs/2201.03545)\n- PoolFormer\n  [\\[github\\]](https://github.com/sail-sg/poolformer)\n  - PoolFormer: MetaFormer is Actually What You Need for Vision.\n    [\\[arXiv:2111.11418\\]](https://arxiv.org/abs/2111.11418)\n- Pooling-based Vision Transformers (PiT)\n  - Rethinking Spatial Dimensions of Vision Transformers.\n    [\\[arXiv:2103.16302\\]](https://arxiv.org/abs/2103.16302)\n- ResNet, ResNeXt, ECA-ResNet, SE-ResNet and friends\n  - Deep Residual Learning for Image Recognition. \n    [\\[arXiv:1512.03385\\]](https://arxiv.org/abs/1512.03385)\n  - Exploring the Limits of Weakly Supervised Pretraining. \n    [\\[arXiv:1805.00932\\]](https://arxiv.org/abs/1805.00932)\n  - Billion-scale Semi-Supervised Learning for Image Classification. \n    [\\[arXiv:1905.00546\\]](https://arxiv.org/abs/1905.00546)\n  - ECA-Net: Efficient Channel Attention for Deep Convolutional Neural Networks. \n    [\\[arXiv1910.03151\\]](https://arxiv.org/abs/1910.03151)\n  - Revisiting ResNets. [\\[arXiv:2103.07579\\]](https://arxiv.org/abs/2103.07579)\n  - Making Convolutional Networks Shift-Invariant Again. (anti-aliasing layer)\n    [\\[arXiv:1904.11486\\]](https://arxiv.org/abs/1904.11486)\n  - Squeeze-and-Excitation Networks. \n    [\\[arXiv:1709.01507\\]](https://arxiv.org/abs/1709.01507)\n  - Big Transfer (BiT): General Visual Representation Learning\n    [\\[arXiv:1912.11370\\]](https://arxiv.org/abs/1912.11370)\n  - Knowledge distillation: A good teacher is patient and consistent\n    [\\[arXiv:2106:05237\\]](https://arxiv.org/abs/2106.05237)\n\n## Profiling\n\nTo understand how big each of the models is, I have done some profiling to measure\n- maximum batch size that fits in GPU memory and\n- throughput in images/second\nfor both inference and backpropagation on K80 and V100 GPUs. For V100, measurements \nwere done for both `float32` and mixed precision.\n\nThe results can be found in the `results/profiling_{k80, v100}.csv` files.\n\nFor backpropagation, we use as loss the mean of model outputs\n\n```python\ndef backprop():\n    with tf.GradientTape() as tape:\n        output = model(x, training=True)\n        loss = tf.reduce_mean(output)\n        grads = tape.gradient(loss, model.trainable_variables)\n    optimizer.apply_gradients(zip(grads, model.trainable_variables))\n```\n\n## License\n\nThis repository is released under the Apache 2.0 license as found in the \n[LICENSE](LICENSE) file.\n\n## Contact\n\nAll things related to `tfimm` can be discussed via \n[Slack](https://join.slack.com/t/tfimm/shared_invite/zt-13dnaf3qo-5JJaCBFIQhugeBXBT3NK8A).',
    'author': 'Martins Bruveris',
    'author_email': 'martins.bruveris@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/martinsbruveris/tensorflow-image-models',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
