# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['auth_code_flow']

package_data = \
{'': ['*']}

install_requires = \
['requests>=2.25.1,<3.0.0', 'urllib3>=1.26.4,<2.0.0']

setup_kwargs = {
    'name': 'auth-code-flow',
    'version': '0.2.1',
    'description': 'Auth Code Flow (`auth-code-flow`) is a utility for obtaining access tokens on behalf of resource owners using the OAuth 2.0 authentication code flow',
    'long_description': '# Auth Code Flow\n\nAuth Code Flow (`auth-code-flow`) is a utility for obtaining access tokens on behalf of resource owners using [the OAuth 2.0 authorization code flow](https://tools.ietf.org/html/rfc6749).\n\n\n## Quick Start\n\nThis is a quick-start tutorial for using `auth-code-flow` to obtain an access token from an OAuth service provider on behalf of its user.\n\nFleshed-out tutorials for doing this using `auth-code-flow` in Python web frameworks like Django and FastAPI are in the works.\n\n\n### First Things First\n\nWe\'ll be walking through the process of obtaining an access token from Stack Exchange on behalf of a user of their service. We\'ll be implementing our utility in conformity with [the Stack Exchange authentication documentation](https://api.stackexchange.com/docs/authentication).\n\nFirst make sure you\'ve created a Stack Exchange developer application, as you\'ll need a developer app\'s client id and client secret for this exercise. Please have a look at the answers to [this question on Stack Exchange](https://meta.stackexchange.com/questions/134532/how-do-you-see-what-applications-youve-authorized-on-stack-exchange-with-oauth) if you can\'t immediately figure out how to create one.\n\n\n### Install Auth Code Flow\n\nCreate a virtual environment with any tool of your choice, activate it and install `auth-code-flow` into it.\n\nA Windows user may do theirs this way:\n\n* Create a virtual environment\n  ```\n  python -m venv env\n  ```\n\n* Activate the virtual environment\n  ```\n  .\\env\\Scripts\\activate\n  ```\n\n* Install `auth-code-flow` into the virtual environment\n  ```\n  pip install auth-code-flow\n  ```\n\n\n### Working with the Flow Manager\n\nInstantiate a flow manager for Stack Exchange.\n\n```python\n# file: flow_managers.py\nfrom auth_code_flow import FlowManager\n\n\nse_flow_manager = FlowManager(\n    base_uri="https://stackoverflow.com",\n    client_id="20146",\n    client_secret="your client secret",  # please read this from env vars for security\n    redirect_uri="http://localhost:8000/oauth/callback/stackexchange",\n    scope="no_expiry",\n    access_token_path="/oauth/access_token/json",\n    authorization_path="/oauth",\n)\n```\n\nIn your auth view you\'re going to present the user with a url they can follow in order to start the process of authorizing your app to do things on Stack Exchange on their behalf. You can build this link using the `get_authorization_endpoint()` method on the manager. You will need to supply a unique state for synchronization &mdash; don\'t worry, you\'ll totally understand what this is all about later.\n\nWe\'ll be using FastAPI views for this documentation.\n\n```python\n# file: app.py\nfrom uuid import uuid4\n\nfrom fastapi import FastAPI\nfrom fastapi.responses import HTMLResponse\n\nfrom .flow_managers import se_flow_manager\n\napp = FastAPI()\n\n@app.get(\'/oauth/providers\')\ndef get_providers():\n    state = str(uuid4())\n    # you\'ll probably store the state in the database against the\n    # logged-in user, so that you can always find out who the state\n    # was created for\n    ...\n\n    # get the authorization url\n    se_auth_url = se_flow_manager.get_authorization_endpoint(state)\n    # display it on the web page for the user\n    html_content = f"""\n    <html>\n        <head>\n            <title>Connect your account</title>\n        </head>\n        <body>\n            <p>Please select any of the links below to connect your account to the provider</p>\n            <p><a href="{se_auth_url}" target="_blank">Connect to Stack Exchange</a></p>\n        </body>\n    </html>\n    """\n    return HTMLResponse(html_content)\n```\n\nWhen the user clicks on the link they will be taken to a dedicated page on Stack Exchange where they can either approve or reject your authorization request. In any case Stack Exchange will redirect to your callback uri with an appropriate response for their action.\n\nIn the case of approval, Stack Exchange will tack onto the callback uri a `state` and a `code` query parameter.\n\nIf it was really Stack Exchange that redirected to this callback uri the `state` parameter will be what you had created and embedded into the authorization endpoint uri in the previous view. You can use this to tell which user Stack Exchange is redirecting for, and make sure they are the currently logged-in user.\n\nIn the view powering the particular callback uri, fetch the user\'s Stack Exchange access token using the `fetch_access_token()` method on the manager.\n\n```python\n# file: app.py\n\n...\n\n@app.get(\'/oauth/callback/stackexchange\')\ndef get_response_from_stack_exchange(state: str, code: str):\n    # check that the returned state was created by\n    # you for the logged-in user\n    ...\n\n    # if the state checks out, fetch the SE access token for the user\n    # note that SE requires posting the parameters to the access\n    # token retrieval endpoint as form data -- application/x-www-form-urlencoded\n    resp = se_flow_manager.fetch_access_token(code, state, post_form_data=True)\n    resp_json = resp.json()\n\n    # you now have an access token to SE services for the SE user\n    # you\'ll probably save it to the database against the\n    # logged-in user...\n    # but we\'ll just display it on a HTML page\n    html_content = f"""\n    <html>\n        <head>\n            <title>Connected to StackExchange</title>\n        </head>\n        <body>\n            <h3>Yayyyyyy</h3>\n            <p>We\'ve successfully obtained your StackExchange access token!</p>\n            <p>{resp_json}</p>\n        </body>\n    </html>\n    """\n    return HTMLResponse(html_content)\n```\n\nCongratulations, you\'ve obtained the user\'s Stack Exchange token. You may use it to make requests to [the Stack Exchange API](https://api.stackexchange.com/docs) on behalf of the user. Responsibly, of course.\n',
    'author': 'Mfon Eti-mfon',
    'author_email': 'mfonetimfon@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/a-thousand-juniors/auth-code-flow',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
