"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const s3_assets = require("aws-cdk-lib/aws-s3-assets");
const runtime_1 = require("./runtime");
/**
 * The code the canary should execute
 */
class Code {
    /**
     * Specify code inline.
     *
     * @param code The actual handler code (limited to 5MB)
     *
     * @returns `InlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * Specify code from a local path. Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param assetPath Either a directory or a .zip file
     *
     * @returns `AssetCode` associated with the specified path.
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * Specify code from an s3 bucket. The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     *
     * @returns `S3Code` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Code", version: "2.39.0-alpha.0" };
/**
 * Canary code from an Asset
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    bind(scope, handler, family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        this.validateCanaryAsset(handler, family);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler, family) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = handler.split('.')[0];
            const nodeFilename = `${filename}.js`;
            const pythonFilename = `${filename}.py`;
            if (family === runtime_1.RuntimeFamily.NODEJS && !fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', nodeFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${nodeFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Nodejs.html)`);
            }
            if (family === runtime_1.RuntimeFamily.PYTHON && !fs.existsSync(path.join(this.assetPath, 'python', pythonFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "python/${pythonFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Python.html)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
_b = JSII_RTTI_SYMBOL_1;
AssetCode[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.AssetCode", version: "2.39.0-alpha.0" };
/**
 * Canary code from an inline string.
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    bind(_scope, handler, _family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-synthetics-alpha.InlineCode", version: "2.39.0-alpha.0" };
/**
 * S3 bucket path to the code zip file
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    bind(_scope, _handler, _family) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.bind);
            }
            throw error;
        }
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_d = JSII_RTTI_SYMBOL_1;
S3Code[_d] = { fqn: "@aws-cdk/aws-synthetics-alpha.S3Code", version: "2.39.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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