# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['c27cache']

package_data = \
{'': ['*']}

install_requires = \
['pytest-asyncio==0.15.1',
 'python-dateutil>=2.8.2,<3.0.0',
 'pytz>=2021.3,<2022.0',
 'redis>=4.1.0,<5.0.0']

setup_kwargs = {
    'name': 'c27cache',
    'version': '0.1.1',
    'description': 'A simple caching library.',
    'long_description': '# C27Cache\n\nC27Cache is a simple HTTP caching library designed to work with [FastAPI](https://fastapi.tiangolo.com/)\n\n## Installation\n\nWhile C27Cache is still early in it\'s development it\'s been used in production on a couple of service. \n\n\n### With pip\n\n```shell\npip install c27cache\n```\n\n### With Poetry\n\n```shell\npoetry add c27cache\n```\n\n## Usage and Examples\n\n### Basic Usage\n\n1. #### Initialize C27Cache\n\n```python\nfrom c27cache.config import C27Cache\nC27Cache.init(redis_url=REDIS_URL, namespace=\'test_namespace\')\n```\n\n2. #### Define your controllers\n\nThe `ttl_in_seconds` expires the cache in 180 seconds. There are other approaches to take with helpers like `expire_end_of_day` and `expires_end_of_week`\n\n```python\n\nimport redis\nfrom datetime import datetime\nfrom fastapi import FastAPI, Request, Response\nfrom fastapi.responses import JSONResponse\nfrom c27cache.config import C27Cache\nfrom c27cache.service import cache\n\n@app.get("/b/home")\n@cache(key="b.home", ttl_in_seconds=180)\nasync def home(request: Request, response: Response):\n    return JSONResponse({"page": "home", "datetime": str(datetime.utcnow())})\n\n@app.get("/b/welcome")\n@cache(key="b.home", end_of_week=True)\nasync def home(request: Request, response: Response):\n    return JSONResponse({"page": "welcome", "datetime": str(datetime.utcnow())})    \n```\n\n### Building keys from parameter objects.\n\nWhile it\'s always possible to explicitly pass keys onto the `key` attribute, there are scenarios where the keys need to be build based on the parameters received by the controller method. For instance, in an authenticated API where the `user_id` is fetched as a controlled `Depends` argument.\n\n```python\n@app.get("/b/logged-in")\n@cache(key="b.logged_in.{}", obj="user", obj_attr="id")\nasync def logged_in(request: Request, response: Response, user=user):\n    return JSONResponse(\n        {"page": "home", "user": user.id, "datetime": str(datetime.utcnow())}\n    )\n```\n\nIn the example above, the key allows room for a dynamic attribute fetched from the object `user`. The key eventually becomes `b.logged_in.112358` if the `user.id` returns `112358` \n\n\n### Explicitly invalidating the cache\n\nThe cache invalidation can be managed using the `@invalidate_cache` decorator. \n\n```python\n@app.post("/b/logged-in")\n@invalidate_cache(\n    key="b.logged_in.{}", obj="user", obj_attr="id", namespace="test_namespace"\n)\nasync def post_logged_in(request: Request, response: Response, user=user):\n    return JSONResponse(\n        {"page": "home", "user": user.id, "datetime": str(datetime.utcnow())}\n    )\n```\n\n## Full Example\n\n```python\nimport os\nimport redis\nfrom datetime import datetime\nfrom fastapi import FastAPI, Request, Response\nfrom fastapi.responses import JSONResponse\nfrom c27cache.config import C27Cache\nfrom c27cache.service import cache, invalidate_cache\n\nREDIS_URL = os.environ.get("REDIS_URL", "redis://localhost:6379/3")\nredis_client = redis.Redis.from_url(REDIS_URL)\n\nclass User:\n    id: str = "112358"\n\nuser = User()\napp = FastAPI()\n\nC27Cache.init(redis_url=REDIS_URL, namespace=\'test_namespace\')\n\n@app.get("/b/home")\n@cache(key="b.home", ttl_in_seconds=180)\nasync def home(request: Request, response: Response):\n    return JSONResponse({"page": "home", "datetime": str(datetime.utcnow())})\n\n\n@app.get("/b/logged-in")\n@cache(key="b.logged_in.{}", obj="user", obj_attr="id")\nasync def logged_in(request: Request, response: Response, user=user):\n    return JSONResponse(\n        {"page": "home", "user": user.id, "datetime": str(datetime.utcnow())}\n    )\n\n\n@app.post("/b/logged-in")\n@invalidate_cache(\n    key="b.logged_in.{}", obj="user", obj_attr="id", namespace="test_namespace"\n)\nasync def post_logged_in(request: Request, response: Response, user=user):\n    return JSONResponse(\n        {"page": "home", "user": user.id, "datetime": str(datetime.utcnow())}\n    )\n\n```\n\n## Caching methods that aren\'t controllers\n\nC27Cache works exactly the same way with regular methods. The example below explains usage of the cache in service objects and application services.\n\n```python\nimport os\nimport redis\nfrom c27cache.config import C27Cache\nfrom c27cache.service import cache, invalidate_cache\n\nREDIS_URL = os.environ.get("REDIS_URL", "redis://localhost:6379/3")\nredis_client = redis.Redis.from_url(REDIS_URL)\n\nclass User:\n    id: str = "112358"\n\nuser = User()\n\n\nC27Cache.init(redis_url=REDIS_URL, namespace=\'test_namespace\')\n\n\n@cache(key=\'cache.me\', ttl_in_seconds=360)\nasync def cache_me(x:int, invoke_count:int):\n    invoke_count = invoke_count + 1\n    result = x * 2\n    return [result, invoke_count]\n````',
    'author': 'ragequilt',
    'author_email': 'sid.ravichandran@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/core-27/c27cache.git',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
