"""A structured set of metadata representing a deployed Cape function.

A :class:`FunctionRef` is intended to capture any/all metadata related to a Cape
function. The metadata is generally user-supplied, provided to them with the output of
the Cape CLI's ``deploy`` command.

**Usage**

::

    fid = "asdf231lkg1324afdg"
    fchecksum = str(b"2l1h21jhgb2k1jh3".hex())
    fref = FunctionRef(fid, fchecksum)

    cape = Cape()
    cape.connect(fref)
"""
import json
import os
import pathlib
from typing import Optional
from typing import Union


class FunctionRef:
    """A reference to a Cape function.

    Args:
        id: Required string denoting the function ID of the deployed Cape function.
            Typically given with the output of the Cape CLI's ``deploy`` command.
        token: Required string containing a Cape function token generated by the Cape
            CLI during ``cape token``.
        checksum: Optional string denoting the checksum of the deployed Cape function.
            If supplied as part of a ``FunctionRef``, the :class:`~pycape.cape.Cape`
            client will verify that enclave responses includes a matching checksum
            whenever the ``FunctionRef`` is included in Cape requests.
    """

    def __init__(
        self,
        id: str,
        token: str,
        checksum: Optional[str] = None,
    ):
        id_ = id
        if not isinstance(id_, str):
            raise TypeError(f"Function id must be a string, found {type(id_)}.")
        if not isinstance(token, str):
            raise TypeError(f"Function token must be a string, found {type(token)}.")
        if checksum is not None and not isinstance(checksum, str):
            raise TypeError(
                f"Function checksum must be a string, found {type(checksum)}."
            )
        self._id = id_
        self._checksum = checksum
        self._token = token

    @property
    def id(self):
        return self._id

    @property
    def checksum(self):
        return self._checksum

    @property
    def token(self):
        return self._token

    @classmethod
    def from_json(cls, token_path: Union[str, os.PathLike]):
        """
        Load a json file containing a function ID, token & checksum.

        Args:
            token_path: a json token file with a function ID, token
            and checksum (optional) generated by the Cape CLI's ``token`` command.

        Returns:
            A :class:`~.function_ref.FunctionRef` representing the deployed Cape
            function.

        Raises:
            ValueError: if the json token file doesn't exist or, the token file
                doesn't contain a `function_id` or a `function_token`.
        """
        if isinstance(token_path, str):
            token_path = pathlib.Path(token_path)

        if token_path.exists():
            with open(token_path, "r") as f:
                token_config = json.load(f)
        else:
            raise ValueError(
                "Couldn't find the token json file with the provided "
                f"path: {str(token_path)}"
            )

        function_id = token_config.get("function_id")
        if function_id is None:
            raise ValueError("Couldn't find a `function_id` in the token file provided")

        function_token = token_config.get("function_token")
        if function_token is None:
            raise ValueError(
                "Couldn't find a `function_token` in the token file provided"
            )

        function_checksum = token_config.get("function_checksum")

        return cls(function_id, function_token, function_checksum)
