"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StackResourceRenamer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
/**
 * StackResourceRenamer renames stack name and stack's subordinate resources' physical names, so that a CDK stack can be used to create multiple stacks in same AWS environment.
 *
 * @stability stable
 */
class StackResourceRenamer {
    /**
     * Construct a new StackResourceRenamer.
     *
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    constructor(renameOper, props = {}) {
        this.renameOper = renameOper;
        //mapping for resources whose physical names donot follow
        //the regular naming conventions: `${resourceType}`+'Name'
        this.irregularNames = {
            Output: 'exportName',
            ScalingPolicy: 'policyName',
            SlackChannelConfiguration: 'configurationName',
            CompositeAlarm: 'alarmName',
            SecurityGroup: 'groupName',
            DBProxy: 'dbProxyName',
        };
        //by default, only rename user provdied custom names
        this.customNameOnly = true;
        this.defaultNameField = 'name';
        if (props.irregularResourceNames) {
            this.irregularNames = {
                ...this.irregularNames,
                ...props.irregularResourceNames,
            };
        }
        this.excludeResTypes = props.excludeResourceTypes;
        this.includeResTypes = props.includeResourceTypes;
        if (props.userCustomNameOnly !== undefined) {
            this.customNameOnly = props.userCustomNameOnly;
        }
    }
    /**
     * Static method to rename a stack and all its subordinate resources.
     *
     * @param stack The stack (and all its children resources) to be renamed.
     * @param renameOper RenameOperation is used to rename stack name and resources' physical names.
     * @param props Properties are set to customize rename operations.
     * @stability stable
     */
    static rename(stack, renameOper, props = {}) {
        cdk.Aspects.of(stack).add(new StackResourceRenamer(renameOper, props));
    }
    /**
     * Implement core.IAspect interface.
     *
     * @param node CFN resources to be renamed.
     * @stability stable
     */
    visit(node) {
        if (node instanceof cdk.Stack) {
            //rename stack
            this.renameResource(node, 'Stack');
        }
        else {
            //rename CFN resources
            let ctorName = node.constructor.name;
            if (ctorName.startsWith('Cfn')) {
                this.renameResource(node, ctorName.substring(3));
            }
        }
    }
    /**
     * Rename a CFN resource or stack.
     *
     * @param node CFN resource or stack.
     * @param resTypeName The type name of CFN resource.
     * @stability stable
     */
    renameResource(node, resTypeName) {
        //check include/exclude
        if (this.excludeResTypes && this.excludeResTypes.length > 0 &&
            this.excludeResTypes.indexOf(resTypeName) !== -1) {
            return;
        }
        if (this.includeResTypes && this.includeResTypes.length > 0 &&
            this.includeResTypes.indexOf(resTypeName) === -1) {
            return;
        }
        //find the specific "name" field for CFN resources
        let physicalName = 'name';
        if (this.irregularNames[resTypeName]) {
            physicalName = this.irregularNames[resTypeName];
        }
        else {
            //decapitalize regular resource names
            let [first, ...rest] = resTypeName;
            let decapName = first.toLowerCase() + rest.join('');
            physicalName = `${decapName}Name`;
        }
        if (physicalName.length === 0) {
            return;
        }
        //some names (eg. stackName, exportName) only has getter,
        //need access protected fields starting with underscore
        let underscoreName = '_' + physicalName;
        //rename
        let b = node;
        if (b[physicalName] && b[physicalName].length > 0) {
            if (isWritable(b, physicalName) && this.isTarget(b[physicalName])) {
                b[physicalName] = this.renameOper.rename(b[physicalName], resTypeName);
            }
            else if (b[underscoreName] && b[underscoreName].length > 0 && isWritable(b, underscoreName) && this.isTarget(b[underscoreName])) {
                b[underscoreName] = this.renameOper.rename(b[underscoreName], resTypeName);
            }
        }
        else if (b[this.defaultNameField] && b[this.defaultNameField].length > 0 &&
            isWritable(b, this.defaultNameField) && this.isTarget(b[this.defaultNameField])) {
            b[this.defaultNameField] = this.renameOper.rename(b[this.defaultNameField], resTypeName);
        }
    }
    /**
     * check if a resName(resource name) is a valid target for rename;
     *
     * @stability stable
     */
    isTarget(resName) {
        let isAWSGenerated = cdk.Token.isUnresolved(resName);
        if (this.customNameOnly && isAWSGenerated) {
            return false;
        }
        return true;
    }
}
exports.StackResourceRenamer = StackResourceRenamer;
_a = JSII_RTTI_SYMBOL_1;
StackResourceRenamer[_a] = { fqn: "cdk-stack-resource-rename.StackResourceRenamer", version: "0.0.13" };
function isWritable(obj, key) {
    let desc;
    for (let o = obj; o != Object.prototype; o = Object.getPrototypeOf(o)) {
        desc = Object.getOwnPropertyDescriptor(o, key);
        if (desc !== undefined) {
            break;
        }
    }
    if (desc === undefined) {
        desc = {};
    }
    return Boolean(desc.writable);
}
//# sourceMappingURL=data:application/json;base64,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