/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.io;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.io.RetainableByteBufferPool;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Pool;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.annotation.ManagedOperation;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class ArrayRetainableByteBufferPool
implements RetainableByteBufferPool,
Dumpable {
    private static final Logger LOG = LoggerFactory.getLogger(ArrayRetainableByteBufferPool.class);
    private final Bucket[] _direct;
    private final Bucket[] _indirect;
    private final int _minCapacity;
    private final int _maxCapacity;
    private final long _maxHeapMemory;
    private final long _maxDirectMemory;
    private final AtomicLong _currentHeapMemory = new AtomicLong();
    private final AtomicLong _currentDirectMemory = new AtomicLong();
    private final Function<Integer, Integer> _bucketIndexFor;

    public ArrayRetainableByteBufferPool() {
        this(0, -1, -1, Integer.MAX_VALUE, -1L, -1L);
    }

    public ArrayRetainableByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize) {
        this(minCapacity, factor, maxCapacity, maxBucketSize, -1L, -1L);
    }

    public ArrayRetainableByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory) {
        this(minCapacity, factor, maxCapacity, maxBucketSize, maxHeapMemory, maxDirectMemory, null, null);
    }

    protected ArrayRetainableByteBufferPool(int minCapacity, int factor, int maxCapacity, int maxBucketSize, long maxHeapMemory, long maxDirectMemory, Function<Integer, Integer> bucketIndexFor, Function<Integer, Integer> bucketCapacity) {
        int f;
        if (minCapacity <= 0) {
            minCapacity = 0;
        }
        if (maxCapacity <= 0) {
            maxCapacity = 65536;
        }
        int n = f = factor <= 0 ? 1024 : factor;
        if (maxCapacity % f != 0 || f >= maxCapacity) {
            throw new IllegalArgumentException("The capacity factor must be a divisor of maxCapacity");
        }
        if (bucketIndexFor == null) {
            bucketIndexFor = c -> (c - 1) / f;
        }
        if (bucketCapacity == null) {
            bucketCapacity = i -> (i + 1) * f;
        }
        int length = bucketIndexFor.apply(maxCapacity) + 1;
        Bucket[] directArray = new Bucket[length];
        Bucket[] indirectArray = new Bucket[length];
        for (int i2 = 0; i2 < directArray.length; ++i2) {
            int capacity = Math.min(bucketCapacity.apply(i2), maxCapacity);
            directArray[i2] = new Bucket(capacity, maxBucketSize);
            indirectArray[i2] = new Bucket(capacity, maxBucketSize);
        }
        this._minCapacity = minCapacity;
        this._maxCapacity = maxCapacity;
        this._direct = directArray;
        this._indirect = indirectArray;
        this._maxHeapMemory = maxHeapMemory;
        this._maxDirectMemory = maxDirectMemory;
        this._bucketIndexFor = bucketIndexFor;
    }

    @ManagedAttribute(value="The minimum pooled buffer capacity")
    public int getMinCapacity() {
        return this._minCapacity;
    }

    @ManagedAttribute(value="The maximum pooled buffer capacity")
    public int getMaxCapacity() {
        return this._maxCapacity;
    }

    @Override
    public RetainableByteBuffer acquire(int size, boolean direct) {
        RetainableByteBuffer buffer;
        Bucket bucket = this.bucketFor(size, direct);
        if (bucket == null) {
            return this.newRetainableByteBuffer(size, direct, byteBuffer -> {});
        }
        Pool.Entry entry = bucket.acquire();
        if (entry == null) {
            Pool.Entry reservedEntry = bucket.reserve();
            if (reservedEntry != null) {
                buffer = this.newRetainableByteBuffer(bucket._capacity, direct, byteBuffer -> {
                    BufferUtil.clear(byteBuffer);
                    reservedEntry.release();
                });
                reservedEntry.enable(buffer, true);
                if (direct) {
                    this._currentDirectMemory.addAndGet(buffer.capacity());
                } else {
                    this._currentHeapMemory.addAndGet(buffer.capacity());
                }
                this.releaseExcessMemory(direct);
            } else {
                buffer = this.newRetainableByteBuffer(size, direct, byteBuffer -> {});
            }
        } else {
            buffer = (RetainableByteBuffer)entry.getPooled();
            buffer.acquire();
        }
        return buffer;
    }

    private RetainableByteBuffer newRetainableByteBuffer(int capacity, boolean direct, Consumer<ByteBuffer> releaser) {
        ByteBuffer buffer = direct ? ByteBuffer.allocateDirect(capacity) : ByteBuffer.allocate(capacity);
        BufferUtil.clear(buffer);
        RetainableByteBuffer retainableByteBuffer = new RetainableByteBuffer(buffer, releaser);
        retainableByteBuffer.acquire();
        return retainableByteBuffer;
    }

    private Bucket bucketFor(int capacity, boolean direct) {
        Bucket[] buckets;
        if (capacity < this._minCapacity) {
            return null;
        }
        int idx = this._bucketIndexFor.apply(capacity);
        Bucket[] bucketArray = buckets = direct ? this._direct : this._indirect;
        if (idx >= buckets.length) {
            return null;
        }
        return buckets[idx];
    }

    @ManagedAttribute(value="The number of pooled direct ByteBuffers")
    public long getDirectByteBufferCount() {
        return this.getByteBufferCount(true);
    }

    @ManagedAttribute(value="The number of pooled heap ByteBuffers")
    public long getHeapByteBufferCount() {
        return this.getByteBufferCount(false);
    }

    private long getByteBufferCount(boolean direct) {
        Bucket[] buckets = direct ? this._direct : this._indirect;
        return Arrays.stream(buckets).mapToLong(Pool::size).sum();
    }

    @ManagedAttribute(value="The number of pooled direct ByteBuffers that are available")
    public long getAvailableDirectByteBufferCount() {
        return this.getAvailableByteBufferCount(true);
    }

    @ManagedAttribute(value="The number of pooled heap ByteBuffers that are available")
    public long getAvailableHeapByteBufferCount() {
        return this.getAvailableByteBufferCount(false);
    }

    private long getAvailableByteBufferCount(boolean direct) {
        Bucket[] buckets = direct ? this._direct : this._indirect;
        return Arrays.stream(buckets).mapToLong(bucket -> bucket.values().stream().filter(Pool.Entry::isIdle).count()).sum();
    }

    @ManagedAttribute(value="The bytes retained by direct ByteBuffers")
    public long getDirectMemory() {
        return this.getMemory(true);
    }

    @ManagedAttribute(value="The bytes retained by heap ByteBuffers")
    public long getHeapMemory() {
        return this.getMemory(false);
    }

    private long getMemory(boolean direct) {
        if (direct) {
            return this._currentDirectMemory.get();
        }
        return this._currentHeapMemory.get();
    }

    @ManagedAttribute(value="The available bytes retained by direct ByteBuffers")
    public long getAvailableDirectMemory() {
        return this.getAvailableMemory(true);
    }

    @ManagedAttribute(value="The available bytes retained by heap ByteBuffers")
    public long getAvailableHeapMemory() {
        return this.getAvailableMemory(false);
    }

    private long getAvailableMemory(boolean direct) {
        Bucket[] buckets = direct ? this._direct : this._indirect;
        long total = 0L;
        for (Bucket bucket : buckets) {
            int capacity = bucket._capacity;
            total += bucket.values().stream().filter(Pool.Entry::isIdle).count() * (long)capacity;
        }
        return total;
    }

    @ManagedOperation(value="Clears this RetainableByteBufferPool", impact="ACTION")
    public void clear() {
        this.clearArray(this._direct, this._currentDirectMemory);
        this.clearArray(this._indirect, this._currentHeapMemory);
    }

    private void clearArray(Bucket[] poolArray, AtomicLong memoryCounter) {
        for (Bucket pool : poolArray) {
            for (Pool.Entry entry : pool.values()) {
                entry.remove();
                memoryCounter.addAndGet(-((RetainableByteBuffer)entry.getPooled()).capacity());
            }
        }
    }

    private void releaseExcessMemory(boolean direct) {
        long excess;
        long maxMemory;
        long l = maxMemory = direct ? this._maxDirectMemory : this._maxHeapMemory;
        if (maxMemory > 0L && (excess = this.getMemory(direct) - maxMemory) > 0L) {
            this.evict(direct, excess);
        }
    }

    private void evict(boolean direct, long excess) {
        Bucket[] buckets;
        if (LOG.isDebugEnabled()) {
            LOG.debug("evicting {} bytes from {} pools", (Object)excess, (Object)(direct ? "direct" : "heap"));
        }
        long now2 = System.nanoTime();
        long totalClearedCapacity = 0L;
        Bucket[] bucketArray = buckets = direct ? this._direct : this._indirect;
        while (totalClearedCapacity < excess) {
            for (Bucket bucket : buckets) {
                Pool.Entry oldestEntry = this.findOldestEntry(now2, bucket);
                if (oldestEntry == null || !oldestEntry.remove()) continue;
                int clearedCapacity = ((RetainableByteBuffer)oldestEntry.getPooled()).capacity();
                if (direct) {
                    this._currentDirectMemory.addAndGet(-clearedCapacity);
                } else {
                    this._currentHeapMemory.addAndGet(-clearedCapacity);
                }
                totalClearedCapacity += (long)clearedCapacity;
            }
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("eviction done, cleared {} bytes from {} pools", (Object)totalClearedCapacity, (Object)(direct ? "direct" : "heap"));
        }
    }

    public String toString() {
        return String.format("%s{min=%d,max=%d,buckets=%d,heap=%d/%d,direct=%d/%d}", super.toString(), this._minCapacity, this._maxCapacity, this._direct.length, this._currentHeapMemory.get(), this._maxHeapMemory, this._currentDirectMemory.get(), this._maxDirectMemory);
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, DumpableCollection.fromArray("direct", this._direct), DumpableCollection.fromArray("indirect", this._indirect));
    }

    private Pool.Entry findOldestEntry(long now2, Pool<RetainableByteBuffer> bucket) {
        Pool.Entry oldestEntry = null;
        for (Pool.Entry entry : bucket.values()) {
            if (oldestEntry != null) {
                long entryAge = now2 - ((RetainableByteBuffer)entry.getPooled()).getLastUpdate();
                if (entryAge <= now2 - ((RetainableByteBuffer)oldestEntry.getPooled()).getLastUpdate()) continue;
                oldestEntry = entry;
                continue;
            }
            oldestEntry = entry;
        }
        return oldestEntry;
    }

    private static class Bucket
    extends Pool<RetainableByteBuffer> {
        private final int _capacity;

        Bucket(int capacity, int size) {
            super(Pool.StrategyType.THREAD_ID, size, true);
            this._capacity = capacity;
        }

        @Override
        public String toString() {
            int entries = 0;
            int inUse = 0;
            for (Pool.Entry entry : this.values()) {
                ++entries;
                if (!entry.isInUse()) continue;
                ++inUse;
            }
            return String.format("%s{capacity=%d,inuse=%d(%d%%)}", super.toString(), this._capacity, inUse, entries > 0 ? inUse * 100 / entries : 0);
        }
    }
}

