"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const cdk = require("aws-cdk-lib");
const kinesis = require("aws-cdk-lib/aws-kinesis");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Default construct has all expected properties', () => {
    const stack = new cdk.Stack();
    const construct = new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.kinesisStream).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
test('Construct deploys Fargate Service in isolated subnets when publicApi is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    // The default isolated VPC should have two subnets, 2 route tables, and no nat/internet gateway, or routes
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::EC2::Subnet', 2);
    template.resourceCountIs('AWS::EC2::RouteTable', 2);
    template.resourceCountIs('AWS::EC2::Route', 0);
    template.resourceCountIs('AWS::EC2::NatGateway', 0);
    template.resourceCountIs('AWS::EC2::InternetGateway', 0);
    const subnet1Resources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: "Isolated"
                }
            ])
        }
    });
    const [subnet1, subnet2] = Object.keys(subnet1Resources);
    // Verify the Fargate Service is deployed into the two isolated subnets
    template.hasResourceProperties('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct deploys Fargate Service in private subnets when publicApi is set to true', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: true,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    // The default public/private VPC should have 4 subnets (two public, two private)
    // 4 route tables, 4 routes, 2 NAT Gateways and 1 Internet Gateway
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::EC2::Subnet', 4);
    template.resourceCountIs('AWS::EC2::RouteTable', 4);
    template.resourceCountIs('AWS::EC2::Route', 4);
    template.resourceCountIs('AWS::EC2::NatGateway', 2);
    template.resourceCountIs('AWS::EC2::InternetGateway', 1);
    const subnet1Resources = template.findResources('AWS::EC2::Subnet', {
        Properties: {
            Tags: assertions_1.Match.arrayWith([
                {
                    Key: "aws-cdk:subnet-type",
                    Value: "Private"
                }
            ])
        }
    });
    const [subnet1, subnet2] = Object.keys(subnet1Resources);
    // Verify the Fargate Service is deployed into the two private subnets
    template.hasResourceProperties('AWS::ECS::Service', {
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                Subnets: [
                    {
                        Ref: subnet1
                    },
                    {
                        Ref: subnet2
                    }
                ]
            },
        }
    });
});
test('Construct uses vpcProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: {
            vpcName: 'my-vpc'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct uses existingVpc when provided', () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack, false, {
        vpcName: 'my-vpc'
    });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingVpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::EC2::VPC', {
        Tags: [
            {
                Key: 'Name',
                Value: 'my-vpc'
            }
        ]
    });
});
test('Construct creates VPC Interface Endpoints for ECR and Kinesis Streams', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.kinesis-streams'
                ]
            ]
        },
    });
    template.hasResourceProperties('AWS::EC2::VPCEndpoint', {
        ServiceName: {
            'Fn::Join': [
                '',
                [
                    'com.amazonaws.',
                    {
                        Ref: 'AWS::Region'
                    },
                    '.ecr.api'
                ]
            ]
        },
    });
});
test('Container has default stream name environment variable', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'KINESIS_DATASTREAM_NAME',
                        Value: {
                            Ref: kinesisStreamId
                        }
                    }
                ],
            }
        ]
    });
});
test('Container stream name environment variable can be overridden', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        streamEnvironmentVariableName: 'my-stream-name'
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: 'my-stream-name',
                        Value: {
                            Ref: kinesisStreamId
                        }
                    }
                ],
            }
        ]
    });
});
test('Kinesis Stream is encrypted with AWS-managed CMK by default', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
});
test('CloudWatch Alarms are created for Kinesis Stream by default', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'GetRecords.IteratorAgeMilliseconds'
    });
    template.hasResourceProperties('AWS::CloudWatch::Alarm', {
        Namespace: 'AWS/Kinesis',
        MetricName: 'ReadProvisionedThroughputExceeded'
    });
});
test('CloudWatch Alarms are not created when createCloudWatchAlarms property is set to false', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        createCloudWatchAlarms: false
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::CloudWatch::Alarm', 0);
});
test('Construct uses existingStreamObj when provided', () => {
    const stack = new cdk.Stack();
    const existingStreamObj = new kinesis.Stream(stack, 'test-stream', {
        streamName: 'my-stream',
    });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        existingStreamObj
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct uses kinesisStreamProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        kinesisStreamProps: {
            streamName: 'my-stream',
            encryption: kinesis.StreamEncryption.UNENCRYPTED
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Kinesis::Stream', {
        Name: 'my-stream'
    });
    defaults.expectNonexistence(stack, 'AWS::Kinesis::Stream', {
        StreamEncryption: {
            EncryptionType: 'KMS',
            KeyId: 'alias/aws/kinesis'
        }
    });
    template.resourceCountIs('AWS::Kinesis::Stream', 1);
});
test('Construct grants PutRecord permission for the Fargate Service to write to the Kinesis Stream', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    // Get Kinesis Data Stream created by the construct
    const kinesisStream = template.findResources('AWS::Kinesis::Stream');
    const [kinesisStreamId] = Object.keys(kinesisStream);
    template.hasResourceProperties('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'kinesis:ListShards',
                        'kinesis:PutRecord',
                        'kinesis:PutRecords'
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            kinesisStreamId,
                            'Arn'
                        ]
                    }
                }
            ]
        }
    });
});
test('Construct defaults to the latest version of the ECR image', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:latest'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses ecrImageVersion when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        ecrImageVersion: 'my-version'
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ECS::TaskDefinition', {
        ContainerDefinitions: [
            {
                Image: {
                    'Fn::Join': [
                        '',
                        [
                            '123456789012.dkr.ecr.us-east-1.',
                            {
                                Ref: 'AWS::URLSuffix'
                            },
                            '/fake-repo:my-version'
                        ]
                    ]
                },
            }
        ]
    });
});
test('Construct uses clusterProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: {
            clusterName: 'my-cluster'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Cluster', 1);
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
});
test('Construct uses containerDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        containerDefinitionProps: {
            containerName: 'my-container'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
});
test('Construct uses fargateTaskDefinitionProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateTaskDefinitionProps: {
            family: 'my-family'
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: 'my-family'
    });
});
test('Construct uses fargateServiceProps when provided', () => {
    const stack = new cdk.Stack();
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        fargateServiceProps: {
            serviceName: 'my-service',
            desiredCount: 7
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: 'my-service',
        DesiredCount: 7
    });
});
test('Construct uses existingFargateServiceObject when provided', () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test-existing-fargate-service', {
        constructVpc: existingVpc,
        clientClusterProps: {
            clusterName: 'my-cluster'
        },
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clientFargateTaskDefinitionProps: {
            family: 'my-family'
        },
        clientContainerDefinitionProps: {
            containerName: 'my-container'
        },
        clientFargateServiceProps: {
            serviceName: 'my-service'
        }
    });
    new lib_1.FargateToKinesisStreams(stack, 'test-fargate-kinesisstreams', {
        publicApi: false,
        existingVpc,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::ECS::Cluster', 1);
    template.hasResourceProperties("AWS::ECS::Cluster", {
        ClusterName: 'my-cluster'
    });
    template.resourceCountIs('AWS::ECS::TaskDefinition', 1);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: 'my-family',
        ContainerDefinitions: [
            {
                Name: 'my-container'
            }
        ]
    });
    template.resourceCountIs('AWS::ECS::Service', 1);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: 'my-service',
    });
});
test('Confirm that CheckVpcProps was called', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "custom-family-name";
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc: defaults.getTestVpc(stack),
        vpcProps: {},
    };
    const app = () => {
        new lib_1.FargateToKinesisStreams(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm that CheckKinesisStreamsProps was called', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const clusterName = "custom-cluster-name";
    const containerName = "custom-container-name";
    const serviceName = "custom-service-name";
    const familyName = "custom-family-name";
    const props = {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        vpcProps: {},
        existingStreamObj: new kinesis.Stream(stack, 'test', {}),
        kinesisStreamProps: {}
    };
    const app = () => {
        new lib_1.FargateToKinesisStreams(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide existingStreamObj or kinesisStreamProps, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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