"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SageMakerCreateTrainingJob = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
const utils_1 = require("./private/utils");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 */
class SageMakerCreateTrainingJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        /**
         * Allows specify security group connections for instances of this fleet.
         */
        this.connections = new ec2.Connections();
        this.securityGroups = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SageMakerCreateTrainingJobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SageMakerCreateTrainingJob);
            }
            throw error;
        }
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the default resource config if not defined.
        this.resourceConfig = props.resourceConfig || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
            volumeSize: core_1.Size.gibibytes(10),
        };
        // set the stopping condition if not defined
        this.stoppingCondition = props.stoppingCondition || {
            maxRuntime: core_1.Duration.hours(1),
        };
        // check that either algorithm name or image is defined
        if (!props.algorithmSpecification.algorithmName && !props.algorithmSpecification.trainingImage) {
            throw new Error('Must define either an algorithm name or training image URI in the algorithm specification');
        }
        // set the input mode to 'File' if not defined
        this.algorithmSpecification = props.algorithmSpecification.trainingInputMode
            ? props.algorithmSpecification
            : { ...props.algorithmSpecification, trainingInputMode: base_types_1.InputMode.FILE };
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.inputDataConfig = props.inputDataConfig.map((config) => {
            if (!config.dataSource.s3DataSource.s3DataType) {
                return {
                    ...config,
                    dataSource: { s3DataSource: { ...config.dataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
                };
            }
            else {
                return config;
            }
        });
        // add the security groups to the connections object
        if (props.vpcConfig) {
            this.vpc = props.vpcConfig.vpc;
            this.subnets = props.vpcConfig.subnets ? this.vpc.selectSubnets(props.vpcConfig.subnets).subnetIds : this.vpc.selectSubnets().subnetIds;
        }
        this.taskPolicies = this.makePolicyStatements();
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    get grantPrincipal() {
        if (this._grantPrincipal === undefined) {
            throw new Error('Principal not available yet--use the object in a Task first');
        }
        return this._grantPrincipal;
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTrainingJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    renderParameters() {
        return {
            TrainingJobName: this.props.trainingJobName,
            EnableNetworkIsolation: this.props.enableNetworkIsolation,
            RoleArn: this._role.roleArn,
            ...this.renderAlgorithmSpecification(this.algorithmSpecification),
            ...this.renderInputDataConfig(this.inputDataConfig),
            ...this.renderOutputDataConfig(this.props.outputDataConfig),
            ...this.renderResourceConfig(this.resourceConfig),
            ...this.renderStoppingCondition(this.stoppingCondition),
            ...this.renderHyperparameters(this.props.hyperparameters),
            ...utils_1.renderTags(this.props.tags),
            ...this.renderVpcConfig(this.props.vpcConfig),
            ...utils_1.renderEnvironment(this.props.environment),
        };
    }
    renderAlgorithmSpecification(spec) {
        return {
            AlgorithmSpecification: {
                TrainingInputMode: spec.trainingInputMode,
                ...(spec.trainingImage ? { TrainingImage: spec.trainingImage.bind(this).imageUri } : {}),
                ...(spec.algorithmName ? { AlgorithmName: spec.algorithmName } : {}),
                ...(spec.metricDefinitions
                    ? { MetricDefinitions: spec.metricDefinitions.map((metric) => ({ Name: metric.name, Regex: metric.regex })) }
                    : {}),
            },
        };
    }
    renderInputDataConfig(config) {
        return {
            InputDataConfig: config.map((channel) => ({
                ChannelName: channel.channelName,
                DataSource: {
                    S3DataSource: {
                        S3Uri: channel.dataSource.s3DataSource.s3Location.bind(this, { forReading: true }).uri,
                        S3DataType: channel.dataSource.s3DataSource.s3DataType,
                        ...(channel.dataSource.s3DataSource.s3DataDistributionType
                            ? { S3DataDistributionType: channel.dataSource.s3DataSource.s3DataDistributionType }
                            : {}),
                        ...(channel.dataSource.s3DataSource.attributeNames ? { AttributeNames: channel.dataSource.s3DataSource.attributeNames } : {}),
                    },
                },
                ...(channel.compressionType ? { CompressionType: channel.compressionType } : {}),
                ...(channel.contentType ? { ContentType: channel.contentType } : {}),
                ...(channel.inputMode ? { InputMode: channel.inputMode } : {}),
                ...(channel.recordWrapperType ? { RecordWrapperType: channel.recordWrapperType } : {}),
            })),
        };
    }
    renderOutputDataConfig(config) {
        return {
            OutputDataConfig: {
                S3OutputPath: config.s3OutputLocation.bind(this, { forWriting: true }).uri,
                ...(config.encryptionKey ? { KmsKeyId: config.encryptionKey.keyArn } : {}),
            },
        };
    }
    renderResourceConfig(config) {
        return {
            ResourceConfig: {
                InstanceCount: config.instanceCount,
                InstanceType: sfn.JsonPath.isEncodedJsonPath(config.instanceType.toString())
                    ? config.instanceType.toString() : `ml.${config.instanceType}`,
                VolumeSizeInGB: config.volumeSize.toGibibytes(),
                ...(config.volumeEncryptionKey ? { VolumeKmsKeyId: config.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderStoppingCondition(config) {
        return {
            StoppingCondition: {
                MaxRuntimeInSeconds: config.maxRuntime && config.maxRuntime.toSeconds(),
            },
        };
    }
    renderHyperparameters(params) {
        return params ? { HyperParameters: params } : {};
    }
    renderVpcConfig(config) {
        return config
            ? {
                VpcConfig: {
                    SecurityGroupIds: core_1.Lazy.list({ produce: () => this.securityGroups.map((sg) => sg.securityGroupId) }),
                    Subnets: this.subnets,
                },
            }
            : {};
    }
    makePolicyStatements() {
        // set the sagemaker role or create new one
        this._grantPrincipal = this._role =
            this.props.role ||
                new iam.Role(this, 'SagemakerRole', {
                    assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                    inlinePolicies: {
                        CreateTrainingJob: new iam.PolicyDocument({
                            statements: [
                                new iam.PolicyStatement({
                                    actions: [
                                        'cloudwatch:PutMetricData',
                                        'logs:CreateLogStream',
                                        'logs:PutLogEvents',
                                        'logs:CreateLogGroup',
                                        'logs:DescribeLogStreams',
                                        'ecr:GetAuthorizationToken',
                                        ...(this.props.vpcConfig
                                            ? [
                                                'ec2:CreateNetworkInterface',
                                                'ec2:CreateNetworkInterfacePermission',
                                                'ec2:DeleteNetworkInterface',
                                                'ec2:DeleteNetworkInterfacePermission',
                                                'ec2:DescribeNetworkInterfaces',
                                                'ec2:DescribeVpcs',
                                                'ec2:DescribeDhcpOptions',
                                                'ec2:DescribeSubnets',
                                                'ec2:DescribeSecurityGroups',
                                            ]
                                            : []),
                                    ],
                                    resources: ['*'],
                                }),
                            ],
                        }),
                    },
                });
        if (this.props.outputDataConfig.encryptionKey) {
            this.props.outputDataConfig.encryptionKey.grantEncrypt(this._role);
        }
        if (this.props.resourceConfig && this.props.resourceConfig.volumeEncryptionKey) {
            this.props.resourceConfig.volumeEncryptionKey.grant(this._role, 'kms:CreateGrant');
        }
        // create a security group if not defined
        if (this.vpc && this.securityGroup === undefined) {
            this.securityGroup = new ec2.SecurityGroup(this, 'TrainJobSecurityGroup', {
                vpc: this.vpc,
            });
            this.connections.addSecurityGroup(this.securityGroup);
            this.securityGroups.push(this.securityGroup);
        }
        const stack = core_1.Stack.of(this);
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTrainingJob', 'sagemaker:DescribeTrainingJob', 'sagemaker:StopTrainingJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'training-job',
                        // If the job name comes from input, we cannot target the policy to a particular ARN prefix reliably...
                        resourceName: sfn.JsonPath.isEncodedJsonPath(this.props.trainingJobName) ? '*' : `${this.props.trainingJobName}*`,
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this._role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTrainingJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTrainingJob = SageMakerCreateTrainingJob;
_a = JSII_RTTI_SYMBOL_1;
SageMakerCreateTrainingJob[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SageMakerCreateTrainingJob", version: "1.192.0" };
SageMakerCreateTrainingJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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