import collections
import json
import setuptools
import distutils.core
import sys
import os

def parse_setup(filename):
    setup_py_filename = os.path.abspath(filename)
    package_path = os.path.dirname(filename)

    setup_args = [None]
    setup_modules = [setuptools, distutils.core]

    # keep track of unpatched functions so we can restore afterwards
    unpatched = [(m, m.setup) for m in setup_modules]

    # patch setup functions to just keep track of arguments passed to them
    def patched_setup(**kwargs):
        setup_args[0] = kwargs

    for module in setup_modules:
        module.setup = patched_setup

    # set up pythonpath/argv like most setup.py scripts expect
    old_path = sys.path
    old_argv = sys.argv
    os.chdir(package_path)
    sys.path = [package_path] + sys.path
    sys.argv = [setup_py_filename, "install"]

    # hack for getsentry/sentry
    sys.modules['__main__'].__file__ = setup_py_filename

    try:
        with open(setup_py_filename) as f:
            exec(f.read(), {
                 "__name__": "__main__",
                 "__builtins__": __builtins__,
                 "__file__": setup_py_filename})
    except ImportError:
        pass
    finally:
        # restore setup functions that have been patched over
        for module, setupfn in unpatched:
            module.setup = setupfn
        sys.path = old_path
        sys.argv = old_argv

    ret = setup_args[0]
    if ret is None:
        raise ValueError("setup wasn't called from setup.py")

    return ret

def get_install_requires(result_dic):
    res_str = ""
    if "install_requires" in result_dic:
        for dep in result_dic["install_requires"]:
            res_str += dep + os.linesep
    return res_str

if __name__ == "__main__":
    import argparse
    parser = argparse.ArgumentParser("Parse setup.py files")
    parser.add_argument('filename')
    args = parser.parse_args()
    results = parse_setup(args.filename)

    print(get_install_requires(results))
