"""Parse 3P xls output files."""

from datetime import datetime

import openpyxl

from pygaps import logger

_META_DICT = {
    'material': {
        'text': ['charge name'],
        'type': 'string'
    },
    'adsorbate': {
        'text': ['adsorbate'],
        'type': 'string'
    },
    'date': {
        'text': ['started time'],
        'type': 'date'
    },
    'material_mass': {
        'text': ['sample weight'],
        'type': 'number'
    },
}

_DATA_DICT = {
    'id': 'measurement',
    'p (': 'pressure',
    'p0 (': 'pressure_saturation',
    'p/p0': 'pressure_relative',
    'v (': 'loading',
    'time': 'point_time',
}


def parse(path):
    """
    Parse an xls file generated by 3P software.

    Parameters
    ----------
    path: str
        The location of an xls file generated by a 3P instrument.

    Returns
    -------
    dict
        A dictionary containing report information.
    """
    meta = {}
    data = {}

    workbook = openpyxl.load_workbook(path, read_only=True, data_only=True)

    # Metadata
    info_sheet = workbook['Info']
    for row in info_sheet.values:
        if not row[0]:
            continue
        key = row[0].lower()
        val = row[1]
        try:
            name = next(
                k for k, v in _META_DICT.items()
                if any(key.startswith(n) for n in v.get('text', []))
            )
        except StopIteration:
            if val:
                meta[key] = val
            continue

        tp = _META_DICT[name]['type']

        if tp == 'number':
            meta[name] = val
        elif tp == 'date':
            meta[name] = _handle_date(val)
        elif tp == 'string':
            meta[name] = _handle_string(val)

    # Data
    data_sheet = workbook['Isotherm']
    # Data headers
    data_val = data_sheet.values
    head, units = _parse_header(list(next(data_val)))
    meta.update(units)
    # Prepare data
    data = []
    branch = 0
    for row in list(data_val):
        # If we reached the desorption branch we change
        if row[0] == "---":
            branch = 1
            continue
        data.append([branch] + list(row))

    data = dict(zip(head, map(lambda *x: list(x), *data)))

    _check(meta, data, path)

    # Set extra metadata
    meta['apparatus'] = '3P'
    meta['temperature'] = 77.4  # TODO where is this stored?
    meta['temperature_unit'] = "K"  # TODO where is this stored?
    meta['pressure_mode'] = 'absolute'
    meta['loading_basis'] = 'molar'
    meta['material_basis'] = 'mass'

    return meta, data


def _handle_date(val):
    """
    Convert date to string.

    Input is a cell of type 'date'.
    """
    if val:
        return datetime.strptime(val, r'%Y-%m-%d %H:%M:%S').isoformat()


def _handle_string(val):
    """
    Replace any newline found.

    Input is a cell of type 'string'.
    """
    return val.replace('\r\n', ' ')


def _parse_header(header_list):
    """Parse an adsorption/desorption header to get columns and units."""
    headers = ['branch']
    units = {}

    for h in header_list:
        txt = next((_DATA_DICT[a] for a in _DATA_DICT if h.lower().startswith(a)), h)
        headers.append(txt)

        if txt == 'loading':
            units['loading_basis'] = 'molar'
            for (u, c) in (
                ('mmol', 'mmol'),
                ('mol', 'mol'),
                ('(cm³/g STP)', 'cm3(STP)'),
            ):
                if u in h:
                    units['loading_unit'] = c
            units['material_basis'] = 'mass'
            for (u, c) in (
                ('/g', 'g'),
                ('/kg', 'kg'),
            ):
                if u in h:
                    units['material_unit'] = c

        if txt == 'pressure':
            units['pressure_mode'] = 'absolute'
            for (u, c) in (
                ('torr', 'torr'),
                ('kPa', 'kPa'),
                ('bar', 'bar'),
            ):
                if u in h:
                    units['pressure_unit'] = c

    return headers, units


def _check(meta, data, path):
    """
    Check keys in data and logs a warning if a key is empty.

    Also logs a warning for errors found in file.
    """
    if 'loading' in data:
        empties = (k for k, v in data.items() if not v)
        for empty in empties:
            logger.info(f"No data collected for {empty} in file {path}.")
    if 'errors' in meta:
        logger.warning('\n'.join(meta['errors']))
