from sage.rings.number_field.number_field import NumberField
from sage.rings.real_mpfr import RR
from sage.matrix.constructor import matrix
from sage.modules.free_module_element import vector, zero_vector
from copy import copy
from sage.plot.graphics import Graphics
from sage.plot.point import points

def to_pt (c):
    """
    Convert a complex number to a couple.

    INPUT:
        - ``c`` - complex number

    OUTPUT:
        A couple (real part, imaginary part)
    """
    try:
        c = CC(c)
    except:
        return c
    return (real(c), imag(c))

def plot_points (lpts, size=1, colormap='hsv'):
    """
    Plot a list of lists of points, with colors.

    INPUT:
        - ``lpts`` - list -- list of lists of points

    OUTPUT:
        A Graphics object.
    """
    try:
        for l in lpts:
            if len(l) > 0:
                l[0][0]
    except:
        lpts = [[to_pt(p) for p in l] for l in lpts]
    from matplotlib import cm
    try:
        colormap = cm.__dict__[colormap]
    except:
        try:
            str(col)
            raise RuntimeError("Color map %s not known (type sorted(colors) for valid names)" % colormap)
        except:
            col = colormap
    else:
        col = []
        for i in range(len(lpts)):
            col.append(colormap(float(i)/float(len(lpts)))[:3])
    g = Graphics()
    for c,l in zip(col, lpts):
        g += points(l, size=size, color=c)
    return g

def usual_projection (m, conform=True):
    """
    Return the projection on the contracting space of m
    (i.e. the subspace generated by eigenvectors for conjugates of the Perron eigenvalue)
    along every other generalized eigenspaces.

    INPUT:
        - ``m`` - a square matrix of size n

    OUTPUT:
        A matrix of size k x n, where k+1 is the degree of the Perron eigenvalue

    EXAMPLES::
        sage: from eigenmorphic import *
        sage: s = WordMorphism("a->ab,b->ac,c->a")
        sage: m = s.incidence_matrix()
        sage: usual_projection(m)
        [  1.00000000000000  -1.41964337760708 -0.771844506346038]
        [ 0.000000000000000  0.606290729207199  -1.11514250803994]
    """
    if conform:
        vp,lv,_ = max(m.eigenvectors_left())
        K = NumberField(vp.minpoly(), 'b', embedding=vp)
        lp = [p for p in K.places() if abs(p(K.gen())) < 1]
        # V = matrix([vector([p(t) for t in lv[0]]) for p in lp]) # to get pts as complex numbers
        V = []
        for p in lp:
            if p.im_gens()[0] in RR:
                V.append([p(t) for t in lv[0]])
            else:
                V.append([p(t).real() for t in lv[0]])
                V.append([p(t).imag() for t in lv[0]])
        V = matrix(V)
    else:
        vp = max(m.eigenvalues())
        P = m.minpoly()/vp.minpoly()
        P *= (P.parent().gen() - vp)
        V = matrix(matrix((P(m)).right_kernel().basis()).transpose().kernel().basis())
        V = matrix(V, ring=RR)
    return V
    
def rauzy_points (u, V, A=None, exchange=False, translate=None, verb=False):
    """
    Compute points of the Rauzy fractal of word u for the projection V.

    INPUT:
        - ``u`` - a finite word on some alphabet A

        - ``V`` - a projection from R^A to R^k, given as a k x len(A) matrix.

        - ``A`` - list (default: ``None``) -- alphabet of the word

        - ``exchange`` - bool (default: ``False``) -- if True, exchange pieces

        - ``translate`` - list (default: ``None``) -- list of vectors of Z^A, translations of the broken line

        - ``verb`` - bool (default: ``False``) -- if > 0, print informations

    OUTPUT:
        A list of points in R^k.
    """
    if A is None:
        A = list(u.parent().alphabet())
    v = zero_vector(len(A))
    if translate is None:
        translate = [copy(v)]
    else:
        translate = [vector(v) for v in translate]
    d = dict()
    for i,a in enumerate(A):
        d[a] = i
    if verb > 0:
        print(d)
    lpts = [[] for _ in A]
    for a in u:
        if exchange:
            v[d[a]] += 1
        for t in translate:
            lpts[d[a]].append(V*(v+t))
        if not exchange:
            v[d[a]] += 1
    return lpts

def rauzy_fractal_plot (u, V, A=None, size=1, colormap='hsv', exchange=False, translate=None):
    """
    Plot the rauzy fractal for the word u, with projection V

    INPUT:
        - ``u`` - a finite word on some alphabet A

        - ``V`` - a projection from R^A to R^k, given as a k x len(A) matrix.

        - ``A`` - list (default: ``None``) -- alphabet of the word

        - ``size`` - float (default: ``1``) -- size of points

        - ``colormap`` - str or map (default: 'hsv') -- colors used to draw each piece of the Rauzy fractal

        - ``exchange`` - bool (default: ``False``) -- if True, exchange pieces

        - ``translate`` - list (default: ``None``) -- list of vectors of Z^A, translations of the broken line

    OUTPUT:
        A Graphics object.

    EXAMPLES::
        sage: from eigenmorphic import *

        # plot the classical Rauzy fractal
        sage: N = 60000
        sage: s = WordMorphism('a->ab,b->ac,c->a')
        sage: u = s.fixed_point('a')
        sage: V = usual_projection(s.incidence_matrix())
        sage: rauzy_fractal_plot(u[:N], V)          # not tested

        # plot a Rauzy fractal for a substitution with a pre-period
        sage: N = 60000
        sage: s = WordMorphism('a->ab,b->ac,c->a')
        sage: t = WordMorphism('a->ba,b->ac,c->a')^5
        sage: u = s.fixed_point('a')
        sage: w = t(u)
        sage: V = usual_projection(s.incidence_matrix())*(t.incidence_matrix()^(-1))
        sage: rauzy_fractal_plot(w[:N], V)      # not tested
    """
    lpts = rauzy_points(u, V, A=A, exchange=exchange, translate=translate)
    return plot_points(lpts, size=size, colormap=colormap)
