"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeleteImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implementation for aws:deleteImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html
 */
class DeleteImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:deleteImage';
        this.deleteImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        const imageId = this.deleteImageStep.imageId.resolveToString(inputs);
        console.log(`DeleteImage: Get snapshots of ${imageId}`);
        const snapshots = this.getSnapshots(imageId);
        console.log(`DeleteImage: Deregister ${imageId}`);
        this.deregisterImage(imageId);
        console.log(`DeleteImage: Delete snapshots for ${imageId}`);
        this.deleteSnapshotsOfImage(snapshots);
        console.log(`DeleteImage: ${imageId} is deleted`);
        return {};
    }
    getSnapshots(imageId) {
        var _b, _c;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeImage', {
            service: 'EC2',
            pascalCaseApi: 'DescribeImages',
            apiParams: {
                ImageIds: [imageId],
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    selector: '$.Images[0]',
                    name: 'Image',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Describe image failed for ${imageId}: ${result.stackTrace}`);
        }
        const image = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['describeImage.Image'];
        return (_c = image === null || image === void 0 ? void 0 : image.BlockDeviceMappings.map(mapping => mapping.Ebs.SnapshotId)) !== null && _c !== void 0 ? _c : [];
    }
    deregisterImage(imageId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deregisterImage', {
            service: 'EC2',
            pascalCaseApi: 'DeregisterImage',
            apiParams: {
                ImageId: imageId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Deregister image failed for ${imageId}: ${result.stackTrace}`);
        }
    }
    deleteSnapshotsOfImage(snapshotIds) {
        for (const snapshotId of snapshotIds) {
            if (snapshotId === null || snapshotId === undefined) {
                continue;
            }
            try {
                this.deleteSnapshot(snapshotId);
            }
            catch (e) {
                if (e.message.indexOf('InvalidSnapshot.InUse') < 0) {
                    throw e;
                }
                console.log('InvalidSnapshot.InUse case found. Attempt to retry after 2 seconds.');
                new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                    sleepSeconds: 2,
                }), this.props).invoke({});
                this.deleteSnapshot(snapshotId);
            }
        }
    }
    deleteSnapshot(snapshotId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deleteSnapshot', {
            service: 'EC2',
            pascalCaseApi: 'DeleteSnapshot',
            apiParams: {
                SnapshotId: snapshotId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Delete snapshot failed for ${snapshotId}: ${result.stackTrace}`);
        }
    }
}
exports.DeleteImageSimulation = DeleteImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
DeleteImageSimulation[_a] = { fqn: "cdk-ssm-documents.DeleteImageSimulation", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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