"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTagsSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:createTags
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html
 */
class CreateTagsSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createTags';
        this.createTagsStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        var _b, _c;
        const resourceType = (_c = (_b = this.createTagsStep.resourceType) === null || _b === void 0 ? void 0 : _b.resolveToString(inputs)) !== null && _c !== void 0 ? _c : 'EC2';
        switch (resourceType) {
            case 'EC2':
                this.createEc2Tags(inputs);
                break;
            case 'ManagedInstance':
            case 'MaintenanceWindow':
            case 'Parameter':
                this.createSsmTags(resourceType, inputs);
                break;
            default:
                throw new Error(`${resourceType} is not a valid resource type for tagging`);
        }
        return {};
    }
    createEc2Tags(inputs) {
        const resourceIds = this.createTagsStep.resourceIds.resolveToStringList(inputs);
        const tags = this.createTagsStep.tags.resolveToMapList(inputs);
        console.log('CreateTags: Create EC2 tags for resources');
        this.callEc2CreateTags(resourceIds, tags);
        console.log('CreateTags: Wait for tags to be completed');
        this.waitForEc2Tags(resourceIds, tags);
        console.log('CreateTags: EC2 tags are associated to the resources');
    }
    callEc2CreateTags(resourceIds, tags) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ec2CreateTags', {
            service: 'EC2',
            pascalCaseApi: 'CreateTags',
            apiParams: {
                Resources: resourceIds,
                Tags: tags,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create EC2 tags failed: ${result.stackTrace}`);
        }
    }
    waitForEc2Tags(resourceIds, tags) {
        while (!this.areEc2TagsAssociatedToTheResources(resourceIds, tags)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    areEc2TagsAssociatedToTheResources(resourceIds, tags) {
        const tagKeys = tags.map(tag => tag.Key);
        const tagValues = tags.map(tag => tag.Value);
        const filters = this.buildEc2TagFilter(resourceIds, tagKeys, tagValues);
        const tagDescriptions = this.ec2DescribeTags(filters);
        for (const tag of tags) {
            const descriptionsOfTag = tagDescriptions.filter(x => x.Key === tag.Key && x.Value === tag.Value);
            const describedResourceIds = new Set(descriptionsOfTag.map(x => x.ResourceId));
            if (!resourceIds.every(x => describedResourceIds.has(x))) {
                return false;
            }
        }
        return true;
    }
    buildEc2TagFilter(resourceIds, tagKeys, tagValues) {
        const potentialFilters = {
            'resource-id': resourceIds,
            'key': tagKeys,
            'value': tagValues,
        };
        const filters = [];
        for (const name of Object.keys(potentialFilters)) {
            if (potentialFilters[name].length !== 0) {
                filters.push({ Name: name, Values: potentialFilters[name] });
            }
        }
        return filters;
    }
    ec2DescribeTags(filters) {
        var _b, _c;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ec2DescribeTags', {
            service: 'EC2',
            pascalCaseApi: 'DescribeTags',
            apiParams: {
                Filters: filters,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    selector: '$.Tags',
                    name: 'Tags',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Describe EC2 tags failed: ${result.stackTrace}`);
        }
        return (_c = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['ec2DescribeTags.Tags']) !== null && _c !== void 0 ? _c : [];
    }
    createSsmTags(resourceType, inputs) {
        const resourceId = this.getSingleResourceId(resourceType, inputs);
        const tags = this.createTagsStep.tags.resolveToMapList(inputs);
        console.log('CreateTags: Create tags for resource');
        this.callSsmAddTagsToResource(resourceType, resourceId, tags);
        console.log('CreateTags: Wait for tags to be completed');
        this.waitForSsmTags(resourceType, resourceId, tags);
        console.log('CreateTags: Tags are associated to the resource');
    }
    callSsmAddTagsToResource(resourceType, resourceId, tags) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ssmAddTagsToResource', {
            service: 'SSM',
            pascalCaseApi: 'AddTagsToResource',
            apiParams: {
                ResourceType: resourceType,
                ResourceId: resourceId,
                Tags: tags,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Add tags failed: ${result.stackTrace}`);
        }
    }
    waitForSsmTags(resourceType, resourceId, tags) {
        while (!this.areAllSsmTagsListed(resourceType, resourceId, tags)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    areAllSsmTagsListed(resourceType, resourceId, tags) {
        const listedTags = this.ssmListTagsForResource(resourceType, resourceId);
        const listedTagMap = new Map(listedTags.map(tag => [tag.Key, tag.Value]));
        return tags.every(tag => listedTagMap.get(tag.Key) === tag.Value);
    }
    ssmListTagsForResource(resourceType, resourceId) {
        var _b, _c;
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'ssmListTagsForResource', {
            service: 'SSM',
            pascalCaseApi: 'ListTagsForResource',
            apiParams: {
                ResourceType: resourceType,
                ResourceId: resourceId,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    selector: '$.TagList',
                    name: 'Tags',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`SSM List tags failed: ${result.stackTrace}`);
        }
        return (_c = (_b = result.outputs) === null || _b === void 0 ? void 0 : _b['ssmListTagsForResource.Tags']) !== null && _c !== void 0 ? _c : [];
    }
    getSingleResourceId(resourceType, inputs) {
        const resourceIds = this.createTagsStep.resourceIds.resolve(inputs);
        if (resourceIds.length !== 1) {
            throw new Error(`Only 1 resource allowed for type ${resourceType}`);
        }
        return resourceIds[0];
    }
}
exports.CreateTagsSimulation = CreateTagsSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateTagsSimulation[_a] = { fqn: "cdk-ssm-documents.CreateTagsSimulation", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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