"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateStackSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const url_1 = require("url");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const stack_status_1 = require("../../domain/stack-status");
const string_variable_1 = require("../../interface/variables/string-variable");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const delete_stack_step_1 = require("../../parent-steps/automation/delete-stack-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implementation for aws:createStack
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html
 */
class CreateStackSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:createStack';
        this.creatStackStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        var _b;
        this.verifyUrlProps(inputs);
        const stackName = this.creatStackStep.stackName.resolveToString(inputs);
        console.log(`CreateStack: Checking that ${stackName} is ready to be created`);
        this.preVerifyStackSubStep(stackName);
        console.log(`CreateStack: Creating ${stackName}`);
        const stackId = this.createStack(inputs);
        console.log(`CreateStack: Waiting for ${stackName} completion`);
        this.waitForStackCompletion(stackName);
        console.log('CreateStack: Stack is created');
        const stack = this.getStackState(stackName);
        return {
            StackId: stackId,
            StackStatus: stack === null || stack === void 0 ? void 0 : stack.StackStatus,
            StackStatusReason: (_b = stack === null || stack === void 0 ? void 0 : stack.StackStatusReason) !== null && _b !== void 0 ? _b : '',
        };
    }
    verifyUrlProps(inputs) {
        return this.verifyUrlProp(this.creatStackStep.templateUrl, inputs)
            && this.verifyUrlProp(this.creatStackStep.stackPolicyUrl, inputs);
    }
    verifyUrlProp(urlToVerify, inputs) {
        return urlToVerify === undefined || new url_1.URL(urlToVerify.resolveToString(inputs));
    }
    preVerifyStackSubStep(stackName) {
        while (!this.preVerifyStack(stackName)) {
            new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
        }
    }
    preVerifyStack(stackName) {
        const state = this.getStackState(stackName);
        if (this.isStackOperationInProgress(state)) {
            return false;
        }
        if ((state === null || state === void 0 ? void 0 : state.StackStatus) !== 'ROLLBACK_COMPLETE') {
            return true;
        }
        console.log(`Deleting stack ${stackName} because we are retrying a create on an existing stack in ROLLBACK_COMPLETE state.`);
        new automation_step_simulation_1.AutomationStepSimulation(new delete_stack_step_1.DeleteStackStep(new aws_cdk_lib_1.Stack(), 'deleteStack', {
            stackNameVariable: new string_variable_1.HardCodedString(stackName),
        }), this.props).invoke({});
        return false;
    }
    isStackOperationInProgress(state) {
        return state !== null && CreateStackSimulation.InProgressStatuses.includes(stack_status_1.StackStatus[state.StackStatus]);
    }
    createStack(inputs) {
        var _b, _c;
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord({
            StackName: this.creatStackStep.stackName,
            Capabilities: this.creatStackStep.capabilities,
            ClientRequestToken: this.creatStackStep.clientRequestToken,
            NotificationARNs: this.creatStackStep.notificationARNs,
            OnFailure: this.creatStackStep.onStackFailure,
            ResourceTypes: this.creatStackStep.resourceTypes,
            RoleARN: this.creatStackStep.roleArn,
            StackPolicyBody: this.creatStackStep.stackPolicyBody,
            StackPolicyURL: this.creatStackStep.stackPolicyUrl,
            Tags: this.creatStackStep.tags,
            TemplateBody: this.creatStackStep.templateBody,
            TemplateURL: this.creatStackStep.templateUrl,
            TimeoutInMinutes: this.creatStackStep.timeoutInMinutes,
        }, x => x.resolve(inputs));
        const parameters = this.resolveSecureStringsInParametersInput((_b = this.creatStackStep.parameters) === null || _b === void 0 ? void 0 : _b.resolveToMapList(inputs));
        if (parameters) {
            apiParams.Parameters = parameters;
        }
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'createStack', {
            service: 'CloudFormation',
            pascalCaseApi: 'CreateStack',
            apiParams: apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING,
                    selector: '$.StackId',
                    name: 'StackId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Create stack failed for ${apiParams.StackName}: ${result.stackTrace}`);
        }
        return (_c = result.outputs) === null || _c === void 0 ? void 0 : _c['createStack.StackId'];
    }
    resolveSecureStringsInParametersInput(parameters) {
        if (!parameters) {
            return null;
        }
        for (const parameter of parameters) {
            const value = parameter.ParameterValue;
            if (!value) {
                continue;
            }
            parameter.ParameterValue = this.props.parameterResolver.resolve(value);
        }
        return parameters;
    }
    getStackState(stackName) {
        var _b, _c;
        const describeResponse = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeStack', {
            service: 'CloudFormation',
            pascalCaseApi: 'DescribeStacks',
            apiParams: {
                StackName: stackName,
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    name: 'Stack',
                    selector: '$.Stacks[0]',
                }],
        }), this.props).invoke({});
        return (_c = (_b = describeResponse.outputs) === null || _b === void 0 ? void 0 : _b['describeStack.Stack']) !== null && _c !== void 0 ? _c : null;
    }
    waitForStackCompletion(stackName) {
        wait_for_and_assert_1.waitForAndAssertStackStatus({
            stackName: stackName,
            waitForStatus: [
                stack_status_1.StackStatus.CREATE_FAILED,
                stack_status_1.StackStatus.CREATE_COMPLETE,
                stack_status_1.StackStatus.ROLLBACK_FAILED,
                stack_status_1.StackStatus.ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.DELETE_IN_PROGRESS,
                stack_status_1.StackStatus.DELETE_FAILED,
                stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_COMPLETE,
                stack_status_1.StackStatus.UPDATE_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_FAILED,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
                stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE,
                stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_COMPLETE,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_IN_PROGRESS,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_FAILED,
                stack_status_1.StackStatus.IMPORT_ROLLBACK_COMPLETE,
            ],
            assertStatus: stack_status_1.StackStatus.CREATE_COMPLETE,
        }, this.props);
    }
}
exports.CreateStackSimulation = CreateStackSimulation;
_a = JSII_RTTI_SYMBOL_1;
CreateStackSimulation[_a] = { fqn: "cdk-ssm-documents.CreateStackSimulation", version: "0.0.12" };
CreateStackSimulation.InProgressStatuses = [
    stack_status_1.StackStatus.ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.DELETE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_IN_PROGRESS,
    stack_status_1.StackStatus.UPDATE_ROLLBACK_COMPLETE_CLEANUP_IN_PROGRESS,
    stack_status_1.StackStatus.REVIEW_IN_PROGRESS,
];
//# sourceMappingURL=data:application/json;base64,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