"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ChangeInstanceStateSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:changeInstanceState
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html
 */
class ChangeInstanceStateSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:changeInstanceState';
        this.changeInstanceStateStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        var _b, _c;
        const checkStateOnly = (_c = (_b = this.changeInstanceStateStep.checkStateOnly) === null || _b === void 0 ? void 0 : _b.resolveToBoolean(inputs)) !== null && _c !== void 0 ? _c : false;
        const desiredState = this.changeInstanceStateStep.desiredState.resolveToString(inputs);
        if (!checkStateOnly) {
            console.log(`ChangeInstanceState: Setting instances state to ${desiredState}`);
            this.changeInstanceState(desiredState, inputs);
        }
        console.log(`ChangeInstanceState: Waiting for instances to be ${desiredState}`);
        this.waitForInstanceState(desiredState, inputs);
        console.log(`ChangeInstanceState: Instances are now ${desiredState}`);
        return {};
    }
    changeInstanceState(desiredState, inputs) {
        switch (desiredState) {
            case 'running':
                return this.changeInstanceStateToRunning(inputs);
            case 'stopped':
                return this.changeInstanceStateToStopped(inputs);
            case 'terminated':
                return this.changeInstanceStateToTerminated(inputs);
            default:
                throw new Error(`Desired state: ${desiredState} is not an end-state to change to.`);
        }
    }
    changeInstanceStateToRunning(inputs) {
        this.changeInstanceStateCall('StartInstances', {
            AdditionalInfo: this.changeInstanceStateStep.additionalInfo,
        }, inputs);
    }
    changeInstanceStateToStopped(inputs) {
        this.changeInstanceStateCall('StopInstances', {
            Force: this.changeInstanceStateStep.force,
        }, inputs);
    }
    changeInstanceStateToTerminated(inputs) {
        this.changeInstanceStateCall('TerminateInstances', {}, inputs);
    }
    changeInstanceStateCall(endpoint, moreParams, inputs) {
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord({
            InstanceIds: this.changeInstanceStateStep.instanceIds,
            ...moreParams,
        }, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'changeInstanceState', {
            service: 'EC2',
            pascalCaseApi: endpoint,
            apiParams,
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Change instance state failed for ${apiParams.InstanceIds}: ${result.stackTrace}`);
        }
    }
    waitForInstanceState(desiredState, inputs) {
        const instanceIds = this.changeInstanceStateStep.instanceIds.resolveToStringList(inputs);
        switch (desiredState) {
            case 'running':
                return this.waitForStateRunningAndStatusOk(instanceIds);
            case 'stopped':
                return this.waitForStateStopped(instanceIds);
            case 'terminated':
                return this.waitForStateTerminated(instanceIds);
            default:
                throw new Error(`Desired state: ${desiredState} is not an end-state to wait for.`);
        }
    }
    waitForStateRunningAndStatusOk(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['shutting-down', 'terminated', 'running'],
            assertState: 'running',
            instanceIds: instanceIds,
        }, this.props);
        // For running, we also check if the state is 'ok'
        wait_for_and_assert_1.waitForAndAssertInstanceStatus({
            waitForStatus: ['ok', 'impaired', 'insufficient-data', 'not-applicable'],
            assertStatus: 'ok',
            instanceIds: instanceIds,
        }, this.props);
    }
    waitForStateStopped(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['shutting-down', 'terminated', 'stopped'],
            assertState: 'stopped',
            instanceIds: instanceIds,
        }, this.props);
    }
    waitForStateTerminated(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['terminated'],
            assertState: 'terminated',
            instanceIds: instanceIds,
        }, this.props);
    }
}
exports.ChangeInstanceStateSimulation = ChangeInstanceStateSimulation;
_a = JSII_RTTI_SYMBOL_1;
ChangeInstanceStateSimulation[_a] = { fqn: "cdk-ssm-documents.ChangeInstanceStateSimulation", version: "0.0.12" };
//# sourceMappingURL=data:application/json;base64,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